/*
 * Copyright (c) 2003, 2004 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using Nemerle.Collections;
using Nemerle.Compiler.Parsetree;

namespace Nemerle.Compiler 
{
  public module Passes 
  {
    internal mutable Solver : Solver;
    public mutable Hierarchy : TypesManager;

    internal mutable tyinfo_counter : int;
    mutable current_tyinfo_count : int;
    mutable current_pb_stage : int;
    
    mutable pb_killed : bool;

    
    // ---- PIPELINES USED IN COMPILATION PROCESS ----

    /** Pipeline function for creating lexer object from given string.

        In default compilation it is a function creating LexerFile from given
        filename. It can be customized to do any other action, like treating
        given string as textual code and creating LexerString from it.
     */
    public mutable LexingPipeline : string -> LexerFile;

    /** Pipeline for performing parsing of lexical tokens into parsetree.
 
        In default compilation it is using provided LexerFile object to
        parse tokens and build list of TopDeclarations
      */
    public mutable ParsingPipeline : LexerBase -> list [TopDeclaration];

    /** Pipline for adding namespaces and types hierarchy from parsed
        declarations.

        The list of declarations denotes the scope of one file, including
        using statements, etc. So the default pipeline passes enriched
        environment through declarations on the list.
     */
    public mutable ScanningPipeline : TopDeclaration -> void;

    public delegate InitDelegate () : void;
    public event OnInit : InitDelegate;

    /** Each function put here is executed once, after the compiler has finished. */
    public CleanupOnce : Queue [void -> void] = Queue ();
    
    /// initialize pipelines with default values
    this () {
      ParsingPipeline = MainParser.Parse;
      ScanningPipeline = ScanTypeHierarchy.ProcessDeclaration;
    }

    public GeneratedAssembly : System.Reflection.Assembly
    {
      get { Hierarchy.GeneratedAssembly }
    }
    

    internal MarkTypeBuilderCompiled () : void
    {
      current_tyinfo_count = current_tyinfo_count + 1;
      ProgressBar (10 + current_tyinfo_count * 90 / tyinfo_counter);
    }

    ProgressBar (stage : int) : void
    {
      when (Options.ProgressBar) {
        def max = 60;
        def act = stage * max / 100;
        def act = if (act > max) max else act;
        def diff = act - current_pb_stage;
        when ((pb_killed && diff > 10) || (!pb_killed && diff > 0)) {
          System.Console.Write ("\r" + System.String ('_', act) + 
                                       System.String ('.', max - act) + "\r");
          current_pb_stage = act;
          pb_killed = false;
        }
      }
    }

    public KillProgressBar () : void
    {
      when (Options.ProgressBar && !pb_killed) {
        System.Console.Write ("\n");
        pb_killed = true;
      }
    }

    internal InitCompiler () : void {
      Stats.Reset ();
      Message.Init ();
      MacroColorizer.Clear ();
      Location.Init ();
      MacroRegistry.Init ();
      LibraryReferenceManager.Init ();     
      NamespaceTree.Init ();
      Util.Init ();
      AttributeCompiler.Init ();
      PreParser.Init ();
      Options.Validate ();
      Passes.Solver = Solver ();
      Hierarchy = null;
      // if it was not initialized outside the Run
      when (LexingPipeline == null)
        if (Options.IndentationSyntax)
          LexingPipeline = fun (fn) {
            def l = LexerFile (fn);
            l.SwitchToIndentationMode ();
            l
          }
        else
          LexingPipeline = LexerFile;
      when (OnInit != null) OnInit ();
      while (! CleanupOnce.IsEmpty)
        (CleanupOnce.Take ()) ();
    }

    internal CleanUp () : void {
      /// some cleanup
      WarningOptions.Clean ();
      while (! CleanupOnce.IsEmpty)
        (CleanupOnce.Take ()) ();
      LexingPipeline = null;
    }
    
    internal LoadExternalLibraries () : void {
      unless (Options.DoNotLoadStdlib) {
        LibraryReferenceManager.AddLibrary ("mscorlib");
        LibraryReferenceManager.AddLibrary ("System");
        LibraryReferenceManager.AddLibrary ("Nemerle");
        unless (Options.GreedyReferences) LibraryReferenceManager.AddLibrary ("System.Xml");
      };
     
      List.Iter (List.Rev (Options.ReferencedLibraries), 
                 LibraryReferenceManager.AddLibrary);

      foreach (file in Options.MacrosToLoad.Reverse())
        LibraryReferenceManager.LoadMacrosFrom (file);

      when (! Options.PersistentLibraries || InternalType.Void == null) {
        SystemType.Init ();                 
        InternalType.InitSystemTypes ();
        unless (Options.DoNotLoadStdlib) InternalType.InitNemerleTypes ();

        unless (Options.DoNotLoadMacros)
          LibraryReferenceManager.LoadMacrosFrom ("Nemerle.Macros");
      }

      // this is because we store macro syntax inside GlobalEnv
      GlobalEnv.Init ();
    }

    
    /** 
     * Run passes of the compiler. 
     */
    public Run () : void    
    {
      try {
      InitCompiler ();
      
      ProgressBar (1);

      LoadExternalLibraries ();

      ProgressBar (2);      

      def trees = List.RevMap (Options.Sources, fun (x) { ParsingPipeline (LexingPipeline (x)) });

      Message.MaybeBailout();    // we have detected multiple files already
      
      ProgressBar (5);

      Hierarchy = TypesManager ();

      // create N.C.TypeBuilders for all parsed types and add them to namespace hierarchy
      foreach (group in trees)
        List.Iter (group, ScanningPipeline);
      
      when (Options.DoNotLoadStdlib) InternalType.InitNemerleTypes ();      
      
      ProgressBar (8);

      Hierarchy.Run ();
      Message.MaybeBailout();

      Hierarchy.CreateAssembly ();  

      ProgressBar (10);

      Hierarchy.EmitAuxDecls ();
      Message.MaybeBailout();

      Hierarchy.EmitDecls ();
      Message.MaybeBailout();
       
      NamespaceTree.FinishMacroContextClass ();

      NamespaceTree.CheckReferencedNamespaces ();
      Hierarchy.CheckForUnusedGlobalSymbols ();
      Hierarchy.CheckFinalization ();

      when (Options.XmlDoc) {
        XmlDoc.Init ();
        Hierarchy.SourceTopIter (XmlDoc.DumpType);
        XmlDoc.Save ();
      }

      unless (Options.CompileToMemory)
        Hierarchy.SaveAssembly ();
        
      Message.MaybeBailout();
        
      KillProgressBar ();
      Stats.Run ();
      }
      finally {
        CleanUp ();
      }
    }  
  }
}
