/*
 * Copyright (c) 2003-2005 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * Pretty printing of the parse and typed trees
 */

using System.Text;
 
using Nemerle.Collections;
using Nemerle.Utility;
using Nemerle.IO;

using Nemerle.Compiler.Parsetree;
using TT = Nemerle.Compiler.Typedtree;

//#define PRINT_COLOR

namespace Nemerle.Compiler
{
  public module PrettyPrint
  {
    /* ------------------------------------------------------------------------ */
    /* -- PARSE TREE ---------------------------------------------------------- */
    /* ------------------------------------------------------------------------ */
    
    SprintExpr (ctx : option[Typer], expr : PExpr, indentation : string,
                acc : StringBuilder) : void 
    {
      def add (x : string) { ignore (acc.Append (x)) };
    
      def expr = 
        match (ctx) {
          | Some (c) => MacroRegistry.expand_macro (c, expr)
          | _ => expr
        };

      def print_tparms (pars) {
        add ("[");
        add (NString.Concat (", ", List.Map (pars, Pair.First)));
        add ("]")
      };

      def print_tconstraints (cts : list [Constraint]) {
        | [] => ()
        | x :: xs =>
          add (sprintf ("where %s : ", Pair.First (x.tyvar)));
          SprintExpr (ctx, x.ty, indentation, acc);
          print_tconstraints (xs)
      };

      def print_funparm (p : Fun_parm) {
        | <[ parameter: $(n : name) : $ty ]> =>
          add (n.Id + " : ");
          SprintExpr (ctx, ty, indentation, acc)

        | <[ parameter: params $(n : name) : $ty ]> =>
          add (sprintf ("params %s : ", n.Id));
          SprintExpr (ctx, ty, indentation, acc)

        | <[ parameter: $(n : name) : $ty = $expr ]> =>
          add (sprintf ("params %s : ", n.Id));
          SprintExpr (ctx, ty, indentation, acc);
          add (" = ");
          SprintExpr (ctx, expr, indentation, acc)

        | _ => add ("spliced funparm name")
      };
     
      def print_funparms (fps) {
        NString.SeparatedCalls (", ", fps, print_funparm, acc)
      };

      def sprint_ss (x) {
        if (x == null) add ("(NULL-MEM)") else
        match (x) {
          | Splicable.Name (n) => add (n.Id);
          #if PRINT_COLOR
          add ("."); add (n.color.ToString ());
          #endif
          | Splicable.Expression (e) =>
            add ("$(");
            SprintExpr (ctx, e, indentation, acc);
            add (")");
          | Splicable.HalfId (n) =>
            add (n + "<COMPL>");
        }
      }

      if (expr == null) add ("(NULL)") else
      match (expr) {
        | <[ $(n : name) ]> =>
          add (n.Id);
          #if PRINT_COLOR
          add ("."); add (n.color.ToString ());
          #endif

        | <[ $obj . $mem ]> => 
          SprintExpr (ctx, obj, indentation, acc); add ("."); sprint_ss (mem)

        | <[ $x :: $xs ]> =>
          SprintExpr (ctx, x, indentation, acc);  
          add (" :: "); SprintExpr (ctx, xs, indentation, acc)

        | <[ $func (.. $parms) ]> =>
          def print_parm (p : PExpr) {
            match (p) {
              | <[ $(n : name) = $expr ]> =>
                add (n.Id + " = ");
                SprintExpr (ctx, expr, indentation + "  ", acc)
              | _ =>
                SprintExpr (ctx, p, indentation + "  ", acc)
            }
          };
          match ((func, parms)) {
            | (<[ $(x : name) ]>, [e]) when LexerBase.IsOperator (x.Id) =>
              add (x.Id);
              print_parm (e)

            | (<[ $(x : name) ]>, [e1, e2]) when LexerBase.IsOperator (x.Id) =>
              print_parm (e1);
              add (" " + x.Id + " ");
              print_parm (e2);            

            | _ =>
              SprintExpr (ctx, func, indentation, acc); add (" (");
              NString.SeparatedCalls (", ", parms, print_parm, acc);             
              add (")");
          }

        | <[ $target = $source ]> =>
          SprintExpr (ctx, target, indentation, acc); add (" = ");
          SprintExpr (ctx, source, indentation, acc);

        | <[ def $n = $val ]> =>
          add ("def "); SprintExpr (ctx, n, indentation, acc);
          add (" = "); SprintExpr (ctx, val, indentation, acc)

        | <[ mutable $n = $val ]> =>
          add ("mutable "); SprintExpr (ctx, n, indentation, acc);
          add (" = "); SprintExpr (ctx, val, indentation, acc)

        | <[ $expr :> $ty ]> =>
          add ("("); SprintExpr (ctx, expr, indentation, acc); add (" :> ");
          SprintExpr (ctx, ty, indentation, acc); add (")");

        | <[ $expr is $ty ]> =>
          add ("("); SprintExpr (ctx, expr, indentation, acc);
          add (" is "); SprintExpr (ctx, ty, indentation, acc); add (")")

        | <[ $e1 where $e2 ]> =>
          SprintExpr (ctx, e1, indentation, acc);
          add (" where "); SprintExpr (ctx, e2, indentation, acc)
        
        | <[ ( $expr : $ty ) ]> =>
          add ("("); SprintExpr (ctx, expr, indentation, acc);
          add (" : "); SprintExpr (ctx, ty, indentation, acc); add (")")
        
        | PExpr.TypedPattern => ()
        | PExpr.TypedType => ()
        | PExpr.As  => ()

        | <[ [..$elems] ]> =>
          add ("[");
          NString.SeparatedCalls (", ", elems, fun (x) { 
            SprintExpr (ctx, x, indentation + " ", acc); 
          }, acc);
          add ("]");
          
        | <[ match ($expr) {.. $cases } ]> =>
          def print_case (c : MatchCase) {
            def <[ case: | ..$guards => $expr ]> = c;
            foreach (g in guards) {
              add (sprintf ("\n%s| ", indentation + "  "));
              SprintExpr (ctx, g, indentation, acc);
            }
            add (sprintf (" => \n%s", indentation + "    "));
            SprintExpr (ctx, expr, indentation + "    ", acc)
          };

          add ("match ("); 
          SprintExpr (ctx, expr, indentation, acc); add (") {");
          List.Iter (cases, print_case);
          add (sprintf ("\n%s}", indentation))

        | <[ throw $exc ]> =>
          add ("throw "); SprintExpr (ctx, exc, indentation, acc);

        | <[ try $body catch { $exn is $exn_ty => $handler } ]> => 
          add (sprintf ("try {\n%s  ", indentation));
          SprintExpr (ctx, body, indentation + "  ", acc);
          add (sprintf ("\n%s}\n%scatch {\n%s", indentation, indentation, indentation + "  "));
          sprint_ss (exn); add (" : ");
          SprintExpr (ctx, exn_ty, indentation, acc);
          add (sprintf ("=>\n%s", indentation + "    "));
          SprintExpr (ctx, handler, indentation + "    ", acc);
          add (sprintf ("\n%s}", indentation));

        | <[ try $body finally $handler ]> =>
          add (sprintf ("try {\n%s  ", indentation));
          SprintExpr (ctx, body, indentation + "  ", acc);
          add (sprintf ("\n%s}\n%sfinally {\n%s  ", indentation, indentation, indentation));
          SprintExpr (ctx, handler, indentation + "  ", acc);
          add (sprintf ("\n%s}", indentation));

        | PExpr.Literal (lit) => add (lit.ToString ())

        | <[ this ]> => add ("this")

        | <[ base ]> => add ("base")

        | <[ typeof ($t) ]> =>
          add ("typeof ("); SprintExpr (ctx, t, indentation, acc); add (")")

        | <[ {.. $seq } ]> =>
          match (seq) {
            | [e] =>
              SprintExpr (ctx, e, indentation, acc)
            | _ =>
              add ("{");
              NString.SeparatedCalls (";", seq, fun (x) { 
                add (sprintf ("\n%s", indentation + "  ")); 
                SprintExpr (ctx, x, indentation + "  ", acc); 
              }, acc);
              add (sprintf ("\n%s}", indentation))
          }

        | <[ (.. $args) ]> =>
          add ("(");
          NString.SeparatedCalls (", ", args, fun (x) { 
            SprintExpr (ctx, x, indentation, acc) 
          }, acc);
          add (")")

        | <[ array (..$args) ]> =>
          add ("array (");
          NString.SeparatedCalls (", ", args, fun (x) { 
            SprintExpr (ctx, x, indentation, acc) 
          }, acc);

        | <[ ref $e ]> => add ("ref "); SprintExpr (ctx, e, indentation, acc)

        | <[ out $e ]> => add ("ref "); SprintExpr (ctx, e, indentation, acc)

        | <[ array $args ]> =>
          add ("array ");
          SprintExpr (ctx, args, indentation, acc);

        | <[ array .[ $rank ] $args ]> =>
          add ("array .[");
          SprintExpr (ctx, rank, indentation, acc);
          add ("] ");
          SprintExpr (ctx, args, indentation, acc);

        | <[ $obj .[..$args] ]> =>
          SprintExpr (ctx, obj, indentation, acc);
          add (".[");
          NString.SeparatedCalls (", ", args, fun (x) { 
            SprintExpr (ctx, x, indentation, acc) 
          }, acc);
          add ("]")
          
        | <[ $obj [.. $args] ]> =>
          SprintExpr (ctx, obj, indentation, acc);
          add ("[");
          NString.SeparatedCalls (", ", args, fun (x) { 
            SprintExpr (ctx, x, indentation, acc) 
          }, acc);
          add ("]")

        | PExpr.Lambda (fd) =>
          add ("fun "); print_tparms (fd.header.typarms.tyvars);
          add (" ("); print_funparms (fd.header.parms); add (") ");
          add (": "); SprintExpr (ctx, fd.header.ret_type, indentation, acc); add (" ");
          print_tconstraints (fd.header.typarms.constraints); add (" ");
          SprintExpr (ctx, fd.body, indentation, acc)

        | <[ def ..$funs ]> =>
          def print_fun (f : Function_decl) {
            | <[ fundecl: $(n : name) [ ..$tparms] (..$args)
                 where .. $tconstrs $body ]> =>
              add (n.Id); add (" "); print_tparms (tparms);
              add (" ("); print_funparms (args); add (") ");
              print_tconstraints (tconstrs); add (" ");
              SprintExpr (ctx, body, indentation, acc)
            | _ => add ("spliced fun name")
          };
        
          add ("def ");
          NString.SeparatedCalls ("\nand ", funs, print_fun, acc)

        | PExpr.Wildcard => add ("_ ");

        | PExpr.Void => add ("void ");

        | PExpr.MacroCall (_, namespc, parms) =>
          match (namespc.Value) {
            | NamespaceTree.TypeInfoCache.MacroCall (m) =>
              def (rules, _) = m.SyntaxExtension ();
              mutable parms_left = parms;
              add (rules.ToString () + " ");
              def rules = rules.Next;

              def print_gel (x) {
                match (x) {
                  | GrammarElement.Keyword (k) => add (" "); add (k); add (" ")
                  | GrammarElement.Operator (")") => add (") ")
                  | GrammarElement.Operator ("}") => add ("} ")                
                  | GrammarElement.Operator (o) => add (o)
                  | GrammarElement.Optional (g) => print_gel (g)
                  | GrammarElement.RawToken => 
                    match (parms_left) {
                      | SyntaxElement.RawToken (t) :: xs =>
                        add (t.ToString ());
                        parms_left = xs;
                      | _ =>
                        Message.Error ("expected raw token as parameter of macro "
                                       + Util.QidOfList (namespc.Name))
                    };

                  | GrammarElement.Expression =>
                    match (parms_left) {
                      | SyntaxElement.Expression (e) :: xs =>
                        SprintExpr (ctx, e, indentation, acc);
                        parms_left = xs;
                      | _ =>
                        Message.Error ("expected expression as parameter of macro "
                                       + Util.QidOfList (namespc.Name) + " got " +
                                       parms_left.ToString ())
                    };
                  | GrammarElement.Parameter =>
                    match (parms_left) {
                      | SyntaxElement.Parameter (p) :: xs =>
                        print_funparm (p);
                        parms_left = xs;
                      | _ =>
                        Message.Error ("expected function parameter as parameter of macro "
                                       + Util.QidOfList (namespc.Name))
                    };

                  | GrammarElement.ExpressionList =>
                    NString.SeparatedCalls (", ", parms_left, fun (_) {
                      | SyntaxElement.Expression (e) => SprintExpr (ctx, e, indentation, acc)
                      | _ =>
                        Message.Error ("expected expression in macro parameters: "
                                       + Util.QidOfList (namespc.Name))
                    }, acc);

                  | GrammarElement.Branch | GrammarElement.End => Util.ice ("invalid node");
                }
                when (x.Next != null)
                  print_gel (x.Next)
              };
              print_gel (rules);
            | _ => add ("macro_call")
          }

        | PExpr.ToComplete (n) =>
          add (n + "<COMPL>");

        | PExpr.Spliced (e) =>
          add ("$("); SprintExpr (ctx, e, indentation, acc); add (")");

        | PExpr.Ellipsis (e) =>
          add (".."); SprintExpr (ctx, e, indentation, acc); 

        | PExpr.Quoted (quot) =>
          add ("<[ ");
          add (quot.ToString ());
          add (" ]>");

        | PExpr.Typed (e) => SprintTyExpr (ctx, e, None (), false, indentation, acc)

        | PExpr.Error => ()
      }
    }

    [Nemerle.Assertions.Ensures (value != null)]
    public SprintExpr (ctx : option[Typer], expr : PExpr) : string
    {
      def result = StringBuilder ();
      SprintExpr (ctx, expr, "", result);
      result.ToString ()    
    }

    public PrintExpr (ctx : option[Typer], expr : PExpr) : void
    {
      printf ("%s\n", SprintExpr (ctx, expr))
    }


    /* ------------------------------------------------------------------------ */
    /* -- TYPED TREE ---------------------------------------------------------- */
    /* ------------------------------------------------------------------------ */


    /* -- MATCHING ---------------------------------------------------------- */
    
    /**
     * Pretty prints a match ('expr') { 'match_cases' } instruction.
     * This is a string generating wrapper for SprintTyMatch/6.
     */
    public SprintTyMatch (ctx : option [Typer],
                          expr : TT.TExpr,
                          match_cases : list [TT.Match_case]) : string
    {
      def result = StringBuilder ();
      ignore (result.Append ("\n"));
      SprintTyMatch (ctx, expr, None (), match_cases, "", result);
      ignore (result.Append ("\n"));
      result.ToString ()          
    }


    public SprintPattern (pattern : TT.Pattern) : string
    {
      def result = StringBuilder ();

      def append (x : string) { ignore (result.Append (x)) }
        
      def print_pattern (pattern : TT.Pattern)
      {
        | TT.Pattern.Error => append ("(ERROR)")
        | TT.Pattern.Wildcard => append ("_")
        | TT.Pattern.As (TT.Pattern.HasType (tycon), decl) =>
          append (decl.Name + " : " + tycon.FullName)          
        | TT.Pattern.As (pat /* Pattern */, decl /* LocalValue */) =>
          append ("(");
          print_pattern (pat);
          append (") as " + decl.Name)
        | TT.Pattern.HasType (tc) =>
          append ($ "_ is $tc");
        | TT.Pattern.Tuple (args /* list [Pattern] */) =>
          append ("(");
          def loop (args) {
            | [] => ()
            | [arg] =>
              print_pattern (arg)
            | arg :: args =>
              print_pattern (arg);
              append (", ");
              loop (args)
          };
          loop (args);
          append (")")         

        // records
        | TT.Pattern.Record (args /* list <IField * TT.Pattern> */) =>
          def print_record_field (fld, pat)
          {
            append (fld.Name + " = ");
            print_pattern (pat)
          }          

          def loop (args) {
            | [] => ()
            | [(fld, pat)] =>
              print_record_field (fld, pat)
            | (fld, pat) :: rest =>
              print_record_field (fld, pat);
              append ("; ");
              loop (rest)
          }
          
          append ("{");
          loop (args);
          append ("}")

        // variant constructors
        | TT.Pattern.Application (name /* TypeInfo */, TT.Pattern.Wildcard /* Pattern */) =>
          append (name.FullName)
        | TT.Pattern.Application (name /* TypeInfo */, (TT.Pattern.Record) as arg /* Pattern */) =>
          append (name.FullName + " ");
          print_pattern (arg);   
        | TT.Pattern.Application (name /* TypeInfo */, arg /* Pattern */) =>
          append (name.FullName + " (");          
          print_pattern (arg);
          append (")")

        // literals
        | TT.Pattern.Literal (lit /* Literal */) =>
          append (lit.ToString ())
          
        | TT.Pattern.Enum (fld, _) =>
          append (fld.Name)
      }
      
      print_pattern (pattern);
      result.ToString ()
    }
    

    /**
     * Pretty prints a match ('expr') { 'match_cases' } instruction.
     */
    public SprintTyMatch (ctx : option [Typer],
                          expr : TT.TExpr,
                          current_fun_name : option [string],
                          match_cases : list [TT.Match_case],
                          indentation : string,
                          result : StringBuilder) : void
    {
      def append (x : string) { ignore (result.Append (x)) }
      def indent () { append (indentation) }

      def recurse (expr : TT.TExpr) {
        SprintTyExpr (ctx, expr, current_fun_name, false, indentation, result)
      }
      def recurse_and_indent (expr : TT.TExpr) {
        SprintTyExpr (ctx, expr, current_fun_name, true, indentation + "    ", result)
      }
      def recurse_and_short_indent (expr : TT.TExpr) {
        SprintTyExpr (ctx, expr, current_fun_name, true, indentation + "  ", result)
      }
        
      def print_pattern (pattern : TT.Pattern)
      {
        append (SprintPattern (pattern))
      }
      
      def print_patterns (patterns)
      {
        def do_print (pattern, guard, assigns : list [_], terminator)
        {
          indent ();
          append ("  | ");
          print_pattern (pattern);

          match (guard) {
            | TT.TExpr.Literal (Literal.Bool (true)) => ()
            | _ =>
              append (" when ");
              recurse (guard);
          }

          unless (assigns.IsEmpty) {
            append (" with (");
            foreach ((name, value) in assigns) {
              append (name.Name);
              append (" = ");
              recurse (value);
              append (", ");
            }
          }

          append (terminator)
        }
        
        match (patterns) {
          | [(pattern, guard, assigns)] =>
            do_print (pattern, guard, assigns, " =>\n")
          | (pattern, guard, assigns) :: rest =>
            do_print (pattern, guard, assigns, "\n");
            print_patterns (rest)
          | _ => ()
        }
      }
      
      def print_match_cases (cases : list [TT.Match_case])
      {
        | TT.Match_case where (patterns, body, _) :: rest =>
          print_patterns (patterns);
          recurse_and_indent (body);
          append ("\n");
          print_match_cases (rest)          
        | _ => ()      
      }

      def collapse_match_like_constructions () : bool
      {
        def expr_is_bool (expr : TT.TExpr)
        {
          expr.Type.IsFixed &&
          match (expr.Type.FixedValue) {
            | MType.Class (tycon, []) =>
              tycon.FullName == "System.Boolean" ||
              tycon.FullName == "Nemerle.Core.bool"
            | _ =>
              false
          }
        }

        def case_is_true (case : TT.Match_case) {
          (case.patterns is [(Literal (Bool (true)),
                              Literal (Bool (true)), [])])
        }
        def case_is_false (case : TT.Match_case) {
          (case.patterns is [(Literal (Bool (false)),
                              Literal (Bool (true)), [])])
        }
        def case_is_wildcard (case : TT.Match_case) {
          (case.patterns is [(Wildcard, Literal (Bool (true)), [])])
        }
        def case_is_ty_check (case : TT.Match_case) {
          (case.patterns is [(As (HasType, _), Literal (Bool (true)), [])])
        }
        def expr_is_true (expr : TT.TExpr) {
          (expr is TT.TExpr.Literal (Literal.Bool (true)))
        }
        def expr_is_false (expr : TT.TExpr) {
          (expr is TT.TExpr.Literal (Literal.Bool (false)))
        }        
        def expr_is_unit (expr : TT.TExpr) {
          (expr is TT.TExpr.Literal (Literal.Void))
        }

        def matching_over_bool =
          expr_is_bool (expr);
        
        match (match_cases) {
          | [then_case, else_case] =>
            def then_is_true = case_is_true (then_case);
            def then_is_false = case_is_false (then_case);
            def then_is_ty_check = case_is_ty_check (then_case);
            def else_is_wildcard = case_is_wildcard (else_case);

            def then_body_is_true = expr_is_true (then_case.body);
            def then_body_is_false = expr_is_false (then_case.body);
            def else_body_is_unit = expr_is_unit (else_case.body);
            def else_body_is_true = expr_is_true (else_case.body);
            def else_body_is_false = expr_is_false (else_case.body);
            def else_body_is_bool = expr_is_bool (else_case.body);

            def then_pattern_count = then_case.patterns.Length;
            def else_pattern_count = then_case.patterns.Length;

            def is_if =
              matching_over_bool && then_is_true && else_is_wildcard;
            def is_when =
              matching_over_bool && then_is_true &&
              else_is_wildcard && else_body_is_unit;
            def is_unless =
              matching_over_bool && then_is_false &&
              else_is_wildcard && else_body_is_unit;

            def is_and_and =
              matching_over_bool && then_is_false &&
              then_body_is_false && else_is_wildcard && else_body_is_bool;

            def is_or_or =
              matching_over_bool && then_is_true &&
              then_body_is_true && else_is_wildcard && else_body_is_bool;

            def is_is =
              then_body_is_true && then_is_ty_check &&
              else_is_wildcard && else_body_is_false;

            def is_matches =
              then_body_is_true && else_is_wildcard && else_body_is_false &&
              then_pattern_count == 1 && else_pattern_count == 1;
                  
            def is_redundant =
              matching_over_bool &&
              then_is_true && then_body_is_true &&
              else_is_wildcard && else_body_is_false ||
              then_is_false && then_body_is_false &&
              else_is_wildcard && else_body_is_true;

            if (is_redundant) {
              append ("(");
              recurse (expr);
              append (")");
              true
            }
            else if (is_or_or || is_and_and) {
              append ("(");
              recurse (expr);
              append (if (is_or_or) " || " else " && ");
              recurse (else_case.body);
              append (")");
              true
            }
            else if (is_when || is_unless) {
              append ((if (is_when) "when" else "unless") + " (");
              recurse (expr);
              append (") {\n");
              recurse_and_short_indent (then_case.body);
              append ("\n" + indentation + "}");
              true
            }
            else if (is_if)
            {
              append ("if (");
              recurse (expr);
              append (") {\n");
              recurse_and_short_indent (then_case.body);
              append ("\n" + indentation + "}\n" + indentation + "else {\n");
              recurse_and_short_indent (else_case.body);
              append ("\n" + indentation + "}");
              true
            }
            else if (is_is)
            {
              append ("(");
              recurse (expr);
              append (" is ");
              match (then_case.patterns) {
                | [(As (HasType (ty), _), _, [])] =>
                  append (strip_nemerle_core_dot_prefix (ty.FullName));
                | _ =>
                  Util.ice ("SprintTyMatch: is_is")
              }
              append (")");
              true
            }
            else if (is_matches)
            {
              append ("(");
              recurse (expr);
              append (" matches ");
              match (then_case.patterns) {
                | [(then_pat, _, [])] =>
                  print_pattern (then_pat)
                | _ =>
                  Util.ice ("SprintTyMatch: is_matches")
              }
              append (")");
              true
            }
            else
              false

          | _ =>
            false
        }
      }

      unless (collapse_match_like_constructions ())
      {
        append ("match (");
        recurse (expr);
        append (") {\n");
      
        print_match_cases (match_cases);

        indent ();
        append ("}")
      }
    }


    /* -- EXPRESSIONS ------------------------------------------------------- */

    /**
     * Pretty prints a typed tree expression.
     */
    public SprintTyExpr (ctx : option [Typer], expr : TT.TExpr) : string
    {
      def result = StringBuilder ();
      ignore (result.Append ("\n"));
      SprintTyExpr (ctx, expr, None (), false, "", result);
      ignore (result.Append ("\n"));
      result.ToString ()    
    }

    /**
     * Pretty prints a typed tree expression.
     */
    public SprintTyExpr (expr : TT.TExpr) : string
    {
      def result = StringBuilder ();
      SprintTyExpr (None (), expr, None (), false, "", result);
      result.ToString ()    
    }
    
    
    /**
     * Pretty prints a typed tree expression.
     */
    public SprintTyExpr (ctx : option [Typer],
                         expr : TT.TExpr,
                         current_fun_name : option [string],
                         is_top_level : bool,
                         indentation : string,
                         result : StringBuilder) : void
    {
      def append (x : string) { ignore (result.Append (x)) }
      def indent () { append (indentation) }
      //append($"{$(expr.GetHashCode())}");

      def recurse (expr : TT.TExpr) {
        SprintTyExpr (ctx, expr, current_fun_name, false, indentation, result)
      }
      def recurse_no_indent (expr : TT.TExpr) {
        SprintTyExpr (ctx, expr, current_fun_name, true, indentation, result)
      }
      def recurse_and_indent (expr : TT.TExpr) {
        SprintTyExpr (ctx, expr, current_fun_name, true, indentation + "  ", result)
      }

      def print_type (ty : TyVar) {
        append (ty.ToString ())
      }
      
      def print_expr_list (begin : string, args : list [TT.TExpr],
                           separator : string, end : string)
      {
        append (begin);
        NString.SeparatedCalls (separator, args, recurse, result);
        append (end)
      }      

      def print_fun_call_parms (parms : list [TT.Parm])
      {
        NString.SeparatedCalls (", ", parms, fun (parm : TT.Parm) { recurse (parm.expr) }, result)
      }
 
      def is_list_cons (mem : IMember)
      {
        mem.DeclaringType.FullName == "Nemerle.Core.list.Cons" && mem.Name == ".ctor"
      }

      def is_list_nil (mem : IMember)
      {
        mem.DeclaringType.FullName == "Nemerle.Core.list.Nil" && mem.Name == "_N_constant_object"
      }
      
      // checks if an expression needs to be put in curly braces
      def need_curly_braces (expr : TT.TExpr)
      {
        | TT.TExpr.DefValIn => true
        | TT.TExpr.DefFunctionsIn => true
        | TT.TExpr.Sequence => true
        | TT.TExpr.FieldMember (obj, _)
        | TT.TExpr.PropertyMember (obj, _)
        | TT.TExpr.MethodRef (obj, _, _, _)
        | TT.TExpr.ArrayIndexer (obj, _) =>
          need_curly_braces (obj)
        | TT.TExpr.Assign (target, source) =>
          need_curly_braces (target) || need_curly_braces (source)
        | TT.TExpr.Throw (exn) =>
          need_curly_braces (exn)
        | _ => false
      }

      
      // prints a list, using the [,] and :: shortcuts for Cons where appropriate
      def print_list_constructors (cons : TT.TExpr)
      {
        def walk_tree (cons : TT.TExpr, acc : list [TT.TExpr]) : list [TT.TExpr] * bool
        {
          match (cons) {
            | TT.TExpr.Call (TT.TExpr.StaticRef (_, mem, _), parms, _) when is_list_cons (mem) =>
              match (parms) {
                | [head, tail] =>
                  walk_tree (tail.expr, head.expr :: acc)
                | _ =>
                  Util.ice ("SprintTyExpr/flatten_list/walk_list_tree")
                }
              
            | TT.TExpr.StaticRef (_, mem, _) when is_list_nil (mem) =>
              (List.Rev (acc), true)

            | _ =>
              (List.Rev (cons :: acc), false)
          }
        }

        def (flattened_list, ended_with_nil) =
          walk_tree (cons, []);

        if (ended_with_nil)
          print_expr_list ("[", flattened_list, ", ", "]")
        else
          print_expr_list ("", flattened_list, " :: ", "")
      }


      // prints a local function declaration
      def print_local_fun_decl (title : string, fun_decl : TT.Fun_header)
      {      
        append (title + " " +
                print_fun_typarms (fun_decl.typarms) +
                fun_decl.name + " " +
                print_fun_parms (fun_decl.parms) + " : " +
                fun_decl.ret_type.ToString () + " {\n");

        // change the current function's name when recursing
        match (fun_decl.body) {
          | FunBody.Typed (body) =>
            SprintTyExpr (ctx, body, Some (fun_decl.name),
                          true, indentation + "  ", result);
          | _ => {}
        }

        append ("\n" + indentation + "}\n")
      }   


      // indent the top level expressions
      when (is_top_level) indent ();

      if (expr == null) append ("[[[NULL]]]") else
      match (expr) {
        // reference building
        | TT.TExpr.LocalFunRef (decl /* LocalValue */, _)
        | TT.TExpr.LocalRef (decl /* LocalValue */) =>
          append (decl.Name)
          
        | TT.TExpr.StaticRef (from, mem /* IMember */, tp) =>
          def ty_name =
            strip_nemerle_core_dot_prefix (from.ToString ());
          def co_name = mem.Name;
          
          if (is_list_nil (mem)) {
            append ("[]")
          }
          else {
            append (ty_name);

            unless (co_name == ".ctor") {
              append ("." + co_name)
            }
          }

          unless (tp.IsEmpty)
            append ($".$tp");
          
        | TT.TExpr.FieldMember (obj /* PExpr */, fld /* IField */) =>
          recurse (obj);
          append ("." + fld.Name)
          
        | TT.TExpr.ConstantObjectRef (_, mem /* IField */) =>
          append (mem.DeclaringType.FullName + "." + mem.Name);
          
        | TT.TExpr.ImplicitValueTypeCtor =>
          append (expr.Type.ToString () + " ()");

        | TT.TExpr.DefaultValue =>
          append ("DEFAULT")

        | TT.TExpr.NotNull (e) =>
          append ($"when ($e == null) throw NullMatchException ();")
          
        | TT.TExpr.Switch (indexing, defl, cases) =>
          append ($ "switch ($indexing) $defl $cases")

        | TT.TExpr.If (cond, e1, e2) =>
          append ("if.real (");
          recurse (cond);
          append (") {\n");
          recurse_and_indent (e1);
          append ("\n" + indentation + "}\n" + indentation + "else {\n");
          recurse_and_indent (e2);
          append ("\n" + indentation + "}");

        | TT.TExpr.HasType (e, t) =>
          append ($ "($e is $t)")
          
        | TT.TExpr.PropertyMember (obj /* PExpr */, prop /* IProperty */) =>
          recurse (obj);
          append ("." + prop.Name)
          
        | TT.TExpr.StaticPropertyRef (_, prop /* IProperty */) =>
          append (prop.DeclaringType.FullName + "." + prop.Name)

        | TT.TExpr.EventMember (obj /* PExpr */, ev /* IEvent */) =>
          recurse (obj);
          append ("." + ev.Name)
          
        | TT.TExpr.StaticEventRef (_, ev /* IEvent */) =>
          append (ev.DeclaringType.FullName + "." + ev.Name)

        // FIXME: handle the 'notvirtual' flag
        | TT.TExpr.MethodRef (obj /* PExpr */, meth /* IMethod */, vars, _ /* notvirtual : bool */) =>
          recurse (obj);
          append ("." + meth.Name);

          unless (vars.IsEmpty)
            append ($".$vars");

        // the special case for unary operators
        | TT.TExpr.Call (TT.TExpr.OpCode (name), [parm], _) =>
          append (name);
          append ("(");
          recurse (parm.expr);
          append (")");

        // the special case for infix binary operators
        | TT.TExpr.Call (TT.TExpr.OpCode (name), [parm1, parm2], _) =>
          append ("(");
          recurse (parm1.expr);
          append (" " + name + " ");
          recurse (parm2.expr);
          append (")")

        // pretty print the list constructors
        | TT.TExpr.Call (TT.TExpr.StaticRef (_, mem, _), _, _) when is_list_cons (mem) =>
          print_list_constructors (expr)
              
        // write all the other calls in prefix form 
        | TT.TExpr.Call (func /* PExpr */, parms /* list [Parm] */, _) =>
          recurse (func);
          append (" (");
          print_fun_call_parms (parms);
          append (")")
                  
        | TT.TExpr.SelfTailCall (parms /* list [Parm] */) =>
          match (current_fun_name) {
            | Some (fun_name) => append (fun_name)
            | _ =>
              match (ctx) {
                | Some (ctx) =>
                  append (ctx.CurrentFunction.name)
                | _ =>
                  append ("SELF");
              }
          }          
          append (" (");
          print_fun_call_parms (parms);
          append (")")


        // assignment
        | TT.TExpr.Assign (target /* PExpr */, source /* PExpr */) =>
          recurse (target);
          append (" = ");
          recurse (source);


        | TT.TExpr.MultipleAssign (assigns) =>
          append ($ "ASSIGNS $assigns");


        | TT.TExpr.Label (k, body) =>
          append ($ "l$k:\n");
          recurse_and_indent (body);


        | TT.TExpr.Block (jump, body) =>
          append ($ "block ($(jump.Name)) : ");
          recurse_and_indent (body);


        | TT.TExpr.Goto (id, t) =>
          append ($ "goto l$id [$t];");
          
          
        // local definitions
        | TT.TExpr.DefValIn (name /* LocalValue */, val /* PExpr */, body /* PExpr */) =>
          if (need_curly_braces (val))
          {
            append ("def " + name.Name + " = {\n");
            recurse_and_indent (val);
            append ("\n" + indentation + "};\n");
            recurse_no_indent (body)
          }
          else
          {
            append ("def " + name.Name + " = ");
            recurse (val);
            append (";\n");
            recurse_no_indent (body)
          }

        // local function definitions can be grouped using the 'and' keyword
        | TT.TExpr.DefFunctionsIn (funs /* list [Function_decl] */, body /* PExpr */) =>
          match (funs) {
            | fun_decl :: rest =>
              print_local_fun_decl ("def", fun_decl);
              List.Iter (rest, fun (fun_decl) { print_local_fun_decl ("and", fun_decl) })
            | _ => Util.ice ("TT.TExpr.DefFunctionsIn with no function declarations")
          }
          recurse_no_indent (body)

          
        // the 'match' instruction is handled separately
        | TT.TExpr.Match (expr /* PExpr */, cases /* list [Match_case] */) =>
          SprintTyMatch (ctx, expr, current_fun_name, cases, indentation, result)


        // exception handling
        | TT.TExpr.Throw (exn /* PExpr */) =>
          append ("throw ");
          recurse (exn)
          
        | TT.TExpr.TryFault (body, handler) =>
          append ("try {\n");
          recurse_and_indent (body);
          append ("\n" + indentation + "} fault {\n");
          recurse_and_indent (handler);
          append ("\n" + indentation + "}")
        
        | TT.TExpr.TryWith (body /* PExpr */, exn /* LocalValue */, handler /* PExpr */) =>
          append ("try {\n");
          recurse_and_indent (body);
          append ("\n" + indentation + "} catch {\n  " + indentation + exn.Name + " : ");
          print_type (exn.Type);
          append (" =>\n");
          recurse_and_indent (handler);
          append ("\n" + indentation + "}")
          
        | TT.TExpr.TryFinally (body /* PExpr */, handler /* PExpr */) =>
          append ("try {\n");
          recurse_and_indent (body);
          append ("\n" + indentation +"} finally {\n");
          recurse_and_indent (handler);
          append ("\n" + indentation + "}")


        // 'this' and 'base' objects
        | TT.TExpr.This =>
          append ("this");
          
        | TT.TExpr.Base /* (base_ctor : IMethod) */ =>
          append ("base")


        // type related nodes
        // FIXME: checked/unchecked
        | TT.TExpr.TypeConversion (expr /* TExpr */, target_type /* TType */, kind) =>
          append ("(");
          recurse (expr);
          match (kind) {
            | TT.ConversionKind.Nop
            | TT.ConversionKind.UpCast =>
              append (" : ");
            | _ =>
              append (" :> ");
          }
          print_type (target_type);
          append (")")

        | TT.TExpr.TypeOf (target_type /* TType */) =>
          append ("typeof (");
          print_type (target_type);
          append (")")

          
        // array related nodes
        | TT.TExpr.Array (args : list [TT.TExpr], _ /* dimensions : list [int] */) =>
          append ("array ");
          print_expr_list (" [", args, ", ", "]")
                    
        | TT.TExpr.ArrayIndexer (obj /* PExpr */, args /* list [TExpr] */) =>
          recurse (obj);
          print_expr_list (" [", args, ", ", "]")
          
        | TT.TExpr.TupleIndexer (obj, k, _) =>
          recurse (obj);
          append ($ " [$k]");
          

        // loading of a literal
        | TT.TExpr.Literal (val /* Literal */) =>
          append (val.ToString ())


        // other nodes
        | TT.TExpr.Sequence (e1 /* PExpr */, e2 /* PExpr */) =>
          recurse (e1);
          append (";\n");
          recurse_no_indent (e2);
          
        | TT.TExpr.Tuple (args /* list [PExpr] */) =>
          print_expr_list ("(", args, ", ", ")")                    

        | TT.TExpr.OpCode (name /* string */) =>
          append (name)

        | TT.TExpr.Error =>
          append ("(ERROR)")

        | TT.TExpr.MethodAddress (_, meth, is_virt, _) =>
          append ($ "ADDR($meth, $is_virt)")

        | TT.TExpr.Delayed (dt) =>
          if (dt.IsResolved)
            SprintTyExpr (ctx, dt.ResolutionResult, current_fun_name, is_top_level, 
                          indentation, result)
          else
            append ("(Delayed)")
      }

      //append ($"{$(expr.Type)}");
    }
    

    public TyVarToParseTree (ty : TyVar) : PExpr
    {
      if (ty.IsFixed) 
        MTypeToParseTree (ty.FixedValue)
      else
        MTypeToParseTree (ty.Fix ())
    }
    

    public MTypeToParseTree (ty : MType) : PExpr
    {
      match (ty) {
        | MType.Class (tycon, []) =>
          Util.ExprOfQid (tycon.FullName)

        | MType.Class (tycon, args) =>
          PExpr.Indexer (Util.ExprOfQid (tycon.FullName),
                         List.Map (args, TyVarToParseTree))

        | MType.TyVarRef (t) => PExpr.Ref (Name (t.Name))

        | MType.Fun (from, to) =>
          <[ $(TyVarToParseTree (from)) -> $(TyVarToParseTree (to)) ]>

        | MType.Void => PExpr.Void ()

        | MType.Tuple (all) =>
          <[ @* (..$( List.Map (all, TyVarToParseTree))) ]>

        | MType.Array (ty, rank) =>
          <[ array [ $(rank : int), $(TyVarToParseTree (ty)) ] ]>

        | MType.Ref (ty) =>
          <[ ref $(TyVarToParseTree (ty)) ]>

        | MType.Out (ty) =>
          <[ out $(TyVarToParseTree (ty)) ]>

        | MType.Intersection => assert (false)
      }
    }
    
    
    /* -- PRIVATE METHODS --------------------------------------------------- */

    /**
     * Strips the 'Nemerle.Core.' prefix from type names
     */
    private strip_nemerle_core_dot_prefix (tyname : string) : string
    {
      if (tyname.StartsWith ("Nemerle.Core."))
        tyname.Substring (13)
      else
        tyname
    }


    /**
     * Prints a type parameters of a function declaration
     */
    private print_fun_typarms (typarms : list [StaticTyVar]) : string
    {
      typarms.ToString ()
    }


    /**
     * Prints the parameters of a function declaration
     */
    private print_fun_parms (parms : list [TT.Fun_parm]) : string
    {      
      def result = StringBuilder ();
      
      def append (x : string) { ignore (result.Append (x)) }

      append ("(");
      NString.SeparatedCalls (", ", parms,
                              fun (parm : TT.Fun_parm) {
                                append ($ "$(parm.name) : $(parm.ty)")
                              }, result);
      append (")");

      result.ToString ()
    }
  }
}

/*** END OF FILE ***/
