/*
 * Copyright (c) 2003, 2004 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using Nemerle.Assertions;
  
using SC = System.Collections;

namespace Nemerle.Collections
{
  /**
   * This is a subtype of Generic.Dictionary, which provides some additional
   *  functional-style methods to its base type.
   */
  [System.Serializable]
  public class Hashtable ['a,'b] 
             : SC.Generic.Dictionary ['a, 'b]
    //           , IDictionary ['a,'b]
  {
    /* -- PUBLIC CONSTRUCTORS ---------------------------------------------- */

    /**
     * Creates an empty hashtable
     */
    public this ()
    {
    }

    
    /**
     * Creates an empty hashtable and sets its default capacity
     */
//    [Requires (capacity > 0)]
    public this (capacity : int)
    {
      base (capacity);
    }

    
    /**
     * Creates an empty hashtable and sets its default capacity
     * and the load factor.
     */
    public this (capacity : int, loadFactor : float)
    requires capacity > 0 && loadFactor >= 0.1f && loadFactor <= 1.0f
    {
      // S.C.G.Dictinary doesn't seem to support that
      base (capacity);
    }

    
    /* -- COPYING CONSTRUCTOR ---------------------------------------------- */

    /**
     * Creates a shallow copy of this hashtable
     */
    public this ([NotNull] ht : Hashtable ['a, 'b])
    {
      base (ht);
    }

    public this (generator : SC.Generic.IEnumerable ['a * 'b])
    {
      base ();
      foreach ((key, val) in generator)
        Add (key, val)
    }
    
    /* -- PUBLIC METHODS --------------------------------------------------- */
        
    /**
     * Returns an optional value associated with the specified key.
     */
    public Get (key : 'a) : option ['b]
    {
      if (ContainsKey (key))
        Some (this [key])
      else
        None ()
    }

    
    /**
     * This is different from add, which can fail if the key is
     * already in the underlying Framework hashtable.
     */
    public Set (key : 'a, val : 'b) : void
    {
      this [key] = val;
    }

    
    /**
     * Clones this hashtable.
     */
    public Clone () : Hashtable ['a,'b]
    {
      Hashtable (this)
    }


    
    /**
     * Returns `true' if the hashtable contains the specified key.
     *
     * NOTE: this is the same as ContainsKey.
     */
    public Contains (key : 'a) : bool
    {
      ContainsKey (key)
    }

    
    /**
     * Folds a function over the key/value pairs.
     */
    public Fold ['c] (s : 'c, f : ('a * 'b * 'c) -> 'c) : 'c 
    {
      mutable acc = s;
      
      foreach (x in this)
        acc = f (x.Key, x.Value, acc);
        
      acc
    }

    
    /**
     * Iterates a function over the key/value pairs in the hashtable.
     */
    public Iter (f : 'a * 'b -> void) : void
    {
      foreach (x in this)
        f (x.Key, x.Value)
    }

    
    /**
     * Maps a given function defined of key-value pairs to the contents
     * of this hashtable. A new hashtable object is created, containing
     * the results of the application.
     */
    public Map ['c, 'd] (f : 'a * 'b -> 'c * 'd) : Hashtable ['c,'d]
    {
      def ht = Hashtable (Count);

      foreach (x in this)
      {
        def (k, v) = f (x.Key, x.Value);
        
        ht.Add (k, v)
      }
      
      ht
    }

    /**
     * Unsafe access to the underlying hashtable
     */
    public new Item [key : 'a] : 'b
    {
      get
      {
        if (ContainsKey (key))
          base [key]
        else
          Nemerle.Extensions.DefaultValue ('b)
      }

      set
      {
        base [key] = value
      }
    }

    public new Remove (key : 'a) : void
    {
      _ = base.Remove (key)
    }


    /**
     * Returns a collection of the key/value pairs from this hashtable
     */
    public KeyValuePairs : ICollection ['a * 'b]
    {
      get
      {
        def result = LinkedList ();
        foreach (kv in this)
          result.Add ((kv.Key, kv.Value));

        result
      }
    }
    
    public override ToString () : string
    {
      def better_to_string (x : object) {
        match (x) {
          | str is string => "\"" + str + "\""
          | ch is char => "'" + ch.ToString () + "'"
          | _ => x.ToString ()
        }
      }
      
      def sb = System.Text.StringBuilder ("{");
      foreach (x in this) {
        _ = sb.Append (better_to_string (x.Key));
        _ = sb.Append (": ");
        _ = sb.Append (better_to_string (x.Value));
        _ = sb.Append (", ");
      }
      // cut the last ', ' if it exists
      when (sb.Length > 1)
        sb.Length = sb.Length - 2;
      _ = sb.Append ('}');
      sb.ToString ()
    }
  } /* end of class Hashtable ('a,'b) */
} /* end of namespace */

