/* ===========================================================================
*
*                            PUBLIC DOMAIN NOTICE
*               National Center for Biotechnology Information
*
*  This software/database is a "United States Government Work" under the
*  terms of the United States Copyright Act.  It was written as part of
*  the author's official duties as a United States Government employee and
*  thus cannot be copyrighted.  This software/database is freely available
*  to the public for use. The National Library of Medicine and the U.S.
*  Government have not placed any restriction on its use or reproduction.
*
*  Although all reasonable efforts have been taken to ensure the accuracy
*  and reliability of the software and data, the NLM and the U.S.
*  Government do not and cannot warrant the performance or results that
*  may be obtained by using this software or data. The NLM and the U.S.
*  Government disclaim all warranties, express or implied, including
*  warranties of performance, merchantability or fitness for any particular
*  purpose.
*
*  Please cite the author in any work or product based on this material.
*
* ===========================================================================*/

/** @file matrix_frequency_data.c
 * Joint probabilities for specific matrices.
 *
 * @author Yi-Kuo Yu, Alejandro Schaffer, E. Michael Gertz
 */
#ifndef SKIP_DOXYGEN_PROCESSING
static char const rcsid[] =
    "$Id: matrix_frequency_data.c 90713 2006-09-27 17:36:18Z gertz $";
#endif /* SKIP_DOXYGEN_PROCESSING */

#include <stdlib.h>
#include <algo/blast/core/ncbi_std.h>
#include <algo/blast/composition_adjustment/composition_constants.h>
#include <algo/blast/composition_adjustment/matrix_frequency_data.h>


/*
 * Joint and background probabilities for the BLOSUM series of
 * matrices.
 *
 * These data are the actual joint probabilities originally used to
 * generate the BLOSUM matrices, printed at full precision (because we
 * can.)  The blocks data was obtained from
 *
 *    ftp://ftp.ncbi.nih.gov/repository/blocks/unix/blosum/blocks_5.0.dat.Z
 *
 * and used to generate the joint probabilities for the BLOSUM
 * matrices, using a small modification to the 'blosum' program
 * supplied in 
 *
 *    ftp://ftp.ncbi.nih.gov/repository/blocks/unix/blosum/blosum.tar.Z
 *
 * The 'blosum' program generates files that contain joint
 * probabilities, e.g. 'blosum62.qij', but these probabilities are by
 * default printed at low accuracy.  We modified the print statements
 * the generate blosum??.qij to print at full precision. The specific
 * modification to the Oct 18, 1992 version of the file blosum.c is
 *
 * 229c229
 * <        fprintf(fout, "%.4f ", (double) fij / totpairs);
 * ---
 * >        fprintf(fout, "%.16e ", (double) fij / totpairs);
 *
 * Internally, the 'blosum' program uses exactly the higher-precision
 * values that we print out to compute matrices.
 */


/** Joint probabilities for BLOSUM45 */
static const double
BLOSUM45_JOINT_PROBS[COMPO_NUM_TRUE_AA][COMPO_NUM_TRUE_AA] =
{
  {1.8067859192199651e-02, 2.8632970280029189e-03, 2.5852716092104864e-03,
   2.6802872542729885e-03, 1.4721761726112878e-03, 2.4357016851740214e-03,
   3.7509741193044975e-03, 6.3392371060867767e-03, 1.2691974410907309e-03,
   3.6179059863255059e-03, 5.2267067349121834e-03, 3.6916191127573101e-03,
   1.6118871872377726e-03, 2.1340510849853227e-03, 2.3620871424910785e-03,
   6.2253573100243412e-03, 4.1104541288450981e-03, 5.9150588019808183e-04,
   1.6885104821419373e-03, 5.5306101689579796e-03},
  {2.8632970280029189e-03, 1.2981201352308378e-02, 1.9658257815459023e-03,
   2.0049066095474076e-03, 5.7991132007379289e-04, 2.4961427104605001e-03,
   3.0703156814048727e-03, 2.2360083825481514e-03, 1.2617224002813071e-03,
   1.5810689324881372e-03, 2.9473459585734607e-03, 6.0953893700066292e-03,
   1.0175430650195330e-03, 1.4313644854523438e-03, 1.2956809668774457e-03,
   2.5563801601305607e-03, 1.9632608045077912e-03, 4.0556397557578358e-04,
   1.4263902102458958e-03, 2.1140813134646885e-03},
  {2.5852716092104864e-03, 1.9658257815459023e-03, 7.8501470034711886e-03,
   3.1259558053206788e-03, 6.6941554412093584e-04, 1.5524817804651980e-03,
   2.1724722312522963e-03, 3.3237456813136586e-03, 1.2666696726132058e-03,
   1.4625067095661482e-03, 1.8988677499765833e-03, 2.8106528553236663e-03,
   7.1583261905988334e-04, 1.1127971670944336e-03, 1.1422087671867729e-03,
   3.0730446623880589e-03, 2.4501354524355920e-03, 2.1811180797041274e-04,
   8.5687995159461643e-04, 1.6162116532974411e-03},
  {2.6802872542729885e-03, 2.0049066095474076e-03, 3.1259558053206788e-03,
   1.3237563732845571e-02, 6.2265374552703391e-04, 1.7098653558570117e-03,
   4.7436012699642992e-03, 2.7602948240886133e-03, 1.2034182798676748e-03,
   1.3552305161392132e-03, 2.1559462215056625e-03, 2.8090778444402500e-03,
   6.1643397738889020e-04, 9.9489940177469407e-04, 1.5334105263610980e-03,
   2.8268709927001139e-03, 2.2865632447178858e-03, 2.4814784831918892e-04,
   1.0831280925706914e-03, 1.7172231262216453e-03},
  {1.4721761726112878e-03, 5.7991132007379289e-04, 6.6941554412093584e-04,
   6.2265374552703391e-04, 9.4412874076739879e-03, 4.1042025902182758e-04,
   7.4458585782220040e-04, 9.8139056130103924e-04, 2.8594866937112776e-04,
   7.8981156978924646e-04, 1.4823588205010103e-03, 7.7106114710675307e-04,
   3.5586318613686682e-04, 6.5657489327527010e-04, 4.1203914885321426e-04,
   1.1459765491404468e-03, 1.0139516933242551e-03, 1.0481557906271918e-04,
   3.8818429938079952e-04, 1.1756651305043552e-03},
  {2.4357016851740214e-03, 2.4961427104605001e-03, 1.5524817804651980e-03,
   1.7098653558570117e-03, 4.1042025902182758e-04, 5.6843729268296912e-03,
   3.1938622849651262e-03, 1.8500970336630773e-03, 1.0245276847880554e-03,
   1.2849197264462630e-03, 2.1618326111720785e-03, 2.8660992672576535e-03,
   8.4702447612542125e-04, 7.3960293189886505e-04, 1.0946565407497567e-03,
   2.3989430846357322e-03, 1.4716698819858265e-03, 3.0973861280282215e-04,
   1.0059662375582828e-03, 1.3753940441066036e-03},
  {3.7509741193044975e-03, 3.0703156814048727e-03, 2.1724722312522963e-03,
   4.7436012699642992e-03, 7.4458585782220040e-04, 3.1938622849651262e-03,
   1.3069658477215567e-02, 2.5107356320500625e-03, 1.3852382529276910e-03,
   1.7450831790645173e-03, 3.1086857110181228e-03, 4.4504838476236725e-03,
   8.9571496470550454e-04, 1.3425662787911086e-03, 2.2536863740022671e-03,
   3.2415108480856065e-03, 2.5397857261280324e-03, 4.0327648497408492e-04,
   1.2098752538504236e-03, 2.2567534430266742e-03},
  {6.3392371060867767e-03, 2.2360083825481514e-03, 3.3237456813136586e-03,
   2.7602948240886133e-03, 9.8139056130103924e-04, 1.8500970336630773e-03,
   2.5107356320500625e-03, 2.8511624448995753e-02, 1.2310524422167546e-03,
   1.9327062043969442e-03, 3.1643945035180797e-03, 3.0499894759105127e-03,
   1.0995162153881082e-03, 1.6723363525115967e-03, 2.2429535327172330e-03,
   4.8536849682850143e-03, 2.7272360099530155e-03, 5.9271120428223316e-04,
   1.3911803421160626e-03, 2.5146676607186078e-03},
  {1.2691974410907309e-03, 1.2617224002813071e-03, 1.2666696726132058e-03,
   1.2034182798676748e-03, 2.8594866937112776e-04, 1.0245276847880554e-03,
   1.3852382529276910e-03, 1.2310524422167546e-03, 5.8487547639124796e-03,
   6.9585846800609714e-04, 1.5573467613102150e-03, 1.3245694120032801e-03,
   5.7090651699323862e-04, 7.8182832078834081e-04, 6.9844962409397991e-04,
   1.2960325177514280e-03, 9.1931607877249958e-04, 1.4976351019639964e-04,
   1.2330522291934879e-03, 7.9246698200683503e-04},
  {3.6179059863255059e-03, 1.5810689324881372e-03, 1.4625067095661482e-03,
   1.3552305161392132e-03, 7.8981156978924646e-04, 1.2849197264462630e-03,
   1.7450831790645173e-03, 1.9327062043969442e-03, 6.9585846800609714e-04,
   1.2388694439401509e-02, 9.2647566985432136e-03, 1.9773203824746490e-03,
   2.2593292403687306e-03, 3.0575457844642532e-03, 1.6077385068018789e-03,
   2.3421848303183976e-03, 2.7542711781141610e-03, 4.6749463838911354e-04,
   1.8956764459114616e-03, 9.4267822167510788e-03},
  {5.2267067349121834e-03, 2.9473459585734607e-03, 1.8988677499765833e-03,
   2.1559462215056625e-03, 1.4823588205010103e-03, 2.1618326111720785e-03,
   3.1086857110181228e-03, 3.1643945035180797e-03, 1.5573467613102150e-03,
   9.2647566985432136e-03, 2.6346530417480200e-02, 3.1142098082328640e-03,
   4.0711638827261041e-03, 5.6719761997807835e-03, 1.9088185348168649e-03,
   3.1873279813785520e-03, 3.8426846085017905e-03, 8.4082046153240706e-04,
   3.0567323782711214e-03, 8.7503010016331175e-03},
  {3.6916191127573101e-03, 6.0953893700066292e-03, 2.8106528553236663e-03,
   2.8090778444402500e-03, 7.7106114710675307e-04, 2.8660992672576535e-03,
   4.4504838476236725e-03, 3.0499894759105127e-03, 1.3245694120032801e-03,
   1.9773203824746490e-03, 3.1142098082328640e-03, 1.1978396089560937e-02,
   1.1104403771137466e-03, 1.4742483750212233e-03, 1.9968025222690676e-03,
   3.2668812435946099e-03, 2.7862613227875948e-03, 4.5043772800615431e-04,
   1.4940888302424689e-03, 2.4843185667476147e-03},
  {1.6118871872377726e-03, 1.0175430650195330e-03, 7.1583261905988334e-04,
   6.1643397738889020e-04, 3.5586318613686682e-04, 8.4702447612542125e-04,
   8.9571496470550454e-04, 1.0995162153881082e-03, 5.7090651699323862e-04,
   2.2593292403687306e-03, 4.0711638827261041e-03, 1.1104403771137466e-03,
   2.5874360305861857e-03, 1.2371987255888946e-03, 6.8765677463444332e-04,
   1.0131873446874034e-03, 1.1327846844696357e-03, 2.1265849704555571e-04,
   8.6680034269080448e-04, 2.1688399348504966e-03},
  {2.1340510849853227e-03, 1.4313644854523438e-03, 1.1127971670944336e-03,
   9.9489940177469407e-04, 6.5657489327527010e-04, 7.3960293189886505e-04,
   1.3425662787911086e-03, 1.6723363525115967e-03, 7.8182832078834081e-04,
   3.0575457844642532e-03, 5.6719761997807835e-03, 1.4742483750212233e-03,
   1.2371987255888946e-03, 1.2391203870190888e-02, 8.9236221263296761e-04,
   1.7312816295078565e-03, 1.7453918151105632e-03, 8.4095245274993082e-04,
   3.4263744717066835e-03, 3.0946984716836139e-03},
  {2.3620871424910785e-03, 1.2956809668774457e-03, 1.1422087671867729e-03,
   1.5334105263610980e-03, 4.1203914885321426e-04, 1.0946565407497567e-03,
   2.2536863740022671e-03, 2.2429535327172330e-03, 6.9844962409397991e-04,
   1.6077385068018789e-03, 1.9088185348168649e-03, 1.9968025222690676e-03,
   6.8765677463444332e-04, 8.9236221263296761e-04, 1.6004098119159461e-02,
   1.9539606804874826e-03, 1.9148086102650892e-03, 2.9896951765680344e-04,
   6.8907122080648483e-04, 1.6099029423219423e-03},
  {6.2253573100243412e-03, 2.5563801601305607e-03, 3.0730446623880589e-03,
   2.8268709927001139e-03, 1.1459765491404468e-03, 2.3989430846357322e-03,
   3.2415108480856065e-03, 4.8536849682850143e-03, 1.2960325177514280e-03,
   2.3421848303183976e-03, 3.1873279813785520e-03, 3.2668812435946099e-03,
   1.0131873446874034e-03, 1.7312816295078565e-03, 1.9539606804874826e-03,
   1.0412392255159685e-02, 4.7291383160678064e-03, 3.5033107367116276e-04,
   1.4588588725561497e-03, 3.1161052123834937e-03},
  {4.1104541288450981e-03, 1.9632608045077912e-03, 2.4501354524355920e-03,
   2.2865632447178858e-03, 1.0139516933242551e-03, 1.4716698819858265e-03,
   2.5397857261280324e-03, 2.7272360099530155e-03, 9.1931607877249958e-04,
   2.7542711781141610e-03, 3.8426846085017905e-03, 2.7862613227875948e-03,
   1.1327846844696357e-03, 1.7453918151105632e-03, 1.9148086102650892e-03,
   4.7291383160678064e-03, 8.6497284024982021e-03, 3.1868671219664247e-04,
   1.3184355271303671e-03, 3.8213223234837145e-03},
  {5.9150588019808183e-04, 4.0556397557578358e-04, 2.1811180797041274e-04,
   2.4814784831918892e-04, 1.0481557906271918e-04, 3.0973861280282215e-04,
   4.0327648497408492e-04, 5.9271120428223316e-04, 1.4976351019639964e-04,
   4.6749463838911354e-04, 8.4082046153240706e-04, 4.5043772800615431e-04,
   2.1265849704555571e-04, 8.4095245274993082e-04, 2.9896951765680344e-04,
   3.5033107367116276e-04, 3.1868671219664247e-04, 5.3077625858668461e-03,
   8.1328347217338593e-04, 4.4278005713663357e-04},
  {1.6885104821419373e-03, 1.4263902102458958e-03, 8.5687995159461643e-04,
   1.0831280925706914e-03, 3.8818429938079952e-04, 1.0059662375582828e-03,
   1.2098752538504236e-03, 1.3911803421160626e-03, 1.2330522291934879e-03,
   1.8956764459114616e-03, 3.0567323782711214e-03, 1.4940888302424689e-03,
   8.6680034269080448e-04, 3.4263744717066835e-03, 6.8907122080648483e-04,
   1.4588588725561497e-03, 1.3184355271303671e-03, 8.1328347217338593e-04,
   6.5658130332894053e-03, 1.9130855013886519e-03},
  {5.5306101689579796e-03, 2.1140813134646885e-03, 1.6162116532974411e-03,
   1.7172231262216453e-03, 1.1756651305043552e-03, 1.3753940441066036e-03,
   2.2567534430266742e-03, 2.5146676607186078e-03, 7.9246698200683503e-04,
   9.4267822167510788e-03, 8.7503010016331175e-03, 2.4843185667476147e-03,
   2.1688399348504966e-03, 3.0946984716836139e-03, 1.6099029423219423e-03,
   3.1161052123834937e-03, 3.8213223234837145e-03, 4.4278005713663357e-04,
   1.9130855013886519e-03, 1.4058608660834061e-02}};

/** Background frequencies for BLOSUM45 */
static const double
BLOSUM45_bg[COMPO_NUM_TRUE_AA] =
{7.8254696826829992e-02, 5.2293400508515497e-02, 4.1869234505207159e-02,
 4.9715478669430609e-02, 2.3504091554598174e-02, 3.5913319135963809e-02,
 5.8088865918176615e-02, 7.4985562582061308e-02, 2.4796120028184831e-02,
 6.1906885653760506e-02, 9.3758807045384390e-02, 6.0002347578480664e-02,
 2.5078218042817218e-02, 4.6429254925009635e-02, 4.2599362265185327e-02,
 6.1179450532953900e-02, 5.2495886521295561e-02, 1.3367812099806363e-02,
 3.3781387194819178e-02, 6.9979818411519248e-02};


/** Joint probabilities for BLOSUM50 */
static const double
BLOSUM50_JOINT_PROBS[COMPO_NUM_TRUE_AA][COMPO_NUM_TRUE_AA] =
{
  {1.9177376165900414e-02, 2.7240376714546557e-03, 2.4031492992281512e-03,
   2.5950400141435646e-03, 1.5161141503466080e-03, 2.1916331677173605e-03,
   3.4448278914639618e-03, 6.2251471153775524e-03, 1.2127849526349368e-03,
   3.5067963781079273e-03, 4.8064175187919377e-03, 3.3118156445774184e-03,
   1.5929928229676674e-03, 2.0111462534334831e-03, 2.2367690269707355e-03,
   6.2077372467778114e-03, 3.9182728585714704e-03, 4.8631739510959277e-04,
   1.5289255743243066e-03, 5.4421693501580945e-03},
  {2.7240376714546557e-03, 1.5249518561669190e-02, 1.9839339217171100e-03,
   1.8569646551144466e-03, 5.0841145558634812e-04, 2.5241992415035875e-03,
   2.9346714600391238e-03, 2.0330929248566329e-03, 1.2916661377324677e-03,
   1.4850904029213015e-03, 2.8200506536265085e-03, 6.3631786906688794e-03,
   8.8044244163237157e-04, 1.1827555704613821e-03, 1.1291633722052721e-03,
   2.5480707779992882e-03, 2.0755623893205866e-03, 3.8597740102525838e-04,
   1.2968650803478032e-03, 1.9767324973351923e-03},
  {2.4031492992281512e-03, 1.9839339217171100e-03, 1.0064274648985122e-02,
   3.4938528463521034e-03, 5.8526580393535813e-04, 1.5772225567522585e-03,
   2.2654955215314441e-03, 3.1227790949497000e-03, 1.5032392778514345e-03,
   1.2604525924282989e-03, 1.7212395367610532e-03, 2.6781179272946298e-03,
   6.5072987349609041e-04, 9.2077070726659231e-04, 1.0780835837505738e-03,
   3.1777703084905646e-03, 2.5607235691682607e-03, 2.2470939491808472e-04,
   9.0102455753719311e-04, 1.4642269583191670e-03},
  {2.5950400141435646e-03, 1.8569646551144466e-03, 3.4938528463521034e-03,
   1.6142284245400377e-02, 4.9055796014848500e-04, 1.7355972391605227e-03,
   4.8092932462930728e-03, 2.8256049847966923e-03, 1.1224039610689225e-03,
   1.2187267471037043e-03, 1.8320202151810498e-03, 2.6483031701007680e-03,
   5.4518900860208049e-04, 8.3372775443144032e-04, 1.4911089792375384e-03,
   2.8168184837142236e-03, 2.2443557004503045e-03, 2.1629521399451062e-04,
   8.6838200454238037e-04, 1.6071086977600263e-03},
  {1.5161141503466080e-03, 5.0841145558634812e-04, 5.8526580393535813e-04,
   4.9055796014848500e-04, 9.0790816399874165e-03, 3.7509694198458956e-04,
   5.7443167694834051e-04, 8.6416125741413223e-04, 2.8532822032727102e-04,
   8.3949095461630769e-04, 1.3866946702884293e-03, 5.9394346268641564e-04,
   3.7056882781629923e-04, 6.0643258121975913e-04, 3.6775301298071699e-04,
   1.1014893301583900e-03, 9.6593119839638185e-04, 9.5473415415687051e-05,
   4.0265275985100583e-04, 1.3071530803028910e-03},
  {2.1916331677173605e-03, 2.5241992415035875e-03, 1.5772225567522585e-03,
   1.7355972391605227e-03, 3.7509694198458956e-04, 5.7288780946700701e-03,
   3.2946407159359019e-03, 1.6542835961738131e-03, 1.0434512306085959e-03,
   1.1246276733803604e-03, 1.8792097500823605e-03, 2.8672107479796771e-03,
   8.1749695236423632e-04, 7.0039848968711364e-04, 1.0716022846036992e-03,
   2.2199500917807818e-03, 1.4976248429072652e-03, 3.4211307677156259e-04,
   8.6801232355735846e-04, 1.4099696811263587e-03},
  {3.4448278914639618e-03, 2.9346714600391238e-03, 2.2654955215314441e-03,
   4.8092932462930728e-03, 5.7443167694834051e-04, 3.2946407159359019e-03,
   1.4137738860962050e-02, 2.2506310350989874e-03, 1.2713631289270268e-03,
   1.5027392767795753e-03, 2.5813459827982222e-03, 4.3484287857007489e-03,
   8.3289745544177065e-04, 1.2381529956170115e-03, 1.9357892959573676e-03,
   2.9614835299989701e-03, 2.4465955340418752e-03, 4.2400226067181517e-04,
   1.2320532047293646e-03, 1.9587224017250621e-03},
  {6.2251471153775524e-03, 2.0330929248566329e-03, 3.1227790949497000e-03,
   2.8256049847966923e-03, 8.6416125741413223e-04, 1.6542835961738131e-03,
   2.2506310350989874e-03, 3.1590334199549770e-02, 1.0972793065146165e-03,
   1.7523277816042203e-03, 2.6606416500206632e-03, 2.7990435215878512e-03,
   9.3546770862675606e-04, 1.4703871152100261e-03, 1.8725532583338258e-03,
   4.3892726706643180e-03, 2.4867404596474229e-03, 5.1106914401047939e-04,
   1.1558215242534139e-03, 2.1542687026690753e-03},
  {1.2127849526349368e-03, 1.2916661377324677e-03, 1.5032392778514345e-03,
   1.1224039610689225e-03, 2.8532822032727102e-04, 1.0434512306085959e-03,
   1.2713631289270268e-03, 1.0972793065146165e-03, 6.3804882118271523e-03,
   6.5152626785360663e-04, 1.2931025418026663e-03, 1.3522962310675180e-03,
   4.5805146362189403e-04, 8.9702000483341665e-04, 5.8791282749163360e-04,
   1.2089586814863836e-03, 8.5349236685497410e-04, 1.6088475111162087e-04,
   1.2994388028749182e-03, 7.4034388655116329e-04},
  {3.5067963781079273e-03, 1.4850904029213015e-03, 1.2604525924282989e-03,
   1.2187267471037043e-03, 8.3949095461630769e-04, 1.1246276733803604e-03,
   1.5027392767795753e-03, 1.7523277816042203e-03, 6.5152626785360663e-04,
   1.3999757001274281e-02, 1.0390816757632199e-02, 1.7363767070665819e-03,
   2.2467399442463766e-03, 3.0498186875911964e-03, 1.3390359572833434e-03,
   2.0791955470136677e-03, 2.9182353019045755e-03, 4.5881135439627285e-04,
   1.7726237165460320e-03, 1.0732172589653342e-02},
  {4.8064175187919377e-03, 2.8200506536265085e-03, 1.7212395367610532e-03,
   1.8320202151810498e-03, 1.3866946702884293e-03, 1.8792097500823605e-03,
   2.5813459827982222e-03, 2.6606416500206632e-03, 1.2931025418026663e-03,
   1.0390816757632199e-02, 3.0409269773873687e-02, 2.6510115013545226e-03,
   4.1722701893507309e-03, 5.8008510370760203e-03, 1.7339408813938210e-03,
   2.9030148705129709e-03, 3.8403853123581615e-03, 7.8809411669902049e-04,
   2.7481481703784771e-03, 9.1689424893657400e-03},
  {3.3118156445774184e-03, 6.3631786906688794e-03, 2.6781179272946298e-03,
   2.6483031701007680e-03, 5.9394346268641564e-04, 2.8672107479796771e-03,
   4.3484287857007489e-03, 2.7990435215878512e-03, 1.3522962310675180e-03,
   1.7363767070665819e-03, 2.6510115013545226e-03, 1.2999587798271149e-02,
   9.6710745846319019e-04, 1.2387025901401758e-03, 1.8070838729286381e-03,
   3.0739875836290694e-03, 2.5875240617244397e-03, 3.6617320485888769e-04,
   1.2944286682925433e-03, 2.1916787299081464e-03},
  {1.5929928229676674e-03, 8.8044244163237157e-04, 6.5072987349609041e-04,
   5.4518900860208049e-04, 3.7056882781629923e-04, 8.1749695236423632e-04,
   8.3289745544177065e-04, 9.3546770862675606e-04, 4.5805146362189403e-04,
   2.2467399442463766e-03, 4.1722701893507309e-03, 9.6710745846319019e-04,
   2.8637130884961001e-03, 1.2334264856852042e-03, 5.4319510273222540e-04,
   9.8962843407290630e-04, 1.1386579188117374e-03, 2.5453642381012927e-04,
   7.4966082446689317e-04, 2.1431247883570799e-03},
  {2.0111462534334831e-03, 1.1827555704613821e-03, 9.2077070726659231e-04,
   8.3372775443144032e-04, 6.0643258121975913e-04, 7.0039848968711364e-04,
   1.2381529956170115e-03, 1.4703871152100261e-03, 8.9702000483341665e-04,
   3.0498186875911964e-03, 5.8008510370760203e-03, 1.2387025901401758e-03,
   1.2334264856852042e-03, 1.5352527520022995e-02, 7.3889079577318679e-04,
   1.5819841076161790e-03, 1.4517880681388670e-03, 8.8872966327839206e-04,
   3.9057132596179516e-03, 2.9949355095799514e-03},
  {2.2367690269707355e-03, 1.1291633722052721e-03, 1.0780835837505738e-03,
   1.4911089792375384e-03, 3.6775301298071699e-04, 1.0716022846036992e-03,
   1.9357892959573676e-03, 1.8725532583338258e-03, 5.8791282749163360e-04,
   1.3390359572833434e-03, 1.7339408813938210e-03, 1.8070838729286381e-03,
   5.4319510273222540e-04, 7.3889079577318679e-04, 1.7056411057608904e-02,
   1.8468214541146135e-03, 1.6196921560667427e-03, 2.3541955953234630e-04,
   6.4111850267928403e-04, 1.5531163167895289e-03},
  {6.2077372467778114e-03, 2.5480707779992882e-03, 3.1777703084905646e-03,
   2.8168184837142236e-03, 1.1014893301583900e-03, 2.2199500917807818e-03,
   2.9614835299989701e-03, 4.3892726706643180e-03, 1.2089586814863836e-03,
   2.0791955470136677e-03, 2.9030148705129709e-03, 3.0739875836290694e-03,
   9.8962843407290630e-04, 1.5819841076161790e-03, 1.8468214541146135e-03,
   1.1119087492164673e-02, 4.7130010640939217e-03, 3.1984588103465904e-04,
   1.2969411223575139e-03, 2.9238790831836834e-03},
  {3.9182728585714704e-03, 2.0755623893205866e-03, 2.5607235691682607e-03,
   2.2443557004503045e-03, 9.6593119839638185e-04, 1.4976248429072652e-03,
   2.4465955340418752e-03, 2.4867404596474229e-03, 8.5349236685497410e-04,
   2.9182353019045755e-03, 3.8403853123581615e-03, 2.5875240617244397e-03,
   1.1386579188117374e-03, 1.4517880681388670e-03, 1.6196921560667427e-03,
   4.7130010640939217e-03, 9.9511225720898735e-03, 3.6462369214353146e-04,
   1.2243502023766173e-03, 4.0630285182107874e-03},
  {4.8631739510959277e-04, 3.8597740102525838e-04, 2.2470939491808472e-04,
   2.1629521399451062e-04, 9.5473415415687051e-05, 3.4211307677156259e-04,
   4.2400226067181517e-04, 5.1106914401047939e-04, 1.6088475111162087e-04,
   4.5881135439627285e-04, 7.8809411669902049e-04, 3.6617320485888769e-04,
   2.5453642381012927e-04, 8.8872966327839206e-04, 2.3541955953234630e-04,
   3.1984588103465904e-04, 3.6462369214353146e-04, 5.8917653327232667e-03,
   8.1035438644761436e-04, 4.8135675107584730e-04},
  {1.5289255743243066e-03, 1.2968650803478032e-03, 9.0102455753719311e-04,
   8.6838200454238037e-04, 4.0265275985100583e-04, 8.6801232355735846e-04,
   1.2320532047293646e-03, 1.1558215242534139e-03, 1.2994388028749182e-03,
   1.7726237165460320e-03, 2.7481481703784771e-03, 1.2944286682925433e-03,
   7.4966082446689317e-04, 3.9057132596179516e-03, 6.4111850267928403e-04,
   1.2969411223575139e-03, 1.2243502023766173e-03, 8.1035438644761436e-04,
   7.7325651605433090e-03, 1.7646048865917325e-03},
  {5.4421693501580945e-03, 1.9767324973351923e-03, 1.4642269583191670e-03,
   1.6071086977600263e-03, 1.3071530803028910e-03, 1.4099696811263587e-03,
   1.9587224017250621e-03, 2.1542687026690753e-03, 7.4034388655116329e-04,
   1.0732172589653342e-02, 9.1689424893657400e-03, 2.1916787299081464e-03,
   2.1431247883570799e-03, 2.9949355095799514e-03, 1.5531163167895289e-03,
   2.9238790831836834e-03, 4.0630285182107874e-03, 4.8135675107584730e-04,
   1.7646048865917325e-03, 1.6356187478795130e-02}};

/** Background frequencies for BLOSUM50 */
static const double
BLOSUM50_bg[COMPO_NUM_TRUE_AA] =
{7.6539470498057649e-02, 5.3250385307217402e-02, 4.3637061980733187e-02,
 5.1393635127596211e-02, 2.2316032400410832e-02, 3.4923218698747475e-02,
 5.6445304260661695e-02, 7.3850907051359951e-02, 2.4711032253042223e-02,
 6.4065361639403171e-02, 9.5587467619348249e-02, 5.7876000358301254e-02,
 2.4385897213061735e-02, 4.8098159196680347e-02, 4.0885461298433984e-02,
 5.9478937760864596e-02, 5.2921707787277797e-02, 1.3706552419028580e-02,
 3.3493684732315711e-02, 7.2433722397458003e-02};


/** Joint probabilities for BLOSUM62 */
static const double
BLOSUM62_JOINT_PROBS[COMPO_NUM_TRUE_AA][COMPO_NUM_TRUE_AA] =
{
  {2.1497573378347484e-02, 2.3470224274721213e-03, 1.9493235258876179e-03,
   2.1674844853066858e-03, 1.5903351423026848e-03, 1.9242657898716525e-03,
   2.9879059292799641e-03, 5.8158526388051033e-03, 1.1076584657559144e-03,
   3.1880644746334580e-03, 4.4186245468471547e-03, 3.3466571942021082e-03,
   1.3412107617355408e-03, 1.6360627863999076e-03, 2.1568959784943114e-03,
   6.2524987419815400e-03, 3.7180506975672363e-03, 4.0281679108936688e-04,
   1.2999956675626666e-03, 5.0679056444508912e-03},
  {2.3470224274721213e-03, 1.7757465118386322e-02, 1.9786027128591904e-03,
   1.5865480081162602e-03, 3.9365984789376245e-04, 2.4858611089731411e-03,
   2.6933867548771758e-03, 1.7221140903704937e-03, 1.2407382229440791e-03,
   1.2435878276496955e-03, 2.4193952633248727e-03, 6.2339060289407083e-03,
   8.0309461712520876e-04, 9.3181986323789834e-04, 9.5783034332718700e-04,
   2.2660898636037261e-03, 1.7802796534180537e-03, 2.6571979312581875e-04,
   9.2634607111251918e-04, 1.5810185245264004e-03},
  {1.9493235258876179e-03, 1.9786027128591904e-03, 1.4140291972553610e-02,
   3.7201973506001745e-03, 4.3845466068066216e-04, 1.5304436972610567e-03,
   2.2097156829738759e-03, 2.8591871815612977e-03, 1.4301072616183181e-03,
   9.9437221166923172e-04, 1.3690958423974782e-03, 2.4402105140841090e-03,
   5.2943633069226512e-04, 7.5004227978192801e-04, 8.6016459857770028e-04,
   3.1466019144814608e-03, 2.2360795375444384e-03, 1.6159545671597605e-04,
   7.0048422794024819e-04, 1.2014015528772706e-03},
  {2.1674844853066858e-03, 1.5865480081162602e-03, 3.7201973506001745e-03,
   2.1274574617480089e-02, 3.9909227141697264e-04, 1.6481246723433428e-03,
   4.9158017471929655e-03, 2.5221102126636373e-03, 9.5384849402143984e-04,
   1.2347404942429857e-03, 1.5202051791453383e-03, 2.4453087721980561e-03,
   4.6429229320514104e-04, 7.6023722413111566e-04, 1.2373315413524663e-03,
   2.8035127901697272e-03, 1.8961512776990257e-03, 1.6218020183662784e-04,
   5.9842263937853702e-04, 1.3158365660538270e-03},
  {1.5903351423026848e-03, 3.9365984789376245e-04, 4.3845466068066216e-04,
   3.9909227141697264e-04, 1.1931352277704348e-02, 3.0937204045913537e-04,
   3.8338775043186374e-04, 7.6951976030099293e-04, 2.2976387481074697e-04,
   1.0956590131781735e-03, 1.5682982157153873e-03, 5.0124929379033781e-04,
   3.7717165634097634e-04, 5.1389991547056834e-04, 3.6111795849154795e-04,
   1.0432626586831986e-03, 9.3041313726939057e-04, 1.4474923964368156e-04,
   3.4603772624580643e-04, 1.3606607271146112e-03},
  {1.9242657898716525e-03, 2.4858611089731411e-03, 1.5304436972610567e-03,
   1.6481246723433428e-03, 3.0937204045913537e-04, 7.3292255467189687e-03,
   3.5385780499965817e-03, 1.3683038039160171e-03, 1.0489026828741754e-03,
   8.9102936026571569e-04, 1.6174411456311808e-03, 3.0968229715707327e-03,
   7.3993258722701268e-04, 5.4255147972143906e-04, 8.4668181752066874e-04,
   1.8931125300036275e-03, 1.3796838284921874e-03, 2.2737931366728891e-04,
   6.7584155312457842e-04, 1.1660966117775285e-03},
  {2.9879059292799641e-03, 2.6933867548771758e-03, 2.2097156829738759e-03,
   4.9158017471929655e-03, 3.8338775043186374e-04, 3.5385780499965817e-03,
   1.6133927472163669e-02, 1.9380952488713059e-03, 1.3667885452189439e-03,
   1.2192061706431622e-03, 2.0030316026648431e-03, 4.1322603720305197e-03,
   6.7909745467514783e-04, 8.5179405867513139e-04, 1.4216207127018586e-03,
   2.9539180653600089e-03, 2.0493063257644955e-03, 2.6488552587183780e-04,
   8.7044186256788659e-04, 1.6987763526262680e-03},
  {5.8158526388051033e-03, 1.7221140903704937e-03, 2.8591871815612977e-03,
   2.5221102126636373e-03, 7.6951976030099293e-04, 1.3683038039160171e-03,
   1.9380952488713059e-03, 3.7804346453413303e-02, 9.5813607255887238e-04,
   1.3849118546156933e-03, 2.0864716056392773e-03, 2.5392537741810947e-03,
   7.3281559749652399e-04, 1.1976708695723554e-03, 1.3641171883713547e-03,
   3.8342830901664762e-03, 2.1858459940987062e-03, 4.0740829083805248e-04,
   8.3467413018106177e-04, 1.8218235950233687e-03},
  {1.1076584657559144e-03, 1.2407382229440791e-03, 1.4301072616183181e-03,
   9.5384849402143984e-04, 2.2976387481074697e-04, 1.0489026828741754e-03,
   1.3667885452189439e-03, 9.5813607255887238e-04, 9.2802502369336622e-03,
   5.8089627083019206e-04, 9.8696608463236094e-04, 1.1873625842258938e-03,
   3.8264639620910225e-04, 8.1041076335565583e-04, 4.7770135861914477e-04,
   1.1052034635193162e-03, 7.4371746073077327e-04, 1.5168037757411286e-04,
   1.5213771111755425e-03, 6.4882907765797669e-04},
  {3.1880644746334580e-03, 1.2435878276496955e-03, 9.9437221166923172e-04,
   1.2347404942429857e-03, 1.0956590131781735e-03, 8.9102936026571569e-04,
   1.2192061706431622e-03, 1.3849118546156933e-03, 5.8089627083019206e-04,
   1.8441526588740136e-02, 1.1382470627796603e-02, 1.5655862274689192e-03,
   2.5081290988482057e-03, 3.0458868657559346e-03, 1.0068164685944146e-03,
   1.7225081689171561e-03, 2.6953622613315018e-03, 3.6183761166072852e-04,
   1.3821121844492116e-03, 1.1972663837662637e-02},
  {4.4186245468471547e-03, 2.4193952633248727e-03, 1.3690958423974782e-03,
   1.5202051791453383e-03, 1.5682982157153873e-03, 1.6174411456311808e-03,
   2.0030316026648431e-03, 2.0864716056392773e-03, 9.8696608463236094e-04,
   1.1382470627796603e-02, 3.7141460156350926e-02, 2.4634345023228079e-03,
   4.9293545515183088e-03, 5.4151301166464015e-03, 1.4146090399381900e-03,
   2.4277107072013821e-03, 3.3238031308707055e-03, 7.3206640617832933e-04,
   2.2096734692836624e-03, 9.4786263030457313e-03},
  {3.3466571942021082e-03, 6.2339060289407083e-03, 2.4402105140841090e-03,
   2.4453087721980561e-03, 5.0124929379033781e-04, 3.0968229715707327e-03,
   4.1322603720305197e-03, 2.5392537741810947e-03, 1.1873625842258938e-03,
   1.5655862274689192e-03, 2.4634345023228079e-03, 1.6113385590544604e-02,
   9.0876633395557617e-04, 9.4875149773685364e-04, 1.5773020912564391e-03,
   3.1016069999481111e-03, 2.3467014804084987e-03, 2.7198500003555514e-04,
   9.9908866586876396e-04, 1.9360424083099779e-03},
  {1.3412107617355408e-03, 8.0309461712520876e-04, 5.2943633069226512e-04,
   4.6429229320514104e-04, 3.7717165634097634e-04, 7.3993258722701268e-04,
   6.7909745467514783e-04, 7.3281559749652399e-04, 3.8264639620910225e-04,
   2.5081290988482057e-03, 4.9293545515183088e-03, 9.0876633395557617e-04,
   4.0477309321969848e-03, 1.1901770463553603e-03, 4.0824445213456919e-04,
   8.5603787638552766e-04, 1.0095451907679563e-03, 1.9872537223131380e-04,
   5.7145288352831449e-04, 2.3123361470140736e-03},
  {1.6360627863999076e-03, 9.3181986323789834e-04, 7.5004227978192801e-04,
   7.6023722413111566e-04, 5.1389991547056834e-04, 5.4255147972143906e-04,
   8.5179405867513139e-04, 1.1976708695723554e-03, 8.1041076335565583e-04,
   3.0458868657559346e-03, 5.4151301166464015e-03, 9.4875149773685364e-04,
   1.1901770463553603e-03, 1.8277684015431908e-02, 5.2528021756783813e-04,
   1.1939618185901600e-03, 1.1624184369750680e-03, 8.4917468952377874e-04,
   4.2392005745634370e-03, 2.5763052227920180e-03},
  {2.1568959784943114e-03, 9.5783034332718700e-04, 8.6016459857770028e-04,
   1.2373315413524663e-03, 3.6111795849154795e-04, 8.4668181752066874e-04,
   1.4216207127018586e-03, 1.3641171883713547e-03, 4.7770135861914477e-04,
   1.0068164685944146e-03, 1.4146090399381900e-03, 1.5773020912564391e-03,
   4.0824445213456919e-04, 5.2528021756783813e-04, 1.9066033679132538e-02,
   1.6662567934883051e-03, 1.3511005665728870e-03, 1.4152209821874487e-04,
   4.5224391125285910e-04, 1.2451325046931832e-03},
  {6.2524987419815400e-03, 2.2660898636037261e-03, 3.1466019144814608e-03,
   2.8035127901697272e-03, 1.0432626586831986e-03, 1.8931125300036275e-03,
   2.9539180653600089e-03, 3.8342830901664762e-03, 1.1052034635193162e-03,
   1.7225081689171561e-03, 2.4277107072013821e-03, 3.1016069999481111e-03,
   8.5603787638552766e-04, 1.1939618185901600e-03, 1.6662567934883051e-03,
   1.2585947097159817e-02, 4.7004857686835334e-03, 2.8731729176487776e-04,
   1.0299846310599138e-03, 2.3587292053265561e-03},
  {3.7180506975672363e-03, 1.7802796534180537e-03, 2.2360795375444384e-03,
   1.8961512776990257e-03, 9.3041313726939057e-04, 1.3796838284921874e-03,
   2.0493063257644955e-03, 2.1858459940987062e-03, 7.4371746073077327e-04,
   2.6953622613315018e-03, 3.3238031308707055e-03, 2.3467014804084987e-03,
   1.0095451907679563e-03, 1.1624184369750680e-03, 1.3511005665728870e-03,
   4.7004857686835334e-03, 1.2514818886617953e-02, 2.8575770858467209e-04,
   9.4161039895612720e-04, 3.6402328079338207e-03},
  {4.0281679108936688e-04, 2.6571979312581875e-04, 1.6159545671597605e-04,
   1.6218020183662784e-04, 1.4474923964368156e-04, 2.2737931366728891e-04,
   2.6488552587183780e-04, 4.0740829083805248e-04, 1.5168037757411286e-04,
   3.6183761166072852e-04, 7.3206640617832933e-04, 2.7198500003555514e-04,
   1.9872537223131380e-04, 8.4917468952377874e-04, 1.4152209821874487e-04,
   2.8731729176487776e-04, 2.8575770858467209e-04, 6.4699301717154852e-03,
   8.8744160259272527e-04, 3.5578318710317554e-04},
  {1.2999956675626666e-03, 9.2634607111251918e-04, 7.0048422794024819e-04,
   5.9842263937853702e-04, 3.4603772624580643e-04, 6.7584155312457842e-04,
   8.7044186256788659e-04, 8.3467413018106177e-04, 1.5213771111755425e-03,
   1.3821121844492116e-03, 2.2096734692836624e-03, 9.9908866586876396e-04,
   5.7145288352831449e-04, 4.2392005745634370e-03, 4.5224391125285910e-04,
   1.0299846310599138e-03, 9.4161039895612720e-04, 8.8744160259272527e-04,
   1.0246100213822419e-02, 1.5489827890922993e-03},
  {5.0679056444508912e-03, 1.5810185245264004e-03, 1.2014015528772706e-03,
   1.3158365660538270e-03, 1.3606607271146112e-03, 1.1660966117775285e-03,
   1.6987763526262680e-03, 1.8218235950233687e-03, 6.4882907765797669e-04,
   1.1972663837662637e-02, 9.4786263030457313e-03, 1.9360424083099779e-03,
   2.3123361470140736e-03, 2.5763052227920180e-03, 1.2451325046931832e-03,
   2.3587292053265561e-03, 3.6402328079338207e-03, 3.5578318710317554e-04,
   1.5489827890922993e-03, 1.9631915140537640e-02}};

/** Background frequencies for BLOSUM62 */
static const double
BLOSUM62_bg[COMPO_NUM_TRUE_AA] =
{7.4216205067993410e-02, 5.1614486141284638e-02, 4.4645808512757915e-02,
 5.3626000838554413e-02, 2.4687457167944848e-02, 3.4259650591416023e-02,
 5.4311925684587502e-02, 7.4146941452644999e-02, 2.6212984805266227e-02,
 6.7917367618953756e-02, 9.8907868497150955e-02, 5.8155682303079680e-02,
 2.4990197579643110e-02, 4.7418459742284751e-02, 3.8538003320306206e-02,
 5.7229029476494421e-02, 5.0891364550287033e-02, 1.3029956129972148e-02,
 3.2281512313758580e-02, 7.2919098205619245e-02};


/** Joint probabilities for BLOSUM80 */
static const double
BLOSUM80_JOINT_PROBS[COMPO_NUM_TRUE_AA][COMPO_NUM_TRUE_AA] =
{
  {2.5190801261944146e-02, 2.0340063576029794e-03, 1.5934450456787860e-03,
   1.7567882204091898e-03, 1.5342531535382758e-03, 1.7000666571091031e-03,
   2.7868757423594017e-03, 5.3312141672192219e-03, 9.4989551434358397e-04,
   2.6511420856442343e-03, 3.6482390472164553e-03, 2.8776040054767438e-03,
   1.1106592033970583e-03, 1.3579821676123010e-03, 2.1407549535065536e-03,
   6.3814552201747433e-03, 3.5676984041129148e-03, 3.2555946863964375e-04,
   1.1044305938818220e-03, 4.6043558088638440e-03},
  {2.0340063576029794e-03, 2.0957619284835711e-02, 1.6758487367421928e-03,
   1.2888167784480715e-03, 3.3826512646630144e-04, 2.3643445818728338e-03,
   2.2882006533010476e-03, 1.4590947777562337e-03, 1.1860735000568496e-03,
   1.0137588625981279e-03, 1.8206526954490602e-03, 6.0560763047425702e-03,
   6.2337540140620625e-04, 6.8229547250252861e-04, 8.5854176274829876e-04,
   2.0042877649097124e-03, 1.5124193439838248e-03, 2.1495826193954981e-04,
   7.3405490506437917e-04, 1.3053875072860433e-03},
  {1.5934450456787860e-03, 1.6758487367421928e-03, 1.6571102507116985e-02,
   3.6538256029687933e-03, 3.7275314562150083e-04, 1.3857458787672526e-03,
   1.9028611939066358e-03, 2.5087252791082940e-03, 1.2287120285654496e-03,
   7.4457563298143141e-04, 1.0685540535108611e-03, 2.2109484306704398e-03,
   4.0104114114639772e-04, 5.5233725060967317e-04, 6.5378230775305264e-04,
   2.8681766931253736e-03, 1.9573168921473445e-03, 1.3788261408849363e-04,
   5.7618561566013431e-04, 9.3483242504674316e-04},
  {1.7567882204091898e-03, 1.2888167784480715e-03, 3.6538256029687933e-03,
   2.6192308612565114e-02, 3.3106691118318270e-04, 1.3768335120404338e-03,
   4.7855109048933571e-03, 2.2232114513351937e-03, 8.1024112807968755e-04,
   7.7107432004201270e-04, 1.0524113745173926e-03, 1.9891158615195038e-03,
   3.3025592590943029e-04, 5.9228818844263222e-04, 9.2722402594094192e-04,
   2.3761553355682980e-03, 1.6423964465390574e-03, 1.1250313032631603e-04,
   4.5827153005301055e-04, 9.6057858837385150e-04},
  {1.5342531535382758e-03, 3.3826512646630144e-04, 3.7275314562150083e-04,
   3.3106691118318270e-04, 1.7242275829262476e-02, 2.8646474074708189e-04,
   2.8282275916935814e-04, 6.0199758435290119e-04, 1.6234838273827443e-04,
   1.1265997431595498e-03, 1.4150540356314203e-03, 3.8133934535800398e-04,
   3.5227309589971249e-04, 5.3668272671786679e-04, 2.9686417147468063e-04,
   9.5169015207393563e-04, 8.7056289887733830e-04, 1.2638354788184329e-04,
   3.0970164723884082e-04, 1.3402441983659027e-03},
  {1.7000666571091031e-03, 2.3643445818728338e-03, 1.3857458787672526e-03,
   1.3768335120404338e-03, 2.8646474074708189e-04, 9.4347765835274943e-03,
   3.4983547675341785e-03, 1.0949548858705770e-03, 1.1246735029182945e-03,
   6.9825772263661188e-04, 1.3621164757556699e-03, 2.8087671852638188e-03,
   7.2881923779495811e-04, 4.5182331847654853e-04, 6.9151715980429068e-04,
   1.6537440549432947e-03, 1.2213788117405877e-03, 1.9879174677422644e-04,
   5.4136767114461271e-04, 1.0117238998697397e-03},
  {2.7868757423594017e-03, 2.2882006533010476e-03, 1.9028611939066358e-03,
   4.7855109048933571e-03, 2.8282275916935814e-04, 3.4983547675341785e-03,
   2.0820161672110919e-02, 1.6705768658803739e-03, 1.2497919872780897e-03,
   9.6840513155216357e-04, 1.4960914225340589e-03, 3.5721908719380593e-03,
   5.7149419172049785e-04, 6.3988473796368903e-04, 1.2122196186289647e-03,
   2.6389347134212079e-03, 1.8749387967046983e-03, 1.9036380224872525e-04,
   6.3290036404780182e-04, 1.4751061012869160e-03},
  {5.3312141672192219e-03, 1.4590947777562337e-03, 2.5087252791082940e-03,
   2.2232114513351937e-03, 6.0199758435290119e-04, 1.0949548858705770e-03,
   1.6705768658803739e-03, 4.6339669111671353e-02, 7.5449961042330647e-04,
   9.4996724611759150e-04, 1.6066592192185663e-03, 2.0310505314366269e-03,
   5.3609624387219209e-04, 8.9864424405597940e-04, 1.0176342727297918e-03,
   3.4421697217907042e-03, 1.8920241991862482e-03, 2.9373791998702517e-04,
   6.1327956332154923e-04, 1.4084342214988788e-03},
  {9.4989551434358397e-04, 1.1860735000568496e-03, 1.2287120285654496e-03,
   8.1024112807968755e-04, 1.6234838273827443e-04, 1.1246735029182945e-03,
   1.2497919872780897e-03, 7.5449961042330647e-04, 1.0381617825381334e-02,
   4.4036371570812095e-04, 7.9508382719793930e-04, 1.0307766502203480e-03,
   2.6842308809614442e-04, 6.8472295590456679e-04, 4.0762982846324675e-04,
   9.6226821486916257e-04, 6.8751986685796880e-04, 1.4371608217276840e-04,
   1.6115232208711219e-03, 5.3755732656986873e-04},
  {2.6511420856442343e-03, 1.0137588625981279e-03, 7.4457563298143141e-04,
   7.7107432004201270e-04, 1.1265997431595498e-03, 6.9825772263661188e-04,
   9.6840513155216357e-04, 9.4996724611759150e-04, 4.4036371570812095e-04,
   2.1961426293375986e-02, 1.1130703369690390e-02, 1.1532712305851239e-03,
   2.4830606782385335e-03, 2.6597759616518901e-03, 7.3360010098323712e-04,
   1.4571406164888326e-03, 2.3588073239766052e-03, 3.3397592349852914e-04,
   1.2624333898651637e-03, 1.2271122820238527e-02},
  {3.6482390472164553e-03, 1.8206526954490602e-03, 1.0685540535108611e-03,
   1.0524113745173926e-03, 1.4150540356314203e-03, 1.3621164757556699e-03,
   1.4960914225340589e-03, 1.6066592192185663e-03, 7.9508382719793930e-04,
   1.1130703369690390e-02, 4.4208543625750947e-02, 1.9460828295282577e-03,
   5.1637253946183711e-03, 5.2243696624605148e-03, 1.1528201208432873e-03,
   2.0991567569779157e-03, 2.7981964830167271e-03, 6.3277768036502599e-04,
   2.0134515231625665e-03, 8.8913964764739482e-03},
  {2.8776040054767438e-03, 6.0560763047425702e-03, 2.2109484306704398e-03,
   1.9891158615195038e-03, 3.8133934535800398e-04, 2.8087671852638188e-03,
   3.5721908719380593e-03, 2.0310505314366269e-03, 1.0307766502203480e-03,
   1.1532712305851239e-03, 1.9460828295282577e-03, 1.8992311051287718e-02,
   7.1570376147636942e-04, 7.2977429729635017e-04, 1.2494764855845110e-03,
   2.5724600423967150e-03, 2.0209702186968510e-03, 1.9168209136920497e-04,
   7.8006357431686252e-04, 1.4847749228956784e-03},
  {1.1106592033970583e-03, 6.2337540140620625e-04, 4.0104114114639772e-04,
   3.3025592590943029e-04, 3.5227309589971249e-04, 7.2881923779495811e-04,
   5.7149419172049785e-04, 5.3609624387219209e-04, 2.6842308809614442e-04,
   2.4830606782385335e-03, 5.1637253946183711e-03, 7.1570376147636942e-04,
   5.3068580972096564e-03, 1.0285774291964053e-03, 3.3805426284931307e-04,
   6.9732452132326552e-04, 9.2870717652810894e-04, 1.9855470215781828e-04,
   4.6842552580683944e-04, 2.1900560317616967e-03},
  {1.3579821676123010e-03, 6.8229547250252861e-04, 5.5233725060967317e-04,
   5.9228818844263222e-04, 5.3668272671786679e-04, 4.5182331847654853e-04,
   6.3988473796368903e-04, 8.9864424405597940e-04, 6.8472295590456679e-04,
   2.6597759616518901e-03, 5.2243696624605148e-03, 7.2977429729635017e-04,
   1.0285774291964053e-03, 2.1050081047094053e-02, 4.2695305789420819e-04,
   9.9526342159352650e-04, 1.0513746746114907e-03, 7.4360162788489814e-04,
   4.5643669065202081e-03, 2.2374943285650862e-03},
  {2.1407549535065536e-03, 8.5854176274829876e-04, 6.5378230775305264e-04,
   9.2722402594094192e-04, 2.9686417147468063e-04, 6.9151715980429068e-04,
   1.2122196186289647e-03, 1.0176342727297918e-03, 4.0762982846324675e-04,
   7.3360010098323712e-04, 1.1528201208432873e-03, 1.2494764855845110e-03,
   3.3805426284931307e-04, 4.2695305789420819e-04, 2.2132147568842062e-02,
   1.4255613382498733e-03, 1.0735194808918437e-03, 9.8582208625861730e-05,
   3.4299989987846913e-04, 1.0356422797820938e-03},
  {6.3814552201747433e-03, 2.0042877649097124e-03, 2.8681766931253736e-03,
   2.3761553355682980e-03, 9.5169015207393563e-04, 1.6537440549432947e-03,
   2.6389347134212079e-03, 3.4421697217907042e-03, 9.6226821486916257e-04,
   1.4571406164888326e-03, 2.0991567569779157e-03, 2.5724600423967150e-03,
   6.9732452132326552e-04, 9.9526342159352650e-04, 1.4255613382498733e-03,
   1.6728326838562371e-02, 4.7707748874885403e-03, 2.2457429196472549e-04,
   9.2159836957901727e-04, 2.0684030169239110e-03},
  {3.5676984041129148e-03, 1.5124193439838248e-03, 1.9573168921473445e-03,
   1.6423964465390574e-03, 8.7056289887733830e-04, 1.2213788117405877e-03,
   1.8749387967046983e-03, 1.8920241991862482e-03, 6.8751986685796880e-04,
   2.3588073239766052e-03, 2.7981964830167271e-03, 2.0209702186968510e-03,
   9.2870717652810894e-04, 1.0513746746114907e-03, 1.0735194808918437e-03,
   4.7707748874885403e-03, 1.5594999664688735e-02, 2.0707952088013908e-04,
   8.2888908190237710e-04, 3.2712773918631040e-03},
  {3.2555946863964375e-04, 2.1495826193954981e-04, 1.3788261408849363e-04,
   1.1250313032631603e-04, 1.2638354788184329e-04, 1.9879174677422644e-04,
   1.9036380224872525e-04, 2.9373791998702517e-04, 1.4371608217276840e-04,
   3.3397592349852914e-04, 6.3277768036502599e-04, 1.9168209136920497e-04,
   1.9855470215781828e-04, 7.4360162788489814e-04, 9.8582208625861730e-05,
   2.2457429196472549e-04, 2.0707952088013908e-04, 8.7238920952867408e-03,
   1.0281252180987448e-03, 3.6291275126003661e-04},
  {1.1044305938818220e-03, 7.3405490506437917e-04, 5.7618561566013431e-04,
   4.5827153005301055e-04, 3.0970164723884082e-04, 5.4136767114461271e-04,
   6.3290036404780182e-04, 6.1327956332154923e-04, 1.6115232208711219e-03,
   1.2624333898651637e-03, 2.0134515231625665e-03, 7.8006357431686252e-04,
   4.6842552580683944e-04, 4.5643669065202081e-03, 3.4299989987846913e-04,
   9.2159836957901727e-04, 8.2888908190237710e-04, 1.0281252180987448e-03,
   1.4809695895202394e-02, 1.2479627747244684e-03},
  {4.6043558088638440e-03, 1.3053875072860433e-03, 9.3483242504674316e-04,
   9.6057858837385150e-04, 1.3402441983659027e-03, 1.0117238998697397e-03,
   1.4751061012869160e-03, 1.4084342214988788e-03, 5.3755732656986873e-04,
   1.2271122820238527e-02, 8.8913964764739482e-03, 1.4847749228956784e-03,
   2.1900560317616967e-03, 2.2374943285650862e-03, 1.0356422797820938e-03,
   2.0684030169239110e-03, 3.2712773918631040e-03, 3.6291275126003661e-04,
   1.2479627747244684e-03, 2.4557979462295820e-02}};

/** Background frequencies for BLOSUM80 */
static const double
BLOSUM80_bg[COMPO_NUM_TRUE_AA] =
{7.2647227078731003e-02, 5.0418078079712533e-02, 4.2998652475215837e-02,
 5.3630877849155469e-02, 2.8859643195758448e-02, 3.3634522394591609e-02,
 5.4557686298480158e-02, 7.6673641116832586e-02, 2.5417438256716121e-02,
 6.7169462169032662e-02, 9.9526086073919384e-02, 5.4794439692059750e-02,
 2.4441485110408976e-02, 4.7108293477054419e-02, 3.8215524905474578e-02,
 5.7239465972425123e-02, 5.0130851564694504e-02, 1.4489654685450317e-02,
 3.4849727270340386e-02, 7.3197242333946158e-02};


/** Joint probabilities for BLOSUM90 */
static const double
BLOSUM90_JOINT_PROBS[COMPO_NUM_TRUE_AA][COMPO_NUM_TRUE_AA] =
{
  {2.8941466476520718e-02, 1.8963172918503119e-03, 1.4296539454877236e-03,
   1.6120541525071418e-03, 1.4378045557717186e-03, 1.5518753295389346e-03,
   2.6715149842246304e-03, 4.9176944070822995e-03, 8.3429471034916194e-04,
   2.2806166994472367e-03, 3.1400426239163565e-03, 2.5717145234657262e-03,
   9.7190459199705012e-04, 1.1707482997247202e-03, 1.9972723941546994e-03,
   6.1234827650497045e-03, 3.4990592936902896e-03, 3.1628263576966828e-04,
   9.5781456962094815e-04, 4.2309560726043610e-03},
  {1.8963172918503119e-03, 2.2892163644406403e-02, 1.4545259958564671e-03,
   1.1118856442462741e-03, 3.0043550945087249e-04, 2.0799816656266574e-03,
   1.9867845182036542e-03, 1.2814494483589008e-03, 1.0659803319010067e-03,
   8.7747376727426759e-04, 1.5563382766215293e-03, 5.6726070720652388e-03,
   5.5601248568691477e-04, 6.0014287440373390e-04, 7.7860213318230740e-04,
   1.7926007027022699e-03, 1.3337993839260983e-03, 1.8822355540425045e-04,
   6.3568077704304765e-04, 1.1439035919666126e-03},
  {1.4296539454877236e-03, 1.4545259958564671e-03, 1.7753653137209658e-02,
   3.4122251157479486e-03, 3.5262434817793110e-04, 1.2595133876871906e-03,
   1.7189800468094042e-03, 2.1939494568804173e-03, 1.1268322842671913e-03,
   6.2771773800865524e-04, 9.0162895039764071e-04, 1.9751605656395136e-03,
   3.4437514139917542e-04, 4.5695294787233016e-04, 5.6210182826678802e-04,
   2.6415583911000790e-03, 1.8096241176560255e-03, 1.1361193219385239e-04,
   5.3166878528705414e-04, 7.8744578895086331e-04},
  {1.6120541525071418e-03, 1.1118856442462741e-03, 3.4122251157479486e-03,
   2.8905985529761216e-02, 2.8205144393017687e-04, 1.2470706135571605e-03,
   4.8098591474193932e-03, 2.0787428065125128e-03, 7.3570523495442291e-04,
   6.1360733950558593e-04, 8.9746213145538078e-04, 1.7935339070485811e-03,
   2.8515554540729039e-04, 4.9706948686116547e-04, 8.3650620447870902e-04,
   2.1902661040309438e-03, 1.4770487412630099e-03, 9.6154148977656466e-05,
   4.1238640135342840e-04, 8.1983005205147035e-04},
  {1.4378045557717186e-03, 3.0043550945087249e-04, 3.5262434817793110e-04,
   2.8205144393017687e-04, 1.8963924833314688e-02, 2.4062393656260663e-04,
   2.4018396300677406e-04, 5.1191591124580998e-04, 1.4262451141854431e-04,
   1.0187249201952210e-03, 1.2204913461061227e-03, 3.4081454298601579e-04,
   3.0855994042178013e-04, 5.4061743869703049e-04, 2.6341296370767072e-04,
   9.0081780182465609e-04, 8.3760722742725604e-04, 1.1402385673074935e-04,
   3.0875531409262481e-04, 1.2790175783090351e-03},
  {1.5518753295389346e-03, 2.0799816656266574e-03, 1.2595133876871906e-03,
   1.2470706135571605e-03, 2.4062393656260663e-04, 1.1258473852655668e-02,
   3.3929068022966746e-03, 1.0135072388571143e-03, 1.0428112436817220e-03,
   6.2154432511249121e-04, 1.2200582014470405e-03, 2.6612615004061048e-03,
   6.9614541817054399e-04, 4.3847028249800182e-04, 6.3169039064483155e-04,
   1.5347331351459369e-03, 1.1633090191045159e-03, 1.8118969468635933e-04,
   4.8090269093961543e-04, 8.6038475380201306e-04},
  {2.6715149842246304e-03, 1.9867845182036542e-03, 1.7189800468094042e-03,
   4.8098591474193932e-03, 2.4018396300677406e-04, 3.3929068022966746e-03,
   2.4082550387620767e-02, 1.5173914448105995e-03, 1.0609424122148584e-03,
   8.6556272233095357e-04, 1.2693446983220930e-03, 3.1728712431242272e-03,
   4.8675933771578960e-04, 5.2592632830518644e-04, 1.1283633992741310e-03,
   2.4358395159902540e-03, 1.7460695145653721e-03, 1.7186658965667723e-04,
   5.3762929629223334e-04, 1.3537458445130380e-03},
  {4.9176944070822995e-03, 1.2814494483589008e-03, 2.1939494568804173e-03,
   2.0787428065125128e-03, 5.1191591124580998e-04, 1.0135072388571143e-03,
   1.5173914448105995e-03, 5.2105577140795424e-02, 6.6116372972342219e-04,
   7.9279793619671047e-04, 1.4084669216931525e-03, 1.8091800840455807e-03,
   4.5905141534639837e-04, 7.2640563736237702e-04, 9.1606958049118977e-04,
   3.2250721573096343e-03, 1.6376154647606311e-03, 2.8505580568873712e-04,
   5.3377971995623207e-04, 1.1777891482397749e-03},
  {8.3429471034916194e-04, 1.0659803319010067e-03, 1.1268322842671913e-03,
   7.3570523495442291e-04, 1.4262451141854431e-04, 1.0428112436817220e-03,
   1.0609424122148584e-03, 6.6116372972342219e-04, 1.1500733945516558e-02,
   3.8185691425617459e-04, 6.9827301155994540e-04, 9.0371184948679912e-04,
   2.4699125636645672e-04, 5.9147458541551113e-04, 3.8444740537595942e-04,
   8.5932297611334600e-04, 6.0598617290751988e-04, 1.4705149603278306e-04,
   1.4431522344877262e-03, 4.5696999458730438e-04},
  {2.2806166994472367e-03, 8.7747376727426759e-04, 6.2771773800865524e-04,
   6.1360733950558593e-04, 1.0187249201952210e-03, 6.2154432511249121e-04,
   8.6556272233095357e-04, 7.9279793619671047e-04, 3.8185691425617459e-04,
   2.4545975084797667e-02, 1.0187721075530867e-02, 1.0015355309232945e-03,
   2.3855654456497641e-03, 2.3760161494061206e-03, 6.6596344099825570e-04,
   1.2754928334842836e-03, 2.1146482135048364e-03, 2.8885067061466341e-04,
   1.1275668044226116e-03, 1.2010991497586472e-02},
  {3.1400426239163565e-03, 1.5563382766215293e-03, 9.0162895039764071e-04,
   8.9746213145538078e-04, 1.2204913461061227e-03, 1.2200582014470405e-03,
   1.2693446983220930e-03, 1.4084669216931525e-03, 6.9827301155994540e-04,
   1.0187721075530867e-02, 4.7314665506091480e-02, 1.6493087064879096e-03,
   4.9150964217477052e-03, 4.7389371254676835e-03, 1.0204208998598171e-03,
   1.9093387711173759e-03, 2.5387072113370619e-03, 5.5567682513643440e-04,
   1.7591338828767365e-03, 8.0719383241327648e-03},
  {2.5717145234657262e-03, 5.6726070720652388e-03, 1.9751605656395136e-03,
   1.7935339070485811e-03, 3.4081454298601579e-04, 2.6612615004061048e-03,
   3.1728712431242272e-03, 1.8091800840455807e-03, 9.0371184948679912e-04,
   1.0015355309232945e-03, 1.6493087064879096e-03, 2.1110006225715187e-02,
   6.5644301788968489e-04, 6.8011754565553524e-04, 1.1433484348853926e-03,
   2.2696044144588988e-03, 1.8742456882945837e-03, 1.4351856033992664e-04,
   6.5878172544710722e-04, 1.2879210940900179e-03},
  {9.7190459199705012e-04, 5.5601248568691477e-04, 3.4437514139917542e-04,
   2.8515554540729039e-04, 3.0855994042178013e-04, 6.9614541817054399e-04,
   4.8675933771578960e-04, 4.5905141534639837e-04, 2.4699125636645672e-04,
   2.3855654456497641e-03, 4.9150964217477052e-03, 6.5644301788968489e-04,
   6.7525741637165784e-03, 9.2071349348378116e-04, 2.9821836819244002e-04,
   6.2250823423415716e-04, 8.5497291624530018e-04, 1.8480270544289832e-04,
   4.1789997362558619e-04, 2.0390401916086623e-03},
  {1.1707482997247202e-03, 6.0014287440373390e-04, 4.5695294787233016e-04,
   4.9706948686116547e-04, 5.4061743869703049e-04, 4.3847028249800182e-04,
   5.2592632830518644e-04, 7.2640563736237702e-04, 5.9147458541551113e-04,
   2.3760161494061206e-03, 4.7389371254676835e-03, 6.8011754565553524e-04,
   9.2071349348378116e-04, 2.2759929778368840e-02, 3.8203945097730409e-04,
   9.0070409285055049e-04, 9.3733303528124379e-04, 7.0215975437974915e-04,
   4.5967694652672827e-03, 1.9728777259904423e-03},
  {1.9972723941546994e-03, 7.7860213318230740e-04, 5.6210182826678802e-04,
   8.3650620447870902e-04, 2.6341296370767072e-04, 6.3169039064483155e-04,
   1.1283633992741310e-03, 9.1606958049118977e-04, 3.8444740537595942e-04,
   6.6596344099825570e-04, 1.0204208998598171e-03, 1.1433484348853926e-03,
   2.9821836819244002e-04, 3.8203945097730409e-04, 2.4044365492630580e-02,
   1.3333728667143045e-03, 9.5861529112551574e-04, 1.0367126137135345e-04,
   2.9369391163915916e-04, 9.1754627765824664e-04},
  {6.1234827650497045e-03, 1.7926007027022699e-03, 2.6415583911000790e-03,
   2.1902661040309438e-03, 9.0081780182465609e-04, 1.5347331351459369e-03,
   2.4358395159902540e-03, 3.2250721573096343e-03, 8.5932297611334600e-04,
   1.2754928334842836e-03, 1.9093387711173759e-03, 2.2696044144588988e-03,
   6.2250823423415716e-04, 9.0070409285055049e-04, 1.3333728667143045e-03,
   2.0222291957641345e-02, 4.8820444720790194e-03, 2.0564914018856946e-04,
   8.5252084753556067e-04, 1.9182160488867347e-03},
  {3.4990592936902896e-03, 1.3337993839260983e-03, 1.8096241176560255e-03,
   1.4770487412630099e-03, 8.3760722742725604e-04, 1.1633090191045159e-03,
   1.7460695145653721e-03, 1.6376154647606311e-03, 6.0598617290751988e-04,
   2.1146482135048364e-03, 2.5387072113370619e-03, 1.8742456882945837e-03,
   8.5497291624530018e-04, 9.3733303528124379e-04, 9.5861529112551574e-04,
   4.8820444720790194e-03, 1.9324325014986026e-02, 1.8008131088087498e-04,
   7.7364223266285920e-04, 3.0481291734621883e-03},
  {3.1628263576966828e-04, 1.8822355540425045e-04, 1.1361193219385239e-04,
   9.6154148977656466e-05, 1.1402385673074935e-04, 1.8118969468635933e-04,
   1.7186658965667723e-04, 2.8505580568873712e-04, 1.4705149603278306e-04,
   2.8885067061466341e-04, 5.5567682513643440e-04, 1.4351856033992664e-04,
   1.8480270544289832e-04, 7.0215975437974915e-04, 1.0367126137135345e-04,
   2.0564914018856946e-04, 1.8008131088087498e-04, 1.0080756092360078e-02,
   9.7237191290610822e-04, 3.3849359263814259e-04},
  {9.5781456962094815e-04, 6.3568077704304765e-04, 5.3166878528705414e-04,
   4.1238640135342840e-04, 3.0875531409262481e-04, 4.8090269093961543e-04,
   5.3762929629223334e-04, 5.3377971995623207e-04, 1.4431522344877262e-03,
   1.1275668044226116e-03, 1.7591338828767365e-03, 6.5878172544710722e-04,
   4.1789997362558619e-04, 4.5967694652672827e-03, 2.9369391163915916e-04,
   8.5252084753556067e-04, 7.7364223266285920e-04, 9.7237191290610822e-04,
   1.7299655473877330e-02, 1.0599282277832859e-03},
  {4.2309560726043610e-03, 1.1439035919666126e-03, 7.8744578895086331e-04,
   8.1983005205147035e-04, 1.2790175783090351e-03, 8.6038475380201306e-04,
   1.3537458445130380e-03, 1.1777891482397749e-03, 4.5696999458730438e-04,
   1.2010991497586472e-02, 8.0719383241327648e-03, 1.2879210940900179e-03,
   2.0390401916086623e-03, 1.9728777259904423e-03, 9.1754627765824664e-04,
   1.9182160488867347e-03, 3.0481291734621883e-03, 3.3849359263814259e-04,
   1.0599282277832859e-03, 2.8697002670368957e-02}};

/** Background frequencies for BLOSUM90 */
static const double
BLOSUM90_bg[COMPO_NUM_TRUE_AA] =
{7.2552570322773416e-02, 4.9204908670176822e-02, 4.1453803904895913e-02,
 5.4114599751069477e-02, 2.9605031943377283e-02, 3.3576453482421185e-02,
 5.5175092196696710e-02, 7.9252675455356913e-02, 2.4890326300616415e-02,
 6.6060229109246138e-02, 9.6973050911305106e-02, 5.3375686232455331e-02,
 2.4402790064347962e-02, 4.6515405498268582e-02, 3.8659721995628657e-02,
 5.8095437228457614e-02, 5.1596863495160232e-02, 1.5369491541399532e-02,
 3.5653734247116535e-02, 7.3472127649230390e-02};


/** Joint probabilities for PAM30 */
static const double
PAM30_JOINT_PROBS[COMPO_NUM_TRUE_AA][COMPO_NUM_TRUE_AA] =
 {{5.9127839295455724e-02, 3.2440614389978224e-04, 1.0035946285397422e-03,
   1.2937791530855702e-03, 3.1489290812535621e-04, 7.8441998276182750e-04,
   1.9549522618591804e-03, 4.4450248999927649e-03, 2.4408739253648561e-04,
   6.3834521530456922e-04, 8.5734186469652062e-04, 6.6918201622206063e-04,
   2.4336822178217660e-04, 1.9674172887918488e-04, 2.6202120968515422e-03,
   5.3063352404797842e-03, 4.2185799566371992e-03, 8.9313082115936708e-06,
   1.8225253115472477e-04, 2.6924708725231086e-03},
  {3.2440614389978224e-04, 3.1662308120430839e-02, 2.3762837035855522e-04,
   5.6563724512424312e-05, 9.3129983865266353e-05, 9.1528180957759745e-04,
   8.6839478291009283e-05, 1.4407449709189928e-04, 8.0713273132343639e-04,
   2.3918939892306258e-04, 1.8565923197220291e-04, 3.6581384858227193e-03,
   1.4663859770241663e-04, 6.8103183598496806e-05, 5.5932452083258294e-04,
   1.0092107276753980e-03, 2.5480695878483489e-04, 2.2193510725567831e-04,
   3.6904551277914591e-05, 1.9661468480294445e-04},
  {1.0035946285397422e-03, 2.3762837035855522e-04, 2.3942402666531522e-02,
   3.3284186062777833e-03, 3.2538350231327974e-05, 4.6226978079510854e-04,
   9.2846484862398504e-04, 1.2584257731785510e-03, 1.5206567357122051e-03,
   2.6711601073772237e-04, 3.0916547891024103e-04, 2.2688803763503840e-03,
   2.7842591589795833e-05, 8.1379367658642105e-05, 2.9181042514696590e-04,
   2.7347990528488269e-03, 1.2577236167229159e-03, 2.6993374401853017e-05,
   2.6944461441064744e-04, 1.8273086910009501e-04},
  {1.2937791530855702e-03, 5.6563724512424312e-05, 3.3284186062777833e-03,
   3.1273803507946972e-02, 1.2103234943893133e-05, 7.6617484041782485e-04,
   5.4382363415210683e-03, 1.3553903446762449e-03, 4.2748415829840124e-04,
   1.3808202761146380e-04, 5.5405232219940619e-05, 8.2915092340941237e-04,
   1.5837617592492976e-05, 1.2183210262808245e-05, 1.6175221090296296e-04,
   9.3274720682462713e-04, 5.4799551604074552e-04, 2.6698310744553494e-06,
   2.7630471952142814e-05, 1.9652464806266956e-04},
  {3.1489290812535621e-04, 9.3129983865266353e-05, 3.2538350231327974e-05,
   1.2103234943893133e-05, 3.0911985925451389e-02, 1.0422909419121936e-05,
   1.3045150483146400e-05, 1.2331325932473484e-04, 8.7403624792468633e-05,
   1.4700821602711548e-04, 1.7204998362221924e-05, 2.0967486284459295e-05,
   4.6716406093025683e-06, 1.5450615839479970e-05, 1.0731935426699396e-04,
   8.8294519055560439e-04, 1.3276608390145865e-04, 1.6423510327768317e-06,
   2.5794057350472414e-04, 2.8675549934525132e-04},
  {7.8441998276182750e-04, 9.1528180957759745e-04, 4.6226978079510854e-04,
   7.6617484041782485e-04, 1.0422909419121936e-05, 2.6544636091699370e-02,
   2.8557369473315988e-03, 3.2014451569115343e-04, 1.7486269599932410e-03,
   9.9252658549817602e-05, 6.0406050527793496e-04, 1.1831024043470412e-03,
   1.5335846864532110e-04, 1.7100059826303178e-05, 7.2911060660364247e-04,
   4.4648205428150529e-04, 3.3667241069963721e-04, 4.8464660724673283e-06,
   1.9368757924195069e-05, 2.5436874878957899e-04},
  {1.9549522618591804e-03, 8.6839478291009283e-05, 9.2846484862398504e-04,
   5.4382363415210683e-03, 1.3045150483146400e-05, 2.8557369473315988e-03,
   3.3517661571345295e-02, 1.0164440084156755e-03, 3.0223151556800276e-04,
   2.7356364677374762e-04, 1.8090379077740425e-04, 9.0138008178105558e-04,
   6.7559973423236150e-05, 1.5496057150416647e-05, 3.8539043828203593e-04,
   7.7702860107301409e-04, 3.7793602875380548e-04, 1.6817714350602577e-06,
   8.3785905046764608e-05, 3.5213850363434608e-04},
  {4.4450248999927649e-03, 1.4407449709189928e-04, 1.2584257731785510e-03,
   1.3553903446762449e-03, 1.2331325932473484e-04, 3.2014451569115343e-04,
   1.0164440084156755e-03, 7.3132026674414694e-02, 1.4265572852210342e-04,
   7.8900361203388875e-05, 2.1034403031175677e-04, 6.1289535748612200e-04,
   7.4245325157996467e-05, 1.5674229370359372e-04, 5.4766647831609131e-04,
   3.4088067901566015e-03, 7.0070498686195871e-04, 5.2853781924046885e-06,
   2.4544028062357452e-05, 8.5444888317695144e-04},
  {2.4408739253648561e-04, 8.0713273132343639e-04, 1.5206567357122051e-03,
   4.2748415829840124e-04, 8.7403624792468633e-05, 1.7486269599932410e-03,
   3.0223151556800276e-04, 1.4265572852210342e-04, 2.5912762590464825e-02,
   5.4172593088878169e-05, 3.5367937307928028e-04, 3.2368239575447064e-04,
   1.3814951744043517e-05, 1.7593656715401314e-04, 4.2250645207888871e-04,
   3.0759142376355817e-04, 1.6958020182452686e-04, 2.8647046074837808e-05,
   3.2560382462402542e-04, 2.5015804822114957e-04},
  {6.3834521530456922e-04, 2.3918939892306258e-04, 2.6711601073772237e-04,
   1.3808202761146380e-04, 1.4700821602711548e-04, 9.9252658549817602e-05,
   2.7356364677374762e-04, 7.8900361203388875e-05, 5.4172593088878169e-05,
   2.5353285377410485e-02, 2.0257509751816058e-03, 3.7763773851518320e-04,
   4.3126365297883206e-04, 6.9390890433520820e-04, 9.9551479665834663e-05,
   2.4863350821043576e-04, 9.5951641417652196e-04, 3.5757593419236457e-06,
   1.3084289995157492e-04, 4.6260165509845463e-03},
  {8.5734186469652062e-04, 1.8565923197220291e-04, 3.0916547891024103e-04,
   5.5405232219940619e-05, 1.7204998362221924e-05, 6.0406050527793496e-04,
   1.8090379077740425e-04, 2.1034403031175677e-04, 3.5367937307928028e-04,
   2.0257509751816058e-03, 7.3000355069562095e-02, 4.1727413572248076e-04,
   1.5642375920382058e-03, 1.4111427667651418e-03, 3.9148184818805121e-04,
   3.4625426602777944e-04, 5.1411236346458541e-04, 1.1411927900827321e-04,
   2.4268846460818286e-04, 2.5562624925597006e-03},
  {6.6918201622206063e-04, 3.6581384858227193e-03, 2.2688803763503840e-03,
   8.2915092340941237e-04, 2.0967486284459295e-05, 1.1831024043470412e-03,
   9.0138008178105558e-04, 6.1289535748612200e-04, 3.2368239575447064e-04,
   3.7763773851518320e-04, 4.1727413572248076e-04, 6.4693883189538823e-02,
   6.6794275671927127e-04, 2.7382721623874425e-05, 4.3433200079450959e-04,
   1.4551172602609017e-03, 1.5845617070211647e-03, 1.4791603367426559e-05,
   1.0031590199504370e-04, 2.4096483294034894e-04},
  {2.4336822178217660e-04, 1.4663859770241663e-04, 2.7842591589795833e-05,
   1.5837617592492976e-05, 4.6716406093025683e-06, 1.5335846864532110e-04,
   6.7559973423236150e-05, 7.4245325157996467e-05, 1.3814951744043517e-05,
   4.3126365297883206e-04, 1.5642375920382058e-03, 6.6794275671927127e-04,
   1.0141188024147338e-02, 1.4660178199119104e-04, 4.8450925637158114e-05,
   1.6515644650970699e-04, 2.2685456319283340e-04, 2.0008298266352844e-06,
   9.3883281308262965e-06, 6.0233931141128079e-04},
  {1.9674172887918488e-04, 6.8103183598496806e-05, 8.1379367658642105e-05,
   1.2183210262808245e-05, 1.5450615839479970e-05, 1.7100059826303178e-05,
   1.5496057150416647e-05, 1.5674229370359372e-04, 1.7593656715401314e-04,
   6.9390890433520820e-04, 1.4111427667651418e-03, 2.7382721623874425e-05,
   1.4660178199119104e-04, 3.3860977281888867e-02, 7.1900609504981099e-05,
   3.1517744951616293e-04, 1.1969179294550922e-04, 8.8987284001973936e-05,
   2.1252281905202656e-03, 1.7152669910871525e-04},
  {2.6202120968515422e-03, 5.5932452083258294e-04, 2.9181042514696590e-04,
   1.6175221090296296e-04, 1.0731935426699396e-04, 7.2911060660364247e-04,
   3.8539043828203593e-04, 5.4766647831609131e-04, 4.2250645207888871e-04,
   9.9551479665834663e-05, 3.9148184818805121e-04, 4.3433200079450959e-04,
   4.8450925637158114e-05, 7.1900609504981099e-05, 4.0604206849577246e-02,
   2.0119339905674016e-03, 7.1262984113373655e-04, 4.7982296290983556e-06,
   1.4865662060036108e-05, 4.6067081670332334e-04},
  {5.3063352404797842e-03, 1.0092107276753980e-03, 2.7347990528488269e-03,
   9.3274720682462713e-04, 8.8294519055560439e-04, 4.4648205428150529e-04,
   7.7702860107301409e-04, 3.4088067901566015e-03, 3.0759142376355817e-04,
   2.4863350821043576e-04, 3.4625426602777944e-04, 1.4551172602609017e-03,
   1.6515644650970699e-04, 3.1517744951616293e-04, 2.0119339905674016e-03,
   4.3704313725631293e-02, 4.6622796190467931e-03, 1.3084508692320411e-04,
   2.0094952100191171e-04, 5.3046884366373238e-04},
  {4.2185799566371992e-03, 2.5480695878483489e-04, 1.2577236167229159e-03,
   5.4799551604074552e-04, 1.3276608390145865e-04, 3.3667241069963721e-04,
   3.7793602875380548e-04, 7.0070498686195871e-04, 1.6958020182452686e-04,
   9.5951641417652196e-04, 5.1411236346458541e-04, 1.5845617070211647e-03,
   2.2685456319283340e-04, 1.1969179294550922e-04, 7.1262984113373655e-04,
   4.6622796190467931e-03, 4.0080501117597984e-02, 7.7221567079978049e-06,
   1.9614260534399206e-04, 1.4808862501356529e-03},
  {8.9313082115936708e-06, 2.2193510725567831e-04, 2.6993374401853017e-05,
   2.6698310744553494e-06, 1.6423510327768317e-06, 4.8464660724673283e-06,
   1.6817714350602577e-06, 5.2853781924046885e-06, 2.8647046074837808e-05,
   3.5757593419236457e-06, 1.1411927900827321e-04, 1.4791603367426559e-05,
   2.0008298266352844e-06, 8.8987284001973936e-05, 4.7982296290983556e-06,
   1.3084508692320411e-04, 7.7221567079978049e-06, 9.7663430469613309e-03,
   5.4428834502404551e-05, 3.4341494426087616e-06},
  {1.8225253115472477e-04, 3.6904551277914591e-05, 2.6944461441064744e-04,
   2.7630471952142814e-05, 2.5794057350472414e-04, 1.9368757924195069e-05,
   8.3785905046764608e-05, 2.4544028062357452e-05, 3.2560382462402542e-04,
   1.3084289995157492e-04, 2.4268846460818286e-04, 1.0031590199504370e-04,
   9.3883281308262965e-06, 2.1252281905202656e-03, 1.4865662060036108e-05,
   2.0094952100191171e-04, 1.9614260534399206e-04, 5.4428834502404551e-05,
   2.5458081876085920e-02, 1.5581070490829142e-04},
  {2.6924708725231086e-03, 1.9661468480294445e-04, 1.8273086910009501e-04,
   1.9652464806266956e-04, 2.8675549934525132e-04, 2.5436874878957899e-04,
   3.5213850363434608e-04, 8.5444888317695144e-04, 2.5015804822114957e-04,
   4.6260165509845463e-03, 2.5562624925597006e-03, 2.4096483294034894e-04,
   6.0233931141128079e-04, 1.7152669910871525e-04, 4.6067081670332334e-04,
   5.3046884366373238e-04, 1.4808862501356529e-03, 3.4341494426087616e-06,
   1.5581070490829142e-04, 4.8623010168479329e-02}};


/** Background frequencies for PAM30 */
static const double
PAM30_bg[COMPO_NUM_TRUE_AA] =
 {8.7126757718998679e-02, 4.0903890307999048e-02, 4.0432285538126674e-02,
  4.6871932807633972e-02, 3.3473507356366124e-02, 3.8255436978704142e-02,
  4.9530476921569762e-02, 8.8612083613937145e-02, 3.3618414314618796e-02,
  3.6885613388971901e-02, 8.5357443758733623e-02, 8.0481583375956817e-02,
  1.4752761600830055e-02, 3.9771658566274730e-02, 5.0679914836743105e-02,
  6.9577076005018168e-02, 5.8541664190993777e-02, 1.0493678893463984e-02,
  2.9916218247065893e-02, 6.4717601577993583e-02};


/** Joint probabilities for PAM70 */
static const double
PAM70_JOINT_PROBS[COMPO_NUM_TRUE_AA][COMPO_NUM_TRUE_AA] =
 {{3.7194198649883180e-02, 8.0875430987034625e-04, 2.0736464600905839e-03,
   2.5255595007111849e-03, 7.0597624705127243e-04, 1.5518877282373696e-03,
   3.3281196505185301e-03, 7.8361919930011979e-03, 6.4846822847107975e-04,
   1.3965782283733507e-03, 1.8572709391302328e-03, 1.7482550126124556e-03,
   4.8307675706277730e-04, 4.6722180304288895e-04, 4.5500317017840861e-03,
   8.1815610080975230e-03, 6.7687959354540955e-03, 3.9553416320695907e-05,
   4.0198038575772689e-04, 4.5596297635283793e-03},
  {8.0875430987034625e-04, 2.2852256980208891e-02, 6.3934638630944640e-04,
   2.6565526087426409e-04, 2.0716330531302742e-04, 1.6017418992100730e-03,
   3.6247967443696756e-04, 4.4251315906877157e-04, 1.4545996303067922e-03,
   4.5486189188160571e-04, 4.7958376254792433e-04, 6.3755221317218641e-03,
   2.9453018991274540e-04, 1.6164674287424461e-04, 1.0920151094434257e-03,
   1.7360744550429996e-03, 6.7924379689915479e-04, 4.2527485122100103e-04,
   1.0517171198521535e-04, 4.6545505887026243e-04},
  {2.0736464600905839e-03, 6.3934638630944640e-04, 1.2675260298758369e-02,
   4.5473263538386658e-03, 1.3072655726453243e-04, 9.8476934026016437e-04,
   1.9319282404800279e-03, 2.3638339620501124e-03, 2.2789366146196482e-03,
   4.8031915042001503e-04, 6.3784180585582998e-04, 3.6406443267426908e-03,
   1.0839896462648394e-04, 2.1124510348244946e-04, 7.0593648786013011e-04,
   3.8434299995336323e-03, 2.1278266383167998e-03, 5.8506014024617153e-05,
   4.7128194649955374e-04, 5.2108088709319821e-04},
  {2.5255595007111849e-03, 2.6565526087426409e-04, 4.5473263538386658e-03,
   1.9552340613266873e-02, 6.2308454328698375e-05, 1.6541837501367655e-03,
   7.7926300406749079e-03, 2.6556442405114718e-03, 9.4505604487949357e-04,
   3.3373212803700417e-04, 2.5428746885488489e-04, 1.8252265778963348e-03,
   7.1571301890192352e-05, 6.2699722593189186e-05, 5.0522979322628484e-04,
   1.9557928207999562e-03, 1.2131330526185470e-03, 1.2993511235541354e-05,
   1.1201266335522796e-04, 5.2454950790431021e-04},
  {7.0597624705127243e-04, 2.0716330531302742e-04, 1.3072655726453243e-04,
   6.2308454328698375e-05, 2.7825141122194648e-02, 5.0435716826920468e-05,
   6.4183020849751415e-05, 3.4312217142660280e-04, 1.8549298854202793e-04,
   3.0428065563695795e-04, 8.6272008280351356e-05, 1.0319775490385661e-04,
   2.1845897888507003e-05, 7.3005996818964865e-05, 2.6943290953722234e-04,
   1.5492275564647287e-03, 3.6530353715738469e-04, 7.8068100250333932e-06,
   5.2405357404561922e-04, 5.9453107180990322e-04},
  {1.5518877282373696e-03, 1.6017418992100730e-03, 9.8476934026016437e-04,
   1.6541837501367655e-03, 5.0435716826920468e-05, 1.6787102517888230e-02,
   4.2787747314313521e-03, 8.0400665120515828e-04, 2.8143703113243106e-03,
   2.6711588850824816e-04, 1.1454480615912710e-03, 2.2219191879465955e-03,
   2.7431052994878819e-04, 7.9884925062964878e-05, 1.3266385181226064e-03,
   9.9419711566964102e-04, 7.3752923373982926e-04, 2.2068032648929309e-05,
   8.4534917138966498e-05, 5.7451792180619309e-04},
  {3.3281196505185301e-03, 3.6247967443696756e-04, 1.9319282404800279e-03,
   7.7926300406749079e-03, 6.4183020849751415e-05, 4.2787747314313521e-03,
   2.1230224844830631e-02, 2.1586929726556546e-03, 7.9951624929977820e-04,
   5.1817013748561565e-04, 4.9671333905053393e-04, 1.8583368487577027e-03,
   1.5522247207982098e-04, 7.5329038569461125e-05, 8.7224608153784204e-04,
   1.6494397215528771e-03, 9.8039692363547162e-04, 9.8627013698496747e-06,
   1.7654014099528279e-04, 7.9167009135776051e-04},
  {7.8361919930011979e-03, 4.4251315906877157e-04, 2.3638339620501124e-03,
   2.6556442405114718e-03, 3.4312217142660280e-04, 8.0400665120515828e-04,
   2.1586929726556546e-03, 5.7388208411163856e-02, 4.3236801169914066e-04,
   3.4974476371122900e-04, 6.1249836569746629e-04, 1.4966553087392409e-03,
   1.8860818518822553e-04, 3.5290430601106194e-04, 1.3849768155457195e-03,
   5.9445746107512484e-03, 1.9215787027991949e-03, 2.5013133615378750e-05,
   1.1604677012097751e-04, 1.7949010789751496e-03},
  {6.4846822847107975e-04, 1.4545996303067922e-03, 2.2789366146196482e-03,
   9.4505604487949357e-04, 1.8549298854202793e-04, 2.8143703113243106e-03,
   7.9951624929977820e-04, 4.3236801169914066e-04, 1.8560540662079211e-02,
   1.7329804084653826e-04, 7.4343007070569922e-04, 9.0662732602859596e-04,
   6.0703685726216408e-05, 3.6980044394832400e-04, 8.3973781481707591e-04,
   7.6422387310404486e-04, 4.5318311225648759e-04, 6.5639640674803356e-05,
   6.1489346188958391e-04, 5.0752810340000398e-04},
  {1.3965782283733507e-03, 4.5486189188160571e-04, 4.8031915042001503e-04,
   3.3373212803700417e-04, 3.0428065563695795e-04, 2.6711588850824816e-04,
   5.1817013748561565e-04, 3.4974476371122900e-04, 1.7329804084653826e-04,
   1.5987197880539435e-02, 3.7542109440494597e-03, 7.9047697379200301e-04,
   7.4501029384275877e-04, 1.2410414311848890e-03, 2.9830750222722838e-04,
   6.5095629385579479e-04, 1.6435015047341363e-03, 1.6913577722248789e-05,
   3.0951431979519975e-04, 7.1703817823281675e-03},
  {1.8572709391302328e-03, 4.7958376254792433e-04, 6.3784180585582998e-04,
   2.5428746885488489e-04, 8.6272008280351356e-05, 1.1454480615912710e-03,
   4.9671333905053393e-04, 6.1249836569746629e-04, 7.4343007070569922e-04,
   3.7542109440494597e-03, 5.9951531802758418e-02, 1.0493781626637116e-03,
   2.6814583560349986e-03, 2.7966586120273659e-03, 8.5989610618916832e-04,
   8.8237114340186434e-04, 1.2009021086866177e-03, 2.3930650357563717e-04,
   6.1025491368789397e-04, 5.0181292839441248e-03},
  {1.7482550126124556e-03, 6.3755221317218641e-03, 3.6406443267426908e-03,
   1.8252265778963348e-03, 1.0319775490385661e-04, 2.2219191879465955e-03,
   1.8583368487577027e-03, 1.4966553087392409e-03, 9.0662732602859596e-04,
   7.9047697379200301e-04, 1.0493781626637116e-03, 4.9293839020253936e-02,
   1.1255105312350092e-03, 1.3216275164777875e-04, 1.0446567434093407e-03,
   2.9144037757683188e-03, 2.9220724161787321e-03, 6.6880694143482477e-05,
   2.4262835127628017e-04, 7.2318948023856685e-04},
  {4.8307675706277730e-04, 2.9453018991274540e-04, 1.0839896462648394e-04,
   7.1571301890192352e-05, 2.1845897888507003e-05, 2.7431052994878819e-04,
   1.5522247207982098e-04, 1.8860818518822553e-04, 6.0703685726216408e-05,
   7.4501029384275877e-04, 2.6814583560349986e-03, 1.1255105312350092e-03,
   6.2216058278953433e-03, 2.9832445346669831e-04, 1.3153451000560772e-04,
   3.2741424213960572e-04, 4.2984024762048522e-04, 9.2547509155780498e-06,
   4.2944764664016277e-05, 1.0815956386861806e-03},
  {4.6722180304288895e-04, 1.6164674287424461e-04, 2.1124510348244946e-04,
   6.2699722593189186e-05, 7.3005996818964865e-05, 7.9884925062964878e-05,
   7.5329038569461125e-05, 3.5290430601106194e-04, 3.6980044394832400e-04,
   1.2410414311848890e-03, 2.7966586120273659e-03, 1.3216275164777875e-04,
   2.9832445346669831e-04, 2.7609457395367236e-02, 1.7973871152134819e-04,
   5.9661742408822192e-04, 3.2037468566685034e-04, 1.8874366992597410e-04,
   4.0197936157261114e-03, 5.3500773324884835e-04},
  {4.5500317017840861e-03, 1.0920151094434257e-03, 7.0593648786013011e-04,
   5.0522979322628484e-04, 2.6943290953722234e-04, 1.3266385181226064e-03,
   8.7224608153784204e-04, 1.3849768155457195e-03, 8.3973781481707591e-04,
   2.9830750222722838e-04, 8.5989610618916832e-04, 1.0446567434093407e-03,
   1.3153451000560772e-04, 1.7973871152134819e-04, 3.0491164657684509e-02,
   3.4531398828496866e-03, 1.5743611594360388e-03, 2.2239711690711994e-05,
   7.0790644298591358e-05, 1.0078399755563739e-03},
  {8.1815610080975230e-03, 1.7360744550429996e-03, 3.8434299995336323e-03,
   1.9557928207999562e-03, 1.5492275564647287e-03, 9.9419711566964102e-04,
   1.6494397215528771e-03, 5.9445746107512484e-03, 7.6422387310404486e-04,
   6.5095629385579479e-04, 8.8237114340186434e-04, 2.9144037757683188e-03,
   3.2741424213960572e-04, 5.9661742408822192e-04, 3.4531398828496866e-03,
   2.5194623995400022e-02, 6.9036292775662960e-03, 2.3646234303355702e-04,
   4.3310777955182051e-04, 1.3658286863463751e-03},
  {6.7687959354540955e-03, 6.7924379689915479e-04, 2.1278266383167998e-03,
   1.2131330526185470e-03, 3.6530353715738469e-04, 7.3752923373982926e-04,
   9.8039692363547162e-04, 1.9215787027991949e-03, 4.5318311225648759e-04,
   1.6435015047341363e-03, 1.2009021086866177e-03, 2.9220724161787321e-03,
   4.2984024762048522e-04, 3.2037468566685034e-04, 1.5743611594360388e-03,
   6.9036292775662960e-03, 2.5145073963504398e-02, 3.3504725726214485e-05,
   3.9901555652894171e-04, 2.7223976124681726e-03},
  {3.9553416320695907e-05, 4.2527485122100103e-04, 5.8506014024617153e-05,
   1.2993511235541354e-05, 7.8068100250333932e-06, 2.2068032648929309e-05,
   9.8627013698496747e-06, 2.5013133615378750e-05, 6.5639640674803356e-05,
   1.6913577722248789e-05, 2.3930650357563717e-04, 6.6880694143482477e-05,
   9.2547509155780498e-06, 1.8874366992597410e-04, 2.2239711690711994e-05,
   2.3646234303355702e-04, 3.3504725726214485e-05, 8.8772945001325369e-03,
   1.1825556331632613e-04, 1.8104742145890378e-05},
  {4.0198038575772689e-04, 1.0517171198521535e-04, 4.7128194649955374e-04,
   1.1201266335522796e-04, 5.2405357404561922e-04, 8.4534917138966498e-05,
   1.7654014099528279e-04, 1.1604677012097751e-04, 6.1489346188958391e-04,
   3.0951431979519975e-04, 6.1025491368789397e-04, 2.4262835127628017e-04,
   4.2944764664016277e-05, 4.0197936157261114e-03, 7.0790644298591358e-05,
   4.3310777955182051e-04, 3.9901555652894171e-04, 1.1825556331632613e-04,
   2.0713264962639342e-02, 3.5013220379328771e-04},
  {4.5596297635283793e-03, 4.6545505887026243e-04, 5.2108088709319821e-04,
   5.2454950790431021e-04, 5.9453107180990322e-04, 5.7451792180619309e-04,
   7.9167009135776051e-04, 1.7949010789751496e-03, 5.0752810340000398e-04,
   7.1703817823281675e-03, 5.0181292839441248e-03, 7.2318948023856685e-04,
   1.0815956386861806e-03, 5.3500773324884835e-04, 1.0078399755563739e-03,
   1.3658286863463751e-03, 2.7223976124681726e-03, 1.8104742145890378e-05,
   3.5013220379328771e-04, 3.4391130954492435e-02}};


/** Background frequencies for PAM70 */
static const double
PAM70_bg[COMPO_NUM_TRUE_AA] =
 {8.7126757718998679e-02, 4.0903890307999048e-02, 4.0432285538126674e-02,
  4.6871932807633972e-02, 3.3473507356366124e-02, 3.8255436978704142e-02,
  4.9530476921569762e-02, 8.8612083613937145e-02, 3.3618414314618796e-02,
  3.6885613388971901e-02, 8.5357443758733623e-02, 8.0481583375956817e-02,
  1.4752761600830055e-02, 3.9771658566274730e-02, 5.0679914836743105e-02,
  6.9577076005018168e-02, 5.8541664190993777e-02, 1.0493678893463984e-02,
  2.9916218247065893e-02, 6.4717601577993583e-02};


/** Joint probabilities for PAM250 */
static const double
PAM250_JOINT_PROBS[COMPO_NUM_TRUE_AA][COMPO_NUM_TRUE_AA] =
 {{1.1506395421657883e-02, 2.4975231508904631e-03, 3.6632346972757428e-03,
   4.3697565048539024e-03, 1.8276942838570587e-03, 3.0112866662942396e-03,
   4.6380059574065739e-03, 1.0339526813131699e-02, 2.1445971886243646e-03,
   2.8561290389349852e-03, 4.8067900424585462e-03, 5.3728656072800076e-03,
   9.9018955937298613e-04, 1.5464751871669552e-03, 5.7141459312891330e-03,
   7.8179771536648658e-03, 6.7187128037285646e-03, 2.4101326295153331e-04,
   1.1721100314806740e-03, 5.8923284166788933e-03},
  {2.4975231508904631e-03, 6.8223547287428144e-03, 1.6533152696263029e-03,
   1.4242726301819985e-03, 5.8944515652700497e-04, 2.0890675656647919e-03,
   1.5846263196146995e-03, 2.0115937172949889e-03, 1.9669335302920869e-03,
   9.4903575242628803e-04, 1.7492451507985533e-03, 7.1821276038123629e-03,
   5.4579896329619748e-04, 5.8361963932467844e-04, 1.9905178391619333e-03,
   2.6418797202180837e-03, 1.9640564224658449e-03, 7.0871113466232773e-04,
   4.6349536482262227e-04, 1.4862706481749135e-03},
  {3.6632346972757428e-03, 1.6533152696263029e-03, 2.5955847457262279e-03,
   3.0517594529028400e-03, 5.8568302165395883e-04, 1.8510873859350727e-03,
   2.7895276558308215e-03, 3.8931580947872417e-03, 1.9571964875229838e-03,
   9.8420613349048088e-04, 1.7794215836961103e-03, 4.0932232786158613e-03,
   3.9994286569269551e-04, 7.1880113475369782e-04, 1.8301977419577379e-03,
   3.2997070393949880e-03, 2.6196442519984238e-03, 1.6728058239964415e-04,
   7.5085733385690172e-04, 1.7484567810093011e-03},
  {4.3697565048539024e-03, 1.4242726301819985e-03, 3.0517594529028400e-03,
   5.3450782499419218e-03, 4.8131585819575790e-04, 2.6206774727375219e-03,
   5.1008808895958635e-03, 4.7650879451660638e-03, 1.8490514559930820e-03,
   1.0026061088056078e-03, 1.5930790839216963e-03, 3.8473023950370976e-03,
   3.7899432160262535e-04, 5.1061942897298563e-04, 1.9136209919861855e-03,
   3.4857847560291560e-03, 2.6547044011729533e-03, 1.0462445958257217e-04,
   5.1745954208780515e-04, 1.8552568588661234e-03},
  {1.8276942838570587e-03, 5.8944515652700497e-04, 5.8568302165395883e-04,
   4.8131585819575790e-04, 1.7511424258914964e-02, 3.7076129096329572e-04,
   4.8903939170975950e-04, 1.3623489692838801e-03, 5.0846386896381476e-04,
   7.2550893644428554e-04, 7.0799236631215092e-04, 7.7075174636654406e-04,
   1.4785573890415589e-04, 4.9387975479971797e-04, 8.9473603104580409e-04,
   2.3049495795083078e-03, 1.1784720532548663e-03, 5.9299054281549242e-05,
   1.0798150304862418e-03, 1.3840709648926645e-03},
  {3.0112866662942396e-03, 2.0890675656647919e-03, 1.8510873859350727e-03,
   2.6206774727375219e-03, 3.7076129096329572e-04, 3.7185262078390084e-03,
   3.3568227763936856e-03, 2.5676676841601132e-03, 2.5232383252931120e-03,
   8.8236895610109610e-04, 2.1731391610102509e-03, 3.6449762674855470e-03,
   4.4907751867065491e-04, 5.2205033209354096e-04, 2.0463950617354261e-03,
   2.3688704455266710e-03, 1.8654198667353624e-03, 1.3455315022844615e-04,
   4.5154514263611947e-04, 1.6079057012005865e-03},
  {4.6380059574065739e-03, 1.5846263196146995e-03, 2.7895276558308215e-03,
   5.1008808895958635e-03, 4.8903939170975950e-04, 3.3568227763936856e-03,
   5.9222965859089486e-03, 4.5816393201343689e-03, 1.9402998565554576e-03,
   1.1442983475073972e-03, 1.9598706476253135e-03, 3.9374858628132975e-03,
   4.4836711718981544e-04, 5.6555863311724848e-04, 2.2081467402101113e-03,
   3.4413512797094705e-03, 2.6498050070285850e-03, 1.0428284199341159e-04,
   5.5113323096536227e-04, 2.1170384602596944e-03},
  {1.0339526813131699e-02, 2.0115937172949889e-03, 3.8931580947872417e-03,
   4.7650879451660638e-03, 1.3623489692838801e-03, 2.5676676841601132e-03,
   4.5816393201343689e-03, 2.3482233614097934e-02, 1.8337281555648979e-03,
   1.8208983421724295e-03, 2.9852908206479617e-03, 4.8388854328488646e-03,
   6.8680090301959161e-04, 1.1712470257125083e-03, 4.0151722887942186e-03,
   7.8821587248468251e-03, 5.1811218591430661e-03, 1.8631167134196649e-04,
   7.9425725644547004e-04, 4.2129549753424219e-03},
  {2.1445971886243646e-03, 1.9669335302920869e-03, 1.9571964875229838e-03,
   1.8490514559930820e-03, 5.0846386896381476e-04, 2.5232383252931120e-03,
   1.9402998565554576e-03, 1.8337281555648979e-03, 5.0577823911620960e-03,
   7.0758217809190201e-04, 1.7792986667457807e-03, 2.6790420483436923e-03,
   3.0226127635083285e-04, 8.8185158980207221e-04, 1.6125778990215547e-03,
   1.9419581808088138e-03, 1.4578210132932265e-03, 1.9201336290789017e-04,
   9.8449170102575230e-04, 1.2982251382555629e-03},
  {2.8561290389349852e-03, 9.4903575242628803e-04, 9.8420613349048088e-04,
   1.0026061088056078e-03, 7.2550893644428554e-04, 8.8236895610109610e-04,
   1.1442983475073972e-03, 1.8208983421724295e-03, 7.0758217809190201e-04,
   3.8506568985748504e-03, 5.5072715793686080e-03, 1.9022872797907308e-03,
   8.9779553718262028e-04, 1.8514043008626233e-03, 1.1725443795985299e-03,
   1.8542542117578613e-03, 2.1953086037143862e-03, 1.1715327832769289e-04,
   8.8263202943021396e-04, 5.5816714963892811e-03},
  {4.8067900424585462e-03, 1.7492451507985533e-03, 1.7794215836961103e-03,
   1.5930790839216963e-03, 7.0799236631215092e-04, 2.1731391610102509e-03,
   1.9598706476253135e-03, 2.9852908206479617e-03, 1.7792986667457807e-03,
   5.5072715793686080e-03, 2.8633158451940706e-02, 3.5641037178981254e-03,
   2.9437898388951153e-03, 5.1517088250987598e-03, 2.4053702086781481e-03,
   3.1126304674457994e-03, 3.3831513688247950e-03, 5.7942468517701131e-04,
   2.0809174228785228e-03, 8.4617896693113381e-03},
  {5.3728656072800076e-03, 7.1821276038123629e-03, 4.0932232786158613e-03,
   3.8473023950370976e-03, 7.7075174636654406e-04, 3.6449762674855470e-03,
   3.9374858628132975e-03, 4.8388854328488646e-03, 2.6790420483436923e-03,
   1.9022872797907308e-03, 3.5641037178981254e-03, 1.8953839887039359e-02,
   1.3056068288886324e-03, 9.5430576257042143e-04, 3.1427838314461703e-03,
   5.3849029169230475e-03, 4.6910200930370578e-03, 3.8031958293778897e-04,
   8.6636282903806235e-04, 2.9693904037835185e-03},
  {9.9018955937298613e-04, 5.4579896329619748e-04, 3.9994286569269551e-04,
   3.7899432160262535e-04, 1.4785573890415589e-04, 4.4907751867065491e-04,
   4.4836711718981544e-04, 6.8680090301959161e-04, 3.0226127635083285e-04,
   8.9779553718262028e-04, 2.9437898388951153e-03, 1.3056068288886324e-03,
   9.5839891336186689e-04, 6.1182400928263853e-04, 4.6431155948741784e-04,
   7.1706174187243107e-04, 7.5402578765302252e-04, 5.8044986217193111e-05,
   2.5054194683598869e-04, 1.4420721870535345e-03},
  {1.5464751871669552e-03, 5.8361963932467844e-04, 7.1880113475369782e-04,
   5.1061942897298563e-04, 4.9387975479971797e-04, 5.2205033209354096e-04,
   5.6555863311724848e-04, 1.1712470257125083e-03, 8.8185158980207221e-04,
   1.8514043008626233e-03, 5.1517088250987598e-03, 9.5430576257042143e-04,
   6.1182400928263853e-04, 1.2698002251905573e-02, 7.0713747400886877e-04,
   1.3236969321879562e-03, 1.1372790252379261e-03, 4.4935713151489045e-04,
   5.9204171542750639e-03, 1.9724229735869326e-03},
  {5.7141459312891330e-03, 1.9905178391619333e-03, 1.8301977419577379e-03,
   1.9136209919861855e-03, 8.9473603104580409e-04, 2.0463950617354261e-03,
   2.2081467402101113e-03, 4.0151722887942186e-03, 1.6125778990215547e-03,
   1.1725443795985299e-03, 2.4053702086781481e-03, 3.1427838314461703e-03,
   4.6431155948741784e-04, 7.0713747400886877e-04, 9.8665046135635783e-03,
   4.3847634418626652e-03, 3.1909511773661765e-03, 1.4643278579496693e-04,
   4.8653577087297381e-04, 2.4870690688613543e-03},
  {7.8179771536648658e-03, 2.6418797202180837e-03, 3.2997070393949880e-03,
   3.4857847560291560e-03, 2.3049495795083078e-03, 2.3688704455266710e-03,
   3.4413512797094705e-03, 7.8821587248468251e-03, 1.9419581808088138e-03,
   1.8542542117578613e-03, 3.1126304674457994e-03, 5.3849029169230475e-03,
   7.1706174187243107e-04, 1.3236969321879562e-03, 4.3847634418626652e-03,
   6.9740330873263399e-03, 5.5479545040527179e-03, 4.1290844979042010e-04,
   1.0821257035278605e-03, 3.5981076685639816e-03},
  {6.7187128037285646e-03, 1.9640564224658449e-03, 2.6196442519984238e-03,
   2.6547044011729533e-03, 1.1784720532548663e-03, 1.8654198667353624e-03,
   2.6498050070285850e-03, 5.1811218591430661e-03, 1.4578210132932265e-03,
   2.1953086037143862e-03, 3.3831513688247950e-03, 4.6910200930370578e-03,
   7.5402578765302252e-04, 1.1372790252379261e-03, 3.1909511773661765e-03,
   5.5479545040527179e-03, 6.1907639011039494e-03, 1.8788403040115726e-04,
   9.2861947637862599e-04, 4.0449485444032200e-03},
  {2.4101326295153331e-04, 7.0871113466232773e-04, 1.6728058239964415e-04,
   1.0462445958257217e-04, 5.9299054281549242e-05, 1.3455315022844615e-04,
   1.0428284199341159e-04, 1.8631167134196649e-04, 1.9201336290789017e-04,
   1.1715327832769289e-04, 5.7942468517701131e-04, 3.8031958293778897e-04,
   5.8044986217193111e-05, 4.4935713151489045e-04, 1.4643278579496693e-04,
   4.1290844979042010e-04, 1.8788403040115726e-04, 5.7988503825631863e-03,
   3.0440102514689768e-04, 1.6081303524350391e-04},
  {1.1721100314806740e-03, 4.6349536482262227e-04, 7.5085733385690172e-04,
   5.1745954208780515e-04, 1.0798150304862418e-03, 4.5154514263611947e-04,
   5.5113323096536227e-04, 7.9425725644547004e-04, 9.8449170102575230e-04,
   8.8263202943021396e-04, 2.0809174228785228e-03, 8.6636282903806235e-04,
   2.5054194683598869e-04, 5.9204171542750639e-03, 4.8653577087297381e-04,
   1.0821257035278605e-03, 9.2861947637862599e-04, 3.0440102514689768e-04,
   9.2529814445245183e-03, 1.0955188103504248e-03},
  {5.8923284166788933e-03, 1.4862706481749135e-03, 1.7484567810093011e-03,
   1.8552568588661234e-03, 1.3840709648926645e-03, 1.6079057012005865e-03,
   2.1170384602596944e-03, 4.2129549753424219e-03, 1.2982251382555629e-03,
   5.5816714963892811e-03, 8.4617896693113381e-03, 2.9693904037835185e-03,
   1.4420721870535345e-03, 1.9724229735869326e-03, 2.4870690688613543e-03,
   3.5981076685639816e-03, 4.0449485444032200e-03, 1.6081303524350391e-04,
   1.0955188103504248e-03, 1.1301289775766317e-02}};


/** Background frequencies for PAM250 */
static const double
PAM250_bg[COMPO_NUM_TRUE_AA] =
 {8.7126757718998679e-02, 4.0903890307999048e-02, 4.0432285538126674e-02,
  4.6871932807633972e-02, 3.3473507356366124e-02, 3.8255436978704142e-02,
  4.9530476921569762e-02, 8.8612083613937145e-02, 3.3618414314618796e-02,
  3.6885613388971901e-02, 8.5357443758733623e-02, 8.0481583375956817e-02,
  1.4752761600830055e-02, 3.9771658566274730e-02, 5.0679914836743105e-02,
  6.9577076005018168e-02, 5.8541664190993777e-02, 1.0493678893463984e-02,
  2.9916218247065893e-02, 6.4717601577993583e-02};


/** The structure used to store frequency data; used only in
    matrix_frequency_data.c . */
typedef struct Compo_FrequencyData {
    const char * name;   /**< name of the matrix */
    const double (*joint_probs)[COMPO_NUM_TRUE_AA]; /**< joint probabilities */
    const double * background; /**< background frequencies */
} Compo_FrequencyData;

/** The static frequency data for matrix MAT
 *  
 *  We use a macro here to minimize the chance that a typo will lead
 *  to the use of inconsistent matrix data */
#define MATRIX_DATA(MAT) \
    { #MAT, MAT##_JOINT_PROBS, MAT##_bg }

/** Number of matrices for which frequency data is available */
#define NUM_SUPPORTED_MATRICES 8
/** An array available frequency data */
static Compo_FrequencyData
s_FrequencyData[NUM_SUPPORTED_MATRICES] =
    {
        MATRIX_DATA(BLOSUM62),
        MATRIX_DATA(BLOSUM80),
        MATRIX_DATA(BLOSUM50),
        MATRIX_DATA(BLOSUM90),
        MATRIX_DATA(BLOSUM45),
        MATRIX_DATA(PAM30),
        MATRIX_DATA(PAM70),
        MATRIX_DATA(PAM250)
    };


/** Given a standard matrix name, locate the static frequency data
 * for the matrix */
static Compo_FrequencyData *
s_LocateFrequencyData(const char * matrix) 
{
    int i;
    for (i = 0;  i < NUM_SUPPORTED_MATRICES;  i++) {
        if (0 == strcmp(s_FrequencyData[i].name, matrix)) {
            return &s_FrequencyData[i];
        }
    }
    return NULL;
}


/* Documented in matrix_frequency_data.h. */
int Blast_FrequencyDataIsAvailable(const char *matrix_name)
{
    return NULL != s_LocateFrequencyData(matrix_name);
}


/* Documented in matrix_frequency_data.h. */
const double *
Blast_GetMatrixBackgroundFreq(const char *matrix_name)
{
    Compo_FrequencyData * data = s_LocateFrequencyData(matrix_name);
    if (NULL != data) {
        return data->background;
    } else {
        fprintf(stderr, "matrix %s is not supported "
                "for RE based adjustment\n", matrix_name);
        return NULL;
    }
}


/* Documented in matrix_frequency_data.h. */
int
Blast_GetJointProbsForMatrix(double ** probs, double row_sums[],
                             double col_sums[], const char *matrix_name)
{
    int i, j;
    Compo_FrequencyData * data = s_LocateFrequencyData(matrix_name);
    if (NULL == data) {
        fprintf(stderr, "matrix %s is not supported "
                "for RE based adjustment\n", matrix_name);
        return -1;
    }
    for (j = 0;  j < COMPO_NUM_TRUE_AA;  j++) {
        col_sums[j] = 0.0;
    }
    for (i = 0;  i < COMPO_NUM_TRUE_AA;  i++) {
        row_sums[i] = 0.0;
        for (j = 0;  j < COMPO_NUM_TRUE_AA;  j++) {
            probs[i][j] = data->joint_probs[i][j];
            row_sums[i] += probs[i][j];
            col_sums[j] += probs[i][j];
        }
    }
    return 0;
}
