/*
 * NASPRO - The NASPRO Architecture for Sound Processing
 * Core library
 *
 * Copyright (C) 2007-2011 NASPRO core development team
 *
 * See the COPYING file for license conditions.
 */

#include "internal.h"

#define MAX(a, b)	(((a) > (b)) ? (a) : (b))
#define MIN(a, b)	(((a) < (b)) ? (a) : (b))

_NACORE_DEF size_t
nacore_char_utf8_encode(char *utf8c, uint32_t cp)
{
	uint8_t *c;

	if ((cp > 0x10ffff) || ((cp >= 0xd800) && (cp <= 0xdfff))
	    || (cp == 0xfeff) || (cp == 0xfffe))
		return 0;

	c = (uint8_t *)utf8c;

	if (cp < 0x80)
	  {
		if (c != NULL)
			c[0] = cp & 0xff;
		return 1;
	  }
	if (cp < 0x800)
	  {
		if (c != NULL)
		  {
			c[0] = (cp >> 6) | 0xc0;
			c[1] = (cp & 0x3f) | 0x80;
		  }
		return 2;
	  }
	if (cp < 0x10000)
	  {
		if (c != NULL)
		  {
			c[0] = (cp >> 12) | 0xe0;
			c[1] = ((cp >> 6) & 0x3f) | 0x80;
			c[2] = (cp & 0x3f) | 0x80;
		  }
		return 3;
	  }

	if (c != NULL)
	  {
		c[0] = (cp >> 18) | 0xf0;
		c[1] = ((cp >> 12) & 0x3f) | 0x80;
		c[2] = ((cp >> 6) & 0x3f) | 0x80;
		c[3] = (cp & 0x3f) | 0x80;
	  }
	return 4;
}

_NACORE_DEF size_t
nacore_char_utf8_decode(const char *utf8c, uint32_t *cp)
{
	const uint8_t *c;

	c = (const uint8_t *)utf8c;

	if (c[0] < 0x80)			/* one byte */
	  {
		if (cp != NULL)
			*cp = c[0];
		return 1;
	  }

	if ((c[0] & 0xe0) == 0xc0)		/* two bytes */
	  {
		if ((c[1] & 0xc0) != 0x80)
			return 0;
		if (cp != NULL)
			*cp = ((c[0] & 0x1f) << 6) | (c[1] & 0x3f);
		return 2;
	  }
	if ((c[0] & 0xf0) == 0xe0)		/* three bytes */
	  {
		/* this does also reject surrogates (U+D800 to U+DFFF), BOM
		 * (U+FFEF) and inverse BOM (U+FFFE) */
		if ((((c[1] & 0xc0) != 0x80) || ((c[2] & 0xc0) != 0x80))
		    || ((c[0] == 0xed) && (c[1] >= 0xa0))
		    || ((c[0] == 0xef) && (c[1] == 0xbb) && (c[2] == 0xbf))
		    || ((c[0] == 0xef) && (c[1] == 0xbf) && (c[2] == 0xbe)))
			return 0;
		if (cp != NULL)
			*cp = ((c[0] & 0x0f) << 12) | ((c[1] & 0x3f) << 6)
			      | (c[2] & 0x3f);
		return 3;
	  }
	if ((c[0] & 0xf8) == 0xf0)		/* four bytes */
	  {
		if (((c[1] & 0xc0) != 0x80) || ((c[2] & 0xc0) != 0x80)
		    || ((c[3] & 0xc0) != 0x80))
			return 0;
		if (cp != NULL)
			*cp = ((c[0] & 0x07) << 18) | ((c[1] & 0x3f) << 12)
			      | ((c[2] & 0x3f) << 6) | (c[3] & 0x3f);
	  }

	return 0;
}

_NACORE_DEF size_t
nacore_char_utf16le_encode(char *utf16lec, uint32_t cp)
{
	uint8_t *c;

	if ((cp > 0x10ffff) || ((cp >= 0xd800) && (cp <= 0xdfff))
	    || (cp == 0xfeff) || (cp == 0xfffe))
		return 0;

	c = (uint8_t *)utf16lec;

	if (cp < 0x10000)
	  {
		if (c != NULL)
		  {
			c[0] = cp & 0xff;
			c[1] = (cp >> 8) & 0xff;
		  }
		return 2;
	  }

	if (c != NULL)
	  {
		cp -= 0x10000;
		c[0] = (cp >> 10) & 0xff;
		c[1] = 0xd8 | ((cp >> 18) & 0x3);
		c[2] = cp & 0xff;
		c[3] = 0xdc | ((cp >> 8) & 0x3);
	  }
	return 4;
}

_NACORE_DEF size_t
nacore_char_utf16le_decode(const char *utf16lec, uint32_t *cp)
{
	const uint8_t *c;
	uint32_t u;

	c = (const uint8_t *)utf16lec;

	u = c[0] | (c[1] << 8);
	if ((u == 0xfeff) || (u == 0xfffe))
		return 0;
	if ((u >= 0xd800) && (u <= 0xdbff))
	  {
		if (cp != NULL)
			*cp =  0x10000 + (((u & 0x3ff) << 10) | c[2]
					  | ((c[3] & 0x3) << 8));
		return 4;
	  }

	if (cp != NULL)
		*cp = u;
	return 2;
}

_NACORE_DEF char *
nacore_string_utf8_to_utf16le(const char *str_utf8)
{
	const char *c8;
	char *c16;
	char *ret;
	uint32_t cp;
	size_t size, s;

	/* count needed bytes */
	for (size = 0, c8 = str_utf8; *c8 != '\0'; )
	  {
		s = nacore_char_utf8_decode(c8, &cp);
		if (s == 0)
		  {
			c8++;
			continue;
		  }

		size += nacore_char_utf16le_encode(NULL, cp);
		c8 += s;
	  }

	/* allocate */
	ret = malloc(size + 2);
	if (ret == NULL)
		return NULL;

	/* convert */
	for (c8 = str_utf8, c16 = ret; *c8 != '\0'; )
	  {
		s = nacore_char_utf8_decode(c8, &cp);
		if (s == 0)
		  {
			c8++;
			continue;
		  }

		size = nacore_char_utf16le_encode(c16, cp);
		c8 += s;
		c16 += size;
	  }

	/* null termination */
	c16[0] = 0;
	c16[1] = 0;

	return ret;
}

_NACORE_DEF char *
nacore_string_utf16le_to_utf8(const char *str_utf16le)
{
	const char *c16;
	char *c8;
	char *ret;
	uint32_t cp;
	size_t size, s;

	/* count needed bytes */
	for (size = 0, c16 = str_utf16le; *c16 != '\0'; )
	  {
		s = nacore_char_utf16le_decode(c16, &cp);
		if (s == 0)
		  {
			c16 += 2;
			continue;
		  }

		size += nacore_char_utf8_encode(NULL, cp);
		c16 += s;
	  }

	/* allocate */
	ret = malloc(size + 1);
	if (ret == NULL)
		return NULL;

	/* convert */
	for (c16 = str_utf16le, c8 = ret; *c16 != '\0'; )
	  {
		s = nacore_char_utf16le_decode(c16, &cp);
		if (s == 0)
		  {
			c16 += 2;
			continue;
		  }

		size = nacore_char_utf8_encode(c8, cp);
		c16 += s;
		c8 += size;
	  }

	/* null termination */
	*c8 = '\0';

	return ret;
}

_NACORE_DEF size_t
nacore_string_get_size(const char *s, void *unused)
{
	return strlen(s) + 1;
}

_NACORE_DEF size_t
nacore_strnlen(const char *s, size_t maxlen)
{
	size_t ret;

	for (ret = 0; s[ret] != '\0'; ret++)
		if (ret == maxlen)
			break;

	return ret;
}

/* Code point category type for grapheme breaking. */
typedef enum
  {
	cp_other = 0,
	cp_cr,
	cp_lf,
	cp_control,
	cp_extend,
	cp_prepend,
	cp_spacing_mark,
	cp_l,
	cp_v,
	cp_t,
	cp_lv,
	cp_lvt
  } code_point_category;

/* Rules for breaking graphemes. Each index of the array represents the current
 * code point category, each bit position inside an array element represents the
 * next code point category, both according to the code_point_category
 * enumeration. When the bit is set to 1 there is no grapheme break. */
static uint16_t
grapheme_breaking_chart[12] =
  {0x50, 0x04, 0, 0, 0x50, 0xff1, 0x50, 0xdd0, 0x350, 0x250, 0x350, 0x250};

/* Gets the category for grapheme breaking of the code point pointed by s.
 * 
 * TODO: needs a reimplementation using some FSM (finite state machine)
 * representation. */
static code_point_category
get_code_point_category(const char *s)
{
	uint32_t v;

	if (*s == 0xd)
		return cp_cr;
	else if (*s == 0xa)
		return cp_lf;
	else if ((*s & 0x80) == 0)	/* one byte */
	  {
		if (((*s >= 0x0) && (*s <= 0x9)) || (*s == 0xb) || (*s == 0xc)
		    || ((*s >= 0xe) && (*s <= 0x1f)) || (*s == 0x7f))
			return cp_control;
	  }
	else if ((*s & 0xe0) == 0xc0)	/* two bytes */
	  {
		v = ((s[0] & 0x1f) << 6) | (s[1] & 0x3f);

		if (((v >= 0x600) && (v <= 0x603))
		    || (v == 0x6dd) || (v == 0x70f))
			return cp_control;

		if (((v >= 0x300) && (v <= 0x36f))
		    || ((v >= 0x483) && (v <= 0x489))
		    || ((v >= 0x591) && (v <= 0x5bd))
		    || ((v >= 0x5bf) && (v <= 0x5c2))
		    || (v == 0x5c4) || (v == 0x5c5) || (v == 0x5c7)
		    || ((v >= 0x610) && (v <= 0x61a))
		    || ((v >= 0x64b) && (v <= 0x65f)) || (v == 0x670)
		    || ((v >= 0x646) && (v <= 0x6dc))
		    || ((v >= 0x6df) && (v <= 0x6e4))
		    || (v == 0x6e7) || (v == 0x6e8)
		    || ((v >= 0x6ea) && (v <= 0x6ed)) || (v == 0x711)
		    || ((v >= 0x730) && (v <= 0x74a))
		    || ((v >= 0x7a6) && (v <= 0x7b0))
		    || ((v >= 0x7eb) && (v <= 0x7f3)))
			return cp_extend;
	  }
	else if ((*s & 0xf0) == 0xe0)	/* three bytes */
	  {
		v = ((s[0] & 0xf) << 12) | ((s[1] & 0x3f) << 6) | (s[2] & 0x3f);

		if ((v == 0x17b4) || (v == 0x17b5) || (v == 0x200b)
		    || (v == 0x200e) || (v == 0x200f) || (v == 0x2028)
		    || ((v >= 0x2029) && (v <= 0x202e))
		    || ((v >= 0x2060) && (v <= 0x2064))
		    || ((v >= 0x206a) && (v <= 0x206f))
		    || (v == 0xfeff) || ((v >= 0xfff9) && (v <= 0xfffb)))
			return cp_control;

		if (((v >= 0x816) && (v <= 0x82d)  && (v != 0x820)
		     && (v != 0x824) && (v != 0x828))
		    || ((v >= 0x859) && (v <= 0x85b))
		    || ((v >= 0x900) && (v <= 0x902))
		    || (v == 0x93a) || (v == 0x93c)
		    || ((v >= 0x941) && (v <= 0x948)) || (v == 0x94d)
		    || ((v >= 0x951) && (v <= 0x957))
		    || (v == 0x962) || (v == 0x963) || (v == 0x981)
		    || (v == 0x9bc) || (v == 0x9bc) || (v == 0x9be)
		    || ((v >= 0x9c1) && (v <= 0x9c4))
		    || (v == 0x9cd) || (v == 0x9d7) || (v == 0x9e2)
		    || (v == 0x9e3) || (v == 0xa01) || (v == 0xa02)
		    || (v == 0xa3c) || (v == 0xa41) || (v == 0xa42)
		    || (v == 0xa47) || (v == 0xa48)
		    || ((v >= 0xa4b) && (v <= 0xa4d)) || (v == 0xa51)
		    || (v == 0xa70) || (v == 0xa71) || (v == 0xa75)
		    || (v == 0xa81) || (v == 0xa82) || (v == 0xabc)
		    || ((v >= 0xac1) && (v <= 0xac8) && (v != 0xac6))
		    || (v == 0xacd) || (v == 0xae2) || (v == 0xae3)
		    || (v == 0xb01) || (v == 0xb3c) || (v == 0xb3e)
		    || (v == 0xb3f) || ((v >= 0xb41) && (v <= 0xb44))
		    || (v == 0xb4d) || (v == 0xb56) || (v == 0xb57)
		    || (v == 0xb62) || (v == 0xb63) || (v == 0xb82)
		    || (v == 0xbbe) || (v == 0xbc0) || (v == 0xbcd)
		    || (v == 0xbd7) || ((v >= 0xc3e) && (v <= 0xc40))
		    || ((v >= 0xc46) && (v <= 0xc48))
		    || ((v >= 0xc4a) && (v <= 0xc4d))
		    || (v == 0xc55) || (v == 0xc56) || (v == 0xc62)
		    || (v == 0xc63) || (v == 0xcbc) || (v == 0xcbf)
		    || (v == 0xcc2) || (v == 0xcc6) || (v == 0xccc)
		    || (v == 0xccd) || (v == 0xcd5) || (v == 0xcd6)
		    || (v == 0xce2) || (v == 0xce3) || (v == 0xd3e)
		    || ((v >= 0xd41) && (v <= 0xd44)) || (v == 0xd4d)
		    || (v == 0xd57) || (v == 0xd62) || (v == 0xd63)
		    || (v == 0xdca) || (v == 0xdcf)
		    || ((v >= 0xdd2) && (v <= 0xdd4)) || (v == 0xdd6)
		    || (v == 0xddf) || (v == 0xe31)
		    || ((v >= 0xe34) && (v <= 0xe3a))
		    || ((v >= 0xe47) && (v <= 0xe4e)) || (v == 0xeb1)
		    || ((v >= 0xeb4) && (v <= 0xeb9))
		    || (v == 0xebb) || (v == 0xebc)
		    || ((v >= 0xec8) && (v <= 0xecd)) || (v == 0xf18)
		    || (v == 0xf19) || (v == 0xf35) || (v == 0xf37)
		    || (v == 0xf39)
		    || ((v >= 0xf71) && (v <= 0xf87) && (v != 0xf7f)
			&& (v != 0xf85))
		    || ((v >= 0xf8d) && (v <= 0xfbc) && (v != 0xf98))
		    || (v == 0xfc6)
		    || ((v >= 0x102d) && (v <= 0x103a) && (v != 0x1031)
			&& (v != 0x1038)) || (v == 0x103d) || (v == 0x103e)
		    || (v == 0x1058) || (v == 0x1059)
		    || ((v >= 0x105e) && (v <= 0x1060))
		    || ((v >= 0x1071) && (v <= 0x1074)) || (v == 0x1082)
		    || (v == 0x1085) || (v == 0x1086) || (v == 0x108d)
		    || (v == 0x109d) || ((v >= 0x135d) && (v <= 0x135f))
		    || ((v >= 0x1712) && (v <= 0x1714))
		    || ((v >= 0x1732) && (v <= 0x1734)) || (v == 0x1752)
		    || (v == 0x1753) || (v == 0x1772) || (v == 0x1773)
		    || ((v >= 0x17b7) && (v <= 0x17bd)) || (v == 0x17c6)
		    || ((v >= 0x17c9) && (v <= 0x17d3)) || (v == 0x17dd)
		    || ((v >= 0x180b) && (v <= 0x180d)) || (v == 0x18a9)
		    || ((v >= 0x1920) && (v <= 0x1922)) || (v == 0x1927)
		    || (v == 0x1928) || (v == 0x1932)
		    || ((v >= 0x1939) && (v <= 0x193b)) || (v == 0x1a17)
		    || (v == 0x1a18) || (v == 0x1a56)
		    || ((v >= 0x1a58) && (v <= 0x1a5e)) || (v == 0x1a60)
		    || (v == 0x1a62) || ((v >= 0x1a65) && (v <= 0x1a6c))
		    || ((v >= 0x1a73) && (v <= 0x1a7c)) || (v == 0x1a7f)
		    || ((v >= 0x1b00) && (v <= 0x1b03))
		    || ((v >= 0x1b34) && (v <= 0x1b3c) && (v != 0x1b35)
			&& (v != 0x1b3b)) || (v == 0x1b42)
		    || ((v >= 0x1b6b) && (v <= 0x1b73)) || (v == 0x1b80)
		    || (v == 0x1b81) || ((v >= 0x1ba2) && (v <= 0x1ba5))
		    || (v == 0x1ba8) || (v == 0x1ba9) || (v == 0x1be6)
		    || (v == 0x1be8) || (v == 0x1be9)
		    || ((v >= 0x1bed) && (v <= 0x1bf1) && (v != 0x1bee))
		    || ((v >= 0x1c2c) && (v <= 0x1c33))
		    || (v == 0x1c36) || (v == 0x1c37)
		    || ((v >= 0x1cd0) && (v <= 0x1ce8) && (v != 0x1cd3)
			&& (v != 0x1ce1)) || (v == 0x1ced)
		    || ((v >= 0x1dc0) && (v <= 0x1de6))
		    || ((v >= 0x1dfc) && (v <= 0x1dff))
		    || (v == 0x200c) || (v == 0x200d)
		    || ((v >= 0x20d0) && (v <= 0x20f0))
		    || ((v >= 0x2cef) && (v <= 0x2cf1))
		    || (v == 0x2d7f) || ((v >= 0x2de0) && (v <= 0x2dff))
		    || ((v >= 0x302a) && (v <= 0x302f))
		    || ((v >= 0x3099) && (v <= 0x309a))
		    || ((v >= 0xa66f) && (v <= 0xa672)) || (v == 0xa67c)
		    || (v == 0xa67d) || (v == 0xa6f0) || (v == 0xa6f1)
		    || (v == 0xa802) || (v == 0xa806) || (v == 0xa80b)
		    || (v == 0xa825) || (v == 0xa826) || (v == 0xa8c4)
		    || ((v >= 0xa8e0) && (v <= 0xa8f1))
		    || ((v >= 0xa926) && (v <= 0xa92d))
		    || ((v >= 0xa947) && (v <= 0xa951))
		    || ((v >= 0xa980) && (v <= 0xa982)) || (v == 0xa9b3)
		    || ((v >= 0xa9b6) && (v <= 0xa9b9)) || (v == 0xa9bc)
		    || ((v >= 0xaa29) && (v <= 0xaa2e)) || (v == 0xaa31)
		    || (v == 0xaa32) || (v == 0xaa35) || (v == 0xaa36)
		    || (v == 0xaa45) || (v == 0xaa4c) || (v == 0xaab0)
		    || ((v >= 0xaab2) && (v <= 0xaab4)) || (v == 0xaab7)
		    || (v == 0xaab8) || (v == 0xaabe) || (v == 0xaabf)
		    || (v == 0xaac1) || (v == 0xabe5) || (v == 0xabe8)
		    || (v == 0xabed) || (v == 0xfb1e)
		    || ((v >= 0xfe00) && (v <= 0xfe0f))
		    || ((v >= 0xfe20) && (v <= 0xfe26))
		    || (v == 0xff9e) || (v == 0xff9f))
			return cp_extend;

		if (((v >= 0xe40) && (v <= 0xe44))
		    || ((v >= 0xec0) && (v <= 0xec4))
		    || (v == 0xaab5) || (v == 0xaab6)
		    || ((v >= 0xaab9) && (v <= 0xaabc) && (v != 0xaaba)))
			return cp_prepend;

		if ((v == 0x903) || (v == 0x93b)
		    || ((v >= 0x93e) && (v <= 0x940))
		    || ((v >= 0x949) && (v <= 0x94c)) || (v == 0x94e)
		    || (v == 0x94f) || (v == 0x982) || (v == 0x983)
		    || ((v >= 0x9bf) && (v <= 0x9c8)) || (v == 0x9cb)
		    || (v == 0x9cc) || (v == 0xa03)
		    || ((v >= 0xa3e) && (v <= 0xa40)) || (v == 0xa83)
		    || ((v >= 0xabe) && (v <= 0xac0))
		    || ((v >= 0xac9) && (v <= 0xacc) && (v != 0xaca))
		    || (v == 0xb02) || (v == 0xb03) || (v == 0xb40)
		    || (v == 0xb47) || (v == 0xb48) || (v == 0xb4b)
		    || (v == 0xb4c) || (v == 0xbbf) || (v == 0xbc1)
		    || (v == 0xbc2) || ((v >= 0xbc6) && (v <= 0xbc8))
		    || ((v >= 0xbca) && (v <= 0xbcc))
		    || ((v >= 0xc01) && (v <= 0xc03))
		    || ((v >= 0xc41) && (v <= 0xc44)) || (v == 0xc82)
		    || (v == 0xc83) || (v == 0xcbe) || (v == 0xcc0)
		    || (v == 0xcc1) || (v == 0xcc3) || (v == 0xcc4)
		    || (v == 0xcc7) || (v == 0xcc8) || (v == 0xcca)
		    || (v == 0xccb) || (v == 0xd02) || (v == 0xd03)
		    || (v == 0xd3f) || (v == 0xd40)
		    || ((v >= 0xd46) && (v <= 0xd4c) && (v != 0xd49))
		    || (v == 0xd82) || (v == 0xd83) || (v == 0xdd0)
		    || (v == 0xdd1) || ((v >= 0xdd8) && (v <= 0xdde))
		    || (v == 0xdf2) || (v == 0xdf3) || (v == 0xe30)
		    || (v == 0xe32) || (v == 0xe33) || (v == 0xe45)
		    || (v == 0xeb0) || (v == 0xeb2) || (v == 0xeb3)
		    || (v == 0xf3e) || (v == 0xf3f) || (v == 0xf7f)
		    || (v == 0x120b) || (v == 0x120c) || (v == 0x1031)
		    || (v == 0x1038) || (v == 0x103b) || (v == 0x103c)
		    || (v == 0x1056) || (v == 0x1057)
		    || ((v >= 0x1062) && (v <= 0x1064))
		    || ((v >= 0x1067) && (v <= 0x106d))
		    || (v == 0x1083) || (v == 0x1084)
		    || ((v >= 0x1087) && (v <= 0x108c))
		    || ((v >= 0x108f) && (v <= 0x109c)) || (v == 0x17b6)
		    || ((v >= 0x17be) && (v <= 0x17c8) && (v != 0x17c6))
		    || ((v >= 0x1923) && (v <= 0x1926))
		    || ((v >= 0x1929) && (v <= 0x192b)) || (v == 0x1930)
		    || (v == 0x1931) || ((v >= 0x1933) && (v <= 0x1938))
		    || ((v >= 0x19b0) && (v <= 0x19c0)) || (v == 0x19c8)
		    || (v == 0x19c9) || ((v >= 0x1a19) && (v <= 0x1a1b))
		    || (v == 0x1a55) || (v == 0x1a57) || (v == 0x1a61)
		    || (v == 0x1a63) || (v == 0x1a64)
		    || ((v >= 0x1a6d) && (v <= 0x1a72))
		    || (v == 0x1b04) || (v == 0x1b35)
		    || ((v >= 0x1b3b) && (v <= 0x1b44) && (v != 0x1b3c)
			&& (v != 0x1b42))
		    || (v == 0x1b82) || (v == 0x1ba1) || (v == 0x1ba6)
		    || (v == 0x1ba7) || (v == 0x1baa) || (v == 0x1be7)
		    || ((v >= 0x1bea) && (v <= 0x1bec)) || (v == 0x1bee)
		    || (v == 0x1bf2) || (v == 0x1bf3)
		    || ((v >= 0x1c24) && (v <= 0x1c2b)) || (v == 0x1c34)
		    || (v == 0x1c35) || (v == 0x1ce1) || (v == 0x1cf2)
		    || (v == 0xa823) || (v == 0xa824) || (v == 0xa827)
		    || (v == 0xa880) || (v == 0xa881)
		    || ((v >= 0xa8b4) && (v <= 0xa8c3)) || (v == 0xa952)
		    || (v == 0xa953) || (v == 0xa983) || (v == 0xa9b4)
		    || (v == 0xa9b5) || (v == 0xa9ba) || (v == 0xa9bb)
		    || ((v >= 0xa9bd) && (v <= 0xa9c0)) || (v == 0xaa2f)
		    || (v == 0xaa30) || (v == 0xaa33) || (v == 0xaa34)
		    || (v == 0xaa4d) || (v == 0xaa7b)
		    || ((v >= 0xabe3) && (v <= 0xabec) && (v != 0xabe5)
			&& (v != 0xabe8) && (v != 0xabeb)))
			return cp_spacing_mark;

		if (((v >= 0x1100) && (v <= 0x115f))
		    || ((v >= 0xa960) && (v <= 0xa97c)))
			return cp_l;

		if (((v >= 0x1160) && (v <= 0x11a7))
		    || ((v >= 0xd7b0) && (v <= 0xd7c6)))
			return cp_v;

		if (((v >= 0x11a8) && (v <= 0x11ff))
		    || ((v >= 0xd7cb) && (v <= 0xd7fb)))
			return cp_t;

		if ((v >= 0xac00) && (v <= 0xd788) && ((v % 0x1c) == 0x10))
			return cp_lv;

		if ((v >= 0xac01) && (v <= 0xd7a3))
			return cp_lvt;
	  }
	else				/* four bytes */
	  {
		v = ((s[0] & 0x7) << 18) | ((s[1] & 0x3f) << 12)
		    | ((s[2] & 0x3f) << 6) | (s[3] & 0x3f);

		if ((v == 0x110bd) || (v == 0x1d173) || (v == 0x1d174)
		    || (v == 0xe0001) || ((v >= 0xe0020) && (v <= 0xe007f)))
			return cp_control;

		if ((v == 0x101fd)
		    || ((v >= 0x10a01) && (v <= 0x10a06) && (v != 0x10a04))
		    || ((v >= 0x10a0c) && (v <= 0x10a0f))
		    || ((v >= 0x10a38) && (v <= 0x10a3a))
		    || (v == 0x10a3f) || (v == 0x11001)
		    || ((v >= 0x11038) && (v <= 0x11046))
		    || (v == 0x11080) || (v == 0x11081)
		    || ((v >= 0x110b3) && (v <= 0x110b6))
		    || ((v >= 0x110b9) && (v <= 0x110ba))
		    || ((v >= 0x1d165) && (v <= 0x1d169) && (v != 0x1d166))
		    || ((v >= 0x1d16e) && (v <= 0x1d172))
		    || ((v >= 0x1d17b) && (v <= 0x1d182))
		    || ((v >= 0x1d185) && (v <= 0x1d18b))
		    || ((v >= 0x1d1aa) && (v <= 0x1d1ad))
		    || ((v >= 0x1d242) && (v <= 0x1d244))
		    || ((v >= 0xe0100) && (v <= 0x1e01ef)))
			return cp_extend;

		if ((v == 0x11000) || (v == 0x11002) || (v == 0x11082)
		    || ((v >= 0x110b0) && (v <= 0x110b2)) || (v == 0x110b7)
		    || (v == 0x110b8) || (v == 0x1d166) || (v == 0x1d16d))
			return cp_spacing_mark;
	  }

	return cp_other;
}

_NACORE_DEF size_t
nacore_strgraphemes(const char *s)
{
	size_t ret;
	const char *c, *n;
	code_point_category cpc, cpn;

	if (*s == '\0')
		return 0;

	ret = 0;
	n = s;
	c = NULL;
	cpc = 0;
	while (*n != '\0')
	  {
		cpn = get_code_point_category(n);

		if (c != NULL)
			if (!(grapheme_breaking_chart[cpc] & (1 << cpn)))
				ret++;

		c = n;
		cpc = cpn;

		n += (*((unsigned char *)n) < 0x80)
		     ? 1 : ((*((unsigned char *)n) < 0xe0)
			   ? 2 : ((*((unsigned char *)n) < 0xf0) ? 3 : 4));
	  }

	return ret + 1;
}

_NACORE_DEF size_t
nacore_strngraphemes(const char *s, size_t max)
{
	size_t ret;
	const char *c, *n;
	code_point_category cpc, cpn;

	if ((*s == '\0') || (max == 0))
		return 0;

	ret = 0;
	n = s;
	c = NULL;
	cpc = 0;
	max--;
	while ((*n != '\0') && (ret < max))
	  {
		cpn = get_code_point_category(n);

		if (c != NULL)
			if (!(grapheme_breaking_chart[cpc] & (1 << cpn)))
				ret++;

		c = n;
		cpc = cpn;

		n += (*((unsigned char *)n) < 0x80)
		     ? 1 : ((*((unsigned char *)n) < 0xe0)
			   ? 2 : ((*((unsigned char *)n) < 0xf0) ? 3 : 4));
	  }

	return ret + 1;
}

_NACORE_DEF char *
nacore_astrcpy(const char *s, void *unused)
{
	char *ret;
	size_t len;

	len = strlen(s) + 1;
	ret = malloc(len);
	if (ret == NULL)
		return NULL;
	memcpy(ret, s, len);

	return ret;
}

/* Length modifier enumeration. */
typedef enum
  {
	lm_none,
	lm_hh,
	lm_h,
	lm_l,
	lm_ll,
	lm_L,
	lm_j,
	lm_z,
	lm_t
  } length_mod;

/* Conversion specifier enumeration. */
typedef enum
  {
	cs_d = 0,
	cs_o,
	cs_u,
	cs_x,
	cs_X,
	cs_e,
	cs_E,
	cs_f,
	cs_F,
	cs_g,
	cs_G,
	cs_a,
	cs_A,
	cs_c,
	cs_s,
	cs_p,
	cs_n,
	cs_pc
  } conv_specifier;

/* Conversion flags. */
#define FLAG_DASH	1
#define FLAG_ZERO	(1 << 1)
#define FLAG_MINUS	(1 << 2)
#define FLAG_SPACE	(1 << 3)
#define FLAG_PLUS	(1 << 4)

/* Conversion specification type. */
typedef struct
  {
	char			flags;
	int			min_width;
	int			precision;	/* -1 == no precision given */
	length_mod		lm;
	conv_specifier		cs;
  } conv_spec;

/*
 * Parses minimum field width or precision values (also handles the "*" and
 * "*m$" cases).
 * 
 * *c should point to the beginning of the string to parse and will be set to
 * point to the character right after the value specification.
 * 
 * ap should point to the current argument which is used in the "*" and "*m$"
 * cases and will advance one position only in the "*" case.
 */
static int
get_width_prec(const char **c, va_list *ap)
{
	int ret;
	int m;
	int i;
	va_list m_ap;

	ret = 0;
	switch (**c)
	  {
		case '-':
			do
				(*c)++;
			while (isdigit(**c));
			break;
		case '*':
			(*c)++;
			if (isdigit(**c))
			  {
				m = 0;
				while (**c != '$')
				  {
					  m = 10 * m + **c - '0';
					  (*c)++;
				  }
				(*c)++;

				va_copy(m_ap, *ap);
				for (i = 0; i < m; i++)
					ret = va_arg(m_ap, int);
				va_end(m_ap);
			  }
			else
				ret = va_arg(*ap, int);
			break;
		default:
			while (isdigit(**c))
			  {
				ret = 10 * ret + **c - '0';
				(*c)++;
			  }
			break;
	  }

	return ret;
}

/*
 * Parses conversion specifications (not including the first '%' character).
 *
 * *c should point to the beginning of the string to parse and will be set to
 * point to the character right after the conversion specification.
 *
 * ap should point to the current argument which is used in the "*" and "*m$"
 * cases of precision and/or minimum width specifications and will advance only
 * in "*" cases.
 */
static conv_spec
get_conv_spec(const char **c, va_list *ap)
{
	conv_spec ret =
	  {
		/* .flags	= */ 0,
		/* .min_width	= */ 0,
		/* .precision	= */ -1,
		/* .lm		= */ lm_none,
		/* .cs		= */ cs_pc
	  };

	/* %% */
	if (**c == '%')
	  {
		(*c)++;
		return ret;
	  }

	/* flags */
	while (1)
	  {
		switch (**c)
		  {
			  case '#':
				ret.flags |= FLAG_DASH;
				break;
			  case '0':
				ret.flags |= FLAG_ZERO;
				break;
			  case '-':
				ret.flags |= FLAG_MINUS;
				break;
			  case ' ':
				ret.flags |= FLAG_SPACE;
				break;
			  case '+':
				ret.flags |= FLAG_PLUS;
				break;
			  default:
				goto flags_loop_end;
				break;
		  }

		(*c)++;
	  }
flags_loop_end:

	/* minimum field width */
	if (isdigit(**c) || (**c == '*'))
		ret.min_width = get_width_prec(c, ap);

	/* precision */
	if (**c == '.')
	  {
		(*c)++;
		ret.precision = get_width_prec(c, ap);
	  }

	/* length modifier */
	switch (**c)
	  {
		case 'h':
			(*c)++;
			if (**c == 'h')
			  {
				(*c)++;
				ret.lm = lm_hh;
			  }
			else
				ret.lm = lm_h;
			break;
		case 'l':
			(*c)++;
			if (**c == 'l')
			  {
				(*c)++;
				ret.lm = lm_ll;
			  }
			else
				ret.lm = lm_l;
			break;
		case 'L':
			(*c)++;
			ret.lm = lm_L;
			break;
		case 'j':
			(*c)++;
			ret.lm = lm_j;
			break;
		case 't':
			(*c)++;
			ret.lm = lm_t;
			break;
		case 'z':
			(*c)++;
			ret.lm = lm_z;
			break;
		default:
			ret.lm = lm_none;
			break;
	  }

	/* conversion specifier */
	switch (**c)
	  {
		case 'd':
		case 'i':
			ret.cs = cs_d;
			break;
		case 'u':
			ret.cs = cs_u;
			break;
		case 'o':
			ret.cs = cs_o;
			break;
		case 'x':
			ret.cs = cs_x;
			break;
		case 'X':
			ret.cs = cs_X;
			break;
		case 'e':
			ret.cs = cs_e;
			break;
		case 'E':
			ret.cs = cs_E;
			break;
		case 'f':
			ret.cs = cs_f;
			break;
		case 'F':
			ret.cs = cs_F;
			break;
		case 'g':
			ret.cs = cs_g;
			break;
		case 'G':
			ret.cs = cs_G;
			break;
		case 'a':
			ret.cs = cs_a;
			break;
		case 'A':
			ret.cs = cs_A;
			break;
		case 'c':
			ret.cs = cs_c;
			break;
		case 's':
			ret.cs = cs_s;
			break;
		case 'p':
			ret.cs = cs_p;
			break;
		default: /* 'n' */
			ret.cs = cs_n;
			break;
	  }

	(*c)++;

	return ret;
}

/* Gets a signed integer value from ap according to the given length
 * modifier. */
static intmax_t
conv_val_lli(length_mod lm, va_list *ap)
{
	intmax_t v;

	switch (lm)
	  {
		case lm_hh:
			v = (char)va_arg(*ap, int);
			break;
		case lm_h:
			v = (short int)va_arg(*ap, int);
			break;
		case lm_l:
			v = va_arg(*ap, long int);
			break;
		case lm_ll:
			v = va_arg(*ap, long long int);
			break;
		case lm_j:
			v = va_arg(*ap, intmax_t);
			break;
		case lm_z:
			v = va_arg(*ap, size_t);
			break;
		case lm_t:
			v = va_arg(*ap, ptrdiff_t);
			break;
		default: /* lm_none */
			v = va_arg(*ap, int);
			break;
	  }

	return v;
}

/* Gets an unsigned integer value from ap according to the given length
 * modifier. */
static uintmax_t
conv_val_ulli(length_mod lm, va_list *ap)
{
	uintmax_t v;

	switch (lm)
	  {
		case lm_hh:
			v = (unsigned char)va_arg(*ap, unsigned int);
			break;
		case lm_h:
			v = (unsigned short int)va_arg(*ap, unsigned int);
			break;
		case lm_l:
			v = va_arg(*ap, unsigned long int);
			break;
		case lm_ll:
			v = va_arg(*ap, unsigned long long int);
			break;
		case lm_j:
			v = va_arg(*ap, uintmax_t);
			break;
		case lm_z:
			v = va_arg(*ap, size_t);
			break;
		case lm_t:
			v = va_arg(*ap, ptrdiff_t);
			break;
		default: /* lm_none */
			v = va_arg(*ap, unsigned int);
			break;
	  }

	return v;
}

/* Gets a floating point value from ap according to the given length
 * modifier. */
static long double
conv_val_ld(length_mod lm, va_list *ap)
{
	return (lm == lm_L) ? va_arg(*ap, long double) : va_arg(*ap, double);
}

/* Adds left padding to *str according to the given conversion specification cs
 * and the length of the unpadded string indicated by len. *str will point to
 * the character after the padding spaces. */
static void
pad_left(char **str, conv_spec *cs, int len)
{
	int i;

	if (!(cs->flags & (FLAG_ZERO | FLAG_MINUS)))
	  {
		for (i = len; i < cs->min_width; i++)
		  {
			**str = ' ';
			(*str)++;
		  }
	  }
}

/* Adds zero padding to *str according to the given conversion specification cs
 * and the length of the unpadded string indicated by len. *str will point to
 * the character after the padding zeros. */
static void
pad_zero(char **str, conv_spec *cs, int len)
{
	int i;

	if (cs->flags & FLAG_ZERO)
	  {
		for (i = len; i < cs->min_width; i++)
		  {
			**str = '0';
			(*str)++;
		  }
	  }
}

/* Adds right padding to *str according to the given conversion specification cs
 * and the length of the unpadded string indicated by len. *str will point to
 * the character after the padding spaces. */
static void
pad_right(char **str, conv_spec *cs, int len)
{
	int i;

	if (cs->flags & FLAG_MINUS)
	  {
		for (i = len; i < cs->min_width; i++)
		  {
			**str = ' ';
			(*str)++;
		  }
	  }
}

/* Conversion functions. */

static int
conv_d(char *str, conv_spec *cs, va_list *ap)
{
	intmax_t v, c;
	int ret, i;

	v = conv_val_lli(cs->lm, ap);

	ret = 0;
	c = v;
	do
	  {
		c /= 10;
		ret++;
	  }
	while (c != 0);

	if ((cs->precision == 0) && (v == 0))
		ret = 0;
	else
		ret = MAX(ret, cs->precision);

	if ((cs->flags & (FLAG_SPACE | FLAG_PLUS)) || (v < 0))
		ret++;

	if (str == NULL)
		return MAX(ret, cs->min_width);

	pad_left(&str, cs, ret);

	if (v < 0)
	  {
		*str = '-';
		str++;
	  }
	else if (cs->flags & FLAG_PLUS)
	  {
		*str = '+';
		str++;
	  }
	else if (cs->flags & FLAG_SPACE)
	  {
		*str = ' ';
		str++;
	  }

	pad_zero(&str, cs, ((cs->flags & (FLAG_PLUS | FLAG_SPACE)) || (v < 0))
			   ? ret + 1 : ret);

	if (!((cs->precision == 0) && (v == 0)))
	  {
		if (v < 0)
			for (i = 1, c = -1; c >= v; i++, c *= 10) ;
		else
			for (i = 1, c = 1; c < v; i++, c *= 10) ;

		if (c != v)
		  {
			c /= 10;
			i--;
		  }
		
		if (i < cs->precision)
			for (; i < cs->precision; cs->precision--)
			  {
				*str = '0';
				str++;
			  }

		if (v == 0)
		  {
			*str = '0';
			str++;
		  }
		else
		  {
			do
			  {
				*str = v / c + '0';
				str++;
				v %= c;
				c /= 10;
			  }
			while (c != 0);
		  }
	  }

	pad_right(&str, cs, ret);

	return MAX(ret, cs->min_width);
}

static int
conv_o(char *str, conv_spec *cs, va_list *ap)
{
	uintmax_t v, c;
	int ret, i;

	v = conv_val_ulli(cs->lm, ap);

	ret = 0;
	c = v;
	do
	  {
		c >>= 3;
		ret++;
	  }
	while (c != 0);

	if ((cs->precision == 0) && (v == 0))
		ret = 0;
	else if (cs->precision > ret)
		ret = cs->precision;
	else if (cs->flags & FLAG_DASH)
		ret++;

	if (str == NULL)
		return MAX(ret, cs->min_width);

	pad_left(&str, cs, ret);
	pad_zero(&str, cs, ret);

	if (!((cs->precision == 0) && (v == 0)))
	  {
		for (i = 1, c = 1; c < v; i++, c <<= 3) ;

		if (c != v)
		  {
			c >>= 3;
			i--;
		  }

		if (i < cs->precision)
			for (; i < cs->precision; cs->precision--)
			  {
				*str = '0';
				str++;
			  }
		else if (cs->flags & FLAG_DASH)
		  {
			*str = '0';
			str++;
		  }

		if (v == 0)
		  {
			*str = '0';
			str++;
		  }
		else
		  {
			do
			  {
				*str = v / c + '0';
				str++;
				v %= c;
				c >>= 3;
			  }
			while (c != 0);
		  }
	  }

	pad_right(&str, cs, ret);

	return MAX(ret, cs->min_width);
}

static int
conv_u(char *str, conv_spec *cs, va_list *ap)
{
	uintmax_t v, c;
	int ret, i;

	v = conv_val_ulli(cs->lm, ap);

	ret = 0;
	c = v;
	do
	  {
		c /= 10;
		ret++;
	  }
	while (c != 0);

	if ((cs->precision == 0) && (v == 0))
		ret = 0;
	else
		ret = MAX(ret, cs->precision);

	if (str == NULL)
		return MAX(ret, cs->min_width);

	pad_left(&str, cs, ret);
	pad_zero(&str, cs, ret);

	if (!((cs->precision == 0) && (v == 0)))
	  {
		for (i = 1, c = 1; c < v; i++, c *= 10) ;

		if (c != v)
		  {
			c /= 10;
			i--;
		  }

		if (i < cs->precision)
			for (; i < cs->precision; cs->precision--)
			  {
				*str = '0';
				str++;
			  }

		if (v == 0)
		  {
			*str = '0';
			str++;
		  }
		else
		  {
			do
			  {
				*str = v / c + '0';
				str++;
				v %= c;
				c /= 10;
			  }
			while (c != 0);
		  }
	  }

	pad_right(&str, cs, ret);

	return MAX(ret, cs->min_width);
}

static int
conv_xX(char *str, conv_spec *cs, va_list *ap, char X)
{
	uintmax_t v, c;
	int ret, i;
	char x;

	v = conv_val_ulli(cs->lm, ap);

	ret = 0;
	c = v;
	do
	  {
		c >>= 4;
		ret++;
	  }
	while (c != 0);

	if ((cs->precision == 0) && (v == 0))
		ret = 0;
	else
		ret = MAX(ret, cs->precision)
		      + ((cs->flags & FLAG_DASH) ? 2 : 0);;

	if (str == NULL)
		return MAX(ret, cs->min_width);

	pad_left(&str, cs, ret);
	pad_zero(&str, cs, ret);

	if (!((cs->precision == 0) && (v == 0)))
	  {
		for (i = 1, c = 1; c < v; i++, c <<= 4) ;

		if (c != v)
		  {
			c >>= 4;
			i--;
		  }

		if (cs->flags & FLAG_DASH)
		  {
			*str = '0';
			str++;
			*str = X ? 'X' : 'x';
			str++;
		  }

		if (i < cs->precision)
			for (; i < cs->precision; cs->precision--)
			  {
				*str = '0';
				str++;
			  }

		if (v == 0)
		  {
			*str = '0';
			str++;
		  }
		else
		  {
			do
			  {
				x = v / c;
				*str = x + ((x < 10)
					    ? '0' : ((X ? 'A' : 'a') - 10));
				str++;
				v %= c;
				c >>= 4;
			  }
			while (c != 0);
		  }
	  }

	pad_right(&str, cs, ret);

	return MAX(ret, cs->min_width);
}

static int
conv_x(char *str, conv_spec *cs, va_list *ap)
{
	return conv_xX(str, cs, ap, 0);
}

static int
conv_X(char *str, conv_spec *cs, va_list *ap)
{
	return conv_xX(str, cs, ap, 1);
}

static int
conv_eE(char *str, conv_spec *cs, va_list *ap, char E)
{
	long double v, p, d;
	short int exp, dexp, pe, de;
	int ret;

	v = conv_val_ld(cs->lm, ap);

	if (cs->precision == -1)
		cs->precision = 6;

	exp = 0; 
	dexp = 2;

	if (isinf(v) || isnan(v))
		ret = 3;
	else
	  {
		if (v < -0.0)
		  {
			exp = floor(log10l(-v));
			p = powl(10.0, exp - cs->precision);
			v = roundl(v / p) * p;
			exp = floor(log10l(-v));
		  }
		else if (v > 0.0)
		  {
			exp = floor(log10l(v));
			p = powl(10.0, exp - cs->precision);
			v = roundl(v / p) * p;
			exp = floor(log10l(v));
		  }

		if ((cs->precision == 0) && !(cs->flags & FLAG_DASH))
			ret = 3;
		else
			ret = 4 + cs->precision;

		if (exp < 0)
			dexp = floor(log10(-exp) + 1.0);
		else if (exp > 0)
			dexp = floor(log10(exp) + 1.0);
		
		if (dexp < 2)
			dexp = 2;

		ret += dexp;
	  }

	if (signbit(v) || (cs->flags & FLAG_PLUS) || (cs->flags & FLAG_SPACE))
		ret++;

	if (str == NULL)
		return MAX(ret, cs->min_width);

	pad_left(&str, cs, ret);

	if (signbit(v))
	  {
		*str = '-';
		str++;
	  }
	else if (cs->flags & FLAG_PLUS)
	  {
		*str = '+';
		str++;
	  }
	else if (cs->flags & FLAG_SPACE)
	  {
		*str = ' ';
		str++;
	  }

	if (!isnan(v) && !isinf(v))
		pad_zero(&str, cs,
			 ((cs->flags & (FLAG_PLUS | FLAG_SPACE)) || signbit(v))
			 ? ret + 1 : ret);

	if (isnan(v))
	  {
		strncpy(str, E ? "NAN" : "nan", 3);
		str += 3;
	  }
	else if (isinf(v))
	  {
		strncpy(str, E ? "INF" : "inf", 3);
		str += 3;
	  }
	else if ((v == 0.0) || (v == -0.0))
	  {
		*str = '0';
		str++;

		if ((cs->precision != 0) || (cs->flags & FLAG_DASH))
		  {
			*str = '.';
			str++;

			for (; cs->precision != 0; cs->precision--)
			  {
				*str = '0';
				str++;
			  }
		  }

		strncpy(str, E ? "E+00" : "e+00", 4);
		str += 4;
	  }
	else
	  {
		if (signbit(v))
			v = -v;

		p = powl(10.0, exp);

		d = v / p;

		*str = (char)d + '0';
		str++;

		if ((cs->precision != 0) || (cs->flags & FLAG_DASH))
		  {
			*str = '.';
			str++;

			for (; cs->precision != 0; cs->precision--)
			  {
				v -= (long double)((char)d) * p;
				p /= 10.0L;
				d = v / p;
				*str = (char)d + '0';
				str++;
			  }
		  }

		*str = E ? 'E' : 'e';
		str++;

		if (exp >= 0)
			*str = '+';
		else
			*str = '-';
		str++;

		if (exp < 0)
			exp = -exp;

		for (pe = 1, dexp--; dexp != 0; pe *= 10, dexp--) ;

		for (; pe != 0; pe /= 10)
		  {
			de = exp / pe;
			*str = (char)de + '0';
			str++;
			exp -= de * pe;
		  }
	  }

	pad_right(&str, cs, ret);

	return MAX(ret, cs->min_width);
}

static int
conv_e(char *str, conv_spec *cs, va_list *ap)
{
	return conv_eE(str, cs, ap, 0);
}

static int
conv_E(char *str, conv_spec *cs, va_list *ap)
{
	return conv_eE(str, cs, ap, 1);
}

static int
conv_fF(char *str, conv_spec *cs, va_list *ap, char F)
{
	long double v, p;
	short int d;
	char c;
	int ret;

	v = conv_val_ld(cs->lm, ap);

	if (cs->precision == -1)
		cs->precision = 6;

	ret = 0;
	d = 1;

	if (isinf(v) || isnan(v))
		ret = 3;
	else
	  {
		if (v > 0.0)
		  {
			p = powl(10.0, -cs->precision);
			v = roundl(v / p) * p;
		  }
		else if (v < -0.0)
		  {
			p = powl(10.0, -cs->precision);
			v = roundl(v / p) * p;
		  }
		
		if (v <= -10.0)
			d = floor(log10l(-v) + 1.0);
		else if (v >= 10.0)
			d = floor(log10l(v) + 1.0);

		ret += d;

		if ((cs->precision != 0) || (cs->flags & FLAG_DASH))
			ret += 1 + cs->precision;
	  }

	if (signbit(v) || (cs->flags & FLAG_PLUS) || (cs->flags & FLAG_SPACE))
		ret++;

	if (str == NULL)
		return MAX(ret, cs->min_width);

	pad_left(&str, cs, ret);

	if (signbit(v))
	  {
		*str = '-';
		str++;
	  }
	else if (cs->flags & FLAG_PLUS)
	  {
		*str = '+';
		str++;
	  }
	else if (cs->flags & FLAG_SPACE)
	  {
		*str = ' ';
		str++;
	  }

	if (!isnan(v) && !isinf(v))
		pad_zero(&str, cs,
			 ((cs->flags & (FLAG_PLUS | FLAG_SPACE)) || signbit(v))
			 ? ret + 1 : ret);

	if (isnan(v))
	  {
		strncpy(str, F ? "NAN" : "nan", 3);
		str += 3;
	  }
	else if (isinf(v))
	  {
		strncpy(str, F ? "INF" : "inf", 3);
		str += 3;
	  }
	else
	  {
		if (signbit(v))
			v = -v;

		p = powl(10.0, d - 1);

		if (p < 1.0)
		  {
			*str = '0';
			str++;
		  }

		for (; p >= 1.0; p /= 10.0L)
		  {
			c = v / p;
			*str = c + '0';
			str++;
			v -= (long double)c * p;
		  }

		if ((cs->precision != 0) || (cs->flags & FLAG_DASH))
		  {
			*str = '.';
			str++;

			for (; cs->precision != 0; cs->precision--)
			  {
				c = v / p;
				*str = c + '0';
				str++;
				v -= (long double)c * p;
				p /= 10.0L;
			  }
		  }
	  }

	pad_right(&str, cs, ret);

	return MAX(ret, cs->min_width);
}

static int
conv_f(char *str, conv_spec *cs, va_list *ap)
{
	return conv_fF(str, cs, ap, 0);
}

static int
conv_F(char *str, conv_spec *cs, va_list *ap)
{
	return conv_fF(str, cs, ap, 1);
}

static int
conv_gG(char *str, conv_spec *cs, va_list *ap, char G)
{
	long double v, vc, p, d, min;
	short int exp, dexp, pe, de;
	int pc, ret;

	v = conv_val_ld(cs->lm, ap);

	if (cs->precision == -1)
		cs->precision = 6;
	else if (cs->precision == 0)
		cs->precision = 1;

	exp = 0;
	dexp = 2;

	if (isinf(v) || isnan(v))
		ret = 3;
	else
	  {
		vc = v; /* make gcc happy */

		if (v < -0.0)
		  {
			exp = floor(log10l(-v));
			p = powl(10.0, exp - cs->precision);
			vc = roundl(v / p) * p;
			exp = floor(log10l(-vc));
		  }
		else if (v > 0.0)
		  {
			exp = floor(log10l(v));
			p = powl(10.0, exp - cs->precision);
			vc = roundl(v / p) * p;
			exp = floor(log10l(vc));
		  }

		if ((exp < -4) || (exp >= cs->precision))
		  {
			v = vc;

			if (cs->flags & FLAG_DASH)
				ret = 4 + cs->precision;
			else
			  {
				ret = 3;

				p = powl(10.0, exp);
				vc = v;
				d = vc / p;
				vc -= (long double)((char)d) * p;
				p /= 10.0L;

				min = powl(10.0, exp - cs->precision + 1);

				if ((vc >= min) || (vc <= -min))
				  {
					ret++;

					for (pc = cs->precision - 1;
					     ((vc >= min) || (vc <= -min))
					     && (pc != 0); pc--, ret++)
					  {
						d = vc / p;
						vc -= (long double)((char)d)
						      * p;
						p /= 10.0L;
					  }
				  }
			  }

			if (exp < 0)
				dexp = floor(log10(-exp) + 1.0);
			else if (exp > 0)
				dexp = floor(log10(exp) + 1.0);
			
			if (dexp < 2)
				dexp = 2;

			ret += dexp;
		  }
		else
		  {
			p = powl(10.0, -cs->precision);
			v = roundl(v / p) * p;

			if (v <= -10.0)
				ret = floor(log10l(-v) + 1.0);
			else if (v >= 10.0)
				ret = floor(log10l(v) + 1.0);
			else
				ret = 1;

			if (cs->flags & FLAG_DASH)
				ret += 1 + cs->precision;
			else
			  {
				vc = fabsl(vc);
				vc -= floorl(vc);

				min = powl(10.0, ret - cs->precision);
				if ((v < 1.0) && (v > -1.0))
					min /= 10.0L;

				if (vc >= min)
				  {
					ret++;

					p = 0.1L;
					for (pc = cs->precision;
					     (vc >= min) && (pc != 0); pc--,
					     ret++)
					  {
						d = vc / p;
						vc -= (long double)((char)d)
						      * p;
						p /= 10.0L;
					  }
				  }
			  }
		  }
	  }

	if (signbit(v) || (cs->flags & FLAG_PLUS) || (cs->flags & FLAG_SPACE))
		ret++;

	if (str == NULL)
		return MAX(ret, cs->min_width);

	pad_left(&str, cs, ret);

	if (signbit(v))
	  {
		*str = '-';
		str++;
	  }
	else if (cs->flags & FLAG_PLUS)
	  {
		*str = '+';
		str++;
	  }
	else if (cs->flags & FLAG_SPACE)
	  {
		*str = ' ';
		str++;
	  }

	if (!isnan(v) && !isinf(v))
		pad_zero(&str, cs,
			 ((cs->flags & (FLAG_PLUS | FLAG_SPACE)) || signbit(v))
			 ? ret + 1 : ret);

	if (isnan(v))
	  {
		strncpy(str, G ? "NAN" : "nan", 3);
		str += 3;
	  }
	else if (isinf(v))
	  {
		strncpy(str, G ? "INF" : "inf", 3);
		str += 3;
	  }
	else
	  {
		if (signbit(v))
			v = -v;

		if ((exp < -4) || (exp >= cs->precision))
		  {
			p = powl(10.0, exp);

			d = v / p;
			v -= (long double)((char)d) * p;
			p /= 10.0L;

			*str = (char)d + '0';
			str++;

			min = powl(10.0, exp - cs->precision + 1);

			if ((cs->flags & FLAG_DASH)
			    || ((v >= min) || (v <= -min)))
			  {
				*str = '.';
				str++;

				for (; (cs->precision != 0)
				       && ((cs->flags & FLAG_DASH)
					   || (!(cs->flags & FLAG_DASH)
					       && ((v >= min)
					           || (v <= -min))));
				     cs->precision--)
				  {
					d = v / p;
					*str = (char)d + '0';
					str++;
					v -= (long double)((char)d) * p;
					p /= 10.0L;
				  }
			  }

			*str = G ? 'E' : 'e';
			str++;

			if (exp >= 0)
				*str = '+';
			else
				*str = '-';
			str++;

			if (exp < 0)
				exp = -exp;

			for (pe = 1, dexp--; dexp != 0; pe *= 10, dexp--) ;

			for (; pe != 0; pe /= 10)
			  {
				de = exp / pe;
				*str = (char)de + '0';
				str++;
				exp -= de * pe;
			  }
		  }
		else
		  {
			if (v != 0.0)
				p = powl(10.0, floor(log10l(v) + 1.0) - 1.0);
			else
				p = 1.0;

			min = p / powl(10.0, cs->precision);
			if ((v < 1.0) && (v > -1.0))
				min /= 10.0L;

			if (p < 1.0)
			  {
				*str = '0';
				str++;
			  }

			for (; p >= 1.0; p /= 10.0L)
			  {
				de = v / p;
				*str = (char)de + '0';
				str++;
				v -= (long double)de * p;
			  }

			if ((cs->flags & FLAG_DASH)
			    || ((v <= -min) || (v >= min)))
			  {
				*str = '.';
				str++;

				p = 0.1L;
				for (; (cs->precision != 0)
				       && ((cs->flags & FLAG_DASH)
					   || (!(cs->flags & FLAG_DASH)
					       && ((v >= min)
					           || (v <= -min))));
				     cs->precision--)
				  {
					de = v / p;
					*str = (char)de + '0';
					str++;
					v -= (long double)de * p;
					p /= 10.0L;
				  }
			  }
		  }
	  }

	pad_right(&str, cs, ret);

	return MAX(ret, cs->min_width);
}

static int
conv_g(char *str, conv_spec *cs, va_list *ap)
{
	return conv_gG(str, cs, ap, 0);
}

static int
conv_G(char *str, conv_spec *cs, va_list *ap)
{
	return conv_gG(str, cs, ap, 1);
}

static int
conv_aA(char *str, conv_spec *cs, va_list *ap, char A)
{
	long double v, p, d, vc;
	short int exp, dexp, pe, de;
	int ret;

	v = conv_val_ld(cs->lm, ap);

	exp = 0;
	dexp = 1;

	if (isinf(v) || isnan(v))
		ret = 3;
	else
	  {
		if (v < -0.0)
			exp = floor(log2l(-v));
		else if (v > 0.0)
			exp = floor(log2l(v));
		
		if (cs->precision != -1)
		  {
			if (v < -0.0)
			  {
				p = powl(16.0, (double)exp / 4.0
					       - (double)cs->precision);
				v = roundl(v / p) * p;
				exp = floor(log2l(-v));
			  }
			else if (v > 0.0)
			  {
				p = powl(16.0, (double)exp / 4.0
					       - (double)cs->precision);
				v = roundl(v / p) * p;
				exp = floor(log2l(v));
			  }

			if ((cs->precision == 0) && !(cs->flags & FLAG_DASH))
				ret = 5;
			else
				ret = 6 + cs->precision;
		  }
		else
		  {
			ret = 5;

			p = powl(16.0, (double)exp / 4.0);

			vc = v;
			d = vc / p;
			vc -= (long double)((char)d) * p;

			if (((vc != 0.0) && (vc != -0.0))
			    || (cs->flags & FLAG_DASH))
				ret++;

			while ((vc != 0.0) && (vc != -0.0))
			  {
				p /= 16.0L;
				d = vc / p;
				vc -= (long double)((char)d) * p;
				ret++;
			  }
		  }

		if (exp < 0)
			dexp = floor(log10(-exp) + 1.0);
		else if (exp > 0)
			dexp = floor(log10(exp) + 1.0);
		
		if (dexp < 1)
			dexp = 1;

		ret += dexp;
	  }

	if (signbit(v) || (cs->flags & FLAG_PLUS) || (cs->flags & FLAG_SPACE))
		ret++;

	if (str == NULL)
		return MAX(ret, cs->min_width);

	pad_left(&str, cs, ret);

	if (signbit(v))
	  {
		*str = '-';
		str++;
	  }
	else if (cs->flags & FLAG_PLUS)
	  {
		*str = '+';
		str++;
	  }
	else if (cs->flags & FLAG_SPACE)
	  {
		*str = ' ';
		str++;
	  }

	if (!isnan(v) && !isinf(v))
		pad_zero(&str, cs,
			 ((cs->flags & (FLAG_PLUS | FLAG_SPACE)) || signbit(v))
			 ? ret + 1 : ret);

	if (isnan(v))
	  {
		strncpy(str, A ? "NAN" : "nan", 3);
		str += 3;
	  }
	else if (isinf(v))
	  {
		strncpy(str, A ? "INF" : "inf", 3);
		str += 3;
	  }
	else
	  {
		if (signbit(v))
			v = -v;

		*str = '0';
		str++;
		*str = A ? 'X' : 'x';
		str++;

		p = powl(16.0, (double)exp / 4.0);

		d = v / p;
		*str = (char)d + ((d < 10.0) ? '0' : (((A ? 'A' : 'a') - 10)));
		str++;
		v -= (long double)((char)d) * p;

		if ((cs->precision != -1)
		    && ((cs->precision != 0) || (cs->flags & FLAG_DASH)))
		  {
			*str = '.';
			str++;

			for (; cs->precision != 0; cs->precision--)
			  {
				p /= 16.0L;
				d = v / p;
				*str = (char)d + ((d < 10.0)
						  ? '0' : (((A ? 'A' : 'a')
							    - 10)));
				str++;
				v -= (long double)((char)d) * p;
			  }
		  }
		else if ((v != 0.0) && (v != -0.0) && (cs->precision == -1))
		  {
			*str = '.';
			str++;

			while ((v != 0.0) && (v != -0.0))
			  {
				p /= 16.0L;
				d = v / p;
				*str = (char)d + ((d < 10.0)
						  ? '0' : (((A ? 'A' : 'a')
							    - 10)));
				str++;
				v -= (long double)((char)d) * p;
			  }
		  }

		*str = A ? 'P' : 'p';
		str++;

		if (exp >= 0)
			*str = '+';
		else
			*str = '-';
		str++;

		if (exp < 0)
			exp = -exp;

		for (pe = 1, dexp--; dexp != 0; pe *= 10, dexp--) ;

		for (; pe != 0; pe /= 10)
		  {
			de = exp / pe;
			*str = (char)de + '0';
			str++;
			exp -= de * pe;
		  }
	  }

	pad_right(&str, cs, ret);

	return MAX(ret, cs->min_width);
}

static int
conv_a(char *str, conv_spec *cs, va_list *ap)
{
	return conv_aA(str, cs, ap, 0);
}

static int
conv_A(char *str, conv_spec *cs, va_list *ap)
{
	return conv_aA(str, cs, ap, 1);
}

/* FIXME: wint_t ("%lc") not supported */
static int
conv_c(char *str, conv_spec *cs, va_list *ap)
{
	unsigned char v;

	v = va_arg(*ap, int);

	if (str == NULL)
		return MAX(1, cs->min_width);

	pad_left(&str, cs, 1);

	*str = v;
	str++;
	
	pad_right(&str, cs, 1);

	return MAX(1, cs->min_width);
}

/* FIXME: const wchar_t * ("%ls") not supported */
static int
conv_s(char *str, conv_spec *cs, va_list *ap)
{
	const char *v;
	int ret;

	v = va_arg(*ap, const char *);

	ret = (cs->precision >= 0) ? nacore_strnlen(v, cs->precision)
				   : strlen(v);

	if (str == NULL)
		return MAX(ret, cs->min_width);

	pad_left(&str, cs, ret);

	memcpy(str, v, ret);
	str += ret;

	pad_right(&str, cs, ret);

	return MAX(ret, cs->min_width);
}

static int
conv_p(char *str, conv_spec *cs, va_list *ap)
{
	cs->flags |= FLAG_DASH;
	cs->lm = lm_l;

	return conv_x(str, cs, ap);
}

/* Conversion functions' table (index of function according to
 * enum conv_specifier. */
static int (*conv_func[])(char *str, conv_spec *cs, va_list *ap) =
  {
	conv_d, conv_o, conv_u, conv_x, conv_X, conv_e, conv_E, conv_f, conv_F,
	conv_g, conv_G, conv_a, conv_A, conv_c, conv_s, conv_p
  };

_NACORE_DEF NACORE_FORMAT_VPRINTF(2) int
nacore_vasprintf(char **strp, const char *fmt, va_list ap)
{
	int ret;
	const char *c;
	char *s, *rets;
	va_list ap2;
	conv_spec cs;
	int *n;
	int i;

	/* calculate needed memory space */
	va_copy(ap2, ap);

	c = fmt;
	ret = 0;
	while (*c != '\0')
	  {
		if (*c != '%')
		  {
			/* not a conversion specification */
			ret++, c++;
			continue;
		  }

		/* conversion specification, get needed space */
		c++;
		cs = get_conv_spec(&c, &ap2);
		if (cs.cs != cs_n)
			ret += conv_func[cs.cs](NULL, &cs, &ap2);
	  }

	va_end(ap2);

	if (strp == NULL)
		return ret;

	/* allocate memory and perform conversions */
	rets = malloc(ret + 1);
	if (rets == NULL)
	  {
		*strp = NULL;
		return ret;
	  }

	va_copy(ap2, ap);

	s = rets;
	c = fmt;
	ret = 0;
	while (*c != '\0')
	  {
		if (*c != '%')
		  {
			/* not a conversion specification */
			*s = *c;
			ret++, c++, s++;
			continue;
		  }

		/* conversion specification, perform conversion */
		c++;
		cs = get_conv_spec(&c, &ap2);
		if (cs.cs == cs_n)
		  {
			/* %n */
			n = va_arg(ap, int *);
			*n = ret;
		  }
		else
		  {
			i = conv_func[cs.cs](s, &cs, &ap2);
			ret += i;
			s += i;
		  }
	  }

	*s = '\0';
	*strp = rets;

	va_end(ap2);

	return ret;
}

_NACORE_DEF NACORE_FORMAT_PRINTF(2, 3) int
nacore_asprintf(char **strp, const char *fmt, ...)
{
	va_list ap;
	int ret;

	va_start(ap, fmt);
	ret = nacore_vasprintf(strp, fmt, ap);
	va_end(ap);

	return ret;
}

_NACORE_DEF nacore_list
nacore_string_split(const char *s, const char *sep, nacore_filter_cb filter_cb,
		    void *filter_opaque)
{
	nacore_list ret;
	const char *p;
	char *buf, *tmp;
	size_t buf_size, s_len, sep_len;

	ret = nacore_list_new((nacore_get_size_cb)nacore_string_get_size);
	if (ret == NULL)
		return NULL;

	for (buf = NULL, buf_size = 0, sep_len = strlen(sep),
	     p = strstr(s, sep); p != NULL; s = p + sep_len, p = strstr(s, sep))
	  {
		s_len = (size_t)(p - s);

		if (buf_size <= s_len)
		  {
			tmp = realloc(buf, s_len + 1);
			if (tmp == NULL)
				goto err;

			buf = tmp;
			buf_size = s_len + 1;
		  }

		memcpy(buf, s, s_len);
		buf[s_len] = '\0';

		if (filter_cb != NULL)
			if (filter_cb(buf, filter_opaque) == 0)
				continue;

		if (nacore_list_append(ret, NULL, buf) == NULL)
			goto err;
	  }

	if (buf != NULL)
	  {
		free(buf);
		buf = NULL;
	  }

	if (filter_cb != NULL)
		if (filter_cb(s, filter_opaque) == 0)
			return ret;

	if (nacore_list_append(ret, NULL, (void *)s) == NULL)
		goto err;

	return ret;

err:
	if (buf != NULL)
		free(buf);
	nacore_list_free(ret, NULL, NULL);
	errno = ENOMEM;
	return NULL;
}
