/*
 * NASPRO - NASPRO Architecture for Sound Processing
 * Core library
 *
 * Copyright (C) 2007-2010 Stefano D'Angelo <zanga.mail@gmail.com>
 *
 * See the COPYING file for license conditions.
 */

#include <stdlib.h>
#include <string.h>

#include <NASPRO/core/lib.h>

struct avl_tree_node
  {
	size_t			 father;
	size_t			 left;
	size_t			 right;
	size_t			 l_height;
	size_t			 r_height;
	void			*content;
  };

struct _nacore_avl_tree
  {
	size_t			 root;
	struct avl_tree_node	*nodes;
	size_t			 count;

	int	(*content_cmp)		(void *c1, void *c2);
	int	(*key_cmp)		(void *content, void *key);
  };

nacore_avl_tree_t
nacore_avl_tree_new(int (*content_cmp)	(void *c1, void *c2),
		    int (*key_cmp)	(void *content, void *key))
{
	nacore_avl_tree_t tree;

	tree = malloc(sizeof(struct _nacore_avl_tree));
	if (tree == NULL)
		return NULL;

	tree->root = 0;
	tree->nodes = NULL;
	tree->count = 0;
	tree->content_cmp = content_cmp;
	tree->key_cmp = key_cmp;

	return tree;
}

#define max(x, y)	(((x) > (y)) ? (x) : (y))

#if 0
static void
node_dump(nacore_avl_tree_t tree, struct avl_tree_node *node)
{
	printf("  node: %lu (%p), father: %lu (%p), left: %lu (%p), right: %lu (%p), l_height: %lu, r_height: %lu\n",
		node - tree->nodes + 1,  node,
		node->father, tree->nodes + node->father - 1,
		node->left, tree->nodes + node->left - 1,
		node->right, tree->nodes + node->right - 1,
		node->l_height, node->r_height);
	if (node->left != 0)
		node_dump(tree, tree->nodes + node->left - 1);
	if (node->right != 0)
		node_dump(tree, tree->nodes + node->right - 1);
}

static void
tree_dump(nacore_avl_tree_t tree)
{
	printf("Tree: %p, nodes: %lu, root: %lu (%p)\n", tree, tree->count, tree->root, tree->nodes + tree->root - 1);
	if (tree->root != 0)
	node_dump(tree, tree->nodes + tree->root - 1);
}
#endif

static void
adjust_heights(nacore_avl_tree_t tree, struct avl_tree_node *p)
{
	struct avl_tree_node *p2;
	
	for (p2 = p, p = (p->father == 0) ? NULL : tree->nodes + p->father - 1;
	     p != NULL;
	     p2 = p, p = (p->father == 0) ? NULL : tree->nodes + p2->father - 1)
	  {
		if (p->left == p2 - tree->nodes + 1)
			p->l_height = max(p2->l_height, p2->r_height) + 1;
		else
			p->r_height = max(p2->l_height, p2->r_height) + 1;
	  }
}

void
nacore_avl_tree_add(nacore_avl_tree_t tree, void *content)
{
	struct avl_tree_node *p, *tmp, *tmp2;

	/* Allocation */
	tmp = realloc(tree->nodes,
		      (tree->count + 1) * sizeof(struct avl_tree_node));
	if (tmp == NULL)
		return;

	tree->nodes = tmp;
	tmp = tree->nodes + tree->count;

	tmp->content = content;
	tmp->father = 0;
	tmp->left = 0;
	tmp->right = 0;
	tmp->l_height = 0;
	tmp->r_height = 0;
	tree->count++;

	/* Insertion */
	if (tree->root == 0)
	  {
		tree->root = 1;
		return;
	  }

	p = tree->nodes + tree->root - 1;
	while (1)
	  {
		if (tree->content_cmp(content, p->content) > 0)
		  {
			if (p->right != 0)
				p = tree->nodes + p->right - 1;
			else
			  {
				p->r_height = 1;
				p->right = tree->count;
				tmp->father = p - tree->nodes + 1;
				adjust_heights(tree, p);
				break;
			  }
		  }
		else
		  {
			if (p->left != 0)
				p = tree->nodes + p->left - 1;
			else
			  {
				p->l_height = 1;
				p->left = tree->count;
				tmp->father = p - tree->nodes + 1;
				adjust_heights(tree, p);
				break;
			  }
		  }
	  }

	/* Balancing */
	while (p->father != 0)
	  {
		p = tree->nodes + p->father - 1;

		if ((p->l_height - p->r_height) == 2)
		  {
			if (tree->nodes[p->left - 1].l_height
			    == (p->l_height - 1))
			  {
				/* right rotation */
				
				/* p is root, tmp is pivot */
				tmp = tree->nodes + p->left - 1;
				
				p->left = tmp->right;
				tmp->right = p - tree->nodes + 1;

				p->l_height = tmp->r_height;
				tmp->r_height = max(p->l_height, p->r_height)
						+ 1;

				tmp->father = p->father;
				p->father = tmp - tree->nodes + 1;

				if (p->left != 0)
					tree->nodes[p->left - 1].father =
						p - tree->nodes + 1;

				if (tmp->father != 0)
				  {
					if (tree->nodes[tmp->father - 1].left
					    == p - tree->nodes + 1)
						tree->nodes[tmp->father
							    - 1].left =
							tmp - tree->nodes + 1;
					else
						tree->nodes[tmp->father
							    - 1].right =
							tmp - tree->nodes + 1;
				  }

				adjust_heights(tree, p);
			  }
			else
			  {
				/* left-right rotation */

				/* p is root's father, tmp is pivot,
				 * tmp2 is root */
				tmp = tree->nodes
				      + tree->nodes[p->left - 1].right - 1;
				tmp2 = tree->nodes + tmp->father - 1;
				
				p->left = tmp->right;
				tmp2->right = tmp->left;
				tmp->left = tmp2 - tree->nodes + 1;
				tmp->right = p - tree->nodes + 1;

				p->l_height = tmp->r_height;
				tmp2->r_height = tmp->l_height;
				tmp->l_height = max(tmp2->l_height,
						    tmp2->r_height) + 1;
				tmp->r_height = max(p->l_height, p->r_height)
						+ 1;

				tmp->father = p->father;
				p->father = tmp - tree->nodes + 1;
				tmp2->father = tmp - tree->nodes + 1;

				if (tmp2->right != 0)
					tree->nodes[tmp2->right - 1].father =
						tmp2 - tree->nodes + 1;
				if (p->left != 0)
					tree->nodes[p->left - 1].father =
						p - tree->nodes + 1;

				if (tmp->father != 0)
				  {
					if (tree->nodes[tmp->father - 1].left
					    == p - tree->nodes + 1)
						tree->nodes[tmp->father
							    - 1].left =
							tmp - tree->nodes + 1;
					else
						tree->nodes[tmp->father
							    - 1].right =
							tmp - tree->nodes + 1;
				  }

				adjust_heights(tree, p);
			  }
		  }
		else if ((p->r_height - p->l_height) == 2)
		  {
			if (tree->nodes[p->right - 1].r_height
			    == (p->r_height - 1))
			  {
				/* left rotation */

				/* p is root, tmp is pivot */
				
				tmp = tree->nodes + p->right - 1;
				
				p->right = tmp->left;
				tmp->left = p - tree->nodes + 1;

				p->r_height = tmp->l_height;
				tmp->l_height = max(p->l_height, p->r_height)
						+ 1;

				tmp->father = p->father;
				p->father = tmp - tree->nodes + 1;

				if (p->right != 0)
					tree->nodes[p->right - 1].father =
						p - tree->nodes + 1;

				if (tmp->father != 0)
				  {
					if (tree->nodes[tmp->father - 1].left
					    == p - tree->nodes + 1)
						tree->nodes[tmp->father
							    - 1].left =
							tmp - tree->nodes + 1;
					else
						tree->nodes[tmp->father
							    - 1].right =
							tmp - tree->nodes + 1;
				  }

				adjust_heights(tree, p);
			  }
			else
			  {
				/* right-left rotation */

				/* p is root's father, tmp is pivot,
				 * tmp2 is root */
				
				tmp = tree->nodes +
				      tree->nodes[p->right - 1].left - 1;
				tmp2 = tree->nodes + tmp->father - 1;

				p->right = tmp->left;
				tmp2->left = tmp->right;
				tmp->left = p - tree->nodes + 1;
				tmp->right = tmp2 - tree->nodes + 1;

				p->r_height = tmp->l_height;
				tmp2->l_height = tmp->r_height;
				tmp->l_height = max(p->l_height, p->r_height)
						+ 1;
				tmp->r_height = max(tmp2->l_height,
						    tmp2->r_height) + 1;

				tmp->father = p->father;
				p->father = tmp - tree->nodes + 1;
				tmp2->father = tmp - tree->nodes + 1;

				if (tmp2->left != 0)
					tree->nodes[tmp2->left - 1].father =
						tmp2 -tree->nodes + 1;
				if (p->right != 0)
					tree->nodes[p->right - 1].father =
						p -tree->nodes + 1;

				if (tmp->father != 0)
				  {
					if (tree->nodes[tmp->father - 1].left
					    == p - tree->nodes + 1)
						tree->nodes[tmp->father
							    - 1].left =
							tmp - tree->nodes + 1;
					else
						tree->nodes[tmp->father
							    - 1].right =
							tmp - tree->nodes + 1;
				  }

				adjust_heights(tree, p);
			  }
		  }
	  }

	/* Fix root node */
	p = tree->nodes + tree->root - 1;
	while (p->father != 0)
		p = tree->nodes + p->father - 1;

	tree->root = p - tree->nodes + 1;
}

void
nacore_avl_tree_for_each(nacore_avl_tree_t tree,
			 void (*callback)(void *content, void *data),
			 void *data)
{
	size_t i;

	for (i = 0; i < tree->count; i++)
		callback(tree->nodes[i].content, data);
}

void *
nacore_avl_tree_find(nacore_avl_tree_t tree, void *key)
{
	struct avl_tree_node *p;
	int cmp;
	size_t next;

	if (tree->root == 0)
		return NULL;

	next = tree->root;
	do
	  {
		p = tree->nodes + next - 1;
		cmp = tree->key_cmp(p->content, key);
		if (cmp == 0)
			return p->content;
		if (cmp < 0)
			next = p->right;
		else
			next = p->left;
	  }
	while (next != 0);

	return NULL;
}

size_t
nacore_avl_tree_get_nodes_count(nacore_avl_tree_t tree)
{
	return tree->count;
}

void
nacore_avl_tree_free(nacore_avl_tree_t tree)
{
	free(tree->nodes);
	free(tree);
}

int
nacore_content_cmp_descriptor_by_uri(void *c1, void *c2)
{
	return strcmp(((struct nacore_descriptor *)c1)->uri,
		      ((struct nacore_descriptor *)c2)->uri);
}

int
nacore_key_cmp_descriptor_by_uri(void *content, void *key)
{
	return strcmp(((struct nacore_descriptor *)content)->uri, (char *)key);
}
