/*
    FUSE: Filesystem in Userspace
    Copyright (C) 2001-2005  Miklos Szeredi <miklos@szeredi.hu>

    MythTV FUSE Module
    Copyright (C) 2005-2007  Kees Cook <kees@outflux.net>

    This program can be distributed under the terms of the GNU GPL.
    See the file COPYING.

    This module implements a filename overlay between MythTV and
    Galleon.

    TODO:
        * switch to GString for all the string manipulations
        * add option for logging to syslog
        * allow for configurable filename format

*/

#include <config.h>

#ifdef linux
/* For pread()/pwrite() */
#define _XOPEN_SOURCE 500
#endif

// report on the directory aiming calculations
#undef MYTHTVFS_DEBUG_AIM
// report on the calling paths
#undef MYTHTVFS_DEBUG_FUNC

#include <glib.h>
#include <fuse.h>
#include <fuse_opt.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <dirent.h>
#include <errno.h>
#include <sys/statfs.h>
#ifdef HAVE_SETXATTR
#include <sys/xattr.h>
#endif

#include <malloc.h>  // realloc
#include <stdlib.h>  // exit
#include <strings.h> // rindex
#include <time.h>    // strftime
#include <string.h>  // memset

// networking
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>

// elements we're interested in from the MythTV database of programs
struct show_t {
    char * title;
    char * subtitle;
    char * description;
    char * callsign;
    char * pathname;
    char * starttime;
    char * endtime;
    char * airdate;
};

// options specific to mythtvfs
enum {
    KEY_HELP,
    KEY_VERSION,
};

// global state structure
struct mythtvfs {
    /* configurable */
    char * base_path;
    char * host;
    int    port;
    int short_display_flag;
    char * logfile_name;

    /* runtime state trackers */
    size_t base_len;  // cached strlen(base_path)
    int backend;      // backend fd
    int program_cols; // how many columns are reported from this backend version
    FILE * logfile;
};

static struct mythtvfs mythtvfs;

#define MYTHTVFS_OPT(t, p, v) { t, offsetof(struct mythtvfs, p), v }

static struct fuse_opt mythtvfs_opts[] = {
    /* -o options */
    MYTHTVFS_OPT("host=%s",         host, 0),
    MYTHTVFS_OPT("port=%d",         port, 0),
    MYTHTVFS_OPT("short-display",   short_display_flag, 1),
    MYTHTVFS_OPT("logfile=%s",      logfile_name, 0),
  
    FUSE_OPT_KEY("-V",                KEY_VERSION),
    FUSE_OPT_KEY("--version",         KEY_VERSION),
    FUSE_OPT_KEY("-h",                KEY_HELP),
    FUSE_OPT_KEY("--help",            KEY_HELP),
    FUSE_OPT_END
};

static void usage(const char *progname)
{
    fprintf(stderr,
        "Usage: %s [options] original-path mythtvfs-mount\n"
        "\n"
        "general options:\n"
        "    -o opt,[opt...]        mount options\n"
        "    -h   --help            print help\n"
        "    -V   --version         print version\n"
        "\n"
        "MythTVfs options:\n"
        "    -o host=HOST          mythtv backend hostname\n"
        "    -o port=PORT          mythtv backend host port\n"
        "    -o short-display      shorter filename display\n"
        "    -o logfile=FILE       write verbose logging to FILE\n"
        "\n",progname);
}


//
// Filesystem overlay handling
//
static char * aimed_path = NULL;
static size_t aimed_len  = 0;
static char * aim_to_base(const char * oldpath)
{
    size_t len = 0;
    char * path = (char *)oldpath;

#ifdef MYTHTVFS_DEBUG_AIM
    if (mythtvfs.logfile) fprintf(mythtvfs.logfile,"aim '%s'\n",oldpath);
#endif

    /* skip leading slashes */
    while (path && *path=='/') path++;
    /* make sure path is NULL for "empty" */
    if (path && *path=='\0') path=NULL;
    
    /* calculate new path length */
    if (path) len = strlen(path); 
    len += mythtvfs.base_len + 1 /* "/" */ + 1 /* NULL */;
    
    /* grow path length */
    if (aimed_len < len) {
        if (!(aimed_path = realloc(aimed_path,len))) {
            perror("realloc");
            exit(1);
        }
        aimed_len = len;
    }

    snprintf(aimed_path,aimed_len,"%s%s%s",
             mythtvfs.base_path,
             path ? "/" : "",
             path ? path : "");

#ifdef MYTHTVFS_DEBUG_AIM
    if (mythtvfs.logfile) fprintf(mythtvfs.logfile,"aimed '%s'\n",aimed_path);
#endif

    return aimed_path;
}

static char * fixed_path = NULL;
static size_t fixed_len  = 0;
static char * fixup(const char * oldpath)
{
    size_t len = 0;
    char * path = (char *)oldpath;

#ifdef MYTHTVFS_DEBUG_AIM
    if (mythtvfs.logfile) fprintf(mythtvfs.logfile,"fix '%s'\n",oldpath);
#endif

    /* make sure path is "" for "empty" */
    if (!path) path="";
    
    /* calculate new path length */
    len = strlen(path) + 1 /* NULL */; 

    if (fixed_len < len) {
        if (!(fixed_path = realloc(fixed_path,len))) {
            perror("realloc");
            exit(1);
        }
        fixed_len = len;
    }

    /* trim metadata */
    char * mark;
    strncpy(fixed_path,path,len);
    if (!(mark = rindex(fixed_path,'}'))) {
        return aim_to_base(fixed_path);
    }
    path = mark+1;
    if (!(mark = rindex(path,'.'))) {
        return aim_to_base(fixed_path);
    }
    *mark='\0';

#ifdef MYTHTVFS_DEBUG_AIM
    if (mythtvfs.logfile) fprintf(mythtvfs.logfile,"fixed '%s'\n",path);
#endif

    return aim_to_base(path);
}

//
// Backend communication handling
//
void backend_close()
{
    if (mythtvfs.backend != -1) close(mythtvfs.backend);
    mythtvfs.backend = -1;
}

void backend_client_check();

int backend_init()
{
    struct sockaddr_in host_addr;
    struct hostent *hostinfo;

    if (mythtvfs.backend != -1) close(mythtvfs.backend);

    if ((mythtvfs.backend = socket(AF_INET, SOCK_STREAM, 0))<0) {
        perror("socket");
        exit(1);
    }
    if (!(hostinfo = gethostbyname(mythtvfs.host))) {
        perror(mythtvfs.host);
        exit(1);
    }
    bzero((char*)&host_addr,sizeof(host_addr));
    host_addr.sin_family = AF_INET;
    bcopy((char*)hostinfo->h_addr,
          (char*)&host_addr.sin_addr.s_addr,
          hostinfo->h_length);
    host_addr.sin_port = htons(mythtvfs.port);
    if (connect(mythtvfs.backend,(struct sockaddr*)&host_addr,sizeof(host_addr))<0) {
        fprintf(stderr,"%s:%d ",mythtvfs.host,mythtvfs.port);
        perror("connect");
        backend_close();
        return 0;
    }

    // must attach as a client
    backend_client_check();

    return 1;
}

int backend_cmd(char * cmd, char ** output)
{
    char * send;
    uint32_t len, cmdlen, bytes;
    char size[8];

    if (!output) return 0;

    if (mythtvfs.backend == -1 && !backend_init()) return 0;

    if (mythtvfs.logfile) fprintf(mythtvfs.logfile,"Sending '%s'\n",cmd);

    cmdlen = strlen(cmd);
    len = cmdlen + 8 /* length prefix */ + 1 /* NULL */;
    if (!(send = malloc(len))) {
        perror("malloc");
        exit(1);
    }
    snprintf(send,len,"%-8d%s",cmdlen,cmd);
    len--; /* don't send the NULL */

    if ((bytes = write(mythtvfs.backend,send,len)) != len) {
        free(send);
        perror("write");
        backend_close();
        return 0;
    }
    free(send);

    if (!strcmp(cmd,"DONE")) return 1;

    if ((bytes = read(mythtvfs.backend,size,8)) != 8) {
        perror("read size");
        backend_close();
        return 0;
    }

    len = atoi(size);

    if (!(*output=realloc(*output,len+1))) {
        perror("realloc");
        exit(1);
    }

    bytes = 0;
    while (bytes != len) {
        int got = 0;
        if ((got = read(mythtvfs.backend,*output+bytes,len))<0) {
            perror("read data");
            backend_close();
            return 0;
        }
        bytes+=got;
    }
    // terminate it
    (*output)[len]='\0';

    if (mythtvfs.logfile) fprintf(mythtvfs.logfile,"Got %d '%s'\n",len,*output);

    return 1;
}

void backend_client_check()
{
    char *check = NULL;

    if (!backend_cmd("ANN Playback myself 0",&check)) {
        fprintf(stderr,"Cannot register with backend!\n");
        exit(1);
    }
    if (strcmp(check,"OK")) {
        fprintf(stderr,"Backend would not accept another playback client\n");
        exit(1);
    }
    if (check) free(check);
}

#define MYTH_SEP "[]:[]"
#define MYTH_PROGRAM_COLS  (mythtvfs.program_cols)

void backend_version_check()
{
    char *check = NULL, *result, *version="unknown";
    char * versions[] = { "31", "26", "15", NULL };
    char version_request[80];
    int idx;
    int found=0;
    
    for (idx = 0; !found && versions[idx]; idx++) {
        snprintf(version_request, 80, "MYTH_PROTO_VERSION %s", versions[idx]);
        if (!backend_cmd(version_request, &check)) {
            fprintf(stderr,"Cannot check backend version!\n");
            exit(1);
        }
        result = check;
        version = strstr(check,MYTH_SEP);
        *version = '\0';
        version += strlen(MYTH_SEP);
        if (strcmp(result,"ACCEPT")) {
            backend_close();
            continue;
        }

        found = 1;
        switch (atoi(version)) {
            case 15: mythtvfs.program_cols = 39; break;
            case 26: mythtvfs.program_cols = 41; break;
            case 31: mythtvfs.program_cols = 42; break;
            default:
                fprintf(stderr,"Unknown protocol version (backend=%s)\n",version);
                found = 0;
        }
    }

    if (!found) {
        fprintf(stderr,"Could not find compatible protocol version (mythtvfs=%s, backend=%s)\n",versions[0],version);
        exit(1);
    }

    if (check) free(check);
}

void backend_done()
{
    char * answer = NULL;
    backend_cmd("DONE",&answer);
    if (answer) free(answer);
    backend_close();
}

char * backend_query()
{
    char * answer = NULL;
    backend_cmd("QUERY_RECORDINGS View",&answer);

    return answer;
}

void clear_show(struct show_t * show)
{
    if (show->title) free(show->title);
    if (show->subtitle) free(show->subtitle);
    if (show->description) free(show->description);
    if (show->callsign) free(show->callsign);
    if (show->pathname) free(show->pathname);
    if (show->starttime) free(show->starttime);
    if (show->endtime) free(show->endtime);
    if (show->airdate) free(show->airdate);
    memset(show,0,sizeof(*show));
}

char * backend_pop_program(char * buffer, struct show_t * show)
{
    int col;

    for (col = 0; buffer && col < MYTH_PROGRAM_COLS; col++) {
        char * sep = NULL;
        char * data = buffer;
        if ((sep = strstr(data,MYTH_SEP))!=NULL) {
            *sep = '\0';
            sep+=strlen(MYTH_SEP);
        }

        // make sure we're not null
        if (!data) data="";

        switch (col) {
            case 0: /* unknown */ break;
            case 1: show->title = strdup(data); break;
                    //if (mythtvfs.logfile) fprintf(mythtvfs.logfile,"\ttitle '%s'\n",show->title); break;
            case 2: show->subtitle = strdup(data); break;
            case 3: show->description = strdup(data); break;
            /*
            case 4: show->category = strdup(data); break;
            case 5: show->chanid = strdup(data); break;
            case 6: show->channum = strdup(data); break;
            */
            case 7: show->callsign = strdup(data); break;
            /*
            case 8: show->channame = strdup(data); break;
            */
            case 9: show->pathname = strdup(data); break;
            /*
            case 10: show->fs_high = strdup(data); break;
            case 11: show->fs_low = strdup(data); break;
            */
            case 12: show->starttime = strdup(data); break;
            case 13: show->endtime = strdup(data); break;
            /*
            case 17: show->hostname = strdup(data); break;
            case 18: show->sourceid = strdup(data); break;
            case 19: show->cardid = strdup(data); break;
            case 20: show->inputid = strdup(data); break;
            case 21: show->recpriority = strdup(data); break;
            case 22: show->recstatus = strdup(data); break;
            case 23: show->recordid = strdup(data); break;
            case 24: show->rectype = strdup(data); break;
            case 25: show->dupin = strdup(data); break;
            case 26: show->dupmethod = strdup(data); break;
            case 27: show->recstartts = strdup(data); break;
            case 28: show->recendts = strdup(data); break;
            case 29: show->repeat = strdup(data); break;
            case 30: show->flags = strdup(data); break;
            case 31: show->recgroup = strdup(data); break;
            case 32: show->commfree = strdup(data); break;
            case 33: show->outputfilters = strdup(data); break;
            case 34: show->seriesid = strdup(data); break;
            case 35: show->programid = strdup(data); break;
            case 36: show->lastmodified = strdup(data); break;
            case 37: show->recpriority = strdup(data); break;
            */
            case 38: show->airdate = strdup(data); break;
            default:
                break;
        }
        buffer = sep;
    }

    return buffer;
}
    


static int mythtvfs_getattr(const char *path, struct stat *stbuf)
{
    int res;
#ifdef MYTHTVFS_DEBUG_FUNC
    if (mythtvfs.logfile) fprintf(mythtvfs.logfile,"%s\n",__FUNCTION__);
#endif

    res = lstat(fixup(path), stbuf);
    if(res == -1)
        return -errno;

    return 0;
}

static int mythtvfs_readlink(const char *path, char *buf, size_t size)
{
    int res;
#ifdef MYTHTVFS_DEBUG_FUNC
    if (mythtvfs.logfile) fprintf(mythtvfs.logfile,"%s\n",__FUNCTION__);
#endif

    res = readlink(fixup(path), buf, size - 1);
    if(res == -1)
        return -errno;

    buf[res] = '\0';
    return 0;
}


static int mythtvfs_readdir(const char *path, void *buf, fuse_fill_dir_t filler,
                       off_t offset, struct fuse_file_info *fi)
{
    (void) offset;
    (void) fi;
    (void) path;
    struct stat st;

#ifdef MYTHTVFS_DEBUG_FUNC
    if (mythtvfs.logfile) fprintf(mythtvfs.logfile,"%s\n",__FUNCTION__);
#endif

    memset(&st, 0, sizeof(st));

    st.st_ino++;
    st.st_mode = S_IFDIR|S_IRUSR|S_IRGRP|S_IROTH|S_IXUSR|S_IXGRP|S_IXOTH;
    filler(buf, ".", &st, 0);
    st.st_ino++;
    filler(buf, "..", &st, 0);

    struct show_t show;
    memset(&show, 0, sizeof(show));

    char * buffer = backend_query();
    char * data = buffer;
    while ( (data = backend_pop_program(data,&show)) ) {
        char * metaname = NULL;

        /*
        printf("Title   : %s\n",show.title);
        printf("Subtitle: %s\n",show.subtitle);
        printf("Callsign: %s\n",show.callsign);
        printf("Pathname: %s\n",show.pathname);
        printf("Start   : %s\n",show.starttime);
        printf("End     : %s\n",show.endtime);
        printf("Airdate : %s\n",show.airdate);
        printf("\n");
        */

        char * filename;
        if (!(filename = rindex(show.pathname,'/'))) {
            clear_show(&show);
            continue;
        }

        filename++;
        /* make sure we can read the file */
        if (access(aim_to_base(filename),R_OK)) {
            if (mythtvfs.logfile) fprintf(mythtvfs.logfile,"Skipping '%s'\n",filename);
            clear_show(&show);
            continue;
        }

        if ( mythtvfs.short_display_flag ) {
            // non-standard (SHORT) filename display
            char duration[32];
            char start[32];
            char air[32];
            time_t start_time = atoi(show.starttime);
            time_t end_time = atoi(show.endtime);
            time_t air_time = atoi(show.airdate);
            snprintf(duration,32,"%ld",(end_time-start_time));
            strftime(start,32,"%c",localtime(&start_time));
            strftime(air,32,"%Y-%m-%d",localtime(&air_time));

            int len = 2+strlen(show.title)+
                2+strlen(air)+
                2+strlen(show.subtitle)+
                2+strlen(start)+
                2+strlen(show.callsign)+
                2+strlen(duration)+
                strlen(filename)+
                4+ /* ".mpg" */
                1; /* NULL */
            if (!(metaname=malloc(len))) {
                perror("malloc");
                exit(1);
            }
            snprintf(metaname,len,"{%s}{%s}{%s}{%s}{%s}{%s}%s.mpg",
                show.title,
                air,
                show.subtitle,
                start,
                show.callsign,
                duration,
                filename);
        } else {
            // standard (LONG) filename display
            char duration[32];
            char start[32];
            char air[32];
            time_t start_time = atoi(show.starttime);
            time_t end_time = atoi(show.endtime);
            time_t air_time = atoi(show.airdate);
            snprintf(duration,32,"%ld",(end_time-start_time));
            strftime(start,32,"%I.%M %p %a %b %d, %Y",localtime(&start_time));
            strftime(air,32,"%Y-%m-%d",localtime(&air_time));

            int len = 2+strlen(show.title)+
                      2+strlen(air)+
                      2+strlen(show.subtitle)+
                      2+strlen(start)+
                      2+strlen(show.callsign)+
                      2+strlen(duration)+
                      2+strlen(show.description)+
                      strlen(filename)+
                      4+ /* ".mpg" */
                      1; /* NULL */
            if (!(metaname=malloc(len))) {
                perror("malloc");
                exit(1);
            }
            snprintf(metaname,len,"{%s}{%s}{%s}{%s}{%s}{%s}{%s}%s.mpg",
                     show.title,
                     air,
                     show.subtitle,
                     start,
                     show.callsign,
                     duration,
                     show.description,
                     filename);
        }

        st.st_ino++;
        st.st_mode = S_IFREG|S_IRUSR|S_IRGRP|S_IROTH;

        filler(buf, metaname, &st, 0);
        free(metaname);

        clear_show(&show);
    }
    free(buffer);

    return 0;
}

static int mythtvfs_mknod(const char *path, mode_t mode, dev_t rdev)
{
    int res;
#ifdef MYTHTVFS_DEBUG_FUNC
    if (mythtvfs.logfile) fprintf(mythtvfs.logfile,"%s\n",__FUNCTION__);
#endif

    res = mknod(fixup(path), mode, rdev);
    if(res == -1)
        return -errno;

    return 0;
}

static int mythtvfs_mkdir(const char *path, mode_t mode)
{
    int res;
#ifdef MYTHTVFS_DEBUG_FUNC
    if (mythtvfs.logfile) fprintf(mythtvfs.logfile,"%s\n",__FUNCTION__);
#endif

    res = mkdir(fixup(path), mode);
    if(res == -1)
        return -errno;

    return 0;
}

static int mythtvfs_unlink(const char *path)
{
    int res;
#ifdef MYTHTVFS_DEBUG_FUNC
    if (mythtvfs.logfile) fprintf(mythtvfs.logfile,"%s\n",__FUNCTION__);
#endif

    res = unlink(fixup(path));
    if(res == -1)
        return -errno;

    return 0;
}

static int mythtvfs_rmdir(const char *path)
{
    int res;
#ifdef MYTHTVFS_DEBUG_FUNC
    if (mythtvfs.logfile) fprintf(mythtvfs.logfile,"%s\n",__FUNCTION__);
#endif

    res = rmdir(fixup(path));
    if(res == -1)
        return -errno;

    return 0;
}

static int mythtvfs_symlink(const char *from, const char *to)
{
    int res;
    char * fixed_from = NULL;
    char * fixed_to = NULL;
#ifdef MYTHTVFS_DEBUG_FUNC
    if (mythtvfs.logfile) fprintf(mythtvfs.logfile,"%s\n",__FUNCTION__);
#endif

    fixed_from = strdup(fixup(from));
    fixed_to   = strdup(fixup(to));

    res = symlink(fixed_from, fixed_to);

    free(fixed_from);
    free(fixed_to);

    if(res == -1)
        return -errno;

    return 0;
}

static int mythtvfs_rename(const char *from, const char *to)
{
    int res;
    char * fixed_from = NULL;
    char * fixed_to = NULL;
#ifdef MYTHTVFS_DEBUG_FUNC
    if (mythtvfs.logfile) fprintf(mythtvfs.logfile,"%s\n",__FUNCTION__);
#endif

    fixed_from = strdup(fixup(from));
    fixed_to   = strdup(fixup(to));

    res = rename(fixed_from, fixed_to);

    free(fixed_from);
    free(fixed_to);

    if(res == -1)
        return -errno;

    return 0;
}

static int mythtvfs_link(const char *from, const char *to)
{
    int res;
    char * fixed_from = NULL;
    char * fixed_to = NULL;
#ifdef MYTHTVFS_DEBUG_FUNC
    if (mythtvfs.logfile) fprintf(mythtvfs.logfile,"%s\n",__FUNCTION__);
#endif

    fixed_from = strdup(fixup(from));
    fixed_to   = strdup(fixup(to));

    res = link(from, to);

    free(fixed_from);
    free(fixed_to);

    if(res == -1)
        return -errno;

    return 0;
}

static int mythtvfs_chmod(const char *path, mode_t mode)
{
    int res;
#ifdef MYTHTVFS_DEBUG_FUNC
    if (mythtvfs.logfile) fprintf(mythtvfs.logfile,"%s\n",__FUNCTION__);
#endif

    res = chmod(fixup(path), mode);
    if(res == -1)
        return -errno;

    return 0;
}

static int mythtvfs_chown(const char *path, uid_t uid, gid_t gid)
{
    int res;
#ifdef MYTHTVFS_DEBUG_FUNC
    if (mythtvfs.logfile) fprintf(mythtvfs.logfile,"%s\n",__FUNCTION__);
#endif

    res = lchown(fixup(path), uid, gid);
    if(res == -1)
        return -errno;

    return 0;
}

static int mythtvfs_truncate(const char *path, off_t size)
{
    int res;
#ifdef MYTHTVFS_DEBUG_FUNC
    if (mythtvfs.logfile) fprintf(mythtvfs.logfile,"%s\n",__FUNCTION__);
#endif

    res = truncate(fixup(path), size);
    if(res == -1)
        return -errno;

    return 0;
}

static int mythtvfs_utime(const char *path, struct utimbuf *buf)
{
    int res;
#ifdef MYTHTVFS_DEBUG_FUNC
    if (mythtvfs.logfile) fprintf(mythtvfs.logfile,"%s\n",__FUNCTION__);
#endif

    res = utime(fixup(path), buf);
    if(res == -1)
        return -errno;

    return 0;
}


static int mythtvfs_open(const char *path, struct fuse_file_info *fi)
{
    int res;
#ifdef MYTHTVFS_DEBUG_FUNC
    if (mythtvfs.logfile) fprintf(mythtvfs.logfile,"%s\n",__FUNCTION__);
#endif

    res = open(fixup(path), fi->flags);
    if(res == -1)
        return -errno;

    close(res);
    return 0;
}

static int mythtvfs_read(const char *path, char *buf, size_t size, off_t offset,
                    struct fuse_file_info *fi)
{
    int fd;
    int res;
#ifdef MYTHTVFS_DEBUG_FUNC
    if (mythtvfs.logfile) fprintf(mythtvfs.logfile,"%s\n",__FUNCTION__);
#endif

    (void) fi;
    fd = open(fixup(path), O_RDONLY);
    if(fd == -1)
        return -errno;

    res = pread(fd, buf, size, offset);
    if(res == -1)
        res = -errno;

    close(fd);
    return res;
}

static int mythtvfs_write(const char *path, const char *buf, size_t size,
                     off_t offset, struct fuse_file_info *fi)
{
    int fd;
    int res;
#ifdef MYTHTVFS_DEBUG_FUNC
    if (mythtvfs.logfile) fprintf(mythtvfs.logfile,"%s\n",__FUNCTION__);
#endif

    (void) fi;
    fd = open(fixup(path), O_WRONLY);
    if(fd == -1)
        return -errno;

    res = pwrite(fd, buf, size, offset);
    if(res == -1)
        res = -errno;

    close(fd);
    return res;
}

#if FUSE_VERSION >= 25
# define whichever_stat    statvfs
#else
# define whichever_stat    statfs
#endif

static int mythtvfs_statfs(const char *path, struct whichever_stat *stbuf)
{
    int res;
#ifdef MYTHTVFS_DEBUG_FUNC
    if (mythtvfs.logfile) fprintf(mythtvfs.logfile,"%s\n",__FUNCTION__);
#endif

    res = whichever_stat(fixup(path), stbuf);
    if(res == -1)
        return -errno;

    return 0;
}

static int mythtvfs_release(const char *path, struct fuse_file_info *fi)
{
    /* Just a stub.  This method is optional and can safely be left
       unimplemented */
#ifdef MYTHTVFS_DEBUG_FUNC
    if (mythtvfs.logfile) fprintf(mythtvfs.logfile,"%s\n",__FUNCTION__);
#endif

    (void) path;
    (void) fi;
    return 0;
}

static int mythtvfs_fsync(const char *path, int isdatasync,
                     struct fuse_file_info *fi)
{
    /* Just a stub.  This method is optional and can safely be left
       unimplemented */
#ifdef MYTHTVFS_DEBUG_FUNC
    if (mythtvfs.logfile) fprintf(mythtvfs.logfile,"%s\n",__FUNCTION__);
#endif

    (void) path;
    (void) isdatasync;
    (void) fi;
    return 0;
}

#ifdef HAVE_SETXATTR
/* xattr operations are optional and can safely be left unimplemented */
static int mythtvfs_setxattr(const char *path, const char *name, const char *value,
                        size_t size, int flags)
{
#ifdef MYTHTVFS_DEBUG_FUNC
    if (mythtvfs.logfile) fprintf(mythtvfs.logfile,"%s\n",__FUNCTION__);
#endif
    int res = lsetxattr(fixup(path), name, value, size, flags);
    if(res == -1)
        return -errno;
    return 0;
}

static int mythtvfs_getxattr(const char *path, const char *name, char *value,
                    size_t size)
{
#ifdef MYTHTVFS_DEBUG_FUNC
    if (mythtvfs.logfile) fprintf(mythtvfs.logfile,"%s\n",__FUNCTION__);
#endif
    int res = lgetxattr(fixup(path), name, value, size);
    if(res == -1)
        return -errno;
    return res;
}

static int mythtvfs_listxattr(const char *path, char *list, size_t size)
{
#ifdef MYTHTVFS_DEBUG_FUNC
    if (mythtvfs.logfile) fprintf(mythtvfs.logfile,"%s\n",__FUNCTION__);
#endif
    int res = llistxattr(fixup(path), list, size);
    if(res == -1)
        return -errno;
    return res;
}

static int mythtvfs_removexattr(const char *path, const char *name)
{
#ifdef MYTHTVFS_DEBUG_FUNC
    if (mythtvfs.logfile) fprintf(mythtvfs.logfile,"%s\n",__FUNCTION__);
#endif
    int res = lremovexattr(fixup(path), name);
    if(res == -1)
        return -errno;
    return 0;
}
#endif /* HAVE_SETXATTR */

static struct fuse_operations mythtvfs_oper = {
    .getattr     = mythtvfs_getattr,
    .readlink    = mythtvfs_readlink,
    .readdir     = mythtvfs_readdir,
    .mknod       = mythtvfs_mknod,
    .mkdir       = mythtvfs_mkdir,
    .symlink     = mythtvfs_symlink,
    .unlink      = mythtvfs_unlink,
    .rmdir       = mythtvfs_rmdir,
    .rename      = mythtvfs_rename,
    .link        = mythtvfs_link,
    .chmod       = mythtvfs_chmod,
    .chown       = mythtvfs_chown,
    .truncate    = mythtvfs_truncate,
    .utime       = mythtvfs_utime,
    .open        = mythtvfs_open,
    .read        = mythtvfs_read,
    .write       = mythtvfs_write,
    .statfs      = mythtvfs_statfs,
    .release     = mythtvfs_release,
    .fsync       = mythtvfs_fsync,
#ifdef HAVE_SETXATTR
    .setxattr    = mythtvfs_setxattr,
    .getxattr    = mythtvfs_getxattr,
    .listxattr   = mythtvfs_listxattr,
    .removexattr = mythtvfs_removexattr,
#endif
};

static int mythtvfs_opt_proc(void *data, const char *arg, int key,
                             struct fuse_args *outargs)
{
    (void) data;
    
    switch (key) {
    case FUSE_OPT_KEY_OPT:
        return 1;

    case FUSE_OPT_KEY_NONOPT:
        if (!mythtvfs.base_path) {
            mythtvfs.base_path = strdup(arg);
            mythtvfs.base_len  = strlen(mythtvfs.base_path);
            return 0;
        }
        
        return 1;

    case KEY_HELP:
        usage(outargs->argv[0]);
        fuse_opt_add_arg(outargs, "-ho");
        fuse_main(outargs->argc, outargs->argv, &mythtvfs_oper, NULL);
        exit(1);

    case KEY_VERSION:
        fprintf(stderr, "MythTVfs version %s\n", PACKAGE_VERSION);
#if FUSE_VERSION >= 25
        fuse_opt_add_arg(outargs, "--version");
        fuse_main(outargs->argc, outargs->argv, &mythtvfs_oper, NULL);
#endif
        exit(0);
    
    default:
        fprintf(stderr, "internal error\n");
        abort();
    }
}

int main(int argc, char *argv[])
{
    int rc;
    struct fuse_args args = FUSE_ARGS_INIT(argc, argv);

    // setup "defaults"
    mythtvfs.base_path = NULL;
    mythtvfs.host      = NULL;
    mythtvfs.port      = 6543;
    mythtvfs.short_display_flag = 0;
    mythtvfs.backend   = -1;
    mythtvfs.base_len  = 0;
    mythtvfs.logfile   = NULL;
    mythtvfs.logfile_name = NULL;
    mythtvfs.program_cols = 0;

    // process options
    if (fuse_opt_parse(&args,&mythtvfs,mythtvfs_opts,mythtvfs_opt_proc) == -1)
        exit(1);

    // validate required options (would be nicer to dump usage...)
    if (!mythtvfs.host) {
        fprintf(stderr,"Error: backend not specified!  Run with --help\n");
        exit(1);
    }
    if (!mythtvfs.base_path) {
        fprintf(stderr,"Error: original path not specified!  Run with --help\n");
        exit(1);
    }

    if (mythtvfs.logfile_name) {
        if (!(mythtvfs.logfile = fopen(mythtvfs.logfile_name,"a"))) {
            perror(mythtvfs.logfile_name);
            exit(1);
        }
        setvbuf(mythtvfs.logfile,NULL,_IONBF,0);
    }

    umask(0);

    // attempt to talk to mythtv backend before daemonizing
    backend_version_check();

    rc = fuse_main(args.argc, args.argv, &mythtvfs_oper, NULL);

    backend_done();
    if (mythtvfs.logfile) fclose(mythtvfs.logfile);
    return rc;
}

