# -*- coding: utf-8 -*-
#
# «core» - Mythbuntu Control Centre base class
#
# Portions from GDebi copyright Michael Vogt
# Copyright (C) 2007, Mario Limonciello, for Mythbuntu
#
#
# Mythbuntu is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free
# Software Foundation; either version 2 of the License, or at your option)
# any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this application; if not, write to the Free Software Foundation, Inc., 51
# Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
##################################################################################

import os
import sys
import pygtk
import re
import string
pygtk.require("2.0")

import gtk
import gtk.glade

#MySQL testing support
import MySQLdb

#Case for change
from MythbuntuControlCentre.changer import Applicator

#To Store Current/Old settings
import ConfigParser

#Checking package status cleanly
import warnings
warnings.filterwarnings("ignore", "apt API not stable yet", FutureWarning)
import apt_pkg

# for shutil.rmtree
import shutil

#Package reloading support
from softwareproperties.gtk.SoftwarePropertiesGtk import DialogCacheOutdated

from aptsources import distro
from aptsources.sourceslist import SourcesList, SourceEntry
from softwareproperties.SoftwareProperties import SoftwareProperties

import commands

#mythbuntu common functionality
from mythbuntu_common.lirc import LircHandler
from mythbuntu_common.debconftalk import debconftalk
from mythbuntu_common.dictionaries import *

# needed for paul cannon's interface magic
# eg http://aspn.activestate.com/ASPN/Cookbook/Python/Recipe/439093
import socket
import fcntl
import struct
import array

def get_ip_address(ifname):
    s = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
    return socket.inet_ntoa(fcntl.ioctl(
        s.fileno(),
        0x8915,  # SIOCGIFADDR
        struct.pack('256s', ifname[:15])
    )[20:24])

GLADEDIR = '/usr/share/mythbuntu-control-centre/glade'

# global variables
ltsp_base_dir='/opt/ltsp/'

# Define what tabs of the UI we want to load.
SUBTABS = [
    "tab_system_roles",
    "tab_mythtv_configuration",
    "tab_applications_plugins",
    "tab_proprietary_drivers",
    "tab_proprietary_codecs",
    "tab_remote_control",
    "tab_system_services",
    "tab_themes",
    "tab_artwork_behavior",
    "tab_network_shares",
    "tab_advanced",
    "tab_diskless"
]

class ControlCentre():
    def __init__(self):
        """Initializes the control center app"""
        def add_subpage(self, tabs, name):
            """Inserts a subtab into the notebook.  This assumes the file
            shares the same base name as the page you are looking for."""
            gladefile = GLADEDIR + '/' + name + '.glade'
# See http://faq.pygtk.org/index.py?req=show&file=faq22.002.htp
# for internationalisation support
            gladexml = gtk.glade.XML(gladefile, name, 'mythbuntu-control-centre')
            widget = gladexml.get_widget(name)
            tabs.append_page(widget)
            add_widgets(self, gladexml)
            gladexml.signal_autoconnect(self)

        def add_widgets(self, glade):
            """Makes all widgets callable by the toplevel."""
            for widget in glade.get_widget_prefix(""):
                setattr(self, widget.get_name(), widget)
                # We generally want labels to be selectable so that people can
                # easily report problems in them
                # (https://launchpad.net/bugs/41618), but GTK+ likes to put
                # selectable labels in the focus chain, and I can't seem to turn
                # this off in glade and have it stick. Accordingly, make sure
                # labels are unfocusable here.
                if isinstance(widget, gtk.Label):
                    widget.set_property('can-focus', False)
        def populate_lirc(self):
            """Fills the lirc pages with the appropriate data"""
            self.remote_count = 0
            self.transmitter_count = 0
            self.lirc=LircHandler()
            for item in self.lirc.get_possible_devices("remote"):
                self.remote_list.append_text(item)
                self.remote_count = self.remote_count + 1
            for item in self.lirc.get_possible_devices("transmitter"):
                self.transmitter_list.append_text(item)
                self.transmitter_count = self.transmitter_count + 1
            self.remote_list.set_active(0)
            self.transmitter_list.set_active(0)

        def populate_user_ids(self):
            """Returns all valid user id's for autostart"""
            self.user_count=0
            in_f=open("/etc/passwd")
            for line in in_f:
                try:
                    fields =  string.split(line,":")
                    id=int(fields[2])
                    if id > 999 and id < 10000:
                        self.autologin_user.append_text(fields[0])
                        self.user_count+=1
                except IndexError:
                    continue 
            #double check to see if the user 'mythtv' is in usage too.
            if os.path.exists("/etc/gdm/gdm-cdd.conf"):
                in_f=open("/etc/gdm/gdm-cdd.conf")
                for line in in_f:
                    if re.compile("^AutomaticLogin=").search(line):
                        user=string.split(string.split(line,"=")[1],'\n')[0]
                        if user == "mythtv":
                            self.autologin_user.append_text(user)
                            self.user_count+=1
                        break

        # Initialize apt
        apt_pkg.init()
        self.cache = apt_pkg.GetCache()

        #Initialize configuration comparer
        self.config = ConfigParser.ConfigParser()
        self.config.add_section("mythbuntu")

        # load the main interface
        self.glade = gtk.glade.XML(GLADEDIR + '/' + 'mythbuntu_control_centre.glade')
        add_widgets(self, self.glade)

        # load subtabs of interface
        tabs = self.glade.get_widget("tabs")
        for tab in SUBTABS:
            add_subpage(self, tabs, tab)
        self.glade.signal_autoconnect(self)

        #Populate statically detected interface content
        populate_lirc(self)
        populate_user_ids(self)

        #debconf communicator
        self.communicator = debconftalk()

        #Populate dynamically detected interface content
        self.revert_gui()


    def destroy(self, widget, data=None):
        gtk.main_quit()

    def display_error(self,message):
        """Displays an error message"""
        dialog = gtk.MessageDialog(self.main_window, gtk.DIALOG_MODAL,
                                   gtk.MESSAGE_ERROR, gtk.BUTTONS_CLOSE,
                                   message)
        dialog.run()
        dialog.hide()

    def run(self):
        self.main_window.show()
        gtk.main()


#Helper Functions
    def find_case_for_change(self):
        """Finds and marks all changes made within the interface"""

        to_install = []
        to_remove = []
        to_reconfigure = {}
        text = ""

        #Gather Information
        ##Backend Role
        old_backend=self.config.get("mythbuntu","backend")
        if self.primary_backend_radio.get_active():
            new_backend="primary"
        elif self.secondary_backend_radio.get_active():
            new_backend="secondary"
        else:
            new_backend="none"
        ##Frontend Role
        old_frontend=self.config.get("mythbuntu","frontend")
        if self.frontend_radio.get_active():
            new_frontend="frontend"
        else:
            new_frontend="none"
        #mysql.txt
        old_user=self.config.get("mythbuntu","mythtv_mysql_user")
        new_user=self.mysql_user.get_text()
        old_pass=self.config.get("mythbuntu","mythtv_mysql_password")
        new_pass=self.mysql_pass.get_text()
        old_database=self.config.get("mythbuntu","mythtv_mysql_database")
        new_database=self.mysql_database.get_text()
        old_server=self.config.get("mythbuntu","mythtv_mysql_server")
        new_server=self.mysql_server.get_text()
        # mythweb auth
        # shouldn't we get old values from debconf instead? :(
        old_mythwebauthenable=self.config.get("mythbuntu","mythweb_auth_activated")
        if old_mythwebauthenable=="False":
            old_mythwebauthenable=False
        else:
            old_mythwebauthenable=True
        new_mythwebauthenable=self.mythweb_password_checkbutton.get_active()
        old_mythwebauthuser=self.config.get("mythbuntu","mythweb_auth_username")
        new_mythwebauthuser=self.mythweb_username.get_text()
        old_mythwebauthpass=self.config.get("mythbuntu","mythweb_auth_password")
        new_mythwebauthpass=self.mythweb_password.get_text()

        #remote control
        old_remote=self.config.get("mythbuntu","remote")
        if old_remote == "Custom":
            old_remote_modules=self.config.get("mythbuntu","remote_modules")
            old_remote_driver=self.config.get("mythbuntu","remote_driver")
            old_remote_device=self.config.get("mythbuntu","remote_device")
            old_remote_lircd_conf=self.config.get("mythbuntu","remote_lircd_conf")
        new_remote=self.remote_list.get_active_text()
        if new_remote == "Custom":
            new_remote_modules=self.remote_modules.get_text()
            new_remote_device=self.remote_device.get_text()
            new_remote_driver=self.remote_driver.get_text()
            new_remote_lircd_conf=self.browse_remote_lircd_conf.get_filename()
        #transmitter control
        old_transmitter=self.config.get("mythbuntu","transmitter")
        if old_transmitter == "Custom":
            old_transmitter_modules=self.config.get("mythbuntu","transmitter_modules")
            old_transmitter_driver=self.config.get("mythbuntu","transmitter_driver")
            old_transmitter_device=self.config.get("mythbuntu","transmitter_device")
            old_transmitter_lircd_conf=self.config.get("mythbuntu","transmitter_lircd_conf")
        new_transmitter=self.transmitter_list.get_active_text()
        if new_transmitter == "Custom":
            new_transmitter_modules=self.transmitter_modules.get_text()
            new_transmitter_device=self.transmitter_device.get_text()
            new_transmitter_driver=self.transmitter_driver.get_text()
            new_transmitter_lircd_conf=self.browse_transmitter_lircd_conf.get_filename()
        #Medibuntu
        old_medibuntu=self.config.get("mythbuntu","medibuntu")
        if old_medibuntu=="False":
            old_medibuntu=False
        else:
            old_medibuntu=True
        new_medibuntu=self.enable_medibuntu.get_active()

        #Automatic Login
        old_autologin=self.config.get("mythbuntu","autologin")
        old_autologin_user=self.config.get("mythbuntu","autologin_user")
        if old_autologin=="False":
            old_autologin=False
        else:
            old_autologin=True
        new_autologin=self.enableautologin.get_active()
        new_autologin_user=self.autologin_user.get_active_text()
        #Artwork
        old_artwork=self.config.get("mythbuntu","artwork")
        if old_artwork=="False":
            old_artwork=False
        else:
            old_artwork=True
        new_artwork=self.enable_artwork.get_active()

        #Generate text and action lists
        ##Roles
        if old_backend != new_backend or old_frontend != new_frontend:
            text+="System Roles:\n"
        ###Backend
        if old_backend != new_backend:
            if new_backend == "none":
                text+="   Remove " + old_backend + " backend role from system.\n"
                if old_backend == "primary":
                    to_remove.append("mythtv-backend-master")
                    to_remove.append("mythtv-database")
                to_remove.append("mythtv-backend")
                to_remove.append("mythweb")
            elif old_backend == "none":
                text+="   Add " + new_backend + " backend role to the system.\n"
                if new_backend == "primary":
                    to_install.append("mythtv-backend-master")
                else:
                    to_install.append("mythtv-backend")
            else:
                text+="   Switch " + old_backend + " backend role into a " + new_backend + " role.\n"
                if new_backend == "primary":
                    to_install.append("mythtv-backend-master")
                else:
                    to_remove.append("mythtv-backend-master")
                    to_remove.append("mythtv-database")
        if new_backend == "primary" or new_backend == "secondary":
            to_reconfigure["mythtv-backend"]=True
        else:
            to_reconfigure["mythtv-backend"]=False

        ##Desktop Roles
        seen_title = False
        list=get_desktop_dictionary(self)
        for item in list:
            result=self.config.get("mythbuntu",item)
            active=list[item].get_active()
            if result=="True" and not active:
                if not seen_title:
                    text+="Desktop Roles:\n"
                    seen_title=True
                to_remove.append(item)
                text+="   Remove " + item + " from system.\n"
            elif result=="False" and active:
                if not seen_title:
                    text+="Desktop Roles:\n"
                    seen_title=True
                to_install.append(item)
                text+="   Add " + item + " to system.\n"

        ###Frontend
        if old_frontend != new_frontend:
            if new_frontend == "none":
                text+="   Remove frontend role from system.\n"
                to_remove.append("mythtv-frontend")
                to_remove.append("mythtv-themes")
            else:
                text+="   Add frontend role to system.\n"
                to_install.append("mythtv-frontend")
                to_install.append("mythtv-themes")
        if new_frontend == "frontend":
            to_reconfigure["mythtv-frontend"]=True
        else:
            to_reconfigure["mythtv-frontend"]=False
        ##MythTV Configuration
        if old_pass != new_pass or old_user != new_user or old_server != new_server or old_database != new_database:
            text+="MythTV Configuration\n"
            to_reconfigure["mythtv_mysql_pass"]=new_pass
            to_reconfigure["mythtv_mysql_database"]=new_database
            to_reconfigure["mythtv_mysql_user"]=new_user
            to_reconfigure["mythtv_mysql_server"]=new_server
            if old_user != new_user:
                text +="   Change MythTV MySQL user name.\n"
            if old_pass != new_pass:
                text +="   Change MythTV MySQL user password.\n"
            if old_database != new_database:
                text +="   Change MythTV MySQL database.\n"
            if old_server != new_server:
                text +="   Change MythTV MySQL server.\n"

        #Checkbox items (Plugins, Media Players, Themes, And Codecs, MythWeb auth)
        seen_title=False
        mythweb_do_not_configure=False
        if old_medibuntu != new_medibuntu:
            if not seen_title:
                text+="Plugins, Media Players, Themes and Codecs:\n"
                seen_title=True
            if new_medibuntu:
                text+="   Enabling medibuntu repository.\n"
                to_reconfigure["medibuntu"]=True
            else:
                text+="   Disabling medibuntu repository.\n"
                to_reconfigure["medibuntu"]=False

        for list in get_official_theme_dictionary(self), get_community_theme_dictionary(self), get_media_app_dictionary(self), get_nonfree_dictionary(self), get_frontend_plugin_dictionary(self), get_backend_plugin_dictionary(self):
            for item in list:
                result=self.config.get("mythbuntu",item)
                active=list[item].get_active()
                # if the plugin was previously installed and is not checked anymore in the GUI, remove it
                if result=="True" and not active:
                    if not seen_title:
                        text+="Plugins, Media Players, Themes, and Codecs:\n"
                        seen_title=True
                    to_remove.append(item)
                    if item == "mplayer":
                        to_remove.append("mplayer-nogui")
                    if item == "mythweb":
                       # we mustn't reconfigure mythweb auth if it's to be removed
                        mythweb_do_not_configure = True
                    text+="   Remove " + item + " from system.\n"
                elif result=="False" and active:
                    if not seen_title:
                        text+="Plugins, Media Players, Themes, and Codecs:\n"
                        seen_title=True
                    to_install.append(item)
                    text+="   Add " + item + " to system.\n"
                if item == "mplayer" or item == "xine-ui" or item == "vlc" or\
                   item == "mythvideo" or item == "mythgallery" or item == "mythmusic" or\
                   item == "mythbuntu_apple_trailers":
                    to_reconfigure[item]=active
        # mythweb auth
        # we only want to change anything if mythweb is installed
        # maybe we should wrap other mythweb code as well, but this should be sufficient
        if self.query_installed("mythweb") and not mythweb_do_not_configure:
            if old_mythwebauthenable != new_mythwebauthenable or old_mythwebauthuser != new_mythwebauthuser  or old_mythwebauthpass != new_mythwebauthpass:
                # FIXME: the following strings are likely not translatable yet
                text+="MythWeb Authentication\n"
                to_reconfigure["mythweb_auth_activated"]=new_mythwebauthenable
                to_reconfigure["mythweb_auth_username"]=new_mythwebauthuser
                to_reconfigure["mythweb_auth_password"]=new_mythwebauthpass
                if old_mythwebauthenable != new_mythwebauthenable:
                    text +="   Enable/Disable MythWeb authentication\n"
                if old_mythwebauthuser  != new_mythwebauthuser:
                    text +="   Change user name for MythWeb access\n"
                # old_mythwebauthpas should always be an empty string since we clear that debconf question in mythweb.postinst
                if old_mythwebauthpass != new_mythwebauthpass:
                    text +="   Change password for MythWeb access\n"

        seen_tweaks_title = False
        # mysql tweaks
        # yes, that's a string comparison here
        old_mysql_tweaks = self.config.get("mythbuntu","mysql_tweaks_enabled")
        if self.enable_mysql_tweaks.get_active() == True:
             # remember, config.get doesn't like booleans, we're working with strings here
             new_mysql_tweaks = "True"
        else:
             new_mysql_tweaks = "False"
        if old_mysql_tweaks != new_mysql_tweaks:
            to_reconfigure["mysql_tweaks_enabled"]=new_mysql_tweaks
            if not seen_tweaks_title:
                text+="Various Tweaks/Optimizations:\n"
                seen_tweaks_title = True
            text +="   Enable/Disable MySQL tweaks\n"

        old_defrag = self.config.get("mythbuntu","enable_nightly_defrag")
        # again, remember: we're converting boolean to strings here
        if self.enable_nightly_defrag.get_active() == True:
            new_defrag = "True"
        else:
            new_defrag = "False"
        if new_defrag != old_defrag:
            to_reconfigure["enable_nightly_defrag"]=new_defrag
            if not seen_tweaks_title:
                text+="Various Tweaks/Optimizations:\n"
                seen_tweaks_title = True
            text +="   Enable/Disable daily XFS cron job\n"
            if new_defrag == "True":
                # xfsdump contains xfs_fsr
                # BTW: we also need ionice which is in schedutils. ubuntu-minimal depends on schedutils so we should be safe
                if not self.query_installed("xfsdump"):
                    to_install.append("xfsdump")
            # if new_defrag == "False":
                # It feels wrong to remove xfsdump because the user might need it for other things
                # if self.query_installed("xfsdump"):
                #    to_remove.append("xfsdump")


        old_mysql_repair = self.config.get("mythbuntu","enable_mysql_repair")
        if self.enable_mysql_repair.get_active() == True:
            new_mysql_repair = "True"
        else:
            new_mysql_repair = "False"

        if old_mysql_repair != new_mysql_repair:
            to_reconfigure["new_mysql_repair"]=new_mysql_repair
            if not seen_tweaks_title:
                text+="Various Tweaks/Optimizations:\n"
                seen_tweaks_title = True
            text += "   Enable/Disable mysql repair/optimization cron job\n"

        old_ivtv_tweak = self.config.get("mythbuntu","enable_ivtv_tweak")
        if self.enable_ivtv_tweak.get_active() == True:
            new_ivtv_tweak = "True"
        else:
            new_ivtv_tweak = "False"

        if old_ivtv_tweak != new_ivtv_tweak:
            to_reconfigure["enable_ivtv_tweak"]=new_ivtv_tweak
            if not seen_tweaks_title:
                text+="Various Tweaks/Optimizations:\n"
                seen_tweaks_title = True
            text += "   Enable/Disable IVTV buffer overflow tweak\n"

        old_rtc_tweak = self.config.get("mythbuntu","enable_rtc_tweak")
        if self.enable_rtc_tweak.get_active() == True:
            new_rtc_tweak = "True"
        else:
            new_rtc_tweak = "False"

        if old_rtc_tweak != new_rtc_tweak:
            to_reconfigure["enable_rtc_tweak"]=new_rtc_tweak
            if not seen_tweaks_title:
                text+="Various Tweaks/Optimizations:\n"
                seen_tweaks_title = True
            text += "   Enable/Disable RTC Video Tweak\n"

        # Mythbuntu Diskless Server
        seen_title=False
        list=get_diskless_dictionary(self)
        for item in list:
            result=self.config.get("mythbuntu",item)
            active=list[item].get_active()
            # if the plugin was previously installed and is not checked anymore in the GUI, remove it
            if result=="True" and not active:
                if not seen_title:
                    text+="Mythbuntu Diskless Server:\n"
                    seen_title=True
                to_remove.append(item)
                text+="   Remove " + item + " from system.\n"
            elif result=="False" and active:
                if not seen_title:
                    text+="Mythbuntu Diskless Server:\n"
                    seen_title=True
                try:
                    to_reconfigure['master-backend-address']=self.get_db_setting('MasterServerIP',None)
                #except _mysql_exceptions.OperationalError ?!
                except:
                    basemsg = "I couldn\'t talk to your MySQL server. Is it running?\nIs the connection information in the 'MythTV Configuration' tab correct?"
                    if item == "mythbuntu-diskless-server":
                        self.display_error("Error while configuring mythbuntu-diskless-server.\nPlease update /etc/exports manually or run 'sudo dpkg-reconfigure mythbuntu-diskless-server'\n" + basemsg)
                    elif item == "dhcp3-server":
                        self.display_error("Error while configuring dhcp3-server.\nPlease create /etc/ltsp/dhcpd.conf manually.\n" + basemsg)

                to_install.append(item)
                text+="   Add " + item + " to system.\n"
                # we only have two items and we want to "reconfigure" them both on installation
                # if item == "mythbuntu-diskless-server":
                to_reconfigure[item]=active


        #build or delete diskless images
        if self.diskless_build_image_button.get_active() or self.diskless_delete_image_button.get_active():
            if not seen_title:
                text+="Mythbuntu Diskless Server:\n"
                seen_title=True

            if self.diskless_build_image_button.get_active():
                to_reconfigure["diskless_build"]=True
                to_reconfigure["diskless_sign"]=self.diskless_unsigned_packages.get_active()
                to_reconfigure["diskless_arch"]=self.diskless_arch.get_active_text()
                text+="   Build " + to_reconfigure["diskless_arch"]  + " diskless image.\n"
            else:
                to_reconfigure["diskless_delete"]=True
                to_reconfigure["diskless_arch"]=self.diskless_arch.get_active_text()
                text+="   Delete " + to_reconfigure["diskless_arch"]  + " diskless image.\n"

        #diskless pen drive
        if self.diskless_write_drive_button.get_active():
            if not seen_title:
                text+="Mythbuntu Diskless Server:\n"
                seen_title=True
            to_reconfigure["diskless_pen"]=True
            to_reconfigure["diskless_arch"]=self.diskless_chroot.get_active_text()
            to_reconfigure["diskless_block_device"]=self.diskless_pendrive.get_active_text()
            to_reconfigure["diskless_nbd_device"]=self.diskless_nbd_server_nic.get_active_text()
            if self.diskless_wol_checkbox.get_active():
                to_reconfigure["diskless_wol"]=self.diskless_wol_entry.get_text()
            text+="   Write " + to_reconfigure["diskless_arch"]  + " flash drive image.\n"

        #Services
        seen_title=False
        list = get_services_dictionary(self)
        for item in list:
            result=self.config.get("mythbuntu",item)
            active=list[item].get_active()
            if result=="True" and active == 0:
                if not seen_title:
                    text+="Services:\n"
                    seen_title=True
                if item != "mysql-server":
                    to_remove.append(item)
                if item == "x11vnc" or item == "mysql-server":
                    to_reconfigure[item]=False
                text+="   Remove " + item + " service.\n"
            elif result=="True" and active == 2:
                if not seen_title:
                    text+="Services:\n"
                    seen_title=True
                to_reconfigure[item]=True
                text+="   Reconfigure " + item + " service.\n"
                if item == "x11vnc":
                    to_reconfigure["vnc_password"] = self.vnc_password.get_text()
                if item == "nfs-kernel-server":
                    try:
                        to_reconfigure['master-backend-address']=self.get_db_setting('MasterServerIP',None)
                    except:
                        # there should be one common error message for this
                        message1 = "I could not get the IP address of your master backend.\nIs your MySQL server running?\nIs the connection information in the 'MythTV configuration' tab correct?\n"
                        message2 = "Your NFS share will be exported to the world. Access is\nnot limited to your local network. Please fix\n /etc/exports manually"
                        self.display_error(message1 + message2)

            elif result=="False" and active == 1:
                if not seen_title:
                    text+="Services:\n"
                    seen_title=True
                to_install.append(item)
                to_reconfigure[item]=True
                text+="   Add " + item + " service.\n"
                if item == "x11vnc":
                    to_reconfigure["vnc_password"] = self.vnc_password.get_text()
                if item == "nfs-kernel-server":
                    try:
                        to_reconfigure['master-backend-address']=self.get_db_setting('MasterServerIP',None)
                    except:
                        # there should be one common error message for this
                        message1 = "I could not get the IP address of your master backend.\nIs your MySQL server running?\nIs the connection information in the 'MythTV configuration' tab correct?\n"
                        message2 = "Your NFS share will be exported to the world. Access is\nnot limited to your local network. Please fix\n /etc/exports manually"
                        self.display_error(message1 + message2)

        #remote control
        seen_title=False
        if old_remote != new_remote:
            seen_title=True
            text+="Infrared Devices:\n"
            if not self.remotecontrol.get_active():
                text+="   Disable " + old_remote + " remote control.\n"
                to_reconfigure["remote"]='None'
            else:
                if old_remote != "None":
                    text+="   Change to " + new_remote + " remote control.\n"
                else:
                    text+="   Add to " + new_remote + " remote control.\n"
                to_reconfigure["remote"]=new_remote
                if new_remote == "Custom":
                    to_reconfigure["remote_modules"]=new_remote_modules
                    to_reconfigure["remote_device"]=new_remote_device
                    to_reconfigure["remote_driver"]=new_remote_driver
                    to_reconfigure["remote_lircd_conf"]=new_remote_lircd_conf
        elif (new_remote == "Custom") and (old_remote_modules != new_remote_modules or old_remote_device != new_remote_device or old_remote_driver != new_remote_driver or old_remote_lircd_conf != new_remote_lircd_conf):
                text+="   Customized Custom remote.\n"
                to_reconfigure["remote"]=new_remote
                to_reconfigure["remote_modules"]=new_remote_modules
                to_reconfigure["remote_device"]=new_remote_device
                to_reconfigure["remote_driver"]=new_remote_driver
                to_reconfigure["remote_lircd_conf"]=new_remote_lircd_conf
        if old_transmitter != new_transmitter:
            if not seen_title:
                text+="Infrared Devices:\n"
                seen_title=True
            if not self.transmittercontrol.get_active():
                text+="   Disable " + old_transmitter + " transmitter control.\n"
                to_reconfigure["transmitter"]='None'
            else:
                if old_transmitter != "None":
                    text+="   Change to " + new_transmitter + " transmitter control.\n"
                else:
                    text+="   Add to " + new_transmitter + " transmitter control.\n"
                to_reconfigure["transmitter"]=new_transmitter
                if new_transmitter == "Custom":
                    to_reconfigure["transmitter_modules"]=new_transmitter_modules
                    to_reconfigure["transmitter_device"]=new_transmitter_device
                    to_reconfigure["transmitter_driver"]=new_transmitter_driver
                    to_reconfigure["transmitter_lircd_conf"]=new_transmitter_lircd_conf
        elif (new_transmitter == "Custom") and (old_transmitter_modules != new_transmitter_modules or old_transmitter_device != new_transmitter_device or old_transmitter_driver != new_transmitter_driver or old_transmitter_lircd_conf != new_transmitter_lircd_conf):
                text+="   Customized Custom transmitter.\n"
                to_reconfigure["transmitter"]=new_transmitter
                to_reconfigure["transmitter_modules"]=new_transmitter_modules
                to_reconfigure["transmitter_device"]=new_transmitter_device
                to_reconfigure["transmitter_driver"]=new_transmitter_driver
                to_reconfigure["transmitter_lircd_conf"]=new_transmitter_lircd_conf

        if self.generate_lircrc_checkbox.get_active():
            if not seen_title:
                seen_title=True
                text+="Remote Control:\n"
            to_reconfigure["lircrc"]=True
            text+="   Generate application specific lirc settings.\n"
        else:
            to_reconfigure["lircrc"]=False
        #automatic login
        seen_title=False
        if old_autologin != new_autologin or old_autologin_user != new_autologin_user:
            if not seen_title:
                text+="Automatic Login & Artwork\n"
                seen_title=True
            if new_autologin_user is not None and new_autologin_user != "[Disabled]":
                if old_autologin != new_autologin and  new_autologin:
                    text+="   Activate automatic login for " + new_autologin_user + ".\n"
                    to_reconfigure["autologin"]=True
                    to_reconfigure["autologin_user"]=new_autologin_user
                    if not self.query_installed("mythbuntu-desktop"):
                        to_install.append("mythbuntu-desktop")
                elif old_autologin_user != new_autologin_user:
                    text+="   Change automatic login from " + old_autologin_user + " to " + new_autologin_user + ".\n"
                    to_reconfigure["autologin"]=True
                    to_reconfigure["autologin_user"]=new_autologin_user

            elif old_autologin != new_autologin and old_autologin:
                text+="   Disable automatic login.\n"
                to_reconfigure["autologin"]=False
                to_reconfigure["autologin_user"]=old_autologin_user
                if self.query_installed("ubuntu-mythtv-frontend"):
                    to_remove.append("ubuntu-mythtv-frontend")
        #artwork
        if old_artwork != new_artwork:
            if not seen_title:
                text+="Automatic Login & Artwork\n"
                seen_title=True
            if new_artwork:
                text+="   Add Mythbuntu artwork.\n"
                to_install.append("mythbuntu-desktop")
            else:
                text+="   Remove Mythbuntu artwork.\n"
                to_remove.append("mythbuntu-default-settings")
                to_remove.append("mythbuntu-gdm-theme")
                to_remove.append("mythbuntu-artwork-usplash")
                to_remove.append("mythbuntu-desktop")
                #we can't have autologin without artwork, take out autologin if we take out artwork
                to_reconfigure["autologin"]=False
                to_reconfigure["autologin_user"]=old_autologin_user
                if self.query_installed("ubuntu-mythtv-frontend"):
                    to_remove.append("ubuntu-mythtv-frontend")

        if self.nvidia_settings_checkbox.get_active():
            text+="Proprietary Drivers:\n"
            text+="   Install NVIDIA Settings.\n"
            to_install.append("nvidia-settings")

        for child in self.summary_text.get_children():
            self.summary_text.remove(child)

        if text == "":
            text = "No changes detected."
            self.apply_button3.set_sensitive(False)
        else:
            self.apply_button3.set_sensitive(True)
        summary_buffer = gtk.TextBuffer()
        summary_buffer.set_text(text)
        self.summary_text.set_buffer(summary_buffer)

        return [to_install,to_remove,to_reconfigure]

    def query_installed(self,package):
        """Determines if a single package is installed"""
        try:
            return self.cache[package].CurrentVer
        except KeyError:
            return False

    def set_gui_defaults(self):
        """Loads default settings"""

        def empty_combobox(boxy):
            """Empties out a combo box"""
            model = boxy.get_model()
            boxy.set_model(None)
            model.clear()
            boxy.set_model(model)
            boxy.append_text("[None]")
            boxy.set_active(0)

        #MythTV Setup
        self.mythtv_setup_vbox.set_sensitive(False)
        #Remote frontend/backend connection info
        self.mysql_config_vbox.set_sensitive(False)
        self.mysql_user.set_text("mythtv")
        self.mysql_pass.set_text("")
        self.mysql_database.set_text("mythconverg")
        self.mysql_server.set_text("")
        self.mysql_test_hbox.hide()
        #Remote Controls
        self.remotecontrol.set_active(False)
        self.transmittercontrol.set_active(False)
        #Plugins
        self.selectall_plugins_checkbox.set_active(False)
        self.toggle_meta(self.selectall_plugins_checkbox)
        for list in get_frontend_plugin_dictionary(self),get_backend_plugin_dictionary(self):
            for item in list:
                list[item].set_sensitive(False)
                list[item].set_active(False)
        #Proprietary Codecs
        self.enable_medibuntu.set_active(False)
        #Themes, Applications, Codecs
        self.officialthemes.set_active(False)
        self.communitythemes.set_active(False)
        for list in get_official_theme_dictionary(self),get_community_theme_dictionary(self), get_media_app_dictionary(self), get_nonfree_dictionary(self):
            for item in list:
                list[item].set_active(False)
        #Roles
        self.no_backend_radio.set_active(True)
        self.no_frontend_radio.set_active(True)
        #Services
        self.vnc_installed=False
        self.vnc_pass_hbox.set_sensitive(False)
        self.vnc_password.set_text("")
        self.mysql_vbox.set_sensitive(False)
        list = get_services_dictionary(self)
        for item in list:
            list[item].set_active(0)
        #Proprietary Drivers
        self.amd_vbox.set_sensitive(False)
        self.nvidia_vbox.set_sensitive(False)
        self.nvidia_settings_button.hide()
        self.nvidia_settings_checkbox.hide()
        self.nvidia_settings_checkbox.set_active(False)
        self.xorg_vbox.set_sensitive(False)

        #Autologin user
        self.autologin_user.set_active(0)
        self.enableautologin.set_active(False)
        #Artwork
        self.enable_artwork.set_active(False)
        self.artwork_vbox.set_sensitive(False)
        # mythweb auth
        # FIXME
        # self.mythweb_password_checkbutton.set_active(True)
        # mysqld tweaks
        self.enable_mysql_tweaks.set_active(False)
        # xfs defrag
        self.enable_nightly_defrag.set_active(False)
        # mythtv database optimization/repair
        self.enable_mysql_repair.set_active(False)
        # IVTV Driver Memory Tweak
        self.enable_ivtv_tweak.set_active(False)
        # RTC Video Tweak
        self.enable_rtc_tweak.set_active(False)

        # diskless
        self.diskless_server_install.set_active(False)
        empty_combobox(self.diskless_chroot)
        empty_combobox(self.diskless_arch)
        empty_combobox(self.diskless_pendrive)
        empty_combobox(self.diskless_nbd_server_nic)
        self.diskless_delete_image_hbox.set_sensitive(False)
        self.diskless_build_image_hbox.set_sensitive(False)
        self.diskless_unsigned_packages.set_sensitive(False)
        self.diskless_maintenance_box.set_sensitive(False)
        self.diskless_pendrive_action_hbox.set_sensitive(False)
        self.diskless_wol_entry.set_sensitive(False)
        self.diskless_wol_checkbox.set_active(False)


    def query_system_state(self):
        """Determines the state of the system, and what settings to activate"""
        #Roles

        ##Backend / Frontend Roles
        master_backend=self.query_installed("mythtv-backend-master")
        slave_backend=self.query_installed("mythtv-backend")
        frontend=self.query_installed("mythtv-frontend")

        enable_experimental_options_vbox = False
        if master_backend or slave_backend:
            self.mythtv_setup_vbox.set_sensitive(True)
            self.mythweb_checkbox.set_sensitive(True)
            enable_experimental_options_vbox = True

            if master_backend:
                self.config.set("mythbuntu","backend","primary")
                self.primary_backend_radio.set_active(True)
            else:
                self.config.set("mythbuntu","backend","secondary")
                self.secondary_backend_radio.set_active(True)
                self.mysql_config_vbox.set_sensitive(True)
        else:
            self.config.set("mythbuntu","backend","none")
            enable_experimental_options_vbox = False

        if frontend:
            self.config.set("mythbuntu","frontend","frontend")
            self.frontend_radio.set_active(True)
            if not master_backend:
                self.mysql_config_vbox.set_sensitive(True)
            list = get_frontend_plugin_dictionary(self)
            for plugin in list:
                list[plugin].set_sensitive(True)
        else:
            self.config.set("mythbuntu","frontend","none")

        #Medibuntu Items
        if self.find_medibuntu():
            self.enable_medibuntu.set_active(True)
            self.enable_w32codecs.set_sensitive(True)
            self.enable_w64codecs.set_sensitive(True)
            self.enable_ffmpeg.set_sensitive(True)
            self.enable_libdvdcss2.set_sensitive(True)
            self.config.set("mythbuntu","medibuntu","True")
        else:
            self.enable_w32codecs.set_sensitive(False)
            self.enable_w64codecs.set_sensitive(False)
            self.enable_libdvdcss2.set_sensitive(False)
            self.enable_ffmpeg.set_sensitive(True)
            self.config.set("mythbuntu","medibuntu","False")
        arch = commands.getstatusoutput('uname -m')[1]
        if arch != "x86_64":
            self.enable_w64codecs.set_sensitive(False)
        if arch != "i686":
            self.enable_w32codecs.set_sensitive(False)
        #Checkbox items (Plugins, Media Players, Proprietary Codecs, Themes, Desktop, & Diskless Dictionaries)
        for list in get_frontend_plugin_dictionary(self), get_backend_plugin_dictionary(self), get_official_theme_dictionary(self), get_community_theme_dictionary(self), get_media_app_dictionary(self), get_nonfree_dictionary(self), get_desktop_dictionary(self), get_diskless_dictionary(self):
            for item in list:
                result = self.query_installed(item)
                if item == "mplayer" and not result:
                    result = self.query_installed("mplayer-nogui")
                #turn on and off the diskless page if we have that role
                elif item =="mythbuntu-diskless-server":
                    if result:
                        self.diskless_button.set_sensitive(True)
                    else:
                        self.diskless_button.set_sensitive(False)
                # we also want to have a good configuration file in place
                elif item == "dhcp3-server":
                    if result and not os.path.isfile('/etc/ltsp/dhcpd.conf'):
                        result = False
                if result:
                    list[item].set_active(True)
                    self.config.set("mythbuntu",item,"True")
                else:
                    list[item].set_active(False)
                    self.config.set("mythbuntu",item,"False")

        #Services
        self.enablesamba.remove_text(2)
        self.enablevnc.remove_text(2)
        self.enablenfs.remove_text(2)
        if self.query_installed("samba"):
            self.enablesamba.set_active(1)
            self.enablesamba.append_text("Reconfigure")
            self.config.set("mythbuntu","samba","True")
        else:
            self.config.set("mythbuntu","samba","False")
        if self.query_installed("x11vnc"):
            self.vnc_installed=True
            self.enablevnc.set_active(1)
            self.enablevnc.append_text("Reconfigure")
            self.config.set("mythbuntu","x11vnc","True")
        else:
            self.config.set("mythbuntu","x11vnc","False")
        if self.query_installed("nfs-kernel-server"):
            self.enablenfs.set_active(1)
            self.enablenfs.append_text("Reconfigure")
            self.config.set("mythbuntu","nfs-kernel-server","True")
        else:
            self.config.set("mythbuntu","nfs-kernel-server","False")
        if self.query_installed("openssh-server"):
            self.enablessh.set_active(1)
            self.config.set("mythbuntu","openssh-server","True")
        else:
            self.config.set("mythbuntu","openssh-server","False")
        if self.query_installed("mythtv-database") and self.query_installed("mysql-server"):
            try:
                in_f=open("/etc/mysql/conf.d/mythtv.cnf")
                self.mysql_vbox.set_sensitive(True)
                found=False
                for line in in_f:
                    if re.compile("^bind-address").search(line):
                        self.enablemysql.set_active(1)
                        found=True
                        self.config.set("mythbuntu","mysql-server","True")
                if not found:
                    self.config.set("mythbuntu","mysql-server","False")
                in_f.close()
            except IOError:
                self.config.set("mythbuntu","mysql-server","False")
                self.display_error("/etc/mysql/conf.d/mythtv.cnf is missing.\nDisabling modifications to MySQL Service")
        else:
            self.config.set("mythbuntu","mysql-server","False")
        #Proprietary Drivers
        driver = self.find_proprietary_driver()
        if  driver == 'amdcccle':
            self.amd_vbox.set_sensitive(True)
            self.nvidia_vbox.set_sensitive(False)
            self.nvidia_settings_checkbox.set_active(False)
        elif driver == 'nvidia-settings':
            self.amd_vbox.set_sensitive(False)
            self.nvidia_vbox.set_sensitive(True)
            self.nvidia_settings_button.show()
            self.nvidia_settings_checkbox.set_active(False)
            self.nvidia_settings_checkbox.hide()
        else:
            self.nvidia_vbox.set_sensitive(True)
            self.nvidia_settings_button.hide()
            self.nvidia_settings_checkbox.show()
        if os.path.exists('/usr/bin/displayconfig-gtk'):
            self.xorg_vbox.set_sensitive(True)
        #Advanced tab
        if os.path.exists('/usr/bin/update-manager'):
            self.update_manager_vbox.set_sensitive(True)
        if os.path.exists('/usr/sbin/synaptic'):
            self.synaptic_vbox.set_sensitive(True)
        #Medibuntu Items
        if self.find_medibuntu():
            self.enable_medibuntu.set_active(True)
            self.config.set("mythbuntu","medibuntu","True")
        else:
            self.enable_w32codecs.set_sensitive(False)
            self.enable_w64codecs.set_sensitive(False)
            self.enable_libdvdcss2.set_sensitive(False)
            self.enable_ffmpeg.set_sensitive(False)
            self.config.set("mythbuntu","medibuntu","False")
        #Backend connection info
        try:
            in_f=open("/etc/mythtv/mysql.txt")
            for line in in_f:
                if re.compile("^DBHostName").search(line):
                    text=string.split(string.split(line,"=")[1],'\n')[0]
                    self.mysql_server.set_text(text)
                elif re.compile("^DBUserName").search(line):
                    text=string.split(string.split(line,"=")[1],'\n')[0]
                    self.mysql_user.set_text(text)
                elif re.compile("^DBName").search(line):
                    text=string.split(string.split(line,"=")[1],'\n')[0]
                    self.mysql_database.set_text(text)
                elif re.compile("^DBPassword").search(line):
                    text=string.split(string.split(line,"=")[1],'\n')[0]
                    self.mysql_pass.set_text(text)
            in_f.close()
        except IOError:
             self.display_error("/etc/mythtv/mysql.txt is missing.\nUsing default settings.")
        self.config.set("mythbuntu","mythtv_mysql_server",self.mysql_server.get_text())
        self.config.set("mythbuntu","mythtv_mysql_user",self.mysql_user.get_text())
        self.config.set("mythbuntu","mythtv_mysql_database",self.mysql_database.get_text())
        self.config.set("mythbuntu","mythtv_mysql_password",self.mysql_pass.get_text())
        #Remote Control info
        self.lirc.read_hardware_conf()
        remote_dict=self.lirc.get_device_dictionary("remote")
        transmitter_dict=self.lirc.get_device_dictionary("transmitter")
        if remote_dict["remote"] != "None":
            found = False
            for item in range(self.remote_count):
                self.remote_list.set_active(item)
                if self.remote_list.get_active_text() == remote_dict["remote"]:
                    found = True
                    self.config.set("mythbuntu","remote",self.remote_list.get_active_text())
                    self.remotecontrol.set_active(True)
                    break
            if not found:
                self.remote_list.set_active(0)
                self.generate_lircrc_checkbox.set_active(False)
                self.config.set("mythbuntu","remote","None")
            else:
                self.generate_lircrc_checkbox.set_active(not self.find_lircrc())
        else:
            self.config.set("mythbuntu","remote","None")
        #misc remote info
        self.remote_modules.set_text(remote_dict["remote_modules"])
        self.remote_driver.set_text(remote_dict["remote_driver"])
        self.remote_device.set_text(remote_dict["remote_device"])
        self.browse_remote_lircd_conf.select_filename("/usr/share/lirc/remotes")
        if os.path.exists(remote_dict["remote_lircd_conf"]):
            self.browse_remote_lircd_conf.select_filename(remote_dict["remote_lircd_conf"])
        elif os.path.exists("/usr/share/lirc/remotes/" + remote_dict["remote_lircd_conf"]):
            self.browse_remote_lircd_conf.select_filename("/usr/share/lirc/remotes/" + remote_dict["remote_lircd_conf"])
        #if we have a custom remote, store this stuff
        if remote_dict["remote"] == "Custom":
            self.config.set("mythbuntu","remote_modules",remote_dict["remote_modules"])
            self.config.set("mythbuntu","remote_lircd_conf",remote_dict["remote_lircd_conf"])
            self.config.set("mythbuntu","remote_device",remote_dict["remote_device"])
            self.config.set("mythbuntu","remote_driver",remote_dict["remote_driver"])
        #Transmitter
        if transmitter_dict["transmitter"] != "None":
            found = False
            for item in range(self.transmitter_count):
                self.transmitter_list.set_active(item)
                if self.transmitter_list.get_active_text() == transmitter_dict["transmitter"]:
                    found = True
                    self.config.set("mythbuntu","transmitter",self.transmitter_list.get_active_text())
                    self.transmittercontrol.set_active(True)
                    break
            if not found:
                self.transmitter_list.set_active(0)
                self.config.set("mythbuntu","transmitter","None")
        else:
            self.config.set("mythbuntu","transmitter","None")
        #misc transmitter info
        self.transmitter_modules.set_text(transmitter_dict["transmitter_modules"])
        self.transmitter_driver.set_text(transmitter_dict["transmitter_driver"])
        self.transmitter_device.set_text(transmitter_dict["transmitter_device"])
        self.browse_transmitter_lircd_conf.select_filename("/usr/share/lirc/transmitters")
        if os.path.exists(transmitter_dict["transmitter_lircd_conf"]):
            self.browse_transmitter_lircd_conf.select_filename(transmitter_dict["transmitter_lircd_conf"])
        elif os.path.exists("/usr/share/lirc/transmitters/" + transmitter_dict["transmitter_lircd_conf"]):
            self.browse_transmitter_lircd_conf.select_filename("/usr/share/lirc/transmitters/" + transmitter_dict["transmitter_lircd_conf"])
        #if we have a custom transmitter, store this stuff
        if transmitter_dict["transmitter"] == "Custom":
            self.config.set("mythbuntu","transmitter_modules",transmitter_dict["transmitter_modules"])
            self.config.set("mythbuntu","transmitter_lircd_conf",transmitter_dict["transmitter_lircd_conf"])
            self.config.set("mythbuntu","transmitter_device",transmitter_dict["transmitter_device"])
            self.config.set("mythbuntu","transmitter_driver",transmitter_dict["transmitter_driver"])
        #Automatic Login
        if os.path.exists("/etc/gdm/gdm-cdd.conf"):
            in_f=open("/etc/gdm/gdm-cdd.conf")
            pattern=re.compile("^AutomaticLoginEnable=true")
            pattern2=re.compile("^AutomaticLogin=")
            found_autologin=False
            found_user=False
            for line in in_f:
                if pattern.search(line):
                    self.enableautologin.set_active(True)
                    self.config.set("mythbuntu","autologin","True")
                    self.artwork_vbox.set_sensitive(False)
                    found_autologin=True
                if pattern2.search(line):
                    user=string.split(string.split(line,"=")[1],'\n')[0]
                    for item in range(1,self.user_count+1):
                        self.autologin_user.set_active(item)
                        if self.autologin_user.get_active_text() == user:
                            self.config.set("mythbuntu","autologin_user",user)
                            found_user=True
                            break
                if found_autologin and found_user:
                    break
            if not found_autologin:
                self.config.set("mythbuntu","autologin","False")
            if not found_user:
                self.config.set("mythbuntu","autologin_user","[Disabled]")
                self.autologin_user.set_active(0)
        else:
            self.config.set("mythbuntu","autologin","False")
            self.config.set("mythbuntu","autologin_user","[Disabled]")
        #Artwork Package
        if self.query_installed("mythbuntu-desktop"):
            self.enable_artwork.set_active(True)
            self.config.set("mythbuntu","artwork","True")
        else:
            self.config.set("mythbuntu","artwork","False")
        #Role not toggled
        self.role_toggled=False

        # mythweb auth
        if self.query_installed("mythweb"):
            # FIXME: looks like m-c-c is not supposed to install mythweb unless this is a master or slave backend
            # if the user installs mythweb manually, this could lead to an inconsistent GUI
            #
            self.mythweb_password_checkbutton.set_sensitive(True)

            if self.communicator.run("get","mythweb/enable") == "true":
                self.config.set("mythbuntu","mythweb_auth_activated","True")
            elif self.mythweb_password_checkbutton.get_active() == True:
                self.config.set("mythbuntu","mythweb_auth_activated","True")
            else:
                self.config.set("mythbuntu","mythweb_auth_activated","False")

            if self.communicator.run("get","mythweb/username"):
                self.config.set("mythbuntu","mythweb_auth_username",self.communicator.run("get","mythweb/username"));
                self.mythweb_username.set_text(self.communicator.run("get","mythweb/username"))
            elif  self.mythweb_username.get_text():
                self.config.set("mythbuntu","mythweb_auth_username",self.mythweb_username.get_text());
            else:
                self.config.set("mythbuntu","mythweb_auth_username","")

            if self.communicator.run("get","mythweb/password"):
                self.config.set("mythbuntu","mythweb_auth_password",self.mythweb_password.get_text());
            elif self.mythweb_password.get_text():
               self.config.set("mythbuntu", "mythweb_auth_password",self.mythweb_password.get_text())
            else:
               self.config.set("mythbuntu", "mythweb_auth_password","")

        else:
            self.mythweb_password_checkbutton.set_sensitive(False)
            self.config.set("mythbuntu","mythweb_auth_activated","False")
            self.config.set("mythbuntu","mythweb_auth_username","")
            self.config.set("mythbuntu", "mythweb_auth_password","")

        if self.config.get("mythbuntu","mythweb_auth_activated") == "True":
            self.mythweb_password_checkbutton.set_active(True)

        # mysql tweaks
        # maybe I should check if there is a master backend instead?
        # setting mysql_tweaks_enabled to false here so we don't break if mysql-server is not installed
        self.config.set("mythbuntu","mysql_tweaks_enabled","False")
        if self.query_installed("mysql-server"):
            self.enable_mysql_tweaks.set_sensitive(True)
            # I really should be doing this via debconf, but I can't yet.
            if os.path.exists("/etc/mysql/conf.d/mythtv-tweaks.cnf"):
                self.enable_mysql_tweaks.set_active(True)
                self.config.set("mythbuntu","mysql_tweaks_enabled","True")
            else:
                self.enable_mysql_tweaks.set_active(False)
                self.config.set("mythbuntu","mysql_tweaks_enabled","False")
        else:
            self.enable_mysql_tweaks.set_sensitive(False)

        # xfs defrag
        # check if xfsdump and util-linux are installed as well?
        # util-linux is part of ubuntu-minimal
        if os.path.exists("/etc/cron.daily/mythtv-xfs-defrag"):
            self.enable_nightly_defrag.set_active(True)
            self.config.set("mythbuntu","enable_nightly_defrag","True")
        else:
            self.enable_nightly_defrag.set_active(False)
            self.config.set("mythbuntu","enable_nightly_defrag","False")

        # mythtv database optimization/repair
        # optimize_mythdb.pl is in package mythtv-backend
        # maybe it'd be more future-proof if we checked for the existence of the file as well
        # FIXME: what happens if the user removes mythtv-backend? Maybe we should remove the cron job from the system then
        if os.path.exists("/etc/cron.daily/optimize_mythdb.pl"):
            self.enable_mysql_repair.set_active(True)
            self.config.set("mythbuntu","enable_mysql_repair","True")
        else:
            self.enable_mysql_repair.set_active(False)
            self.config.set("mythbuntu","enable_mysql_repair","False")

        # button to run optimize_mythdb.pl
        if os.path.exists("/usr/share/doc/mythtv-backend/contrib/optimize_mythdb.pl") and self.query_installed("mysql-server"):
            self.mysql_tweak_button.set_sensitive(True)
        else:
            self.mysql_tweak_button.set_sensitive(False)

        #Sysctl tweaks
        self.config.set("mythbuntu","enable_ivtv_tweak","False")
        self.config.set("mythbuntu","enable_rtc_tweak","False")
        if os.path.exists("/etc/sysctl.conf"):
            in_f=open("/etc/sysctl.conf")
            pattern=re.compile("^vm.min_free_kbytes=16384")
            pattern2=re.compile("^dev.rtc.max-user-freq=1024")
            for line in in_f:
                if pattern.search(line):
                    self.enable_ivtv_tweak.set_active(True)
                    self.config.set("mythbuntu","enable_ivtv_tweak","True")
                if pattern2.search(line):
                    self.enable_rtc_tweak.set_active(True)
                    self.config.set("mythbuntu","enable_rtc_tweak","True")
            in_f.close()
        else:
            self.enable_rtc_tweak.set_sensitive(False)
            self.enable_ivtv_tweak.set_sensitive(False)

        #experimental options
        # This is still being used - Foxbuntu
        if enable_experimental_options_vbox:
            self.experimental_options_vbox.set_sensitive(True)
        else:
            self.experimental_options_vbox.set_sensitive(False)
            self.enable_nightly_defrag.set_active(False)
            self.enable_mysql_tweaks.set_active(False)
            self.enable_mysql_repair.set_active(False)

        # mythbuntu-diskless
        # find all chroots for which we can do stuff
        if os.path.exists(ltsp_base_dir):
            for i in os.listdir(ltsp_base_dir):
                if os.path.isfile(ltsp_base_dir + i + '/etc/fstab'):
                    self.diskless_chroot.append_text(i)

        # find all architectures for which we can create chroots
        myarch = apt_pkg.Config.Find("APT::Architecture")
        if myarch == 'amd64':
            self.diskless_arch.insert_text(1,'i386')
            self.diskless_arch.insert_text(2,'amd64')
        elif myarch == 'i386':
            self.diskless_arch.insert_text(1,'i386')
        elif myarch == 'ppc':
            self.diskless_arch.insert_text(1,'ppc')

        self.build_pendrive_list()

    def build_pendrive_list(self,widget=None):
        """Builds a list of pendrives"""

        #if we are called from a widget, then empty the list first
        if widget != None:
            self.set_gui_defaults.empty_combobox(self.diskless_pendrive)

        # build list of removable media
        blockpath =  '/sys/block/'
        try:
            for i in os.listdir(blockpath):
                remov = open(blockpath + i + '/removable')
                modelpath = blockpath + i + '/device/model'
                if remov.readline().rstrip() == '1' and os.path.exists(modelpath) and not "sr" in i:
                    self.diskless_pendrive.append_text(i)
                remov.close()
        except OSError:
            print "Trying to look in a nonexistant /sys/block"

        # build list of network interfaces
        # thanks to paul cannon
        # http://aspn.activestate.com/ASPN/Cookbook/Python/Recipe/439093
        nicpath = '/sys/class/net/'
        try:
            for i in os.listdir(nicpath):
                if i != 'lo':
                    try:
                        get_ip_address(i)
                        self.diskless_nbd_server_nic.append_text(i)
                    except IOError:
                        # looks like this device doesn't have an IP address
                        # or we've encountered another error
                        pass
        except OSError:
            print "Trying to look in a nonexistant /sys/class"


    def find_medibuntu(self):
        """Determines if the medibuntu repo is active"""
        in_f=open("/etc/apt/sources.list")
        pattern=re.compile("^deb http://packages.medibuntu.org/ hardy free non-free")
        found=False
        for line in in_f:
            if pattern.search(line):
                found=True
        if not found and os.path.exists("/etc/apt/sources.list.d/medibuntu.list"):
            found=True
        return found

    def find_proprietary_driver(self):
        """Determines if amdcccle or nvidia-settings are avail"""
        if os.path.exists('/usr/bin/amdcccle'):
            return 'amdcccle'
        elif os.path.exists('/usr/bin/nvidia-settings'):
            return 'nvidia-settings'

    def find_lircrc(self):
        """Determines if there is current a lircrc for the current user"""
        return os.path.exists(os.getenv('HOME') + '/.lircrc')

#Signal call back
    def apply_pressed(self,widget):
        """Asks the user if they want to continue"""

        def call_applicator(self,to_install,to_remove,to_reconfigure):
            """Calls the application routine"""
            self.main_window.hide()
            apply = Applicator(to_install,to_remove,to_reconfigure)
            if not apply.get_init_error():
                apply.commit_changes()
            self.cache = apt_pkg.GetCache()
            self.main_window.show()

        (to_install,to_remove,to_reconfigure) = self.find_case_for_change()
        if (widget is not None):
            #apply button from main window
            if (widget.get_name() == "apply_button"):
                self.apply_dialog.run()
                self.apply_dialog.hide()
            #apply button from summary window
            elif (widget.get_name() == "apply_button3"):
                self.apply_dialog.hide()
                call_applicator(self,to_install,to_remove,to_reconfigure)
                self.revert_gui()
            #apply button from forced apply window
            elif (widget.get_name() == "apply_button2"):
                self.quick_apply_dialog.hide()
                call_applicator(self,to_install,to_remove,to_reconfigure)
                self.revert_gui()

    def revert_gui(self,widget=None):
        """Resets the GUI to old settings"""
        self.set_gui_defaults()
        self.query_system_state()

    def tab_change(self,widget):
        """Called whenever a request to change tabs is made"""
        if widget is not None:
            if self.role_toggled:
                self.quick_apply_dialog.run()
                self.quick_apply_dialog.hide()
                self.role_toggled=False
            name = widget.get_name()
            if name == 'system_roles_button':
                self.tabs.set_current_page(self.tabs.page_num(self.tab_system_roles))
            elif name == 'mythtv_configuration_button':
                self.tabs.set_current_page(self.tabs.page_num(self.tab_mythtv_configuration))
            elif name == 'remote_control_button':
                self.tabs.set_current_page(self.tabs.page_num(self.tab_remote_control))
            elif name == 'applications_plugins_button':
                self.tabs.set_current_page(self.tabs.page_num(self.tab_applications_plugins))
            elif name == 'proprietary_drivers_button':
                self.tabs.set_current_page(self.tabs.page_num(self.tab_proprietary_drivers))
            elif name == 'system_services_button':
                self.tabs.set_current_page(self.tabs.page_num(self.tab_system_services))
            elif name == 'artwork_behavior_button':
                self.tabs.set_current_page(self.tabs.page_num(self.tab_artwork_behavior))
            elif name == 'network_shares_button':
                self.tabs.set_current_page(self.tabs.page_num(self.tab_network_shares))
            elif name == 'proprietary_codecs_button':
                self.tabs.set_current_page(self.tabs.page_num(self.tab_proprietary_codecs))
            elif name == 'advanced_button':
                self.tabs.set_current_page(self.tabs.page_num(self.tab_advanced))
            elif name == 'themes_button':
                self.tabs.set_current_page(self.tabs.page_num(self.tab_themes))
            elif name == 'diskless_button':
                self.tabs.set_current_page(self.tabs.page_num(self.tab_diskless))
            self.buttons_area.set_sensitive(True)

    def toggle_autologin(self,widget):
        """Called whenever autologin is toggled"""
        if widget is not None:
            if widget.get_name() == "enableautologin":
                active = widget.get_active()
                no_desktop = not (self.ubuntu_desk_check.get_active() or \
                self.kubuntu_desk_check.get_active() or \
                self.xubuntu_desk_check.get_active())
                self.autologin_users_hbox.set_sensitive(active)
                if not active:
                    self.autologin_user.set_active(0)
                    if no_desktop:
                        self.artwork_vbox.set_sensitive(False)
                    else:
                        self.artwork_vbox.set_sensitive(True)
                else:
                    self.autologin_user.set_active(1)
                    self.artwork_vbox.set_sensitive(False)

            elif widget.get_name() == "autologin_user":
                if widget.get_active() == 0:
                    self.enableautologin.set_active(False)
                if widget.get_active_text() == "mythtv":
                    self.mythtv_user_warning.show()
                else:
                    self.mythtv_user_warning.hide()

    def toggle_meta(self,widget):
        """Called whenever a request to enable / disable all plugins"""
        if widget is not None:
            list = []
            name = widget.get_name()
            if (widget.get_name() == 'selectall_plugins_checkbox'):
                list = get_frontend_plugin_dictionary(self)
                temp = get_backend_plugin_dictionary(self)
                for theme in temp:
                    list[theme]=temp[theme]
            elif (name == 'officialthemes'):
                list = get_official_theme_dictionary(self)
            elif (name == 'communitythemes'):
                list = get_community_theme_dictionary(self)

            toggle = widget.get_active()
            for item in list:
                if list[item].flags() & gtk.SENSITIVE:
                    list[item].set_active(toggle)


    def on_mythweb_password_checkbutton_toggled(self,widget):
        """Called whenever the advanced configuration button for plugins is pressed"""
        # show the username/password fields
        self.password_table.set_sensitive(self.mythweb_password_checkbutton.get_active())

    def toggle_medibuntu(self,widget):
        """Called whenever medibuntu is toggled"""
        if (widget is not None and widget.get_name() == "enable_medibuntu"):
            self.medibuntu_vbox.set_sensitive(widget.get_active())
            #Revert old package settings


    def toggle_vnc(self,widget):
        """Called whenever VNC service is toggled"""
        if (widget is not None and widget.get_name() == 'enablevnc'):
            iteration=1
            if self.vnc_installed:
                iteration = 2
            if widget.get_active() == iteration:
                self.vnc_pass_hbox.set_sensitive(True)
                self.vnc_error_image.show()
                self.apply_button.set_sensitive(False)
            else:
                self.vnc_pass_hbox.set_sensitive(False)
                self.vnc_error_image.hide()
                self.apply_button.set_sensitive(True)
                self.vnc_password.set_text("")
                self.vnc_error_image.hide()
                self.apply_button.set_sensitive(True)

    def toggle_vnc_password(self,widget):
        if (widget is not None and widget.get_name() == 'vnc_password'):
            password= widget.get_text().split(' ')[0]
            if len(password) >= 6:
                self.vnc_error_image.hide()
                self.apply_button.set_sensitive(True)
            else:
                self.vnc_error_image.show()
                self.apply_button.set_sensitive(False)

    def toggle_ir(self,widget):
        """Called whenever a request to enable/disable remote is called"""
        if widget is not None:
            #turn on/off IR remote
            if widget.get_name() == 'remotecontrol':
                self.remote_hbox.set_sensitive(widget.get_active())
                if widget.get_active():
                    self.generate_lircrc_checkbox.set_active(not self.find_lircrc())
                    if self.remote_list.get_active() == 0:
                        self.remote_list.set_active(1)
                else:
                    self.remote_list.set_active(0)
                    self.generate_lircrc_checkbox.set_active(False)
            #turn on/off IR transmitter
            elif widget.get_name() == "transmittercontrol":
                self.transmitter_hbox.set_sensitive(widget.get_active())
                if widget.get_active():
                    if self.transmitter_list.get_active() == 0:
                        self.transmitter_list.set_active(1)
                else:
                    self.transmitter_list.set_active(0)
            #if our selected remote itself changed
            elif widget.get_name() == 'remote_list':
                self.generate_lircrc_checkbox.set_active(True)
                if self.remote_list.get_active() == 0:
                    custom = False
                    self.remotecontrol.set_active(False)
                    self.generate_lircrc_checkbox.set_active(False)
                elif self.remote_list.get_active_text() == "Custom":
                    custom = True
                else:
                    custom = False
                    self.remote_driver.set_text("")
                    self.remote_modules.set_text("")
                    self.remote_device.set_text("")
                self.remote_driver_hbox.set_sensitive(custom)
                self.remote_modules_hbox.set_sensitive(custom)
                self.remote_device_hbox.set_sensitive(custom)
                self.remote_configuration_hbox.set_sensitive(custom)
                self.browse_remote_lircd_conf.set_filename("/usr/share/lirc/remotes")
            #if our selected transmitter itself changed
            elif widget.get_name() == 'transmitter_list':
                if self.transmitter_list.get_active() == 0:
                    custom = False
                    self.transmittercontrol.set_active(False)
                elif self.transmitter_list.get_active_text() == "Custom":
                    custom = True
                else:
                    custom = False
                    self.transmitter_driver.set_text("")
                    self.transmitter_modules.set_text("")
                    self.transmitter_device.set_text("")
                self.transmitter_driver_hbox.set_sensitive(custom)
                self.transmitter_modules_hbox.set_sensitive(custom)
                self.transmitter_device_hbox.set_sensitive(custom)
                self.transmitter_configuration_hbox.set_sensitive(custom)
                self.browse_transmitter_lircd_conf.set_filename("/usr/share/lirc/transmitters")

    def do_connection_test(self,widget):
        """Tests to make sure that the backend is accessible"""
        self.mysql_test_hbox.show()
        host = self.mysql_server.get_text()
        database = self.mysql_database.get_text()
        user = self.mysql_user.get_text()
        password = self.mysql_pass.get_text()
        try:
            db = MySQLdb.connect(host=host, user=user, passwd=password,db=database)
            cursor = db.cursor()
            cursor.execute("SELECT NULL")
            result = cursor.fetchone()
            cursor.close()
            db.close()
            result = "Successful"
            self.fail_mysql.hide()
            self.pass_mysql.show()
        except:
            result = "Failure"
            self.fail_mysql.show()
            self.pass_mysql.hide()
        self.connection_results_label.show()
        self.connection_results.set_text(result)

    def get_db_setting(self,val,hostname):
        """Get entries om the settings table in the MySQL database"""
        """First argument is the entry in the 'value' column"""
        """The second argument is the host name (can be None)"""
        """The first matching 'data' entry will be returned"""
        # exception handling is done by the caller
        host = self.mysql_server.get_text()
        database = self.mysql_database.get_text()
        user = self.mysql_user.get_text()
        password = self.mysql_pass.get_text()
        db = MySQLdb.connect(host=host, user=user, passwd=password,db=database)
        cursor = db.cursor()
        query = 'select data from settings where value = \'' + val + '\''
        if hostname is not None:
            query = query + ' and hostname = \'' + hostname + '\''
        cursor.execute(query)
        res = cursor.fetchone()[0]
        cursor.close()
        db.close()
        return res


    def toggle_role(self,widget):
        """Called when a role change occurs"""
        if widget is not None:
            name=widget.get_name()
            activate_artwork=False
            if name == "no_backend_radio" or name == "primary_backend_radio" or name == "secondary_backend_radio":
                old_active=self.config.get("mythbuntu","backend")
                if (self.no_backend_radio.get_active() and old_active != "none"):
                    self.role_toggled=True
                elif (self.primary_backend_radio.get_active() and old_active != "primary") or \
                   (self.secondary_backend_radio.get_active() and old_active != "secondary"):
                    activate_artwork=True
                    self.role_toggled=True
            elif name == "no_frontend_radio" or name == " frontend_radio":
                old_active=self.config.get("mythbuntu","frontend")
                if (self.no_frontend_radio.get_active() and old_active != "none"):
                    self.role_toggled=True
                elif (self.frontend_radio.get_active() and old_active != "frontend" ):
                    activate_artwork=True
                    self.role_toggled=True
            if activate_artwork:
                self.enable_artwork.set_active(True)

    def launch_app(self,widget):
        """Launches an external application"""
        def getUserApp():
            uid = os.getenv('SUDO_UID')
            if uid == None:
                return ""
            else:
                in_f=open("/etc/passwd")
                for line in in_f:
                    fields =  string.split(line,":")
                    if fields[2] == uid:
                        return "su " + fields[0] + ' -c '

        if widget is not None:
            self.main_window.hide()
            while gtk.events_pending():
                gtk.main_iteration()
            if widget.get_name() == 'restricted_drivers_button':
                os.system('/usr/bin/jockey-gtk')
            elif widget.get_name() == 'amdcccle_button':
                os.system('/usr/bin/amdcccle')
            elif widget.get_name() == 'nvidia_settings_button':
                os.system('/usr/bin/nvidia-settings')
            elif widget.get_name() == 'mythtv_setup_button':
                os.system(getUserApp() + '"/usr/bin/mythtv-setup"')
            elif widget.get_name() == 'displayconfig_button':
                os.system("/usr/bin/displayconfig-gtk")
            elif widget.get_name() == 'synaptic_button':
                os.system("/usr/sbin/synaptic")
            elif widget.get_name() == 'updatemanager_button':
                os.system("/usr/bin/update-manager")
            elif widget.get_name() == 'terminal_button':
                os.system(getUserApp() + '"x-terminal-emulator"')
            elif widget.get_name() == 'mysql_tweak_button':
                os.system(getUserApp() + '"/usr/bin/x-terminal-emulator -e perl /usr/share/doc/mythtv-backend/contrib/optimize_mythdb.pl"')
            self.main_window.show()


    def on_mythweb_checkbox_toggled(self,widget):
        if self.mythweb_checkbox.get_active() == False:
            self.mythweb_password_checkbutton.set_active(False)
            self.mythweb_password_checkbutton.set_sensitive(False)
        else:
            # only make mythweb auth UI available if mythweb is installed!
            # for some reason, config.get does not work here. query_installed might be slower but.. oh well.
            if self.query_installed("mythweb"):
                self.mythweb_password_checkbutton.set_sensitive(True)
        # Note: it's not a good idea to enable the mythweb auth UI unless mythweb is completely installed
        # The UI will be activated once m-c-c reloads its config

    # helper procedure to toggle diskless_arch combo box
    def diskless_arch_toggle(self):
        item = self.diskless_arch.get_active()
        self.diskless_arch.set_active(0)
        self.diskless_arch.set_active(item)

    def on_diskless_arch_changed(self,widget):
        if  self.diskless_arch.get_active() != 0:
            myarch = self.diskless_arch.get_active_text()
            self.diskless_preparation_action_table.set_sensitive(True)
            if os.path.exists(ltsp_base_dir + myarch):
                self.diskless_delete_image_hbox.set_sensitive(True)
                self.diskless_build_image_hbox.set_sensitive(False)
                self.diskless_unsigned_packages.set_sensitive(False)
            else:
                self.diskless_delete_image_hbox.set_sensitive(False)
                self.diskless_build_image_hbox.set_sensitive(True)
                self.diskless_unsigned_packages.set_sensitive(True)
        else:
            self.diskless_preparation_action_table.set_sensitive(False)

        #Either way, we don't know what we are doing to the new selection
        self.diskless_delete_image_button.set_active(False)
        self.diskless_build_image_button.set_active(False)

    def on_diskless_pendrive_changed(self,widget):
        device = self.diskless_pendrive.get_active_text()
        mypath =  '/sys/block/'
        modelpath = mypath + device + '/device/model'
        if self.diskless_pendrive.get_active() != 0:
            if self.diskless_nbd_server_nic.get_active() != 0:
                self.diskless_pendrive_action_hbox.set_sensitive(True)
            model = open(modelpath)
            self.diskless_pendrive_model.set_text(model.readline().rstrip())
            model.close()
        elif self.diskless_pendrive.get_active() == 0:
            self.diskless_pendrive_model.set_text(self.diskless_pendrive.get_active_text())
            self.diskless_pendrive_action_hbox.set_sensitive(False)

    def on_diskless_nbd_server_nic_changed(self,widget):
        if self.diskless_nbd_server_nic.get_active() != 0 and self.diskless_pendrive.get_active() != 0:
            self.diskless_pendrive_action_hbox.set_sensitive(True)
        else:
            self.diskless_pendrive_action_hbox.set_sensitive(False)

    def on_diskless_chroot_changed(self,widget):
        if self.diskless_chroot.get_active() == 0:
            self.diskless_maintenance_box.set_sensitive(False)
            self.diskless_commit.set_sensitive(False)
            self.diskless_pendrive.set_active(0)
            self.diskless_nbd_server_nic.set_active(0)
            self.diskless_write_drive_button.set_active(False)
        else:
            self.diskless_maintenance_box.set_sensitive(True)
            self.diskless_commit.set_sensitive(True)

    def on_diskless_shell_clicked(self,widget):
        if self.diskless_chroot.get_active() != 0:
            chroot = ltsp_base_dir +  self.diskless_chroot.get_active_text() + '/'
            if self.diskless_prepare_x(chroot) == 0:
                os.system('xterm -e chroot ' + chroot )
            # clean up even with diskless_prepare_x failed. Maybe we need better error handling,
            # eg only clean up things we have already done before something failed
            self.diskless_unprepare_x(chroot)
        else:
            self.display_error('Please choose a chroot!')

    def on_diskless_mcc_clicked(self,widget):
        if self.diskless_chroot.get_active() != 0:
            chroot = ltsp_base_dir +  self.diskless_chroot.get_active_text() + '/'
            if self.diskless_prepare_x(chroot) == 0:
                if os.path.exists(chroot + '/usr/bin/mythbuntu-control-centre'):
                    os.system('chroot ' + chroot + ' /usr/bin/mythbuntu-control-centre')
                else:
                    self.display_error('Sorry, I couldn\'t find mythbuntu-control-centre in the chroot!')
            # clean up even with diskless_prepare_x failed. Maybe we need better error handling,
            # eg only clean up things we have already done before something failed
            self.diskless_unprepare_x(chroot)
        else:
            self.display_error('Please choose a chroot!')

    def on_diskless_commit_clicked(self,widget):
        if self.diskless_chroot.get_active() != 0:
            chroot = self.diskless_chroot.get_active_text()
            os.system('xterm -hold -e ltsp-update-image -b ' + ltsp_base_dir + ' -a ' + chroot)
        else:
            self.display_error('Please choose a chroot!')

    def diskless_prepare_x(self,chroot):
        if os.system('xhost +local:') != 0:
            self.display_error('Sorry, I couln\'t run prepare your X server to accept connections from the chroot.\n (xhost failed)')
            return 1
        if os.system('mkdir -p ' + chroot + '/tmp/.X11-unix') != 0:
            self.display_error('Sorry, I couldn\'t create ' + chroot + '/tmp/.X11-unix')
            return 2
        if os.system('mount --bind /tmp/.X11-unix ' +  chroot + '/tmp/.X11-unix') != 0:
            self.display_error('Sorry, I couldn\'t bind-mount /tmp/.X11-unix to ' +  chroot + '/tmp/.X11-unix')
            return 3
        if os.system('mount --bind /proc ' + chroot + '/proc') != 0:
            self.display_error('Sorry, I coudln\'t bind mount the proc filesystem')
            return 4
        if os.system('mount --bind /sys ' + chroot + '/sys') != 0:
            self.display_error('Sorry, I coudln\'t bind mount the sys filesystem')
            return 5
        if os.system('mount --bind /dev ' + chroot + '/dev') != 0:
            self.display_error('Sorry, I coudln\'t bind mount the dev filesystem')
        return 0

    def diskless_unprepare_x(self,chroot):
        if os.system('xhost -local:') != 0:
            self.display_error('Sorry, I couldn\'t remove non-network local connections from the access list of your X server. Please check \'xhost\' manually')
            # we do not return here because we want to clean up as much as possible
        if os.system('umount  ' +  chroot + '/tmp/.X11-unix') != 0:
            self.display_error('Sorry, I couldn\'t unmount ' + chroot + '/tmp/.X11-unix. Please unmount it manually and be careful when deleting this chroot.')
        if os.system('umount ' + chroot + '/proc') != 0:
            self.display_error('Sorry, I couldn\'t umount the proc filesystem')
        if os.system('umount ' + chroot + '/sys') != 0:
            self.display_error('Sorry, I couldn\'t umount the sys filesystem')
        if os.system('umount ' + chroot + '/dev') != 0:
            self.display_error('Sorry, I couldn\'t umount the dev filesystem')

    def on_diskless_wol_checkbox_toggled(self,widget):
        if self.diskless_wol_checkbox.get_active():
            self.diskless_wol_entry.set_sensitive(True)
        else:
            self.diskless_wol_entry.set_sensitive(False)

    # dependency handling for dhcp server installation - we need mythbuntu-diskless-server
    def on_diskless_server_install_toggled(self,widget=None):
        if self.diskless_server_install.get_active():
            self.diskless_server_dhcp.set_sensitive(True)
            # we always want to force reconfiguration if we don't have /etc/ltsp/dhcpd.conf
            if self.query_installed("dhcp3-server") and os.path.isfile('/etc/ltsp/dhcpd.conf'):
                self.diskless_server_dhcp.set_activce(True)
            else: 
                self.diskless_server_dhcp.set_active(False)
        else:
            self.diskless_server_dhcp.set_sensitive(False)
            self.diskless_server_dhcp.set_active(False)

