/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */


#include "MGResultSetModel.h"

#include "MGCellRendererBlob.h" // data2str()

    //XXX TODO fix the model stuff so that there's only 1 action per cell
    // when it's submited to apply_actions

MGResultSetModel::MGResultSetModel(const Gtk::TreeModelColumnRecord& columns,
                                   MYX_RESULTSET *resultset)
: Glib::ObjectBase(typeid(MGResultSetModel)), //register a custom GType.
  Glib::Object(), //The custom GType is actually registered here.
  _stamp(1), //When the model's stamp != the iterator's stamp then that iterator is invalid and should be ignored. Also, 0=invalid
  _resultset(resultset),
  _edit_enabled(false),
  _disable_checks(true),
  _new_rows(0)
{
//  Gtk::TreeModel::add_interface(Glib::Object::get_type());

//  _column_record.add();
}


MGResultSetModel::~MGResultSetModel()
{ 
  for (unsigned int i= 0; i < _diff_colors.size(); i++)
    g_free(_diff_colors[i]);
  for (unsigned int i= 0; i < _state_colors.size(); i++)
    g_free(_state_colors[i]);
}


Glib::RefPtr<MGResultSetModel> MGResultSetModel::create(const Gtk::TreeModelColumnRecord& columns,
                                                        MYX_RESULTSET *resultset)
{
  return Glib::RefPtr<MGResultSetModel>(new MGResultSetModel(columns, resultset));
}


Gtk::TreeModelFlags MGResultSetModel::get_flags_vfunc()
{
  return Gtk::TREE_MODEL_ITERS_PERSIST;
}


int MGResultSetModel::get_n_columns_vfunc()
{
  return _resultset->columns_num*2+(int)SpecialLast;
}


GType MGResultSetModel::get_column_type_vfunc(int index)
{
  if (index & 1)
    return G_TYPE_STRING;
  else
  {
    switch (_resultset->columns[index/2].column_type)
    {
    case MYX_RSCT_BLOB:
    case MYX_RSCT_TEXT:
      return G_TYPE_STRING;
    default:
      return G_TYPE_STRING;
    }
  }
}


bool MGResultSetModel::iter_next_vfunc(GtkTreeIter* iter)
{
  if (_disable_checks || treeiter_valid(iter))
  {
    iter->stamp= _stamp;

    iter->user_data= (gpointer)((int)iter->user_data+1);

    if ((unsigned)iter->user_data > _resultset->rows_num)
      g_message("QEWQWE!!!");

    return (int)iter->user_data < (int)_resultset->rows_num+_new_rows ? true : false;
  }
  iter->stamp= 0;
  return false;
}


bool MGResultSetModel::iter_children_vfunc(GtkTreeIter* iter, const GtkTreeIter* parent)
{
  g_message("!");
  iter->stamp= 0;
  //XXX
  return false;
}


bool MGResultSetModel::iter_has_child_vfunc(const GtkTreeIter* iter)
{
  if (!iter && _resultset && _resultset->rows_num+_new_rows>0)
  {
    return true;
  }
  return false;
}


int MGResultSetModel::iter_n_children_vfunc(const GtkTreeIter* iter)
{
  if (!iter)
  {
    return _resultset->rows_num + _new_rows;
  }
  return 0;
}


bool MGResultSetModel::iter_nth_child_vfunc(GtkTreeIter* iter, const GtkTreeIter* parent, int n)
{
  if (!parent)
  {
    iter->stamp= _stamp;
    iter->user_data= (gpointer)n;
    return true;
  }
  g_message("!!! %p %i",parent,n);
  iter->stamp= 0;
  return false;
}


bool MGResultSetModel::iter_parent_vfunc(GtkTreeIter* iter, const GtkTreeIter* child)
{
  g_message("++");
  iter->stamp= 0;
  return false;
}


void MGResultSetModel::ref_node_vfunc(GtkTreeIter* iter)
{
//  g_message("REFNODE");
}


void MGResultSetModel::unref_node_vfunc(GtkTreeIter* iter)
{
//  g_message("unrefnode");
}


Gtk::TreeModel::Path MGResultSetModel::get_path_vfunc(const Gtk::TreeModel::iterator& iter)
{
  if (treeiter_valid(iter))
  {
    Path path;

    path.push_back((int)iter.gobj()->user_data);
    
    return path;
  }
  return Path();
}


bool MGResultSetModel::get_iter_vfunc(GtkTreeIter* iter, const Gtk::TreeModel::Path& path)
{
  int len= path.size();

  if (len != 1)
  {
    g_warning("bad path");
    iter->stamp= 0;
    return false;
  }

  unsigned int row_index= path[0];

  if (row_index >= _resultset->rows_num)
  {
    if (row_index >= _resultset->rows_num + _new_rows)
    {
      iter->stamp= 0;
      return false;
    }
  }
  iter->stamp= _stamp;
  iter->user_data= (gpointer)row_index;

  return true;
}



bool MGResultSetModel::get_value(const Gtk::TreeModel::iterator& row, int column, gpointer &value, guint &value_length)
{
  const GtkTreeIter *iter= row.gobj();

  if (_disable_checks || treeiter_valid(iter))
  {
    int r= (int)iter->user_data;

    if (column < (int)_resultset->columns_num*2)
    {
      // check if there's some changed version of the data
      switch (column % 2)
      {
      case 0: // data
        {
          const char *svalue;
          if (!_actions.empty() && find_change(r, column/2, svalue, value_length))
          {
            value= (gpointer)svalue;
            return true;
          }
          
          if ((unsigned int)iter->user_data < _resultset->rows_num)
          {
            MYX_RS_ROW *row= _resultset->rows + r;
            
            if (!row->fields)
            {
              value= NULL;
              value_length= 0;
            }
            else
            {
              value= row->fields[(column/2)].value;
              value_length= row->fields[(column/2)].value_length;
            }
          }
          else
          {
            value= NULL;
            value_length= 0;
          }
        }
        break;
      case 1: // color
        {
          char *color= NULL;

          // 1st check diff
          if (r < (int)_resultset->rows_num)
          {
            MYX_RS_ROW *row= _resultset->rows + r;
            switch (row->diff&MYX_RD_MASK)
            {
            case MYX_RD_OTHER_ONLY:
              color= _diff_colors[0];
              break;
            case MYX_RD_THIS_ONLY:
              color= _diff_colors[1];
              break;
            case MYX_RD_DIFFERS:
              // check column specific differences
              if ((row->diff>>4) & (1<<(column/2)))
                color= _diff_colors[2];
              break;
            }
          }

          // 2nd. changes color overrides diff color
          if (_actions.find(r)!=_actions.end()) // find out state
          {
            MYX_RS_ACTION *action= get_action(*_actions[r].begin());
            if (action && action->action == MYX_RSA_DELETE)
            {
              color= _state_colors[0];
            }
            else if (!action || action->action == MYX_RSA_ADD)
              color= _state_colors[1];
            else
              color= _state_colors[2];
          }
          else
          {
            // nothing here, use default color
          }

          value= color;
          value_length= 0;
        }
        break;
      }
      return true;
    }
  }
  else
    g_warning("invalid iterator!");
  
  return false;
}


void MGResultSetModel::get_value_vfunc(const Gtk::TreeModel::iterator& iter_, int column, GValue* value)
{
  const GtkTreeIter *iter= iter_.gobj();

  if (_disable_checks || treeiter_valid(iter))
  {
    int r= (int)iter->user_data;
    
    // from 0 to 2*_resultset->columns_num, the columns correspond
    // to the content (text or blob ptr) and color of the resultset,
    // alternately.
    // values bigger than that, are special attributes as defined
    // in enum SpecialColumns
    
    if (column >= (int)_resultset->columns_num*2)
    {
      // hack: this is for manually setting special column values
      switch (column-(int)_resultset->columns_num*2)
      {
      case RowEditable:
        {
          bool editable= _edit_enabled;

          /* this is for the editable flag */
          if (editable)
          {
            // check whether this particular row can be edited
            if (_actions.find(r)!=_actions.end())
            {
              MYX_RS_ACTION *action= get_action(*_actions[r].begin());
              if (action && action->action == MYX_RSA_DELETE)
              {
                editable= false;
              }
            }
          }
          g_value_init(value, G_TYPE_BOOLEAN);
          g_value_set_boolean(value, (gboolean)editable);
        }
        break;
      case RowStatusIcon:
//        break;
      default:
        g_value_init(value, G_TYPE_OBJECT);
        g_value_set_object(value, NULL);
        break;
      }
    }
    else if (column < (int)_resultset->columns_num*2)
    {
      bool tmp= _disable_checks;
      gpointer data= NULL;
      guint size= 0;
      bool is_blob;
      switch (_resultset->columns[column/2].column_type)
      {
      case MYX_RSCT_BLOB:
      case MYX_RSCT_TEXT:
        is_blob= true;
        break;
      default:
        is_blob= false;
      }

      g_value_init(value, G_TYPE_STRING);
      
      _disable_checks= true;
      if (get_value(iter_, column, data, size))
      {
        if ((column % 2) == 1) // color
        {
          g_value_set_string(value, (char*)data);
        }
        else
        {
          if (is_blob)
            g_value_set_string(value, data2str(data, size).c_str());
          else
            g_value_set_static_string(value, (gchar*)data);
        }
      }
      else
      {
        if (is_blob)
          g_value_set_string(value, data2str(NULL,0).c_str());
      }
      _disable_checks= tmp;
    }
  }
  else
    g_warning("invalid iterator!");
}


void MGResultSetModel::set_value_impl(const Gtk::TreeModel::iterator& row, int column, const Glib::ValueBase& value)
{
  // dummy, not used by our code
}


void MGResultSetModel::set_value(const Gtk::TreeModel::iterator& row, int column, const gpointer value, guint value_length)
{
  switch ((column%2))
  {
  case 0:
    add_change((unsigned int)row.gobj()->user_data, column/2, value, value_length);
    break;
  case 1:
    // color, ignore
    break;
  }
}


bool MGResultSetModel::treeiter_valid(const Gtk::TreeModel::iterator &iter)
{
  return iter.gobj()->stamp == _stamp;
}


bool MGResultSetModel::treeiter_valid(const GtkTreeIter *iter)
{
  return iter->stamp == _stamp;
}


void MGResultSetModel::notify_resultset_changed()
{
  _stamp++;
}


void MGResultSetModel::set_editable(bool flag)
{
  _edit_enabled= flag;
}


bool MGResultSetModel::get_editable()
{
  return _edit_enabled;
}


void MGResultSetModel::set_state_colors(const std::vector<Glib::ustring> &colors)
{
  for (unsigned int i= 0; i < _state_colors.size(); i++)
    g_free(_state_colors[i]);

  _state_colors.reserve(colors.size());
  for (unsigned int i= 0; i < colors.size(); i++)
  {
    _state_colors[i]= g_strdup(colors[i].c_str());
  }
}


void MGResultSetModel::set_diff_colors(const std::vector<Glib::ustring> &colors)
{
  for (unsigned int i= 0; i < _diff_colors.size(); i++)
    g_free(_diff_colors[i]);

  _diff_colors.reserve(colors.size());
  for (unsigned int i= 0; i < colors.size(); i++)
  {
    _diff_colors[i]= g_strdup(colors[i].c_str());
  }
}


MYX_RS_ROW *MGResultSetModel::get_iter_row(const Gtk::TreeModel::iterator &iter)
{
  return _resultset->rows + (int)iter.gobj()->user_data;
}


MYX_RS_ACTION *MGResultSetModel::get_action(int action_index)
{
  if (action_index < 0) // -1 = new row marker
    return NULL;
  return _resultset->actions->actions+action_index;
}


void MGResultSetModel::add_change(int row, int column, const gpointer data, guint size)
{
  MYX_RS_ACTION *action;
  int index;

  // check whether the row was deleted
  if (_actions.find(row)!=_actions.end())
  {
    action= get_action(*_actions[row].begin());
    if (action && action->action == MYX_RSA_DELETE)
    {
      g_warning("trying to modify a deleted row!");
      return;
    }
  }

  // check whether the new value is the same as the old
  if (row < (int)_resultset->rows_num)
  {
    if (_resultset->rows[row].fields == NULL)
    {
      if (size == 0)
        return;
    } else if (_resultset->rows[row].fields[column].value_length == size
               && memcmp(_resultset->rows[row].fields[column].value, data, size) == 0)
        return; // no change
  }
  else
  {
    const char *d;
    unsigned int s;

    // check whether the new value is the same as the old
    if (find_change(row, column, d, s) && s == size && memcmp(d, data, s)==0)
      return; // no change
  }

  action= myx_query_create_action(row >= (int)_resultset->rows_num ? MYX_RSA_ADD : MYX_RSA_UPDATE,
                                  row, _resultset->columns+column,
                                  size, (const char*)data);

  index= myx_query_add_action(_resultset, action);

  myx_query_free_action(action);

  _actions[row].push_front(index);
}


bool MGResultSetModel::find_change(int row, int column,
                                   const char *&data, unsigned int &size)
{
  ActionList::iterator rowa= _actions.find(row);

  if (rowa!=_actions.end())
  {
    for (std::list<int>::const_iterator iter= rowa->second.begin();
         iter != rowa->second.end(); ++iter)
    {
      MYX_RS_ACTION *action= get_action(*iter);
      if (!action)
      {
        // new row
        data= "";
        size= 0;
        return true;
      } 
      else if (action->action == MYX_RSA_DELETE)
        return false;
      else if (action->column - _resultset->columns == column)
      {
        data= action->new_value;
        size= action->new_value_length;
        return true;
      }
    }
  }
  return false;
}


void MGResultSetModel::apply_edits()
{
  Gtk::TreeModel::Path path;

  for (ActionList::iterator iter= _actions.begin();
       iter != _actions.end(); ++iter)
  {
    MYX_RS_ACTION *action;
    path.clear();
    path.push_back(iter->first);
    action= get_action(*iter->second.begin());
    if (action && action->action == MYX_RSA_DELETE && action->status == MYX_RSAS_APPLIED)
    {
      row_deleted(path);
    }
  }
  _actions.clear();
  _new_rows= 0;

  myx_query_update_resultset(_resultset);

  if (_resultset->actions)
  {
    for (unsigned int i= 0; i < _resultset->actions->actions_num; i++)
    {
      MYX_RS_ACTION *action= _resultset->actions->actions+i;

      _actions[action->row].push_front(i);
      if (action->row >= _resultset->rows_num)
        _new_rows= std::max((unsigned int)_new_rows, action->row - _resultset->rows_num + 1);
    }
  }
}


void MGResultSetModel::discard_edits()
{
  Gtk::TreeModel::Path path;

  for (ActionList::const_iterator iter= _actions.begin();
       iter != _actions.end(); ++iter)
  {
    MYX_RS_ACTION *action;
    path.clear();
    path.push_back(iter->first);
    action= get_action(*iter->second.begin());
    if (!action || action->action == MYX_RSA_ADD)
      row_deleted(path);
    else
      row_changed(path, get_iter(path));
  }

  _actions.clear();
  myx_query_discard_actions(_resultset);
  _new_rows= 0;
}


Gtk::TreeModel::iterator MGResultSetModel::append()
{
  Gtk::TreeModel::iterator iter(this);
  int new_row;

  new_row= _resultset->rows_num + _new_rows;
  _new_rows++;

  if (_actions.find(new_row)!=_actions.end())
    g_warning("trying to add row that already exists?");

  // insert new row marker
  _actions[new_row].push_front(-1);

  // make new iterator
  iter.gobj()->stamp= _stamp;
  iter.gobj()->user_data= (gpointer)new_row;

  // emit signal
  row_inserted(get_path(iter), iter);

  return iter;
}


void MGResultSetModel::erase(const Gtk::TreeModel::iterator &iter)
{
  unsigned int row= (unsigned int)iter.gobj()->user_data;
  MYX_RS_ACTION *action;
  int index;

  action= myx_query_create_action(MYX_RSA_DELETE, row, NULL, 0, NULL);

  index= myx_query_add_action(_resultset, action);

  myx_query_free_action(action);

  _actions[row].push_front(index);
}


void MGResultSetModel::undo(const Gtk::TreeModel::iterator &iter)
{
  int row= (int)iter.gobj()->user_data;

  if (_actions.find(row)==_actions.end())
  {
    g_warning("trying to undo when there's nothing to undo");
    return;
  }

  if (*_actions[row].begin() >= 0)
    myx_query_delete_action(_resultset, *_actions[row].begin());

  int index= *_actions[row].begin();
  
  _actions[row].erase(_actions[row].begin());
  if (_actions[row].empty())
    _actions.erase(_actions.find(row));

  // now this sucks... but we have to update all indexes > index
  // since they were shifted in _resultset->actions
  for (ActionList::iterator it= _actions.begin(); it != _actions.end(); ++it)
  {
    for (std::list<int>::iterator jter= it->second.begin();
         jter != it->second.end(); ++jter)
    {
      if (*jter > index)
        (*jter)--;
    }
  }

  if (row > (int)_resultset->rows_num && _actions.find(row)==_actions.end())
  {
    // emit deleted signal if this was a new row
    row_deleted(get_path(iter));
  }
  else
  {
    row_changed(get_path(iter), iter);
  }
}


bool MGResultSetModel::row_undoable(const Gtk::TreeModel::iterator &iter)
{
  int row= (int)iter.gobj()->user_data;
  if (_actions.find(row)==_actions.end())
    return false;
  return true;
}

