unit Main;

// This is a small demonstration program to show how to use the UniCodeEdit control.
// Implemented is a simple text editor which can do syntax highlighting. Despite the Unicode
// ability there's nothing special with the editor. It acts quite similar to Notepad.
// Note: Only fixed pitched fonts produce correct display results.
//
// (c) 2000, Dipl. Ing. Mike Lischke

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms,
  UCEDelphiHighlighter, UCECPPHighlighter, ExtCtrls, StdCtrls,
  ActnList, Buttons, Menus, Unicode, StdActns, Dialogs,
  ImgList, ComCtrls, ToolWin,
  UCESQLHighlighter, UCEHighlighter,
  UCEHTMLHighlighter, UniCodeEditor;

type
  TMainForm = class(TForm)
    MenuImageList: TImageList;
    OpenDialog: TOpenDialog;
    StatusBar: TStatusBar;
    MainMenu1: TMainMenu;
    File1: TMenuItem;
    Edit1: TMenuItem;
    MainControlBar: TControlBar;
    MainActionList: TActionList;
    EditCopyAction: TEditCopy;
    EditCutAction: TEditCut;
    EditPasteAction: TEditPaste;
    MenuToolBar: TToolBar;
    ToolButton4: TToolButton;
    ToolButton5: TToolButton;
    Cut1: TMenuItem;
    Copy1: TMenuItem;
    Paste1: TMenuItem;
    ButtonToolBar: TToolBar;
    ToolButton1: TToolButton;
    ToolButton2: TToolButton;
    ToolButton3: TToolButton;
    HLComboBox: TComboBox;
    FileOpenAction: TAction;
    FileSaveAction: TAction;
    OpenFile1: TMenuItem;
    Save1: TMenuItem;
    FileCloseAction: TAction;
    N1: TMenuItem;
    Close1: TMenuItem;
    FileNewAction: TAction;
    ToolButton6: TToolButton;
    ToolButton7: TToolButton;
    ToolButton8: TToolButton;
    ToolButton9: TToolButton;
    ToolButton10: TToolButton;
    EditControlCharAction: TAction;
    N2: TMenuItem;
    ControlChars1: TMenuItem;
    EditUndoAction: TAction;
    EditRedoAction: TAction;
    ToolButton11: TToolButton;
    ToolButton12: TToolButton;
    ToolButton13: TToolButton;
    N3: TMenuItem;
    EditUndoAction1: TMenuItem;
    EditRedoAction1: TMenuItem;
    FileSaveAsAction: TAction;
    SaveAs1: TMenuItem;
    SaveDialog: TSaveDialog;
    ToolButton14: TToolButton;
    SearchFindAction: TAction;
    Search1: TMenuItem;
    Find1: TMenuItem;
    Findnext1: TMenuItem;
    SearchNextAction: TAction;
    ToolButton16: TToolButton;
    SearchReplaceAction: TAction;
    SearchReplaceAction1: TMenuItem;
    FontDialog: TFontDialog;
    FontAction: TAction;
    ToolButton19: TToolButton;
    SearchToolBar: TToolBar;
    ToolButton21: TToolButton;
    ToolButton22: TToolButton;
    ToolButton23: TToolButton;
    Options1: TMenuItem;
    GroupUndoItem: TMenuItem;
    AutoIndentItem: TMenuItem;
    InsertItem: TMenuItem;
    TabItem: TMenuItem;
    SmartItem: TMenuItem;
    OptimalItem: TMenuItem;
    UnindentItem: TMenuItem;
    CursorTabItem: TMenuItem;
    CursorEOLItem: TMenuItem;
    UndoItem: TMenuItem;
    BlanksItem: TMenuItem;
    HighlightItem: TMenuItem;
    ControlItem: TMenuItem;
    ReadOnlyItem: TMenuItem;
    NumberItem: TMenuItem;
    HintItem: TMenuItem;
    TripleItem: TMenuItem;
    ToolButton15: TToolButton;
    HideSelectionItem: TMenuItem;
    ToolButton17: TToolButton;
    Edit2: TEdit;
    UpDown1: TUpDown;
    ToolButton18: TToolButton;
    SyntaxEdit1: TUniCodeEdit;
    HTMLHighlighter: TUCEHTMLHighlighter;
    SQLHighlighter: TUCESQLHighlighter;
    DelphiHighlighter: TUCEDelphiHighlighter;
    CPPHighlighter: TUCECPPHighlighter;
    procedure FormCreate(Sender: TObject);
    procedure HLComboBoxChange(Sender: TObject);
    procedure SyntaxEdit1CaretChange(Sender: TObject; X, Y: Cardinal);
    procedure SyntaxEdit1SettingChange(Sender: TObject);
    procedure SyntaxEditDragOver(Sender, Source: TObject; X, Y: Integer; State: TDragState; var Accept: Boolean);
    procedure FileNewActionExecute(Sender: TObject);
    procedure FileOpenActionExecute(Sender: TObject);
    procedure EditCopyActionExecute(Sender: TObject);
    procedure EditCutActionExecute(Sender: TObject);
    procedure EditPasteActionExecute(Sender: TObject);
    procedure EditUndoActionExecute(Sender: TObject);
    procedure EditRedoActionExecute(Sender: TObject);
    procedure FileCloseActionExecute(Sender: TObject);
    procedure FileSaveAsActionExecute(Sender: TObject);
    procedure FileSaveActionExecute(Sender: TObject);
    procedure SearchFindActionExecute(Sender: TObject);
    procedure SearchNextActionExecute(Sender: TObject);
    procedure FontActionExecute(Sender: TObject);
    procedure OptionMenuClick(Sender: TObject);
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
    procedure SyntaxEdit1ReplaceText(Sender: TObject; const Search, Replace: WideString; Line, Start, Stop: Integer;
      var ReplaceAction: TReplaceAction);
    procedure Edit2Change(Sender: TObject);
  private
    FFileFilter: WideString;
    FLastSearchOptions: TSearchOptions;
    procedure WMDropFiles(var Msg: TWMDropFiles); message WM_DROPFILES;
    procedure DoLoad(const FileName: WideString; ReadOnly: Boolean = False);
    procedure OnHint(Sender: TObject);
    procedure ConfirmConversion(Sender: TWideStrings; var Allowed: Boolean);
  protected
    procedure UpdateActions; override;
  end;

var
  MainForm: TMainForm;               

//----------------------------------------------------------------------------------------------------------------------

implementation

uses
  Clipbrd, ShellAPI, Find;
                                  
{$R *.DFM}

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.DoLoad(const FileName: WideString; ReadOnly: Boolean = False);

var
  I: Integer;
  Ext: WideString;                    

begin
  SyntaxEdit1.Lines.LoadFromFile(FileName);
  SyntaxEdit1.Info := FileName;
  SyntaxEdit1.Modified := False;
  if ReadOnly then SyntaxEdit1.Options := SyntaxEdit1.Options + [eoReadOnly];
  StatusBar.Panels[1].Text := '';
  Caption := ExtractFileName(FileName) + ' - Editor';
  // try to find a proper highlighter for the newly loaded file
  Ext := ExtractFileExt(FileName);
  with HLComboBox do
  begin
    ItemIndex := 0;
    for I := 1 to Items.Count -1 do
      if Pos(Ext, TUCEHighlighter(Items.Objects[I]).DefaultFilter) > 0 then
      begin
        ItemIndex := I;
        Break;
      end;
    HLComboBoxChange(nil);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.FormCreate(Sender: TObject);

var
  I: Integer;

begin
  FileNewAction.Execute;
  FFileFilter := '';
  // collect all available highlighters
  HLComboBox.Items.AddObject('no highlighter', nil);
  for I := 0 to ComponentCount - 1 do
    if Components[I] is TUCEHighlighter then
      with Components[I] as TUCEHighlighter do
      begin
        FFileFilter := FFileFilter + DefaultFilter + '|';
        HLComboBox.Items.AddObject(LanguageName, Self.Components[I]);
      end;
  FFileFilter := FFileFilter + 'Text files (*.txt)|*.txt|Unicode text files (*.uni)|*.uni|all files (*.*)|*.*';
  OpenDialog.Filter := FFileFilter;
  SaveDialog.Filter := FFileFilter;
  DragAcceptFiles(Handle, True);
  Application.OnHint := OnHint;
  SyntaxEdit1.Lines.OnConfirmConversion := ConfirmConversion;

  // reflect current edit options into menu items
  with SyntaxEdit1 do
  begin
    GroupUndoItem.Checked := eoGroupUndo in Options;
    AutoIndentItem.Checked := eoAutoIndent in Options;
    InsertItem.Checked := eoInserting in Options;
    TabItem.Checked := eoUseTabs in Options;
    SmartItem.Checked := eoSmartTabs in Options;
    OptimalItem.Checked := eoOptimalFill in Options;
    UnindentItem.Checked := eoAutoUnindent in Options;
    CursorTabItem.Checked := eoCursorThroughTabs in Options;
    CursorEOLItem.Checked := eoScrollPastEOL in Options;
    BlanksItem.Checked := eoKeepTrailingBlanks in Options;
    HighlightItem.Checked := eoUseSyntaxHighlighting in Options;
    UndoItem.Checked := eoUndoAfterSave in Options;
    ControlItem.Checked := eoShowControlChars in Options;
    ReadOnlyItem.Checked := eoShowCursorWhileReadOnly in Options;
    NumberItem.Checked := eoLineNumbers in Options;
    HintItem.Checked := eoShowScrollHint in Options;
    TripleItem.Checked := eoTripleClicks in Options;
    HideSelectionItem.Checked := eoHideSelection in Options;
  end;

  // consider command line
  for I := 1 to ParamCount do
    if FileExists(ParamStr(I)) then
    begin
      DoLoad(ParamStr(I));
      Break;
    end;

  UpDown1.Position := SyntaxEdit1.CharWidth;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.HLComboBoxChange(Sender: TObject);

begin
  with HLComboBox, Items do
  begin
    SyntaxEdit1.HighLighter := Objects[ItemIndex] as TUCEHighlighter;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.SyntaxEdit1CaretChange(Sender: TObject; X, Y: Cardinal);

begin
  StatusBar.Panels[0].Text := Format('%d : %d', [Y + 1, X + 1]);
  StatusBar.Update;
  SyntaxEdit1.MarkLine := -1;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.SyntaxEdit1SettingChange(Sender: TObject);

begin
  with SyntaxEdit1 do
    if eoReadOnly in Options then StatusBar.Panels[2].Text := 'Read only'
                             else
      if eoInserting in Options then StatusBar.Panels[2].Text := 'Insert'
                                else StatusBar.Panels[2].Text := 'Overwrite'
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.WMDropFiles(var Msg: TWMDropFiles);

// the common game here if a file has been dropped on the application

var
  Buffer: array[0..MAX_PATH] of Char;
  Count: Integer;

begin
  Count := DragQueryFile(Msg.Drop, DWORD(-1), nil, 0);
  if Count > 0 then
  begin
    DragQueryFile(Msg.Drop, 0, Buffer, MAX_PATH);
    DoLoad(Buffer);
  end;
  DragFinish(Msg.Drop);
  Msg.Result := 0;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.SyntaxEditDragOver(Sender, Source: TObject; X, Y: Integer; State: TDragState; var Accept: Boolean);

begin
  Accept := True;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.FileNewActionExecute(Sender: TObject);

begin
  SyntaxEdit1.ClearAll(False, False);
  SyntaxEdit1.Info := 'New Document.uni';
  SyntaxEdit1.Lines.SaveUnicode := True;
  SyntaxEdit1.Modified := False;
  Caption := 'New document - Editor';
  StatusBar.Panels[1].Text := '';
  StatusBar.Panels[3].Text := '';
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.FileOpenActionExecute(Sender: TObject);

begin
  with OpenDialog do
  begin
    if Execute then DoLoad(FileName, ofReadOnly in Options);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.EditCopyActionExecute(Sender: TObject);

begin
  SyntaxEdit1.CopyToClipboard;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.EditCutActionExecute(Sender: TObject);

begin
  SyntaxEdit1.CutToClipboard;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.EditPasteActionExecute(Sender: TObject);

begin
  SyntaxEdit1.PasteFromClipboard;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.UpdateActions;

const
 ReasonToString: array[TChangeReason] of WideString =
   ('none', 'text insertion', 'text deletion', 'text replacement', 'text movement', 'text copying',
    'cursor movement', 'text overwrite', 'text indentation');

begin
  inherited;
  // misc
  with EditUndoAction do
  begin
    Enabled := SyntaxEdit1.CanUndo;
    if not Enabled then Hint := ''
                   else Hint := 'Undo ' + ReasonToString[SyntaxEdit1.UndoList.GetCurrentUndoReason];
  end;
  with EditRedoAction do
  begin
    Enabled := SyntaxEdit1.CanRedo;
    if not Enabled then Hint := ''
                   else Hint := 'Redo ' + ReasonToString[SyntaxEdit1.UndoList.GetCurrentRedoReason];
  end;

  // file
  FileSaveAction.Enabled := SyntaxEdit1.Modified;

  // edit
  EditCopyAction.Enabled := SyntaxEdit1.SelAvail;
  EditCutAction.Enabled := SyntaxEdit1.SelAvail;
  EditPasteAction.Enabled := Clipboard.HasFormat(CF_TEXT) or
                             Clipboard.HasFormat(CF_UNICODETEXT);
  if SyntaxEdit1.Modified then StatusBar.Panels[1].Text := 'Modified'
                          else StatusBar.Panels[1].Text := '';
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.OnHint(Sender: TObject);

var
  Text: WideString;

begin
  if Mouse.Capture = 0 then
  begin
    Text := GetLongHint(Application.Hint);
    StatusBar.SimpleText := Text;
    StatusBar.SimplePanel := Text <> '';
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.EditUndoActionExecute(Sender: TObject);

begin
  SyntaxEdit1.Undo;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.EditRedoActionExecute(Sender: TObject);

begin
  SyntaxEdit1.Redo;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.FileCloseActionExecute(Sender: TObject);

begin
  Close;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.FileSaveAsActionExecute(Sender: TObject);

begin
  with SaveDialog do
  begin
    if SyntaxEdit1.Info = '' then FileName := 'New document.txt'
                             else FileName := SyntaxEdit1.Info;
    if Execute then
    begin
      SyntaxEdit1.Info := FileName;
      SyntaxEdit1.Lines.SaveToFile(FileName);
      Self.Caption := ExtractFileName(FileName) + ' - Editor';
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.FileSaveActionExecute(Sender: TObject);

begin
  if FileExists(SyntaxEdit1.Info) then
  begin
    SyntaxEdit1.Lines.SaveToFile(SyntaxEdit1.Info);
  end
  else FileSaveAsAction.Execute;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.SearchFindActionExecute(Sender: TObject);

var
  Result: TModalResult;
  CurrentWord: WideString;
  BB, BE: TPoint;

begin
  // the dialog must appear at second place otherwise the edit will be shown modally
  with SyntaxEdit1, SyntaxFindDialog do
  begin
    if (Sender as TAction).Tag = 0 then SearchRadioButton.Checked := True;
    if (Sender as TAction).Tag = 1 then ReplaceRadioButton.Checked := True;
    if not SelAvail then
    begin
      GlobalRadioButton.Checked := True;
      CurrentWord := GetWordAndBounds(CaretXY, BB, BE);
      if (BB.X <= CaretX) and (CaretX <= BE.X) then
        if CurrentWord <> '' then FindTextComboBox.Text := CurrentWord;
        // otherwise use last text
    end
    else
    begin
      FindTextComboBox.Text := SelText;
      SelectedRadioButton.Checked := True;
    end;

    Result:= ShowModal;
    if Result <> mrCancel then
    begin
      FLastSearchOptions := [];
      if CaseCheckBox.Checked then Include(FLastSearchOptions, soMatchCase);
      if WholeWordCheckBox.Checked then Include(FLastSearchOptions, soWholeWord);
      if BackwardRadioButton.Checked then Include(FLastSearchOptions, soBackwards);
      if HomeRadioButton.Checked then Include(FLastSearchOptions, soEntireScope);
      if SelectedRadioButton.Checked then Include(FLastSearchOptions, soSelectedOnly);
      if NonSpacingCheckBox.Checked then Include(FLastSearchOptions, soIgnoreNonSpacing);
      if CompressCheckBox.Checked then Include(FLastSearchOptions, soSpaceCompress);
      if RECheckBox.Checked then Include(FLastSearchOptions, soRegularExpression);
      if PromptCheckBox.Checked then Include(FLastSearchOptions, soPrompt);
      if ReplaceRadioButton.Checked then
      begin
        Include(FLastSearchOptions, soReplace);
        if Result = mrYesToAll then Include(FlastSearchOptions, soReplaceAll);
      end;

      if SyntaxEdit1.SearchReplace(FindTextComboBox.Text, NewTextComboBox.Text, FLastSearchOptions) = 0 then
        MessageDlg('No further occurence of ''' + FindTextComboBox.Text + ''' found.',
                   mtInformation, [mbOK], 0);
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.SearchNextActionExecute(Sender: TObject);

begin
  with SyntaxFindDialog do
  begin
    if FindTextComboBox.Text = '' then SearchFindAction.Execute
                                  else
    begin
        // use the same options as before but start at current cursor position
      if SyntaxEdit1.SearchReplace(FindTextComboBox.Text, NewTextComboBox.Text,
                                   FLastSearchOptions - [soEntireScope, soSelectedOnly]) = 0 then
        MessageDlg('No further occurence of ''' + FindTextComboBox.Text + ''' found.',
                   mtInformation, [mbOK], 0);
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.FontActionExecute(Sender: TObject);

begin
  FontDialog.Font := SyntaxEdit1.Font;
  if FontDialog.Execute then
  begin
    SyntaxEdit1.Font := FontDialog.Font;
    SyntaxEdit1.CharWidth := UpDown1.Position;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.ConfirmConversion(Sender: TWideStrings; var Allowed: Boolean);

var
  Result: Word;

begin
  Result := MessageDlg('You are about to save a file as ANSI text which contains Unicode.'#13 +
              'Do you want to save the text as Unicode, cancel the action or ignore potential losses'#13 +
              'and save as ANSI text?',
              mtConfirmation, [mbYes, mbCancel, mbIgnore], 0);
  case Result of
    mrYes:
      begin
        Sender.SaveUnicode := True;
        Allowed := True;
      end;
    mrIgnore:
      begin
        Allowed := True;
      end;
  else
    Allowed := False;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.OptionMenuClick(Sender: TObject);

begin
  with Sender as TMenuItem do
  begin
    Checked := not Checked;
  end;
  with SyntaxEdit1 do
  begin
    if UndoItem.Checked then Options := Options + [eoGroupUndo]
                        else Options := Options - [eoGroupUndo];
    if AutoIndentItem.Checked then Options := Options + [eoAutoIndent]
                              else Options := Options - [eoAutoIndent];
    if InsertItem.Checked then Options := Options + [eoInserting]
                          else Options := Options - [eoInserting];
    if TabItem.Checked then Options := Options + [eoUseTabs]
                       else Options := Options - [eoUseTabs];
    if SmartItem.Checked then Options := Options + [eoSmartTabs]
                         else Options := Options - [eoSmartTabs];
    if OptimalItem.Checked then Options := Options + [eoOptimalFill]
                           else Options := Options - [eoOptimalFill];
    if UnindentItem.Checked then Options := Options + [eoAutoUnindent]
                            else Options := Options - [eoAutoUnindent];
    if CursorTabItem.Checked then Options := Options + [eoCursorThroughTabs]
                             else Options := Options - [eoCursorThroughTabs];
    if CursorEOLItem.Checked then Options := Options + [eoScrollPastEOL]
                             else Options := Options - [eoScrollPastEOL];
    if BlanksItem.Checked then Options := Options + [eoKeepTrailingBlanks]
                          else Options := Options - [eoKeepTrailingBlanks];
    if HighlightItem.Checked then Options := Options + [eoUseSyntaxHighlighting]
                             else Options := Options - [eoUseSyntaxHighlighting];
    if UndoItem.Checked then Options := Options + [eoUndoAfterSave]
                        else Options := Options - [eoUndoAfterSave];
    if ControlItem.Checked then Options := Options + [eoShowControlChars]
                           else Options := Options - [eoShowControlChars];
    if ReadOnlyItem.Checked then Options := Options + [eoShowCursorWhileReadOnly]
                            else Options := Options - [eoShowCursorWhileReadOnly];
    if NumberItem.Checked then Options := Options + [eoLineNumbers]
                          else Options := Options - [eoLineNumbers];
    if HintItem.Checked then Options := Options + [eoShowScrollHint]
                        else Options := Options - [eoShowScrollHint];
    if TripleItem.Checked then Options := Options + [eoTripleClicks]
                          else Options := Options - [eoTripleClicks];
    if HideSelectionItem.Checked then Options := Options + [eoHideSelection]
                                 else Options := Options - [eoHideSelection];
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.FormCloseQuery(Sender: TObject; var CanClose: Boolean);

var
  Result: Word;

begin
  CanClose := not SyntaxEdit1.Modified;
  if not CanClose then
  begin
    Result := MessageDlg(Format('Save changes to %s?', [SyntaxEdit1.Info]),
                         mtConfirmation, [mbYes, mbNo, mbCancel], 0);
    if Result = mrYes then FileSaveAction.Execute;
    CanClose := Result <> mrCancel;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.SyntaxEdit1ReplaceText(Sender: TObject; const Search, Replace: WideString; Line, Start,
  Stop: Integer; var ReplaceAction: TReplaceAction);

begin
  with Sender as TUniCodeEdit do
  begin
    case MessageDlg('Replace this occurence of ''' + Search + '''?', mtConfirmation,
                    [mbYes, mbNo, mbCancel, mbAll], 0) of
      mrCancel:
        ReplaceAction := raCancel;
      mrYes:
        ReplaceAction := raReplace;
      mrYesToAll:
        ReplaceAction := raReplaceAll;
    else
      // no
      ReplaceAction := raSkip;
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.Edit2Change(Sender: TObject);

begin
  SyntaxEdit1.CharWidth := UpDown1.Position;
end;

//----------------------------------------------------------------------------------------------------------------------

end.
