/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

#ifdef ENABLE_LUA_MODULES

#include "myx_grt_private.h"

#include "myx_grt_lua_private.h"


static MYX_GRT_ERROR lua_call_function(MYX_GRT_FUNCTION *function, MYX_GRT_VALUE *value, MYX_GRT_VALUE **retval);
static MYX_GRT_ERROR lua_init_module(MYX_GRT_MODULE_LOADER *loader, const char *file, MYX_GRT_MODULE **retmodule);

static int l_log_error(lua_State *lua);
static int l_log_warning(lua_State *lua);
static int l_log_message(lua_State *lua);


static const luaL_reg lualibs[] = {
        { "base",       luaopen_base },
        { "table",      luaopen_table },
        { "io",         luaopen_io },
        { "string",     luaopen_string },
        { "math",       luaopen_math },
        { "debug",      luaopen_debug },
        { "loadlib",    luaopen_loadlib },
        { NULL,         NULL }
};


static MYX_GRT *get_grt(lua_State *L)
{
  MYX_GRT **ctx;
  lua_getglobal(L, "__GRT");

  ctx= (MYX_GRT**)luaL_checkudata(L, -1, "MYX_GRT");
  if (ctx)
  {
    lua_pop(L, 1);
    return *ctx;
  }
  return NULL;
}


MYX_GRT_MODULE_LOADER *myx_lua_init_loader(MYX_GRT *grt, MYX_GRT_ERROR *error)
{
  MYX_GRT_MODULE_LOADER *loader= g_new0(MYX_GRT_MODULE_LOADER, 1);
  MYX_LUA_LOADER *priv= g_new0(MYX_LUA_LOADER, 1);
  static char *file_extensions[]= {
    ".lua"
  };

  *error= MYX_GRT_NO_ERROR;

  loader->grt= grt;
  loader->loader_type= MYX_LUA_MODULE_TYPE;
  loader->priv= priv;
  loader->init_module= lua_init_module;
  loader->call_function= lua_call_function;
  loader->extensions_num= 1;
  loader->extensions= file_extensions;

  priv->lua= lua_open();
  
  // register a global __GRT variable 
  {
    MYX_GRT **userdata;
    lua_pushstring(priv->lua, "__GRT");
    userdata= (MYX_GRT**)lua_newuserdata(priv->lua, sizeof(MYX_GRT*));
    *userdata= grt;
    luaL_newmetatable(priv->lua, "MYX_GRT");
    lua_setmetatable(priv->lua, -2);
    lua_settable(priv->lua, LUA_GLOBALSINDEX);
  }
  // register some libs
  {
    const luaL_reg *lib;

    for (lib = lualibs; lib->func != NULL; lib++) 
    {
      lib->func(priv->lua);
      lua_settop(priv->lua, 0);
    }
  }

  // register logging functions
  lua_register(priv->lua, "logerror", l_log_error);
  lua_register(priv->lua, "logwarning", l_log_warning);
  lua_register(priv->lua, "logmessage", l_log_message);

  g_assert(lua_gettop(priv->lua) == 0);
  
  return loader;
}


static void lua_push_fallback_table(lua_State *l)
{
  lua_newtable(l);
  lua_pushstring(l, "__index");
  lua_getglobal(l, "_G");
  lua_settable(l, -3);
}


static MYX_GRT_ERROR lua_init_module(MYX_GRT_MODULE_LOADER *loader, const char *file, MYX_GRT_MODULE **retmodule)
{
  MYX_GRT_MODULE *module;
  lua_State *l= loader->priv->lua;
  MYX_GRT_VALUE *module_info;
  const char *module_name, *extends;
  MYX_GRT_LIST *module_functions;
  int status;
  unsigned int i;

  // create a new table which will be the environment for the
  // loaded module
  lua_pushstring(l, "___tmp");
  lua_newtable(l);
  lua_settable(l, LUA_GLOBALSINDEX);

  // set the global environment as a fallback for the module environment
  lua_getglobal(l, "___tmp");
  lua_push_fallback_table(l);
  lua_setmetatable(l, -2);
  lua_pop(l, 1); // pop __tmp

  // load the module
  status= luaL_loadfile(l, file);
  if (status != 0)
  {
    if (getenv("GRT_VERBOSE"))
    {
      g_warning("Could not load lua module %s: %s", file, lua_tostring(l, -1));
      lua_pop(l, 1);
    }
    return MYX_GRT_BAD_MODULE;
  }

  // fetch the new environment table
  lua_getglobal(l, "___tmp");

  // sets it as the environment for the loaded module
  lua_setfenv(l, -2);

  // execute the module, so that function declarations in it get executed
  status= lua_pcall(l, 0, 0, 0);

  if (status != 0)
  {
    g_warning("error executing lua module %s: %s\n", file, lua_tostring(l, -1));
    lua_pop(l, 1);
    goto error;
  }

  // get module info
  lua_getglobal(l, "___tmp");
  lua_pushstring(l, "getModuleInfo");
  lua_gettable(l, -2);
  status= lua_pcall(l, 0, 1, 0);

  if (status != 0)
  {
    g_warning("error calling getModuleInfo() in lua module %s: %s\n", file,
              lua_tostring(l, -1));
    lua_pop(l, 1);
    goto error;
  }

  module_info= myx_lua_pop_grt_value(l);
  if (!module_info || !myx_grt_value_get_type(module_info)==MYX_DICT_VALUE)
  {
    g_warning("invalid return value calling getModuleInfo() in lua module %s\n", file);
    if (module_info)
      myx_grt_value_release(module_info);
    goto error;
  }
  
  lua_pop(l, 1); // pop __tmp

  // get module data
  module_name= myx_grt_value_as_string(myx_grt_dict_get_item_value(module_info, "name"));

  extends= myx_grt_value_as_string(myx_grt_dict_get_item_value(module_info, "extends"));

  module_functions= myx_grt_value_as_list(myx_grt_dict_get_item_value(module_info, "functions"));
  
  if (!module_name || !module_functions)
  {
    g_warning("bad info returned from getModuleInfo() in lua module %s: %s\n", file,
              lua_tostring(l, -1));
    if (module_info)
      myx_grt_value_release(module_info);
    goto error;
  }

  // rename the ___tmp module table to the definitive name
  lua_pushstring(l, module_name);
  lua_getglobal(l, "___tmp");
  lua_settable(l, LUA_GLOBALSINDEX);

  lua_pushstring(l, "___tmp");
  lua_pushnil(l);
  lua_settable(l, LUA_GLOBALSINDEX);

  
  // init internal module descriptor
  module= g_new0(MYX_GRT_MODULE, 1);

  module->loader= loader;
  module->priv= NULL;
  module->name= g_strdup(module_name);
  module->path= g_strdup(file);
  module->functions_num= module_functions->items_num;
  module->functions= g_new0(MYX_GRT_FUNCTION, module->functions_num);
  for (i= 0; i < module_functions->items_num; i++)
  {
    MYX_GRT_FUNCTION *func= module->functions+i;
    
    func->module= module;
    myx_grt_parse_function_spec(module_functions->items[i]->value.s, func);
    func->priv= NULL;
  }
  module->extends= g_strdup(extends);
  
  myx_grt_value_release(module_info);

  *retmodule= module;
  
  if (getenv("GRT_VERBOSE"))
    g_message("Initialized module %s", file);
  
  return MYX_GRT_NO_ERROR;
  
error:

  return MYX_GRT_MODULE_INIT_ERROR;
}


static MYX_GRT_ERROR lua_call_function(MYX_GRT_FUNCTION *function, MYX_GRT_VALUE *value, MYX_GRT_VALUE **retval)
{
  MYX_GRT_ERROR error= MYX_GRT_NO_ERROR;
  lua_State *lua= function->module->loader->priv->lua;
  
  if (getenv("GRT_VERBOSE"))
    g_message("Calling lua function %s.%s", function->module->name, function->name);

  if (value)
    myx_lua_push_grt_value(lua, value);
  else
    lua_pushnil(lua);

  lua_getglobal(lua, function->module->name);
  lua_pushstring(lua, function->name);
  lua_gettable(lua, -2);
  
  if (lua_pcall(lua, 1, 1, 0) != 0)
  {
    g_warning("Error calling lua function %s.%s", function->module->name, function->name);
    *retval= NULL;
    
    error= MYX_GRT_FUNCTION_CALL_ERROR;
  }
  else
  {
    *retval= myx_lua_pop_grt_value(lua);
  }

  return error;
}





static int l_log_error(lua_State *lua)
{
  MYX_GRT *grt= get_grt(lua);
  const char *message;
  const char *detail;
  
  detail= luaL_checkstring(lua, -1);
  lua_pop(lua, 1);
  message= luaL_checkstring(lua, -1);
  lua_pop(lua, 1);

  grt->logfunc(grt, 2, message, detail);

  return 0;
}


static int l_log_warning(lua_State *lua)
{
  MYX_GRT *grt= get_grt(lua);
  const char *message;
  const char *detail;
  
  detail= luaL_checkstring(lua, -1);
  lua_pop(lua, 1);
  message= luaL_checkstring(lua, -1);
  lua_pop(lua, 1);

  grt->logfunc(grt, 1, message, detail);

  return 0;
}


static int l_log_message(lua_State *lua)
{
  MYX_GRT *grt= get_grt(lua);
  const char *message;
  const char *detail;
  
  detail= luaL_checkstring(lua, -1);
  lua_pop(lua, 1);
  message= luaL_checkstring(lua, -1);
  lua_pop(lua, 1);

  grt->logfunc(grt, 0, message, detail);

  return 0;
}



#endif
