/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/**
 * @file myx_grt.c 
 * @brief GRT environment functions
 * 
 * See also: <a href="../grt.html#GRT">GRT</a>
 */

#include <glib.h>

#include <libxml/xmlmemory.h>
#include <libxml/parser.h>

#include "myx_xml_aux_functions.h"
#include "myx_grt_private.h"


static MYX_GRT_VALUE *unserialize_from_xml(xmlNodePtr node, GHashTable *objects_by_id);
static xmlNodePtr serialize_to_xml(xmlNodePtr parent, MYX_GRT_VALUE *value, GHashTable *saved_ids);


static void default_print(const char *msg, void *data)
{
  printf("%s", msg);
}


static void default_log(MYX_GRT *grt, int code, const char *msg, const char *detail)
{
  printf("%i: %s: %s\n", code, msg, detail?detail:"");
}


/**
 **************************************************************************** 
 * @brief Creates a new GRT environment.
 *
 *  Also:
 *  - sets a default print callback that's equivalent to printf() and
 * registers the built-in module loader type.
 *  - creates a server socket so that modules can connect to it for
 * communicating with the grt (passing progress info, for example)
 *   After creating the environment, you will want to call individual
 * initializers for each module loader you want to support.
 *
 * @return A newly created GRT.
 ****************************************************************************/
MYX_GRT *myx_grt_initialize()
{
  MYX_GRT *grt= g_new0(MYX_GRT, 1);

  grt->print= default_print;
  grt->print_data= NULL;
  
  grt->logfunc= default_log;
  
  grt->root= myx_grt_dict_new(NULL);

  myx_grt_register_module_loader(grt, myx_builtin_init_loader(grt));

#ifdef remove_this_when_this_gets_used
  myx_grt_setup_messaging(grt);
#endif
  return grt;
}


/**
 ****************************************************************************
 * @brief Shutdown a GRT environment and frees all used resources
 *
 *   Will close and free all module loaders, modules and other resources
 *   allocated for it.
 * 
 * @param grt the GRT environment
 *
 * NOTE
 *   This is not freeing anything atm.
 ****************************************************************************/
void myx_grt_finalize(MYX_GRT *grt)
{
  //XXX TODO
}


/**
 ****************************************************************************
 * @brief Sets the function to be used to print values
 * 
 *  Sets the function that is used when GRT needs to print messages.
 *
 * @param grt      the GRT environment
 * @param user_data a pointer to be passed to the callback function. 
 * @param process_output_func the callback function which should take 
 *        the text and the userdata pointer as arguments
 ****************************************************************************/
void myx_grt_set_output_callback(MYX_GRT *grt, void *user_data, MYX_GRT_PRINT_CALLBACK process_output_func)
{
  grt->print= process_output_func;
  grt->print_data= user_data;
}


/**
 ****************************************************************************
 * @brief Sets callbacks for module logging
 * 
 *  This will set the callbacks that will be called when a module calls
 * a logging function, for errors or messages in general.
 *
 * @param grt      the GRT environment
 * @param log_func the function that will handle logging calls from modules
 ****************************************************************************/
void myx_grt_module_set_log_callback(MYX_GRT *grt, MYX_GRT_LOG_CALLBACK log_func)
{
  grt->logfunc= log_func;
}


/**
 ****************************************************************************
 * @brief Registers a module loader with the GRT
 * 
 * Will register an already initialized module loader with the GRT.
 * You can only register only one module loader of each type.
 * After registration, you can scan for modules with myx_grt_scan_for_modules
 * or load them individually with myx_grt_module_init
 * 
 * @param grt    the GRT environment where the loader should be registered
 * @param loader an initialized loader object. 
 *
 * @return MYX_GRT_NO_ERROR, MYX_GRT_INTERNAL_ERROR
 ****************************************************************************/
MYX_GRT_ERROR myx_grt_register_module_loader(MYX_GRT *grt, MYX_GRT_MODULE_LOADER *loader)
{  
  g_return_val_if_fail(grt != NULL, MYX_GRT_INTERNAL_ERROR);
  g_return_val_if_fail(loader != NULL, MYX_GRT_INTERNAL_ERROR);

  grt->loaders_num++;
  grt->loaders= g_realloc(grt->loaders, sizeof(MYX_GRT_MODULE_LOADER*)*grt->loaders_num);

  grt->loaders[grt->loaders_num-1]= loader;

  return MYX_GRT_NO_ERROR;
}

/**
 ****************************************************************************
 * @brief Returns a copied list of the structs registered in the GRT
 * 
 * @param grt the GRT
 *
 * @return A copy of the registered structs
 ****************************************************************************/
MYX_GRT_STRUCTS * myx_grt_structs_get(MYX_GRT *grt)
{
  MYX_GRT_STRUCTS *structs= g_malloc(sizeof(MYX_GRT_STRUCTS));

  g_return_val_if_fail(grt != NULL, NULL);

  structs->structs_num= grt->structs_num;
  structs->structs= g_memdup(grt->structs, sizeof(MYX_GRT_STRUCT)*structs->structs_num);

  return structs;
}

/**
 ****************************************************************************
 * @brief Returns the number of structs registered in the GRT
 * 
 * @param grt the GRT
 *
 * @return The number of structs
 ****************************************************************************/
int myx_grt_struct_get_count(MYX_GRT *grt)
{
  g_return_val_if_fail(grt != NULL, -1);

  return grt->structs_num;
}

/**
 ****************************************************************************
 * @brief Returns the number of structs registered in the GRT that have
 * the struct with the struct_name as their parent
 * 
 * @param grt the GRT
 * @param struct_name The name of the parent struct
 *
 * @return The number of structs
 ****************************************************************************/
int myx_grt_struct_get_child_count(MYX_GRT *grt, const char *struct_name)
{
  unsigned int i, c= 0;

  g_return_val_if_fail(grt != NULL, -1);

  for (i= 0; i<grt->structs_num; i++)
  {
    MYX_GRT_STRUCT *str= grt->structs + i;

    if (strcmp3(str->parent_struct_name, struct_name) == 0)
    {
      c++;
    }
  }

  return c;
}

/**
 ****************************************************************************
 * @brief Returns the GRT struct given by an index
 * 
 * @param grt the GRT
 * @param index the index of the requested struct
 * 
 * @return The struct with the given index
 ****************************************************************************/
MYX_GRT_STRUCT * myx_grt_struct_get_by_index(MYX_GRT *grt, unsigned int index)
{
  g_return_val_if_fail(grt != NULL, NULL);
  g_return_val_if_fail(index < grt->structs_num, NULL);

  return grt->structs + index;
}

/**
 ****************************************************************************
 * @brief Returns the child GRT struct of the struct defined by struct_name 
 * with the given by an
 * 
 * @param grt the GRT
 * @param struct_name the name of the parent strcut
 * @param index the index of the requested struct
 * 
 * @return The struct with the given index or null if not found
 ****************************************************************************/
MYX_GRT_STRUCT * myx_grt_struct_get_child_by_index(MYX_GRT *grt, const char *struct_name, unsigned int index)
{
  unsigned int i, c= 0;

  g_return_val_if_fail(grt != NULL, NULL);

  for (i= 0; i<grt->structs_num; i++)
  {
    MYX_GRT_STRUCT *str= grt->structs + i;

    if (strcmp3(str->parent_struct_name, struct_name) == 0)
    {
      if (c == index)
        return str;

      c++;
    }
  }

  return NULL;
}


/**
 ****************************************************************************
 * @brief Fetches the root object of the GRT
 * 
 * @param grt the GRT
 *
 * @return The root GRT value.
 ****************************************************************************/
MYX_GRT_VALUE *myx_grt_get_root(MYX_GRT *grt)
{
  g_return_val_if_fail(grt != NULL, NULL);

  return grt->root;
}


/**
 ****************************************************************************
 * @brief "Loads" a value into the GRT and makes it the root object/value
 * 
 *   This will replace the root object of the GRT with the passed one.
 * The old object will be completely freed. 
 *
 * The GRT will take over ownership of the whole object passed. That means
 * you should not free it.
 *
 * @param grt the GRT environment
 * @param new_root a GRT object/value that will become the root object. It has to
 *    be a MYX_DICT_VALUE
 *
 * @return MYX_GRT_NO_ERROR if success
 ****************************************************************************/
MYX_GRT_ERROR myx_grt_set_root(MYX_GRT *grt, MYX_GRT_VALUE *new_root)
{
  g_return_val_if_fail(grt != NULL, MYX_GRT_INTERNAL_ERROR);
  g_return_val_if_fail(new_root != NULL, MYX_GRT_INTERNAL_ERROR);

  if (myx_grt_value_get_type(new_root) != MYX_DICT_VALUE)
    return MYX_GRT_BAD_VALUE;

  myx_grt_value_release(grt->root);

  grt->root= new_root;

  myx_grt_value_retain(new_root);
  
  return MYX_GRT_NO_ERROR;
}


/**
 ****************************************************************************
 * @brief Loads a previously stored GRT value from a file
 * 
 *   Loads a GRT value from a GRT XML file. Use myx_grt_store_to_file to
 * create files in that format.
 *   Read myx_grt_store_to_file for more details.
 *
 * @param filename the name of the file to load. The file must be in the GRT XML format.
 *     Usually something stored by myx_grt_store_to_file
 *
 * @return NULL on error and the value if the file could be correctly loaded.
 ****************************************************************************/
MYX_GRT_VALUE *myx_grt_retrieve_from_file(const char *filename)
{
  xmlDocPtr doc;
  xmlNodePtr root;
  MYX_GRT_VALUE *value;

  if (!(doc= myx_xmlParseFile(filename)))
    return NULL;

  root= xmlDocGetRootElement(doc);
  if (root)
  {
    root= root->children;
    while (root && xmlStrcmp(root->name, "value")!=0) root= root->next;
    if (root)
    {
      GHashTable *objects_by_id= g_hash_table_new(g_str_hash, g_str_equal);
      value= unserialize_from_xml(root, objects_by_id);
      g_hash_table_destroy(objects_by_id);
    }
  }

  xmlFreeDoc(doc);

  return value;
}


/**
 ****************************************************************************
 * @brief Stores a GRT value to a file
 * 
 *   This will serialize the value to XML and store it in a file that can 
 * later be retrieved with myx_grt_retrieve_from_file
 *
 * @param  value the GRT value to store
 * @param  filename name of file to store data
 *
 * @return  MYX_GRT_NO_ERROR if there were no errors.
 ****************************************************************************/
MYX_GRT_ERROR myx_grt_store_to_file(MYX_GRT_VALUE *value, const char *filename)
{
  xmlDocPtr doc;
  int res;
  GHashTable *saved_ids;
  
  doc= xmlNewDoc("1.0");
  doc->children= xmlNewDocRawNode(doc, NULL, "data", NULL);
  
  saved_ids= g_hash_table_new(g_str_hash, g_str_equal);
  
  serialize_to_xml(doc->children, value, saved_ids);

  g_hash_table_destroy(saved_ids);
  
  res= myx_xmlSaveFile(filename, doc);

  xmlFreeDoc(doc);

  return res == -1 ? MYX_GRT_CANT_OPEN_FILE : MYX_GRT_NO_ERROR;
}


static char *value_type_strings[]=
{
  "int",
    "bigint",
    "real",
    "string",
    "list",
    "dict"
};


/**
 ****************************************************************************
 * @brief Prints a string using the print callback from GRT
 *
 *  Prints a formated message using the callback function set up in GRT.
 *
 * @param grt the GRT environment. A print callback must have been previously set.
 * @param fmt format string, accepts anything that printf() does.
 * @param ... arguments for the formatted message
 *****************************************************************************/
int myx_grt_printf(MYX_GRT *grt, const char *fmt, ...)
{
  char *tmp;

  va_list args;
  va_start(args, fmt);
  
  tmp= g_strdup_vprintf(fmt, args);
  va_end(args);
  
  MYX_PRINT(grt, tmp);
  g_free(tmp);
  return 0;
}



static xmlNodePtr serialize_to_xml(xmlNodePtr parent, MYX_GRT_VALUE *value,
                                   GHashTable *saved_ids)
{
  unsigned int i;
  char buffer[100];
  const char *id;
  xmlNodePtr node;

  switch (value->type)
  {
  case MYX_INT_VALUE:
    g_snprintf(buffer, sizeof(buffer), "%i", value->value.i);
    node= xmlNewTextChild(parent, NULL, "value", buffer);

    xmlNewProp(node, "type", "int");
    break;
  case MYX_REAL_VALUE:
    g_snprintf(buffer, sizeof(buffer), "%f", value->value.r);
    node= xmlNewTextChild(parent, NULL, "value", buffer);

    xmlNewProp(node, "type", "real");
    break;
  case MYX_STRING_VALUE:
    node= xmlNewTextChild(parent, NULL, "value", value->value.s);

    xmlNewProp(node, "type", "string");
    break;
  case MYX_LIST_VALUE:
    node= xmlNewTextChild(parent, NULL, "value", NULL);
    xmlNewProp(node, "type", "list");
    xmlNewProp(node, "content-type", value_type_strings[value->value.l->content_type]);

    if (value->value.l->content_struct_name)
      xmlNewProp(node, "content-struct-name", value->value.l->content_struct_name);

    for (i= 0; i < value->value.l->items_num; i++)
    {
      g_return_val_if_fail(value->value.l->items[i]->type == value->value.l->content_type, NULL);
      serialize_to_xml(node, value->value.l->items[i], saved_ids);
    }
    break;
  case MYX_DICT_VALUE:
    node= xmlNewTextChild(parent, NULL, "value", NULL);
    xmlNewProp(node, "type", "dict");

    if ((value->value.d->struct_name) && (value->value.d->struct_name[0]))
      xmlNewProp(node, "struct-name", value->value.d->struct_name);

    if ((id= myx_grt_dict_id_item_as_string(value)))
    {
      if (g_hash_table_lookup(saved_ids, id))
      {
        // the object is already in the XML tree, just leave a link to it
        xmlNewProp(node, "link", id);

        return node;
      }
      else
        // object is not yet in XML tree. put it in and record the ID
        g_hash_table_insert(saved_ids, (gpointer)id, "X");      
    }

    for (i= 0; i < value->value.d->items_num; i++)
    {
      xmlNodePtr child;
      
      child= serialize_to_xml(node, value->value.d->items[i].value, saved_ids);
      xmlNewProp(child, "key", value->value.d->items[i].key);
    }
    break;
  }
  return node;
}


static MYX_GRT_VALUE *unserialize_from_xml(xmlNodePtr node, GHashTable *objects_by_id)
{
  int i;
  xmlChar *str;
  xmlChar *node_type= xmlGetProp(node, "type");
  MYX_GRT_VALUE *value= NULL;
  
  if (!node_type)
  {
    g_warning("Node '%s' in xml doesn't have a type property", node->name);
    return NULL;
  }

  if (strcmp(node_type, "int")==0)
  {
    str= xmlNodeGetContent(node);
    value= myx_grt_value_from_int(strtol(str, NULL, 0));
    xmlFree(str);
  }
  else if (strcmp(node_type, "real")==0)
  {
    str= xmlNodeGetContent(node);
    value= myx_grt_value_from_double(strtod(str, NULL));
    xmlFree(str);
  }
  else if (strcmp(node_type, "string")==0)
  {
    str= xmlNodeGetContent(node);
    value= myx_grt_value_from_string(str);
    xmlFree(str);
  }
  else if (strcmp(node_type, "dict")==0)
  {
    xmlChar *node_type= xmlGetProp(node, "struct-name");
    xmlChar *link_id= xmlGetProp(node, "link");
    xmlNodePtr child;

    if (link_id)
    {
      MYX_GRT_VALUE *linked_object;

      linked_object= g_hash_table_lookup(objects_by_id, link_id);
      if (!linked_object)
      {
        g_warning("linked object %s was not yet parsed at the time it was found",
                  link_id);
      }
      else
      {
        myx_grt_value_retain(linked_object);
      }

      value= linked_object;
      xmlFree(link_id);
    }
    else
    {
      value= myx_grt_dict_new(NULL);

      if (node_type && node_type[0])
        value->value.d->struct_name= g_strdup(node_type);

      child= node->children;
      while (child)
      {
        MYX_GRT_VALUE *sub_value;
        
        if (strcmp(child->name, "value")==0)
        {
          xmlChar *key= xmlGetProp(child, "key");

          sub_value= unserialize_from_xml(child, objects_by_id);
          if (sub_value)
          {
            myx_grt_dict_set_item(value, key, sub_value);
            myx_grt_value_release(sub_value);
          }
          else
          {
            myx_grt_value_release(value);
            value= NULL;
            break;
          }

          if (xmlStrcmp(key, "_id")==0 && sub_value)
          {
            if (sub_value->type == MYX_STRING_VALUE)
              g_hash_table_insert(objects_by_id, (gpointer)myx_grt_value_as_string(sub_value),
                                  value);
          }

          xmlFree(key);
        }

        child= child->next;
      }
    }
  }
  else if (strcmp(node_type, "list")==0)
  {
    xmlChar *ctype= xmlGetProp(node, "content-type");
    xmlChar *cstruct_name= xmlGetProp(node, "content-struct-name");
    xmlNodePtr child;

    for (i= 0; sizeof(value_type_strings)/sizeof(char*); i++)
      if (strcmp2(value_type_strings[i], ctype)==0)
      {
        value= myx_grt_list_new(i, NULL);
        break;
      }

    value->value.l->content_struct_name= g_strdup(cstruct_name);

    child= node->children;
    while (child)
    {
      MYX_GRT_VALUE *sub_value;

      if (child->type == XML_ELEMENT_NODE && strcmp(child->name, "value")==0)
      {
        sub_value= unserialize_from_xml(child, objects_by_id);
        if (sub_value)
        {
          myx_grt_list_add_item(value, sub_value);
          myx_grt_value_release(sub_value);
        }
        else
        {
          myx_grt_value_release(value);
          value= NULL;
          break;
        }
      }

      child= child->next;
    }

    xmlFree(ctype);
    xmlFree(cstruct_name);
  }
  xmlFree(node_type);

  return value;
}


/**
 ****************************************************************************
 * @brief Convert a GRT value into a XML string
 *
 * Produces a XML representation of the GRT value.
 * 
 * @param value a GRT value
 *
 * @return String containing the value as a XML or NULL if there's an error.
 * 
 * @see myx_grt_value_from_xml
 *****************************************************************************/
char *myx_grt_value_to_xml(MYX_GRT_VALUE *value)
{
  xmlDocPtr doc;
  xmlNodePtr root;
  xmlChar *buffer= NULL;
  GHashTable *saved_ids;
  int size;
  
  if (value)
  {
    doc= xmlNewDoc("1.0");
    doc->children= root= xmlNewDocRawNode(doc, NULL, "data", NULL);
    
    saved_ids= g_hash_table_new(g_str_hash, g_str_equal);
    
    serialize_to_xml(root, value, saved_ids);
    
    g_hash_table_destroy(saved_ids);

    xmlDocDumpFormatMemory(doc, &buffer, &size, 1);

    xmlFreeDoc(doc);

    return (char*)buffer;
  }
  else
    return NULL;
}


/**
 ****************************************************************************
 * @brief Parse a XML representation of a GRT value.
 *
 * Parses a XML string and rebuilds the GRT value that corresponds to it.
 *
 * @param str the string in XML format containing a serialized GRT value
 * @param size length of the string
 *
 * @return The value corresponding to the passed in string or NULL if there's
 *   an error.
 * 
 * @see myx_grt_value_to_xml
 *****************************************************************************/
MYX_GRT_VALUE *myx_grt_value_from_xml(const char *str, size_t size)
{
  xmlDocPtr doc= xmlParseMemory(str, (int)size);
  xmlNodePtr root;
  MYX_GRT_VALUE *value;

  if (!doc)
  {
    g_warning("Could not parse XML data");
    return NULL;
  }

  root= xmlDocGetRootElement(doc);

  //check if we have a <data> rootnode
  if (root && (xmlStrcmp(root->name, "data")==0))
    root= root->children;

  //skip all nodes that are no <value> nodes
  while (root && xmlStrcmp(root->name, "value")!=0) 
    root= root->next;

  if (root)
  {
    GHashTable *objects_by_id= g_hash_table_new(g_str_hash, g_str_equal);
    value= unserialize_from_xml(root, objects_by_id);
    g_hash_table_destroy(objects_by_id);
  }
  else
    value= NULL;

  xmlFreeDoc(doc);

  return value;
}

/** 
 ****************************************************************************
 * @brief Add a message to the GRT message queue.
 *
 * @param grt
 * @param msg_type  
 * @param message  message string
 * @param details  optional string list for details of the message
 * @param copy_details if 1, the function will make a copy of the details 
 *              parameter
 *
 *****************************************************************************/
void myx_grt_add_msg(MYX_GRT *grt, int msg_type, const char *message, MYX_STRINGLIST *details,
                     int copy_details)
{
  MYX_GRT_MSG *msg;
  
  if (!grt->msgs)
    grt->msgs= g_new0(MYX_GRT_MSGS, 1);

  grt->msgs->msgs_num++;
  grt->msgs->msgs= g_realloc(grt->msgs->msgs,
                             grt->msgs->msgs_num*sizeof(MYX_GRT_MSG));
  msg= grt->msgs->msgs+grt->msgs->msgs_num-1;

  msg->msg_type= msg_type;
  msg->msg= g_strdup(message);
  if (copy_details && details)
  {
    unsigned int i;
    msg->msg_detail= g_new0(MYX_STRINGLIST, 1);
    msg->msg_detail->strings= g_new0(char*, details->strings_num);
    for (i= 0; i < details->strings_num; i++)
      msg->msg_detail->strings[i]= g_strdup(details->strings[i]);
  }
  else
    msg->msg_detail= details;
}


/** 
 ****************************************************************************
 * @brief Fetches messages from the GRT.
 *
 * @param grt  
 * @param count  number of messages to fetch. 0 will return all messages.
 *
 * @return Fetched messages or NULL, if there's none. The returned struct
 * must be freed with myx_grt_free_msgs()
 *****************************************************************************/
MYX_GRT_MSGS* myx_grt_get_msgs(MYX_GRT *grt, unsigned int count)
{
  //unsigned int i;
  MYX_GRT_MSGS *msgs;
  
  // fetch messages from modules
  myx_grt_fetch_module_messages(grt);
  
  // return what was requested

  if (!grt->msgs)
    return NULL;
  
  if (count == 0)
  {
    msgs= grt->msgs;
    grt->msgs= NULL;
  }
  else
  {
    unsigned int i;

    if (count > grt->msgs->msgs_num)
      count= grt->msgs->msgs_num;

    msgs= g_new0(MYX_GRT_MSGS, 1);
    msgs->msgs= g_new0(MYX_GRT_MSG, count);
    for (i= 0; i < count; i++)
    {
      msgs->msgs[i]= grt->msgs->msgs[i];
    }
    memmove(grt->msgs->msgs, grt->msgs->msgs+count,
            (grt->msgs->msgs_num-count)*sizeof(MYX_GRT_MSG));
    grt->msgs->msgs_num-= count;
  }

  return msgs;
}


/** 
 ****************************************************************************
 * @brief 
 *
 * @param msgs  messages to be freed
 *****************************************************************************/
void myx_grt_free_msgs(MYX_GRT_MSGS *msgs)
{
  unsigned int i;
  
  for (i= 0; i < msgs->msgs_num; i++)
  {
    g_free(msgs->msgs[i].msg);
    if (msgs->msgs[i].msg_detail)
      myx_free_stringlist(msgs->msgs[i].msg_detail);
  }
  g_free(msgs->msgs);
  g_free(msgs);
}

