/* Copyright (C) 2003,2004 MySQL AB

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */


#include <myx_library.h>
#include <glib.h>
#include <libxml/xmlmemory.h>
#include <libxml/parser.h>
#include <libxml/xpath.h>
#include <assert.h>
#include <myx_xml_aux_functions.h>
#include <string.h>
#include <stdlib.h>
#if defined(__WIN__) || defined(_WIN32) || defined(_WIN64)
#include <Wincrypt.h>
#endif

/** @defgroup Connection_management_private internal stuff
 *  @ingroup Connection_management */

/** @addtogroup Connection_management_private
 *  @{ */

/*static variables*/
static xmlXPathContextPtr context; /* used by all XPath operations*/

// forward declarations
static xmlChar *obscure(const xmlChar* str);
static xmlChar *retrieve_password(xmlChar *password,
                                  enum myx_password_storage_type storage_type);
static xmlChar* save_password(xmlChar *password,
                              enum myx_password_storage_type storage_type);
static 
  void parse_user_connection(xmlDocPtr doc, xmlNodePtr user_conn,
                             MYX_USER_CONNECTION *out,
                             enum myx_password_storage_type pwd_storage_type);

/*
 * Public functions
 */

///////////////////////////////////////////////////////////////////////////////
/** @brief get int value stored at xpath of the document
    @param doc document where int is stored
    @param xpath path to node with value (it MUST have single child)
    @return found value

    calls atoi_and_free()
*//////////////////////////////////////////////////////////////////////////////
int get_xpath_int_value(xmlDocPtr doc, const char *xpath)
{
  xmlXPathObjectPtr result= xmlXPathEval(xpath, context);
  xmlNodeSetPtr node_set= result->nodesetval;
  int res= atoi_and_free(xmlNodeListGetString(doc,
                                              node_set->nodeTab[0]->children,
                                              1));
  assert(node_set->nodeNr == 1);
  xmlXPathFreeObject(result);
  return res;
}

///////////////////////////////////////////////////////////////////////////////
/** @brief create a new connections struct and initialize it's fields
    @return created connections struct
*//////////////////////////////////////////////////////////////////////////////
MYX_USER_CONNECTIONS * new_user_connections()
{
  MYX_USER_CONNECTIONS * user_conns= 
    (MYX_USER_CONNECTIONS*)g_malloc(sizeof(MYX_USER_CONNECTIONS));
  user_conns->last_connection= -1;
  user_conns->user_connections= NULL;
  user_conns->user_connections_num= 0;
  return user_conns;
}

///////////////////////////////////////////////////////////////////////////////
/** @brief load predefined connection stored in xml-file
    @ingroup Connection_management

    @param filename path to xml-file with stored connections
    @param error_code returned error code, <BR>
           possible values are:
             - \link MYX_LIB_ERROR::MYX_NO_ERROR MYX_NO_ERROR \endlink
             - \link MYX_LIB_ERROR::MYX_XML_PARSE_ERROR 
                                                  MYX_XML_PARSE_ERROR \endlink
             - \link MYX_LIB_ERROR::MYX_XML_EMPTY_DOCUMENT 
                                               MYX_XML_EMPTY_DOCUMENT \endlink
             - \link MYX_LIB_ERROR::MYX_XML_NO_VALID_DOCUMENT
                  MYX_XML_NO_VALID_DOCUMENT \endlink 
                  if the root of the document wasn't equal to
                  "user_connections"
    @return loaded MYX_USER_CONNECTIONS

    uses DOM model for parsing <BR>
    calls 
      get_xpath_int_value() and
      parse_user_connection()
*//////////////////////////////////////////////////////////////////////////////
MYX_USER_CONNECTIONS* myx_load_user_connections(const char *filename,
                                                MYX_LIB_ERROR *error_code)
{
  MYX_USER_CONNECTIONS *user_conns;
  xmlDocPtr doc;
  xmlNodePtr root;
  int count, i;
  xmlNodeSetPtr node_set;
  xmlXPathObjectPtr result;
  enum myx_password_storage_type pwd_storage_type;

  *error_code= MYX_NO_ERROR;

  if (! file_exists(filename) )
    return new_user_connections();

  if ( !(doc= myx_xmlParseFile(filename)) )
  {
    *error_code= MYX_XML_PARSE_ERROR;
    return NULL;
  }

  if ( !(root= xmlDocGetRootElement(doc)) )
  {
    *error_code= MYX_XML_EMPTY_DOCUMENT;
    xmlFreeDoc(doc);
    return NULL;
  }
    
  if (xmlStrcmp(root->name, (const xmlChar *) "user_connections"))
  {
    *error_code= MYX_XML_NO_VALID_DOCUMENT;
    xmlFreeDoc(doc);
    return NULL;
  }

  /*create a new context for use with the XPath functions*/
  context= xmlXPathNewContext(doc); 
  user_conns= (MYX_USER_CONNECTIONS*)g_malloc(sizeof(MYX_USER_CONNECTIONS));

  user_conns->last_connection=
    get_xpath_int_value(doc,"/user_connections/last_connection");
  pwd_storage_type=
    get_xpath_int_value(doc,"/user_connections/password_storage_type");

  /*retrieve the user_connections*/
  result= xmlXPathEval("/user_connections/user_connection", context);
  node_set= result->nodesetval;

  count= node_set->nodeNr;
  user_conns->user_connections_num= count;
  if (!count)
  {
    user_conns->user_connections= NULL;
  }
  else
  {
    user_conns->user_connections=
      (MYX_USER_CONNECTION*)g_malloc(sizeof(MYX_USER_CONNECTION) * count);
    for (i=0; i< count; i++)
    {
      parse_user_connection(doc,node_set->nodeTab[i],
                            user_conns->user_connections+i, pwd_storage_type);
    }
  }

  xmlXPathFreeObject(result);
  xmlXPathFreeContext(context);
  xmlFreeDoc(doc);

  return user_conns;
}

///////////////////////////////////////////////////////////////////////////////
/** @brief store connections to xml-file
    @ingroup Connection_management

    @param user_connections connections to store
    @param pwd_storage_type method of storing password
    @param filename path to file to store in
    @return -1 if file was not written, else 0

    calls NewTextChild_int_content to store int values
*//////////////////////////////////////////////////////////////////////////////
int myx_store_user_connections(MYX_USER_CONNECTIONS* user_connections,
                               MYX_PASSWORD_STORAGE_TYPE pwd_storage_type,
                               const char *filename)
{
  xmlDocPtr doc;
  xmlNodePtr root_node, node, ucon_node;
  int res, j;
  MYX_USER_CONNECTION *ucon, *ucon_end;

  if (user_connections == NULL) return -1;

  doc= xmlNewDoc("1.0");
  root_node= doc->children= xmlNewDocRawNode(doc,NULL,
                                             "user_connections", NULL);

  NewTextChild_int_content(root_node,NULL,"last_connection",
                           user_connections->last_connection);
  NewTextChild_int_content(root_node,NULL,"password_storage_type",
                           pwd_storage_type);

  ucon= user_connections->user_connections;
  ucon_end= ucon + user_connections->user_connections_num;
  for (; ucon != ucon_end; ucon++)
  {
    char *tmp;

    ucon_node= xmlNewTextChild(root_node, NULL, "user_connection", NULL);
#define STRNULL(s) (s)?(s):""
    xmlNewTextChild(ucon_node, NULL, "connection_name",
                    STRNULL(ucon->connection_name));
    xmlNewTextChild(ucon_node, NULL, "username", ucon->username);
    xmlNewTextChild(ucon_node, NULL, "hostname", ucon->hostname);
    NewTextChild_int_content(ucon_node, NULL, "port", ucon->port);
    xmlNewTextChild(ucon_node, NULL, "schema", ucon->schema);

    node= xmlNewTextChild(ucon_node, NULL, "advanced_options", NULL);
    for (j=0; j < (int)ucon->advanced_options_num; j++)
      xmlNewTextChild(node, NULL, "advanced_option",ucon->advanced_options[j]);

    xmlNewTextChild(ucon_node,NULL,"storage_path",STRNULL(ucon->storage_path));
    xmlNewTextChild(ucon_node,NULL,"notes", STRNULL(ucon->notes));
#undef STRNULL
    NewTextChild_int_content(ucon_node, NULL, "connection_type",
                             ucon->connection_type);
    NewTextChild_int_content(ucon_node,NULL,"storage_type",ucon->storage_type);

    tmp= save_password(ucon->password, pwd_storage_type);
    xmlNewTextChild(ucon_node, NULL, "password", tmp);
    g_free(tmp);
  }

  res= myx_xmlSaveFile(filename, doc);
  xmlFreeDoc(doc);
  return (res == -1) ? -1 : 0;
}

///////////////////////////////////////////////////////////////////////////////
/** @brief free memory for user connections
    @ingroup Connection_management
    @param user_connections user connections to free
    @return 0 always
*//////////////////////////////////////////////////////////////////////////////
int myx_free_user_connections(MYX_USER_CONNECTIONS *user_connections)
{
  unsigned int i;
  if (user_connections != NULL)
  {
    MYX_USER_CONNECTION *ucon= user_connections->user_connections;
    MYX_USER_CONNECTION *ucon_end= ucon+user_connections->user_connections_num;
    for (i=0; ucon != ucon_end; ucon++)
      myx_free_user_connection_content(ucon);
    g_free(user_connections);
  }
  return 0;
}

///////////////////////////////////////////////////////////////////////////////
/** @brief add a new connection to the set of connection
    @ingroup Connection_management
    @param user_connections set of connection to add to
    @param new_connection new connection to add
    @return index of the new connection 
               if there wasn't a connection with the same name
               or index of the existed connection with the same name
*//////////////////////////////////////////////////////////////////////////////
int myx_add_user_connection(MYX_USER_CONNECTIONS **user_connections,
                            MYX_USER_CONNECTION *new_connection)
{
  unsigned int i;
  int found= 0;

  if (!(*user_connections))
  {
    *user_connections= new_user_connections();
  }
  else
  {
    for (i= 0; i < (*user_connections)->user_connections_num; i++)
    {
      if ((*user_connections)->user_connections[i].connection_name &&
          strcmp2((*user_connections)->user_connections[i].connection_name,
                  new_connection->connection_name)==0)
      {
        found= 1;
        break;
      }
    }
  }
  if (!found)
  {
    MYX_USER_CONNECTION *ptr;
    unsigned int count= (*user_connections)->user_connections_num + 1;

    (*user_connections)->user_connections_num= count;
    (*user_connections)->user_connections= (MYX_USER_CONNECTION*)
      g_realloc((*user_connections)->user_connections,
                sizeof(MYX_USER_CONNECTION)*count);
    ptr= (*user_connections)->user_connections + count - 1;

#define DUPN(s) (s != NULL ? xmlMemStrdup(s) : NULL)
    
    ptr->connection_name= DUPN(new_connection->connection_name);
    ptr->username= DUPN(new_connection->username);
    ptr->password= DUPN(new_connection->password);
    ptr->hostname= DUPN(new_connection->hostname);
    ptr->port= new_connection->port;
    ptr->schema= DUPN(new_connection->schema);
    ptr->advanced_options_num= new_connection->advanced_options_num;
    if (ptr->advanced_options_num <= 0)
    {
      ptr->advanced_options= NULL;
    }
    else
    {
      char **option= (char**)g_malloc(sizeof(char*)*ptr->advanced_options_num);
      char **option_end= option + ptr->advanced_options_num;
      char **orig_option= new_connection->advanced_options;
      ptr->advanced_options= option;
      for (; option != option_end ; option++, orig_option++)
        *option= xmlMemStrdup(*orig_option);
    } 
    ptr->storage_path= DUPN(new_connection->storage_path);
    ptr->notes= DUPN(new_connection->notes);
    ptr->connection_type= new_connection->connection_type;
    ptr->storage_type= new_connection->storage_type;

#undef DUPN

    return (*user_connections)->user_connections_num-1;
  }

  return i;
}

///////////////////////////////////////////////////////////////////////////////
/** @brief free memory for user connection
    @ingroup Connection_management
    @param uc user connection to free
    @return 0 always
*//////////////////////////////////////////////////////////////////////////////
int myx_free_user_connection_content(MYX_USER_CONNECTION *uc)
{
  char **aopt_pos, **aopt_end;

  xmlFree(uc->connection_name);
  xmlFree(uc->username);
  xmlFree(uc->password);
  xmlFree(uc->hostname);
  xmlFree(uc->schema);

  aopt_pos= uc->advanced_options;
  aopt_end= aopt_pos + uc->advanced_options_num;
  for (; aopt_pos != aopt_end; aopt_pos++)
    xmlFree(*aopt_pos);
  xmlFree(uc->storage_path);
  xmlFree(uc->notes);

  g_free(uc->advanced_options);
  return 0;
}

/*
 * Private functions
 */

///////////////////////////////////////////////////////////////////////////////
/** @brief Pseudo-encryption
    @param str string to encrypt
    @return encrypted string
    simple binary invertion for now

    This function is used for encrypting as well as decrypting
*//////////////////////////////////////////////////////////////////////////////
static xmlChar *obscure(const xmlChar* str)
{
  if (!str)
  {
    return 0;
  }
  else
  {
    gulong len= (gulong)strlen(str);
    xmlChar *result= (xmlChar*)g_malloc(len+1);
    xmlChar *res_pos= result;
    xmlChar *res_end= result + len;
    for (; res_pos != res_end; res_pos++, str++)
      *res_pos= ~(*str);
    *res_pos= 0;
    return result;
  }
}

#if defined(__WIN__) || defined(_WIN32) || defined(_WIN64)
///////////////////////////////////////////////////////////////////////////////
/** @brief Crypt uprotected password
    @param cipher uprotected password
    @return crypted string
*//////////////////////////////////////////////////////////////////////////////
static xmlChar* win_load_password(xmlChar *cipher)
{
  DATA_BLOB data_in;
  DATA_BLOB data_out;
  xmlChar *plaintext;

  data_in.pbData= hex_decode(cipher, &data_in.cbData);

  if (! CryptUnprotectData(&data_in,
                           NULL,
                           NULL,        // Optional entropy
                           NULL,        // Reserved
                           NULL,
                           0,
                           &data_out))
  {
    return NULL;
  }

  g_free(data_in.pbData);

  plaintext= g_strdup(data_out.pbData);
  LocalFree(data_out.pbData);

  return plaintext;
}

///////////////////////////////////////////////////////////////////////////////
/** @brief Crypt protected password
    @param protected password
    @return crypted string

    Stores the password using Microsoft's DPAPI (which is a subset of the 
    Cryptographic-API)
    This means that the encryption key is derived from the user-credentials.
    Only the same user can decrypt it.
    In case of an error NULL is returned.
*//////////////////////////////////////////////////////////////////////////////
static xmlChar* win_store_password(xmlChar *password)
{
  DATA_BLOB data_in;
  DATA_BLOB data_out;
  xmlChar *cipher;

  data_in.pbData= password;
  // include the trailing 0 so we get a null-terminated string back 
  //  after decrypting
  data_in.cbData= (DWORD) strlen(password) +1;
  
  if (!CryptProtectData(&data_in,           
                                                  // A description string:
                        L"This is the description string.", 
                        NULL,                     // Optional entropy not used.
                        NULL,                     // Reserved.
                        NULL,                      
                        CRYPTPROTECT_UI_FORBIDDEN,
                        &data_out))
  {
    int error= GetLastError();
    return NULL;
  }

  //hex-encode it so we get a string that we can save in
  //an xml-file.
  cipher= hex_encode(data_out.pbData, data_out.cbData);
  LocalFree(data_out.pbData);

  return cipher;
}
#endif

///////////////////////////////////////////////////////////////////////////////
/** @brief used for retreiving of password from xml-plaintext
    @param password string contained password to retrieve
    @param storage_type method of retreiving
    @return retrieved password (allocated by xmlStrdup)

    calls hex_decode() and obscure() to retrieve encrypted password <BR>
*//////////////////////////////////////////////////////////////////////////////
static xmlChar *retrieve_password(xmlChar *password,
                                  enum myx_password_storage_type storage_type)
{
  xmlChar *tmp, *tmp2, *plaintext_password;

  if (!password) return NULL;

  switch (storage_type)
  {
  case MYX_PASSWORD_NOT_STORED: return NULL;     break;
  case MYX_PASSWORD_PLAINTEXT:  return xmlStrdup(password); break;

  case MYX_PASSWORD_OBSCURED:
    {
      //since the old hex_encode function produced a single 0 at
      //the end of the str (instead of two), remove it
      unsigned int password_len= (unsigned int)strlen(password);
      if (password_len % 2 != 0 && password[password_len-1] == '0')
        password[password_len-1]= 0;
    }
    tmp= hex_decode(password, NULL);
    tmp2= obscure(tmp);
    plaintext_password= xmlStrdup(tmp2);
    g_free(tmp);
    g_free(tmp2);
    return plaintext_password;
    
    break;

  case MYX_PASSWORD_OS_SPECIFIC:
#if defined(__WIN__) || defined(_WIN32) || defined(_WIN64)
    tmp= win_load_password(password);
    plaintext_password= xmlStrdup(tmp);
    g_free(tmp);
    return plaintext_password;
#endif
    break;
  }

  return password;
}

///////////////////////////////////////////////////////////////////////////////
/** @brief store password to xml string
    @param password password to store
    @param storage_type method of storing
    @return glib allocated string

    calls obscure() and hex_encode() to encrypt password
*//////////////////////////////////////////////////////////////////////////////
static xmlChar* save_password(xmlChar *password,
                              enum myx_password_storage_type storage_type)
{
  if (!password || !(*password)) return NULL;

  switch (storage_type)
  {
  case MYX_PASSWORD_NOT_STORED: return NULL;     break;
  case MYX_PASSWORD_PLAINTEXT:  return g_strdup(password); break;

  case MYX_PASSWORD_OBSCURED:
    {
      xmlChar *tmp= obscure(password);
      xmlChar *encrypted_password= hex_encode(tmp,-1);
      g_free(tmp);
      return encrypted_password;
    }
  case MYX_PASSWORD_OS_SPECIFIC:
#if defined(__WIN__) || defined(_WIN32) || defined(_WIN64)
    return win_store_password(password);
#endif
    break;
  }
  return password;
}

///////////////////////////////////////////////////////////////////////////////
/** @brief used for parsing a user_connection from the xml-dom-tree
    @param doc parsed xml-document
    @param user_conn Node in parsed xml-document contained the Connection
    @param out Connection struct to fill
    @param pwd_storage_type method of reading of passsord

    is called from myx_load_user_connections() <BR>
    calls try_to_get_string_field(), try_to_get_int_field(), 
        retrieve_password()
*//////////////////////////////////////////////////////////////////////////////
static 
  void parse_user_connection(xmlDocPtr doc, xmlNodePtr user_conn,
                             MYX_USER_CONNECTION *out,
                             enum myx_password_storage_type pwd_storage_type)
{
  xmlNodePtr cur;
  int i;
  xmlXPathObjectPtr result;
  xmlNodeSetPtr node_set;

  assert( !xmlStrcmp(user_conn->name, "user_connection") );
  memset(out, 0, sizeof(MYX_USER_CONNECTION));
  for (cur= user_conn->children; cur != NULL; cur= cur->next)
  {
    try_to_get_string_field(doc,cur,"connection_name",&out->connection_name);
    try_to_get_string_field(doc,cur,"username",       &out->username);
    try_to_get_string_field(doc,cur,"hostname",       &out->hostname);
    try_to_get_string_field(doc,cur,"schema",         &out->schema);
    try_to_get_string_field(doc,cur,"storage_path",   &out->storage_path);
    try_to_get_string_field(doc,cur,"notes",          &out->notes);

    try_to_get_int_field(doc,cur,"port",           (int*)&out->port);
    try_to_get_int_field(doc,cur,"connection_type",
                         (int*)&out->connection_type);
    try_to_get_int_field(doc,cur,"storage_type",   (int*)&out->storage_type);

    if ( !xmlStrcmp(cur->name, "password") )
    {
      xmlChar *tmp;
      if (out->password)
        xmlFree(out->password);
      tmp= xmlNodeListGetString(doc, cur->children,1);
      out->password= retrieve_password(tmp, pwd_storage_type);
      xmlFree(tmp);
    }
  }

  context->node= user_conn;
  result= xmlXPathEval("advanced_options/advanced_option", context);
  node_set= result->nodesetval;
  out->advanced_options_num= !node_set ? 0 : node_set->nodeNr;
  if (out->advanced_options_num == 0)
  {
    out->advanced_options= NULL;
  }
  else
  {
    out->advanced_options= (char**)g_malloc(sizeof(char*) * node_set->nodeNr);
    for (i=0; i< (int)out->advanced_options_num; i++)
    {
      out->advanced_options[i]=
        xmlNodeListGetString(doc, node_set->nodeTab[i]->children, 1);
    }
  }
}

/** @} */ // end of Connection_management_private

