unit ApplicationDataModule;

interface

uses
  gnugettext, SysUtils, Classes, MyxError, Options, Forms, AuxFuncs, Dialogs,
  myx_public_interface, StrUtils;

type
  TMyxGrtShOptions = class;

  TApplicationDM = class(TDataModule)
    procedure DataModuleCreate(Sender: TObject);
    procedure DataModuleDestroy(Sender: TObject);

    procedure CheckCommandlineParameter;

    procedure OnApplicationException(Sender: TObject; E: Exception);

    procedure LoadOptions;
    function GetLastFileDialogPaths(DlgName: WideString): WideString;
    procedure SetLastFileDialogPaths(DlgName: WideString; Path: WideString);
  private
    FOptions: TMyxGrtShOptions;
    FOptionProvider: IOptionProvider;
  public
    ApplicationIsTerminating: Boolean;
    property Options: TMyxGrtShOptions read FOptions;
  end;

  TMyxGrtShOptions = class(TMyxOptions)
    constructor Create; reintroduce; overload;
    destructor Destroy; override;

    procedure LoadOptions; override;
    procedure StoreOptions; override;
  public
    Verbose,
      RemoteDebug: Boolean;
    JvmLibrary,
      StartScriptFile: WideString;

    SideBarWidth: Integer;

    LastFileDialogPaths: TStringList;

    CreateWindowsStyleLineBreaks: Boolean;

    UseToolbarGradient: Boolean;
  end;

  EMyxQueryBrowserLibError = class(EMyxLibraryError);

var
  ApplicationDM: TApplicationDM;

implementation

{$R *.dfm}

procedure TApplicationDM.DataModuleCreate(Sender: TObject);
var
  s: WideString;
begin
  //Exception Handling
  Application.OnException := OnApplicationException;

  ApplicationIsTerminating := False;

  //DLL Version check
  if (libmysqlx_PUBLIC_INTERFACE_VERSION <> myx_get_public_interface_version) then
    s := Format(_('There is a incompatible version of the ' +
      'library %s installed (Version %s). Please update the library to version %s.'),
      ['libmysqlx.dll', FormatLibraryVersion(myx_get_public_interface_version),
      FormatLibraryVersion(libmysqlx_PUBLIC_INTERFACE_VERSION)]) + #13#10#13#10;

  if (s <> '') then
    if (ShowModalDialog('Library version mismatch',
      trim(s), myx_mtError, 'Quit' + #13#10 + 'Ignore') = 1) then
    begin
      ApplicationIsTerminating := True;
      Application.Terminate;
    end;
end;

procedure TApplicationDM.DataModuleDestroy(Sender: TObject);

begin
  FOptions := nil;
  FOptionProvider := nil;
end;

procedure TApplicationDM.CheckCommandlineParameter;

var
  i: integer;

begin
  i := 1;
  while (i <= ParamCount) do
  begin
    if (WideSameText(ParamStr(i), '-verbose')) then
      FOptions.Verbose := True
    else
      if (WideSameText(ParamStr(i), '-debug')) then
        FOptions.RemoteDebug := True
      else
        if (WideSameText(ParamStr(i), '-jvm')) and (i < ParamCount) then
        begin
          FOptions.JvmLibrary := ParamStr(i + 1);
          inc(i);
        end
        else
          FOptions.StartScriptFile := ParamStr(i);

    inc(i);
  end;
end;

procedure TApplicationDM.OnApplicationException(Sender: TObject; E: Exception);
begin
  if (not (Application.Terminated)) then
    ShowError(E);
end;

procedure TApplicationDM.LoadOptions;
begin
  //Load Options
  FOptions := TMyxGrtShOptions.Create;
  FOptionProvider := FOptions; // Increment reference counter.
end;

function TApplicationDM.GetLastFileDialogPaths(DlgName: WideString): WideString;
begin
  Result := FOptions.LastFileDialogPaths.Values['DlgName'];
end;

procedure TApplicationDM.SetLastFileDialogPaths(DlgName: WideString; Path: WideString);
begin
  FOptions.LastFileDialogPaths.Values['DlgName'] := Path;
end;

// -------------------------------------------------
// MYXOptions

constructor TMyxGrtShOptions.Create;
begin
  LastFileDialogPaths := TStringList.Create;

  inherited Create;
end;

destructor TMyxGrtShOptions.Destroy;
begin
  inherited Destroy;

  LastFileDialogPaths.Free;
end;

procedure TMyxGrtShOptions.LoadOptions;
var
  POptions: PMYX_APPLICATION_OPTIONS;
  Options: TMYX_APPLICATION_OPTIONS;
  i, j: integer;
  error: MYX_LIB_ERROR;
  OptionGroupName, OptionName, OptionValue: WideString;
  ExePath: WideString;
begin
  ExePath := ExtractFilePath(Application.ExeName);

  //------------------------------------------------------------
  //Initialize Values

  LastFileDialogPaths.Text := '';

  SideBarWidth := 230;

  CreateWindowsStyleLineBreaks := True;

  UseToolbarGradient := True;

  //------------------------------------------------------------
  //Read Options file
  if (FileExists(MYXCommonOptions.UserDataDir + 'mysqlgrtsh_options.xml')) then
  begin
    POptions := myx_get_application_options(
      MYXCommonOptions.UserDataDir + 'mysqlgrtsh_options.xml',
      @error);
    try
      if (error <> MYX_NO_ERROR) then
      begin
        ShowModalDialog('XML Error',
          'Error while loading Options file ''' +
          MYXCommonOptions.UserDataDir + 'mysqlgrtsh_options.xml' + ''''#13#10 +
          'Error Number ' + IntToStr(Ord(error)),
          myx_mtError);
      end
      else
      begin
        Options := TMYX_APPLICATION_OPTIONS.Create(POptions);
        try
          for i := 0 to Options.option_groups.Count - 1 do
            for j := 0 to Options.option_groups[i].name_value_pairs.Count - 1 do
            begin
              OptionGroupName := Options.option_groups[i].name;
              OptionName := Options.option_groups[i].name_value_pairs[j].name;
              OptionValue := Options.option_groups[i].name_value_pairs[j].value;

              // TODO: Once only the interface is used this cast will be unncessary.
              (Self as IOptionProvider).OptionAsString[OptionName] := OptionValue;
              if (CompareText(OptionGroupName, 'General') = 0) then
              begin
                if (CompareText(OptionName, 'LastFileDialogPaths') = 0) then
                  LastFileDialogPaths.Text := AnsiReplaceText(OptionValue,
                    '', #13#10);
              end
              else
                if (CompareText(OptionGroupName, 'Text') = 0) then
                begin
                  if (CompareText(OptionName, 'CreateWindowsStyleLineBreaks') = 0) then
                    CreateWindowsStyleLineBreaks := (OptionValue = '1');
                end
                else
                  if (CompareText(OptionGroupName, 'WindowPos') = 0) then
                  begin
                    WindowPosList.AddObject(OptionName, TMyxWindowPos.Create(OptionValue));
                  end
                  else
                    if (CompareText(OptionGroupName, 'UI') = 0) then
                    begin
                      if (CompareText(OptionName, 'SideBarWidth') = 0) then
                        SideBarWidth := StrToIntDef(OptionValue, 230)
                      else
                        if (CompareText(OptionName, 'UseToolbarGradient') = 0) then
                          UseToolbarGradient := (OptionValue = '1');
                    end;

            end;
        finally
          Options.Free;
        end;
      end;
    finally
      //Free Application Options
      myx_free_application_options(POptions);
    end;
  end;
end;

procedure TMyxGrtShOptions.StoreOptions;
var
  Options: TMYX_APPLICATION_OPTIONS;
  OptionGroup: TMYX_OPTION_GROUP;
  ExePath: WideString;
  i: integer;
begin
  if (MYXCommonOptions.UserDataDir='') then
    Exit;

  ExePath := ExtractFilePath(Application.ExeName);

  //Create Application Options
  Options := TMYX_APPLICATION_OPTIONS.create;
  try
    //-----------------------------------------------
    //Create Option Group
    OptionGroup := TMYX_OPTION_GROUP.create('General');
    Options.option_groups.Add(OptionGroup);

    AddParam(OptionGroup, 'LastFileDialogPaths',
      AnsiReplaceText(LastFileDialogPaths.Text, #13#10, ''));

    //-----------------------------------------------
    //Create Option Group
    OptionGroup := TMYX_OPTION_GROUP.create('Text');
    Options.option_groups.Add(OptionGroup);

    AddParam(OptionGroup, 'CreateWindowsStyleLineBreaks',
      IntToStr(Ord(CreateWindowsStyleLineBreaks)));

    //-----------------------------------------------
    //Create Option Group
    OptionGroup := TMYX_OPTION_GROUP.create('UI');
    Options.option_groups.Add(OptionGroup);

    AddParam(OptionGroup, 'SideBarWidth',
      IntToStr(SideBarWidth));

    AddParam(OptionGroup, 'UseToolbarGradient',
      IntToStr(Ord(UseToolbarGradient)));

    //-----------------------------------------------
    //Create Option Group
    OptionGroup := TMYX_OPTION_GROUP.create('WindowPos');
    Options.option_groups.Add(OptionGroup);

    //Store all window positions
    for i := 0 to WindowPosList.Count - 1 do
      AddParam(OptionGroup, WindowPosList[i],
        TMyxWindowPos(WindowPosList.Objects[i]).AsWideString);

    // Convert all options (stored only in the list) to the needed name-value-pairs.
    // TODO: Once everything works via the list this will be the main save method.
    StoreListOptions(Options.option_groups);

    //-----------------------------------------------
    //Store Options to file
    myx_store_application_options(Options.get_record_pointer,
      MYXCommonOptions.UserDataDir + 'mysqlgrtsh_options.xml');
  finally
    Options.Free;
  end;
end;

end.

