-- ----------------------------------------------------------------------------------------
-- Copyright (C) 2004 MySQL AB
--
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
-- ----------------------------------------------------------------------------------------


-- ----------------------------------------------------------------------------------------
-- @file RdbmsConnections.lua
-- @brief Module that contains functionality for database drivers
-- ----------------------------------------------------------------------------------------

-- ----------------------------------------------------------------------------------------
-- @brief Returns the information about this module
--
--   Every Grt module has to implement this function to return information about the 
-- module. Note that new functions that should be exposed to the Grt have to be listed 
-- here. Function that are not exposed should start with a underscore.
--
-- @return A dict that contains the name and the function names of the module
-- ----------------------------------------------------------------------------------------

function getModuleInfo()
  local moduleInfo= 
    {
      name= "RdbmsConnections", 
      functions= {
        "getConnManagement::",
        "storeConns::"
      }, 
      extends= ""
    }

  return moduleInfo
end

-- ----------------------------------------------------------------------------------------
-- @brief Returns a dict containing connection management information
--
--   Returns a structed dict that contains a list of all drivers and stored connections
--
-- @return a structed dict that contains a list of all drivers and stored connections
-- ----------------------------------------------------------------------------------------
function getConnManagement()
  local connMgmt= grtV.newObj("db.conn.Management", "driverManagement", grt.newGuid(), "")

  -- Get driver information
  grtV.insert(connMgmt.driverInfos, getDriversMysql())
  grtV.insert(connMgmt.driverInfos, getDriversOracle())
  grtV.insert(connMgmt.driverInfos, getDriversMssql())
  grtV.insert(connMgmt.driverInfos, getDriversAccess())
  grtV.insert(connMgmt.driverInfos, getDriversGeneric())

  -- Prepare a list of stored connections
  local storedConnsFilename= grt.getResLua(Base:getAppDataDir()) .. "mysqlx_connections.xml"

  if grt.fileExists(storedConnsFilename) then
    connMgmt.storedConns= grtV.load(storedConnsFilename)
  end

  return grt.success(connMgmt)
end

-- ----------------------------------------------------------------------------------------
-- @brief Fills the given list with driver information
--
--   Helper function to fill the given list with driver information
--
-- @param drivers a typed list that will hold the drivers
-- ----------------------------------------------------------------------------------------
function storeConns(args)
  if args == nil then
    return grt.error("The first argument has to be the global path of stored connections")
  end

  local storedConnsFilename= grt.getResLua(Base:getAppDataDir()) .. "mysqlx_connections.xml"
  grtV.save(grtV.getGlobal(args[1]), storedConnsFilename)

  return grt.success()
end

-- ----------------------------------------------------------------------------------------
-- @brief Returns a new created driver parameter
--
--   Helper function to create a new driver parameter
--
-- @param owner the owner GRT value (the driver)
-- @param name the name of the parameter (username, password, port, host, ...)
-- @param caption the caption of the parameter
-- @param desc the description
-- @param paramType the type of the param (can be "string", "int", "file")
-- @param row the layout row in which to create the widget for this parameter
-- @param width the width of the widget
-- @param defaultValue the default value or "" if there is no default value
-- 
-- @return a new created GRT value of struct "db.conn.DriverParameter"
-- ----------------------------------------------------------------------------------------
function getDriverParameter(owner, name, caption, desc, paramType,
  row, width, defaultValue, layoutAdvanced, required)

  local param= grtV.newObj("db.conn.DriverParameter", name, grt.newGuid(), grtV.toLua(owner._id))

  param.caption= caption
  param.description= desc
  param.paramType= paramType
  param.defaultValue= defaultValue
  param.layoutAdvanced= 0
  param.layoutRow= row
  param.layoutWidth= width
  param.layoutAdvanced= layoutAdvanced
  param.required= required

  return param
end

-- ----------------------------------------------------------------------------------------
-- @brief Adds default driver parameter to the driver parameter list
--
--   Helper function to add the default parameters host, port, username, password to
-- the given driver parameter list
--
-- @param owner the owner GRT value (the driver)
-- @param params the parameter list GRT value
-- @param startRow in which row the default parameter should begin (set to 2 to allow
--   another parameter to be in the first line)
-- @param defaultPort the default port number
-- ----------------------------------------------------------------------------------------
function addDriverParamDefaults(owner, params, startRow, defaultPort)
  grtV.insert(params, getDriverParameter(owner, "host", "Hostname:", 
    "Name or IP address of the server machine", "string", startRow, 118, "", 0, 1))

  grtV.insert(params, getDriverParameter(owner, "port", "Port:", 
    "TCP/IP port", "int", startRow, 46, defaultPort, 0, 1))

  grtV.insert(params, getDriverParameter(owner, "username", "Username:", 
    "Name of the user to connect with.", "string", startRow + 1, 218, "", 0, 1))

  grtV.insert(params, getDriverParameter(owner, "password", "Password:", 
    "The user's password.", "password", startRow + 2, 218, "", 0, 1))
end



-- ----------------------------------------------------------------------------------------
-- @brief Function to get all MySQL drivers
--
--   Helper function to return all available MySQL drivers
-- 
-- @return a new created GRT value of struct "db.conn.Rdbms" containing the drivers infos
-- ----------------------------------------------------------------------------------------
function getDriversMysql()
  -- create Rdbms object
  local rdbms= grtV.newObj("db.conn.Rdbms", "Mysql", "{6D75781B-52CF-4252-9B3D-9F28B75C09F7}", "")

  rdbms.caption= "MySQL Server"

  -- add driver to the Rdbms' list of drivers
  grtV.insert(rdbms.drivers, getDriverMysqlNative(rdbms))
  grtV.insert(rdbms.drivers, getDriverMysqlJdbc(rdbms))
  grtV.insert(rdbms.drivers, getDriverMysqlEmbedded(rdbms))

  rdbms.defaultDriver= rdbms.drivers[0]._id

  return rdbms
end

-- ----------------------------------------------------------------------------------------
-- @brief Function to get the native MySQL driver
--
--   Helper function to return infos about the native MySQL driver
-- 
-- @param owner the Grt value of the Rdbms
--
-- @return a new created GRT value of struct "db.conn.Driver" containing the driver infos
-- ----------------------------------------------------------------------------------------
function getDriverMysqlNative(owner)
  -- create driver object
  local driver= grtV.newObj("db.conn.NativeDriver", "MysqlNative", 
    "{D65C7567-0B84-4AC6-A1A4-0A4BB8C9F3F2}", grtV.toLua(owner._id))

  -- set driver values
  driver.caption= "MySQL Native Driver"
  driver.description= "MySQL native driver using the MySQL client library"
  driver.schemaStructs= "db.mysql"

  driver.filesTarget= "."
  grtV.insert(driver.files, "libmysql.dll")
  driver.downloadUrl= "http://dev.mysql.com/downloads/mysql/"

  -- add driver parameters
  addDriverParamDefaults(driver, driver.parameters, 1, "3306")

  -- add additional parameters
  local param= getDriverParameter(owner, "schema", "Default Schema:", 
    "The schema that will be used as default schema", "string", -1, 218, "", 0, 0)
  param.lookupValueModule= "ReverseEngineeringMysql"
  param.lookupValueMethod= "getSchemataList"
  grtV.insert(driver.parameters, param)

  -- advanced parameters
  grtV.insert(driver.parameters, getDriverParameter(owner, "useCompression", "Use compression protocol", 
    "Select this option for WAN connections", "boolean", -1, 318, "", 1, 0))
  grtV.insert(driver.parameters, getDriverParameter(owner, "useSSL", 
    "Use SSL if available (the client library needs to support it)", 
    "This option turns on SSL encryption if the client library supports it", "boolean", -1, 318, "", 1, 0))
  grtV.insert(driver.parameters, getDriverParameter(owner, "useAnsiQuotes", "Use ANSI quotes to quote identifiers", 
    "If enabled this option overwrites the serverside settings", "tristate", -1, 318, "", 1, 0))
  if grt.getResLua(Base:getOsTypeName()) == "WINDOWS" then
    grtV.insert(driver.parameters, getDriverParameter(owner, "socket", "Named Pipe:", 
      "Use the specified named pipe instead of a TCP/IP connection. This only works on localhost.", 
      "string", -1, 218, "", 1, 0))
  else
    grtV.insert(driver.parameters, getDriverParameter(owner, "socket", "Socket Name:", 
      "Use the specified socket instead of a TCP/IP connection. This only works on localhost.", 
      "string", -1, 218, "", 1, 0))
  end

  driver.defaultModules= 
    {
      ReverseEngineeringModule= "ReverseEngineeringMysql",
      MigrationModule= "MigrationGeneric",
      TransformationModule= "TransformationMysql"
    }

  driver.isInstalled= 1

  return driver
end

-- ----------------------------------------------------------------------------------------
-- @brief Function to get the Jdbc MySQL driver
--
--   Helper function to return infos about the Jdbc driver
-- 
-- @param owner the Grt value of the Rdbms
--
-- @return a new created GRT value of struct "db.conn.Driver" containing the driver infos
-- ----------------------------------------------------------------------------------------
function getDriverMysqlJdbc(owner)
  -- create driver object
  local driver= grtV.newObj("db.conn.JdbcDriver", "MysqlJdbc31", 
    "{8E33CDBA-2B8D-4221-96C4-506D398BC377}", grtV.toLua(owner._id))

  -- set driver values
  driver.caption= "MySQL JDBC Driver 3.1"
  driver.description= "MySQL JDBC Driver"
  driver.schemaStructs= "db.mysql"

  driver.filesTarget= "./java/lib/"
  grtV.insert(driver.files, "mysql-connector-java-3.1.6-bin.jar")
  driver.downloadUrl= "http://dev.mysql.com/downloads/connector/j/3.1.html"

  -- Jdbc specific settings
  driver.className= "com.mysql.jdbc.Driver"
  driver.connectionStringTemplate= "jdbc:mysql://%host%:%port%/?user=%username%&password=%password%&useServerPrepStmts=false"

  -- add driver parameters
  addDriverParamDefaults(driver, driver.parameters, 1, "3306")

  -- add additional parameters
  local param= getDriverParameter(owner, "schema", "Default Schema:", 
    "The schema that will be used as default schema", "string", -1, 218, "", 0, 0)
  param.lookupValueModule= "ReverseEngineeringMysqlJdbc"
  param.lookupValueMethod= "getSchemataList"
  grtV.insert(driver.parameters, param)

  -- advanced parameters
  grtV.insert(driver.parameters, getDriverParameter(owner, "jdbcConnStr", "Connection String:", 
    "Jdbc Connection String", "string", -1, 218, "", 1, 0))

  driver.defaultModules= 
    {
      ReverseEngineeringModule= "ReverseEngineeringMysqlJdbc",
      MigrationModule= "MigrationGeneric",
      TransformationModule= "TransformationMysqlJdbc"
    }

  driver.isInstalled= 1

  return driver
end

-- ----------------------------------------------------------------------------------------
-- @brief Function to get the embedded MySQL driver
--
--   Helper function to return infos about the embedded MySQL driver
-- 
-- @param owner the Grt value of the Rdbms
--
-- @return a new created GRT value of struct "db.conn.Driver" containing the driver infos
-- ----------------------------------------------------------------------------------------
function getDriverMysqlEmbedded(owner)
  -- create driver object
  local driver= grtV.newObj("db.conn.NativeDriver", "MysqlEmbedded", 
    "{64B098D9-57F1-44C6-9B6D-41852D159A38}", grtV.toLua(owner._id))

  -- set driver values
  driver.caption= "MySQL Native Driver for Embedded"
  driver.description= "MySQL native driver for the use with Embedded"
  driver.schemaStructs= "db.mysql"

  driver.filesTarget= "."
  grtV.insert(driver.files, "libmysqld.dll")
  driver.downloadUrl= "http://dev.mysql.com/downloads/mysql/"

  -- add driver parameters
  grtV.insert(driver.parameters,getDriverParameter(owner, "datadir", "Data Directory:", 
    "The MySQL data directory to use", "dir", 1, 218, "", 0, 1))

  -- add additional parameters
  local param= getDriverParameter(owner, "schema", "Default Schema:", 
    "The schema that will be used as default schema", "string", 2, 218, "", 0, 0)
  param.lookupValueModule= "ReverseEngineeringMysql"
  param.lookupValueMethod= "getSchemataList"
  grtV.insert(driver.parameters, param)

  driver.defaultModules= 
    {
      ReverseEngineeringModule= "ReverseEngineeringMysql",
      MigrationModule= "MigrationGeneric",
      TransformationModule= "TransformationMysql"
    }

  driver.isInstalled= 1

  return driver
end



-- ----------------------------------------------------------------------------------------
-- @brief Function to get all Oracle drivers
--
--   Helper function to return all available Oracle drivers
-- 
-- @return a new created GRT value of struct "db.conn.Rdbms" containing the drivers infos
-- ----------------------------------------------------------------------------------------
function getDriversOracle()
  -- create Rdbms object
  local rdbms= grtV.newObj("db.conn.Rdbms", "Oracle", "{B9E7D193-5761-461A-84DF-3D31A78D8546}", "")

  rdbms.caption= "Oracle Database Server"

  -- add driver to the Rdbms' list of drivers
  grtV.insert(rdbms.drivers, getDriverOracleJdbcSid(rdbms))
  grtV.insert(rdbms.drivers, getDriverOracleJdbcService(rdbms))

  rdbms.defaultDriver= rdbms.drivers[0]._id

  return rdbms
end

-- ----------------------------------------------------------------------------------------
-- @brief Function to get the Oracle driver using Sid
--
--   Helper function to return infos about the Jdbc driver
-- 
-- @param owner the Grt value of the Rdbms
--
-- @return a new created GRT value of struct "db.conn.Driver" containing the driver infos
-- ----------------------------------------------------------------------------------------
function getDriverOracleJdbcSid(owner)
  -- create driver object
  local driver= grtV.newObj("db.conn.JdbcDriver", "OracleThinSid", 
    "{C72E3867-3115-41BE-9DCF-102DDB62093F}", grtV.toLua(owner._id))

  -- set driver values
  driver.caption= "Oracle Thin JDBC Driver using SID"
  driver.description= "Oracle Thin JDBC driver to connect to Oracle 9i and Oracle 10g servers."
  driver.schemaStructs= "db.oracle"

  driver.filesTarget= "./java/lib/"
  grtV.insert(driver.files, "ojdbc14.jar")
  driver.downloadUrl= "http://www.oracle.com/technology/software/tech/java/sqlj_jdbc/htdocs/jdbc101020.html"

  -- Jdbc specific settings
  driver.className= "oracle.jdbc.OracleDriver"
  driver.connectionStringTemplate= "jdbc:oracle:thin:%username%/%password%@%host%:%port%:%sid%"

  -- add driver parameters
  grtV.insert(driver.parameters, getDriverParameter(owner, "sid", "SID:", 
    "Oracle system identifier", "string", 1, 218, "", 0, 1))

  addDriverParamDefaults(driver, driver.parameters, 2, "1521")

  -- advanced parameters
  grtV.insert(driver.parameters, getDriverParameter(owner, "jdbcConnStr", "Connection String:", 
    "Jdbc Connection String", "string", -1, 218, "", 1, 0))
  
  driver.defaultModules= 
    {
      ReverseEngineeringModule= "ReverseEngineeringOracle",
      MigrationModule= "MigrationOracle",
      TransformationModule= ""
    }

  driver.isInstalled= grt.getRes(BaseJava:javaClassExists(driver.className))

  return driver
end

-- ----------------------------------------------------------------------------------------
-- @brief Function to get the Oracle driver using Service
--
--   Helper function to return infos about the Jdbc driver
-- 
-- @param owner the Grt value of the Rdbms
--
-- @return a new created GRT value of struct "db.conn.Driver" containing the driver infos
-- ----------------------------------------------------------------------------------------
function getDriverOracleJdbcService(owner)
  -- create driver object
  local driver= grtV.newObj("db.conn.JdbcDriver", "OracleThinService", 
    "{212808F5-EE3A-44F1-A437-40CA6A9BD63A}", grtV.toLua(owner._id))

  -- set driver values
  driver.caption= "Oracle Thin JDBC Driver using Service"
  driver.description= "Oracle Thin JDBC driver to connect to Oracle 9i and Oracle 10g servers."
  driver.schemaStructs= "db.oracle"

  driver.filesTarget= "./java/lib/"
  grtV.insert(driver.files, "ojdbc14.jar")
  driver.downloadUrl= "http://www.oracle.com/technology/software/tech/java/sqlj_jdbc/htdocs/jdbc101020.html"

  -- Jdbc specific settings
  driver.className= "oracle.jdbc.OracleDriver"
  driver.connectionStringTemplate= "jdbc:oracle:thin:%username%/%password%@//%host%:%port%/%service%"

  -- add driver parameters
  grtV.insert(driver.parameters, getDriverParameter(owner, "service", "Service:", 
    "Oracle service name", "string", 1, 218, "", 0, 1))

  addDriverParamDefaults(driver, driver.parameters, 2, "1521")

  -- advanced parameters
  grtV.insert(driver.parameters, getDriverParameter(owner, "jdbcConnStr", "Connection String:", 
    "Jdbc Connection String", "string", -1, 218, "", 1, 0))
  

  driver.defaultModules= 
    {
      ReverseEngineeringModule= "ReverseEngineeringOracle",
      MigrationModule= "MigrationOracle",
      TransformationModule= ""
    }

  driver.isInstalled= grt.getRes(BaseJava:javaClassExists(driver.className))

  return driver
end




-- ----------------------------------------------------------------------------------------
-- @brief Function to get all Mssql drivers
--
--   Helper function to return all available Oracle drivers
-- 
-- @return a new created GRT value of struct "db.conn.Rdbms" containing the drivers infos
-- ----------------------------------------------------------------------------------------
function getDriversMssql()
  -- create Rdbms object
  local rdbms= grtV.newObj("db.conn.Rdbms", "Mssql", "{C31762DD-0C04-45A0-8030-C05A2382E004}", "")

  rdbms.caption= "MS SQL Server"

  -- add driver to the Rdbms' list of drivers
  grtV.insert(rdbms.drivers, getDriverMssqlJdbc(rdbms))

  rdbms.defaultDriver= rdbms.drivers[0]._id

  return rdbms
end

-- ----------------------------------------------------------------------------------------
-- @brief Function to get the MS SQL driver
--
--   Helper function to return infos about the Jdbc driver
-- 
-- @param owner the Grt value of the Rdbms
--
-- @return a new created GRT value of struct "db.conn.Driver" containing the driver infos
-- ----------------------------------------------------------------------------------------
function getDriverMssqlJdbc(owner)
  -- create driver object
  local driver= grtV.newObj("db.conn.JdbcDriver", "jTDS102", 
    "{9437F83E-FF9E-4831-9132-B4502303C229}", grtV.toLua(owner._id))

  -- set driver values
  driver.caption= "MS SQL JDBC Driver"
  driver.description= "JDBC driver to connect to MS SQL Server 2000 and 2005."
  driver.schemaStructs= "db.mssql"

  driver.filesTarget= "./java/lib/"
  grtV.insert(driver.files, "jtds-1.0.2.jar")
  driver.downloadUrl= "http://sourceforge.net/project/showfiles.php?group_id=33291"

  -- Jdbc specific settings
  driver.className= "net.sourceforge.jtds.jdbc.Driver"
  driver.connectionStringTemplate= "jdbc:jtds:sqlserver://%host%:%port%/%database%;user=%username%;password=%password%"

  -- add driver parameters
  addDriverParamDefaults(driver, driver.parameters, 1, "1433")

  local param= getDriverParameter(owner, "database", "Database:", 
    "Name of the database, e.g. Northwind.", "string", -1, 218, "", 0, 1)
  param.lookupValueModule= "ReverseEngineeringMssql"
  param.lookupValueMethod= "getCatalogList"
  grtV.insert(driver.parameters, param)


  -- advanced parameters
  grtV.insert(driver.parameters, getDriverParameter(owner, "jdbcConnStr", "Connection String:", 
    "Jdbc Connection String", "string", -1, 218, "", 1, 0))
  

  driver.defaultModules= 
    {
      ReverseEngineeringModule= "ReverseEngineeringMssql",
      MigrationModule= "MigrationMssql",
      TransformationModule= ""
    }

  driver.isInstalled= grt.getRes(BaseJava:javaClassExists(driver.className))

  return driver
end



-- ----------------------------------------------------------------------------------------
-- @brief Function to get all MsAccess drivers
--
--   Helper function to return all available Oracle drivers
-- 
-- @return a new created GRT value of struct "db.conn.Rdbms" containing the drivers infos
-- ----------------------------------------------------------------------------------------
function getDriversAccess()
  -- create Rdbms object
  local rdbms= grtV.newObj("db.conn.Rdbms", "Access", "{D150C038-494B-444F-9E96-4BF35E58762B}", "")

  rdbms.caption= "MS Access"

  -- add driver to the Rdbms' list of drivers
  grtV.insert(rdbms.drivers, getDriverAccessJdbc(rdbms))

  rdbms.defaultDriver= rdbms.drivers[0]._id

  return rdbms
end

-- ----------------------------------------------------------------------------------------
-- @brief Function to get the MS SQL driver
--
--   Helper function to return infos about the Jdbc driver
-- 
-- @param owner the Grt value of the Rdbms
--
-- @return a new created GRT value of struct "db.conn.Driver" containing the driver infos
-- ----------------------------------------------------------------------------------------
function getDriverAccessJdbc(owner)
  -- create driver object
  local driver= grtV.newObj("db.conn.JdbcDriver", "Access", 
    "{A8F2E8C2-415A-48C5-B8F8-95EE6E7D4FDB}", grtV.toLua(owner._id))

  -- set driver values
  driver.caption= "MS Access"
  driver.description= "JDBC driver to connect to MS Access."
  driver.schemaStructs= "db"

  -- Jdbc specific settings
  driver.className= "sun.jdbc.odbc.JdbcOdbcDriver"
  driver.connectionStringTemplate= "jdbc:odbc:Driver={Microsoft Access Driver (*.mdb)};DBQ=%databaseFile%;DriverID=22;READONLY=true;UID=%username%;PWD=%password%}"

  -- add driver parameters
  local param= getDriverParameter(owner, "databaseFile", "Database File:", 
    "MS Access database file.", "file", 1, 218, "", 0, 1)
  param.paramTypeDetails=
    {
      fileType= "MS Access Files",
      fileExtension= "mdb",
      fileOpenDialogCaption, "Open MS Access File ..."
    }
  grtV.insert(driver.parameters, param)

  grtV.insert(driver.parameters, getDriverParameter(owner, "username", "Username:", 
    "Name of the user to connect with.", "string", 2, 218, "", 0, 0))

  grtV.insert(driver.parameters, getDriverParameter(owner, "password", "Password:", 
    "The user's password.", "password", 3, 218, "", 0, 0))

  -- advanced parameters
  grtV.insert(driver.parameters, getDriverParameter(owner, "jdbcConnStr", "Connection String:", 
    "Jdbc Connection String", "string", -1, 218, "", 1, 0))

  driver.defaultModules= 
    {
      ReverseEngineeringModule= "ReverseEngineeringAccess",
      MigrationModule= "MigrationAccess",
      TransformationModule= ""
    }

  driver.isInstalled= 1

  return driver
end

-- ----------------------------------------------------------------------------------------
-- @brief Function to get all generic drivers
--
--   Helper function to return all available generic drivers
-- 
-- @return a new created GRT value of struct "db.conn.Rdbms" containing the drivers infos
-- ----------------------------------------------------------------------------------------
function getDriversGeneric()
  -- create Rdbms object
  local rdbms= grtV.newObj("db.conn.Rdbms", "GenericJdbc", "{76FD8EDF-45C7-4A4C-9014-F7FF485BA904}", "")

  rdbms.caption= "Generic Jdbc"

  -- add driver to the Rdbms' list of drivers
  grtV.insert(rdbms.drivers, getDriverGenericJdbc(rdbms))

  rdbms.defaultDriver= rdbms.drivers[0]._id

  return rdbms
end

-- ----------------------------------------------------------------------------------------
-- @brief Function to get the MS SQL driver
--
--   Helper function to return infos about the Jdbc driver
-- 
-- @param owner the Grt value of the Rdbms
--
-- @return a new created GRT value of struct "db.conn.Driver" containing the driver infos
-- ----------------------------------------------------------------------------------------
function getDriverGenericJdbc(owner)
  -- create driver object
  local driver= grtV.newObj("db.conn.JdbcDriver", "GenericJdbc", 
    "{27A7FC95-2459-444E-901A-3458CF82D808}", grtV.toLua(owner._id))

  -- set driver values
  driver.caption= "Generic Jdbc"
  driver.description= "Generic Jdbc driver connection"

  driver.schemaStructs= "db"

  -- Jdbc specific settings
  driver.className= ""
  driver.connectionStringTemplate= ""

  -- add driver parameters
  grtV.insert(driver.parameters, getDriverParameter(owner, "classname", "Class Name:", 
    "Classname of the driver to use.", "string", 1, 218, "", 0, 1))

  grtV.insert(driver.parameters, getDriverParameter(owner, "jdbcConnStr", "Connection String:", 
    "Jdbc Connection String", "string", 2, 218, "", 0, 1))

  driver.defaultModules= 
    {
      ReverseEngineeringModule= "ReverseEngineeringGeneric",
      MigrationModule= "MigrationGeneric",
      TransformationModule= ""
    }

  driver.isInstalled= 1

  return driver
end
