unit XGrtForms;

// Copyright (C) 2003, 2004 MySQL AB
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

interface

uses
  gnugettext, Windows, Messages, SysUtils, Forms, Controls,
  Classes, IniFiles, TntSysUtils, TntClasses, StdCtrls, TntStdCtrls,
  ExtCtrls, TntExtCtrls, Buttons, TntButtons, ComCtrls, TntComCtrls,
  Menus, TntMenus, TntForms, SyncObjs,
  TypInfo, Variants, AuxFuncs, myx_grt_public_interface;

type
  TXGrtFormApplication = class
    constructor Create(PGrt: Pointer; PModule: PMYX_GRT_MODULE);
    destructor Destroy; override;

    procedure BuildPathCacheLists;

    function CreateBridgeRoot(Argument: Pointer): Pointer;

    function InitDict(PValue: Pointer): Pointer;
    function Update(PValue: Pointer): Pointer;
    function SetValue(PValue: Pointer): Pointer;
    function GetValue(PValue: Pointer): Pointer;
    function DelValue(PValue: Pointer): Pointer;

    procedure SetValueSynchronized;
  protected
    function GetObjectById(Id: String): TObject;
    function GetIdByObject(Obj: TObject): String;
  private
    { Private declarations }
    FPGrt: Pointer;
    FPModule: PMYX_GRT_MODULE;

    FPathApplicationCache,
    FPathFormCache,
    FPathAnchorsCache,
    FPathConstraintsCache: TTntStringList;

    FInitializing: Boolean;

    FPValueToSet: Pointer;

    FIdCache: THashedStringList;
  end;

implementation

uses Grt;

// -----------------------------------------------------------------------------

constructor TXGrtFormApplication.Create(PGrt: Pointer; PModule: PMYX_GRT_MODULE);

begin
  FPGrt := PGrt;
  FPModule := PModule;

  FInitializing := False;

  BuildPathCacheLists;

  FIdCache := THashedStringList.Create;
end;

// -----------------------------------------------------------------------------

destructor TXGrtFormApplication.Destroy;

begin
  FPathApplicationCache.Free;
  FPathFormCache.Free;
  FPathAnchorsCache.Free;
  FPathConstraintsCache.Free;

  FIdCache.Free;
end;

// -----------------------------------------------------------------------------

function TXGrtFormApplication.CreateBridgeRoot(argument: Pointer): Pointer;

var
  Path: WideString;
  PValue: Pointer;

begin
  Path := GetGrtFunctionArgumentAsString(argument);

  FInitializing := True;
  try
    PValue := myx_grt_dict_new_obj(FPGrt, 'forms.Application',
      TntApplication.Title, GetIdByObject(Application), '');
  finally
    FInitializing := False;
  end;

  if (myx_grt_dict_item_set_by_path(
    myx_grt_get_root(FPGrt), Path, PValue) <> 0) then
  begin
    raise Exception.Create(Format(
      _('The object cannot be created at the given path %s.'),
      [Path]));
  end;

  Result := PValue;
end;

// -----------------------------------------------------------------------------

function TXGrtFormApplication.InitDict(PValue: Pointer): Pointer;

var
  Data: Pointer;
  StructName: WideString;
  NewObj: TObject;

begin
  StructName := myx_grt_dict_struct_get_name(PValue);

  if (StructName = 'forms.Application') then
  begin
    // Assign the global Application object
    myx_grt_value_bridge_data_object_set(PValue, Application)
  end
  else
    if (StructName = 'forms.Form') then
    begin
      Data := myx_grt_value_bridge_data_object_get(PValue);

      // if there is no bridge_data assigned yet, a new form
      // has to be created
      if (Data = nil) then
      begin
        FInitializing := True;
        try
          NewObj := TTntForm.Create(nil);
          myx_grt_value_bridge_data_object_set(PValue, NewObj);

          myx_grt_dict_item_set_value_from_string(PValue, '_id',
            GetIdByObject(NewObj));
        finally
          FInitializing := False;
        end;
      end;
    end;

  Result := nil;
end;

// -----------------------------------------------------------------------------

function TXGrtFormApplication.Update(PValue: Pointer): Pointer;

var
  Data: Pointer;
  Key: String;
  I,
    J: Integer;
  GrtListCount: Integer;
  PGrtListItem,
  PGrtListItemData: Pointer;
  Found: Boolean;

begin
  Result := nil;

  Data := myx_grt_value_bridge_data_owner_get(PValue);

  if (TObject(Data) is TApplication) then
  begin
    Key := _myx_grt_value_bridge_dict_key_get(PValue);

    // sync forms list
    if (Key = 'forms') then
    begin
      GrtListCount := myx_grt_bridge_list_item_count(PValue);

      // check if all forms are in the GRT list
      for I := 0 to Screen.FormCount - 1 do
      begin
        Found := False;

        for J := 0 to GrtListCount - 1 do
        begin
          PGrtListItem := myx_grt_bridge_list_item_get(PValue, J, 0);
          PGrtListItemData := myx_grt_value_bridge_data_object_get(PGrtListItem);

          if (PGrtListItemData = Screen.Forms[I]) then
          begin
            Found := True;
            break;
          end;
        end;

        // if it is not in the list, add it
        if (Not(Found)) then
        begin
          FInitializing := True;
          try
            // create new dict with the form as bridge_data
            PGrtListItem := myx_grt_bridge_dict_new(FPGrt,
              'forms.Form', Screen.Forms[I]);

            // initialize the form's members
            myx_grt_dict_init_obj(FPGrt, PGrtListItem,
              Screen.Forms[I].Name, GetIdByObject(Screen.Forms[I]), '');

            // add the form to the list
            myx_grt_bridge_list_item_insert(PValue, -1, PGrtListItem, 0);
          finally
            FInitializing := False;
          end;
        end;
      end;

      // check if the GRT list has a form that is no longer available
      for J := GrtListCount - 1 downto 0 do
      begin
        Found := False;

        PGrtListItem := myx_grt_bridge_list_item_get(PValue, J, 0);
        PGrtListItemData := myx_grt_value_bridge_data_object_get(PGrtListItem);

        for I := 0 to Screen.FormCount - 1 do
        begin
          if (PGrtListItemData = Screen.Forms[I]) then
          begin
            Found := True;
            break;
          end;
        end;

        // if it is no longer available, remove it from the GRT list
        if (Not(Found)) then
          myx_grt_bridge_list_item_del(PValue, J, 0);
      end;
    end;
  end
  else
    // deal with typed dicts
    if (TObject(Data) is TTntForm) then
    begin
      Key := _myx_grt_value_bridge_dict_key_get(PValue);

      if (Key = 'menuitems') then
      begin
        
      end;
    end;
end;

// -----------------------------------------------------------------------------

function TXGrtFormApplication.SetValue(PValue: Pointer): Pointer;

begin
  Result := nil;

  if (FInitializing) then
    Exit;

  FPValueToSet := PValue;

  TThread.Synchronize(nil, SetValueSynchronized);
end;

// -----------------------------------------------------------------------------

procedure TXGrtFormApplication.SetValueSynchronized;

var
  PValue,
    Data: Pointer;
  Key: String;
  Index: Integer;
  Obj: TObject;

begin
  PValue := FPValueToSet;

  Data := myx_grt_value_bridge_data_owner_get(PValue);

  // handle application dict
  if (TObject(Data) is TApplication) then
  begin
    Key := _myx_grt_value_bridge_dict_key_get(PValue);

    Index := FPathApplicationCache.IndexOf(Key);

    case Index of
      0:
        TntApplication.Title := myx_grt_bridge_value_as_string(PValue);
      4:
        SetCursorPos(myx_grt_bridge_value_as_int(PValue), Mouse.CursorPos.Y);
      5:
        SetCursorPos(Mouse.CursorPos.X, myx_grt_bridge_value_as_int(PValue));
      8:
        begin
          Obj := GetObjectById(myx_grt_bridge_value_as_string(PValue));

          if (Obj <> nil) then
            TForm(Obj).SetFocus;
        end;
    end;
  end
  else
    // handle form dict
    if (TObject(Data) is TTntForm) then
    begin
      Key := _myx_grt_value_bridge_dict_key_get(PValue);

      Index := FPathFormCache.IndexOf(Key);

      case Index of
        0:
          TTntForm(Data).Name := myx_grt_bridge_value_as_string(PValue);
        3:
          TTntForm(Data).Left := myx_grt_bridge_value_as_int(PValue);
        4:
          TTntForm(Data).Top := myx_grt_bridge_value_as_int(PValue);
        5:
          TTntForm(Data).Width := myx_grt_bridge_value_as_int(PValue);
        6:
          TTntForm(Data).Height := myx_grt_bridge_value_as_int(PValue);
        8:
          TTntForm(Data).Caption := myx_grt_bridge_value_as_string(PValue);
        9:
          TTntForm(Data).Color := HtmlColorToColor(myx_grt_bridge_value_as_string(PValue));
        10:
          TTntForm(Data).Enabled := myx_grt_bridge_value_as_int(PValue) = 1;
        11:
          TTntForm(Data).Visible := myx_grt_bridge_value_as_int(PValue) = 1;
      end;
    end;
end;

// -----------------------------------------------------------------------------

function TXGrtFormApplication.GetValue(PValue: Pointer): Pointer;

var
  Data: Pointer;
  Key: String;
  Index: Integer;

begin
  Result := nil;

  Data := myx_grt_value_bridge_data_owner_get(PValue);

  // handle application dict
  if (TObject(Data) is TApplication) then
  begin
    Key := _myx_grt_value_bridge_dict_key_get(PValue);

    Index := FPathApplicationCache.IndexOf(Key);

    case Index of
      0:
        myx_grt_bridge_value_change_string(PValue, TntApplication.Title);
      3:
        myx_grt_bridge_value_change_int(PValue, Ord(Application.Active));
      4:
        myx_grt_bridge_value_change_int(PValue, Mouse.CursorPos.X);
      5:
        myx_grt_bridge_value_change_int(PValue, Mouse.CursorPos.Y);
      6:
        begin
        end;
      8:
        myx_grt_bridge_value_change_string(PValue,
          GetIdByObject(Screen.ActiveForm));
    end;
  end
  else
    // handle forms dict
    if (TObject(Data) is TTntForm) then
    begin
      Key := _myx_grt_value_bridge_dict_key_get(PValue);

      Index := FPathFormCache.IndexOf(Key);

      case Index of
        0:
          myx_grt_bridge_value_change_string(PValue, TTntForm(Data).Name);
        3:
          myx_grt_bridge_value_change_int(PValue, TTntForm(Data).Left);
        4:
          myx_grt_bridge_value_change_int(PValue, TTntForm(Data).Top);
        5:
          myx_grt_bridge_value_change_int(PValue, TTntForm(Data).Width);
        6:
          myx_grt_bridge_value_change_int(PValue, TTntForm(Data).Height);
        8:
          myx_grt_bridge_value_change_string(PValue, TTntForm(Data).Caption);
        9:
          myx_grt_bridge_value_change_string(PValue, ColorToHtmlColor(TTntForm(Data).Color));
        10:
          myx_grt_bridge_value_change_int(PValue, Ord(TTntForm(Data).Enabled));
        11:
          myx_grt_bridge_value_change_int(PValue, Ord(TTntForm(Data).Visible));
      end;
    end;
end;

// -----------------------------------------------------------------------------

function TXGrtFormApplication.DelValue(PValue: Pointer): Pointer;

begin
  Result := nil;
end;

// -----------------------------------------------------------------------------

procedure TXGrtFormApplication.BuildPathCacheLists;

var
  PathObjectCache,
    PathControlCache: WideString;

begin
  // Build Path caches
  PathObjectCache :=
    'name'#13#10 +
    '_id'#13#10 +
    'parent'#13#10;

  FPathApplicationCache := TTntStringList.Create;
  FPathApplicationCache.Text :=
    PathObjectCache +
    'active'#13#10 +
    'mouseX'#13#10 +
    'mouseY'#13#10 +
    'forms'#13#10 +
    'mainForm'#13#10 +
    'activeForm'#13#10;

  PathControlCache :=
    'left'#13#10 +
    'top'#13#10 +
    'width'#13#10 +
    'height'#13#10 +
    'autosize'#13#10 +
    'caption'#13#10 +
    'color'#13#10 +
    'enabled'#13#10 +
    'visible'#13#10 +
    'anchors'#13#10 +
    'constraints'#13#10 +
    'parent'#13#10 +
    'controls'#13#10 +
    'eventOnClick'#13#10 +
    'eventOnDblClick'#13#10 +
    'eventOnResize'#13#10 +
    'eventOnMouseDown'#13#10 +
    'eventOnMouseMove'#13#10 +
    'eventOnMouseUp'#13#10 +
    'eventOnMouseIn'#13#10 +
    'eventOnMouseOut'#13#10 +
    'eventOnKeyDown'#13#10 +
    'eventOnKeyUp'#13#10;

  FPathFormCache := TTntStringList.Create;
  FPathFormCache.Text :=
    PathObjectCache +
    PathControlCache +
    'active'#13#10 +
    'activeControl'#13#10 +
    'alphaBlend'#13#10 +
    'alphaBlendValue'#13#10 +
    'windowState'#13#10 +
    'mainMenu'#13#10 +
    'eventOnCreate'#13#10 +
    'eventOnDestroy'#13#10 +
    'eventOnActivate'#13#10 +
    'eventOnDeactivate'#13#10 +
    'eventOnClose'#13#10 +
    'menuitems'#13#10 +
    'panels'#13#10 +
    'buttons'#13#10 +
    'edits'#13#10 +
    'checkboxes'#13#10 +
    'radiobuttons'#13#10 +
    'listboxes'#13#10 +
    'dropdowns'#13#10 +
    'pageControls'#13#10;

  //FPathFormCache_TypedFormControlsListStart := 37;

  FPathAnchorsCache := TTntStringList.Create;
  FPathAnchorsCache.Text :=
    PathObjectCache +
    'left'#13#10 +
    'top'#13#10 +
    'right'#13#10 +
    'bottom'#13#10;

  FPathConstraintsCache := TTntStringList.Create;
  FPathConstraintsCache.Text :=
    PathObjectCache +
    'minWidth'#13#10 +
    'maxWidth'#13#10 +
    'minHeight'#13#10 +
    'maxHeight'#13#10;

  // Instead of having individual lists for all typed dicts
  // for the form's controls, they are keept in a seperate
  // stringlist
  {FPathTypedFormControlsList := TTntStringList.Create;

  // Add menuitems
  FPathTypedFormControlsList.AddObject('/menuitems',
    TXGrtFormWidgetType.Create('forms.Menuitem',
    PathObjectCache +
    PathControlCache +
    'checked'#13#10 +
    'icon'#13#10,
    [TTntMenuItem, TMenuItem]));

  // Add panels
  FPathTypedFormControlsList.AddObject('/panels',
    TXGrtFormWidgetType.Create('forms.Panel',
    PathObjectCache +
    PathControlCache +
    'border'#13#10,
    [TTntPanel, TPanel]));

  // Add buttons
  FPathTypedFormControlsList.AddObject('/buttons',
    TXGrtFormWidgetType.Create(
    'forms.Button',
    PathObjectCache +
    PathControlCache +
    'isDefault'#13#10 +
    'icon'#13#10,
    [TButton, TSpeedButton, TTntButton, TTntSpeedButton]));

  // Add edits
  FPathTypedFormControlsList.AddObject('/edits',
    TXGrtFormWidgetType.Create('forms.Edit',
    PathObjectCache +
    PathControlCache +
    '/border'#13#10 +
    '/text'#13#10 +
    '/readOnly'#13#10 +
    '/passwordChar'#13#10,
    [TTntEdit, TEdit]));

  // Add checkboxes
  FPathTypedFormControlsList.AddObject('/checkboxes',
    TXGrtFormWidgetType.Create('forms.Checkbox',
    PathObjectCache +
    PathControlCache +
    '/checked'#13#10,
    [TTntCheckBox, TCheckBox]));

  // Add radiobuttons
  FPathTypedFormControlsList.AddObject('/radiobuttons',
    TXGrtFormWidgetType.Create('forms.Radiobutton',
    PathObjectCache +
    PathControlCache +
    '/group'#13#10 +
    '/checked'#13#10,
    [TTntRadioButton, TRadioButton]));

  // Add listboxes
  FPathTypedFormControlsList.AddObject('/listboxes',
    TXGrtFormWidgetType.Create('forms.Listbox',
    PathObjectCache +
    PathControlCache +
    '/multiselect'#13#10 +
    '/lastSelectedItem'#13#10 +
    '/items'#13#10 +
    '/selectedItems'#13#10,
    [TTntListBox, TListBox]));

  // Add dropdowns
  FPathTypedFormControlsList.AddObject('/dropdowns',
    TXGrtFormWidgetType.Create('forms.Dropdown',
    PathObjectCache +
    PathControlCache +
    '/fixedList'#13#10 +
    '/text'#13#10 +
    '/selectedItem'#13#10 +
    '/items'#13#10,
    [TTntComboBox, TComboBox]));

  // Add pagecontrols
  FPathTypedFormControlsList.AddObject('/pagecontrols',
    TXGrtFormWidgetType.Create('forms.Pagecontrol',
    PathObjectCache +
    PathControlCache +
    '/activePage'#13#10 +
    '/pages'#13#10,
    [TTntPageControl, TPageControl]));}
end;

// -----------------------------------------------------------------------------

function TXGrtFormApplication.GetObjectById(Id: String): TObject;

var
  Index: Integer;

begin
  Index := FIdCache.IndexOf(Id);

  if (Index > -1) then
    Result := FIdCache.Objects[Index]
  else
    Result := nil;
end;

// -----------------------------------------------------------------------------

function TXGrtFormApplication.GetIdByObject(Obj: TObject): String;

var
  Index: Integer;

begin
  Index := FIdCache.IndexOfObject(Obj);

  if (Index > -1) then
    Result := FIdCache[Index]
  else
  begin
    Result := '{' + GetNewGUID + '}';

    FIdCache.AddObject(Result, Obj);
  end;
end;

end.

