unit UniCodeEditor;

// Version 2.1.8
//
// UniCodeEditor, a Unicode Source Code Editor for Delphi.
//
// UniCodeEditor is released under the MIT license:
// Copyright (c) 1999-2005 Mike Lischke (support@soft-gems.net, www.soft-gems.net).
//
// Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated
// documentation files (the "Software"), to deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to
// permit persons to whom the Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in all copies or substantial portions of the
// Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
// WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS
// OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
// OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//
// You are asked to give the author(s) the due credit. This means that you acknowledge the work of the author(s)
// in the product documentation, about box, help or wherever a prominent place is.
//
//----------------------------------------------------------------------------------------------------------------------
//
// April 2005
//   - Bug fix: Internal line index of a line not updated when a new line is inserted.
//
// February 2005
//   - Bug fix: key handled flag removed, as it does not work with multiple keypresses.
//
// January 2005
//   - Change: UCE is now released under the MIT license.
//   - Bug fix: Cursor was not visible when moved to the right border.
//   - Bug fix: Thumb scrolling does not work correctly for large scroll ranges.
//   - Improvement: Introduction of local index in lines for quick lookup.
//
// December 2004
//   - Improvement: new option eoLineNumbersZeroBased to display line numbers in the gutter starting with 0 or 1.
//
// November 2004
//   - Bug fix: Correct insertion point for CutToClipboard.
//   - Bug fix: Wrong selection state when cutting/deleting text.
//   - Bug fix: Selection setting
//   - Improvement: Added bookmark change event.
//   - Bug fix: Selection end must be set too when start is set.
//   - Improvement: ClearSelection
//
//   Version 2.1, 2003-08-17, Mike Zinner
//     Added LineHeight property, ControlStyle csNeedsBorderPaint for XP border
//     Added properties SelStart, SelEnd, method SetCaretToEditorBottom
//     Added support for WMCopy, WMCut, WMPaste support
//     Bugfixed SetSelStart
//   Version 2.0, 2003-08-17, Mike Zinner
//     Repackaged for D7
//     Renamed to UniCodeEditor
//   Version 1.2, 2000-03-01, Mike Lischke
//     search and replace, bug fixes
//   Version 1.2, 1999-12-01, Mike Lischke
//     switched entirely to Unicode (some code already was using Unicode, e.g. output)
//     basic IME handling
//     mouse wheel support, undo/redo, bookmarks, painting optimization, scroll hint window, bug fixes
//   Version 1.1, 1999-07-01, Mike Lischke
//     line numbers, structural rework, clean up, bug fixes
//   Version 1.0, 1999-03-10, Mike Lischke
//     First implementation from a freeware lib (mwEdit), significant changes due to speed and needed functionality
//     (removed lots of redundancies in string handling, total rework of output, undo / redo, tab support, control char
//     output, first Unicode support etc.)
//
// Credits for their valuable assistance and code donations go to:
//   Mike Zinner, Tom-Vidar Nilsen
//
// Open issues:
//   - Because the the control depends on Unicode.pas, which is still beta, also the syntax editor must be considered
//     as being beta. In particular not all search (and replace) functionality is fully implemented and tested.
//   - Although the edit works with wide strings, the highlighter do not yet, because new tools must to be created
//     to build new highlighter classes which fully support wide strings (see also my homepage for DCG, the
//     Delphi Compiler Generator, which will soon be extended to create Unicode highlighters). The impact on the current
//     implementation can be seen where the edit needs to act on the text like looking for word boundaries etc.
//
//----------------------------------------------------------------------------------------------------------------------

interface

{$Booleval off} // Use fastest possible boolean evaluation.

{$I Compilers.inc}

{$ifdef COMPILER_7_UP}
  // For some things to work we need code, which is classified as being unsafe for .NET.
  {$warn UNSAFE_TYPE off}
  {$warn UNSAFE_CAST off}
  {$warn UNSAFE_CODE off}
{$endif COMPILER_7_UP}

uses
  Windows, Messages, SysUtils, Classes, Controls, Graphics, Math,
  ExtCtrls, Forms, StdCtrls, Clipbrd, Unicode, Contnrs,
  UCEEditorKeyCommands, UCEHighlighter, UCEShared;

const
  UCEVersion = '2.1.8';

  // Self defined cursors for drag'n'drop.
  crDragMove = 2910;
  crDragCopy = 3110;

type
  // eoSmartTabs and eoUseTabs are mutual exclusive (avoid activating both, as this confuses the input).
  // To use any of the tab options you must also have eoWantTabs enabled, otherwise the TAB char
  // will never even reach the edit.
  TUniCodeEditOption = (
    eoAutoIndent,                 // Do auto indentation when inserting a new line.
    eoAutoUnindent,               // Do auto undindentation when deleting a character which
                                  // is only preceded by whitespace characters.
    eoCursorThroughTabs,          // Move cursor as if tabulators were (TabSize) spaces.
    eoGroupUndo,                  // While executing undo/redo handle all continous changes of
                                  // the same kind in one call instead undoing/redoing each command separately.
    eoHideSelection,              // Hide selection when unfocused.
    eoInserting,                  // Denotes whether the control is in insert or in overwrite mode
                                  // (this option in superseeded by eoReadOnly).
    eoKeepTrailingBlanks,         // Don't automatically remove white spaces at the end of a line.
    eoLineNumbers,                // Show line numbers in gutter.
    eoLineNumbersZeroBased,       // Line numbers in the gutter start with 0. Only meaningfull with eoLineNumbers.
    eoOptimalFill,                // Not yet implemented.
    eoReadOnly,                   // Prevent the content of the control from being changed.
    eoReadOnlySelection,          // Show selection if in read-only mode (useful for simulation of static text).
    eoScrollPastEOL,              // Allow the cursor to go past the end of the line.
    eoShowControlChars,           // Show tabulators, spaces and line breaks (only if not eoScrollPastEOL) as glyphs.
    eoShowCursorWhileReadOnly,    // Don't hide the cursor if control is in read-only mode.
    eoShowScrollHint,             // Show a hint window with the current top line while scrolling with the thumb.
    eoSmartTabs,                  // Automatically put in enough spaces when TAB has been pressed to move
                                  // the cursor to the next non-white space character of the previous line
                                  // (actually only the cursor is moved, the spaces are inserted later).
    eoTripleClicks,               // Allow selecting an entire line with a triple click.
    eoUndoAfterSave,              // Don't clear the undo/redo stack after the control has been saved.
    eoUseUndoRedo,                // If set then undo and redo functionality is enabled.
    eoUseTabs,                    // Don't covert TABs to spaces but use TAB with fixed distances (see TabSize).
    eoUseSyntaxHighlighting,      // Switch on syntax highlighting (ignored if no highlighter is assigned).
    eoWantTabs                    // Use TABs for input rather than moving the focus to the next control.
  );
  TUniCodeEditOptions = set of TUniCodeEditOption;

const
  DefaultTabGlyph       = WideChar($2192);
  DefaultLineBreakGlyph = WideChar('');
  DefaultSpaceGlyph     = WideChar($2219); // Note: $B7 should not be used here because it might be interpreted
                                           //       as valid identifier character (e.g. in Catalan)

  DefaultOptions = [eoAutoIndent, eoAutoUnindent, eoCursorThroughTabs, eoGroupUndo, eoHideSelection,
    eoInserting, eoScrollPastEOL, eoSmartTabs, eoTripleClicks, eoUseSyntaxHighlighting, eoWantTabs];

type
  TReplaceAction = (raCancel, raSkip, raReplace, raReplaceAll);

  TCustomUnicodeEdit = class;

  TBookmark = record
    Visible: Boolean;
    X, Y: Integer;
  end;

  TPaintEvent = procedure(Sender: TCustomUnicodeEdit; ACanvas: TCanvas) of object;
  TCaretEvent = procedure(Sender: TCustomUnicodeEdit; X, Y: Cardinal) of object;
  TReplaceTextEvent = procedure(Sender: TCustomUnicodeEdit; const Search, Replace: WideString; Line, Start, Stop: Integer;
    var Action: TReplaceAction) of object;
  TProcessCommandEvent = procedure(Sender: TCustomUnicodeEdit; var Command: TEditorCommand; var AChar: WideChar;
    Data: Pointer) of object;
  TGutterMouseDownEvent = procedure(Sender: TCustomUnicodeEdit; Button: TMouseButton; Shift: TShiftState; X, Y,
    Line: Integer) of object;
  TBookmarkChangeEvent = procedure(Sender: TCustomUniCodeEdit; Index, X, Y: Integer; var Allowed: Boolean) of object;

  TCaretType = (
    ctVerticalLine,
    ctHorizontalLine,
    ctHalfBlock,
    ctBlock
  );

  // Undo / redo reasons.
  TChangeReason = (
    ecrNone,
    ecrInsert,
    ecrDelete,
    ecrReplace,
    ecrDragMove,
    ecrDragCopy,
    ecrCursorMove,
    ecrOverwrite,
    ecrIndentation
  );

  // This record keeps the details of a change in the editor.
  PChange = ^TChange;
  TChange = record
    Reason: TChangeReason;
    OldText: WideString;
    OldCaret,
    OldSelStart,
    OldSelEnd: TPoint;
    NewText: WideString;
    NewCaret,
    NewSelStart,
    NewSelEnd: TPoint;
  end;

  // Used to record the current edit state for registering an undo/redo action.
  TEditState = record
    Text: WideString;
    Caret,
    SelStart,
    SelEnd: TPoint;
  end;

  TUniCodeEditorContent = class;

  // Undo and redo in one list.
  TUndoList = class
  private
    FList: TList;
    FCurrent: Integer;
    FMaxUndo: Integer;
    FOwner: TCustomUniCodeEdit;
    function GetCanRedo: Boolean;
    function GetCanUndo: Boolean;
    procedure SetMaxUndo(Value: Integer);
  protected
    procedure RemoveChange(Index: Integer);
  public
    constructor Create(AOwner: TCustomUniCodeEdit);
    destructor Destroy; override;

    procedure AddChange(AReason: TChangeReason; const AOldText: WideString; AOldCaret, AOldSelStart, AOldSelEnd:
      PPoint;
      const ANewText: WideString; ANewCaret, ANewSelStart, ANewSelEnd: PPoint); overload;
    procedure AddChange(AReason: TChangeReason; const AOldText: WideString; AOldCaret, AOldSelStart, AOldSelEnd:
      PPoint); overload;
    function GetUndoChange(var Change: PChange): TChangeReason;
    function GetRedoChange(var Change: PChange): TChangeReason;
    function GetCurrentRedoReason: TChangeReason;
    function GetCurrentUndoReason: TChangeReason;
    procedure ClearList;

    property CanRedo: Boolean read GetCanRedo;
    property CanUndo: Boolean read GetCanUndo;
    property MaxUndo: Integer read FMaxUndo write SetMaxUndo;
  end;

  // States that describe the look of a line.
  TUCELineStates = set of (
    lsFolded,                // The line is folded. Use the line info record to know what to display and
                             // how large the folded area is.
    lsValidated,             // If not set then line has not been validated (parsed, bounds are not determined etc.).
    lsSelected,              // Set when the whole line is selected.
    lsWrapped                // Line is wrapped at the the right margin.
  );

  // Default implementation of the custom style interface. This is mainly used for certain fixed properties
  // that must be set at design time (e.g. selection style, scroll hint).
  TUCELineStyle = class(TInterfacedPersistent, IUCELineStyle)
  private
    FForeground: TColor;
    FBackground: TColor;
    FForceFontStyles: Boolean;
    FFontStyles: TFontStyles;
  protected
    function GetBackground: TColor; virtual;
    function GetFontStyles: TFontStyles; virtual;
    function GetForceFontStyles: Boolean; virtual;
    function GetForeground: TColor; virtual;
    procedure SetBackground(const Color: TColor); virtual;
    procedure SetFontStyles(const Styles: TFontStyles); virtual;
    procedure SetForceFontStyles(Force: Boolean); virtual;
    procedure SetForeground(const Color: TColor); virtual;
  published
    property Background: TColor read GetBackground write SetBackground;
    property Foreground: TColor read GetForeground write SetForeground;
    property FontStyles: TFontStyles read GetFontStyles write SetFontStyles;
    property ForceFontStyles: Boolean read GetForceFontStyles write SetForceFontStyles;
  end;

  // The actual data of one line.
  TStyleStack = class(TStack)
  end;

  TUCELine = class
  private
    FContent: TUniCodeEditorContent;
    FLexerState: Integer;                        // Lexer state to be used when this line is lexed. This is the state
                                                 // that the lexer was when it finished the previous line.
    FText: WideString;                           // The text of the line.
    FIndex: Integer;                             // The index in the parent list.
    FStates: TUCELineStates;
    FBounds: TRect;
    FMarkers: TInterfaceList;                    // A list of markers to show in the gutter of the edit.
    FStyles: TStyleStack;                        // A stack of custom draw styles for the line. Only the top style is actually applied.
    FData: TObject;
    procedure SetText(const Value: WideString);  // Application defined data.
  protected
    procedure Changed;
    procedure DrawMarkers(Index: Integer; Canvas: TCanvas; X, Y: Integer); virtual;
  public
    constructor Create(Owner: TUniCodeEditorContent); virtual;
    destructor Destroy; override;

    procedure Assign(Source: TUCELine);
    function AddMarker(Marker: IUCELineMarker): Integer;
    function HasMarker(Marker: IUCELineMarker): Boolean;
    procedure RemoveMarker(Marker: IUCELineMarker);
    procedure ReplaceMarker(OldMarker, NewMarker: IUCELineMarker);
    function PeekStyle: IUCELineStyle;
    procedure PopStyle;
    procedure PushStyle(Style: IUCELineStyle);
    procedure RemoveStyle(Style: IUCELineStyle);

    property Bounds: TRect read FBounds;
    property Data: TObject read FData write FData;
    property Index: Integer read FIndex;
    property LexerState: Integer read FLexerState write FLexerState;
    property States: TUCELineStates read FStates;
    property Text: WideString read FText write SetText;
  end;

  // The format of the text to load or write.
  TTextFormat = (
    tfANSI,        // Plain ANSI text. A language must be specified to allow conversion.
    tfUTF8,        // Unicode encoded as UTF 8.
    tfUTF16,       // UTF 16 with the system default byte order (on Windows this is LSB first, little-endian).
    tfUTF16LE,     // UTF 16 (little-endian) with LSB first.
    tfUTF16BE      // UTF 16 (big-endian) with MSB first.
  );

  TLineEvent = procedure(Sender: TObject; Line: TUCELine) of object;
  
  // This is an internal class for the editor and should not be used outside. It contains the actual text along
  // with additional info for each line.
  TUniCodeEditorContent = class
  private
    FOwner: TCustomUniCodeEdit;
    FLines: array of TUCELine;                   // The line storage.
    FCount: Integer;                             // The actual number of used lines. Might differ from the length of FLines.
    FChanged: Boolean;

    FOnChangeLine,
    FOnValidateLine,
    FOnDeleteLine: TLineEvent; 
    function GetLine(Index: Integer): TUCELine;
    function GetLineNoInit(Index: Integer): TUCELine;
    function GetText: WideString;
    procedure Grow;
    procedure SetChanged(const Value: Boolean);
    procedure SetText(const Value: WideString);
  protected
    procedure DoChangeLine(Line: TUCELine); virtual;
    procedure DoDeleteLine(Line: TUCELine); virtual;
    procedure DoValidateLine(Line: TUCELine); virtual;
    procedure SetCapacity(NewCapacity: Integer);
  public
    constructor Create(AOwner: TCustomUniCodeEdit); virtual;
    destructor Destroy; override;

    function AddLine(const Text: WideString): TUCELine; overload;
    function AddLine(const Line: TUCELine): TUCELine; overload;
    procedure AddStrings(const Strings: TStrings); overload;
    procedure AddStrings(const Strings: TWideStrings); overload;
    procedure AssignTo(Destination: TObject);
    procedure Clear;
    function CollectText(Start, Stop: TPoint): WideString;
    procedure DeleteLine(Index: Integer);
    procedure Error(const Msg: string; Data: Integer);
    procedure Exchange(Index1, Index2: Integer);
    function InsertLine(Index: Integer; const Text: WideString): TUCELine;
    procedure LoadFromStream(Stream: TStream; Format: TTextFormat; Language: LCID = 0);
    procedure RevalidateLine(Index: Integer); overload;
    procedure RevalidateLine(Line: TUCELine); overload;
    procedure SaveToStream(Stream: TStream; Format: TTextFormat; Language: LCID = 0; WithBOM: Boolean = True);

    property Changed: Boolean read FChanged write SetChanged;
    property Count: Integer read FCount;
    property Line[Index: Integer]: TUCELine read GetLine; default;
    property LineNoInit[Index: Integer]: TUCELine read GetLineNoInit;
    property Text: WideString read GetText write SetText;

    property OnChangeLine: TLineEvent read FOnChangeLine write FOnChangeLine;
    property OnDeleteLine: TLineEvent read FOnDeleteLine write FOnDeleteLine;
    property OnValidateLine: TLineEvent read FOnValidateLine write FOnValidateLine;
  end;

  TBookmarkOptions = class(TPersistent)
  private
    FEnableKeys: Boolean;
    FGlyphsVisible: Boolean;
    FLeftMargin: Integer;
    FOwner: TCustomUniCodeEdit;
    procedure SetGlyphsVisible(Value: Boolean);
    procedure SetLeftMargin(Value: Integer);
  public
    constructor Create(AOwner: TCustomUniCodeEdit);
  published
    property EnableKeys: Boolean read FEnableKeys write FEnableKeys default True;
    property GlyphsVisible: Boolean read FGlyphsVisible write SetGlyphsVisible default True;
    property LeftMargin: Integer read FLeftMargin write SetLeftMargin default 2;
  end;

  TScrollHintWindow = class(THintWindow)
  protected
    procedure Paint; override;
  end;

  TDistanceArray = array of Cardinal;

  TSearchOption = (
    soBackwards,             // Search backwards instead of forward.
    soEntireScope,           // Search in entire text, not only in selected text.
    soIgnoreNonSpacing,      // Ignore non-spacing characters in search.
    soMatchCase,             // Case sensitive search.
    soPrompt,                // Ask user for each replace action.
    soRegularExpression,     // Search using regular expressions.
    soReplace,               // Replace, not simple search.
    soReplaceAll,            // Replace all occurences
    soSelectedOnly,          // Search in selected text only.
    soSpaceCompress,         // Handle several consecutive white spaces as one white space,
                             // so "ab   cd" will match "ab cd" and "ab        cd".
    soWholeWord              // Match entire words only.
  );
  TSearchOptions = set of TSearchOption;

  // Miscellanous events.
  // Note: the scroll event uses pixel values for both horizontal and vertical movements.
  TUCEScrollEvent = procedure(Sender: TCustomUniCodeEdit; DeltaX, DeltaY: Integer) of object;

  TCustomUniCodeEdit = class(TCustomControl)
  private
    FBlockBegin: TPoint;
    FBlockEnd: TPoint;
    FBookmarkOptions: TBookmarkOptions;
    FBookMarks: array[0..9] of TBookMark;
    FBorderStyle: TBorderStyle;
    FCaretOffset: TPoint;
    FCaretVisible,
    FDragWasStarted: Boolean;
    FCaretX: Integer;
    FCaretY: Integer;
    FCharsInWindow: Integer;
    FCharWidth: Integer;
    FCharWidthArray: TDistanceArray;
    FContent: TUniCodeEditorContent;
    FData: TObject;                    // This is a placeholder for application defined data.
                                       // the selection block end or just the cursor position.
    FDoubleClickTime: UINT;
    FExtraLineSpacing: Integer;
    FFontDummy,
    FLineNumberFont: TFont;
    FGutterColor: TColor;
    FGutterRect: TRect;
    FGutterWidth: Integer;
    FHighLighter: TUCEHighlighter;
    FIndentSize: Integer;
    FInsertCaret: TCaretType;
    FInternalBMList: TImageList;
    FDropTarget: Boolean;              // Needed to know in the scroll timer event whether to change
    FKeyStrokes: TKeyStrokes;
    FLastCaret: TPoint;
    FLastDblClick: UINT;
    FLastValidLine: Integer;
    FLinesInWindow: Integer;
    FMarginColor: TColor;
    FMaxRightPos: Integer;
    FModified: Boolean;
    FMultiClicked: Boolean;
    FOffsetX,
    FOffsetY: Integer;
    FOnBookmarkChange: TBookmarkChangeEvent;
    FOnCaretChange: TCaretEvent;
    FOnGutterMouseDown: TGutterMouseDownEvent;
    FOnPaint: TPaintEvent;
    FOnProcessCommand: TProcessCommandEvent;
    FOnProcessUserCommand: TProcessCommandEvent;
    FOnReplaceText: TReplaceTextEvent;
    // Miscellanous events
    FOnScroll: TUCEScrollEvent;                   // Called when one or both window offsets changed.
    FOnSettingChange: TNotifyEvent;
    FOptions: TUniCodeEditOptions;
    FOverwriteCaret: TCaretType;
    FRightMargin: Integer;
    FScrollBars: TScrollStyle;
    FScrollHint: TScrollHintWindow;
    FScrollTimer: TTimer;
    FSelectedColor,
    FScrollHintColor: TUCELineStyle;
    FTabSize: Integer;
    FTextHeight: Integer;
    FUndoList: TUndoList;
    FUpdateCount: Integer;
    procedure CalcCharWidths(Index: Integer); overload;
    procedure CalcCharWidths(const S: WideString; TabSize, CharWidth: Cardinal); overload;
    function CaretXPix: Integer;
    function CaretYPix: Integer;
    function CharIndexToColumn(X: Cardinal): Cardinal; overload;
    function CharIndexToColumn(X: Cardinal; const S: WideString; TabSize, CharWidth: Cardinal): Cardinal; overload;
    function CharIndexToColumn(Pos: TPoint): Cardinal; overload;
    function ColumnToCharIndex(X: Cardinal): Cardinal; overload;
    function ColumnToCharIndex(Pos: TPoint): Cardinal; overload;
    procedure ComputeCaret(X, Y: Integer);
    function CopyOnDrop(Source: TObject): Boolean;
    procedure FontChanged(Sender: TObject);
    function GetBlockBegin: TPoint;
    function GetBlockEnd: TPoint;
    function GetCanRedo: Boolean;
    function GetCanUndo: Boolean;
    function GetCaretXY: TPoint;
    function GetFont: TFont;
    function GetLineEnd: Cardinal; overload;
    function GetLineEnd(Index: Cardinal): Cardinal; overload;
    function GetLineText: WideString;
    function GetMaxRightChar: Integer;
    function GetMaxUndo: Integer;
    function GetSelectedText: WideString;
    function GetSelectionAvailable: Boolean;
    function GetSelEnd: Integer;
    function GetSelStart: Integer;
    function GetText: WideString;
    function GetTopLine: Integer;
    function LeftSpaces(const S: WideString): Cardinal;
    procedure LineNumberFontChanged(Sender: TObject);
    function NextCharPos(ThisPosition: Cardinal; ForceNonDefault: Boolean = False): Cardinal; overload;
    function NextCharPos(ThisPosition, Line: Cardinal; ForceNonDefault: Boolean = False): Cardinal; overload;
    function NextTabPos(Index: Integer): Integer;
    procedure OnScrollTimer(Sender: TObject);
    function PosInSelection(Pos: TPoint): Boolean;
    function PreviousCharPos(ThisPosition: Cardinal): Cardinal; overload;
    function PreviousCharPos(ThisPosition, Line: Cardinal): Cardinal; overload;
    function RecordState(IncludeText: Boolean): TEditState;
    procedure ReplaceLeftTabs(var S: WideString);
    procedure ScanFrom(Index: Integer);
    procedure SetBlockBegin(Value: TPoint);
    procedure SetBlockEnd(Value: TPoint);
    procedure SetBorderStyle(Value: TBorderStyle);
    procedure SetCaretX(Value: Integer);
    procedure SetCaretXY(const Value: TPoint);
    procedure SetCaretY(Value: Integer);
    procedure SetCharWidth(const Value: Integer);
    procedure SetExtraLineSpacing(const Value: Integer);
    procedure SetFont(const Value: TFont);
    procedure SetGutterColor(Value: TColor);
    procedure SetGutterWidth(Value: Integer);
    procedure SetHighlighter(const Value: TUCEHighlighter);
    procedure SetIndentSize(Value: integer);
    procedure SetInsertCaret(const Value: TCaretType);
    procedure SetKeystrokes(const Value: TKeyStrokes);
    procedure SetLineNumberFont(const Value: TFont);
    procedure SetLineText(Value: WideString);
    procedure SetMarginColor(const Value: TColor);
    procedure SetMaxRightChar(const Value: Integer);
    procedure SetMaxUndo(const Value: Integer);
    procedure SetModified(const Value: Boolean);
    procedure SetOffsetX(Value: Integer);
    procedure SetOffsetY(Value: Integer);
    procedure SetOptions(const Value: TUniCodeEditOptions);
    procedure SetOverwriteCaret(const Value: TCaretType);
    procedure SetRightMargin(Value: Integer);
    procedure SetScrollBars(const Value: TScrollStyle);
    procedure SetScrollHintColor(const Value: TUCELineStyle);
    procedure SetSelectedColor(const Value: TUCELineStyle);
    procedure SetSelText(const Value: WideString);
    procedure SetSelTextExternal(const Value: WideString);
    procedure SetTabSize(Value: Integer);
    procedure SetText(const Value: WideString);
    procedure SetTopLine(Value: Integer);
    procedure SetUpdateState(Updating: Boolean);
    procedure SetWordBlock(Value: TPoint);
    procedure TripleClick;
    function Unindent(X, Y: Cardinal): Cardinal;
    procedure UpdateCaret;
    procedure UpdateScrollBars;
    procedure CMMouseWheel(var Message: TCMMouseWheel); message CM_MOUSEWHEEL;
    procedure CMSysColorChange(var Message: TMessage); message CM_SYSCOLORCHANGE;
    procedure WMCopy(var Message: TWMCopy); message WM_COPY;
    procedure WMCut(var Message: TWMCut); message WM_CUT;
    procedure WMEraseBkgnd(var Message: TMessage); message WM_ERASEBKGND;
    procedure WMGetDlgCode(var Msg: TWMGetDlgCode); message WM_GETDLGCODE;
    procedure WMHScroll(var Message: TWMScroll); message WM_HSCROLL;
    procedure WMImeComposition(var Message: TMessage); message WM_IME_COMPOSITION;
    procedure WMImeNotify(var Message: TMessage); message WM_IME_NOTIFY;
    procedure WMKillFocus(var Message: TWMKillFocus); message WM_KILLFOCUS;
    procedure WMPaste(var Message: TWMPaste); message WM_PASTE;
    procedure WMSetCursor(var Message: TWMSetCursor); message WM_SETCURSOR;
    procedure WMSetFocus(var Message: TWMSetFocus); message WM_SETFOCUS;
    procedure WMSize(var Message: TWMSize); message WM_SIZE;
    procedure WMVScroll(var Message: TWMScroll); message WM_VSCROLL;
  protected
    procedure CreateParams(var Params: TCreateParams); override;
    procedure DeleteSelection(NeedRescan: Boolean);
    procedure DoCaretChange; virtual;
    procedure DoGutterMouseDown(Button: TMouseButton; Shift: TShiftState; X, Y, Line: Integer); virtual;
    procedure DoScroll(DeltaX, DeltaY: Integer); virtual;
    procedure DoSettingChanged; virtual;
    procedure DoStartDrag(var DragObject: TDragObject); override;
    procedure DragCanceled; override;
    procedure DragOver(Source: TObject; X, Y: Integer; State: TDragState; var Accept: Boolean); override;
    procedure HideCaret;
    procedure InitializeCaret;
    procedure InsertText(const Value: WideString);
    procedure InvalidateLine(Index: Integer);
    procedure InvalidateLines(Start, Stop: Integer);
    procedure InvalidateToBottom(Index: Integer);
    function IsIdentChar(const AChar: WideChar; IdChars: TIdentChars): Boolean;
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    procedure KeyPress(var Key: Char); override;
    procedure LinesChanged;
    procedure Loaded; override;
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
    procedure MouseMove(Shift: TShiftState; X, Y: Integer); override;
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
    function NextSmartTab(X, Y: Integer; SkipNonWhite: Boolean): Integer;
    procedure Paint; override;
    procedure PaintGutter(TextCanvas: TCanvas); virtual;
    procedure PaintHighlighted(TextCanvas: TCanvas);
    procedure PaintText(TextCanvas: TCanvas); virtual;
    procedure ProcessCommand(var Command: TEditorCommand; var AChar: WideChar; Data: Pointer); virtual;
    // If the translations requires Data, memory will be allocated for it via a
    // GetMem call. The client must call FreeMem on Data if it is not nil.
    procedure ResetCaret; virtual;
    procedure ShowCaret;
    function TextFromClipboard: WideString; virtual;
    procedure TextToClipboard(Text: WideString); virtual;
    function TranslateKeyCode(Code: Word; Shift: TShiftState; var Data: Pointer): TEditorCommand;

    property BookMarkOptions: TBookmarkOptions read FBookmarkOptions write FBookmarkOptions;
    property BorderStyle: TBorderStyle read FBorderStyle write SetBorderStyle default bsSingle;
    property CharWidth: Integer read FCharWidth write SetCharWidth;
    property ExtraLineSpacing: Integer read FExtraLineSpacing write SetExtraLineSpacing default 0;
    property Font: TFont read GetFont write SetFont;
    property GutterColor: TColor read FGutterColor write SetGutterColor;
    property GutterWidth: Integer read FGutterWidth write SetGutterWidth;
    property HighLighter: TUCEHighlighter read FHighLighter write SetHighlighter;
    property IndentSize: Integer read FIndentSize write SetIndentSize default 2;
    property InsertCaret: TCaretType read FInsertCaret write SetInsertCaret default ctVerticalLine;
    property Keystrokes: TKeyStrokes read FKeyStrokes write SetKeystrokes;
    property LineNumberFont: TFont read FLineNumberFont write SetLineNumberFont;
    property MarginColor: TColor read FMarginColor write SetMarginColor default clSilver;
    property MaxRightChar: Integer read GetMaxRightChar write SetMaxRightChar;
    property MaxUndo: Integer read GetMaxUndo write SetMaxUndo;
    property OnBookmarkChange: TBookmarkChangeEvent read FOnBookmarkChange write FOnBookmarkChange;
    property OnCaretChange: TCaretEvent read FOnCaretChange write FOnCaretChange;
    property OnGutterMouseDown: TGutterMouseDownEvent read FOnGutterMouseDown write FOnGutterMouseDown;
    property OnPaint: TPaintEvent read FOnPaint write FOnPaint;
    property OnProcessCommand: TProcessCommandEvent read FOnProcessCommand write FOnProcessCommand;
    property OnProcessUserCommand: TProcessCommandEvent read FOnProcessUserCommand write FOnProcessUserCommand;
    property OnReplaceText: TReplaceTextEvent read FOnReplaceText write FOnReplaceText;
    property OnScroll: TUCEScrollEvent read FOnScroll write FOnScroll;
    property OnSettingChange: TNotifyEvent read FOnSettingChange write FOnSettingChange;
    property Options: TUniCodeEditOptions read FOptions write SetOptions default DefaultOptions;
    property OverwriteCaret: TCaretType read FOverwriteCaret write SetOverwriteCaret default ctBlock;
    property RightMargin: Integer read FRightMargin write SetRightMargin default 80;
    property ScrollBars: TScrollStyle read FScrollBars write SetScrollBars default ssBoth;
    property ScrollHintColor: TUCELineStyle read FScrollHintColor write SetScrollHintColor;
    property SelectedColor: TUCELineStyle read FSelectedColor write SetSelectedColor;
    property TabSize: Integer read FTabSize write SetTabSize default 8;
    property TabStop default True;
    property UpdateLock: Integer read FUpdateCount;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    
    procedure BeginUpdate;
    procedure BlockIndent;
    procedure BlockUnindent;
//    function ActivateKeyboard(const C: WideChar): Boolean;
    function CharPositionToRowColumn(Position: Cardinal): TPoint;
    procedure ClearAll(RegisterUndo, KeepUndoList: Boolean);
    procedure ClearSelection;
    procedure ClearUndo;
    procedure CommandProcessor(Command: TEditorCommand; AChar: WideChar; Data: Pointer); virtual;
    procedure CopyToClipboard;
    procedure CutToClipboard;
    procedure DragDrop(Source: TObject; X, Y: Integer); override;
    procedure EndUpdate;
    procedure EnsureCursorPosVisible;
    function GetBookMark(BookMark: Integer; var X, Y: Integer): Boolean;
    function GetWordAndBounds(Pos: TPoint; var BB, BE: TPoint): WideString;
    procedure GotoBookMark(BookMark: Integer);
    function IsBookmark(BookMark: Integer): Boolean;
    function LastWordPos: TPoint; virtual;
    function LineFromPos(Pos: TPoint): Integer;
    procedure LoadFromFile(const FileName: WideString; TextFormat: TTextFormat);
    function NextWordPos: TPoint; virtual;
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
    procedure PasteFromClipboard;
    function PositionFromPoint(X, Y: Integer): TPoint;
    procedure Redo;
    procedure RemoveBookmark(BookMark: Integer);
    function RowColumnToCharPosition(P: TPoint): Cardinal;
    procedure SaveToFile(const FileName: WideString; TextFormat: TTextFormat);
    function SearchReplace(const SearchText, ReplaceText: WideString; Options: TSearchOptions): Integer;
    procedure SelectAll;
    procedure SetBookMark(BookMark: Integer; X: Integer; Y: Integer);
    procedure SetCaretToEditorBottom;
    procedure SetDefaultKeystrokes; virtual;
    procedure SetSelEnd(const Value: Integer);
    procedure SetSelStart(const Value: Integer);
    procedure Undo;
    procedure WndProc(var Msg: TMessage); override;
    function WordAtPos(X, Y: Integer): WideString;
    property BlockBegin: TPoint read GetBlockBegin write SetBlockBegin;
    property BlockEnd: TPoint read GetBlockEnd write SetBlockEnd;
    property CanRedo: Boolean read GetCanRedo;
    property CanUndo: Boolean read GetCanUndo;
    property CaretX: Integer read FCaretX write SetCaretX;
    property CaretXY: TPoint read GetCaretXY write SetCaretXY;
    property CaretY: Integer read FCaretY write SetCaretY;
    property CharsInWindow: Integer read FCharsInWindow;
    property Content: TUniCodeEditorContent read FContent;
    property Data: TObject read FData write FData;
    property LineHeight: integer read FTextHeight;
    property LinesInWindow: Integer read FLinesInWindow;
    property LineText: WideString read GetLineText write SetLineText;
    property Modified: Boolean read FModified write SetModified;
    property OffsetX: Integer read FOffsetX write SetOffsetX;
    property OffsetY: Integer read FOffsetY write SetOffsetY;
    property SelectedText: WideString read GetSelectedText write SetSelTextExternal;
    property SelectionAvailable: Boolean read GetSelectionAvailable;
    property SelEnd: Integer read GetSelEnd write SetSelEnd;
    property SelStart: Integer read GetSelStart write SetSelStart;
    property Text: WideString read GetText write SetText;
    property TopLine: Integer read GetTopLine write SetTopLine;
    property UndoList: TUndoList read FUndoList;
  end;

  TUniCodeEdit = class(TCustomUniCodeEdit)
  published
    property Align;
    property Anchors;
    property Constraints;
    property BevelEdges;
    property BevelInner;
    property BevelOuter;
    property BevelKind;
    property BevelWidth;
    property BorderWidth;
    property BorderStyle;
    property BookMarkOptions;
    property CharWidth;
    property Color;
    property Ctl3D;
    property Enabled;
    property ExtraLineSpacing;
    property Font;
    property GutterColor;
    property GutterWidth;
    property Height;
    property HighLighter;
    property IndentSize;
    property InsertCaret;
    property Keystrokes;
    property LineNumberFont;
    property MarginColor;
    property MaxRightChar;
    property MaxUndo;
    property Name;
    property Options;
    property OverwriteCaret;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property RightMargin;
    property ShowHint;
    property ScrollBars;
    property ScrollHintColor;
    property SelectedColor;
    property TabOrder;
    property TabSize;
    property TabStop default True;
    property Tag;
    property Visible;
    property Width;

    property OnBookmarkChange;
    property OnCaretChange;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnGutterMouseDown;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnPaint;
    property OnProcessCommand;
    property OnProcessUserCommand;
    property OnReplaceText;
    property OnScroll;
    property OnSettingChange;
    property OnStartDrag;
  end;

//----------------------------------------------------------------------------------------------------------------------

implementation

uses
  {$ifdef COMPILER_6_UP}
    RtlConsts,
  {$else}
    Consts,
  {$endif COMPILER_6_UP}
  CommCtrl, ImgList, Dialogs, IMM, StringContainers;

{$R UniCodeEditor.res}

resourcestring
  SGeneralIMEError = 'General IME error encountered. No specific error info is available, though.';

var
  PlatformIsUnicode: Boolean;
  
//----------------------------------------------------------------------------------------------------------------------

procedure AssignError(ClassName: string; Source: TObject);

var
  SourceName: string;
  
begin
  if Assigned(Source) then
    SourceName := Source.ClassName
  else
    SourceName := 'nil';
  raise EConvertError.CreateResFmt(@SAssignError, [SourceName, ClassName]);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure ConvertAndAddImages(IL: TImageList);

// Loads the internal bookmark image list.

var
  OneImage: TBitmap;
  I: Integer;
  MaskColor: TColor;
  Name: string;

begin
  // Since we want our bookmark images looking in the correct system colors,
  // we have to remap their colors. I really would like to use
  // IL.GetInstRes(HInstance, rtBitmap, Resname, 16, [lrMap3DColors], clWhite) for this task,
  // but could not get it working under NT. It seems that image lists are not very well
  // supported under WinNT 4.
  OneImage := TBitmap.Create;
  try
    // it is assumed that the image height determines also the width of
    // one entry in the image list
    IL.Clear;
    IL.Height := 16;
    IL.Width := 16;
    for I := 0 to 9 do
    begin
      Name := 'MARK_' + IntToStr(I);
      OneImage.Handle := CreateGrayMappedRes(HInstance, PChar(Name));
      MaskColor := OneImage.Canvas.Pixels[0, 0];
      IL.AddMasked(OneImage, MaskColor);
    end;
  finally
    OneImage.Free;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

function Min(X, Y: Integer): Integer;

begin
  if X < Y then
    Result := X
  else
    Result := Y;
end;

//----------------------------------------------------------------------------------------------------------------------

function MaxPoint(P1, P2: TPoint): TPoint;

begin
  if P1.Y > P2.Y then
    Result := P1
  else
    if P1.Y < P2.Y then
      Result := P2
    else
      if P1.X > P2.X then
        Result := P1
      else
        Result := P2;
end;

//----------------------------------------------------------------------------------------------------------------------

function MaxPointAsReference(var P1, P2: TPoint): PPoint;

begin
  if P1.Y > P2.Y then
    Result := @P1
  else
    if P1.Y < P2.Y then
      Result := @P2
    else
      if P1.X > P2.X then
        Result := @P1
      else
        Result := @P2;
end;

//----------------------------------------------------------------------------------------------------------------------

function MinPoint(P1, P2: TPoint): TPoint;

begin
  if P1.Y < P2.Y then
    Result := P1
  else
    if P1.Y > P2.Y then
      Result := P2
    else
      if P1.X < P2.X then
        Result := P1
      else
        Result := P2;
end;

//----------------------------------------------------------------------------------------------------------------------

function MinPointAsReference(var P1, P2: TPoint): PPoint;

begin
  if P1.Y < P2.Y then
    Result := @P1
  else
    if P1.Y > P2.Y then
      Result := @P2
    else
      if P1.X < P2.X then
        Result := @P1
      else
        Result := @P2;
end;

//----------------------------------------------------------------------------------------------------------------------

function PointsAreEqual(P1, P2: TPoint): Boolean;

begin
  Result := (P1.X = P2.X) and (P1.Y = P2.Y);
end;

//----------------- TUndoList ------------------------------------------------------------------------------------------

constructor TUndoList.Create(AOwner: TCustomUniCodeEdit);

begin
  inherited Create;
  FOwner := AOwner;
  FList := TList.Create;
  FMaxUndo := 10;
  FCurrent := -1;
end;

//----------------------------------------------------------------------------------------------------------------------

destructor TUndoList.Destroy;

begin
  ClearList;
  FList.Free;
  inherited Destroy;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TUndoList.AddChange(AReason: TChangeReason; const AOldText: WideString; AOldCaret, AOldSelStart,
  AOldSelEnd: PPoint; const ANewText: WideString; ANewCaret, ANewSelStart, ANewSelEnd: PPoint);

// adds a new change into the undo list;
// The parameter list looks rather messy, but should serve all possible combination
// which can appear. By using pointers instead of just the records we can pass
// nil values where we don't need a value in the undo list. This is particularly useful
// to avoid creating dummy records just to fill the paramter list.
// Note: There's an overloaded funtion variant to avoid passing in the current
//       edit state (text, caret, selection).

var
  NewChange: PChange;

begin
  if FMaxUndo > 0 then
  begin
    if FList.Count >= FMaxUndo then
      RemoveChange(0);

    // a new undo record deletes all redo records if there are any
    while FCurrent < FList.Count - 1 do
      RemoveChange(FList.Count - 1);

    // add a new entry
    New(NewChange);
    FillChar(NewChange^, SizeOf(NewChange^), 0);
    with NewChange^ do
    begin
      Reason := AReason;
      OldText := AOldText;
      if Assigned(AOldCaret) then
        OldCaret := AOldCaret^;
      if Assigned(AOldSelStart) then
        OldSelStart := AOldSelStart^;
      if Assigned(AOldSelEnd) then
        OldSelEnd := AOldSelEnd^;
      NewText := ANewText;
      if Assigned(ANewCaret) then
        NewCaret := ANewCaret^;
      if Assigned(ANewSelStart) then
        NewSelStart := ANewSelStart^;
      if Assigned(ANewSelEnd) then
        NewSelEnd := ANewSelEnd^;
    end;
    FList.Add(NewChange);
    // advance current undo index
    Inc(FCurrent);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TUndoList.AddChange(AReason: TChangeReason; const AOldText: WideString; AOldCaret, AOldSelStart, AOldSelEnd:
  PPoint);

// adds a new change into the undo list, all new values are taken from current state
// Note: There's an overloaded function variant to allow passing other New* values
//       than the ones of the editor.

var
  NewChange: PChange;

begin
  if FMaxUndo > 0 then
  begin
    if FList.Count >= FMaxUndo then
      RemoveChange(0);

    // a new undo record deletes all redo records if there are any
    while FCurrent < FList.Count - 1 do
      RemoveChange(FList.Count - 1);

    // add a new entry
    New(NewChange);
    FillChar(NewChange^, SizeOf(NewChange^), 0);
    with NewChange^ do
    begin
      Reason := AReason;
      OldText := AOldText;
      if Assigned(AOldCaret) then
        OldCaret := AOldCaret^;
      if Assigned(AOldSelStart) then
        OldSelStart := AOldSelStart^;
      if Assigned(AOldSelEnd) then
        OldSelEnd := AOldSelEnd^;
      with FOwner do
      begin
        NewText := GetSelectedText;
        NewCaret := CaretXY;
        NewSelStart := FBlockBegin;
        NewSelEnd := FBlockEnd;
      end;
    end;
    FList.Add(NewChange);
    // advance current undo index
    Inc(FCurrent);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

function TUndoList.GetUndoChange(var Change: PChange): TChangeReason;

begin
  if FCurrent = -1 then
    Result := ecrNone
  else
  begin
    Change := FList.Items[FCurrent];
    Result := Change.Reason;
    Dec(FCurrent);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

function TUndoList.GetRedoChange(var Change: PChange): TChangeReason;

begin
  if FCurrent = FList.Count - 1 then
    Result := ecrNone
  else
  begin
    Inc(FCurrent);
    Change := FList.Items[FCurrent];
    Result := Change.Reason;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

function TUndoList.GetCanRedo: Boolean;

begin
  Result := FCurrent < FList.Count - 1;
end;

//----------------------------------------------------------------------------------------------------------------------

function TUndoList.GetCanUndo: Boolean;

begin
  Result := FCurrent > -1;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TUndoList.SetMaxUndo(Value: Integer);

begin
  if Value > 32767 then
    Value := 32767;
  if FMaxUndo <> Value then
  begin
    FMaxUndo := Value;
    // FCurrent ist automatically updated
    while FMaxUndo < FList.Count - 1 do
      RemoveChange(0);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TUndoList.RemoveChange(Index: Integer);

var
  AChange: PChange;

begin
  if (Index > -1) and (Index < FList.Count) then
  begin
    AChange := FList.Items[Index];
    Dispose(AChange);
    FList.Delete(Index);
  end;
  if FCurrent > FList.Count - 1 then
    FCurrent := FList.Count - 1;
end;

//----------------------------------------------------------------------------------------------------------------------

function TUndoList.GetCurrentUndoReason: TChangeReason;

begin
  if FCurrent = -1 then
    Result := ecrNone
  else
    Result := PChange(FList.Items[FCurrent]).Reason;
end;

//----------------------------------------------------------------------------------------------------------------------

function TUndoList.GetCurrentRedoReason: TChangeReason;

begin
  if FCurrent = FList.Count - 1 then
    Result := ecrNone
  else
    Result := PChange(FList.Items[FCurrent + 1]).Reason;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TUndoList.ClearList;

var
  I: Integer;

begin
  for I := FList.Count - 1 downto 0 do
    RemoveChange(I);
end;

//----------------- TBookmarkOptions -----------------------------------------------------------------------------------

constructor TBookmarkOptions.Create(AOwner: TCustomUniCodeEdit);

begin
  inherited Create;
  FOwner := AOwner;
  FEnableKeys := True;
  FGlyphsVisible := True;
  FLeftMargin := 2;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TBookmarkOptions.SetGlyphsVisible(Value: Boolean);

begin
  if FGlyphsVisible <> Value then
  begin
    FGlyphsVisible := Value;
    FOwner.Invalidate;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TBookmarkOptions.SetLeftMargin(Value: Integer);

begin
  if FLeftMargin <> Value then
  begin
    FLeftMargin := Value;
    FOwner.Invalidate;
  end;
end;

//----------------- TUCELineStyle --------------------------------------------------------------------------------------

function TUCELineStyle.GetBackground: TColor;

begin
  Result := FBackground;
end;

//----------------------------------------------------------------------------------------------------------------------

function TUCELineStyle.GetFontStyles: TFontStyles;

begin
  Result := FFontStyles;
end;

//----------------------------------------------------------------------------------------------------------------------

function TUCELineStyle.GetForceFontStyles: Boolean;

begin
  Result := FForceFontStyles;
end;

//----------------------------------------------------------------------------------------------------------------------

function TUCELineStyle.GetForeground: TColor;

begin
  Result := FForeground;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TUCELineStyle.SetBackground(const Color: TColor);

begin
  FBackground := Color;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TUCELineStyle.SetFontStyles(const Styles: TFontStyles);

begin
  FFontStyles := Styles;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TUCELineStyle.SetForceFontStyles(Force: Boolean);

begin
  FForceFontStyles := Force;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TUCELineStyle.SetForeground(const Color: TColor);

begin
  FForeground := Color;
end;

//----------------- TUCELine -------------------------------------------------------------------------------------------

constructor TUCELine.Create(Owner: TUniCodeEditorContent);

begin
  FContent := Owner;
  
  // Markers and styles member are created on demand.
end;

//----------------------------------------------------------------------------------------------------------------------

destructor TUCELine.Destroy;

begin
  FMarkers.Free;
  FStyles.Free;

  inherited;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TUCELine.SetText(const Value: WideString);

begin
  FText := Value;
  FContent.DoChangeLine(Self);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TUCELine.Changed;

begin
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TUCELine.DrawMarkers(Index: Integer; Canvas: TCanvas; X, Y: Integer);

var
  I: Integer;
  Marker: IUCELineMarker;
  Size: TSize;

begin
  if Assigned(FMarkers) then
    for I := 0 to FMarkers.Count - 1 do
    begin
      Marker := IUCELineMarker(FMarkers[I]);
      Marker.Draw(Index, Canvas, X, Y);
      Size := Marker.GetSize(Index);
      Inc(X, Size.cx);
    end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TUCELine.Assign(Source: TUCELine);

var
  I: Integer;
  Index: Integer;
  
begin
  FLexerState := Source.FLexerState;
  FText := Source.FText;
  FStates := Source.FStates;
  FBounds := Source.FBounds;
  for I := 0 to Source.FMarkers.Count - 1 do
    FMarkers.Add(Source.FMarkers[I]);
  for I := 0 to Source.FStyles.Count - 1 do
  begin
    Index := FStyles.List.Add(Source.FStyles.List[I]);
    IUCELineStyle(FStyles.List[Index])._AddRef;
  end;
  FData := Source.FData;
end;

//----------------------------------------------------------------------------------------------------------------------

function TUCELine.AddMarker(Marker: IUCELineMarker): Integer;

// Adds a new marker at the end of the list of markers for this line.

begin
  if Assigned(Marker) then
  begin
    if FMarkers = nil then
      FMarkers := TInterfaceList.Create;

    // If the given marker already is in the list then remove it first.
    // This way you can simply move it to the end of the list by re-adding it.
    if FMarkers.IndexOf(Marker) > -1 then
      FMarkers.Remove(Marker)
    else
      Marker._AddRef;
    Result := FMarkers.Add(Marker);
    Changed;
  end
  else
    Result := -1;
end;

//----------------------------------------------------------------------------------------------------------------------

function TUCELine.HasMarker(Marker: IUCELineMarker): Boolean;

begin
  Result := Assigned(FMarkers) and (FMarkers.IndexOf(Marker) > -1);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TUCELine.RemoveMarker(Marker: IUCELineMarker);

// Removes the given marker from the list of markers for this line.

var
  Index: Integer;

begin
  if Assigned(FMarkers) then
  begin
    Index := FMarkers.IndexOf(Marker);
    if Index > -1 then
    begin
      FMarkers.Remove(Marker);

      if FMarkers.Count = 0 then
        FreeAndNil(FMarkers);
      Changed;
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TUCELine.ReplaceMarker(OldMarker, NewMarker: IUCELineMarker);

// Removes the given marker from the list of markers for this line.

var
  Index: Integer;

begin
  if Assigned(FMarkers) then
  begin
    Index := FMarkers.IndexOf(OldMarker);
    if Index > -1 then
    begin
      FMarkers[Index] := NewMarker;

      Changed;
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

function TUCELine.PeekStyle: IUCELineStyle;

begin
  if Assigned(FStyles) then
    Result := IUCELineStyle(FStyles.Peek)
  else
    Result := nil;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TUCELine.PopStyle;

// Removes the top entry of the style stack.

var
  Style: IUCELineStyle;

begin
  if Assigned(FStyles) then
  begin
    // When leaving the method this inteface is released once.
    Pointer(Style) := FStyles.Pop;

    if FStyles.Count = 0 then
      FreeAndNil(FStyles);
    Changed;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TUCELine.PushStyle(Style: IUCELineStyle);

// Adds a new style to the style stack for this line. This new style becomes the primary style until it is removed.

begin
  if FStyles = nil then
    FStyles := TStyleStack.Create;

  Style._AddRef; // Need to addref explicitely as we now only handle the interface as pointer.
  FStyles.Push(Pointer(Style));
  Changed;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TUCELine.RemoveStyle(Style: IUCELineStyle);

var
  Index: Integer;
  
begin
  if Assigned(FStyles) then
  begin
    Index := FStyles.List.IndexOf(Pointer(Style));
    if Index > -1 then
    begin
      FStyles.List.Remove(Pointer(Style));
      Style._Release;
      if FStyles.Count = 0 then
        FreeAndNil(FStyles);

      Changed;
    end;
  end;
end;

//----------------- TUniCodeEditorContent ------------------------------------------------------------------------------

constructor TUniCodeEditorContent.Create(AOwner: TCustomUniCodeEdit);

begin
  inherited Create;

  FOwner := AOwner;
end;

//----------------------------------------------------------------------------------------------------------------------

destructor TUniCodeEditorContent.Destroy;

begin
  Clear;
  
  inherited;
end;

//----------------------------------------------------------------------------------------------------------------------

function TUniCodeEditorContent.GetLine(Index: Integer): TUCELine;

begin
  Result := FLines[Index];
  if Assigned(Result) and not (lsValidated in Result.FStates) then
    DoValidateLine(Result);
end;

//----------------------------------------------------------------------------------------------------------------------

function TUniCodeEditorContent.GetLineNoInit(Index: Integer): TUCELine;

begin
  Result := FLines[Index];
end;

//----------------------------------------------------------------------------------------------------------------------

function TUniCodeEditorContent.GetText: WideString;

var
  I: Integer;
  Buffer: TBufferedString;

begin
  Buffer := TBufferedString.Create;
  try
    for I := 0 to FCount - 1 do
    begin
      Buffer.Add(FLines[I].FText);
      if I < FCount - 1 then
        Buffer.AddNewLine;
    end;

    Result := Buffer.AsString;
  finally
    Buffer.Free;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TUniCodeEditorContent.Grow;

var
  Delta,
  Len: Integer;

begin
  Len := Length(FLines);
  if Len > 64 then
    Delta := Len div 4
  else
  begin
    if Len > 8 then
      Delta := 16
    else
      Delta := 4;
  end;
  SetCapacity(Len + Delta);
end;

//----------------------------------------------------------------------------------------------------------------------

function TUniCodeEditorContent.AddLine(const Text: WideString): TUCELine;

begin
  Result := InsertLine(FCount, Text);
end;

//----------------------------------------------------------------------------------------------------------------------

function TUniCodeEditorContent.AddLine(const Line: TUCELine): TUCELine;

begin
  Result := InsertLine(FCount, Text);
  Result.Assign(Line);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TUniCodeEditorContent.AddStrings(const Strings: TStrings);

var
  I: Integer;

begin
  for I := 0 to Strings.Count - 1 do
    AddLine(Strings[I]);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TUniCodeEditorContent.AddStrings(const Strings: TWideStrings);

var
  I: Integer;

begin
  for I := 0 to Strings.Count - 1 do
    AddLine(Strings[I]);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TUniCodeEditorContent.AssignTo(Destination: TObject);

var
  I: Integer;
  WStrings: TUniCodeEditorContent;
  AStrings: TStrings;

begin
  if Destination is TUniCodeEditorContent then
  begin
    WStrings := Destination as TUniCodeEditorContent;
    WStrings.Clear;
    for I := 0 to Count - 1 do
      WStrings.AddLine(FLines[I]);
  end
  else
    if Destination is TStrings then
    begin
      AStrings := Destination as TStrings;
      AStrings.Clear;
      for I := 0 to Count - 1 do
        AStrings.Add(FLines[I].FText);
    end
    else
      AssignError(ClassName, Destination);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TUniCodeEditorContent.Clear;

var
  I: Integer;

begin
  for I := 0 to FCount - 1 do
  begin
    DoDeleteLine(FLines[I]);
    FLines[I].Free;
  end;
  SetCapacity(0);

  if not (csDestroying in FOwner.ComponentState) then
    FOwner.ResetCaret;
end;

//----------------------------------------------------------------------------------------------------------------------

function TUniCodeEditorContent.CollectText(Start, Stop: TPoint): WideString;

// Collects text off of a range of lines given by Start and Stop.
// This method is stable agains out-of-range values and returns as much as possible text that can be retrieved.
// The given coordinates must be character indexes and are all 0-based.

var
  Temp: TPoint;
  Buffer: TBufferedString;

begin
  Buffer := TBufferedString.Create;

  try
    // First do a couple of sanity checks.
    if Start.Y < 0 then
      Start.Y := 0;
    if Start.Y > FCount - 1 then
      Start.Y := FCount - 1;
    if Stop.Y < 0 then
      Stop.Y := 0;
    if Stop.Y > FCount - 1 then
      Stop.Y := FCount - 1;
    if Start.Y > Stop.Y then
    begin
      // Swap start and end point if they are in wrong order.
      Temp := Start;
      Start := Stop;
      Stop := Temp;
    end;

    if Start.X < 0 then
      Start.X := 0;
    if Stop.X < 0 then
      Stop.X := 0;

    if (Start.Y > -1) and (Stop.Y > -1) then
    begin
      if Start.Y = Stop.Y then
      begin
        if Start.X > Stop.X then
        begin
          Temp.X := Start.X;
          Start.X := Stop.X;
          Stop.X := Temp.X;
        end;
        Buffer.Add(Copy(FLines[Start.Y].Text, Start.X + 1, Stop.X - Start.X + 1));
      end
      else
      begin
        Buffer.Add(Copy(FLines[Start.Y].Text, Start.X + 1, Length(FLines[Start.Y].Text)));
        Buffer.AddNewLine;
        Inc(Start.Y);
        while Start.Y < Stop.Y do
        begin
          Buffer.Add(FLines[Start.Y].Text);
          Buffer.AddNewLine;
          Inc(Start.Y);
        end;
        Buffer.Add(Copy(FLines[Stop.Y].Text, 1, Stop.X + 1));
      end;
    end;
    Result := Buffer.AsString;
  finally
    Buffer.Free;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TUniCodeEditorContent.DeleteLine(Index: Integer);

begin
  if (Index < 0) or (Index >= FCount) then
    Error(SListIndexError, Index);

  DoDeleteLine(FLines[Index]);
  FLines[Index].Free;
  Dec(FCount);
  if Index < FCount then
  begin
    System.Move(FLines[Index + 1], FLines[Index], (FCount - Index) * SizeOf(TUCELine));
    FLines[FCount] := nil;
  end;

  // Reindex all following lines.
  while Index  < FCount do
  begin
    Dec(FLines[Index].FIndex);
    Inc(Index);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TUniCodeEditorContent.Error(const Msg: string; Data: Integer);

  //--------------- local function --------------------------------------------

  function ReturnAddr: Pointer;

  asm
          MOV EAX, [EBP + 4]
  end;

  //--------------- end local function ----------------------------------------

begin
  raise EStringListError.CreateFmt(Msg, [Data])at ReturnAddr;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TUniCodeEditorContent.Exchange(Index1, Index2: Integer);

var
  Temp: TUCELine;

begin
  if (Index1 < 0) or (Index1 >= FCount) then
    Error(SListIndexError, Index1);
  if (Index2 < 0) or (Index2 >= FCount) then
    Error(SListIndexError, Index2);

  Temp := FLines[Index1];
  FLines[Index1] := FLines[Index2];
  FLines[Index2] := Temp;

  FLines[Index1].FIndex := Index1;
  FLines[Index2].FIndex := Index2;

  DoChangeLine(FLines[Index1]);
  DoChangeLine(FLines[Index2]);
end;

//----------------------------------------------------------------------------------------------------------------------

function TUniCodeEditorContent.InsertLine(Index: Integer; const Text: WideString): TUCELine;

begin
  if FCount = Length(FLines) then
    Grow;
  if Index < FCount then
    System.Move(FLines[Index], FLines[Index + 1], (FCount - Index) * SizeOf(TUCELine));

  Inc(FCount);
  FLines[Index] := TUCELine.Create(Self);
  FLines[Index].FText := Text;
  FLines[Index].FIndex := Index;

  Result := FLines[Index];

  // Reindex all following lines.
  Inc(Index);
  while Index  < FCount do
  begin
    Inc(FLines[Index].FIndex);
    Inc(Index);
  end;

  DoChangeLine(Result);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TUniCodeEditorContent.DoChangeLine(Line: TUCELine);

begin
  FChanged := True;
  if Line.Index < FOwner.FLastValidLine then
    FOwner.ScanFrom(Line.Index);
  if Assigned(FOnChangeLine) then
    FOnChangeLine(Self, Line);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TUniCodeEditorContent.DoDeleteLine(Line: TUCELine);

begin
  if Assigned(FOnDeleteLine) then
    FOnDeleteLine(Self, Line);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TUniCodeEditorContent.DoValidateLine(Line: TUCELine);

begin
  if Assigned(FOnValidateLine) then
    FOnValidateLine(Self, Line);
  Include(Line.FStates, lsValidated);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TUniCodeEditorContent.SetCapacity(NewCapacity: Integer);

begin
  SetLength(FLines, NewCapacity);
  if NewCapacity < FCount then
    FCount := NewCapacity;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TUniCodeEditorContent.LoadFromStream(Stream: TStream; Format: TTextFormat; Language: LCID = 0);

// Replaces the current content with the text from the given stream.
// Format tells what is in the stream. If this is tfANSI you also must set Language to a value that
// describes the content of the file.

var
  SW: WideString;
  SA: string;
  Size: Integer;
  CodePage: Integer;

begin
  case Format of
    tfANSI,
    tfUTF8:
      begin
        Size := Stream.Size - Stream.Position;
        SetLength(SA, Size);
        Stream.ReadBuffer(PChar(SA)^, Size);
        if Format = tfANSI then
        begin
          // Check if we actually have UTF 8 here, despite the user said if would be ANSI.
          if CompareMem(@BOM_UTF8, PChar(SA), SizeOf(BOM_UTF8)) then
            CodePage := CP_UTF8
          else
            if Language = 0 then
              CodePage := CP_ACP
            else
              CodePage := CodePageFromLocale(Language);
        end
        else
          CodePage := CP_UTF8;

        SW := StringToWideStringEx(SA, CodePage);
        SetText(SW);
      end;
    tfUTF16:
      begin
        Size := Stream.Size - Stream.Position;
        SetLength(SW, Size);
        Stream.ReadBuffer(PWideChar(SW)^, 2 * Size);
        SetText(SW);
      end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TUniCodeEditorContent.RevalidateLine(Index: Integer);

begin
  Exclude(FLines[Index].FStates, lsValidated);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TUniCodeEditorContent.RevalidateLine(Line: TUCELine);

begin
  Exclude(Line.FStates, lsValidated);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TUniCodeEditorContent.SaveToStream(Stream: TStream; Format: TTextFormat; Language: LCID = 0;
  WithBOM: Boolean = True);

// Saves the currently loaded text into the given stream. WithBOM determines whether to write a
// byte order mark or not.
// Note: when saved as ANSI text there will never be a BOM.

const
  ANSINewline: AnsiString = #13#10;
  
var
  BOM: WideString;
  SA: string;
  I: Integer;
  CodePage: Integer;

begin
  case Format of
    tfANSI:
      begin
        // ANSI format does not have a BOM, so none is written out.
        if Language = 0 then
          CodePage := CP_ACP
        else
          CodePage := CodePageFromLocale(Language);
        for I := 0 to FCount - 1 do
        begin
          SA := WideStringToStringEx(FLines[I].Text, CodePage);
          Stream.WriteBuffer(PChar(SA)^, Length(SA));
          if I < FCount - 1 then
            Stream.WriteBuffer(PChar(ANSINewline)^, Length(ANSINewline));
        end;
      end;
    tfUTF8:
      begin
        if WithBOM then
        begin
          BOM := BOM_LSB_FIRST;
          SA := WideStringToStringEx(BOM, CP_UTF8);
          Stream.WriteBuffer(PChar(SA)^, Length(SA));
        end;
        for I := 0 to FCount - 1 do
        begin
          SA := WideStringToStringEx(FLines[I].Text, CP_UTF8);
          Stream.WriteBuffer(PChar(SA)^, Length(SA));
          if I < FCount - 1 then
            Stream.WriteBuffer(PChar(ANSINewline)^, Length(ANSINewline));
        end;
      end;
    tfUTF16:
      begin
        if WithBOM then
        begin
          BOM := BOM_LSB_FIRST;
          Stream.WriteBuffer(PWideChar(BOM)^, Length(BOM));
        end;
        for I := 0 to FCount - 1 do
        begin
          Stream.WriteBuffer(PWideChar(FLines[I].FText)^, 2 * Length(FLines[I].FText));
          if I < FCount - 1 then
            Stream.WriteBuffer(PWideChar(WideCRLF)^, 2 * Length(WideCRLF));
        end;
      end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TUniCodeEditorContent.SetChanged(const Value: Boolean);

begin
  FChanged := Value;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TUniCodeEditorContent.SetText(const Value: WideString);

// Replaces the current content with the text given by Value. Eventual byte order marks are correctly handled.

var
  Head,
  Tail: PWideChar;
  S: WideString;

begin
  Clear;
  Head := PWideChar(Value);
  if (Head^ = BOM_LSB_FIRST) or (Head^ = BOM_MSB_FIRST) then
  begin
    // Consider byte order mark.
    if Head^ = BOM_MSB_FIRST then
      StrSwapByteOrder(Head);
    Inc(Head);
  end;
  while Head^ <> WideNull do
  begin
    Tail := Head;
    while not (Tail^ in [WideNull, WideLineFeed, WideCarriageReturn, WideVerticalTab, WideFormFeed]) and
      (Tail^ <> WideLineSeparator) and (Tail^ <> WideParagraphSeparator) do
      Inc(Tail);
    SetString(S, Head, Tail - Head);
    AddLine(S);
    Head := Tail;
    if Head^ <> WideNull then
    begin
      Inc(Head);
      if (Tail^ = WideCarriageReturn) and (Head^ = WideLineFeed) then
        Inc(Head);
    end;
  end;
end;

//----------------- TScrollHintWindow ----------------------------------------------------------------------------------

procedure TScrollHintWindow.Paint;

var
  R: TRect;

begin
  R := ClientRect;
  Inc(R.Left, 2);
  Inc(R.Top, 2);
  Canvas.Font := Font;
  DrawText(Canvas.Handle, PChar(Caption), -1, R, DT_LEFT or DT_NOPREFIX or DT_WORDBREAK or DrawTextBiDiModeFlagsReadingOnly);
end;

//----------------- TCustomUniCodeEdit ----------------------------------------------------------------------------------

constructor TCustomUniCodeEdit.Create(AOwner: TComponent);

var
  I: Integer;

begin
  inherited Create(AOwner);
  DoubleBuffered := False;
  FOptions := DefaultOptions;
  FContent := TUniCodeEditorContent.Create(Self);

  FFontDummy := TFont.Create;
  FLineNumberFont := TFont.Create;
  FLineNumberFont.Name := 'Terminal';
  FLineNumberFont.Size := 6;
  FLineNumberFont.Color := clBlack;
  FLineNumberFont.Style := [];
  FLineNumberFont.OnChange := LineNumberFontChanged;

  FUndoList := TUndoList.Create(Self);
  FSelectedColor := TUCELineStyle.Create;
  FScrollHintColor := TUCELineStyle.Create;
  FScrollHintColor.FBackground := clAppWorkSpace;
  FScrollHintColor.FForeground := clInfoText;
  FBookmarkOptions := TBookmarkOptions.Create(Self);

  FScrollTimer := TTimer.Create(Self);
  FScrollTimer.Enabled := False;
  FScrollTimer.Interval := 20;
  FScrollTimer.OnTimer := OnScrollTimer;

  FCharsInWindow := 10;
  // FRightMargin, FCharWidth and MaxRight have to be set before FontChanged
  // is called for the first time.
  FRightMargin := 80;
  // This char width value is only a dummy and will be updated later.
  FCharWidth := 8;
  MaxRightChar := 80;
  FMarginColor := clSilver;
  FGutterWidth := 30;
  ControlStyle := ControlStyle + [csOpaque, csSetCaption, csReflector, csNeedsBorderPaint];
  Height := 150;
  Width := 200;
  Cursor := crIBeam;
  Color := clWindow;
  FCaretX := 0;
  FCaretY := 0;
  FCaretOffset := Point(0, 0);
  FGutterColor := clBtnFace;
  Font.OnChange := FontChanged;
  FFontDummy.Name := 'Courier New';
  FFontDummy.Size := 10;
  Font.Assign(FFontDummy);
  ParentFont := False;
  ParentColor := False;
  TabStop := True;
  with FGutterRect do
  begin
    Left := 0;
    Top := 0;
    Right := FGutterWidth - 1;
    Bottom := Height;
  end;
  FTabSize := 8;
  MaxRightChar := 128;
  FUpdateCount := 0;
  FIndentSize := 2;
  FScrollBars := ssBoth;
  FBlockBegin := Point(0, 0);
  FBlockEnd := Point(0, 0);
  FInternalBMList := TImageList.Create(Self);
  ConvertAndAddImages(FInternalBMList);
  for I := 0 to 9 do
    FBookmarks[I].Y := -1;
  FBorderStyle := bsSingle;
  FInsertCaret := ctVerticalLine;
  FOverwriteCaret := ctBlock;
  FKeyStrokes := TKeyStrokes.Create(Self);

  // Retrive double click time to be used in triple clicks.
  // Make this time smaller to avoid triple clicks in cases of double clicks and drag'n drop start.
  FDoubleClickTime := GetDoubleClickTime;
  SetDefaultKeystrokes;

  FScrollHint := TScrollHintWindow.Create(Self);
  FScrollHint.Name := 'Scroll';
end;

//----------------------------------------------------------------------------------------------------------------------

destructor TCustomUniCodeEdit.Destroy;

begin
  FKeyStrokes.Free;
  FLineNumberFont.Free;
  FFontDummy.Free;
  FContent.Free;
  FSelectedColor.Free;
  FScrollHintColor.Free;
  FBookmarkOptions.Free;
  FScrollTimer.Free;
  FUndoList.Free;
  FCharWidthArray := nil;
  // scroll hint window is automatically destroyed
  inherited Destroy;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.BeginUpdate;

begin
  if FUpdateCount = 0 then
    SetUpdateState(True);
  Inc(FUpdateCount);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.BlockIndent;

// Inserts FIndentSize space characters before every line in the current selection.
// Note: A designed side effect is that all leading tab characters in the concerned
//       lines will be replaced by the proper number of spaces!

var
  I, J,
  Last: Integer;
  Line: WideString;
  BB, BE: PPoint;

  SelStart,
  SelEnd,
  OldCaret,
  NewCaret: TPoint;
  T1, T2: TWideStringList;

begin
  if SelectionAvailable then
  begin
    BeginUpdate;
    // By using references instead of a copy we can directly work on the selection
    // variables and have not to worry about whether the block begin is really
    // before the block end.
    BB := MinPointAsReference(FBlockBegin, FBLockEnd);
    BE := MaxPointAsReference(FBlockBegin, FBLockEnd);

    // exclude last selection line if it does not contain any selected chars
    if BE.X > 0 then
      Last := BE.Y
    else
      Last := BE.Y - 1;

    // save state for undo, this needs special care as we are going to replace
    // characters not in the current selection
    OldCaret := CaretXY;
    SelStart := BB^;
    SelEnd := BE^;

    T1 := TWideStringList.Create;
    T1.Capacity := Last - BB.Y + 1;
    T2 := TWideStringList.Create;
    T2.Capacity := Last - BB.Y + 1;

    for I := BB.Y to Last do
    begin
      Line := FContent[I].Text;
      // copy all lines (entirely) which are somehow touched by the current selection
      T1.Add(Line);
      ReplaceLeftTabs(Line);
      Insert(WideStringOfChar(' ', FIndentSize), Line, 1);
      FContent[I].Text := Line;
      // copy also all lines after they have been changed
      T2.Add(Line);

      // adjust cursor if it is at this particular line, this needs to be done only
      // once, hence I put it herein
      if FCaretY = I then
      begin
        J := GetLineEnd(I);
        // cursor beyond end of line?
        if FCaretX >= J then
          CaretX := J
        else
        begin
          // cursor in leading white space area?
          J := LeftSpaces(Line); // don't need TrimRightW here, as it has already been
                                 // verified that the cursor is before a non-white space char
          if FCaretX < J then
            CaretX := J
          else
            CaretX := FCaretX + FIndentSize;
        end;
      end
    end;

    // adjust selection appropriately (the Delphi IDE moves the selection start and
    // end to the first non-whitespace if there's no non-whitespace before the
    // particular X position, hence I'll do it too)
    Inc(BB.X, FIndentSize);
    Line := FContent[BB.Y].Text;
    I := LeftSpaces(Line);
    if I >= BB.X then
      BB.X := I;
    I := GetLineEnd(BB.Y);
    if I < BB.X then
      BB.X := I;

    if BE.X > 0 then
    begin
      Inc(BE.X, FIndentSize);
      Line := FContent[BE.Y].Text;
      I := LeftSpaces(Line);
      if I >= BE.X then
        BE.X := I;
      I := GetLineEnd(BE.Y);
      if I < BE.X then
        BE.X := I;
    end;

    // no further cursor adjustion, if the cusror is, by any means, not on any of
    // the lines in the current selection then it won't be updated at all
    NewCaret := CaretXY;
    FUndoList.AddChange(ecrIndentation, T1.Text, @OldCaret, @SelStart, @SelEnd,
      T2.Text, @NewCaret, @FBlockBegin, @FBlockEnd);

    T1.Free;
    T2.Free;
    EndUpdate;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.BlockUnindent;

// Removes first FIndentSize space characters of every line in the current selection
// or all leading white spaces if their count is less than FIndentSize.
// Note: A designed side effect is that all leading tab characters in the concerned
//       lines will be replaced by the proper number of spaces!

var
  I, J,
    Last: Integer;
  Line: WideString;
  BB, BE: PPoint;

  SelStart,
    SelEnd,
    OldCaret,
    NewCaret: TPoint;
  OldText,
    NewText: WideString;

begin
  if SelectionAvailable then
  begin
    BeginUpdate;

    // By using references instead of a copy we can directly work on the selection
    // variables and have not to worry about whether the block begin is really
    // before the block end.
    BB := MinPointAsReference(FBlockBegin, FBLockEnd);
    BE := MaxPointAsReference(FBlockBegin, FBLockEnd);

    // exclude last selection line if it does not contain any selected chars
    if BE.X > 0 then
      Last := BE.Y
    else
      Last := BE.Y - 1;

    // save state for undo, this needs special care as we are going to replace
    // characters not in the current selection
    OldText := '';
    NewText := '';
    OldCaret := CaretXY;
    SelStart := BB^;
    SelEnd := BE^;

    for I := BB.Y to Last do
    begin
      Line := FContent[I].Text;
      // copy all lines (entirely) which are somehow touched by the current selection
      OldText := OldText + Line + WideCRLF;
      ReplaceLeftTabs(Line);
      J := LeftSpaces(Line);
      if J >= FIndentSize then
        Delete(Line, 1, FIndentSize)
      else
        Delete(Line, 1, J);
      FContent[I].Text := Line;
      // copy also all lines after they have been changed
      NewText := NewText + Line + WideCRLF;

      // adjust cursor if it is at this particular line, this needs to be done only
      // once, hence I put it herein
      if FCaretY = I then
      begin
        J := GetLineEnd(I);
        // cursor beyond end of line?
        if FCaretX >= J then
          CaretX := J
        else
        begin
          // cursor in leading white space area?
          J := LeftSpaces(Line); // don't need TrimRightW here, as it has already been
                                 // verified that the cursor is before a non-white space char
          if FCaretX < J then
            CaretX := J
          else
            CaretX := FCaretX - FIndentSize;
        end;
      end
    end;

    // adjust selection appropriately (the Delphi IDE moves the selection start and
    // end to the first non-whitespace if there's no non-whitespace before the
    // particular X position, hence I'll do it too)
    Dec(BB.X, FIndentSize);
    Line := FContent[BB.Y].Text;
    I := LeftSpaces(Line);
    if I >= BB.X then
      BB.X := I;
    I := GetLineEnd(BB.Y);
    if I < BB.X then
      BB.X := I;

    if BE.X > 0 then
    begin
      Dec(BE.X, FIndentSize);
      Line := FContent[BE.Y].Text;
      I := LeftSpaces(Line);
      if I >= BE.X then
        BE.X := I;
      I := GetLineEnd(BE.Y);
      if I < BE.X then
        BE.X := I;
    end;

    // no further cursor adjustion, if the cursor is, by any mean, not on any of
    // the lines in the current selection then it won't be updated at all
    NewCaret := CaretXY;
    FUndoList.AddChange(ecrIndentation, OldText, @OldCaret, @SelStart, @SelEnd,
      NewText, @NewCaret, @FBlockBegin, @FBlockEnd);

    EndUpdate;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.CalcCharWidths(Index: Integer);

// Creates an array of integer values describing the width of each character
// of a given line in pixels.

var
  I, Run,
  NewPos: Integer;
  P: PWideChar;
  Multiplier: Integer;

begin
  if Index > FContent.Count - 1 then
    Index := FContent.Count - 1;
  if Index = -1 then
    FCharWidthArray := nil
  else
  begin
    P := PWideChar(FContent[Index].Text);
    SetLength(FCharWidthArray, Length(FContent[Index].Text));
    Multiplier := FTabSize * FCharWidth;
    // running term for WideTabulator calculation
    Run := 0;
    I := 0;
    while P^ <> WideNull do
    begin
      if P^ = WideTabulator then
      begin
        // calculate new absolute position
        NewPos := (Run div Multiplier + 1) * Multiplier;
        // make this a relative distance
        FCharWidthArray[I] := NewPos - Run;
      end
      else
      begin
        FCharWidthArray[I] := FCharWidth;
      end;

      Inc(Run, FCharWidthArray[I]);
      Inc(I);
      Inc(P);
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.CalcCharWidths(const S: WideString; TabSize, CharWidth: Cardinal);

// Creates an array of integer values describing the width of each character
// of the given WideString in pixels.

var
  I: Integer;
  Run,
    NewPos: Cardinal;

begin
  SetLength(FCharWidthArray, Length(S));
  // Running term for tabulator calculation.
  Run := 0;
  for I := 0 to Length(S) - 1 do
  begin
    if S[I + 1] = WideTabulator then
    begin
      // calculate new absolute position
      NewPos := (Run div (TabSize * CharWidth) + 1) * TabSize * CharWidth;
      // make this a relative distance
      FCharWidthArray[I] := NewPos - Run;
    end
    else
      FCharWidthArray[I] := FCharWidth;
    Inc(Run, FCharWidthArray[I]);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

function TCustomUniCodeEdit.CaretXPix: Integer;

begin
  Result := FCaretX * FCharWidth + FGutterRect.Right + 1 + FOffsetX;
end;

//----------------------------------------------------------------------------------------------------------------------

function TCustomUniCodeEdit.CaretYPix: Integer;

begin
  Result := (FCaretY + FOffsetY) * FTextHeight;
end;

//----------------------------------------------------------------------------------------------------------------------

function TCustomUniCodeEdit.CharIndexToColumn(X: Cardinal): Cardinal;

begin
  Result := CharIndexToColumn(Point(X, FCaretY));
end;

//----------------------------------------------------------------------------------------------------------------------

function TCustomUniCodeEdit.CharIndexToColumn(X: Cardinal; const S: WideString; TabSize, CharWidth: Cardinal):
  Cardinal;

// calculates the corresponding column from the given character position using the
// text stored in S

var
  I: Cardinal;

begin
  CalcCharWidths(S, TabSize, CharWidth);
  I := 0;
  Result := 0;
  while I < X do
  begin
    if I < Cardinal(Length(FCharWidthArray)) then
      Inc(Result, FCharWidthArray[I])
    else
      Inc(Result, CharWidth);
    Inc(I);
  end;
  Result := Result div Cardinal(CharWidth);
end;

//----------------------------------------------------------------------------------------------------------------------

function TCustomUniCodeEdit.CharIndexToColumn(Pos: TPoint): Cardinal;

// calculates the corresponding column from the given character position using the
// text stored in FContent at position Pos.Y

var
  I: Integer;

begin
  CalcCharWidths(Pos.Y);
  I := 0;
  Result := 0;
  while I < Pos.X do
  begin
    if I < Length(FCharWidthArray) then
      Inc(Result, FCharWidthArray[I])
    else
      Inc(Result, FCharWidth);
    Inc(I);
  end;
  Result := Result div Cardinal(FCharWidth);
end;

//----------------------------------------------------------------------------------------------------------------------

function TCustomUniCodeEdit.CharPositionToRowColumn(Position: Cardinal): TPoint;

// Calculates the total position of the column/row reference point as would all lines be
// one continous WideString.

var
  I: Integer;
  Len: Cardinal;

begin
  Result.Y := 0;
  for I := 0 to FContent.Count - 1 do
  begin
    Len := Length(FContent[I].Text) + 2; // sub 2 for WideCRLF
    if Position < Len then
      Break;
    Inc(Result.Y);
    Dec(Position, Len);
  end;
  Result.X := Position;
  Result.X := ColumnToCharIndex(Result);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.ClearAll(RegisterUndo, KeepUndoList: Boolean);

// Deletes the entire text and (in case KeepUndoList is not True) clears also the
// undo list. If KeepUndoList is True then RegisterUndo has no effect otherwise
// a new undo record is added to make the clear operation reversable.

var
  State: TEditState;
  I: Integer;

begin
  for I := 0 to 9 do
    RemoveBookmark(I);

  if FContent.Count > 0 then
  begin
    BeginUpdate;
    RegisterUndo := RegisterUndo and KeepUndoList;
    try
      if RegisterUndo then
      begin
        SelectAll;
        State := RecordState(True);
      end;
      FContent.Clear;
      FBlockBegin := Point(0, 0);
      FBlockEnd := Point(0, 0);
      CaretXY := Point(0, 0);
      if RegisterUndo then
        with State do
          FUndoList.AddChange(ecrDelete, Text, @Caret, @SelStart, @SelEnd);
    finally
      EndUpdate;
    end;
  end;
  if not KeepUndoList then
    FUndoList.ClearList;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.ClearSelection;

// Removes the current selection mark if there is any.

begin
  FBlockBegin := Point(0, 0);
  FBlockEnd := Point(0, 0);
  Invalidate;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.ClearUndo;

begin
  FUndoList.ClearList;
end;

//----------------------------------------------------------------------------------------------------------------------

function TCustomUniCodeEdit.ColumnToCharIndex(X: Cardinal): Cardinal;

begin
  Result := ColumnToCharIndex(Point(X, FCaretY));
end;

//----------------------------------------------------------------------------------------------------------------------

function TCustomUniCodeEdit.ColumnToCharIndex(Pos: TPoint): Cardinal;

// Calculates the character index into the line given by Y. X and Y are usually a
// caret position. Because of non-default width characters (like tabulators) the
// CaretX position is merely a column value and does not directly correspond to a
// particular character. The result can be used to address a character in the line
// given by Y or is a generic char index if X is beyond any char in Y.

var
  Run: Integer;

begin
  CalcCharWidths(Pos.Y);
  Run := 0;
  Pos.X := Pos.X * FCharWidth;
  Result := 0;
  while Result < Cardinal(Length(FCharWidthArray)) do
  begin
    Inc(Run, FCharWidthArray[Result]);
    if Run > Pos.X then
      Break;
    Inc(Result);
  end;
  // If the run is still smaller than the given column then add "virtual" chars
  // to be able to add a character (far) beyond the current end of the line.
  if Run < Pos.X then
    Inc(Result, (Pos.X - Run) div FCharWidth);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.CommandProcessor(Command: TEditorCommand; AChar: WideChar; Data: Pointer);

var
  CX: Integer;
  Len: Integer;
  Temp: WideString;
  Helper: WideString;
  OldOptions: TUniCodeEditOptions;
  Action: Boolean;
  WP: TPoint;
  Caret: TPoint;
  I, J: Integer;
  OldChar: WideChar;
  BB, BE: TPoint;
  LastState: TEditState;

  //--------------- local functions -------------------------------------------

  procedure RegisterCursorMoveChange;

  // helper function to ease the repetitive registration of the cursor move undo change

  var
    Caret: TPoint;

  begin
    if not PointsAreEqual(CaretXY, LastState.Caret) then
    begin
      Caret := CaretXY;
      FUndoList.AddChange(ecrCursorMove, '', @LastState.Caret, @LastState.SelStart, @LastState.SelEnd,
        '', @Caret, @FBlockBegin, @FBlockEnd);
    end;
  end;

  //---------------------------------------------------------------------------

  procedure RegisterDeleteChange;

  // helper function to ease the repetitive registration of a delete undo change

  begin
    LastState.Caret := Point(FCaretX, FCaretY);
    with LastState do
      FUndoList.AddChange(ecrDelete, Text, @Caret, @SelStart, @SelEnd);
  end;

  //---------------------------------------------------------------------------

  procedure RegisterReplaceChange(NewText: WideString; StartBound, EndBound: TPoint);

  // helper function to ease the repetitive registration of a replace undo change
  // StartBound and EndBound are just to ease making undo and do not correspond
  // to a selection

  var
    Caret: TPoint;

  begin
    Caret := CaretXY;
    FUndoList.AddChange(ecrReplace, LastState.Text, @LastState.Caret, @LastState.SelStart, @LastState.SelEnd,
      NewText, @Caret, @StartBound, @EndBound);
  end;

  //---------------------------------------------------------------------------

  procedure RegisterInsertChange(NewText: WideString; StartBound, EndBound: TPoint);

  // helper function to ease the repetitive registration of an insert undo change,
  // StartBound and EndBound are just to ease making undo and do not correspond
  // to a selection

  var
    Caret: TPoint;

  begin
    Caret := CaretXY;
    FUndoList.AddChange(ecrInsert, '', @LastState.Caret, @LastState.SelStart, @LastState.SelEnd,
      NewText, @Caret, @StartBound, @EndBound);
  end;

  //---------------------------------------------------------------------------

  procedure RegisterOverwriteChange(NewChar: WideChar);

  var
    Caret: TPoint;

  begin
    Caret := CaretXY;
    FUndoList.AddChange(ecrOverwrite, OldChar, @LastState.Caret, nil, nil,
      NewChar, @Caret, nil, nil);
  end;

  //--------------- end local functions ---------------------------------------

begin
  ProcessCommand(Command, AChar, Data);
  if (Command = ecNone) or (Command >= ecUserFirst) then
    Exit;

  case Command of
    ecLeft,
    ecSelLeft:
      begin
        LastState := RecordState(False);
        CX := PreviousCharPos(FCaretX);

        if not (eoScrollPastEOL in FOptions) and (FCaretX = 0) then
        begin
          if FCaretY > 0 then
          begin
            CaretY := FCaretY - 1;
            CaretX := GetLineEnd;
          end;
        end
        else
          CaretX := CX;

        if Command = ecSelLeft then
        begin
          if not SelectionAvailable then
            SetBlockBegin(LastState.Caret);
          SetBlockEnd(CaretXY);
        end
        else
          SetBlockBegin(CaretXY);

        RegisterCursorMoveChange;
      end;

    ecRight,
    ecSelRight:
      begin
        LastState := RecordState(False);
        CX := NextCharPos(FCaretX);

        if not (eoScrollPastEOL in FOptions) and (Cardinal(CX) > GetLineEnd) then
        begin
          if FCaretY < FContent.Count - 1 then
          begin
            CaretX := 0;
            CaretY := FCaretY + 1;
          end;
        end
        else
          CaretX := CX;

        if Command = ecSelRight then
        begin
          if not SelectionAvailable then
            SetBlockBegin(LastState.Caret);
          SetBlockEnd(CaretXY);
        end
        else
          SetBlockBegin(CaretXY);

        RegisterCursorMoveChange;
      end;

    ecUp,
    ecSelUp:
      begin
        LastState := RecordState(False);
        CaretY := FCaretY - 1;
        if (Command = ecSelUp) then
        begin
          if not SelectionAvailable then
            SetBlockBegin(Point(FCaretX, FCaretY + 1));
          SetBlockEnd(CaretXY);
        end
        else
          SetBlockBegin(CaretXY);

        RegisterCursorMoveChange;
      end;

    ecDown,
    ecSelDown:
      begin
        LastState := RecordState(False);
        CaretY := FCaretY + 1;
        if Command = ecSelDown then
        begin
          if not SelectionAvailable then
            SetBlockBegin(Point(FCaretX, FCaretY - 1));
          SetBlockEnd(CaretXY);
        end
        else
          SetBlockBegin(CaretXY);

        RegisterCursorMoveChange;
      end;

    ecWordLeft,
    ecSelWordLeft:
      begin
        LastState := RecordState(False);
        if (Command = ecSelWordLeft) and (not SelectionAvailable) then
          SetBlockBegin(CaretXY);
        CaretXY := LastWordPos;
        if Command = ecSelWordLeft then
          SetBlockEnd(CaretXY)
        else
          SetBlockBegin(CaretXY);
        RegisterCursorMoveChange;
      end;

    ecWordRight,
    ecSelWordRight:
      begin
        LastState := RecordState(False);
        if (Command = ecSelWordRight) and (not SelectionAvailable) then
          SetBlockBegin(CaretXY);
        CaretXY := NextWordPos;
        if Command = ecSelWordRight then
          SetBlockEnd(CaretXY)
        else
          SetBlockBegin(CaretXY);
        RegisterCursorMoveChange;
      end;

    ecLineStart,
    ecSelLineStart:
      begin
        LastState := RecordState(False);
        if (Command = ecSelLineStart) and (not SelectionAvailable) then
          SetBlockBegin(CaretXY);
        CaretX := 0;
        if Command = ecSelLineStart then
          SetBlockEnd(CaretXY)
        else
          SetBlockBegin(CaretXY);
        RegisterCursorMoveChange;
      end;

    ecLineEnd,
    ecSelLineEnd:
      begin
        LastState := RecordState(False);
        if (Command = ecSelLineEnd) and (not SelectionAvailable) then
          SetBlockBegin(CaretXY);
        if eoKeepTrailingBlanks in FOptions then
          Helper := LineText
        else
          Helper := WideTrimRight(LineText);
        if Helper <> LineText then
          LineText := Helper;
        CaretX := GetLineEnd(FCaretY);
        if Command = ecSelLineEnd then
          SetBlockEnd(CaretXY)
        else
          SetBlockBegin(CaretXY);
        RegisterCursorMoveChange;
      end;

    ecPageUp,
    ecSelPageUp:
      begin
        LastState := RecordState(False);
        if (Command = ecSelPageUp) and (not SelectionAvailable) then
          SetBlockBegin(CaretXY);
        Caret := Point(FCaretX, FCaretY - FLinesInWindow);
        if Command = ecSelPageUp then
          SetBlockEnd(Caret)
        else
          SetBlockBegin(Caret);
        TopLine := TopLine - LinesInWindow;
        CaretXY := Caret;
        RegisterCursorMoveChange;
      end;

    ecPageDown,
    ecSelPageDown:
      begin
        LastState := RecordState(False);
        if (Command = ecSelPageDown) and (not SelectionAvailable) then
          SetBlockBegin(CaretXY);
        Caret := Point(FCaretX, FCaretY + FLinesInWindow);
        if (Command = ecSelPageDown) then
          SetBlockEnd(Caret)
        else
          SetBlockBegin(Caret);
        TopLine := TopLine + LinesInWindow;
        CaretXY := Caret;
        RegisterCursorMoveChange;
      end;

    ecPageLeft,
    ecSelPageLeft:
      begin
        LastState := RecordState(False);
        if (Command = ecSelPageLeft) and (not SelectionAvailable) then
          SetBlockBegin(CaretXY);
        OffsetX := FOffsetX + CharsInWindow * FCharWidth;
        CaretX := FCaretX - CharsInWindow;
        if (Command = ecSelPageLeft) then
          SetBlockEnd(CaretXY)
        else
          SetBlockBegin(CaretXY);
        RegisterCursorMoveChange;
      end;

    ecPageRight,
    ecSelPageRight:
      begin
        LastState := RecordState(False);
        if (Command = ecSelPageRight) and (not SelectionAvailable) then
          SetBlockBegin(CaretXY);
        OffsetX := FOffsetX - CharsInWindow * FCharWidth;
        CaretX := FCaretX + CharsInWindow;
        if (Command = ecSelPageRight) then
          SetBlockEnd(CaretXY)
        else
          SetBlockBegin(CaretXY);
        RegisterCursorMoveChange;
      end;

    ecPageTop,
    ecSelPageTop:
      begin
        LastState := RecordState(False);
        if (Command = ecSelPageTop) and (not SelectionAvailable) then
          SetBlockBegin(CaretXY);
        CaretY := TopLine - 1;
        if Command = ecSelPageTop then
          SetBlockEnd(CaretXY)
        else
          SetBlockBegin(CaretXY);
        RegisterCursorMoveChange;
      end;

    ecPageBottom,
    ecSelPageBottom:
      begin
        LastState := RecordState(False);
        if (Command = ecSelPageBottom) and (not SelectionAvailable) then
          SetBlockBegin(CaretXY);
        CaretY := TopLine + LinesInWindow - 2;
        if (Command = ecSelPageBottom) then
          SetBlockEnd(CaretXY)
        else
          SetBlockBegin(CaretXY);
        RegisterCursorMoveChange;
      end;

    ecEditorTop,
    ecSelEditorTop:
      begin
        LastState := RecordState(False);
        if (Command = ecSelEditorTop) and (not SelectionAvailable) then
          SetBlockBegin(CaretXY);
        CaretX := 0;
        CaretY := 0;
        if Command = ecSelEditorTop then
          SetBlockEnd(CaretXY)
        else
          SetBlockBegin(CaretXY);
        RegisterCursorMoveChange;
      end;

    ecEditorBottom,
    ecSelEditorBottom:
      begin
        LastState := RecordState(False);
        if (Command = ecSelEditorBottom) and (not SelectionAvailable) then
          SetBlockBegin(CaretXY);
        CaretY := FContent.Count - 1;
        OffsetY := FLinesInWindow - FContent.Count;
        CaretX := Length(FContent[FContent.Count - 1].Text);
        OffsetX := 0;
        if Command = ecSelEditorBottom then
          SetBlockEnd(CaretXY)
        else
          SetBlockBegin(CaretXY);
        RegisterCursorMoveChange;
      end;

    ecGotoXY,
    ecSelGotoXY:
      if Data <> nil then
      begin
        LastState := RecordState(False);
        if (Command = ecSelGotoXY) and (not SelectionAvailable) then
          SetBlockBegin(CaretXY);
        CaretXY := PPoint(Data)^;
        if Command = ecSelGotoXY then
          SetBlockEnd(CaretXY)
        else
          SetBlockBegin(CaretXY);
        RegisterCursorMoveChange;
      end;

    ecSelectAll:
      SelectAll;

    ecDeleteLastChar:
      if not (eoReadOnly in FOptions) then
      begin
        LastState := RecordState(True);
        if SelectionAvailable then
        begin
          SetSelText('');
          RegisterDeleteChange;
        end
        else
        begin
          Temp := LineText;
          Len := Length(Temp);
          // get char index from column
          CX := ColumnToCharIndex(FCaretX);
          if CX <= Len then
          begin
            if CX > 0 then
            begin
              // current index is somewhere on the line, first check whether we
              // can do auto unindentation
              I := CX - 1;
              while (I > -1) and UnicodeIsWhiteSpace(Word(Temp[I + 1])) do
                Dec(I);
              if (eoAutoUnindent in FOptions) and (I = -1) then
              begin
                // fine, all conditions met to do auto unindentation
                I := Unindent(FCaretX, FCaretY);
                J := ColumnToCharIndex(I);
                Helper := Copy(Temp, J + 1, CX - J);
                Delete(Temp, J + 1, CX - J);
                LineText := Temp;
                CaretX := I;
                LastState.Text := Helper;
                RegisterDeleteChange;
              end
              else
              begin
                // no unindentation, just delete last character;
                // need to test for non-default width character though
                if CharIndexToColumn(CX) = Cardinal(FCaretX) then
                begin
                  Helper := Temp[CX];
                  Delete(Temp, CX, 1);
                  CaretX := PreviousCharPos(FCaretX, FCaretY);
                end
                else
                begin
                  Helper := Temp[CX + 1];
                  Delete(Temp, CX + 1, 1);
                  CaretX := CharIndexToColumn(CX);
                end;
                LineText := Temp;
                LastState.Text := Helper;
                RegisterDeleteChange;
              end;
            end
            else
            begin
              // character index is at line start, but the first char might be a non-default
              // width char and the current column is somewhere inbetween -> check this
              if FCaretX > 0 then
              begin
                // yes, the first char is of non-default width, so delete this char
                // instead of moving the line one step up
                LastState.Text := Copy(Temp, 1, 1);
                Delete(Temp, 1, 1);
                LineText := Temp;
                CaretX := 0;
                RegisterDeleteChange;
              end
              else
                // no, first char is just an ordinary one, so move this line one step up
                if FCaretY > 0 then
                begin
                  FContent.DeleteLine(FCaretY);
                  // need to relocate bookmarks
                  for I := 0 to 9 do
                    if FBookmarks[I].Y >= FCaretY then
                      SetBookmark(I, FBookmarks[I].X, FBookmarks[I].Y - 1);
                  CaretY := CaretY - 1;
                  CaretX := GetLineEnd(FCaretY);
                  LineText := LineText + Temp;
                  LastState.Text := WideCRLF;
                  RegisterDeleteChange;
                  InvalidateToBottom(FCaretY);
                end;
            end;
          end
          else
          begin
            // caret is beyond the end of the current line -> just do a cursor move
            CaretX := CharIndexToColumn(CX - 1);
            RegisterCursorMoveChange;
          end;
        end;
      end;

    ecDeleteChar:
      if not (eoReadOnly in FOptions) then
      begin
        LastState := RecordState(True);
        if SelectionAvailable then
        begin
          SetSelText('');
          SetBlockBegin(CaretXY);
          RegisterDeleteChange;
        end
        else
        begin
          Temp := LineText;
          Len := Length(Temp);
          CX := ColumnToCharIndex(FCaretX);
          if Len > CX then
          begin
            LastState.Text := Temp[CX + 1];
            Delete(Temp, CX + 1, 1);
            LineText := Temp;
            // need a caret change here if it was place on a non-default width char
            CaretX := CharIndexToColumn(CX);
            RegisterDeleteChange;
          end
          else
          begin
            if FCaretY < FContent.Count - 1 then
            begin
              Helper := WideStringOfChar(' ', CX - Len);
              LineText := Temp + Helper + FContent[FCaretY + 1].Text;
              FContent.DeleteLine(FCaretY + 1);
              // need to relocate bookmarks
              for I := 0 to 9 do
                if FBookmarks[I].Y >= FCaretY then
                  SetBookmark(I, FBookmarks[I].X, FBookmarks[I].Y - 1);
              InvalidateToBottom(FCaretY);
              LastState.Text := WideCRLF;
              RegisterDeleteChange;
            end;
          end;
        end;
      end;

    ecDeleteWord:
      if not (eoReadOnly in FOptions) then
      begin
        LastState := RecordState(True);
        WP := NextWordPos;
        if (FCaretX <> WP.X) or (FCaretY <> WP.Y) then
        begin
          SetBlockBegin(CaretXY);
          SetBlockEnd(WP);
          LastState.Text := SelectedText;
          SetSelText('');
          CaretXY := CaretXY; // fix up cursor position
          RegisterDeleteChange;
        end;
      end;

    ecDeleteLastWord:
      if not (eoReadOnly in FOptions) then
      begin
        LastState := RecordState(True);
        WP := LastWordPos;
        if (FCaretX <> WP.X) or (CaretY <> WP.Y) then
        begin
          SetBlockBegin(WP);
          SetBlockEnd(CaretXY);
          LastState.Text := SelectedText;
          SetSelText('');
          CaretXY := CaretXY; // fix up cursor position
          RegisterDeleteChange;
        end;
      end;

    ecDeleteBOL:
      if not (eoReadOnly in FOptions) then
      begin
        LastState := RecordState(True);
        LastState.Text := Copy(LineText, 1, ColumnToCharIndex(FCaretX));
        SetBlockBegin(Point(0, CaretY));
        SetBlockEnd(CaretXY);
        SetSelText('');
        CaretXY := Point(0, CaretY); // fix up cursor position
        RegisterDeleteChange;
      end;

    ecDeleteEOL:
      if not (eoReadOnly in FOptions) then
      begin
        LastState := RecordState(True);
        CX := ColumnToCharIndex(FCaretX);
        LastState.Text := Copy(LineText, CX + 1, Length(LineText) - CX);
        SetBlockBegin(CaretXY);
        SetBlockEnd(Point(GetLineEnd, CaretY));
        SetSelText('');
        CaretXY := CaretXY; // fix up cursor position
        RegisterDeleteChange;
      end;

    ecDeleteLine:
      if not (eoReadOnly in FOptions) then
      begin
        LastState := RecordState(True);
        CaretX := 0;
        if SelectionAvailable then
          SetBlockBegin(CaretXY);
        LastState.Text := PWideChar(LineText + WideCRLF);
        FContent.DeleteLine(FCaretY);
        RegisterDeleteChange;
      end;

    ecClearAll:
      if not (eoReadOnly in FOptions) then
      begin
        ClearAll(True, True);
      end;

    ecInsertLine,
    ecLineBreak:
      if not (eoReadOnly in FOptions) then
      begin
        LastState := RecordState(True);
        if SelectionAvailable then
        begin
          BB := MinPoint(FBlockBegin, FBlockEnd);
          Caret := BB;
          Temp := '';
          if not (eoKeepTrailingBlanks in FOptions) then
          begin
            // include trailing blanks (which would otherwise appear and automatically
            // deleted after the line break has been inserted) into the text being
            // replaced, to be able to restore them while doing undo
            CX := ColumnToCharIndex(BB);
            // keep char index, we need it later
            I := CX;
            Helper := FContent[BB.Y].Text;
            // extent the helper WideString to match the caret position
            if CX > Length(Helper) then
              Helper := Helper + WideStringOfChar(' ', CX - Length(Helper));
            Dec(CX);
            while (CX > -1) and UnicodeIsWhiteSpace(Word(Helper[CX + 1])) do
              Dec(CX);
            Inc(CX);
            BB.X := CharIndexToColumn(Point(CX, BB.Y));
            // copy white spaces potentially to be restored
            if CX < I then
              Temp := Copy(Helper, CX + 1, I - CX);
          end;
          SetSelText(WideCRLF);
          SetBlockBegin(CaretXY);
          // save additional text to be restored
          LastState.Text := Temp + LastState.Text;
          BE := CaretXY;
          // restore caret postion if we deal with a line insertion
          if Command = ecInsertLine then
            CaretXY := Caret;
          RegisterReplaceChange(WideCRLF, BB, BE);
        end
        else
        begin
          Temp := LineText;
          CX := ColumnToCharIndex(FCaretX);
          LineText := Copy(Temp, 1, CX);
          BB := Point(FCaretX, FCaretY);
          Delete(Temp, 1, CX);
          InvalidateToBottom(FCaretY);
          if eoAutoIndent in FOptions then
            Caret := Point(NextSmartTab(0, FCaretY + 1, False), FCaretY + 1)
          else
            Caret := Point(0, FCaretY + 1);
          // the just calculated column corresponds directly to the characters we
          // have to insert, hence no conversion to a char index is needed
          Helper := WideStringOfChar(' ', Caret.X);
          Insert(Helper, Temp, 1);
          FContent.InsertLine(Caret.Y, Temp);
          if Command = ecLineBreak then
            CaretXY := Caret;

          RegisterInsertChange(WideCRLF + Helper, BB, Caret);

          // Need to relocate bookmarks.
          for I := 0 to 9 do
            if (FBookmarks[I].Y + 1 >= CaretY) and (FBookmarks[I].X >= CX) then
              SetBookmark(I, FBookmarks[I].X, FBookmarks[I].Y + 1);
          InvalidateToBottom(FCaretY - 1); 
        end;
      end;

    ecChar:
      // #127 is Ctrl + Backspace
      if ((AChar = WideTabulator) or (AChar >= WideSpace)) and (AChar <> #127) and not (eoReadOnly in FOptions) then
      begin
        LastState := RecordState(True);
        if SelectionAvailable then
        begin
          // Special case TAB -> it does not delete the selection, but enhances
          // it to the next smart tab pos and moves following characters accordingly
          // if the editor is using smart tabs currently
          // (that's the way the Delphi IDE does it, so I'll do it too).
          if (AChar = WideTabulator) and (eoSmartTabs in FOptions) then
          begin
            CX := NextSmartTab(FCaretX, FCaretY, True);
            I := ColumnToCharIndex(Point(FCaretX, FCaretY));
            Temp := LineText;
            Insert(WideStringOfChar(' ', CX - FCaretX), Temp, I + 1);
            LineText := Temp;
            // adjust selection
            BB := MinPoint(FBlockBegin, FBLockEnd);
            BE := MaxPoint(FBlockBegin, FBLockEnd);
            if (FCaretY = BB.Y) and (FCaretX <= BB.X) then
            begin
              SetBlockBegin(Point(BB.X + CX - FCaretX, FCaretY));
              if FCaretY = BE.Y then
                SetBlockEnd(Point(BE.X + CX - FCaretX, FCaretY))
              else
                SetBlockEnd(BE);
            end
            else
              if (FCaretY = BE.Y) and (FCaretX >= BE.X) then
                SetBlockEnd(Point(CX, FCaretY));
            CaretX := CX;
            RegisterReplaceChange(SelectedText, FBlockBegin, FBlockEnd);
          end
          else
          begin
            BB := MinPoint(FBlockBegin, FBlockEnd);
            if (AChar = WideTabulator) and not (eoUseTabs in FOptions) then
              Temp := WideStringOfChar(' ', FTabSize)
            else
              Temp := AChar;
            SetSelText(Temp);
            CaretX := NextCharPos(BB.X, BB.Y);
            SetBlockBegin(CaretXY);
            RegisterReplaceChange(Temp, BB, CaretXY);
          end;
        end
        else
        begin
          Temp := LineText;
          Len := Length(Temp);
          CX := ColumnToCharIndex(FCaretX);
          if Len < CX then
            Temp := Temp + WideStringOfChar(' ', CX - Len);
          OldOptions := FOptions;
          Include(FOptions, eoScrollPastEOL);
          try
            if eoInserting in FOptions then
            begin
              if (AChar = WideTabulator) and not (eoUseTabs in FOptions) then
              begin
                if eoSmartTabs in FOptions then
                  I := NextSmartTab(FCaretX, FCaretY, True)
                else
                  I := FCaretX + FTabSize;
                // If I is still at FCaretX then do nothing (can happen with smart
                // tabs under certain conditions).
                if I > FCaretX then
                begin
                  // Insert as many spaces as the size of the column difference.
                  Helper := WideStringOfChar(' ', I - FCaretX);
                  Insert(Helper, Temp, CX + 1);
                  Action := True;
                end
                else
                  Action := False;
              end
              else
              begin
                // insert the new character (plus some spaces if the current column is
                // within a non-default width character and the new character is not a tab)
                if AChar = WideTabulator then
                begin
                  Helper := AChar;
                  Insert(Helper, Temp, CX + 1);
                  // need to set the line text at this point, although it is redundant (as it will
                  // be set in the action branch below), because the new caret position
                  // needs also to be calculated and this computation is based on the
                  // text
                  LineText := Temp;
                  I := NextCharPos(FCaretX, True);
                end
                else
                begin
                  Helper := WideStringOfChar(' ', Cardinal(FCaretX) - CharIndexToColumn(CX)) + AChar;
                  Insert(Helper, Temp, CX + 1);
                  I := FCaretX + 1;
                end;
                Action := True;
              end;

              if Action then
              begin
                LineText := Temp;
                CaretX := I;
                RegisterInsertChange(Helper, LastState.Caret, CaretXY);
                if FCaretX >= -FOffsetX * FCharWidth + FCharsInWindow then
                  OffsetX := FOffsetX - Min(25, FCharsInWindow - 1) * FCharWidth;
              end;
            end
            else
            begin
              // in overwrite mode does a tab char only advance the cursor
              if AChar = WideTabulator then
              begin
                CaretX := NextTabPos(FCaretX);
                RegisterCursorMoveChange;
              end
              else
              begin
                OldChar := Temp[CX + 1];
                Temp[CX + 1] := AChar;
                CaretX := NextCharPos(FCaretX);
                LineText := Temp;
                RegisterOverwriteChange(AChar);
                if CaretX >= -FOffsetX * FCharWidth + FCharsInWindow then
                  OffsetX := FOffsetX + Min(25, FCharsInWindow - 1) * FCharWidth;
              end;
            end;
          finally
            FOptions := OldOptions;
          end;
        end;
      end;

    ecUndo:
      Undo;

    ecRedo:
      Redo;

    ecGotoMarker0..ecGotoMarker9:
      if BookMarkOptions.EnableKeys then
        GotoBookMark(Command - ecGotoMarker0);

    ecToggleMarker0..ecToggleMarker9:
      begin
        if BookMarkOptions.EnableKeys then
        begin
          CX := Command - ecToggleMarker0;
          Action := FBookMarks[CX].Y <> CaretY;
          RemoveBookmark(CX);
          if Action then
            SetBookMark(CX, FCaretX, CaretY);
        end;
      end;

    ecCut:
      if not (eoReadOnly in FOptions) and SelectionAvailable then
        CutToClipboard;

    ecCopy:
      CopyToClipboard;

    ecPaste:
      if not (eoReadOnly in FOptions) then
        PasteFromClipboard;

    ecScrollUp:
      begin
        TopLine := TopLine - 1;
        if CaretY > TopLine + FLinesInWindow - 2 then
          CaretY := TopLine + LinesInWindow - 2;
        RegisterCursorMoveChange;
      end;

    ecScrollDown:
      begin
        TopLine := TopLine + 1;
        if CaretY < TopLine - 1 then
          CaretY := TopLine - 1;
        RegisterCursorMoveChange;
      end;

    ecScrollLeft:
      begin
        Dec(FOffsetX, FCharWidth);
        if FCaretX > -FOffsetX * FCharWidth + CharsInWindow then
          CaretX := -FOffsetX * FCharWidth + CharsInWindow;
        RegisterCursorMoveChange;
      end;

    ecScrollRight:
      begin
        Inc(FOffsetX, FCharWidth);
        if FCaretX < -FOffsetX * FCharWidth then
          CaretX := -FOffsetX * FCharWidth;
        RegisterCursorMoveChange;
      end;

    ecInsertMode:
      begin
        Include(FOptions, eoInserting);
        InitializeCaret;
        DoSettingChanged;
      end;

    ecOverwriteMode:
      begin
        Exclude(FOptions, eoInserting);
        InitializeCaret;
        DoSettingChanged;
      end;

    ecToggleMode:
      begin
        if eoInserting in FOptions then
          Exclude(FOptions, eoInserting)
        else
          Include(FOptions, eoInserting);
        InitializeCaret;
        DoSettingChanged;
      end;

    ecBlockIndent:
      if not (eoReadOnly in FOptions) then
        BlockIndent;

    ecBlockUnindent:
      if not (eoReadOnly in FOptions) then
        BlockUnindent;
  end;

  // Add macro recorder stuff here?
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.ComputeCaret(X, Y: Integer);

// sets the caret to the line/column indicated by the given pixel position

var
  EditState: TEditState;
  NewCaret: TPoint;

begin
  EditState := RecordState(False);
  CaretXY := PositionFromPoint(X, Y);
  with EditState do
    if not PointsAreEqual(CaretXY, Caret) then
    begin
      NewCaret := CaretXY;
      FUndoList.AddChange(ecrCursorMove, '', @Caret, @SelStart, @SelEnd,
        '', @NewCaret, nil, nil);
    end;
  UpdateWindow(Handle);
end;

//----------------------------------------------------------------------------------------------------------------------

function TCustomUniCodeEdit.CopyOnDrop(Source: TObject): Boolean;

// determines wether a drag'n drop operation is a move or a copy

var
  Shift: TShiftState;

begin
  // determine move or copy operation
  Shift := KeyDataToShiftState(0);
  // w/o modifier keys the operation is determined source and target edit being the same or not
  if Shift = [] then
    Result := Source <> Self
  else
    Result := (ssCtrl in Shift) or (Source <> Self) and not (ssShift in Shift);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.CopyToClipboard;

begin
  // Delphi's TClipboard does not support Unicode (D5 and lower) hence we
  // need to handle the stuff "manually"
  if SelectionAvailable then
    TextToClipboard(SelectedText);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.CreateParams(var Params: TCreateParams);

const
  ScrollBar: array[TScrollStyle] of DWORD = (0, WS_HSCROLL, WS_VSCROLL, WS_HSCROLL or WS_VSCROLL);
  BorderStyles: array[TBorderStyle] of DWORD = (0, WS_BORDER);

  CS_OFF = CS_HREDRAW or CS_VREDRAW;

begin
  inherited CreateParams(Params);
  with Params do
  begin
    Style := Style or ScrollBar[FScrollBars] or BorderStyles[FBorderStyle] or WS_CLIPCHILDREN or WS_CLIPSIBLINGS;
    WindowClass.Style := WindowClass.Style and not CS_OFF;
    if NewStyleControls and Ctl3D and (FBorderStyle = bsSingle) then
    begin
      Style := Style and not WS_BORDER;
      ExStyle := ExStyle or WS_EX_CLIENTEDGE;
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.CutToClipboard;

var
  State: TEditState;
  InsertionPoint: TPoint;

begin
  if SelectionAvailable then
  begin
    State := RecordState(True);
    CopyToClipboard;
    SetSelText('');
    with State do
    begin
      InsertionPoint := MinPoint(SelStart, SelEnd); 
      FUndoList.AddChange(ecrDelete, Text, @InsertionPoint, @SelStart, @SelEnd);
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.DeleteSelection(NeedRescan: Boolean);

// Deletes all characters in the current selection and causes the highlighter
// to rescan modified lines if NeedRescan is True. Set this to False if you are going
// to insert new text at the current cursor position afterwards and start the rescan then.

var
  I,
  Count,
  Start: Integer;
  TempString,
  TempString2: WideString;
  BB, BE: TPoint;
  BeginX,
  EndX: Integer;

begin
  BB := MinPoint(FBlockBegin, FBlockEnd);
  BeginX := ColumnToCharIndex(BB);
  BE := MaxPoint(FBlockBegin, FBlockEnd);
  EndX := ColumnToCharIndex(BE);

  CaretX := BB.X;
  if BB.Y = BE.Y then
  begin
    TempString := FContent[BB.Y].Text;
    Delete(TempString, BeginX + 1, EndX - BeginX);
    FContent[BB.Y].Text := TempString;
  end
  else
  begin
    CaretY := BB.Y;
    TempString := FContent[BB.Y].Text;
    if Length(TempString) < BeginX then
      TempString := TempString + WideStringOfChar(' ', BeginX - Length(TempString))
    else
      Delete(TempString, BeginX + 1, Length(TempString));
    if BE.Y < FContent.Count then
    begin
      TempString2 := FContent[BE.Y].Text;
      Delete(TempString2, 1, EndX);
      TempString := TempString + TempString2;
    end;
    Start := BB.Y;
    Count := BE.Y - BB.Y;
    for I := 0 to 9 do
      if FBookmarks[I].Y >= BE.Y then
        SetBookmark(I, FBookmarks[I].X, FBookmarks[I].Y - Count)
      else
        if FBookmarks[I].Y > BB.Y then
          SetBookmark(I, FBookmarks[I].X, BB.Y);

    try
      for I := Count + Start - 1 downto Start do
        FContent.DeleteLine(I);
    finally
      // Don't rescan if strings are empty now or something is to be inserted afterwards.
      if Assigned(FHighlighter) and (Start > 0) and NeedRescan then
        ScanFrom(Start - 1);
    end;
    // if all lines have been removed then TempString can only be an empty WideString
    if Start < FContent.Count then
      FContent[Start].Text := TempString;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.DoCaretChange;

begin
  {// try setting the correct keyboard layout
  if not (csLButtonDown in ControlState) and
     (FCaretY < FContent.Count) and
     (FCaretX > 0) and
     (FCaretX < Length(FContent[FCaretY])) then
    ActivateKeyboard(FContent[FCaretY][FCaretX]);}
  if Assigned(FOnCaretChange) then
    FOnCaretChange(Self, FCaretX, FCaretY);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.DoGutterMouseDown(Button: TMouseButton; Shift: TShiftState; X, Y, Line: Integer);

begin
  if Assigned(FOnGutterMouseDown) then
    FOnGutterMouseDown(Self, Button, Shift, X, Y, Line);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.DoScroll(DeltaX, DeltaY: Integer);

begin
  if Assigned(FOnScroll) then
    FOnScroll(Self, DeltaX, DeltaY);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.DoSettingChanged;

begin
  if Assigned(FOnSettingChange) then
    FOnSettingChange(Self);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.DoStartDrag(var DragObject: TDragObject);

begin
  FDragWasStarted := False;
  inherited;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.DragCanceled;

begin
  FScrollTimer.Enabled := False;
  CaretXY := FLastCaret; // restore previous caret position
  if not FDragWasStarted then
    SetBlockBegin(FLastCaret);
  inherited;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.DragDrop(Source: TObject; X, Y: Integer);

var
  DoMove,
    DoReplace: Boolean;
  NewCaret: TPoint;
  OldBB,
    OldBE: TPoint;
  OldText,
    NewText: WideString;
  ThisEditState,
    OtherEditState: TEditState;

begin
  if not (eoReadOnly in FOptions) then
  begin
    inherited;

    // determine move or copy operation
    DoMove := not CopyOnDrop(Source);
    ThisEditState := RecordState(True);

    if Source = Self then
    begin
      NewText := GetSelectedText;
      NewCaret := CaretXY;

      OldText := GetSelectedText;
      OldBB := MinPoint(FBlockBegin, FBlockEnd);
      OldBE := MaxPoint(FBlockBegin, FBlockEnd);

      if DoMove then
      begin
        if PosInSelection(NewCaret) then
          NewCaret := OldBB
        else
        begin
          // continue calculation with char indices
          NewCaret.X := ColumnToCharIndex(NewCaret);
          if (NewCaret.Y >= OldBB.Y) and (NewCaret.X >= OldBB.X) then
          begin
            // correct cursor position if the target pos is located after the selection start
            Dec(NewCaret.Y, OldBE.Y - OldBB.Y);
            if NewCaret.Y = OldBB.Y then
              Inc(NewCaret.X, OldBB.X);
            if NewCaret.Y = OldBE.Y then
              Dec(NewCaret.X, OldBE.X);
          end;
          // finally go back to columns
          NewCaret.X := CharIndexToColumn(NewCaret);
        end;
        // delete old text
        SetSelText('');
      end;

      // insert the text at new position...
      CaretXY := NewCaret;
      SetBlockBegin(NewCaret);
      SetSelText(NewText);

      // ...and mark it again as selected
      CaretXY := NewCaret;
      SetBlockBegin(NewCaret);
      SetBlockEnd(Point(NewCaret.X + OldBE.X - OldBB.X, NewCaret.Y + OldBE.Y - OldBB.Y));

      // finally register undo change
      if DoMove then
        FUndoList.AddChange(ecrDragMove, OldText, @FLastCaret, @OldBB, @OldBE)
      else
        FUndoList.AddChange(ecrDragCopy, NewText, @NewCaret, @OldBB, @OldBE);
    end
    else
      if Source is TCustomUniCodeEdit then
      begin
        // drag'n drop operation between different edits
        NewText := TCustomUniCodeEdit(Source).GetSelectedText;
        NewCaret := CaretXY;
        DoReplace := PosInSelection(PositionFromPoint(X, Y));

        OldText := GetSelectedText;
        OldBB := MinPoint(FBlockBegin, FBlockEnd);
        OldBE := MaxPoint(FBlockBegin, FBlockEnd);

        // replace selected text here with selected text from other edit if the drag cursor
        // points to selected text, otherwise insert the new text
        if DoReplace then
          CaretXY := OldBB
        else
          BlockBegin := CaretXY; // make sure nothing is selected anymore
        SetSelText(NewText);
        // if the user wants to move text then additionally delete the selected text in the other edit
        with Source as TCustomUniCodeEdit do
        begin
          if DoMove then
          begin
            // retrieve text, selection bounds and cursor position from other edit for undo registration
            OtherEditState := RecordState(True);
            CaretXY := MinPoint(FBlockBegin, FBlockEnd);
            SetSelText('');
            // register undo action in other edit
            with OtherEditState do
              FUndoList.AddChange(ecrDelete, Text, @FLastCaret, @SelStart, @SelEnd);

          end;
        end;

        // mark the new text as selected
        if DoReplace then
          BlockBegin := OldBB
        else
          BlockBegin := NewCaret;
        BlockEnd := CaretXY;
        // If there's a move operation then there was also a caret update in the code above which
        // set the caret for the source edit. We need now to tell Windows to set back the caret (which
        // is a shared ressource) to the correct coordinates in this edit.
        if DoMove then
          UpdateCaret;

        // finally register undo change in this edit
        with ThisEditState do
          FUndoList.AddChange(ecrInsert, '', @Caret, @SelStart, @SelEnd,
            NewText, @FBlockBegin, @FBlockBegin, @FBlockEnd);
      end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.DragOver(Source: TObject; X, Y: Integer; State: TDragState; var Accept: Boolean);

// drag operations are also allowed between different edits

begin
  FDragWasStarted := True;
  inherited;
  if Accept and (Source is TCustomUniCodeEdit) then
  begin
    case State of
      dsDragLeave:
        begin
          FDropTarget := False;
          FScrollTimer.Enabled := False;
          // give the timer time to process its pending message
          Application.ProcessMessages;
        end;
      dsDragEnter:
        begin
          // remember that we are drag target currently as the originator of the
          // drag operation might be another syntax edit instance
          FDropTarget := True;
          SetFocus;
          FScrollTimer.Enabled := True;
        end;
    end;

    // If source and target are identical then we cannot drop on selected text.
    // If this edit is not the source then we accept the operation and will
    // replace the selected text by the incoming one.
    if Source = Self then
      Accept := not PosInSelection(PositionFromPoint(X, Y))
    else
      Accept := True;

    if Accept then
    begin
      // if Ctrl is pressed or Source is not this control then change
      // cursor to indicate copy instead of move
      if CopyOnDrop(Source) then
        TCustomUniCodeEdit(Source).DragCursor := crDragCopy
      else
        TCustomUniCodeEdit(Source).DragCursor := crDragMove;
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.EndUpdate;

begin
  if FUpdateCount > 0 then
    Dec(FUpdateCount);
  if FUpdateCount = 0 then
  begin
    SetUpdateState(False);
    UpdateScrollBars;
    UpdateCaret;
    DoCaretChange;
    Invalidate;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.EnsureCursorPosVisible;

begin
  // make sure X is visible
  if FCaretX < (-FOffsetX div FCharWidth) then
    OffsetX := -FCaretX * FCharWidth
  else
    if FCaretX >= (FCharsInWindow - FOffsetX div FCharWidth) then
      OffsetX := -(FCaretX - FCharsInWindow + 1) * FCharWidth;

  // make sure Y is visible
  if FCaretY < (TopLine - 1) then
    TopLine := FCaretY + 1
  else
    if FCaretY > (TopLine + (LinesInWindow - 2)) then
      TopLine := FCaretY - (LinesInWindow - 2);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.FontChanged(Sender: TObject);

var
  DC: HDC;
  Save: THandle;
  Metrics: TTextMetric;
  TMFont: HFONT;
  TMLogFont: TLogFont;

begin
  GetObject(Font.Handle, SizeOf(TLogFont), @TMLogFont);

  // We need to get the font with bold and italics set so we have enough room
  // for the widest character it will draw.  This is because fixed pitch fonts
  // always have the same character width, but ONLY when the same attributes
  // are being used.  That is, a bold character can be wider than a non-bold
  // character in a fixed pitch font.
  TMLogFont.lfWeight := FW_BOLD;
  TMLogFont.lfItalic := 1;
  TMFont := CreateFontIndirect(TMLogFont);
  try
    DC := GetDC(0);
    try
      Save := SelectObject(DC, TMFont);
      GetTextMetrics(DC, Metrics);
      SelectObject(DC, Save);
    finally
      ReleaseDC(0, DC);
    end;
  finally
    DeleteObject(TMFont);
  end;

  with Metrics do
  begin
    // Note:  Through trial-and-error I found that tmAveCharWidth should be used
    // instead of tmMaxCharWidth as you would think. I'm basing this behavior
    // on the Delphi IDE editor behavior. If tmMaxCharWidth is used, we end up
    // with chars being much farther apart than the same font in the IDE.
    CharWidth := tmAveCharWidth;

    FTextHeight := tmHeight + tmExternalLeading + FExtraLineSpacing;
    if Assigned(Parent) then
    begin
      FLinesInWindow := ClientHeight div FTextHeight;
      FCharsInWindow := (ClientWidth - FGutterRect.Right + 1) div FCharWidth;
    end;
    if HandleAllocated then
    begin
      if Focused then
      begin
        HideCaret;
        InitializeCaret;
      end;
      UpdateScrollBars;
    end;
    Invalidate;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

function TCustomUniCodeEdit.GetBlockBegin: TPoint;

begin
  Result := MinPoint(FBlockBegin, FBlockEnd);
end;

//----------------------------------------------------------------------------------------------------------------------

function TCustomUniCodeEdit.GetBlockEnd: TPoint;

begin
  Result := MaxPoint(FBlockBegin, FBlockEnd);
end;

//----------------------------------------------------------------------------------------------------------------------

function TCustomUniCodeEdit.GetBookMark(BookMark: Integer; var X, Y: Integer): Boolean;

begin
  Result := False;
  if Bookmark in [0..9] then
  begin
    X := FBookmarks[Bookmark].X;
    Y := FBookmarks[Bookmark].Y;
    Result := True;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

function TCustomUniCodeEdit.GetCanRedo: Boolean;

begin
  Result := (eoUseUndoRedo in FOptions) and FUndoList.CanRedo;
end;

//----------------------------------------------------------------------------------------------------------------------

function TCustomUniCodeEdit.GetCanUndo: Boolean;

begin
  Result := (eoUseUndoRedo in FOptions) and FUndoList.CanUndo;
end;

//----------------------------------------------------------------------------------------------------------------------

function TCustomUniCodeEdit.GetCaretXY: TPoint;

begin
  Result := Point(FCaretX, FCaretY);
end;

//----------------------------------------------------------------------------------------------------------------------

function TCustomUniCodeEdit.GetFont: TFont;

begin
  Result := inherited Font;
end;

//----------------------------------------------------------------------------------------------------------------------

function TCustomUniCodeEdit.GetLineEnd: Cardinal;

begin
  Result := GetLineEnd(FCaretY);
end;

//----------------------------------------------------------------------------------------------------------------------

function TCustomUniCodeEdit.GetLineEnd(Index: Cardinal): Cardinal;

// calculates the first unused column of the line given by Index

var
  I: Integer;

begin
  CalcCharWidths(Index);
  Result := 0;
  for I := 0 to High(FCharWidthArray) do
    Inc(Result, FCharWidthArray[I]);
  Result := Result div Cardinal(FCharWidth);
end;

//----------------------------------------------------------------------------------------------------------------------

function TCustomUniCodeEdit.GetLineText: WideString;

// Returns the text of the current line.
// If there is no line currently the one is created implicitely.

var
  WasModified: Boolean;

begin
  if FContent.Count = 0 then
  begin
    WasModified := Modified;
    FContent.AddLine('');
    // It's a dummy line we just have inserted hence we should restore the
    // previous modification state.
    Modified := WasModified;
  end;
  Result := FContent[FCaretY].Text;
end;

//----------------------------------------------------------------------------------------------------------------------

function TCustomUniCodeEdit.GetMaxRightChar: Integer;

begin
  Result := FMaxRightPos div FCharWidth;
end;

//----------------------------------------------------------------------------------------------------------------------

function TCustomUniCodeEdit.GetMaxUndo: Integer;

begin
  Result := FUndoList.MaxUndo;
end;

//----------------------------------------------------------------------------------------------------------------------

function TCustomUniCodeEdit.GetSelectedText: WideString;

var
  BB, BE: TPoint;
  BeginX,
  EndX: Integer;
  First, Last: Integer;
  Helper: TWideStringList;

begin
  Result := '';
  if SelectionAvailable then
  begin
    Helper := TWideStringList.Create;
    BB := BlockBegin;
    BeginX := ColumnToCharIndex(BB);
    BE := BlockEnd;
    EndX := ColumnToCharIndex(BE);
    if BB.Y = BE.Y then
      Result := Copy(FContent[BB.Y].Text, BeginX + 1, EndX - BeginX)
    else
    begin
      First := BB.Y;
      Last := BE.Y;
      Helper.Add(Copy(FContent[First].Text, BeginX + 1, Length(FContent[First].Text)));
      Inc(First);
      while First < Last do
      begin
        Helper.Add(FContent[First].Text);
        Inc(First);
      end;
      Result := Helper.Text;
      if First < FContent.Count then
        Result := Result + WideCRLF + Copy(FContent[First].Text, 1, EndX);
    end;
    Helper.Free;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

function TCustomUniCodeEdit.GetSelectionAvailable: Boolean;

begin
  Result := ((FBlockBegin.X <> FBlockEnd.X) or (FBlockBegin.Y <> FBlockEnd.Y)) and (FContent.Count > 0);
end;

//----------------------------------------------------------------------------------------------------------------------

function TCustomUniCodeEdit.GetSelEnd: Integer;

var
  Loop: Integer;
  X, Y: Integer;

begin
  X := BlockEnd.X;
  Y := BlockEnd.Y;

  Result := 0;
  Loop := 0;
  while Loop < (Y - 1) do
  begin
    Inc(Result, Length(FContent[Loop].Text) + 2);
    Inc(Loop);
  end;
  Result := Result + X;
end;

//----------------------------------------------------------------------------------------------------------------------

function TCustomUniCodeEdit.GetSelStart: Integer;

var
  Loop: Integer;
  X, Y: Integer;

begin
  Result := 0;

  if SelectionAvailable then
  begin
    X := FBlockBegin.X;
    Y := FBlockBegin.Y;  
    Loop := 0;

    while Loop < Y do
    begin
      Inc(Result, Length(FContent[Loop].Text) + 2);
      Inc(Loop);
    end;

    Result := Result + Min(X, Length(FContent[Loop].Text) + 2);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

function TCustomUniCodeEdit.GetText: WideString;

begin
  Result := FContent.Text;
end;

//----------------------------------------------------------------------------------------------------------------------

function TCustomUniCodeEdit.GetTopLine: Integer;

begin
  Result := -FOffsetY + 1;
end;

//----------------------------------------------------------------------------------------------------------------------

function TCustomUniCodeEdit.GetWordAndBounds(Pos: TPoint; var BB, BE: TPoint): WideString;

// Returns the word and its boundaries at position Pos. All positions are given
// as column/row values.
// Note: Compare Pos with BB and BE to determine whether the word is really at the
//       given position or has been automatically choosen because there was no word
//       at this position.

var
  CX: Integer;
  Line: WideString;
  IdChars: TIdentChars;
  Len: Integer;

begin
  // make sure the given position is valid
  if Pos.Y < 0 then
    Pos.Y := 0;
  if Pos.Y >= FContent.Count then
    Pos.Y := FContent.Count - 1;
  if Pos.X < 0 then
    Pos.X := 0;

  // determine WideString to be searched through
  if Pos.Y > -1 then
    Line := FContent[Pos.Y].Text
  else
    Line := '';
  Len := Length(Line);
  BB.Y := Pos.Y;
  BE.Y := Pos.Y;

  if Len > 0 then
  begin
    // setup initial states
    CX := ColumnToCharIndex(Pos);
    if CX > Len - 1 then
      CX := Len - 1;
    if FHighLighter <> nil then
      IdChars := FHighLighter.IdentChars
    else
      IDchars := [#33..#255];

    // four cases are to be considered:
    // 1. IdChar at current position
    if IsIdentChar(Line[CX + 1], IdChars) then
    begin
      // find start of word
      BB.X := CX;
      while (BB.X > 0) and IsIdentChar(Line[BB.X], IdChars) do
        Dec(BB.X);
      // find end of word
      BE.X := CX + 1;
      while (BE.X < Len) and IsIdentChar(Line[BE.X + 1], IdChars) do
        Inc(BE.X);
      // copy word
      Result := Copy(Line, BB.X + 1, BE.X - BB.X);
    end
    else
    begin
      // 2. no ID char at current position, so search to the left
      BE.X := CX;
      while (BE.X > 0) and not IsIdentChar(Line[BE.X], IdChars) do
        Dec(BE.X);
      if BE.X > 0 then
      begin
        CX := BE.X;
        // find start of word
        while (CX > 0) and IsIdentChar(Line[CX], IdChars) do
          Dec(CX);
        BB.X := CX;
        Result := Copy(Line, BB.X + 1, BE.X - BB.X);
      end
      else
      begin
        // 3. no ID char found to the left, so search to the right
        BB.X := CX + 1;
        while (BB.X < Len) and not IsIdentChar(Line[BB.X + 1], IdChars) do
          Inc(BB.X);
        if BB.X < Len then
        begin
          // find end of word
          BE.X := BB.X + 1;
          while (BE.X < Len) and IsIdentChar(Line[BE.X + 1], IdChars) do
            Inc(BE.X);
          Result := Copy(Line, BB.X + 1, BE.X - BB.X);
        end
        else
        begin
          // 4. nothing found, return all we have
          BB.X := 0;
          BE.X := Len;
          Result := Line;
        end;
      end;
    end;

    // finally turn char indices into columns
    BB.X := CharIndexToColumn(BB);
    BE.X := CharIndexToColumn(BE);
  end
  else
  begin
    Result := '';
    BB.X := 0;
    BE.X := 0;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.GotoBookMark(BookMark: Integer);

begin
  if BookMark in [0..9] then
  begin
    CaretXY := Point(FBookMarks[BookMark].X, FBookMarks[BookMark].Y);
    EnsureCursorPosVisible;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.HideCaret;

begin
  if FCaretVisible then
    FCaretVisible := not Windows.HideCaret(Handle);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.InitializeCaret;

var
  ct: TCaretType;
  cw, ch: Integer;

begin
  if not (eoReadOnly in FOptions) or
    (eoShowCursorWhileReadOnly in FOptions) then
  begin
    // CreateCaret automatically destroys the previous one, so we don't have to
    // worry about cleaning up the old one here with DestroyCaret.
    // Ideally, we will have properties that control what these two carets look like.
    HideCaret;
    if eoInserting in FOptions then
      ct := FInsertCaret
    else
      ct := FOverwriteCaret;
    case ct of
      ctHorizontalLine:
        begin
          cw := FCharWidth;
          ch := 2;
          FCaretOffset := Point(0, FTextHeight - 2);
        end;
      ctHalfBlock:
        begin
          cw := FCharWidth;
          ch := (FTextHeight - 2) div 2;
          FCaretOffset := Point(0, ch);
        end;
      ctBlock:
        begin
          cw := FCharWidth;
          ch := FTextHeight - 2;
          FCaretOffset := Point(0, 0);
        end;
    else // ctVerticalLine
      cw := 2;
      ch := FTextHeight - 2;
      FCaretOffset := Point(0, 0);
    end;
    CreateCaret(Handle, 0, cw, ch);
    UpdateCaret;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.InsertText(const Value: WideString);

var
  I,
  Len: Integer;
  TempString,
  TempString2: WideString;
  Helper: WideString;
  MaxChars: Integer;
  Truncated: Boolean;
  CX: Integer;
  Temp: TWideStringList;
  TempY: Integer;

begin
  if Value <> '' then
  begin
    Temp := TWideStringList.Create;
    try
      Truncated := False;
      Temp.Text := Value + WideCR + WideLF;
      CX := ColumnToCharIndex(FCaretX);

      TempString := LineText;
      Len := CX - Length(TempString);

      if Temp.Count = 1 then
      begin
        if Len > 0 then
          TempString := TempString + WideStringOfChar(' ', Len);
        Insert(Temp[0], TempString, CX + 1);
        LineText := TempString;
        CaretX := CharIndexToColumn(CX + Length(Temp[0]));
      end
      else
      begin
        if Len > 0 then
        begin
          Helper := WideStringOfChar(' ', Len);
          TempString := TempString + Helper;
        end;
        TempString := Copy(TempString, 1, CX) + Temp[0];
        TempString2 := Copy(LineText, CX + 1, Length(LineText));
        MaxChars := FMaxRightPos div FCharWidth;
        if Length(TempString) > MaxChars then
        begin
          Truncated := True;
          TempString := Copy(TempString, 1, MaxChars);
        end;
        FContent[FCaretY].Text := TempString;
        TempY := FCaretY + 1;
        for I := 1 to Temp.Count - 2 do
        begin
          if Length(Temp[I]) > MaxChars then
          begin
            Truncated := True;
            Temp[I] := Copy(Temp[I], 1, MaxChars);
          end;
          FContent.InsertLine(TempY, Temp[I]);
          Inc(TempY);
        end;
        // Relocate bookmarks.
        for I := 0 to 9 do
          if FBookmarks[I].Y >= CaretY then
            SetBookmark(I, FBookmarks[I].X, FBookmarks[I].Y + Temp.Count - 1);


        FContent.InsertLine(TempY, Temp[Temp.Count - 1] + TempString2);
        if Assigned(FHighlighter) then
          ScanFrom(FCaretY);
        CaretXY := Point(Length(Temp[Temp.Count - 1]), TempY);
      end;
      if Truncated then
        MessageDlg('One or more lines have been truncated.!', mtWarning, [mbOk], 0);
    finally
      Temp.Free;
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.InvalidateLine(Index: Integer);

var
  R: TRect;

begin
  R := Rect(0, (Index + FOffsetY) * FTextHeight, ClientWidth, (Index + FOffsetY + 1) * FTextHeight);
  InvalidateRect(Handle, @R, False);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.InvalidateLines(Start, Stop: Integer);

var
  R: TRect;

begin
  if Start <= Stop then
  begin
    Start := Start + FOffsetY;
    if Start < 0 then
      Start := 0;
    Stop := Stop + FOffsetY + 1;
    if Stop > FLinesInWindow then
      Stop := FLinesInWindow;
    // consider partially visible line
    if (ClientHeight mod FLinesInWindow) <> 0 then
      Inc(Stop);
    R := Rect(0, Start * FTextHeight, ClientWidth, Stop * FTextHeight);
  end
  else
  begin
    Stop := Stop + FOffsetY;
    if Stop < 0 then
      Stop := 0;
    Start := Start + FOffsetY + 1;
    if Start > FLinesInWindow then
      Start := FLinesInWindow;
    // consider partially visible line
    if (ClientHeight mod FLinesInWindow) <> 0 then
      Inc(Start);
    R := Rect(0, Stop * FTextHeight, ClientWidth, Start * FTextHeight);
  end;
  InvalidateRect(Handle, @R, False);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.InvalidateToBottom(Index: Integer);

var
  R: TRect;

begin
  R := Rect(0, (Index + FOffsetY) * FTextHeight, ClientWidth, ClientHeight);
  InvalidateRect(Handle, @R, False);
end;

//----------------------------------------------------------------------------------------------------------------------

function TCustomUniCodeEdit.IsBookmark(BookMark: Integer): Boolean;

var
  X, Y: Integer;

begin
  Result := GetBookMark(BookMark, X, Y);
end;

//----------------------------------------------------------------------------------------------------------------------

function TCustomUniCodeEdit.IsIdentChar(const AChar: WideChar; IdChars: TIdentChars): Boolean;

// Compatibility routine to check whether a given character is in the provided (SBCS) characters list.
// Works only if AChar is in the ANSI code page (value is < $100).
// Consider this function as being temporary until highlighters are ready which can deal with Unicode.

begin
  Result := Char(AChar) in IdChars;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.KeyDown(var Key: Word; Shift: TShiftState);

var
  Data: Pointer;
  Cmd: TEditorCommand;

begin
  inherited;

  Data := nil;
  try
    Cmd := TranslateKeyCode(Key, Shift, Data);
    if Cmd <> ecNone then
    begin
      Key := 0;
      CommandProcessor(Cmd, WideNull, Data);
    end;
  finally
    if Data <> nil then
      FreeMem(Data);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.KeyPress(var Key: Char);

var
  CW: WideChar;

begin
  inherited;

  CW := KeyUnicode(Key);
  CommandProcessor(ecChar, CW, nil);
end;

//----------------------------------------------------------------------------------------------------------------------

function TCustomUniCodeEdit.LastWordPos: TPoint;

var
  CX, CY: Integer;
  X: Integer;
  FoundNonWhiteSpace: Boolean;
  Len: Integer;
  Temp: WideString;
  IdChars: TIdentChars;

begin
  Result := CaretXY;
  Len := Length(LineText);
  CX := ColumnToCharIndex(FCaretX);
  if CX > Len then
    CX := Len;
  if CX > 0 then
    CY := FCaretY
  else
  begin
    if FCaretY < 1 then
    begin
      Result := Point(0, 0);
      Exit;
    end
    else
    begin
      Result := Point(GetLineEnd(FCaretY - 1), FCaretY - 1);
      Exit;
    end;
  end;

  if FHighLighter <> nil then
    IdChars := FHighLighter.IdentChars
  else
    IDchars := [#33..#255];

  Temp := FContent[CY].Text;
  FoundNonWhiteSpace := IsIdentChar(Temp[CX], IdChars);
  if not FoundNonWhiteSpace then
    for X := CX downto 2 do
      if IsIdentChar(Temp[X - 1], IdChars) then
      begin
        CX := X - 1;
        FoundNonWhiteSpace := True;
        Break;
      end;

  if FoundNonWhiteSpace then
  begin
    FoundNonWhiteSpace := False;
    for X := CX downto 2 do
      if not IsIdentChar(Temp[X - 1], IdChars) then
      begin
        Result := Point(X - 1, CY);
        FoundNonWhiteSpace := True;
        Break;
      end;

    if not FoundNonWhiteSpace then
      Result := Point(0, CY);
  end
  else
  begin
    Dec(CY);
    if CY = -1 then
      Result := Point(0, 0)
    else
      Result := Point(length(FContent[CY].Text), CY);
  end;
  Result.X := CharIndexToColumn(Result);
end;

//----------------------------------------------------------------------------------------------------------------------

function TCustomUniCodeEdit.LeftSpaces(const S: WideString): Cardinal;

// counts the number of white spaces at the beginning of the WideString;
// a tab is considered as one space here

var
  Run: PWideChar;

begin
  Run := PWideChar(S);
  while UnicodeIsWhiteSpace(Word(Run^)) do
    Inc(Run);
  Result := Run - PWideChar(S);
end;

//----------------------------------------------------------------------------------------------------------------------

function TCustomUniCodeEdit.LineFromPos(Pos: TPoint): Integer;

// returns the index of the line at position Pos (which must be given in client coordinates)

begin
  Result := -FOffsetY + (Pos.Y div FTextHeight) + 1;
  if Result < 0 then
    Result := 0;
  if Result > FContent.Count - 1 then
    Result := FContent.Count - 1;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.LineNumberFontChanged(Sender: TObject);

begin
  if HandleAllocated then
  begin
    InvalidateRect(Handle, @FGutterRect, False);
    UpdateWindow(Handle);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.LinesChanged;

begin
  if HandleAllocated then
  begin
    UpdateScrollBars;
    if Assigned(FHighlighter) then
      ScanFrom(FCaretY);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.Loaded;

begin
  inherited Loaded;
  UpdateScrollBars;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.LoadFromFile(const FileName: WideString; TextFormat: TTextFormat);

var
  Stream: TFileStream;

begin
  Stream := TFileStream.Create(FileName, fmOpenRead);
  try
    FContent.LoadFromStream(Stream, TextFormat);
    CaretX := 0;
    CaretY := 0;
    UpdateScrollbars;
    Invalidate;
  finally
    Stream.Free;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);

var
  Position: TPoint; // Postion in line/column format.

begin
  inherited;

  if Button = mbLeft then
  begin
    if not Focused then
    begin
      // Focus change. Don't use the SetFocus method as this does not work for MDI windows or embedded OLE
      // controls.
      if not Focused and CanFocus then
        Windows.SetFocus(Handle);
      // Don't delete selection if the edit wasn't focused previously.
      if SelectionAvailable then
        Exit;
    end;

    Position := PositionFromPoint(X, Y);
    if PtInRect(FGutterRect, Point(X, Y)) then
      DoGutterMouseDown(Button, Shift, X, Y, Position.Y)
    else
    begin
      // MouseDown will also be called when the mouse has been clicked twice (double click),
      // hence we can handle the triple click stuff herein too
      if ssDouble in Shift then
      begin
        FLastDblClick := GetTickCount;
        if FContent.Count > 0 then
          SetWordBlock(CaretXY);
        FMultiClicked := True;
      end
      else
      begin
        if (eoTripleClicks in FOptions) and (Shift = [ssLeft]) and (FLastDblClick > 0) then
        begin
          if (GetTickCount - FLastDblClick) < FDoubleClickTime then
          begin
            TripleClick;
            Exit;
          end;
          FLastDblClick := 0;
        end;

        SetCapture(Handle);
        // keep last cursor position to restore it in case of cancelled drag operation
        FLastCaret := CaretXY;
        ComputeCaret(X, Y);
        if ((Shift - [ssCtrl]) = [ssLeft]) and PosInSelection(Position) then
          BeginDrag(False)
        else
        begin
          if ssShift in Shift then
            SetBlockEnd(CaretXY)
          else
            SetBlockBegin(CaretXY);
        end;
      end;
    end;
  end
  else
    FScrollTimer.enabled := False;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.MouseMove(Shift: TShiftState; X, Y: Integer);

var
  P: TPoint;

begin
  inherited;
  if X > FGutterWidth then
  begin
    P := PositionFromPoint(X, Y);
    // don't use ssLeft in Shift as this parameter is not always reliable (mouse wheel!)
    if (csLButtonDown in ControlState) and not FMultiClicked then
    begin
      if not FScrollTimer.Enabled
        and ((FLastCaret.X - P.X <> 0) or (FLastCaret.Y - P.Y <> 0)) then
        FScrollTimer.Enabled := True;
    end
    else
      FScrollTimer.enabled := False;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);

begin
  FMultiClicked := False;
  FScrollTimer.enabled := False;
  ReleaseCapture;
  inherited;
end;

//----------------------------------------------------------------------------------------------------------------------

function TCustomUniCodeEdit.NextCharPos(ThisPosition: Cardinal; ForceNonDefault: Boolean): Cardinal;

begin
  Result := NextCharPos(ThisPosition, FCaretY, ForceNonDefault);
end;

//----------------------------------------------------------------------------------------------------------------------

function TCustomUniCodeEdit.NextCharPos(ThisPosition, Line: Cardinal; ForceNonDefault: Boolean): Cardinal;

// Determines the next character position for the given line depending on the
// column given by ThisPosition. The returned index can directly be used to set the caret.

var
  PixelPos, Run: Cardinal;
  I: Cardinal;

begin
  CalcCharWidths(Line);
  // turn column into pixel position
  PixelPos := ThisPosition * Cardinal(FCharWidth);
  if ((eoCursorThroughTabs in FOptions) and not ForceNonDefault) or
    (ThisPosition >= GetLineEnd(Line)) then
    Result := ThisPosition + 1
  else
  begin
    // The task is to find the first column which corresponds directly to a
    // character (which is not always the case, eg. tabs) and is larger than
    // the given position.
    Run := 0;
    // sum up character widths until we find a pixel position larger than the
    // given one
    for I := 0 to High(FCharWidthArray) do
    begin
      // values in the char width array always correspond to an actual character index
      if PixelPos < Run then
        Break
      else
        Inc(Run, FCharWidthArray[I]);
    end;
    Result := Run div Cardinal(FCharWidth);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

function TCustomUniCodeEdit.NextSmartTab(X, Y: Integer; SkipNonWhite: Boolean): Integer;

// Calculates the indent position starting with X and depending on the first
// line above Y which contains a white space char before X.
// X and Result are column values!

var
  I: Integer;
  Len: Integer;
  Line: WideString;

begin
  // cursor at line 0?
  if Y = 0 then
    Result := X // no next tab position available
  else
  begin
    // find a line above the current line which size is greater than X
    repeat
      Dec(Y);
      I := GetLineEnd(Y);
      if I > X then
        Break;
    until Y = 0;

    // found a line?
    if I > X then
    begin
      Line := FContent[Y].Text;
      Len := Length(Line);
      I := ColumnToCharIndex(Point(X, Y));
      // skip non-white spaces if required
      if SkipNonWhite then
        while (I < Len) and not UnicodeIsWhiteSpace(Word(Line[I + 1])) do
          Inc(I);

      // now skip any white space character
      while (I < Len) and UnicodeIsWhiteSpace(Word(Line[I + 1])) do
        Inc(I);
      Result := CharIndexToColumn(Point(I, Y));
    end
    else
      Result := X;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

function TCustomUniCodeEdit.NextTabPos(Index: Integer): Integer;

// calculates next tab stop position

begin
  Result := (Index div FTabSize + 1) * FTabSize;
end;

//----------------------------------------------------------------------------------------------------------------------

function TCustomUniCodeEdit.NextWordPos: TPoint;

// Calculates the caret position of the next word start or the line end if there's
// no further word on the current line and the caret is not already on the line's end.

var
  CX, CY: Integer;
  X: Integer;
  FoundWhiteSpace: Boolean;
  Len: Integer;
  Temp: WideString;
  IdChars: TIdentChars;

begin
  Result := CaretXY;
  // ensure at least an empty line is there
  Temp := LineText;
  CX := ColumnToCharIndex(FCaretX) + 1;
  if CX < Length(FContent[FCaretY].Text) then
    CY := FCaretY
  else
  begin
    CY := FCaretY + 1;
    if CY > FContent.Count - 1 then
      Exit;
    if WideTrim(FContent[CY].Text) = '' then
    begin
      Result := Point(0, CY);
      Exit;
    end;
    CX := 1;
  end;

  if FHighLighter <> nil then
    IdChars := FHighLighter.IdentChars
  else
    IDchars := [#33..#255];

  Temp := FContent[CY].Text;
  Len := Length(Temp);
  FoundWhiteSpace := (not IsIdentChar(Temp[CX], IdChars)) or (CY <> FCaretY);
  if not FoundWhiteSpace then
    for X := CX to Len do
      if not IsIdentChar(Temp[X], IdChars) then
      begin
        CX := X;
        FoundWhiteSpace := True;
        Break;
      end;

  if FoundWhiteSpace then
  begin
    FoundWhiteSpace := False;
    for X := CX to Len do
      if IsIdentChar(Temp[X], IdChars) then
      begin
        Result := Point(X - 1, CY);
        FoundWhiteSpace := True;
        Break;
      end;
  end;

  if not FoundWhiteSpace then
    Result := Point(Len, CY);
  Result.X := CharIndexToColumn(Result);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.Notification(AComponent: TComponent; Operation: TOperation);

begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) and (AComponent = FHighLighter) then
  begin
    FHighLighter := nil;
    Invalidate;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.OnScrollTimer(Sender: TObject);

// do autoscrolling while mouse down and set selection appropriately

var
  SP,
    CP: TPoint;
  NewCaret: TPoint;
  R: TRect;

begin
  Inc(FUpdateCount);
  // get current caret postion
  GetCursorPos(SP);
  GetWindowRect(Handle, R);
  InflateRect(R, -FTextHeight, -FTextHeight);
  CP := ScreenToClient(SP);
  if not PtInRect(R, SP) then
  begin
    // scroll only if mouse is outside the specified rectangle (which is a bit smaller
    // than the client area in order to allow to stop scrolling if the mouse is moved
    // over an other edit to drop text to)
    if SP.X < R.Left then
      OffsetX := FOffsetX + (R.Left - SP.X);
    if SP.X > R.Right then
      OffsetX := FOffsetX + (R.Right - SP.X);
    if SP.Y < R.Top then
      OffsetY := FOffsetY + (R.Top - SP.Y);
    if SP.Y > R.Bottom then
      OffsetY := FOffsetY + (R.Bottom - SP.Y);
  end;
  NewCaret := PositionFromPoint(CP.X, CP.Y);
  if (not (eoScrollPastEOL in FOptions)) and
    (NewCaret.Y > FContent.Count - 1) then // added by MikeZ
    NewCaret.Y := FContent.Count - 1; // added by MikeZ

  Dec(FUpdateCount);
  if not PointsAreEqual(CaretXY, NewCaret) then
  begin
    if not FDropTarget then
      SetBlockEnd(NewCaret);
    CaretXY := NewCaret;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.Paint;

var
  WasVisible: Boolean;
  SaveIndex: Integer;

begin
  WasVisible := FCaretVisible;
  if WasVisible then
    HideCaret;

  Canvas.Lock;
  try
    SaveIndex := SaveDC(Canvas.Handle);
    ExcludeClipRect(Canvas.Handle, 0, 0, FGutterRect.Right, FGutterRect.Bottom);
    Canvas.Font := Font;
    if FContent.Count > 0 then
    begin
      if Assigned(FHighLighter) and
        (eoUseSyntaxHighlighting in FOptions) then
        PaintHighlighted(Canvas)
      else
        PaintText(Canvas);
    end
    else
      with Canvas do
      begin
        Brush.Color := Self.Color;
        FillRect(ClipRect);

        // draw the right margin marker
        Pen.Color := FMarginColor;
        Pen.Width := 1;
        MoveTo(FRightMargin * FCharWidth + FGutterRect.Right + 1 + FOffsetX, 0);
        LineTo(FRightMargin * FCharWidth + FGutterRect.Right + 1 + FOffsetX, ClientHeight);
      end;

    RestoreDC(Canvas.Handle, SaveIndex);
    
    // Draw gutter after the content as it needs already validated lines.
    PaintGutter(Canvas);
    if Assigned(FOnPaint) then
      FOnPaint(Self, Canvas);
    if WasVisible then
      ShowCaret;
  finally
    Canvas.Unlock;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.PaintGutter(TextCanvas: TCanvas);

var
  I: Integer;
  UpdateRect, R: TRect;
  StartLine,
  EndLine,
  LineIndex,
  YOffset: Integer;
  Number: string;
  LineBase: Integer;

begin
  // Draw only if (part of) gutter rect is invalid.
  if IntersectRect(UpdateRect, TextCanvas.ClipRect, FGutterRect) then
  begin
    with TextCanvas do
    begin
      Brush.Color := FGutterColor;
      UpdateRect.Left := FGutterRect.Right - 5;
      FillRect(UpdateRect);

      UpdateRect.Right := FGutterRect.Right - 3;
      Pen.Color := clBtnHighlight;
      Pen.Style := psSolid;
      Pen.Width := 1;
      MoveTo(UpdateRect.Right, 0);
      LineTo(UpdateRect.Right, UpdateRect.Bottom);

      Inc(UpdateRect.Right);
      Pen.Color := clBtnShadow;
      MoveTo(UpdateRect.Right, 0);
      LineTo(UpdateRect.Right, UpdateRect.Bottom);

      Inc(UpdateRect.Right, 2);
      Pen.Color := Self.Color;
      Pen.Width := 2;
      MoveTo(UpdateRect.Right, 0);
      LineTo(UpdateRect.Right, UpdateRect.Bottom);

      StartLine := (UpdateRect.Top div FTextHeight) - FOffsetY;
      EndLine := (UpdateRect.Bottom div FTextHeight) - FOffsetY + 1;
      if EndLine > FContent.Count - 1 then
        EndLine := FContent.Count - 1;
      YOffset := (UpdateRect.Top div FTextHeight) * FTextHeight;
      R := Rect(0, YOffset, FGutterRect.Right - 5, YOffset + FTextHeight);

      if eoLineNumbers in FOptions then
      begin
        Font.Assign(FLineNumberFont);

        // Make the line numbers showing one or zero based, depending on options.
        LineBase := Ord(not (eoLineNumbersZeroBased in FOptions));
        for LineIndex := StartLine + LineBase to EndLine + LineBase do
        begin
          Number := IntToStr(LineIndex);
          FillRect(R);
          DrawText(Handle, PChar(Number), Length(Number), R, DT_VCENTER or DT_RIGHT or DT_SINGLELINE);
          OffsetRect(R, 0, FTextHeight);
        end;
      end;

      // Let the lines draw their images.
      for LineIndex := StartLine to EndLine do
      begin
        FContent[LineIndex].DrawMarkers(LineIndex, TextCanvas, 0, YOffset);
        Inc(YOffset, FTextHeight);
      end;

      // Fill rest of gutter rect if necessary.
      Brush.Color := FGutterColor;
      if R.Top < UpdateRect.Bottom then
      begin
        R.Bottom := UpdateRect.Bottom;
        FillRect(R);
      end;

      if BookMarkOptions.GlyphsVisible then
      begin
        for I := 0 to 9 do
          with FBookMarks[I] do
            if Visible and (Y >= StartLine) and (Y <= EndLine) then
            begin
              FInternalBMList.Draw(TextCanvas, FBookmarkOptions.LeftMargin, (Y + FOffsetY) * FTextHeight, I);
            end;
      end;

      ExcludeClipRect(Handle, 0, 0, FGutterRect.Right, FGutterRect.Bottom);
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.PaintHighlighted(TextCanvas: TCanvas);

// main drawing routine to paint the control if syntax highlighting is used

var
  YOffset,
  LineIndex,
  PaintOffset,
  OldOffset,
  StartIndex,
  StopIndex,
  TempIndex: Integer;
  BB, BE: PPoint;
  TokenData: TTokenData;

  I,
  StartLine,
  EndLine: Integer;
  UpdateRect,
  Dummy,
  R: TRect;
  SelStart,
  SelEnd: Integer;
  CurrentStyle: Cardinal; // to optimize drawing selected/unselected text
  ShowSelection,
  ShowLineBreak: Boolean;
  LineBase: PWideChar;

  CustomStyle: IUCELineStyle;
  
  //--------------- local functions -------------------------------------------

  procedure PaintToken(Start, Len: Cardinal);

  // paints the given string at current paint offset, no control chars are painted,
  // text attributes must be set up already on call;
  // side effects: current paint offset will be updated

  var
    Index,
    Counter: Cardinal;
    P: PWideChar;

  begin
    // replace non-printable characters,
    // calculate update rect for this token (-part) btw.
    P := LineBase + Start;
    OldOffset := PaintOffset;
    Index := Start;
    Counter := Len;
    while Counter > 0 do
    begin
      if P^ < ' ' then
        P^ := ' ';
      Inc(P);
      Inc(PaintOffset, FCharWidthArray[Index]);
      if PaintOffset > UpdateRect.Right then
      begin
        // The token is longer than what we need to draw. Stop looping here
        // and draw only the necessary part.
        Len := Index - Start + 1;
        Break;
      end;
      Inc(Index);
      Dec(Counter);
    end;

    R := Rect(OldOffset, YOffset, PaintOffset, YOffset + FTextHeight);
    // finally paint text
    ExtTextOutW(TextCanvas.Handle, OldOffset, YOffset, ETO_OPAQUE, @R,
      LineBase + Start, Len, @FCharWidthArray[Start]);
  end;

  //---------------------------------------------------------------------------

  procedure PaintTokenWithControlChars(Start, Len: Cardinal);

  // paints the given WideString at current paint offset, considers control characters,
  // text attributes must be set up already on call;
  // side effects: current paint offset will be updated

  var
    I,
    Counter: Cardinal;
    Run,
    TabPos: PWideChar;

  begin
    // replace space characters
    Run := LineBase + Start;
    Counter := Len;
    while Counter > 0 do
    begin
      if Run^ = ' ' then
        Run^ := DefaultSpaceGlyph;
      Inc(Run);
      Dec(Counter);
    end;

    Run := LineBase + Start;
    Counter := Len;
    // TODO: Draw only as much characters as fit in the update rect.
    with TextCanvas do
      while Counter > 0 do
      begin
        // find TAB character
        TabPos := StrScanW(Run, WideTabulator);
        // something to draw before tab?
        if Assigned(TabPos) then
          I := Min(Counter, TabPos - Run)
        else
          I := Counter;
        if I > 0 then
        begin
          OldOffset := PaintOffset;
          Inc(PaintOffset, I * Cardinal(FCharWidth));
          R := Rect(OldOffset, YOffset, PaintOffset, YOffset + FTextHeight);
          ExtTextOutW(Handle, OldOffset, YOffset, ETO_OPAQUE, @R, Run, I, @FCharWidthArray[Start]);
          Inc(Start, I);
          Dec(Counter, I);
          Inc(Run, I);
        end;

        // tab to draw?
        if Assigned(TabPos) then
        begin
          while (TabPos^ = WideTabulator) and (Counter > 0) do
          begin
            OldOffset := PaintOffset;
            Inc(PaintOffset, FCharWidthArray[Start]);
            R := Rect(OldOffset, YOffset, PaintOffset, YOffset + FTextHeight);

            SetTextAlign(Handle, TA_CENTER);
            ExtTextOutW(Handle, (PaintOffset + OldOffset) div 2, YOffset, ETO_OPAQUE, @R, DefaultTabGlyph, 1, nil);
            SetTextAlign(Handle, TA_LEFT);
            Inc(Start);
            Dec(Counter);
            Inc(TabPos);
          end;
          Run := TabPos;
        end;
      end;
  end;

  //----------------------------------------------------------------------------

  procedure PaintLineBreak;

  begin
    R := Rect(PaintOffset, YOffset, PaintOffset + FCharWidth, YOffset + FTextHeight);
    if IntersectRect(Dummy, R, UpdateRect) then
      ExtTextOutW(TextCanvas.Handle, PaintOffset, YOffset, ETO_OPAQUE, @R, DefaultLineBreakGlyph, 1,
        Pointer(FCharWidthArray));
    Inc(PaintOffset, FCharWidth);
  end;

  //----------------------------------------------------------------------------

  procedure PickColors(Foreground, Background: TColor; Reversed: Boolean);

  // Selects the given colors into the target canvas. If CustomStyle is Assigned
  // then use this style instead of the given colors.
  // If Reversed is true then fore- and background colors of the custom style
  // (if assinged) are reversed.
  // If there is a custom style Assigned, which has its font style forcing flag set
  // then also font styles from this custom style are applied.

  begin
    with TextCanvas do
    begin
      if Assigned(CustomStyle) then
      begin
        if Reversed then
        begin
          Font.Color := CustomStyle.Background;
          Brush.Color := CustomStyle.Foreground;
        end
        else
        begin
          Font.Color := CustomStyle.Foreground;
          Brush.Color := CustomStyle.Background;
        end;
        if CustomStyle.ForceFontStyles then
          Font.Style := CustomStyle.FontStyles;
      end
      else
      begin
        Font.Color := Foreground;
        Brush.Color := Background;
      end;
    end;
  end;
  
  //--------------- end local functions ----------------------------------------

var
  S: WideString;
  
begin
  with TextCanvas do
  begin
    UpdateRect := ClipRect;

    // Optimize drawing by using the update area.
    StartLine := (UpdateRect.Top div FTextHeight) - FOffsetY;
    EndLine := (UpdateRect.Bottom div FTextHeight) - FOffsetY + 1;
    if EndLine > FContent.Count - 1 then
      EndLine := FContent.Count - 1;

    // The vertical position we will paint at. Start at top and increment by
    // FTextHeight for each iteration of the loop (see bottom of for loop).
    YOffset := (UpdateRect.Top div FTextHeight) * FTextHeight;

    // Tell highlighter which state it should use from now on.
    if StartLine < FContent.Count then
    begin
      // Scan all lines until the start line so we have a reliable lexer state.
      // If the start line is the very first line in the control then there is no
      // need to scan it. The highlighter will use the default start state in that case. 
      if (StartLine > FLastValidLine) and (StartLine > 0) then
      begin
        // Not yet scanned, so do it now.
        I := FLastValidLine;
        // Scan all lines up to the start line.
        repeat
          // TODO: The highlighters still use ANSI!
          // SetLine will reset the current lexer state so we have to set it afterwards again.
          FHighLighter.SetLine(FContent[I].Text);
          if I > 0 then
            FHighLighter.SetRange(FContent[I].LexerState);
          FHighlighter.Next;
          while not FHighLighter.EOL do
            FHighLighter.Next;
          Inc(I);
          // Keep the start state for this line.
          FContent[I].LexerState := FHighLighter.GetRange;
        until I = StartLine;
      end;
    end;
    
    // Determine correct start and end line of selection.
    BB := MinPointAsReference(FBlockBegin, FBlockEnd);
    BE := MaxPointAsReference(FBlockBegin, FBlockEnd);

    ShowLineBreak := (eoShowControlChars in FOptions) and not (eoScrollPastEOL in FOptions);

    // Determine in advance whether to show selection or not.
    // Note: We have four concurrent options determining when to show selection (apart
    //       from SelectionAvailable). These are: focus state, eoReadOnly, eoReadOnlySelection and
    //       eoHideSelection. Given these states one can derive a logic table with 16 results
    //       and from this table get a logical equation. Optimizing this eq. leads to
    //       the four conditions written and described below. If we name the mentioned
    //       states (in the same order as above) with A, B, C and D, respectively, then
    //       the resulting eq. is: Sel = AB or AC or BD or CD.
    // show selection if...
    ShowSelection := // selection is available and
      SelectionAvailable and (
        // edit is focused and not read only (then both selection opts have no influence) or
        (Focused and not (eoReadOnly in FOptions)) or
        // edit is focused and selection in r/o mode is allowed (then read only and
        // hide selection when unfocused don't matter) or
        (Focused and (eoReadOnlySelection in FOptions)) or
        // edit is not read only and selection will not be hidden when unfocused (then
        // being focused doesn't matter and r/o selection has no influence) or
        ((FOptions * [eoReadOnly, eoHideSelection]) = []) or
        // r/o selection is enabled and selection will not be hidden when unfocused
        // (then being focused or in r/o mode don't matter).
        ((eoReadOnlySelection in FOptions) and not (eoHideSelection in FOptions))
      );

    // Loop through the lines which need repainting.
    for LineIndex := StartLine to EndLine do
    begin
      // Tell highlighter what line we are working on. This will get the first token set up.
      S := FContent[LineIndex].Text;
      CalcCharWidths(LineIndex);

      FHighLighter.SetLine(S);
      if LineIndex > 0 then
        FHighLighter.SetRange(FContent[LineIndex].LexerState);

      // Advance highlighter to first token.
      FHighLighter.Next;
      LineBase := PWideChar(S);

      PaintOffset := FOffsetX + FGutterRect.Right + 1;

      // Get a potential custom style.
      CustomStyle := FContent[LineIndex].PeekStyle;

      // If no selection is to draw then go straight ahead.
      if (LineIndex < BB.Y) or (LineIndex > BE.Y) or not ShowSelection then
      begin
        while not FHighLighter.EOL and (PaintOffset < UpdateRect.Right) do
        begin
          TokenData := FHighLighter.GetTokenInfo;
          with TokenData do
          begin
            Font.Style := Style;
            PickColors(Foreground, Background, False);

            if eoShowControlChars in FOptions then
              PaintTokenWithControlChars(Position, Length)
            else
              PaintToken(Position, Length);
          end;
          FHighLighter.Next;
        end;

        if PaintOffset < UpdateRect.Right then
        begin
          // Determine the colors for the rest of the line.
          PickColors(Self.Font.Color, Self.Color, False);
          if ShowLineBreak then
            PaintLineBreak;
        end;
      end
      else
      begin
        // Selection parts are to be considered, so determine start and end position of
        // the selection on this particular line.
        if BB.Y < LineIndex then
          SelStart := 0
        else
          SelStart := ColumnToCharIndex(Point(BB.X, LineIndex));
        if BE.Y > LineIndex then
          SelEnd := FMaxRightPos // any large number makes it here
        else
          SelEnd := ColumnToCharIndex(Point(BE.X, LineIndex));

        CurrentStyle := 0;
        StopIndex := 0;
        while not FHighLighter.EOL do
        begin
          // Get the current token (string) to be painted.
          TokenData := FHighLighter.GetTokenInfo;
          // Start with the beginning of the token.
          StartIndex := 0;
          // Stop at the end of the token.
          StopIndex := TokenData.Length;
          Font.Style := TokenData.Style;

          // Paint up to selection start. That length will be the StopIndex
          // or the start of the selection mark, whichever is less.
          TempIndex := Min(StopIndex + TokenData.Position, SelStart) - StartIndex - TokenData.Position;
          if TempIndex > 0 then
          begin
            // set needed styles
            with TokenData do
            begin
              PickColors(Foreground, Background, False);
              CurrentStyle := 1;
              if eoShowControlChars in FOptions then
                PaintTokenWithControlChars(Position, TempIndex)
              else
                PaintToken(Position, TempIndex);
            end;
            // update the start index into the line text to skip past the
            // text we just painted.
            Inc(StartIndex, TempIndex);
          end;

          // Paint the selection text. That length will be the StopIndex or the
          // end of the selection mark, whichever is less.
          TempIndex := Min(StopIndex + TokenData.Position, SelEnd) - StartIndex - TokenData.Position;
          if TempIndex > 0 then // Have anything to paint?
          begin
            if CurrentStyle <> 2 then // other than selected style?
            begin
              // set the selection highlight colors
              with FSelectedColor do
                PickColors(Foreground, Background, True);
              CurrentStyle := 2;
            end;
            // Paint the selection text
            with TokenData do
              if eoShowControlChars in FOptions then
                PaintTokenWithControlChars(Position + StartIndex, TempIndex)
              else
                PaintToken(Position + StartIndex, TempIndex);
            // Update the start index into the line text to skip past the
            // text we just painted.
            Inc(StartIndex, TempIndex);
          end;

          // Paint the post-selection text, the length is whatever is left over.
          Font.Style := TokenData.Style;
          TempIndex := StopIndex - StartIndex;
          if TempIndex > 0 then
          begin
            if CurrentStyle <> 1 then // other than unselected style?
            begin
              // set needed styles
              with TokenData do
              begin
                PickColors(Foreground, Background, False);
              end;
              CurrentStyle := 1;
            end;
            with TokenData do
              if eoShowControlChars in FOptions then
                PaintTokenWithControlChars(Position + StartIndex, TempIndex)
              else
                PaintToken(Position + StartIndex, TempIndex);
          end;

          if CurrentStyle <> 2 then
            CurrentStyle := 0;
          FHighLighter.Next;
        end;

        // Prepare drawing of the rest of the line.
        if LineIndex < BE.Y then
        begin
          if Assigned(CustomStyle) then
            Brush.Color := CustomStyle.Foreground
          else
            Brush.Color := FSelectedColor.Background;
        end
        else
          if Assigned(CustomStyle) then
            Brush.Color := CustomStyle.Background
          else
            Brush.Color := Self.Color;
        if ShowLineBreak then
        begin
          if StopIndex < SelEnd then
          begin
            if Assigned(CustomStyle) then
              Font.Color := CustomStyle.Background
            else
              Font.Color := FSelectedColor.Foreground;
          end
          else
          begin
            if Assigned(CustomStyle) then
              Font.Color := CustomStyle.Foreground
            else
              Font.Color := Font.Color;
          end;
          PaintLineBreak;
        end;
      end;

      // Clear background of the rest of the line.
      FillRect(Rect(PaintOffset, YOffset, UpdateRect.Right, YOffset + FTextHeight));

      // Draw the right margin marker.
      Pen.Color := FMarginColor;
      Pen.Width := 1;
      MoveTo(FRightMargin * FCharWidth + FOffsetX + FGutterRect.Right + 1, YOffset);
      LineTo(FRightMargin * FCharWidth + FOffsetX + FGutterRect.Right + 1, YOffset + FTextHeight);

      // This line has been painted, increment the vertical offset and loop back
      // for the next line of text.
      Inc(YOffset, FTextHeight);

      // Store state value to avoid later rescans.
      if (LineIndex + 1) < FContent.Count then
      begin
        while not FHighLighter.EOL do
          FHighLighter.Next;
        FContent[LineIndex + 1].LexerState := FHighLighter.GetRange;
      end;
    end;

    // Finally erase all the space not covered by lines.
    if YOffset < UpdateRect.Bottom then
    begin
      Brush.Color := Self.Color;
      FillRect(Rect(UpdateRect.Left, YOffset, UpdateRect.Right, UpdateRect.Bottom));

      // Draw the right margin marker.
      Pen.Color := FMarginColor;
      Pen.Width := 1;
      MoveTo(FRightMargin * FCharWidth + FOffsetX + FGutterRect.Right + 1, YOffset);
      LineTo(FRightMargin * FCharWidth + FOffsetX + FGutterRect.Right + 1, UpdateRect.Bottom);
    end;

    if EndLine > FLastValidLine then
      FLastValidLine := EndLine;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.PaintText(TextCanvas: TCanvas);

// main drawing routine to paint the control if no syntax highlighting is used

var
  YOffset,
  LineIndex,
  PaintOffset,
  OldOffset,
  StartIndex,
  StopIndex,
  TempIndex: Integer;
  BB, BE: PPoint;
  TokenLen: Integer;

  StartLine,
  EndLine: Integer;
  UpdateRect,
  Dummy,
  R: TRect;
  SelStart,
  SelEnd: Integer;
  ShowSelection: Boolean;
  ShowLineBreak: Boolean;
  LineBase: PWideChar;

  CustomStyle: IUCELineStyle;
  
  //--------------- local functions -------------------------------------------

  procedure PaintToken(Start, Len: Cardinal);

  // paints the given WideString at current paint offset, no control chars are painted,
  // text attributes must be set up already on call;
  // side effects: current paint offset will be updated

  var
    Index,
      Counter: Cardinal;
    P: PWideChar;

  begin
    // replace non-printable characters,
    // calculate update rect for this token (-part) btw.
    P := LineBase + Start;
    OldOffset := PaintOffset;
    Index := Start;
    Counter := Len;
    while Counter > 0 do
    begin
      if P^ < ' ' then
        P^ := ' ';
      Inc(P);
      Inc(PaintOffset, FCharWidthArray[Index]);
      if PaintOffset > UpdateRect.Right then
      begin
        // The token is longer than what we need to draw. Stop looping here
        // and draw only the necessary part.
        Len := Index - Start + 1;
        Break;
      end;
      Inc(Index);
      Dec(Counter);
    end;

    R := Rect(OldOffset, YOffset, PaintOffset, YOffset + FTextHeight);
    // finally paint text
    ExtTextOutW(TextCanvas.Handle, OldOffset, YOffset, ETO_OPAQUE, @R, LineBase + Start, Len, @FCharWidthArray[Start]);
  end;

  //---------------------------------------------------------------------------

  procedure PaintTokenWithControlChars(Start, Len: Cardinal);

  // paints the given string at current paint offset, considers control characters,
  // text attributes must be set up already on call;
  // side effects: current paint offset will be updated

  var
    I,
      Counter: Cardinal;
    Run,
      TabPos: PWideChar;

  begin
    // replace space characters
    Run := LineBase + Start;
    Counter := Len;
    while Counter > 0 do
    begin
      if Run^ = ' ' then
        Run^ := DefaultSpaceGlyph;
      Inc(Run);
      Dec(Counter);
    end;

    Run := LineBase + Start;
    Counter := Len;
    with TextCanvas do
      while Counter > 0 do
      begin
        // find TAB character
        TabPos := StrScanW(Run, WideTabulator);
        // something to draw before tab?
        if Assigned(TabPos) then
          I := Min(Counter, TabPos - Run)
        else
          I := Counter;
        if I > 0 then
        begin
          OldOffset := PaintOffset;
          Inc(PaintOffset, I * Cardinal(FCharWidth));
          R := Rect(OldOffset, YOffset, PaintOffset, YOffset + FTextHeight);
          ExtTextOutW(Handle, OldOffset, YOffset, ETO_OPAQUE, @R, Run, I, @FCharWidthArray[Start]);
          Inc(Start, I);
          Dec(Counter, I);
          Inc(Run, I);
        end;

        // tab to draw?
        if Assigned(TabPos) then
        begin
          while (TabPos^ = WideTabulator) and (Counter > 0) do
          begin
            OldOffset := PaintOffset;
            Inc(PaintOffset, FCharWidthArray[Start]);
            R := Rect(OldOffset, YOffset, PaintOffset, YOffset + FTextHeight);

            SetTextAlign(Handle, TA_CENTER);
            ExtTextOutW(Handle, (PaintOffset + OldOffset) div 2, YOffset, ETO_OPAQUE, @R, DefaultTabGlyph, 1, nil);
            SetTextAlign(Handle, TA_LEFT);
            Inc(Start);
            Dec(Counter);
            Inc(TabPos);
          end;
          Run := TabPos;
        end;
      end;
  end;

  //---------------------------------------------------------------------------

  procedure PaintLineBreak;

  begin
    R := Rect(PaintOffset, YOffset, PaintOffset + FCharWidth, YOffset + FTextHeight);
    if IntersectRect(Dummy, R, UpdateRect) then
      ExtTextOutW(TextCanvas.Handle, PaintOffset, YOffset, ETO_OPAQUE, @R, DefaultLineBreakGlyph, 1,
        Pointer(FCharWidthArray));
    Inc(PaintOffset, FCharWidth);
  end;

  //----------------------------------------------------------------------------

  procedure PickColors(Foreground, Background: TColor; Reversed: Boolean);

  // Selects the given colors into the target canvas. If CustomStyle is Assigned
  // then use this style instead of the given colors.
  // If Reversed is true then fore- and background colors of the custom style
  // (if assinged) are reversed.
  // If there is a custom style Assigned, which has its font style forcing flag set
  // then also font styles from this custom style are applied.

  begin
    with TextCanvas do
    begin
      if Assigned(CustomStyle) then
      begin
        if Reversed then
        begin
          Font.Color := CustomStyle.Background;
          Brush.Color := CustomStyle.Foreground;
        end
        else
        begin
          Font.Color := CustomStyle.Foreground;
          Brush.Color := CustomStyle.Background;
        end;
        if CustomStyle.ForceFontStyles then
          Font.Style := CustomStyle.FontStyles;
      end
      else
      begin
        Font.Color := Foreground;
        Brush.Color := Background;
      end;
    end;
  end;
  
  //--------------- end local functions ----------------------------------------

begin
  with TextCanvas do
  begin
    Font := Self.Font;
    UpdateRect := ClipRect;
    // optimize drawing by using the update area
    StartLine := (UpdateRect.Top div FTextHeight) - FOffsetY;
    EndLine := (UpdateRect.Bottom div FTextHeight) - FOffsetY + 1;
    if EndLine > FContent.Count - 1 then
      EndLine := FContent.Count - 1;

    // The vertical position we will paint at. Start at top and increment by
    // FTextHeight for each iteration of the loop (see bottom of for loop).
    YOffset := (UpdateRect.Top div FTextHeight) * FTextHeight;

    // determine correct start and end line of selection
    BB := MinPointAsReference(FBlockBegin, FBlockEnd);
    BE := MaxPointAsReference(FBlockBegin, FBlockEnd);

    ShowLineBreak := (eoShowControlChars in FOptions) and not (eoScrollPastEOL in FOptions);

    // Determine in advance whether to show selection or not.
    // Note: We have four concurrent options determining when to show selection (apart
    //       from SelectionAvailable). These are: focus state, eoReadOnly, eoReadOnlySelection and
    //       eoHideSelection. Given these states one can derive a logic table with 16 results
    //       and from this table get a logical equation. Optimizing this eq. leads to
    //       the four conditions written and described below. If we name the mentioned
    //       states (in the same order as above) with A, B, C and D, respectively, then
    //       the resulting eq. is: Sel = AB or AC or BD or CD.
    // show selection if...
    ShowSelection := // selection is available and
      SelectionAvailable and (
         // edit is focused and not read only (then both selection opts have no influence) or
        (Focused and not (eoReadOnly in FOptions)) or
         // edit is focused and selection in r/o mode is allowed (then read only and
         // hide selection when unfocused don't matter) or
        (Focused and (eoReadOnlySelection in FOptions)) or
         // edit is not read only and selection will not be hidden when unfocused (then
         // being focused doesn't matter and r/o selection has no influence) or
        ((FOptions * [eoReadOnly, eoHideSelection]) = []) or
         // r/o selection is enabled and selection will not be hidden when unfocused
         // (then being focused or in r/o mode don't matter)
        ((eoReadOnlySelection in FOptions) and not (eoHideSelection in FOptions))
      );

    // Loop from the top line in the window to the last line in the window.
    for LineIndex := StartLine to EndLine do
    begin
      PaintOffset := FOffsetX + FGutterRect.Right + 1;
      LineBase := PWideChar(FContent[LineIndex].Text);
      TokenLen := Length(FContent[LineIndex].Text);

      // Get a potential custom style.
      CustomStyle := FContent[LineIndex].PeekStyle;

      if TokenLen > 0 then
      begin
        CalcCharWidths(LineIndex);

        // If no selection is to draw then go straight ahead.
        if (LineIndex < BB.Y) or (LineIndex > BE.Y) or not ShowSelection then
        begin
          Font.Style := Self.Font.Style;
          PickColors(Self.Font.Color, Self.Color, False);

          if eoShowControlChars in FOptions then
            PaintTokenWithControlChars(0, TokenLen)
          else
            PaintToken(0, TokenLen);

          if ShowLineBreak then
            PaintLineBreak;
        end
        else
        begin
          // Selection parts are to be considered, so determine start and end position of
          // the selection on this particular line.
          if BB.Y < LineIndex then
            SelStart := 0
          else
            SelStart := ColumnToCharIndex(Point(BB.X, LineIndex));
          if BE.Y > LineIndex then
            SelEnd := MaxInt
          else
            SelEnd := ColumnToCharIndex(Point(BE.X, LineIndex));

          // Start with the beginning of the token.
          StartIndex := 0;
          StopIndex := TokenLen;

          // Paint up to selection start. That length will be the StopIndex
          // or the start of the selection mark, whichever is less.
          TempIndex := Min(TokenLen, SelStart);
          if TempIndex > 0 then
          begin
            // Set needed colors and styles.
            PickColors(Self.Font.Color, Self.Color, False);

            if eoShowControlChars in FOptions then
              PaintTokenWithControlChars(0, TempIndex)
            else
              PaintToken(0, TempIndex);
            // update the start index into the line text to skip past the
            // text we just painted.
            Inc(StartIndex, TempIndex);
          end;

          // Paint the selection text. That length will be the StopIndex or the
          // end of the selection mark, whichever is less.
          TempIndex := Min(StopIndex, SelEnd) - StartIndex;
          if TempIndex > 0 then // Have anything to paint?
          begin
            // set the selection highlight colors
            with FSelectedColor do
            begin
              // Set needed colors and styles.
              PickColors(Foreground, Background, True);
            end;
            // paint the selection text
            if eoShowControlChars in FOptions then
              PaintTokenWithControlChars(StartIndex, TempIndex)
            else
              PaintToken(StartIndex, TempIndex);
            // Update the start index into the line text to skip past the
            // text we just painted.
            Inc(StartIndex, TempIndex);
          end;

          // Paint the post-selection text, the length is whatever is left over.
          Font.Style := Self.Font.Style;
          TempIndex := StopIndex - StartIndex;
          if TempIndex > 0 then
          begin
            // Set needed colors and styles.
            PickColors(Self.Font.Color, Self.Color, False);
            if eoShowControlChars in FOptions then
              PaintTokenWithControlChars(StartIndex, TempIndex)
            else
              PaintToken(StartIndex, TempIndex);
          end;

          // Prepare drawing of the rest of the line.
          if ShowSelection and (LineIndex < BE.Y) then
            PickColors(FSelectedColor.Foreground, FSelectedColor.Background, True)
          else
            PickColors(Self.Font.Color, Self.Color, False);
          if ShowLineBreak then
          begin
            if ShowSelection and (StopIndex < SelEnd) then
              PickColors(FSelectedColor.Foreground, FSelectedColor.Background, True)
            else
              PickColors(Self.Font.Color, Self.Color, False);
            PaintLineBreak;
          end;
        end;
      end
      else
        PickColors(Self.Font.Color, Self.Color, False);

      // Clear background of the rest of the line.
      FillRect(Rect(PaintOffset, YOffset, UpdateRect.Right, YOffset + FTextHeight));

      // draw the right margin marker
      Pen.Color := FMarginColor;
      Pen.Width := 1;
      MoveTo(FRightMargin * FCharWidth + FOffsetX + FGutterRect.Right + 1, YOffset);
      LineTo(FRightMargin * FCharWidth + FOffsetX + FGutterRect.Right + 1, YOffset + FTextHeight);

      // This line has been painted, increment the vertical offset and loop back
      // for the next line of text.
      Inc(YOffset, FTextHeight);
    end;

    // finally erase all the space not covered by lines
    if YOffset < UpdateRect.Bottom then
    begin
      Brush.Color := Self.Color;
      FillRect(Rect(UpdateRect.Left, YOffset, UpdateRect.Right, UpdateRect.Bottom));

      // draw the right margin marker
      Pen.Color := FMarginColor;
      Pen.Width := 1;
      MoveTo(FRightMargin * FCharWidth + FOffsetX + FGutterRect.Right + 1, YOffset);
      LineTo(FRightMargin * FCharWidth + FOffsetX + FGutterRect.Right + 1, UpdateRect.Bottom);
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.PasteFromClipboard;

var
  State: TEditState;
  BB, BE: TPoint;

begin
  // ANSI text is automatically converted to Unicode if no Unicode version of this text is available.
  if Clipboard.HasFormat(CF_UNICODETEXT) then
  begin
    State := RecordState(True);
    // keep the position at which the new text will be inserted
    BB := MinPoint(FBlockBegin, FBlockEnd);

    SetSelText(TextFromClipboard);

    // Keep end of text block we just inserted.
    BE := CaretXY;
    with State do
      FUndoList.AddChange(ecrReplace, Text, @Caret, @SelStart, @SelEnd,
        Clipboard.AsText, @BE, @BB, @BE);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

function TCustomUniCodeEdit.PosInSelection(Pos: TPoint): Boolean;

// determines whether the given position (line/column reference) is in the currrent
// selection block

var
  BB, BE: PPoint;

begin
  Result := False;
  if SelectionAvailable then
  begin
    BB := MinPointAsReference(FBlockBegin, FBlockEnd);
    BE := MaxPointAsReference(FBlockBegin, FBlockEnd);
    if BB.Y = BE.Y then
      Result := (Pos.Y = BB.Y) and (Pos.X >= BB.X) and (Pos.X <= BE.X)
    else
      Result := ((Pos.Y > BB.Y) and (Pos.Y < BE.Y)) or
        ((Pos.Y = BB.Y) and (Pos.X >= BB.X)) or
        ((Pos.Y = BE.Y) and (Pos.X <= BE.X));
    // so far so good, let's also take into account when
    // the block end is beyond a line end
    if Result and (Pos.Y = BE.Y) then
      Result := Cardinal(Pos.X) < GetLineEnd(Pos.Y);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

function TCustomUniCodeEdit.PositionFromPoint(X, Y: Integer): TPoint;

// calculates the column/line position from the given pixel coordinates (which must
// be in client coordinates)

begin
  // note: for the column value a real division with round is used (instead of
  // an integer div) to automatically switch to the proper column depending on
  // whether the position is before the half width of the column or after
  Result := Point(Round((X - FOffsetX - FGutterRect.Right + 1) / FCharWidth), Y div FTextHeight - FOffsetY);
end;

//----------------------------------------------------------------------------------------------------------------------

function TCustomUniCodeEdit.PreviousCharPos(ThisPosition: Cardinal): Cardinal;

begin
  Result := PreviousCharPos(ThisPosition, FCaretY);
end;

//----------------------------------------------------------------------------------------------------------------------

function TCustomUniCodeEdit.PreviousCharPos(ThisPosition, Line: Cardinal): Cardinal;

// Determines the previous character position for the given line depending on the
// column given by ThisPosition. The returned index can directly be used to set the caret.

var
  PixelPos, Run: Cardinal;
  I: Cardinal;

begin
  if ThisPosition = 0 then
    Result := 0
  else
  begin
    CalcCharWidths(Line);
    // turn column into pixel position
    PixelPos := ThisPosition * Cardinal(FCharWidth);
    if (eoCursorThroughTabs in FOptions) or
      (ThisPosition > GetLineEnd(Line)) then
      Result := ThisPosition - 1
    else
    begin
      // The task is to find the last column which corresponds directly to a
      // character (which is not always the case, eg. tabs) and is smaller than
      // the given position.
      Run := 0;
      // sum up character widths until we find the largest pixel position smaller
      // than the given one
      for I := 0 to High(FCharWidthArray) do
      begin
        // values in the char width array always correspond to an actual character index
        if PixelPos <= (Run + FCharWidthArray[I]) then
          Break
        else
          Inc(Run, FCharWidthArray[I]);
      end;
      Result := Run div Cardinal(FCharWidth);
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.ProcessCommand(var Command: TEditorCommand; var AChar: WideChar; Data: Pointer);

begin
  if Command < ecUserFirst then
  begin
    if Assigned(FOnProcessCommand) then
      FOnProcessCommand(Self, Command, AChar, Data);
  end
  else
  begin
    if Assigned(FOnProcessUserCommand) then
      FOnProcessUserCommand(Self, Command, AChar, Data);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

function TCustomUniCodeEdit.RecordState(IncludeText: Boolean): TEditState;

// prepare the result structure so it can be used for undo-regstration
begin
  with Result do
  begin
    if IncludeText then
      Text := GetSelectedText
    else
      Text := '';
    Caret := CaretXY;
    SelStart := FBlockBegin;
    SelEnd := FBlockEnd;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.Redo;

var
  Change: PChange;
  LastReason: TChangeReason;

  //--------------- local functions -------------------------------------------

  procedure RedoAction;

  // redo the action currently recorded in Change

  begin
    with Change^ do
      case Reason of
        ecrInsert:
          begin
            CaretXY := OldCaret;
            BlockBegin := CaretXY;
            SetSelText(NewText);
            CaretXY := NewCaret;
            BlockEnd := NewCaret;
          end;
        ecrDelete:
          begin
            SetBlockBegin(OldSelStart);
            SetBlockEnd(OldSelEnd);
            SetSelText('');
            CaretXY := NewCaret;
          end;
        ecrCursorMove:
          begin
            SetBlockBegin(NewSelStart);
            SetBlockEnd(NewSelEnd);
            CaretXY := NewCaret;
          end;
        ecrReplace:
          begin
            CaretXY := OldCaret;
            SetBlockBegin(OldSelStart);
            SetBlockEnd(OldSelEnd);
            SetSelText(NewText);
            CaretXY := NewCaret;
          end;
        ecrDragCopy,
          ecrDragMove:
          begin
            if Reason = ecrDragMove then
            begin
              SetBlockBegin(OldSelStart);
              SetBlockEnd(OldSelEnd);
              SetSelText('');
            end;

            CaretXY := MinPoint(NewSelStart, NewSelEnd);
            SetBlockBegin(NewSelStart);
            SetSelText(NewText);

            CaretXY := NewCaret;
            SetBlockBegin(NewSelStart);
            SetBlockEnd(NewSelEnd);
          end;
        ecrOverwrite:
          begin
            // overwriting text is always replacing it with one char
            CaretXY := OldCaret;
            SetBlockBegin(OldCaret);
            SetBlockEnd(Point(OldCaret.X + 1, OldCaret.Y));
            SetSelText(NewText);
            SetBlockBegin(NewCaret);
            CaretXY := NewCaret;
          end;
        ecrIndentation:
          begin
            SetBlockBegin(Point(0, Min(OldSelStart.Y, OldSelEnd.Y)));
            SetBlockEnd(Point(0, Max(OldSelStart.Y, OldSelEnd.Y) + 1));
            SetSelText(NewText);
            CaretXY := NewCaret;
            SetBlockBegin(NewSelStart);
            SetBlockEnd(NewSelEnd);
          end;
      end;
  end;

  //--------------- end local functions ---------------------------------------

begin
  if CanRedo then
  begin
    BeginUpdate;
    repeat
      // pop last entry from undo stack and get its values
      LastReason := FUndoList.GetRedoChange(Change);
      RedoAction;
    until not (eoGroupUndo in FOptions) or not CanRedo or (LastReason <> FUndoList.GetCurrentRedoReason);
    EndUpdate;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.RemoveBookmark(BookMark: Integer);

var
  Allowed: Boolean;

begin
  if (BookMark in [0..9]) and FBookMarks[BookMark].Visible then
    with FBookMarks[BookMark] do
    begin
      Allowed := True;
      if Assigned(FOnBookmarkChange) then
        FOnBookmarkChange(Self, Bookmark, X, Y, Allowed);
      Visible := not Allowed;
      if Allowed then
      begin
        InvalidateLine(Y);
        Y := -1;
      end;
    end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.ReplaceLeftTabs(var S: WideString);

// replaces all leading tab characters in the given WideString by a number of spaces given
// by the current tab size

var
  Count: Integer;

begin
  Count := LeftSpaces(S);
  while Count > 0 do
  begin
    if S[Count] = WideTabulator then
    begin
      Delete(S, Count, 1);
      Insert(WideStringOfChar(' ', FTabSize), S, Count);
    end;
    Dec(Count);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.ResetCaret;

begin
  if (FCaretX <> 0) or (FCaretY <> 0) then
  begin
    FCaretX := 0;
    FCaretY := 0;
    EnsureCursorPosVisible;
    if FUpdateCount = 0 then
    begin
      UpdateCaret;
      DoCaretChange;
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

function TCustomUniCodeEdit.RowColumnToCharPosition(P: TPoint): Cardinal;

// Calculates the total position of the column/row reference point as would all lines be
// one continous WideString.

var
  I: Integer;

begin
  Result := 0;
  P.X := ColumnToCharIndex(P);
  for I := 0 to Min(P.Y, FContent.Count) - 1 do
    Inc(Result, Length(FContent[I].Text) + 2); // add 2 for WideCRLF
  if P.Y < FContent.Count then
    Inc(Result, P.X);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.SaveToFile(const FileName: WideString; TextFormat: TTextFormat);

var
  Stream: TFileStream;

begin
  Stream := TFileStream.Create(FileName, fmCreate);
  try
    FContent.SaveToStream(Stream, TextFormat, 0, True);
  finally
    Stream.Free;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.ScanFrom(Index: Integer);

begin
  if Index < FContent.Count then
  begin
    if Index < FLastValidLine then
      FLastValidLine := Max(0, Index);
    InvalidateToBottom(Index);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

function TCustomUniCodeEdit.SearchReplace(const SearchText, ReplaceText: WideString; Options: TSearchOptions): Integer;

var
  StartPoint,
    EndPoint: TPoint; // search range
  Current: TPoint; // current search position
  DoBackward,
    DoFromCursor: Boolean;
  DoPrompt: Boolean;
  DoReplace,
    DoReplaceAll,
    Changed: Boolean;
  Action: TReplaceAction;

  //--------------- local functions -------------------------------------------

  function InValidSearchRange(First, Last: Integer): Boolean;

  begin
    Result := not (((Current.Y = StartPoint.Y) and (First < StartPoint.X)) or
      ((Current.Y = EndPoint.Y) and (Last > EndPoint.X)));
  end;

  //---------------------------------------------------------------------------

  procedure DoPlainSearchReplace;

  var
    SearchLen,
    ReplaceLen,
    N: Integer;
    Start,
    Stop,
    ColStart,
    ColStop: Integer;
    Found,
    DoStop: Boolean;
    Searcher: TUTBMSearch;
    SearchOptions: TSearchFlags;
    Offset: Integer;

  begin
    // TODO: Make search engine accept other input.
    //Searcher := TUTBMSearch.Create(FContent);
    Searcher := TUTBMSearch.Create(nil);
    try
      // initialize the search engine
      SearchOptions := [];
      if soMatchCase in Options then
        Include(SearchOptions, sfCaseSensitive);
      if soIgnoreNonSpacing in Options then
        Include(SearchOptions, sfIgnoreNonSpacing);
      if soSpaceCompress in Options then
        Include(SearchOptions, sfSpaceCompress);
      if soWholeWord in Options then
        Include(SearchOptions, sfWholeWordOnly);
      //soWholeWord in Options;
      Searcher.FindPrepare(SearchText, SearchOptions);
      // search while the current search position is inside of the search range
      SearchLen := Length(SearchText);
      ReplaceLen := Length(ReplaceText);
      DoStop := False;

      while not DoStop and (Current.Y >= StartPoint.Y) and (Current.Y <= EndPoint.Y) do
      begin
        // need a running offset because further search results are determined with regard
        // to the unchanged string but when replacing the string changes all the time
        Offset := 0;
        Found := Searcher.FindAll(FContent[Current.Y].Text);
        if DoBackward then
          N := Pred(Searcher.Count)
        else
          N := 0;
        // Operate on all results in this line.
        while not DoStop and Found do
        begin
          if (N < 0) or (N = Searcher.Count) then
            Break;
          Searcher.GetResult(N, Start, Stop);
          Inc(Start, Offset);
          Inc(Stop, Offset);
          // convert character positions to column values,
          // need to invalidate the char width array in every run because there might be non-single characters
          // like tabulators in text or replace string
          ColStart := CharIndexToColumn(Point(Start, Current.Y));
          ColStop := CharIndexToColumn(Point(Stop, Current.Y));
          if DoBackward then
            Dec(N)
          else
            Inc(N);
          // Is the search result entirely in the search range?
          if not InValidSearchRange(Start, Stop) then
            Continue;
          Inc(Result);
          // Select the text, so the user can see it in the OnReplaceText event
          // handler or as the search result.
          Current.X := ColStart;
          BlockBegin := Current;
          if DoBackward then
            CaretXY := Current;
          Current.X := ColStop;
          BlockEnd := Current;
          if not DoBackward then
            CaretXY := Current;
          // if it's a search only we can leave the procedure now
          if not (DoReplace or DoReplaceAll) then
            Abort;

          // Prompt and replace or replace all.  If user chooses to replace
          // all after prompting, turn off prompting.
          if DoPrompt then
          begin
            Action := raCancel;
            Application.ProcessMessages;
            FOnReplaceText(Self, SearchText, ReplaceText, Current.Y, ColStart, ColStop, Action);
            if Action = raCancel then
            begin
              DoStop := True;
              Break;
            end;
          end
          else
            Action := raReplace;
          if Action <> raSkip then
          begin
            // user has been prompted and has requested to silently replace all
            // so turn off prompting
            if Action = raReplaceAll then
            begin
              if not DoReplaceAll then
              begin
                DoReplaceAll := True;
                BeginUpdate;
              end;
              DoPrompt := False;
            end;
            SetSelTextExternal(ReplaceText);
            Changed := True;
          end;
          // calculate position offset (this offset is character index not column based)
          if not DoBackward then
            Inc(Offset, ReplaceLen - SearchLen);
          if not DoReplaceAll then
            DoStop := True;
        end;
        // search next / previous line
        if DoBackward then
          Dec(Current.Y)
        else
          Inc(Current.Y);
      end;
    finally
      Searcher.Free;
    end;
  end;

  //---------------------------------------------------------------------------

  procedure DoRESearchReplace;

  var
    SearchLen,
    ReplaceLen,
    N: Integer;
    Start,
    Stop,
    ColStart,
    ColStop: Integer;
    Found,
    DoStop: Boolean;
    Searcher: TURESearch;
    SearchOptions: TSearchFlags;
    Offset: Integer;

  begin
    // TODO: Make the search engine accept other input.
    //Searcher := TURESearch.Create(FContent);
    Searcher := TURESearch.Create(nil);
    try
      // initialize the search engine
      SearchOptions := [];
      if soMatchCase in Options then
        Include(SearchOptions, sfCaseSensitive);
      if soIgnoreNonSpacing in Options then
        Include(SearchOptions, sfIgnoreNonSpacing);
      if soSpaceCompress in Options then
        Include(SearchOptions, sfSpaceCompress);
      if soWholeWord in Options then
        Include(SearchOptions, sfWholeWordOnly);
      //soWholeWord in Options;
      Searcher.FindPrepare(SearchText, SearchOptions);
      // search while the current search position is inside of the search range
      SearchLen := Length(SearchText);
      ReplaceLen := Length(ReplaceText);
      DoStop := False;

      while not DoStop and (Current.Y >= StartPoint.Y) and (Current.Y <= EndPoint.Y) do
      begin
        // need a running offset because further search results are determined with regard
        // to the unchanged string but when replacing the string changes all the time
        Offset := 0;
        Found := Searcher.FindAll(FContent[Current.Y].Text);
        if DoBackward then
          N := Pred(Searcher.Count)
        else
          N := 0;
        // Operate on all results in this line.
        while not DoStop and Found do
        begin
          if (N < 0) or (N = Searcher.Count) then
            Break;
          Searcher.GetResult(N, Start, Stop);
          Inc(Start, Offset);
          Inc(Stop, Offset);
          // convert character positions to column values,
          // need to invalidate the char width array in every run because there might be non-single characters
          // like tabulators in text or replace string
          ColStart := CharIndexToColumn(Point(Start, Current.Y));
          ColStop := CharIndexToColumn(Point(Stop, Current.Y));
          if DoBackward then
            Dec(N)
          else
            Inc(N);
          // Is the search result entirely in the search range?
          if not InValidSearchRange(Start, Stop) then
            Continue;
          Inc(Result);
          // Select the text, so the user can see it in the OnReplaceText event
          // handler or as the search result.
          Current.X := ColStart;
          BlockBegin := Current;
          if DoBackward then
            CaretXY := Current;
          Current.X := ColStop;
          BlockEnd := Current;
          if not DoBackward then
            CaretXY := Current;
          // if it's a search only we can leave the procedure now
          if not (DoReplace or DoReplaceAll) then
            Abort;

          // Prompt and replace or replace all.  If user chooses to replace
          // all after prompting, turn off prompting.
          if DoPrompt then
          begin
            Action := raCancel;
            Application.ProcessMessages;
            FOnReplaceText(Self, SearchText, ReplaceText, Current.Y, ColStart, ColStop, Action);
            if Action = raCancel then
            begin
              DoStop := True;
              Break;
            end;
          end
          else
            Action := raReplace;
          if Action <> raSkip then
          begin
            // user has been prompted and has requested to silently replace all
            // so turn off prompting
            if Action = raReplaceAll then
            begin
              if not DoReplaceAll then
              begin
                DoReplaceAll := True;
                BeginUpdate;
              end;
              DoPrompt := False;
            end;
            SetSelTextExternal(ReplaceText);
            Changed := True;
          end;
          // calculate position offset (this offset is character index not column based)
          if not DoBackward then
            Inc(Offset, ReplaceLen - SearchLen);
          if not DoReplaceAll then
            DoStop := True;
        end;
        // search next / previous line
        if DoBackward then
          Dec(Current.Y)
        else
          Inc(Current.Y);
      end;
    finally
      Searcher.Free;
    end;
  end;

  //--------------- end local functions ---------------------------------------

begin
  Result := 0;
  // can't search for or replace an empty WideString
  if (Length(SearchText) > 0) and (FContent.Count > 0) then
  begin
    Changed := False;
    // get the text range to search in, ignore search in selection if nothing is selected
    DoBackward := soBackwards in Options;
    DoPrompt := (soPrompt in Options) and Assigned(FOnReplaceText);
    DoReplace := soReplace in Options;
    DoReplaceAll := soReplaceAll in Options;
    DoFromCursor := not (soEntireScope in Options);
    if SelectionAvailable and (soSelectedOnly in Options) then
    begin
      StartPoint := BlockBegin;
      EndPoint := BlockEnd;
      // ignore the cursor position when searching in the selection
      if DoBackward and not (soRegularExpression in Options) then
        Current := EndPoint
      else
        Current := StartPoint;
    end
    else
    begin
      StartPoint := Point(0, 0);
      EndPoint.Y := FContent.Count - 1;
      EndPoint.X := Length(FContent[EndPoint.Y].Text) + 1;
      if DoFromCursor then
        if DoBackward then
          EndPoint := CaretXY
        else
          StartPoint := CaretXY;
      if DoBackward then
        Current := EndPoint
      else
        Current := StartPoint;
    end;
    if DoReplaceAll and not DoPrompt then
      BeginUpdate;

    try
      if soRegularExpression in Options then
        DoRESearchReplace
      else
        DoPlainSearchReplace;
    finally
      if DoReplaceAll and not DoPrompt then
        EndUpdate;
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.SelectAll;

var
  State: TEditState;
  Caret: TPoint;

begin
  if FContent.Count > 0 then
  begin
    State := RecordState(False);
    SetBlockBegin(Point(0, 0));
    SetBlockEnd(Point(Length(FContent[FContent.Count - 1].Text) + 1, FContent.Count - 1));
    Caret := Point(Length(FContent[FContent.Count - 1].Text) + 1, FContent.Count - 1);
    CaretXY := Caret;
    Invalidate;

    FUndoList.AddChange(ecrCursorMove, '', @State.Caret, @State.SelStart, @State.SelEnd,
      '', @Caret, @FBlockBegin, @FBlockEnd);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.SetBlockBegin(Value: TPoint);

begin
  if Value.X < 0 then
    Value.X := 0
  else
    if Value.X > MaxRightChar then
      Value.X := MaxRightChar;
  if Value.Y < 0 then
    Value.Y := 0
  else
    if Value.Y > FContent.Count then
      Value.Y := FContent.Count;
  if (FUpdateCount = 0) and SelectionAvailable then
    InvalidateLines(FBlockBegin.Y, FBlockEnd.Y);
  FBlockBegin := Value;
  FBlockEnd := Value;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.SetBlockEnd(Value: TPoint);

begin
  if Value.X < 0 then
    Value.X := 0
  else
    if Value.X > MaxRightChar then
      Value.X := MaxRightChar;
  if Value.Y < 0 then
    Value.Y := 0
  else
    if Value.Y > FContent.Count then
      Value.Y := FContent.Count;

  if (FBlockEnd.X <> Value.X) or (FBlockEnd.Y <> Value.Y) then
  begin
    if FUpdateCount = 0 then
      InvalidateLines(Value.Y, FBlockEnd.Y);
    FBlockEnd := Value;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.SetBookMark(BookMark: Integer; X: Integer; Y: Integer);

var
  Allowed: Boolean;

begin
  if (BookMark in [0..9]) and (Y <= FContent.Count) then
  begin
    Allowed := True;
    if Assigned(FOnBookmarkChange) then
      FOnBookmarkChange(Self, Bookmark, X, Y, Allowed);
    if Allowed then
    begin
      FBookMarks[BookMark].X := X;
      FBookMarks[BookMark].Y := Y;
      FBookMarks[BookMark].Visible := True;
      InvalidateLine(Y);
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.SetBorderStyle(Value: TBorderStyle);

begin
  if FBorderStyle <> Value then
  begin
    FBorderStyle := Value;
    RecreateWnd;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.SetCaretToEditorBottom;

begin
  CaretY := FContent.Count - 1;
  CaretX := Length(LineText);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.SetCaretX(Value: Integer);

var
  I: Integer;

begin
  if Value < 0 then
    Value := 0;

  if FCaretX <> Value then
  begin
    if (CaretY < FContent.Count) then
      I := Length(FContent[CaretY].Text)
    else
      I := 0;

    if eoScrollPastEOL in FOptions then
    begin
      if Value > MaxRightChar then
        Value := MaxRightChar;
    end
    else
    begin
      if Value > I then
        Value := I;
    end;
    FCaretX := Value;
    EnsureCursorPosVisible;
    if FUpdateCount = 0 then
    begin
      UpdateCaret;
      DoCaretChange;
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.SetCaretXY(const Value: TPoint);

begin
  // set Y first because ScrollPastEOL can effect the X pos, prevent SetCaretX/Y
  // from sending OnChange events
  Inc(FUpdateCount);
  CaretY := Value.Y;
  CaretX := Value.X;
  Dec(FUpdateCount);
  if FUpdateCount = 0 then
  begin
    UpdateCaret;
    UpdateScrollBars;
    UpdateWindow(Handle);
    DoCaretChange;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.SetCaretY(Value: Integer);

var
  Len: Integer;
  S: WideString;

begin
  if (Value < 0) or (FContent.Count = 0) then
    Value := 0;
  if (FContent.Count > 0) and (Value > FContent.Count - 1) then
    Value := FContent.Count - 1;
  if FCaretY <> Value then
  begin
    // Remove trailing blanks in the line we just leaving
    if (FCaretY > -1) and (FCaretY < FContent.Count) and not (eoKeepTrailingBlanks in FOptions) then
    begin
      S := FContent[FCaretY].Text;
      if (Length(S) > 0) and (S[Length(S)] = ' ') then
        FContent[FCaretY].Text := WideTrimRight(S);
    end;
    FCaretY := Value;

    if not (eoScrollPastEOL in FOptions) then
    begin
      Len := Length(LineText);
      if FCaretX > Len then
      begin
        CaretX := Len;
        if FCaretX < -FOffsetX * FCharWidth then
          OffsetX := FCaretX * FCharWidth;
        Exit;
      end;
    end;
    EnsureCursorPosVisible;
    if FUpdateCount = 0 then
    begin
      UpdateCaret;
      DoCaretChange;
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.SetCharWidth(const Value: Integer);

var
  RightChar: Integer;

begin
  if FCharWidth <> Value then
  begin
    RightChar := FMaxRightPos div FCharWidth;
    FCharWidth := Value;
    MaxRightChar := RightChar;
    Invalidate;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.SetDefaultKeystrokes;

begin
  FKeyStrokes.ResetDefaults;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.SetExtraLineSpacing(const Value: Integer);

begin
  FExtraLineSpacing := Value;
  FontChanged(Self);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.SetFont(const Value: TFont);

var
  SavePitch: TFontPitch;

begin
  SavePitch := Value.Pitch;
  Value.Pitch := fpFixed;
  inherited Font := Value;
  Value.Pitch := SavePitch;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.SetGutterColor(Value: TColor);

begin
  if FGutterColor <> Value then
  begin
    FGutterColor := Value;
    Invalidate;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.SetGutterWidth(Value: Integer);

var
  OldValue: Integer;

begin
  OldValue := GutterWidth;
  if OldValue <> Value then
  begin
    if Value < 0 then
      Value := 0;
    FGutterWidth := Value;
    FGutterRect.Right := Value - 1;
    if HandleAllocated then
    begin
      FCharsInWindow := (ClientWidth - FGutterRect.Right + 1) div FCharWidth;
      UpdateCaret;
      UpdateScrollBars;
      Invalidate;
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.SetHighlighter(const Value: TUCEHighlighter);

begin
  if Value <> FHighLighter then
  begin
    if Assigned(FHighLighter) then
      FHighLighter.WindowList.RemoveInstance(Self);
    FHighLighter := Value;
    if Assigned(FHighLighter) then
    begin
      Value.WindowList.AddInstance(Self);
      Value.FreeNotification(Self);
      ScanFrom(0);
    end
    else
      Invalidate;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.SetIndentSize(Value: Integer);

begin
  if Value < 1 then
    Value := 1;
  if FIndentSize <> Value then
  begin
    FIndentSize := Value;
    Invalidate;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.SetInsertCaret(const Value: TCaretType);

begin
  if FInsertCaret <> Value then
  begin
    FInsertCaret := Value;
    InitializeCaret;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.SetKeystrokes(const Value: TKeyStrokes);

begin
  if Value = nil then
    FKeyStrokes.Clear
  else
    FKeyStrokes.Assign(Value);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.SetLineNumberFont(const Value: TFont);

begin
  FLineNumberFont.Assign(Value);
  LineNumberFontChanged(nil);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.SetLineText(Value: WideString);

begin
  if FContent.Count = 0 then
    FContent.AddLine('');
  if FCaretY < FContent.Count then
  begin
    FContent[FCaretY].Text := Value;
    InvalidateLine(FCaretY);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.SetMarginColor(const Value: TColor);

begin
  if FMarginColor <> Value then
  begin
    FMarginColor := Value;
    Invalidate;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.SetMaxRightChar(const Value: Integer);

begin
  if FMaxRightPos <> (Value * FCharWidth) then
  begin
    FMaxRightPos := Value * FCharWidth;
    if HandleAllocated then
    begin
      Invalidate;
      UpdateScrollBars;
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.SetMaxUndo(const Value: Integer);

begin
  if Value > -1 then
  begin
    FUndoList.MaxUndo := Value;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.SetModified(const Value: Boolean);

begin
  if FModified <> Value then
  begin
    FModified := Value;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.SetOffsetX(Value: Integer);

var
  dX: Integer;
  R: TRect;

begin
  if - Value > (FMaxRightPos - ClientWidth + FGutterRect.Right) then
    Value := -(FMaxRightPos - ClientWidth + FGutterRect.Right);
  if Value > 0 then
    Value := 0;
  if FOffsetX <> Value then
  begin
    dX := Value - FOffsetX;
    FOffsetX := Value;
    R := Rect(FGutterRect.Right + 1, 0, ClientWidth, ClientHeight);
    ScrollWindow(Handle, dX, 0, nil, @R);
    if (CaretXPix + FCaretOffset.X) < (FGutterRect.Right + 1) then
      HideCaret
    else
      ShowCaret;
    if FUpdateCount = 0 then
    begin
      UpdateWindow(Handle);
      UpdateScrollbars;
    end;
    DoScroll(dX, 0);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.SetOffsetY(Value: Integer);

// This offset is measured in lines not pixels.

var
  dY: Integer;

begin
  // Range check, order is important here.
  if Value < (FLinesInWindow - FContent.Count) then
    Value := FLinesInWindow - FContent.Count;
  if Value > 0 then
    Value := 0;
  if FOffsetY <> Value then
  begin
    dY := FTextHeight * (Value - FOffsetY);
    FOffsetY := Value;
    ScrollWindow(Handle, 0, dY, nil, nil);
    if FUpdateCount = 0 then
    begin
      UpdateWindow(Handle);
      UpdateScrollbars;
    end;
    DoScroll(0, dY);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.SetOptions(const Value: TUniCodeEditOptions);

var
  ToBeSet,
  ToBeCleared,
  ToChange: TUniCodeEditOptions;

begin
  if FOptions <> Value then
  begin
    ToBeSet := Value - FOptions;
    ToBeCleared := FOptions - Value;
    FOptions := Value;
    ToChange := ToBeSet + ToBeCleared;

    if eoReadOnly in FOptions then
      if eoShowCursorWhileReadOnly in ToBeSet then
        InitializeCaret
      else
      begin
        HideCaret;
        DestroyCaret;
      end;

    if HandleAllocated and ((eoHideSelection in ToChange) or
      (eoShowControlChars in ToChange) or
      (eoUseSyntaxHighlighting in ToChange)) then
      Invalidate;

    if eoLineNumbers in ToChange then
    begin
      FGutterRect.Right := FGutterWidth - 1;
      if HandleAllocated then
      begin
        FCharsInWindow := (ClientWidth - FGutterRect.Right + 1) div FCharWidth;
        Invalidate;
      end;
    end;

    // reset pos in case it's past EOL currently
    if (eoScrollPastEOL in ToBeCleared) and (FCaretX > MaxRightChar) then
      CaretX := MaxRightChar;
    if HandleAllocated and (eoLineNumbers in ToChange) and Focused then
      UpdateCaret;
    DoSettingChanged;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.SetOverwriteCaret(const Value: TCaretType);

begin
  if FOverwriteCaret <> Value then
  begin
    FOverwriteCaret := Value;
    InitializeCaret;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.SetRightMargin(Value: Integer);

begin
  if (FRightMargin <> Value) then
  begin
    FRightMargin := Value;
    Invalidate;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.SetScrollBars(const Value: TScrollStyle);

begin
  if (FScrollBars <> Value) then
  begin
    FScrollBars := Value;
    RecreateWnd;
    UpdateScrollBars;
    Invalidate;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.SetScrollHintColor(const Value: TUCELineStyle);

begin
  FScrollHintColor.FBackground := Value.FBackground;
  FScrollHintColor.FForeground := Value.FForeground;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.SetSelectedColor(const Value: TUCELineStyle);

begin
  FSelectedColor.FBackground := Value.FBackground;
  FSelectedColor.FForeground := Value.FForeground;
  Invalidate;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.SetSelEnd(const Value: Integer);

var
  P: TPoint;
  Loop: Integer;
  Count: Integer;

begin
  Loop := 0;
  Count := 0;
  while (Loop < FContent.Count - 1) and ((Count + Length(FContent[Loop].Text) + 2) < Value) do
  begin
    Inc(Count, Length(FContent[Loop].Text) + 2);
    Inc(loop);
  end;
  P.Y := Loop;
  P.X := Value - Count;
  If P.X > Length(FContent[Loop].Text) then
    P.X := Length(FContent[Loop].Text);
  Blockend := P;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.SetSelStart(const Value: Integer);

var
  Loop: Integer;
  Count: Integer;
  P: TPoint;

begin
  if FContent.Count = 0 then
    P := Point(0, 0)
  else
  begin
    Loop := 0;
    Count := 0;
    while (Loop < FContent.Count) and ((Count + Length(FContent[Loop].Text) + 2) < Value) do
    begin
      Inc(Count, Length(FContent[Loop].Text) + 2);
      Inc(loop);
    end;
    P.X := Value - Count;
    if Loop = FContent.Count then
      Dec(Loop);
    P.Y := Loop;
    if P.X > Length(FContent[Loop].Text) then
      P.X := Length(FContent[Loop].Text);
  end;
  FBlockBegin := P;
  FBlockEnd := P;                             
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.SetSelText(const Value: WideString);

begin
  InvalidateToBottom(Min(FBlockBegin.Y, FBlockEnd.Y));
  if SelectionAvailable then
    DeleteSelection(Value = '');
  InsertText(Value);
  BlockBegin := CaretXY;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.SetSelTextExternal(const Value: WideString);

var
  State: TEditState;
  BB, BE: TPoint;

begin
  State := RecordState(True);
  // keep the position at which the new text will be inserted
  BB := MinPoint(FBlockBegin, FBlockEnd);
  SetSelText(Value);
  SetBlockBegin(CaretXY);
  // keep end of text block just inserted
  BE := CaretXY;
  with State do
    FUndoList.AddChange(ecrReplace, Text, @Caret, @SelStart, @SelEnd, Value, @BE, @BB, @BE);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.SetTabSize(Value: Integer);

begin
  if Value < 1 then
    Value := 1;
  if FTabSize <> Value then
  begin
    FTabSize := Value;
    Invalidate;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.SetText(const Value: WideString);

begin
  FContent.Text := Value;
  Invalidate;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.SetTopLine(Value: Integer);

begin
  if Value < 1 then
    Value := 1;
  if TopLine <> Value then
  begin
    if Value > FContent.Count then
      Value := FContent.Count;
    OffsetY := 1 - Value;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.SetUpdateState(Updating: Boolean);

begin
  SendMessage(Handle, WM_SETREDRAW, Ord(not Updating), 0);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.SetWordBlock(Value: TPoint);

var
  BB, BE: TPoint;

begin
  GetWordAndBounds(Value, BB, BE);
  SetBlockBegin(BB);
  SetBlockEnd(BE);
  CaretXY := BE;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.ShowCaret;

begin
  if not FCaretVisible then
    FCaretVisible := Windows.ShowCaret(Handle);
end;

//----------------------------------------------------------------------------------------------------------------------

function TCustomUniCodeEdit.TextFromClipboard: WideString;

var
  Data: THandle;

begin
  // ANSI text is automatically converted to Unicode if no Unicode version of this text is available.
  if Clipboard.HasFormat(CF_UNICODETEXT) then
  begin
    // Delphi's TClipboard does not support Unicode (D5 and lower) hence we
    // need to handle the stuff manually.
    Data := Clipboard.GetAsHandle(CF_UNICODETEXT);
    try
      if Data <> 0 then
        Result := PWideChar(GlobalLock(Data));
    finally
      if Data <> 0 then
        GlobalUnlock(Data);
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.TextToClipboard(Text: WideString);

var
  Data: THandle;
  DataPtr: Pointer;
  Size: Cardinal;

begin
  // Delphi's TClipboard does not support Unicode (D5 and lower) hence we
  // need to handle the stuff manually.
  Size := Length(Text);
  Data := GlobalAlloc(GMEM_MOVEABLE + GMEM_DDESHARE, 2 * Size + 2);
  try
    DataPtr := GlobalLock(Data);
    try
      Move(PWideChar(Text)^, DataPtr^, 2 * Size + 2);
      Clipboard.SetAsHandle(CF_UNICODETEXT, Data);
    finally
      GlobalUnlock(Data);
    end;
  except
    GlobalFree(Data);
    raise;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

function TCustomUniCodeEdit.TranslateKeyCode(Code: word; Shift: TShiftState; var Data: Pointer): TEditorCommand;

// If the translations requires Data, memory will be allocated for it via a
// GetMem call. The client must call FreeMem on Data if it is not nil.

var
  I: Integer;

begin
  I := Keystrokes.FindKeycode(Code, Shift);
  if I >= 0 then
    Result := Keystrokes[I].Command
  else
    Result := ecNone;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.TripleClick;

// mark entire line if the user clicked three times within the double click interval

begin
  FMultiClicked := True;
  SetBlockBegin(Point(0, CaretY));
  SetBlockEnd(Point(0, CaretY + 1));
  FLastDblClick := 0;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.Undo;

var
  Change: PChange;
  LastReason: TChangeReason;

  //--------------- local functions -------------------------------------------

  procedure UndoAction;

  // undo the action currently recorded in Change

  var
    I, J: Cardinal;
    T: TWideStringList;

  begin
    with Change^ do
      case Reason of
        ecrInsert:
          begin
            SetBlockBegin(NewSelStart);
            SetBlockEnd(NewSelEnd);
            SetSelText('');
            CaretXY := OldCaret;
          end;
        ecrDelete:
          begin
            CaretXY := OldCaret;
            SetBlockBegin(MinPoint(OldSelStart, OldSelEnd));
            SetSelText(OldText);
            SetBlockBegin(OldSelStart);
            SetBlockEnd(OldSelEnd);
          end;
        ecrCursorMove:
          begin
            CaretXY := OldCaret;
            BlockBegin := OldSelStart;
            BlockEnd := OldSelEnd;
          end;
        ecrReplace:
          begin
            CaretXY := OldCaret;
            SetBlockBegin(NewSelStart);
            SetBlockEnd(NewSelEnd);
            SetSelText(OldText);
            SetBlockBegin(OldSelStart);
            SetBlockEnd(OldSelEnd);
          end;
        ecrDragCopy,
          ecrDragMove:
          begin
            SetBlockBegin(NewSelStart);
            SetBlockEnd(NewSelEnd);
            SetSelText('');

            if Reason = ecrDragMove then
            begin
              CaretXY := MinPoint(OldSelStart, OldSelEnd);
              SetSelText(OldText);
            end;

            CaretXY := OldCaret;
            SetBlockBegin(OldSelStart);
            SetBlockEnd(OldSelEnd);
          end;
        ecrOverwrite:
          begin
            // overwriting text is always replacing one char by another
            CaretXY := OldCaret;
            SetBlockBegin(OldCaret);
            SetBlockEnd(Point(OldCaret.X + 1, OldCaret.Y));
            SetSelText(OldText);
            SetBlockBegin(OldCaret);
          end;
        ecrIndentation:
          begin
            // This is a special kind of replacement, as we set back entire lines.
            // With the consideration of some background information, it becomes
            // reasonably fast (no highlighter rescan necessary as we are modifying spaces
            // only, no partial line modifications).
            T := TWideStringList.Create;
            try
              T.Text := OldText;
              J := Min(NewSelStart.Y, NewSelEnd.Y);
              // restore old lines
              for I := 0 to T.Count - 1 do
                FContent[I + J].Text := T[I];
            finally
              T.Free;
            end;
            CaretXY := OldCaret;
            SetBlockBegin(OldSelStart);
            SetBlockEnd(OldSelEnd);
          end;
      end;
  end;

  //--------------- end local functions ---------------------------------------

begin
  if CanUndo then
  begin
    BeginUpdate;
    repeat
      // pop last entry from undo stack and get its values
      LastReason := FUndoList.GetUndoChange(Change);
      UndoAction;
    until not (eoGroupUndo in FOptions) or not CanUndo or (LastReason <> FUndoList.GetCurrentUndoReason);
    EndUpdate;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

function TCustomUniCodeEdit.Unindent(X, Y: Cardinal): Cardinal;

// Calculates the next unindent position starting with X and depending on the
// first line above Y which contains a white space char before X.
// X must be a column value!

  //--------------- local functions -------------------------------------------

  function FindNonWhiteSpaceColumn(X, Y: Cardinal; var NewX: Cardinal): Boolean;

  var
    Index: Integer;
    Line: WideString;

  begin
    NewX := X;
    Result := False;
    Line := FContent[Y].Text;
    while NewX > 0 do
    begin
      Dec(NewX);
      // turn column into its corresponding character index...
      Index := ColumnToCharIndex(Point(NewX, Y));
      // ... and turn it back into the start column of this char
      NewX := CharIndexToColumn(Point(Index, Y));
      // fork out if we found a non-white space char
      if (Index < Length(Line)) and not UnicodeIsWhiteSpace(Word(Line[Index + 1])) then
      begin
        Result := True;
        Break;
      end;
    end;
  end;

  //---------------------------------------------------------------------------

  function FindWhiteSpaceColumn(X, Y: Cardinal; var NewX: Cardinal): Boolean;

  var
    Index: Integer;
    Line: WideString;

  begin
    NewX := X;
    Result := False;
    Line := FContent[Y].Text;
    while NewX > 0 do
    begin
      Dec(NewX);
      // turn column into its corresponding character index...
      Index := ColumnToCharIndex(Point(NewX, Y));
      // ... and turn it back into the start column of this char
      NewX := CharIndexToColumn(Point(Index, Y));
      // fork out if we found a white space
      if (Index < Length(Line)) and UnicodeIsWhiteSpace(Word(Line[Index + 1])) then
      begin
        Result := True;
        Break;
      end;
    end;
  end;

  //--------------- end local functions ---------------------------------------

begin
  // cursor at line 0?
  if (X = 0) or (Y = 0) then
    Result := 0 // no previous tab position available
  else
  begin
    // find a line above the current line, which is not empty and whose first
    // non-white character is before X
    repeat
      // go one line up (picturally speaking, actually we use the previous line)
      Dec(Y);
      // find last non-white space before current column
      if FindNonWhiteSpaceColumn(X, Y, Result) then
      begin
        // we found a non-white space, now search the first white
        // space column before this column
        if FindWhiteSpaceColumn(Result, Y, Result) then
        begin
          // if there's such a column then advance to next column (which is then
          // a non-white space column)
          Result := NextCharPos(Result, Y);
        end;
        // leave loop, since we either found what we were looking for or
        // the line in question has no white chars at the beginning (which means
        // unindentation target column is 0)
        Break;
      end;
      // turn around if there was no unindent column by going up one line
    until Y = 0;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.UpdateCaret;

var
  CX, CY: Integer;

begin
  CX := CaretXPix + FCaretOffset.X;
  CY := CaretYPix + FCaretOffset.Y;
  SetCaretPos(CX - 1, CY);
  if (CX < FGutterRect.Right + 1) or (CX > ClientWidth) or
    (CY < 0) or (CY > ClientHeight - FTextHeight) then
    HideCaret
  else
    ShowCaret;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.UpdateScrollBars;

var
  ScrollInfo: TScrollInfo;
  Page: Integer;

begin
  if (FUpdateCount = 0) and (ClientWidth > 0) then
  begin
    FillChar(ScrollInfo, SizeOf(ScrollInfo), 0);
    ScrollInfo.cbSize := SizeOf(ScrollInfo);
    ScrollInfo.fMask := SIF_ALL;
    ScrollInfo.nMin := 0;
    ScrollInfo.nTrackPos := 0;
    // vertical scrollbar
    if FScrollBars in [ssBoth, ssVertical] then
    begin
      if (FLinesInWindow > 0) and (FContent.Count > FLinesInWindow) then
      begin
        ScrollInfo.nPage := FLinesInWindow + 1;
        ScrollInfo.nMax := FContent.Count;
        ScrollInfo.nPos := -FOffsetY;
      end;
      SetScrollInfo(Handle, SB_VERT, ScrollInfo, True);
    end;

    if FScrollBars in [ssBoth, ssHorizontal] then
    begin
      ScrollInfo.nMax := FMaxRightPos + 1;
      Page := ClientWidth - FGutterRect.Right + 1;
      if Page >= 0 then
        ScrollInfo.nPage := Page
      else
        ScrollInfo.nPage := 0;
      ScrollInfo.nPos := -FOffsetX;
      SetScrollInfo(Handle, SB_HORZ, ScrollInfo, True);
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.WndProc(var Msg: TMessage);

// prevent Alt-Backspace from beeping

begin
  if (Msg.Msg = WM_SYSCHAR) and
    (Msg.wParam = VK_BACK) and
    ((Msg.lParam and $20000000) <> 0) then
  begin
    Msg.Msg := 0;
  end
  else
    inherited;
end;

//----------------------------------------------------------------------------------------------------------------------

function TCustomUniCodeEdit.WordAtPos(X, Y: Integer): WideString;

// returns the word at the given position, X and Y must be given in client coordinates,

var
  BB, BE: TPoint;

begin
  Result := GetWordAndBounds(PositionFromPoint(X, Y), BB, BE);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.CMMouseWheel(var Message: TCMMouseWheel);

var
  ScrollCount: Integer;

begin
  inherited;
  if Message.Result = 0 then
    with Message do
    begin
      Result := 1;
      if ssCtrl in ShiftState then
        ScrollCount := FLinesInWindow * (WheelDelta div WHEEL_DELTA)
      else
        ScrollCount := Mouse.WheelScrollLines * (WheelDelta div WHEEL_DELTA);
      TopLine := TopLine - ScrollCount;
      Update;
    end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.CMSysColorChange(var Message: TMessage);

begin
  inherited;
  ConvertAndAddImages(FInternalBMList);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.WMCopy(var Message: TWMCopy);

var
  Data: THandle;
  DataPtr: Pointer;
  SelectedText: WideString;
  SelLen: integer;

begin
  SelectedText := GetSelectedText;
  SelLen := Length(SelectedText);

  if (SelLen > 0) then
  begin
    Clipboard.Open;
    try
      EmptyClipboard;

      Data := GlobalAlloc(GMEM_MOVEABLE or GMEM_DDESHARE, SelLen * 2 + 2);
      if (Data <> 0) then
      begin
        DataPtr := GlobalLock(Data);
        try
          if (DataPtr <> nil) then
          begin
            Move(PWideChar(SelectedText)^, DataPtr^, SelLen * 2 + 2);
            if (SetClipboardData(CF_UNICODETEXT, Data) = 0) then
              raise Exception.Create('The selected text cannot be copied ' +
                'to the clipboard.');
          end;
        finally
          GlobalUnlock(Data);
        end;
      end;

    finally
      Clipboard.Close;
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.WMCut(var Message: TWMCut);

var
  Msg: TWMCopy;

begin
  Msg.Msg := 769;
  Msg.Result := 0;

  WMCopy(Msg);

  SetSelText('');
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.WMEraseBkgnd(var Message: TMessage);

begin
  Message.Result := 1;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.WMGetDlgCode(var Msg: TWMGetDlgCode);

begin
  Msg.Result := DLGC_WANTARROWS or DLGC_WANTCHARS;
  if eoWantTabs in FOptions then
    Msg.Result := Msg.Result or DLGC_WANTTAB;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.WMHScroll(var Message: TWMScroll);

  //--------------- local functions -------------------------------------------

  function GetRealScrollPosition: Integer;

  var
    SI: TScrollInfo;
    Code: Integer;

  begin
    SI.cbSize := SizeOf(TScrollInfo);
    SI.fMask := SIF_TRACKPOS;
    Code := SB_HORZ;
    GetScrollInfo(Handle, Code, SI);
    Result := SI.nTrackPos;
  end;

  //--------------- end local functions ---------------------------------------

begin
  case Message.ScrollCode of
    SB_BOTTOM:
      OffsetX := -FMaxRightPos;
    SB_ENDSCROLL:
      ;
    SB_LINELEFT:
      if FOffsetX < 0 then
        OffsetX := FOffsetX + FCharWidth;
    SB_LINERIGHT:
      if - FOffsetX < (FMaxRightPos - ClientWidth + FGutterRect.Right) then
        OffsetX := FOffsetX - FCharWidth;
    SB_PAGELEFT:
      OffsetX := FOffsetX + ClientWidth - FGutterRect.Right + 1;
    SB_PAGERIGHT:
      OffsetX := FOffsetX - ClientWidth + FGutterRect.Right + 1;
    SB_THUMBPOSITION,
    SB_THUMBTRACK:
      OffsetX := -GetRealScrollPosition;
    SB_TOP:
      OffsetX := 0;
  end;
  Message.Result := 0;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.WMImeComposition(var Message: TMessage);

var
  IMEContext: HIMC;
  P: Pointer;
  Size: Integer;

begin
  if (Message.LParam and GCS_RESULTSTR) <> 0 then
  begin
    IMEContext := ImmGetContext(Handle);
    try
      // Note: The docs say the return value would always be in bytes, even for Unicode input.
      //       This is obviously not true! The return value is number of characters (without null terminator).
      Size := ImmGetCompositionString(IMEContext, GCS_RESULTSTR, nil, 0);
      if Size < 0 then
      begin
        // A problem occured. IMM_ERROR_NODATA can be ignored but IMM_ERROR_GENERAL cannot.
        if Size = IMM_ERROR_GENERAL then
          raise Exception.Create(SGeneralIMEError);
      end
      else
      begin
        Inc(Size); // Add place for the null terminator.
        P := nil;
        try
          if PlatformIsUnicode then
          begin
            P := AllocMem(2 * Size);
            ImmGetCompositionStringW(IMEContext, GCS_RESULTSTR, P, 2 * Size);
            PWideChar(P)[Size - 1] := #0;
            InsertText(PWideChar(P));
          end
          else
          begin
            // On non-Unicode platforms we get only back meaningful values if the system
            // local corresponds to the locale of the input method editor.
            // In this case we get ANSI strings that must be converted with the current system locale.
            P := AllocMem(Size);
            ImmGetCompositionString(IMEContext, GCS_RESULTSTR, P, Size);
            PChar(P)[Size - 1] := #0;
            InsertText(PChar(P));
          end;
        finally
          FreeMem(P);
        end;
      end;
    finally
      ImmReleaseContext(Handle, IMEContext);
    end;
    Message.Result := 0;
  end
  else
    inherited;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.WMImeNotify(var Message: TMessage);

var
  IMC: HIMC;
  LogFont: TLogFont;

begin
  with Message do
  begin
    case WParam of
      IMN_SETOPENSTATUS:
        begin
          IMC := ImmGetContext(Handle);
          if IMC <> 0 then
          begin
            // need to tell the composition window what font we are using currently
            GetObject(Font.Handle, SizeOf(TLogFont), @LogFont);
            ImmSetCompositionFont(IMC, @LogFont);
            ImmReleaseContext(Handle, IMC);
          end;
          Message.Result := 0;
        end;
    else
      inherited;
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.WMKillFocus(var Message: TWMKillFocus);

begin
  inherited;
  HideCaret;
  DestroyCaret;
  if (eoHideSelection in FOptions) and SelectionAvailable then
    Invalidate;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.WMPaste(var Message: TWMPaste);

var
  Data: THandle;
  S: WideString;
begin
  if (Clipboard.HasFormat(CF_UNICODETEXT)) then
  begin
    Data := Clipboard.GetAsHandle(CF_UNICODETEXT);

    try
      if (Data <> 0) then
        S := PWideChar(GlobalLock(Data));

      SetSelText(S);
    finally
      if (Data <> 0) then
        GlobalUnlock(Data);
    end;
  end
  else
    if (Clipboard.HasFormat(CF_TEXT)) then
      SetSelText(Clipboard.AsText);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.WMSetCursor(var Message: TWMSetCursor);

var
  P: TPoint;

begin
  if csDesigning in ComponentState then
    inherited
  else
  begin
    GetCursorPos(P);
    P := ScreenToClient(P);
    if (Message.HitTest <> HTCLIENT) or
      PtInRect(FGutterRect, P) then
      Windows.SetCursor(Screen.Cursors[crArrow])
    else
    begin
      P := PositionFromPoint(P.X, P.Y);
      if PosInSelection(P) then
        Windows.SetCursor(Screen.Cursors[crArrow])
      else
        Windows.SetCursor(Screen.Cursors[Cursor]);
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.WMSetFocus(var Message: TWMSetFocus);

begin
  inherited;
  InitializeCaret;
  UpdateScrollBars;
  if (eoHideSelection in FOptions) and SelectionAvailable then
    Invalidate;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.WMSize(var Message: TWMSize);

begin
  inherited;
  if HandleAllocated then
  begin
    FCharsInWindow := (ClientWidth - FGutterRect.Right + 1) div FCharWidth;
    FLinesInWindow := ClientHeight div FTextHeight;
    FGutterRect.Bottom := ClientHeight;
    if (FContent.Count - TopLine + 1) < FLinesInWindow then
      TopLine := FContent.Count - FLinesInWindow + 1;
    UpdateScrollBars;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TCustomUniCodeEdit.WMVScroll(var Message: TWMScroll);

  //--------------- local functions -------------------------------------------

  function GetRealScrollPosition: Integer;

  var
    SI: TScrollInfo;
    Code: Integer;

  begin
    SI.cbSize := SizeOf(TScrollInfo);
    SI.fMask := SIF_TRACKPOS;
    Code := SB_VERT;
    GetScrollInfo(Handle, Code, SI);
    Result := SI.nTrackPos;
  end;

  //--------------- end local functions ---------------------------------------

var
  LineString: WideString;
  HintRect: TRect;
  P: TPoint;
  ButtonH: Integer;

begin
  case Message.ScrollCode of
    SB_BOTTOM:
      OffsetY := -FContent.Count;
    SB_ENDSCROLL:
      FScrollHint.ReleaseHandle;
    SB_LINEUP:
      OffsetY := FOffsetY + 1;
    SB_LINEDOWN:
      OffsetY := FOffsetY - 1;
    SB_PAGELEFT:
      OffsetY := FOffsetY + FLinesInWindow;
    SB_PAGERIGHT:
      OffsetY := FOffsetY - FLinesInWindow;
    SB_THUMBPOSITION,
    SB_THUMBTRACK:
      begin
        OffsetY := -GetRealScrollPosition;
        if eoShowScrollHint in FOptions then
        begin
          LineString := IntToStr(-FOffsetY + 1);
          // Calculate optimal size for scroll hint window.
          HintRect := FScrollHint.CalcHintRect(2000, LineString, nil);
          // Place it parallel to the thumb.
          P := ClientToScreen(Point(0, 0));
          ButtonH := GetSystemMetrics(SM_CYVSCROLL) + 8;
          OffsetRect(HintRect,
            P.X + ClientWidth - HintRect.Right - 2,
            ButtonH div 2 + P.Y + Round((ClientHeight - HintRect.Bottom - ButtonH) * -FOffsetY / FContent.Count));

          with FScrollHint do
          begin
            Font.Color := FScrollHintColor.FForeground;
            Color := FScrollHintColor.FBackground;
            ActivateHint(HintRect, LineString);
            UpdateWindow(Handle);
          end;
        end;
      end;
    SB_TOP:
      OffsetY := 0;
  end;
  Message.Result := 0;
end;

//----------------------------------------------------------------------------------------------------------------------

(*
  Must be adjusted for the latest Unicode version.
const
  // associated languages to particular Unicode code block,
  CodeBlockToLanguage: array[TUnicodeBlock] of LCID =
  (
    LANG_NEUTRAL, // Basic Latin
    LANG_NEUTRAL, // Latin-1 Supplement
    LANG_NEUTRAL, // Latin Extended-A
    LANG_NEUTRAL, // Latin Extended-B
    LANG_NEUTRAL, // IPA Extensions
    LANG_NEUTRAL, // Spacing Modifier Letters
    LANG_NEUTRAL, // Combining Diacritical Marks
    LANG_GREEK, // Greek
    LANG_RUSSIAN, // Cyrillic
    LANG_NEUTRAL, // Armenian
    LANG_HEBREW, // Hebrew
    LANG_ARABIC, // Arabic
    LANG_NEUTRAL, // Devanagari
    LANG_NEUTRAL, // Bengali
    LANG_NEUTRAL, // Gurmukhi
    LANG_NEUTRAL, // Gujarati
    LANG_NEUTRAL, // Oriya
    LANG_NEUTRAL, // Tamil
    LANG_NEUTRAL, // Telugu
    LANG_NEUTRAL, // Kannada
    LANG_NEUTRAL, // Malayalam
    LANG_THAI, // Thai
    LANG_NEUTRAL, // Lao
    LANG_NEUTRAL, // Tibetan
    LANG_NEUTRAL, // Georgian
    LANG_NEUTRAL, // Hangul Jamo
    LANG_NEUTRAL, // Latin Extended Additional
    LANG_GREEK, // Greek Extended
    LANG_NEUTRAL, // General Punctuation
    LANG_NEUTRAL, // Superscripts and Subscripts
    LANG_NEUTRAL, // Currency Symbols
    LANG_NEUTRAL, // Combining Marks for Symbols
    LANG_NEUTRAL, // Letterlike Symbols
    LANG_NEUTRAL, // Number Forms
    LANG_NEUTRAL, // Arrows
    LANG_NEUTRAL, // Mathematical Operators
    LANG_NEUTRAL, // Miscellaneous Technical
    LANG_NEUTRAL, // Control Pictures
    LANG_NEUTRAL, // Optical Character Recognition
    LANG_NEUTRAL, // Enclosed Alphanumerics
    LANG_NEUTRAL, // Box Drawing
    LANG_NEUTRAL, // Block Elements
    LANG_NEUTRAL, // Geometric Shapes
    LANG_NEUTRAL, // Miscellaneous Symbols
    LANG_NEUTRAL, // Dingbats
    LANG_NEUTRAL, // CJK Symbols and Punctuation
    LANG_JAPANESE, // Hiragana
    LANG_JAPANESE, // Katakana
    LANG_NEUTRAL, // Bopomofo
    LANG_NEUTRAL, // Hangul Compatibility Jamo
    LANG_NEUTRAL, // Kanbun
    LANG_CHINESE, // Enclosed CJK Letters and Months
    LANG_CHINESE, // CJK Compatibility
    LANG_CHINESE, // CJK Unified Ideographs
    LANG_NEUTRAL, // Hangul Syllables
    LANG_NEUTRAL, // High Surrogates
    LANG_NEUTRAL, // High Private Use Surrogates
    LANG_NEUTRAL, // Low Surrogates
    LANG_NEUTRAL, // Private Use
    LANG_ARABIC, // CJK Compatibility Ideographs
    LANG_NEUTRAL, // Alphabetic Presentation Forms
    LANG_ARABIC, // Arabic Presentation Forms-A
    LANG_NEUTRAL, // Combining Half Marks
    LANG_CHINESE, // CJK Compatibility Forms
    LANG_NEUTRAL, // Small Form Variants
    LANG_ARABIC, // Arabic Presentation Forms-B
    LANG_NEUTRAL, // Halfwidth and Fullwidth Forms
    LANG_NEUTRAL // Specials
    );

function TCustomUniCodeEdit.ActivateKeyboard(const C: WideChar): Boolean;

// Tries to activate a keyboard layout based on the code block which contains C.
// If the layout can be activated (if needed then it is loaded) then Result is True
// otherwise False.
// Consider this function as being experimental as the character to language conversion
// does not work very well. Need to investigate this further...

var
  CodeBlock: TUnicodeBlock;
  KBID: string;

begin
  CodeBlock := CodeBlockFromChar(Word(C));
  Result := ActivateKeyboardLayout(CodeBlockToLanguage[CodeBlock], 0) <> 0;
  if not Result then
  begin
    KBID := Format('%x', [CodeBlockToLanguage[CodeBlock]]);
    Result := LoadKeyboardLayout(PChar(KBID), KLF_SUBSTITUTE_OK) <> 0;
  end;
end;
*)

//----------------------------------------------------------------------------------------------------------------------

initialization
  Screen.Cursors[crDragMove] := LoadCursor(HInstance, 'DRAGMOVE');
  Screen.Cursors[crDragCopy] := LoadCursor(HInstance, 'DRAGCOPY');
  PlatformIsUnicode := (Win32Platform = VER_PLATFORM_WIN32_NT);
finalization
end.

