/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/**
 * @file myx_gc_base.cpp 
 * @brief Implementation of the base generic canvas class.
 * 
 */

#include "myx_gc_base.h"

//----------------- CGCBase --------------------------------------------------------------------------------------------

CGCBase::CGCBase(void) 
{ 
  _className = "CGCBase"; 
  FDestroying = false;
};

//----------------------------------------------------------------------------------------------------------------------

CGCBase::~CGCBase(void)
{
  // Notify listers about the destruction.
  FDestroying = true;
  for (CGCListenerIterator iterator = FListeners.begin(); iterator != FListeners.end(); ++iterator)
    (*iterator)->onDestroy(this);
  FListeners.clear();
};

//----------------------------------------------------------------------------------------------------------------------

/**
 * Adds a listener to the internal list of listeners, if it is not already there.
 *
 * @param listener The new listener to add to the internal list. A listener is only added once.
 */
void CGCBase::addListener(CGCListener* listener) 
{ 
  if (!FDestroying)
    if (FListeners.find(listener) == FListeners.end()) 
      FListeners.insert(listener); 
};

//----------------------------------------------------------------------------------------------------------------------

/**
 * Triggers the onCange event of all registered listeners to notfied them about a particular change.
 *
 * @param origin The object where the change happend.
 * @param reason What was the reason of that change?
 */
void CGCBase::change(CGCBase* origin, TGCChangeReason reason)
{
  CGCListenerIterator iterator = FListeners.begin(); 
  while (iterator != FListeners.end())
  {
    CGCListenerIterator next = iterator; ++next;
    try
    {
      (*iterator)->onChange(this, origin, reason);
    }
    catch(...)
    {
      // If there was an exception while executing the method the listener is removed from our list
      // to avoid further harm.
      FListeners.erase(iterator);
    };
    iterator = next;
  };
};

//----------------------------------------------------------------------------------------------------------------------

/**
 * Triggers the onError event of all registered listeners to notfied them about an error.
 *
 * @param origin The object where the error happend.
 * @param message A message describing the error.
 */
void CGCBase::error(CGCBase* origin, const char* message)
{
  CGCListenerIterator iterator = FListeners.begin(); 
  while (iterator != FListeners.end())
  {
    CGCListenerIterator next = iterator; ++next;
    try
    {
      (*iterator)->onError(this, origin, message);
    }
    catch(...)
    {
      // If there was an exception while executing the method the listener is removed from our list
      // to avoid further harm.
      FListeners.erase(iterator);
    };
    iterator = next;
  };
};

//----------------------------------------------------------------------------------------------------------------------

void CGCBase::removeListener(CGCListener* listener)
{
  if (!FDestroying)
    FListeners.erase(listener);
};

//----------------------------------------------------------------------------------------------------------------------

