/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/**
 * @file myx_gc_layer.h 
 * @brief Implementation of the GC layer class.
 * 
 */

#ifndef __GC_LAYER_H__
#define __GC_LAYER_H__

#include "myx_gc_canvas.h"

//----------------------------------------------------------------------------------------------------------------------

class CGenericCanvas;
class CFigureInstance;
class CFigure;
class CHitResults;

class CInstanceListener: public CGCListener
{
  friend class CLayer;
protected:
  CLayer* layer;
public:
  virtual void __cdecl onChange(CGCBase* sender, CGCBase* origin, TGCChangeReason reason);
  virtual void __cdecl onDestroy(CGCBase* object);
  virtual void __cdecl onError(CGCBase* sender, CGCBase* origin, const char* message);
};

/**
 * This is the base layer class, which is used by views to display their content. There are descendants for special things
 * like feedback, grids and so on.
 */
class GENERIC_CANVAS_API CLayer: public CGCBase
{
  friend class CFigureInstance;
  friend class CFigureInstanceEnumerator;
private:
  wstring FName;                  // The name of the layer.
  CGenericCanvas* FCanvas;        // The canvas to which this layer belongs (if any).
  bool FDirty;                    // True if any of the properties changed that affect the display list.
  float FScaling[3];              // The factors to scale the layer in each direction.
  float FMinimalScale;            // The lesser of the x and y scale factors.
  float FTranslation[3];          // The factors to move the layer.
  CFigureInstances FInstances;    // A list of figures instances.
  CFigureInstances FGroups;       // A list of groups.
  int FUpdateCount;               // See CGenericCanvas.
  bool FVisible;                  // true if the layer is visible, otherwise false. Default is true.
  bool FEnabled;                  // True if the layer answers to mouse and keyboard actions.
  CInstanceListener FListener;
protected:
  void applyTransformations();
  void checkError(void);
  void makeDirty(void);
  void renderFeedback(CFigureInstance* Instance);
  virtual void renderLayerContent(void);
  void validate(void);
  virtual void validateLayerContent(void);
public:
  CLayer(string Name);
  virtual ~CLayer(void);

  virtual void __cdecl addInstance(CFigureInstance* Instance);
  virtual void __cdecl beginUpdate(void);
  virtual CGenericCanvas* __cdecl canvas(void);
  virtual void __cdecl canvas(CGenericCanvas* NewCanvas);
  virtual void __cdecl clear();
  virtual CFigureInstance* __cdecl createInstance(CFigure* Figure);
  void getHitTestInfoAt(CHitResults* Hits, const float X, const float Y, bool SingleHit);
  virtual void __cdecl enabled(bool IsEnabled);
  virtual bool __cdecl enabled(void);
  virtual void __cdecl endUpdate(void);
  virtual bool __cdecl isUpdating(void);
  wstring name(void) { return FName; };
  virtual TGCVariant __cdecl property(const char* name, unsigned int index);
  virtual void __cdecl property(const char* name, unsigned int index, const TGCVariant& value);
  virtual void __cdecl removeInstance(CFigureInstance* Instance);
  virtual void __cdecl render(float CurrentZoom, TBoundingBox Bounds);
  virtual void __cdecl translate(float Tx, float Ty, float Tz, bool Accumulative = false);
  virtual void __cdecl translateV(const float Factor[3], bool Accumulative = false);
  virtual void __cdecl scale(const float Factor[3], bool Accumulative = false);
  virtual void __cdecl scale(float Sx, float Sy, float Sz, bool Accumulative = false);
  virtual void __cdecl visible(bool IsVisible);
  virtual bool __cdecl visible(void);
};

//----------------------------------------------------------------------------------------------------------------------

/**
 * The grid layer is a special layer variant that renders itself as grid.
 */
class GENERIC_CANVAS_API CGridLayer: public CLayer
{
private:
  CGCView* FView;       // The view to which this grid belongs.
protected:
  virtual void renderLayerContent(void);
public:
  CGridLayer(CGCView* View);
};

//----------------- Selection layer and associated structures ----------------------------------------------------------

typedef struct tagSelectionEntry
{
  CFigureInstance* instance;
  bool dirty;
  TBoundingBox bounds;
} TSelectionEntry;

typedef map<CFigureInstance*, TSelectionEntry*> CSelection;
typedef map<CFigureInstance*, TSelectionEntry*>::iterator CSelectionIterator;
typedef map<CFigureInstance*, TSelectionEntry*>::reverse_iterator CSelectionIteratorReverse;

/** Interal states of the selection layer. */
#define GC_FBSTATE_RUBBERBAND 0x0001

/**
 * The selection layer is a special layer variant that renders decorations for selected figures and can be queried
 * for quick hit tests and lists of selected figures.
 */
class GENERIC_CANVAS_API CFeedbackLayer: public CLayer
{
private:
  CSelection FSelection;          // The list of currently selected figure instances.
  float FHandleSize;              // The size of the decoration handles.
  unsigned int FStates;          // A number of flags holding transient states of the layer.
  TBoundingBox FRubberbandBounds; // The current bounds of the rubber band (if active).
  GLuint FRubberband;             // The display list for the rubber band (if active).
  GLuint FSelectionDecoration;    // The display list for the selection decoration.
protected:
  void createSelectionDecoration(void);
  bool internalAddToSelection(CFigureInstance* Instance);
  void internalRemoveFromSelection(CFigureInstance* Instance);
  virtual void renderLayerContent(void);
  virtual void validateLayerContent(void);
public:
  CFeedbackLayer(string Name);
  virtual ~CFeedbackLayer(void);

  virtual void __cdecl addToSelection(CFigureInstance* Instance);
  virtual void __cdecl clearSelection(void);
  virtual TGCDirection __cdecl getSelectionInfo(float X, float Y, float Zoom);
  virtual void __cdecl invalidateBounds(CFigureInstance* Instance);
  virtual void __cdecl removeFromSelection(CFigureInstance* Instance);
  virtual void __cdecl resizeFiguresStart(int X, int Y, TGCDirection Direction);
  virtual void __cdecl resizeFiguresStop(void);
  virtual void __cdecl resizeFiguresTo(int X, int Y);
  virtual void __cdecl rubberbandResize(int X, int Y, TRBSelectionAction Action);
  virtual void __cdecl rubberbandStart(TRubberbandStyle Style, int X, int Y, bool RemoveSelection);
  virtual void __cdecl rubberbandStop(void);
};

//----------------------------------------------------------------------------------------------------------------------

#endif // __GC_LAYER_H__
