/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/**
 * @file myx_gc_base.h 
 * @brief Implementation of the GC base class from which most other GC classes are derived.
 * 
 */

#ifndef __GC_BASE_H__
#define __GC_BASE_H__

#include "myx_gc.h"
#include "myx_gc_datatypes.h"

//----------------------------------------------------------------------------------------------------------------------

class CGCBase;

/**
 * The general listener class is used to notify users of the canvas about general events like repaints and errors.
 * This class is only an abstract class and must get a concrete implemention in the application.
 * All Listener classes are meant to be a means for calling back the application. They are implemented and instantiated
 * in the application and must be freed there. Don't forget to remove the listener class before you free it!
 */
class GENERIC_CANVAS_API CGCListener
{
public:
  virtual void __cdecl onChange(CGCBase* sender, CGCBase* origin, TGCChangeReason reason) = 0;
  virtual void __cdecl onDestroy(CGCBase* sender) = 0;
  virtual void __cdecl onError(CGCBase* sender, CGCBase* origin, const char* message) = 0;
};

typedef set<CGCListener*> CGCListeners;
typedef set<CGCListener*>::iterator CGCListenerIterator;

/**
 * CGCBase serves as general base class for all generic canvas classes.
 */
class GENERIC_CANVAS_API CGCBase
{
private:
   CGCListeners FListeners;
   bool FDestroying;
protected:
  string _className; // Used to determine the actual class.
  void setDestroying(void) { FDestroying = true; }; // Helper to set destroying state explicitely.
public:
  CGCBase(void); 
  virtual ~CGCBase(void);

  virtual void __cdecl addListener(CGCListener* listener); 
  virtual void __cdecl change(CGCBase* origin, TGCChangeReason reason);
  string className(void) { return _className; };
  virtual bool __cdecl destroying(void) { return FDestroying; };
  virtual void __cdecl error(CGCBase* origin, const char* message);
  virtual const char* __cdecl getClassName(void) { return _className.c_str(); };
  virtual TGCVariant __cdecl property(const char* name, unsigned int index) = 0;
  virtual void __cdecl property(const char* name, unsigned int index, const TGCVariant& value) = 0;
  virtual void __cdecl release(void) { delete this; }; 
  virtual void __cdecl removeListener(CGCListener* listener);
};

//----------------------------------------------------------------------------------------------------------------------

#endif // __GC_BASE_H__
