/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */


#ifdef ENABLE_JAVA_MODULES

#include "myx_grt_java.h"

#ifdef __GNUC__
#define JAVA_PATH_SEPARATOR ":"
#else
#define JAVA_PATH_SEPARATOR ";"
#endif


#define GREF(env, r) (*env)->NewGlobalRef(env, r)

static MYX_GRT_ERROR java_call_function(MYX_GRT_FUNCTION *function, MYX_GRT_VALUE *value, MYX_GRT_VALUE **retval);
static MYX_GRT_ERROR java_init_module(MYX_GRT_MODULE_LOADER *loader, const char *file, MYX_GRT_MODULE **retmodule);
static MYX_GRT_ERROR java_fetch_messages(MYX_GRT_MODULE_LOADER *loader, MYX_GRT_MSGS *msgs);

static jobject j_xml_object_from_grt_value(MYX_JAVA_LOADER *jenv, MYX_GRT_VALUE *value);
static MYX_GRT_VALUE *j_grt_value_from_java_xml_object(MYX_JAVA_LOADER *jenv, jobject object);
static jobject j_call_method(JNIEnv *env, jclass theclass, jobject object, const char *name, const char *signature, jvalue *args);
static jobject j_call_static_method(JNIEnv *env, jclass theclass, const char *name, const char *signature, jvalue *args);
static char *j_get_string(JNIEnv *env, jobject strobj);
static MYX_GRT_LIST *j_get_string_list(MYX_JAVA_LOADER *jloader, jobject lobj);


MYX_GRT_MODULE_LOADER *myx_java_init_loader(MYX_GRT *grt, const char *class_path, MYX_GRT_ERROR *error)
{
  MYX_GRT_MODULE_LOADER *loader= g_new0(MYX_GRT_MODULE_LOADER, 1);
  MYX_JAVA_LOADER *priv= g_new0(MYX_JAVA_LOADER, 1);
  JavaVMInitArgs vm_init_args;
  JavaVMOption options[1];
  jclass grt_class;
  static char *file_extensions[]= {
    ".class"
  };
  int res;
  const char *grt_class_path= "java";
  const char *grt_class_path_libraries_dir= "java/lib";
  char *grt_class_path_libraries= NULL;
  GDir *dir;
  const char *entry;

  *error= MYX_GRT_NO_ERROR;

  loader->grt= grt;
  loader->loader_type= MYX_JAVA_MODULE_TYPE;
  loader->priv= priv;
  loader->init_module= java_init_module;
  loader->call_function= java_call_function;
  loader->fetch_messages= java_fetch_messages;
  loader->extensions_num= 1;
  loader->extensions= file_extensions;

  vm_init_args.version = JNI_VERSION_1_4;
  vm_init_args.nOptions = 1;

  //scan for .jars in the lib directory
  dir= g_dir_open(grt_class_path_libraries_dir, 0, NULL);
  if (!dir)
  {
    *error= MYX_GRT_BAD_PATH;
    goto errorfree;
  }

  while ((entry= g_dir_read_name(dir)) != NULL)
  {
    char *path= g_build_filename(grt_class_path_libraries_dir, entry, NULL);

    grt_class_path_libraries= str_g_append_and_free(
        grt_class_path_libraries, g_strdup_printf("%s%s", 
          path, JAVA_PATH_SEPARATOR));

    g_free(path);
  }
  g_dir_close(dir);

  //base class path
  options[0].optionString= g_strdup_printf("-Djava.class.path=%s%s", grt_class_path, JAVA_PATH_SEPARATOR);

  //add library path to class path
  if (grt_class_path_libraries)
    options[0].optionString= str_g_append_and_free(options[0].optionString,
      grt_class_path_libraries);

  //add user defined class_path to class path
  if (class_path)
    options[0].optionString= str_g_append(options[0].optionString,
      class_path); 

  vm_init_args.options = options;
  vm_init_args.ignoreUnrecognized = JNI_FALSE;

  //Try launch the Java VM
  res= JNI_CreateJavaVM(&priv->jvm, (void **)&priv->env, &vm_init_args);

  g_free(options[0].optionString);

  if (res < 0)
    goto error;

  // Search class reference for the java Grt class
  grt_class= (*priv->env)->FindClass(priv->env, "com/mysql/grt/Grt");
  if (!grt_class)
  {
    if (getenv("GRT_VERBOSE"))
      g_warning("Could not load java class com.mysql.grt.Grt.");
    goto error;
  }
  priv->grt_class= (*priv->env)->NewGlobalRef(priv->env, grt_class);

  // Search for the function callModuleFunction in the Grt class
  priv->grt_call_func= (*priv->env)->GetStaticMethodID(priv->env, 
     priv->grt_class, "callModuleFunction", "(Ljava/lang/Class;Ljava/lang/String;Ljava/lang/String;Ljava/lang/String;)Ljava/lang/String;");
  if (!priv->grt_call_func)
  {
    if (getenv("GRT_VERBOSE"))
      g_warning("The java function callModuleFunction cannot be found in the java class com.mysql.grt.Grt.");
    goto error;
  }

  // Register the Logging class
  //XXX
  
  return loader;
  
error:
  *error= MYX_GRT_MODULE_INIT_ERROR;
errorfree:
  g_free(priv);
  g_free(loader);
  return NULL;
}


static MYX_GRT_ERROR java_init_module(MYX_GRT_MODULE_LOADER *loader, const char *file, MYX_GRT_MODULE **retmodule)
{
  MYX_JAVA_LOADER *jloader= loader->priv;
  MYX_GRT_MODULE *module;
  MYX_JAVA_MODULE *jmodule;
  jclass theclass;
  jobject module_info;
  unsigned int i;
  char *name, *ptr, *class_name;
  char *module_name= NULL;
  MYX_GRT_VALUE *module_functions= NULL;
  char *extends= NULL;

  class_name = g_path_get_basename(file);
  
  ptr= strchr(class_name, '.');
  if (ptr)
    *ptr= 0;

  name= g_strdup_printf("com/mysql/grt/modules/%s", class_name);
  g_free(class_name);

  theclass= (*jloader->env)->FindClass(jloader->env, name);
  if (!theclass)
  {
    (*jloader->env)->ExceptionDescribe(jloader->env);
    (*jloader->env)->ExceptionClear(jloader->env);

    if (getenv("GRT_VERBOSE"))
      g_warning("Could not load java class %s", name);
    g_free(name);
    return MYX_GRT_MODULE_INIT_ERROR;
  }

  // fetch info about the module
  module_info= j_call_static_method(jloader->env, theclass,
                                    "getModuleInfo", "()Ljava/lang/String;", NULL);

  if (module_info)
  {
    MYX_GRT_VALUE *grt_info;

    grt_info= j_grt_value_from_java_xml_object(jloader, module_info);

    if (!grt_info || myx_grt_value_get_type(grt_info) != MYX_DICT_VALUE)
      g_warning("could not parse xml response data from %s",
                file);
    else
    {
      char *name_start= g_strrstr(myx_grt_dict_item_as_string(grt_info, "name"), ".");
      const char *extends= myx_grt_dict_item_as_string(grt_info, "extends");

      if (name_start)
        module_name= g_strdup(name_start+1);
      else
        module_name= g_strdup(myx_grt_dict_item_as_string(grt_info, "name"));

      module_functions= myx_grt_dict_get_item_value(grt_info, "functions");
      if (module_functions && myx_grt_value_get_type(module_functions)==MYX_LIST_VALUE)
        myx_grt_value_retain(module_functions);
      else
        module_functions= NULL;

      if ((extends) && (extends[0]))
        extends= g_strdup(extends);
    }
    (*jloader->env)->DeleteLocalRef(jloader->env, module_info);
    
    if (!grt_info)
      return MYX_GRT_BAD_MODULE;

    myx_grt_value_release(grt_info);
  }
  else
  {
    // No exception handling needed here because it is handled
    // directly in j_call_static_method

    if (getenv("GRT_VERBOSE"))
      g_warning("Module %s doesn't implement getModuleInfo", file);
    return MYX_GRT_BAD_MODULE;
  }

  if (!module_name || !module_functions)
  {
    if (getenv("GRT_VERBOSE"))
    {
      if (!module_name)
        g_warning("Module info from %s doesn't contain 'name'", file);
      if (!module_functions)
        g_warning("Module info from %s doesn't contain 'functions'", file);
    }
    g_free(module_name);
    g_free(extends);
    if (module_functions)
      myx_grt_value_release(module_functions);
    g_free(name);
    return MYX_GRT_BAD_MODULE;
  }

  // init internal module descriptor
  module= g_new0(MYX_GRT_MODULE, 1);
  jmodule= g_new0(MYX_JAVA_MODULE, 1);

  module->loader= loader;
  module->priv= jmodule;
  module->name= module_name;
  module->path= name;
  module->functions_num= myx_grt_list_item_num(module_functions);
  module->functions= g_new0(MYX_GRT_FUNCTION, module->functions_num);
  for (i= 0; i < module->functions_num; i++)
  {
    MYX_GRT_FUNCTION *func= module->functions+i;
    MYX_JAVA_FUNCTION *jfunc= g_new0(MYX_JAVA_FUNCTION, 1);
    char *tmp= g_strdup(myx_grt_value_as_string(myx_grt_list_get_item(module_functions, i)));
    char *return_type;
    
    func->module= module;

    // do not use myx_grt_parse_function_spec here
    // since we need special handling for the java signature
    func->name= g_strdup(strtok(tmp, ":"));
    jfunc->java_signature= g_strdup(strtok(NULL, ":"));
    func->param_struct_name= NULL;
    return_type= strtok(NULL, ":");
    if ((return_type) && (return_type[0]))
      func->return_struct_name= g_strdup(return_type);
  
    func->priv= jfunc;

    g_free(tmp);
  }
  myx_grt_value_release(module_functions);
  module->extends= extends;

  // java specific module info
  jmodule->class_ref= (*jloader->env)->NewGlobalRef(jloader->env, theclass);

  *retmodule= module;
  
  if (getenv("GRT_VERBOSE"))
    g_message("Initialized module %s", name);

  return MYX_GRT_NO_ERROR;
}

static MYX_GRT_ERROR java_call_function(MYX_GRT_FUNCTION *function, MYX_GRT_VALUE *value, MYX_GRT_VALUE **retval)
{
  MYX_JAVA_LOADER *jloader= function->module->loader->priv;
  MYX_JAVA_MODULE *jmodule= function->module->priv;
  MYX_JAVA_FUNCTION *jfunc= function->priv;
  JNIEnv *env= jloader->env;
  jobject res= NULL;
  jvalue args[4];

  args[0].l= jmodule->class_ref;
  args[1].l= (*env)->NewStringUTF(env, function->name);
  args[2].l= (*env)->NewStringUTF(env, jfunc->java_signature);
  args[3].l= j_xml_object_from_grt_value(jloader, value);

  res= (*env)->CallStaticObjectMethodA(env, jloader->grt_class, jloader->grt_call_func, args);
  if (res)
  {
    *retval= j_grt_value_from_java_xml_object(jloader, res);
    (*env)->DeleteLocalRef(env, res);
  }
  else
    *retval= NULL;
  (*env)->DeleteLocalRef(env, args[1].l);
  (*env)->DeleteLocalRef(env, args[2].l);
  (*env)->DeleteLocalRef(env, args[3].l);

  if (!res)
  {
    (*env)->ExceptionDescribe(env);
    (*env)->ExceptionClear(env);

    return MYX_GRT_FUNCTION_CALL_ERROR;
  }

  return MYX_GRT_NO_ERROR;
}

/*
static MYX_GRT_ERROR java_call_function(MYX_GRT_FUNCTION *function, MYX_GRT_VALUE *value, MYX_GRT_VALUE **retval)
{
  MYX_JAVA_LOADER *jloader= function->module->loader->priv;
  jmethodID methodID;
  char *signature;
  
  if (getenv("GRT_VERBOSE"))
    g_message("Calling java function %s.%s", function->module->name, function->name);

  signature= "(Ljava/lang/String;)Ljava/lang/String;";

  methodID= (*jloader->env)->GetStaticMethodID(jloader->env, 
                                               function->module->priv->class_ref,
                                               function->name,
                                               signature);
  if (methodID)
  {
    JNIEnv *env= jloader->env;
    jobject res= NULL;
    jvalue args[1];

    args[0].l= j_xml_object_from_grt_value(jloader, value);
    
    res= (*env)->CallStaticObjectMethodA(env, function->module->priv->class_ref, methodID, args);
    if (res)
      *retval= j_grt_value_from_java_xml_object(jloader, res);
    else
      return MYX_GRT_FUNCTION_CALL_ERROR;

    return MYX_GRT_NO_ERROR;
  }

  g_warning("Can't find method ID '%s' for java module '%s' (%s).",
            function->name, function->module->name, function->module->path);

  return MYX_GRT_INTERNAL_ERROR;
}*/


static MYX_GRT_ERROR java_fetch_messages(MYX_GRT_MODULE_LOADER *loader, MYX_GRT_MSGS *msgs)
{
  // fetch messages here and put them in msgs
  
  return MYX_GRT_INTERNAL_ERROR;
}

//-----------------------------------------------------------------------------
// Private Stuff
//-----------------------------------------------------------------------------


static jobject j_call_method(JNIEnv *env, jclass theclass, jobject object, const char *name, const char *signature, jvalue *args)
{
  jmethodID methodID;
  jobject res= NULL;

  methodID= (*env)->GetMethodID(env, theclass, name, signature);
  if (methodID)
    res= (*env)->CallObjectMethodA(env, object, methodID, args);
  else
  {
    if (getenv("GRT_VERBOSE"))
      g_warning("Method '%s' / %s not found in class", name, signature);
  }
    
  return res;
}


static jobject j_call_static_method(JNIEnv *env, jclass theclass, const char *name, const char *signature, jvalue *args)
{
  jobject res = NULL;
  jmethodID methodID;

  methodID= (*env)->GetStaticMethodID(env, theclass, name, signature);
  if (methodID)
    res= (*env)->CallStaticObjectMethodA(env, theclass, methodID, args);
  else
  {
    //Clear the exception if the Method cannot be found
    (*env)->ExceptionClear(env);

    if (getenv("GRT_VERBOSE"))
      g_warning("Static method '%s' / %s not found in class", name, signature);
  }

  return res;
}


static char *j_get_string(JNIEnv *env, jobject strobj)
{
  const char *jstr= (*env)->GetStringUTFChars(env, strobj, 0);
  char *str= NULL;

  if (jstr)
  {
    str= g_strdup(jstr);

    (*env)->ReleaseStringUTFChars(env, strobj, jstr);
  }
  else
    str= g_strdup("");

  return str;
}


static jobject j_xml_object_from_grt_value(MYX_JAVA_LOADER *jenv, MYX_GRT_VALUE *value)
{
  jobject jstr= NULL;
  char *data;

  data= myx_grt_value_to_xml(value);
  if (data)
  {
    jstr= (*jenv->env)->NewStringUTF(jenv->env, data);

    g_free(data);
  }

  return jstr;
}


static MYX_GRT_VALUE *j_grt_value_from_java_xml_object(MYX_JAVA_LOADER *jenv, jobject strobj)
{
  MYX_GRT_VALUE *value= NULL;
  const char *jstr= (*jenv->env)->GetStringUTFChars(jenv->env, strobj, 0);

  if (jstr)
  {
    value= myx_grt_value_from_xml(jstr, strlen(jstr));

    (*jenv->env)->ReleaseStringUTFChars(jenv->env, strobj, jstr);
  }
  return value;
}

#endif
