/*
 * mxallowd
 * (c) 2007-2008 Michael Stapelberg
 * http://michael.stapelberg.de/mxallowd
 *
 *
 * mxallowd is a daemon which uses libnetfilter_queue and iptables
 * to allow (or deny) connections to a mailserver (or similar
 * application) if the remote host hasn't connected to a
 * fake daemon before.
 *
 * This is an improved version of the so-called nolisting 
 * (see http://www.nolisting.org/). The assumption is that spammers
 * are not using RFC 2821-compatible SMTP-clients and are
 * sending fire-and-forget spam (directly to the first or second
 * MX-entry without retrying on error). This direct access is
 * blocked with mxallowd, you'll only get a connection if you
 * retry.
 *
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 * Or, point your browser to http://www.gnu.org/copyleft/gpl.html
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>
#include <unistd.h>
#include <linux/ip.h>
#include <netinet/ip6.h>
#include <linux/netfilter.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <getopt.h>
#include <string.h>
#include <signal.h>
#include <pthread.h>

#include <libnetfilter_queue/libnetfilter_queue.h>

#define _MXALLOWD_INTERNAL
#include "mxallowd.h"
#undef _MXALLOWD_INTERNAL
#include "log.h"
#include "config.h"
#include "whitelist.h"
#include "resolve.h"

#ifndef NFQ_PACKET_BUFFER_SIZE
#define NFQ_PACKET_BUFFER_SIZE	4096
#endif
#define MXALLOWD_VERSION	"1.5"

/*
 * Adds a mailserver to the list of fake or real mailservers (configuration)
 *
 */
void add_mailserver(struct mailserver_list **root, char *mailserver_ip) {
	struct mailserver_list *new = malloc(sizeof(struct mailserver_list));
	if (new == NULL)
		diep("malloc()");
	new->ip_address = strdup(mailserver_ip);
	new->next = NULL;
	if (*root == NULL)
		*root = new;
	else {
		struct mailserver_list *cur = *root;
		while (cur->next != NULL)
			cur = cur->next;
		cur->next = new;
	}
}

/*
 * Checks whether a mailserver is included in the list of fake or real mailservers
 *
 */
bool is_included(struct mailserver_list *root, char *ip_address) {
	struct mailserver_list *cur = root;
	do {
		if (strcmp(cur->ip_address, ip_address) == 0)
			return true;
	} while ((cur = cur->next) != NULL);
	return false;
}

/*
 * Helper function to decide what to do with packets sent by iptables
 *
 */
static int handlePacket(struct nfq_q_handle *qh, struct nfgenmsg *nfmsg,
			 struct nfq_data *nfdata, void *packet_buffer) {
	(void)nfmsg;
	(void)packet_buffer;

	struct nfqnl_msg_packet_hdr *header;
	char *payload = NULL;
	int payload_length;

	if (!(header = nfq_get_msg_packet_hdr(nfdata)))
		/* Skip the packet if the header could not be obtained */
		return -1;

	if ((payload_length = nfq_get_payload(nfdata, (char**)&payload)) == -1)
		/* Skip the packet if it has no payload */
		return -1;

	/* OK, we'll shortly need to decide, so let's clean up the whitelists here */
	cleanup_whitelist();

	char dest_address[INET6_ADDRSTRLEN+1],
	     source_address[INET6_ADDRSTRLEN+1];
	memset(dest_address, '\0', INET6_ADDRSTRLEN+1);
	memset(source_address, '\0', INET6_ADDRSTRLEN+1);

	struct iphdr* ip_header = (struct iphdr*)payload;
	if (ip_header->version == IPVERSION) {
		if (inet_ntop(AF_INET, &(ip_header->daddr), dest_address, sizeof(dest_address)) == NULL ||
		    inet_ntop(AF_INET, &(ip_header->saddr), source_address, sizeof(source_address)) == NULL)
			return -1;
	} else {
#ifdef IPV6
		struct ip6_hdr *ip6_header = (struct ip6_hdr*)payload;
		if (inet_ntop(AF_INET6, &(ip6_header->ip6_dst), dest_address, sizeof(dest_address)) == NULL ||
		    inet_ntop(AF_INET6, &(ip6_header->ip6_src), source_address, sizeof(source_address)) == NULL)
			/* Address could not be read, packet has to be malformed */
			return -1;
#endif
	}

	/* Let's see if the packet was sent to MX1 */
	if (is_included(fake_mailservers, dest_address)) {
		/* This packet was sent to MX1, whitelist the sender for MX2 */
		slog("Successful connection from %s to a fake mailserver, adding to whitelist\n", source_address);
		add_to_whitelist(source_address, NULL, false);
	} else if (is_included(real_mailservers, dest_address)) {
		/* This packet was sent to MX2, let's see what to do */
		if (!is_whitelisted(source_address, true)) {
			/* The sender is not whitelisted, so we drop the packet */
			slog("Dropping connection attempt from %s to a real mailserver\n", source_address);
			blocked_attempts++;
			return nfq_set_verdict(qh, ntohl(header->packet_id), NF_DROP, 0, NULL);
		}
		slog("Successful connection from %s to a real mailserver\n", source_address);
	}
	return nfq_set_verdict(qh, ntohl(header->packet_id), NF_ACCEPT, 0, NULL);
}

/*
 * Handle SIGHUP (reload configuration)
 *
 */
void handle_sighup(int sig) {
	slog("Reloading configuration (upon SIGHUP)\n");
	read_configuration(configfile);
}

/*
 * Handle SIGUSR1 (print out some statistics)
 *
 */
void handle_sigusr1(int sig) {
	cleanup_whitelist();
	slog("Statistics: %lu blocked connection attempts, %lu successful connections, %lu direct-to-fake-mx\n",
		blocked_attempts, successful_connects, direct_to_fake);
}


void print_help(char *executable) {
	printf("\n\
mxallowd %s (c) 2007-2008 Michael Stapelberg\n\
(IPv6 %s in this build)\n\
\n\
Syntax: %s [-F] [-d] [-c configfile] [-t 3600] [-s] "
"[-q]"
"\n\t -f 192.168.1.254 -r 192.168.1.2 -n 23\n\
\n\
-c\t--config\n\
\tSpecify the path to a configuration file (default\n\
\t/etc/mxallowd.conf)\n\n\
-f\t--fake-mailserver\n\
\tSpecify which IP-address of the fake mailserver\n\
\t(connecting will whitelist you for the real server)\n\n\
-F\t--foreground\n\
\tDo not fork into background, stay on console\n\n\
-r\t--real-mailserver\n\
\tSpecify which IP-address the real mailserver has\n\n\
-t\t--whitelist-time\n\
\tSpecify the amount of time (in seconds) until an\n\
\tIP-address will be removed from the whitelist\n\n\
-d\t--no-rdns-whitelist\n\
\tDisable whitelisting all IP-addresses that have\n\
\tthe same RDNS as the connecting one (for google\n\
\tmail it is necessary to enable this!)\n\n\
-s\t--stdout\n\
\tLog to stdout, not to syslog\n\n\
-q\t--quiet\n\
\tDon't log anything but errors\n\n"
"-n\t--queue-num\n\
\tThe queue number which will be used (--queue-num in NFQUEUE target)\n\n"
"-h\t--help\n\
\tDisplay this help\n\
\n", MXALLOWD_VERSION,
#ifdef IPV6
"supported",
#elif
"not supported",
#endif
executable);
}

int main(int argc, char **argv) {
	/* Parse command line options */
	int o, option_index = 0;
	static struct option long_options[] = {
		{"fake-mailserver", required_argument, 0, 'f'},
		{"foreground", required_argument, 0, 'F'},
		{"real-mailserver", required_argument, 0, 'r'},
		{"whitelist-time", required_argument, 0, 't'},
		{"no-rdns-whitelist", no_argument, 0, 'd'},
		{"stdout", no_argument, 0, 's'},
		{"help", no_argument, 0, 'h'},
		{"config", required_argument, 0, 'c'},
		{"queue-num", required_argument, 0, 'n'},
		{"quiet", no_argument, 0, 'q'},
		{0, 0, 0, 0}
	};
	char *options_string = "f:r:t:c:n:shdFq";

	/* First loop is for looking if -c exists and reading the configuration */
	while ((o = getopt_long(argc, argv, options_string, long_options, &option_index)) != -1) {
		if (o == 'c') {
			configfile = optarg;
			break;
		} else continue;
	}       
        
	/* If the parameter was not given, try to read the default config */
	if (!read_configuration(configfile) && argc == 1) {
		fprintf(stderr, "Error: Missing arguments and no configuration file could be read\n");
		print_help(argv[0]);
		return 1;
	}
	/* Restart parsing */
#ifdef __GLIBC__
	optind = 0;
#else
	optind = 1;
#endif
#ifdef HAVE_OPTRESET
	/* For BSD */
	optreset = 1;
#endif

	while ((o = getopt_long(argc, argv, options_string, long_options, &option_index)) != -1) {
		switch (o) {
			case 'f':
				add_mailserver(&fake_mailservers, optarg);
				break;
			case 'F':
				stay_in_foreground = true;
				break;
			case 'r':
				add_mailserver(&real_mailservers, optarg);
				break;
			case 't':
				allow_time = atoi(optarg);
				break;
			case 's':
				to_stdout = true;
				break;
			case 'd':
				rdns_whitelist = false;
				break;
			case 'h':
				print_help(argv[0]);
				return 0;
			case 'q':
				quiet = true;
				break;
			case 'n':
				queue_num = atoi(optarg);
				break;
		}
	}

	if (fake_mailservers == NULL || real_mailservers == NULL) {
		fprintf(stderr, "Error: fake and real mailserver have to be specified\n");
		print_help(argv[0]);
		return 1;
	}

	if (queue_num == -1) {
		fprintf(stderr, "Error: queue-num has to be specified\n");
		print_help(argv[0]);
		return 1;
	}

	if (!stay_in_foreground) {
		slog("Forking into background...\n");
		int pid = fork();
		if (pid < 0)
			diep("fork()");
		else if (pid > 0)
			return 0;
	}

	/* Setup signal handlers */
	if (signal(SIGHUP, handle_sighup) == SIG_ERR)
		diep("signal(SIGHUP)");

	if (signal(SIGUSR1, handle_sigusr1) == SIG_ERR)
		diep("signal(SIGUSR1)");

	if (!quiet) {
		slog("mxallowd %s starting...\n", MXALLOWD_VERSION);
		struct mailserver_list *cur = fake_mailservers;
		do {
			slog("Fake Mailserver: %s\n", cur->ip_address);
		} while ((cur = cur->next) != NULL);
		cur = real_mailservers;
		do {
			slog("Real Mailserver: %s\n", cur->ip_address);
		} while ((cur = cur->next) != NULL);
	}

	/* Create a libnetfilterqueue-handle */
	struct nfq_handle *nfq = nfq_open();
	if (nfq == NULL)
		diem("Error creating an nfq-handle");

	/* Unbind to clean up previous instances */
#ifdef BROKEN_UNBIND
	(void)nfq_unbind_pf(nfq, AF_INET);
#else
	if (nfq_unbind_pf(nfq, AF_INET) != 0)
		diem("Error unbinding AF_INET");
#endif
	/* Bind to IPv4 */
	if (nfq_bind_pf(nfq, AF_INET) != 0)
		diem("Error binding to AF_INET");

#ifdef IPV6
	/* Unbind to clean up previous instances */
#ifdef BROKEN_UNBIND
	(void)nfq_unbind_pf(nfq, AF_INET6);
#else
	if (nfq_unbind_pf(nfq, AF_INET6) != 0)
		diem("Error unbinding AF_INET6");
#endif
	/* Bind to IPv6 */
	if (nfq_bind_pf(nfq, AF_INET6) != 0)
		diem("Error binding to AF_INET6");
#endif

	/* Create queue */
	struct nfq_q_handle *qh;
	if (!(qh = nfq_create_queue(nfq, queue_num, &handlePacket, NULL)))
		dief("Error creating queue %d\n", queue_num);

	/* We need a copy of the packet */
	if (nfq_set_mode(qh, NFQNL_COPY_PACKET, NFQ_PACKET_BUFFER_SIZE) == -1)
		diem("Cannot set mode to NFQNL_COPY_PACKET");

	char *packet_buffer = malloc(sizeof(unsigned char) * (NFQ_PACKET_BUFFER_SIZE + 1));
	if (packet_buffer == NULL)
		diep("malloc()");

	/* Create resolver thread if necessary */
	pthread_t resolv_thread;
	if (	rdns_whitelist &&
		(pthread_mutex_init(&resolv_thread_mutex, NULL) != 0 ||
		 pthread_cond_init(&resolv_new_cond, NULL) != 0 ||
		 pthread_create(&resolv_thread, NULL, resolve_thread, NULL) != 0))
		diep("Cannot create thread");

	int fd = nfq_fd(nfq), rv;
	/* Read a packet in blocking mode */
	while ((rv = recv(fd, packet_buffer, NFQ_PACKET_BUFFER_SIZE, 0)) >= 0)
		nfq_handle_packet(nfq, packet_buffer, rv);

	nfq_destroy_queue(qh);
	nfq_close(nfq);
	free(packet_buffer);
	return 0;
}
