/***************************************************************************
 Mutella - A commandline/HTTP client for the Gnutella filesharing network.

 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation; either version 2
 of the License, or (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 tstring.h  -  STL-like string class, for portability.
 
    begin                : Thu May 31 2001
    copyright            : (C) 2001 by
    email                : maksik@gmx.co.uk
 ***************************************************************************/

#ifndef __TSTRING_H__INCLUDED__
#define __TSTRING_H__INCLUDED__

#include <stdarg.h>
#include <iostream.h>

#ifdef TSTRING_DEBUG

#include <stdio.h>

#define CHECK_CONSIST check_consist();

#else

#define CHECK_CONSIST

#endif // TSTRING_DEBUG

template<class T>
class TString{
public:
	typedef T* LPT;
	typedef const T* LPCT;
	// constructors
	TString() : m_pBuffer(NULL), m_nBufferSize(0), m_nLength(0) {}
	TString(const TString<T>& orig) : m_pBuffer(NULL), m_nBufferSize(0), m_nLength(0) {
		_reallocNreplace(orig.m_nLength, orig.m_pBuffer);
	}
	TString(LPCT psz) : m_pBuffer(NULL), m_nBufferSize(0), m_nLength(0) {
		_reallocNreplace(_strlen(psz), psz);
	}
	TString(T c) : m_pBuffer(NULL), m_nBufferSize(0), m_nLength(0) {
		_reallocNreplace(1, &c);
	}
	TString(const TString<T>& orig, int nSize) : m_pBuffer(NULL), m_nBufferSize(0), m_nLength(0) {
		_reallocNreplace(_min(orig.m_nLength, nSize), orig.m_pBuffer);
	}
	TString(LPCT psz,int nSize) : m_pBuffer(NULL), m_nBufferSize(0), m_nLength(0) {
		_reallocNreplace(_strnlen(psz,nSize), psz);
	}
	TString(const TString<T>& orig, int nStart, int nLen) : m_pBuffer(NULL), m_nBufferSize(0), m_nLength(0) {
		if (nStart >= orig.m_nLength) return;
		if (nLen > orig.m_nLength-nStart) nLen = orig.m_nLength-nStart;
		if (nLen <= 0) return;
		_reallocNreplace( nLen, orig.m_pBuffer+nStart);
	}
	TString(LPCT psz, int nStart, int nLen) : m_pBuffer(NULL), m_nBufferSize(0), m_nLength(0) {
		int nLenOrig = _strnlen(psz,nStart+nLen);
		if (nStart >= nLenOrig) return;
		if (nLen > nLenOrig-nStart) nLen = nLenOrig-nStart;
		if (nLen <= 0) return;
		_reallocNreplace( nLen, psz+nStart);
	}
	// destructor
	~TString(){ clear(); }
	// assignment
	const TString<T>& operator=(const TString<T>& orig) {
		CHECK_CONSIST
		_reallocNreplace(orig.m_nLength, orig.m_pBuffer);
		CHECK_CONSIST
		return *this;
	}
	const TString<T>& operator=(LPCT psz) {
		CHECK_CONSIST
		_reallocNreplace(_strlen(psz), psz);
		CHECK_CONSIST
		return *this;
	}
	const TString<T>& operator=(T c) {
		CHECK_CONSIST
		if (c == _null){
			clear();
			return *this;
		}
		T str[2] = { c, _null };
		_reallocNreplace(1, str);
		CHECK_CONSIST
		return *this;
	}
	// assignments of partial strings
	void assign(const TString<T>& orig, int nSize){
		CHECK_CONSIST
		_reallocNreplace(_min(orig.m_nLength, nSize), orig.m_pBuffer);
		CHECK_CONSIST
	}
	void assign(LPCT psz,int nSize){
		CHECK_CONSIST
		_reallocNreplace(_strnlen(psz,nSize), psz);
		CHECK_CONSIST
	}
	void assign(const TString<T>& orig, int nStart, int nLen) {
		if (nStart >= orig.m_nLength) return;
		if (nLen > orig.m_nLength-nStart) nLen = orig.m_nLength-nStart;
		if (nLen <= 0) return;
		CHECK_CONSIST
		_reallocNreplace( nLen, orig.m_pBuffer+nStart);
		CHECK_CONSIST
	}
	void assign(LPCT psz, int nStart, int nLen){
		int nLenOrig = _strnlen(psz,nStart+nLen);
		if (nStart >= nLenOrig) return;
		if (nLen > nLenOrig-nStart) nLen = nLenOrig-nStart;
		if (nLen <= 0) return;
		CHECK_CONSIST
		_reallocNreplace( nLen, psz+nStart);
		CHECK_CONSIST
	}
	// concatetation
	const TString<T>& operator+=(const TString<T>& rhs) {
		CHECK_CONSIST
		_reallocNtail(rhs.m_nLength, rhs.m_pBuffer);
		CHECK_CONSIST
		return *this;
	}
	const TString<T>& operator+=(LPCT pszRhs) {
		CHECK_CONSIST
		_reallocNtail(_strlen(pszRhs), pszRhs);
		CHECK_CONSIST
		return *this;
	}
	const TString<T>& operator+=(T cRhs) {
		CHECK_CONSIST
		if (cRhs == _null) return *this;
		T str[2] = { cRhs, _null };
		_reallocNtail(1, str);
		CHECK_CONSIST
		return *this;
	}
	// functionality
	int length() const {
		CHECK_CONSIST
		return m_nLength;
	}
	int size() const {
		CHECK_CONSIST
		return m_nLength;
	}
	int buffersize() const {
		CHECK_CONSIST
		return m_nBufferSize;
	}
	int capacity() const {
		CHECK_CONSIST
		return m_nBufferSize-1;
	}
	void reserve(int nMaxLength){
		CHECK_CONSIST
		if (nMaxLength+1>m_nBufferSize)
			_realloc(nMaxLength);
	}
	bool empty() const {
		CHECK_CONSIST
		return 0==m_nLength;
	}
	void clear(){
		CHECK_CONSIST
		//delete [] m_pBuffer;
		free(m_pBuffer);
		m_pBuffer = NULL;
		m_nBufferSize = 0;
		m_nLength = 0;
		CHECK_CONSIST
	}
	void cut(int nPos){
		CHECK_CONSIST
		if (nPos<m_nLength)
			if (nPos<=0)
				clear();
			else
			{
				m_nLength = nPos;
				m_pBuffer[nPos]='\0';
				CHECK_CONSIST
			}
	}
	TString<T>& insert(int nPos, const TString<T>& s) {
		CHECK_CONSIST
		if (nPos > m_nLength)
			nPos = m_nLength;
		LPT pOldBuf = m_pBuffer;
		int nOldLen = m_nLength;
		// fake clear()
		m_nBufferSize = 0;
		m_nLength = 0;
		m_pBuffer = NULL;
		//
		_reallocNreplace(nOldLen+s.m_nLength, NULL);
		memcpy(m_pBuffer, pOldBuf, sizeof(T)*nPos);
		memcpy(m_pBuffer+nPos, s.m_pBuffer, sizeof(T)*s.m_nLength);
		memcpy(m_pBuffer+(nPos+s.m_nLength), pOldBuf+nPos, sizeof(T)*(nOldLen-nPos));
		m_nLength = nOldLen+s.m_nLength;
		m_pBuffer[m_nLength] = _null;
		CHECK_CONSIST
		//
		free(pOldBuf);
		return *this;
	}
	TString<T>& insert(int nPos, LPCT psz) {
		CHECK_CONSIST
		if (nPos > m_nLength)
			nPos = m_nLength;
		int nAddLen = _strlen(psz);
		LPT pOldBuf = m_pBuffer;
		int nOldLen = m_nLength;
		// fake clear()
		m_nBufferSize = 0;
		m_nLength = 0;
		m_pBuffer = NULL;
		//
		_reallocNreplace(nOldLen+nAddLen, NULL);
		memcpy(m_pBuffer, pOldBuf, sizeof(T)*nPos);
		memcpy(m_pBuffer+nPos, psz, sizeof(T)*nAddLen);
		memcpy(m_pBuffer+(nPos+nAddLen), pOldBuf+nPos, sizeof(T)*(nOldLen-nPos));
		m_nLength = nOldLen+nAddLen;
		m_pBuffer[m_nLength] = _null;
		CHECK_CONSIST
		//
		free(pOldBuf);
		return *this;
	}
	// search
	int find(const TString<T>& needle, int nStart=0) const {
		CHECK_CONSIST
		if (m_nLength-nStart < needle.m_nLength)
			return -1;
		int nPos = _strstr(m_pBuffer+nStart, needle.m_pBuffer);
		if (nPos>=0)
			return nStart+nPos;
		return -1;
	}
	int find(LPCT pszNeedle, int nStart=0) const {
		CHECK_CONSIST
		if (m_nLength-nStart < _strlen(pszNeedle))
			return -1;
		int nPos = _strstr(m_pBuffer+nStart, pszNeedle);
		if (nPos>=0)
			return nStart+nPos;
		return -1;
	}
	int find(T needle, int nStart=0) const {
		CHECK_CONSIST
		if (m_nLength-nStart < 1)
			return -1;
		int nPos = _strchr(m_pBuffer+nStart, needle);
		if (nPos>=0)
			return nStart+nPos;
		return -1;
	}
	int rfind(const TString<T>& needle, int nStart=0) const {
		CHECK_CONSIST
		if (m_nLength-nStart < needle.m_nLength)
			return -1;
		int nPos = _strrstr(m_pBuffer+nStart, needle.m_pBuffer);
		if (nPos>=0)
			return nStart+nPos;
		return -1;
	}
	int rfind(LPCT pszNeedle, int nStart=0) const {
		CHECK_CONSIST
		if (m_nLength-nStart < _strlen(pszNeedle))
			return -1;
		int nPos = _strrstr(m_pBuffer+nStart, pszNeedle);
		if (nPos>=0)
			return nStart+nPos;
		return -1;
	}
	int rfind(T needle, int nStart=0) const {
		CHECK_CONSIST
		if (m_nLength-nStart < 1)
			return -1;
		int nPos = _strrchr(m_pBuffer+nStart, needle);
		if (nPos>=0)
			return nStart+nPos;
		return -1;
	}
	// substrings
	TString<T> substr(int nStart, int nLen = INT_MAX) const {
		CHECK_CONSIST
		return TString<T>(*this, nStart, nLen == INT_MAX ? m_nLength : nLen);
	}
	// character conversion
	const TString<T>& make_lower(){
		CHECK_CONSIST
		_makelower(m_pBuffer, m_nLength);
		return *this;
	}
	const TString<T>& make_upper(){
		CHECK_CONSIST
		_makeupper(m_pBuffer, m_nLength);
		return *this;
	}
	//
	LPCT c_str() const {
		CHECK_CONSIST
		return m_pBuffer ? m_pBuffer : &_null;
	}
	const T& operator[](int i) const{
		CHECK_CONSIST
#ifdef CHECK_RANGE
		if (i<0||i>m_nLength)
			fprintf(stderr, "TString::operator[] out of range\n");
#endif
		return m_pBuffer[i];
	}
	T& operator[](int i){
		CHECK_CONSIST
#ifdef CHECK_RANGE
		if (i<0||i>m_nLength)
			fprintf(stderr, "TString::operator[] out of range\n");
#endif
		return m_pBuffer[i];
	}
	// sprintf replacement
	inline const TString<T>& format(const char *fmt, ...){
		va_list ap;
		va_start(ap, fmt);
		format(fmt, ap);
		va_end(ap);
		return *this;
	}
	void format(const char *fmt, va_list ap){
		int n;
		int size = _max(m_nBufferSize,16+_strlen(fmt)); // dont ask me why 16
		//if ((p = malloc (size)) == NULL) return NULL;
		while (1) {
			// Ensure we hawe that much space in the buffer
			_reallocNreplace(size, NULL);
			// Try to print in the allocated space
#ifdef VA_COPY
			va_list tmp;      // required for PPC
			VA_COPY(tmp, ap); // required for PPC
			n = _vsnprintf(m_pBuffer, m_nBufferSize, fmt, tmp);
			va_end(ap);       // required for PPC
#else //VA_COPY
			n = _vsnprintf(m_pBuffer, m_nBufferSize, fmt, ap);
#endif //VA_COPY
			// If that worked, return the string
			if (n > -1 && n < m_nBufferSize)
			{
				// make string structure consistent
				m_nLength = _strlen(m_pBuffer);
				return;
			}
			// Else try again with more space
			if (n > -1)     // glibc 2.1
				size = n+1; // precisely what is needed
			else                        // glibc 2.0
				size = m_nBufferSize*2; // twice the old size
		}
	};
protected:
	// methods
	void _realloc(int nLength) { // allocate buffer to hold string of length nLength (no optimisation)
#ifdef CHECK_RANGE
		if (nLength < 0)
			fprintf(stderr, "TString::_reallocNreplace nLength < 0\n");
#endif
		// TODO: shrink buffer if the string is tooo short (check usage of this function before)
		if (nLength+1>m_nBufferSize) {
			m_nBufferSize = nLength+1; // realloc is exact function
			m_pBuffer = (T*) realloc(m_pBuffer, sizeof(T)*m_nBufferSize);
			ASSERT(m_pBuffer);
		}
	}
	void _reallocNtail(int nAdd, LPCT pTail = NULL){
#ifdef CHECK_RANGE
		if ( nAdd < 0 )
			fprintf(stderr, "TString::_reallocNtail nAdd < 0\n");
#endif
		if (m_nLength+1+nAdd>m_nBufferSize){
			_realloc(m_nLength+nAdd+16); // TODO: optimize reallocations
		}
		if (pTail) {
#ifdef CHECK_RANGE
				if (nAdd > _strnlen(pTail, nAdd))
					fprintf(stderr, "TString::_reallocNtail nAdd > _strlen(pTail)\n");
#endif
				memcpy(m_pBuffer+m_nLength, pTail, sizeof(T)*nAdd); // here we trust that nAdd <= strlen(pTail)
				m_nLength += nAdd;
				m_pBuffer[m_nLength] = _null;
		}
	}
	void _reallocNreplace(int nLength, LPCT pContent = NULL ) { // allocate buffer to hold string of length nLength
#ifdef CHECK_RANGE
		if (nLength < 0)
			fprintf(stderr, "TString::_reallocNreplace nLength < 0\n");
#endif
		// TODO: shrink buffer if the string is tooo short (check usage of this function before)
		if (nLength+1>m_nBufferSize) {
			_realloc(nLength+16); // TODO: optimize this +16
		}
		if (pContent) {
#ifdef CHECK_RANGE
			if (nLength > _strnlen(pContent, nLength))
				fprintf(stderr, "TString::_reallocNreplace nLength > _strlen(pContent)\n");
#endif
			memcpy(m_pBuffer, pContent, sizeof(T)*nLength); // here we trust that nLength <= strlen(pContent)
			m_nLength = nLength;
		} else {
			m_nLength = 0;
		}
		m_pBuffer[m_nLength] = _null;
	}
/*	void _reallocNtail(int nAdd, LPCT pTail = NULL){
#ifdef CHECK_RANGE
		if ( nAdd < 0 )
			fprintf(stderr, "TString::_reallocNtail nAdd < 0\n");
#endif
		if (m_nLength+1+nAdd>m_nBufferSize){
			m_nBufferSize = m_nLength+1+nAdd+16; // TODO: optimize reallocations
			T* pNewBuf = new T[m_nBufferSize];
			if (m_pBuffer){
				if (pTail) {
					memcpy(pNewBuf, m_pBuffer, sizeof(T)*m_nLength);
#ifdef CHECK_RANGE
					if (nAdd > _strnlen(pTail, nAdd))
						fprintf(stderr, "TString::_reallocNtail nAdd > _strlen(pTail)\n");
#endif
					memcpy(pNewBuf+m_nLength, pTail, sizeof(T)*nAdd); // here we trust that nAdd <= strlen(pTail)
					m_nLength += nAdd;
					pNewBuf[m_nLength] = _null;
				} else
					memcpy(pNewBuf, m_pBuffer, sizeof(T)*(m_nLength+1)); // this will also copy trailing 0
				delete [] m_pBuffer;
			} else {
			    if (pTail) {
#ifdef CHECK_RANGE
					if (nAdd > _strnlen(pTail, nAdd))
						fprintf(stderr, "TString::_reallocNtail nAdd > _strlen(pTail)\n");
#endif
					memcpy(pNewBuf, pTail, sizeof(T)*nAdd); // here we trust that nAdd <= strlen(pTail)
					m_nLength = nAdd;
					pNewBuf[m_nLength] = _null;
				} else {
					ASSERT(m_nLength == 0);
					pNewBuf[0] = _null;
				}
			}
			m_pBuffer = pNewBuf;
		} else if (pTail) {
#ifdef CHECK_RANGE
			if (nAdd > _strnlen(pTail, nAdd))
				fprintf(stderr, "TString::_reallocNtail nAdd > _strlen(pTail)\n");
#endif
			memcpy(m_pBuffer+m_nLength, pTail, sizeof(T)*nAdd); // here we trust that nAdd <= strlen(pTail)
			m_nLength += nAdd;
			m_pBuffer[m_nLength] = _null;
		}
	}
	void _reallocNreplace(int nLength, LPCT pContent = NULL ) { // allocate buffer to hold string of length nLength
#ifdef CHECK_RANGE
		if (nLength < 0)
			fprintf(stderr, "TString::_reallocNreplace nLength < 0\n");
#endif
		// TODO: shrink buffer if the string is tooo short (check usage of this function before)
		if (nLength+1>m_nBufferSize) {
			m_nBufferSize = nLength+1+16; // TODO: optimize this +16
			if (m_pBuffer)
				delete [] m_pBuffer;
			m_pBuffer = new T[m_nBufferSize];
		}
		if (pContent) {
#ifdef CHECK_RANGE
			if (nLength > _strnlen(pContent, nLength))
				fprintf(stderr, "TString::_reallocNreplace nLength > _strlen(pContent)\n");
#endif
			memcpy(m_pBuffer, pContent, sizeof(T)*nLength); // here we trust that nLength <= strlen(pContent)
			m_nLength = nLength;
		} else {
			m_nLength = 0;
		}
		m_pBuffer[m_nLength] = _null;
	}*/
	// helpers
	int _min(register int a, register int b) { if ( a<b ) return a; return b; }
	int _max(register int a, register int b) { if ( a>b ) return a; return b; }

#ifdef CHECK_RANGE
	void check_consist() const {
		if ( m_nLength && m_nLength >= m_nBufferSize ||
			 m_nLength != _strlen(m_pBuffer) ||
			 m_nLength && !m_pBuffer )
		{
			fprintf(stderr, "TString::consistency check failed\n");
		}
	}
#endif
	static int _strnlen(LPCT psz, int nMax){
		if (!psz) return 0;
		if (nMax<0 || nMax==INT_MAX ) return _strlen(psz);
		for (int i = 0; i<nMax; i++)
			if (psz[i] == _null)
				return i;
		return nMax;
	}
	// "trait" functionality
	static int _strlen(LPCT psz);
	static int _strstr(LPCT stack, LPCT needle); // returns position of the first occurence, or -1
	static int _strchr(LPCT stack, T needle); // same as above
	static int _strrstr(LPCT stack, LPCT needle);
	static int _strrchr(LPCT stack, T needle);
	static int _strcmp(LPCT left, LPCT right); // returns -1, 0 or 1
	static void _makelower(LPT str, int len);
	static void _makeupper(LPT str, int len);
	static int _vsnprintf(LPT str, int size, const char *format, va_list ap);
	static T _null;
	// data
	T* m_pBuffer;
	int m_nBufferSize;
	int m_nLength;
	// friends
	friend bool operator== <>(const TString<T>& lhs, const TString<T>& rhs);
	friend bool operator== <>(const TString<T>& lhs, typename TString<T>::LPCT pszRhs);
	friend bool operator== <>(typename TString<T>::LPCT pszLhs, const TString<T>& rhs);
	friend bool operator== <>(const TString<T>& lhs,T cRhs);
	friend bool operator== <>(T cLhs,const TString<T>& rhs);
	friend bool operator!= <>(const TString<T>& lhs, const TString<T>& rhs);
	friend bool operator!= <>(const TString<T>& lhs, typename TString<T>::LPCT pszRhs);
	friend bool operator!= <>(typename TString<T>::LPCT pszLhs, const TString<T>& rhs);
	friend bool operator!= <>(const TString<T>& lhs,T cRhs);
	friend bool operator!= <>(T cLhs,const TString<T>& rhs);
	friend bool operator> <>(const TString<T>& lhs, const TString<T>& rhs);
	friend bool operator> <>(const TString<T>& lhs, typename TString<T>::LPCT pszRhs);
	friend bool operator> <>(typename TString<T>::LPCT pszLhs, const TString<T>& rhs);
	friend bool operator> <>(const TString<T>& lhs,T cRhs);
	friend bool operator> <>(T cLhs,const TString<T>& rhs);
	friend bool operator< <>(const TString<T>& lhs, const TString<T>& rhs);
	friend bool operator< <>(const TString<T>& lhs, typename TString<T>::LPCT pszRhs);
	friend bool operator< <>(typename TString<T>::LPCT pszLhs, const TString<T>& rhs);
	friend bool operator< <>(const TString<T>& lhs,T cRhs);
	friend bool operator< <>(T cLhs,const TString<T>& rhs);
	friend TString<T> operator+ <>(const TString<T>& lhs, const TString<T>& rhs);
	friend TString<T> operator+ <>(const TString<T>& lhs,typename TString<T>::LPCT pszRhs);
	friend TString<T> operator+ <>(typename TString<T>::LPCT pszLhs, const TString<T>& rhs);
	friend TString<T> operator+ <>(const TString<T>& lhs, T cRhs);
	friend TString<T> operator+ <>(T cLhs, const TString<T>& rhs);
	friend ostream& operator<< <>(ostream& os, const TString<T>& st);
	friend istream& operator>> <>(istream& is, const TString<T>& st);
};

template<class T>
bool operator==(const TString<T>& lhs, const TString<T>& rhs) {
	if (lhs.m_nLength != rhs.m_nLength)
		return false;
	return 0 == TString<T>::_strcmp(lhs.m_pBuffer, rhs.m_pBuffer);
}
template<class T>
bool operator==(const TString<T>& lhs, typename TString<T>::LPCT pszRhs) {
	if (lhs.m_nLength != TString<T>::_strlen(pszRhs))
		return false;
	return 0 == TString<T>::_strcmp(lhs.m_pBuffer, pszRhs);
}
template<class T>
bool operator==(typename TString<T>::LPCT pszLhs, const TString<T>& rhs) {
	if (TString<T>::_strlen(pszLhs) != rhs.m_nLength)
		return false;
	return 0 == TString<T>::_strcmp(pszLhs, rhs.m_pBuffer);
}
template<class T>
bool operator==(const TString<T>& lhs,T cRhs) {
	if (lhs.m_nLength != 1)
		return false;
	return lhs.m_pBuffer[0] == cRhs;
}
template<class T>
bool operator==(T cLhs,const TString<T>& rhs) {
	if (1 != rhs.m_nLength)
		return false;
	return cLhs == rhs.m_pBuffer[0];
}

template<class T>
bool operator!=(const TString<T>& lhs, const TString<T>& rhs) {
	if (lhs.m_nLength != rhs.m_nLength)
		return true;
	return 0 != TString<T>::_strcmp(lhs.m_pBuffer, rhs.m_pBuffer);
}
template<class T>
bool operator!=(const TString<T>& lhs, typename TString<T>::LPCT pszRhs) {
	if (lhs.m_nLength != TString<T>::_strlen(pszRhs))
		return true;
	return 0 != TString<T>::_strcmp(lhs.m_pBuffer, pszRhs);
}
template<class T>
bool operator!=(typename TString<T>::LPCT pszLhs, const TString<T>& rhs) {
	if (TString<T>::_strlen(pszLhs) != rhs.m_nLength)
		return true;
	return 0 != TString<T>::_strcmp(pszLhs, rhs.m_pBuffer);
}
template<class T>
bool operator!=(const TString<T>& lhs,T cRhs) {
	if (lhs.m_nLength != 1)
		return true;
	return lhs.m_pBuffer[0] != cRhs;
}
template<class T>
bool operator!=(T cLhs,const TString<T>& rhs) {
	if (1 != rhs.m_nLength)
		return true;
	return cLhs != rhs.m_pBuffer[0];
}

template<class T>
bool operator>(const TString<T>& lhs, const TString<T>& rhs) { return 0 < TString<T>::_strcmp(lhs.m_pBuffer, rhs.m_pBuffer); }
template<class T>
bool operator>(const TString<T>& lhs,typename TString<T>::LPCT pszRhs) { return 0 < TString<T>::_strcmp(lhs.m_pBuffer, pszRhs); }
template<class T>
bool operator>(typename TString<T>::LPCT pszLhs, const TString<T>& rhs) { return 0 < TString<T>::_strcmp(pszLhs, rhs.m_pBuffer); }
template<class T>
bool operator>(const TString<T>& lhs, T cRhs) { if (!lhs.m_pBuffer) return true; return lhs.m_pBuffer[0] < cRhs; }
template<class T>
bool operator>(T CLhs, const TString<T>& rhs) { if (!rhs.m_pBuffer) return false; return cLhs < rhs.m_pBuffer[0]; }

template<class T>
bool operator<(const TString<T>& lhs, const TString<T>& rhs) { return 0 > TString<T>::_strcmp(lhs.m_pBuffer, rhs.m_pBuffer); }
template<class T>
bool operator<(const TString<T>& lhs,typename TString<T>::LPCT pszRhs) { return 0 > TString<T>::_strcmp(lhs.m_pBuffer, pszRhs); }
template<class T>
bool operator<(typename TString<T>::LPCT pszLhs, const TString<T>& rhs) { return 0 > TString<T>::_strcmp(pszLhs, rhs.m_pBuffer); }
template<class T>
bool operator<(const TString<T>& lhs, T cRhs) { if (!lhs.m_pBuffer) return false; return lhs.m_pBuffer[0] > cRhs; }
template<class T>
bool operator<(T CLhs, const TString<T>& rhs) { if (!rhs.m_pBuffer) return true; return cLhs > rhs.m_pBuffer[0]; }

template<class T>
TString<T> operator+(const TString<T>& lhs, const TString<T>& rhs) {
	TString<T> res; // no buffer allocated
	res._reallocNreplace(lhs.m_nLength + rhs.m_nLength); // its done here
	res._reallocNreplace(lhs.m_nLength, lhs.m_pBuffer);
	res._reallocNtail(rhs.m_nLength, rhs.m_pBuffer);
	return res;
}
template<class T>
TString<T> operator+(const TString<T>& lhs,typename TString<T>::LPCT pszRhs) {
	TString<T> res;
	int nLRhs = TString<T>::_strlen(pszRhs);
	res._reallocNreplace(lhs.m_nLength + nLRhs);
	res._reallocNreplace(lhs.m_nLength, lhs.m_pBuffer);
	res._reallocNtail(nLRhs,pszRhs);
	return res;
}
template<class T>
TString<T> operator+(typename TString<T>::LPCT pszLhs, const TString<T>& rhs) {
	TString<T> res;
	int nLLhs = TString<T>::_strlen(pszLhs);
	res._reallocNreplace(nLLhs + rhs.m_nLength);
	res._reallocNreplace(nLLhs,pszLhs);
	res._reallocNtail(rhs.m_nLength, rhs.m_pBuffer);
	return res;
}
template<class T>
TString<T> operator+(const TString<T>& lhs, T cRhs) {
	TString<T> res;
	res._reallocNreplace(lhs.m_nLength + 1);
	res._reallocNreplace(lhs.m_nLength, lhs.m_pBuffer);
	if (cRhs != TString<T>::_null){
		T str[2] = { cRhs, TString<T>::_null };
		res._reallocNtail(1,str);
	}
	return res;
}
template<class T>
TString<T> operator+(T cLhs, const TString<T>& rhs) {
	TString<T> res;
	res._reallocNreplace(1 + rhs.m_nLength);
	if (cLhs != TString<T>::_null){
		T str[2] = { cLhs, TString<T>::_null };
		res._reallocNreplace(1,str);
	}
	res._reallocNtail(rhs.m_nLength, rhs.m_pBuffer);
	return res;
}

template<class T>
ostream& operator<<(ostream& os, const TString<T>& st) {
	return os << st.m_pBuffer;
}

template<class T>
istream& operator>>(istream& is, const TString<T>& st){
	ASSERT(0);// not implemented properly
	return is >> st.m_pBuffer;
}

typedef TString<char> CString;
#endif /*__TSTRING_H__INCLUDED__*/

