/***************************************************************************
                          gnuwebcache.cpp  -  description
                             -------------------
    begin                : Fri Sep 27 2002
    copyright            : (C) 2002 by Max Zaitsev
    email                : maksik@gmx.co.uk
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "mutella.h"
#include "structures.h"

#include "asyncsocket.h"
#include "packet.h"

#include "event.h"
#include "messages.h"
#include "gnudirector.h"
#include "gnuwebcache.h"
#include "gnucache.h"

#include "conversions.h"
#include "common.h"

#define RECV_BUFF_SIZE 	0x4000
#define TIMEOUT			60

MGnuWebCache::MGnuWebCache(MGnuDirector* pDirector) :
	m_pDirector(pDirector),
	m_nLastActive(xtime()),
	m_bValid(false),
	m_nBytesReceived(0),
	m_pBuffer(NULL)
{
}

MGnuWebCache::~MGnuWebCache()
{
	delete [] m_pBuffer;
}

bool MGnuWebCache::Connect(const CString& sURL, RequestType request)
{
	m_request = request;
	// parse the URL: remove http://, trailing / and so on
	int nPos;
	m_sURL = sURL;
	m_sHost = sURL;
	m_sHost.make_lower();
	if ((nPos=m_sHost.find("http://"))!=-1)
		m_sHost = m_sHost.substr(nPos+7);
	m_sHost = StripAnyOf(m_sHost, " /");
	// extract host, port number and path
	m_sPath = "/";
	if ((nPos=m_sHost.find("/"))!=-1)
	{
		m_sPath = m_sHost.substr(nPos);
		m_sHost.cut(nPos);
	}
	int nPort = 80;
	if ((nPos=m_sHost.find(":"))!=-1)
	{
		nPort = atol(m_sHost.substr(nPos+1).c_str());
		m_sHost.cut(nPos);
	}
	// connect
	if(!Create(0, SOCK_STREAM, FD_READ|FD_WRITE|FD_CONNECT|FD_CLOSE))
	{
		m_bValid = false;
		return false;
	}
	//cout << "HOST: '" << m_sHost << "'  PORT: " << nPort << endl;
	m_bValid = (MAsyncSocket::Connect(m_sHost.c_str(), nPort) || GetLastError() == EINPROGRESS);
	m_nLastActive = xtime();
	return m_bValid;
}

void MGnuWebCache::OnTimer(int nTimeStamp)
{
	if (nTimeStamp > m_nLastActive + TIMEOUT)
	{
		m_bValid = false;
		//cout << "TIMEOUT\n";
		Close();
	}
}

void MGnuWebCache::OnReceive(int nErrorCode)
{
	MAsyncSocket::OnReceive(nErrorCode);
	// receive and tail the buffer
	if (!m_pBuffer)
		m_pBuffer = new char[RECV_BUFF_SIZE];
	
	int nReceived = Receive(m_pBuffer+m_nBytesReceived, RECV_BUFF_SIZE - m_nBytesReceived - 1); // -1 is for \0

	switch (nReceived)
	{
		case 0:
		case SOCKET_ERROR:
			m_bValid = false;
			Close();
			return;
	}
	//
	m_nLastActive = xtime();
	// add a trailing '\0'
	ASSERT(nReceived>0);
	m_nBytesReceived += nReceived;
	ASSERT(m_nBytesReceived<RECV_BUFF_SIZE);
	m_pBuffer[m_nBytesReceived]='\0';
	// search for \r\n\r\n
	char* pHeaderEnd = strstr(m_pBuffer, "\r\n\r\n");
	if (!pHeaderEnd)
		return;
	// if it's there -- parse the rest and behave accordingly
	// cout << "RECEIVED RESPONCE:\n" << m_pBuffer << ":END\n";

	MakeLower(m_pBuffer);
	// detect HTTP return codes and particularly redirect replies
	if (strncmp(m_pBuffer, "http/1.", 7) == 0)
	{
		int nCode = atol(m_pBuffer+9);
		if (nCode >= 400)
		{
			m_bValid = false;
			Close();
			return;
		}
		if (nCode >= 300)
		{
			// simply close the current connection and initiate a new one
			m_bValid = false;
			Close();
			if ( m_request != HostFile && m_request != UrlFile )
				return;
			//
			// cout << "redirect reply received!\n";
			// get the redirect path
			//Location: http://www.middlebrookvillage.com/error_docs/not_found.html
			char* pLocation = strstr(m_pBuffer, "location:");
			char* pLocEnd = NULL;
			if (pLocation && (pLocEnd=strstr(pLocation, "\r\n")))
			{
				*pLocEnd = '\0';
				pLocation = StripWhite(pLocation+9);
				// first question: relative or absolute path
				if (strncmp(pLocation, "http://",7)==0)
				{
					// cout << "opening " << pLocation << endl;
					m_pDirector->OpenWebCacheConnection(pLocation, Redirect|m_request);
				}
				else
				{
					// this will fail for the absolute path on the same server!
					m_sURL.cut(m_sURL.rfind('/')+1);
					m_sURL+=pLocation;
					// cout << "opening " << m_sURL << endl;
					m_pDirector->OpenWebCacheConnection(m_sURL, Redirect|m_request);
				}
				// the url seems to be OK for future use
				// WARNING: if we are redirected to 404 or whatever reply we are in trouble!
				m_pDirector->AddWebCacheUrl(m_sURL);
			}
			return;
		}
	}
	bool bOK = false;
	MGnuCache* pHostCatcher = m_pDirector->GetHostCatcher();
	char* pLin = pHeaderEnd+4 ;
	char* pLinEnd;
	do {
		pLinEnd = strstr(pLin, "\r\n");
		if (pLinEnd)
		{
			*pLinEnd = '\0';
			pLinEnd += 2;
		}
		else
		{
			pLinEnd = strchr(pLin, '\n');
			if (pLinEnd)
			{
				*pLinEnd = '\0';
				pLinEnd += 1;
			}
		}
		// look at the line
		switch (m_request & 0xf)
		{
			case HostFile : {
					// we expect hosts list here or string "ERROR"
					if (strstr(pLin, "error"))
					{
						pLinEnd = NULL; // quit the loop
						break;
					}
					Node newNode;
					newNode.Ping	   = 0;
					newNode.Speed      = 0;
					newNode.ShareSize  = 0;
					newNode.ShareCount = 0;
					newNode.Distance   = 0;
					newNode.Port = 6346;
					char* pTmp = strchr(pLin, ':');
					if (pTmp)
					{
						newNode.Port = atol(pTmp+1);
						*pTmp = '\0';
					}
					if (Str2Ip(pLin, newNode.Host))
					{
						pHostCatcher->UpdateCache(newNode);
						bOK = true;
					}
				}
				break;
			case UrlFile  : {
				// we expect url list here or string "ERROR"
				if (strstr(pLin, "error"))
				{
					pLinEnd = NULL; // quit the loop
					break;
				}
				//
				if (strncmp(pLin, "http://", 7) == 0)
				{
					m_pDirector->AddWebCacheUrl(pLin);
					bOK = true;
				}
				break;
			}
			
			case Update   :
				if (strstr(pLin, "error"))
				{
					pLinEnd = NULL; // quit the loop
					bOK = false;
					break;
				}
				if (strncmp(pLin, "ok", 2))
					bOK = true;
				break;

			case Ping     : if (strstr(pLin, "pong")) bOK = true; break;
		}
		// go to the next
		pLin = pLinEnd;
	} while (pLin && *pLin);
	if (bOK && (m_request & Redirect) == 0) // the URL has replied and we were able to understand that!
		m_pDirector->AddWebCacheUrl(m_sURL);
	//
	m_bValid = false;
}

void MGnuWebCache::OnSend(int nErrorCode)
{
	MAsyncSocket::OnSend(nErrorCode);
	m_nLastActive = xtime();
	// make a request
	CString sRequest = "GET " + m_sPath;
	switch (m_request)
	{
		case HostFile : sRequest += "?hostfile=1"; break;
		case UrlFile  : sRequest += "?urlfile=1"; break;
		case Update   : sRequest += "?ip=" +
									Ip2Str(m_pDirector->GetPublicIP()) + ":" +
									DWrdtoStr(m_pDirector->GetPublicPort());
			break;
		case Ping     : sRequest += "?ping=1"; break;
		case HostRedirect:
		case UrlRedirect:
			break;
	}
	sRequest += " HTTP/1.0\r\nUser-Agent: Mutella\r\nHost: " + m_sHost + "\r\nAccept: text/html, */*\r\n\r\n";
	// send a request
	Send(sRequest.c_str(), sRequest.length());
	// modify select flags
	ModifySelectFlags(0, FD_WRITE);
}

void MGnuWebCache::OnConnect(int nErrorCode)
{
	MAsyncSocket::OnConnect(nErrorCode);
	// why we need that at all?
	m_nLastActive = xtime();
}

void MGnuWebCache::OnClose(int nErrorCode)
{
	MAsyncSocket::OnClose(nErrorCode);
	// check the reason and mark URL as BAD?
	m_bValid = false; 
}

