/***************************************************************************
 Mutella - A commandline/HTTP client for the Gnutella filesharing network.

 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation; either version 2
 of the License, or (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 gnushare.cpp  -  Definition of representation of a shared files folder

 the original version of this file was taken from Gnucleus (http://gnucleus.sourceforge.net)

    begin                : Tue May 29 2001
    copyright            : (C) 2001 by
    email                : maksik@gmx.co.uk
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "mutella.h"
#include "structures.h"

#include "asyncsocket.h"
#include "packet.h"

#include "event.h"
#include "messages.h"
#include "gnudirector.h"
#include "gnushare.h"
#include "gnudownload.h"
#include "dir.h"
#include "asyncfile.h"
#include "common.h"
#include "conversions.h"
#include "property.h"
#include "preferences.h"
#include "gnuwordhash.h"
#include "sha1thread.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <dirent.h>
#include <ctype.h>

#define SHARE_QUEUE_MAX 100

class MEventQueueShare : public MAsyncEventReceiver {
public:
	MEventQueueShare() : MAsyncEventReceiver(INT_MAX) {}
protected:
	virtual bool IsOfInterest(MEvent* p){
		ASSERT(p);
		switch (p->GetID())
		{
			case MSG_SHARE_COMMAND: return true; // commands should never be lost
			case MSG_DOWNLOAD_FILE_MOVED: return true; // it is like a command -- should never be lost
			case MSG_UPLOAD_FILE_OPEN_FAILED: return true; // it is like a command as well
			case MSG_SHA1_PROGRESS_UPDATE: return true; // we are in progress of hashing a large file
			case MSG_SHA1_READY: return true; // we have calculated a hash
			case MSG_SHARE_REQUEST: if(QueuedEvents() < SHARE_QUEUE_MAX) return true;
		}
		return false;
	}
};

enum ShareCommand
{
	SHCMD_RELOAD,
	SHCMD_UPDATE,
	SHCMD_LOAD,
	SHCMD_STOP
};

class MShareCmd : public TSimpleEvent<ShareCommand>{
public:
	MShareCmd(ShareCommand cmd) : TSimpleEvent<ShareCommand>(ET_MESSAGE, ES_NONE, cmd, MSG_SHARE_COMMAND) {}
	virtual CString Format(){
		switch (m_value){
			case SHCMD_RELOAD: return "SHARE-RELOAD";
			case SHCMD_UPDATE: return "SHARE-UPDATE";
			case SHCMD_LOAD: return "SHARE-LOAD";
			case SHCMD_STOP: return "SHARE-STOP";
		}
		return "WRONG";
	}
};

class MShareThread : public MThread {
public:
	MShareThread(MGnuShare* pS){
		m_pShare    = pS;
		ASSERT(m_pShare->m_pDirector);
		m_pWordHash = m_pShare->m_pDirector->GetWordTable();
		ASSERT(m_pWordHash);

		m_dwTotalFiles = 0;
		m_dwTotalSize  = 0;
		ED().AddReceiver(&m_EventQueue);
	}
	~MShareThread()
	{
		ASSERT(finished());
		ED().RemoveReceiver(&m_EventQueue); // this does not hurt
	}
	// interlocked syncronous operations
	CString GetPath(DWORD);
	void AddSharedDir(CString path, bool bDeep);
	void ResetSharedDirs();
	//
	bool GetFile(DWORD index, CString &file, CString &filePath, bool bCheckName);
	int  GetIndexBySha1(const CString& sSha1);
	void GetShareCopy(vector<SharedFile>&);
	void AddFile(CString sFileName, const CString& sFilePath, DWORD dwFileSize, bool bLock = true);
	// delayed operations
	void StopThread(){
		ED().SendEvent( new MShareCmd(SHCMD_STOP) ); // TODO: check if it really works
	}
protected:	
	void LoadFiles();
	void RecurseLoad(CString, CString, bool, DWORD &, long long &);
private:
	MMutex m_filesmutex;
	vector<SharedFile> m_SharedFiles;   // Each shared file
	vector<SharedDirectory> m_SharedDirectories;
	DWORD m_dwTotalFiles;
	DWORD m_dwTotalSize;

	MEventQueueShare m_EventQueue;

	MGnuShare*    m_pShare;
	MGnuWordHash* m_pWordHash;
	
	MShareThread();
	void run();
};

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

MGnuShare::MGnuShare(MGnuDirector* pCntr)
{	
	m_pDirector = pCntr;

	m_dwTotalFiles = 0;
	m_dwTotalSize  = 0;
	m_dwUltrapeerSizeMarker = 8;
	m_pDirector->AttachShare(this);
	m_pSearchSharedThread = NULL;
}

MGnuShare::~MGnuShare()
{
	m_pDirector->DetachShare(this);
	if (m_pSearchSharedThread->finished())
		delete m_pSearchSharedThread;
	else
	{
		TRACE("WARNING: failed to stop search thread");
	}
}

void MGnuShare::UpdateSizeAndCount(DWORD dwSize, DWORD dwCount)
{
	m_dwTotalFiles = dwCount;
	m_dwTotalSize  = dwSize;
	// Put ultrapeer marker on shared file size
	m_dwUltrapeerSizeMarker = 8;
	while(m_dwUltrapeerSizeMarker < m_dwTotalSize && m_dwUltrapeerSizeMarker)
		m_dwUltrapeerSizeMarker <<= 1;
}

void MGnuShare::InitShare()
{
	m_pSearchSharedThread = new MShareThread(this);
	ASSERT(m_pSearchSharedThread);
	m_pSearchSharedThread->start();
	bool bPost = false;
	if (strlen(m_pDirector->GetPrefs()->m_szSharePath))
	{
		m_pSearchSharedThread->AddSharedDir(m_pDirector->GetPrefs()->m_szSharePath,true);
		bPost = true;
	}
	if (m_pDirector->GetPrefs()->m_bShareDownloadDir && strlen(m_pDirector->GetPrefs()->m_szDownloadPath))
	{
		m_pSearchSharedThread->AddSharedDir(m_pDirector->GetPrefs()->m_szDownloadPath,false);
		bPost = true;
	}
	if (bPost)
		ED().PostEvent( new MShareCmd(SHCMD_LOAD) );
}

void MGnuShare::CloseShare()
{
	TRACE("Stoping search thread...");
	m_pSearchSharedThread->StopThread();
}

void MGnuShare::Rescan()
{
	m_pSearchSharedThread->ResetSharedDirs();
	m_pSearchSharedThread->AddSharedDir(m_pDirector->GetPrefs()->m_szSharePath,true);
	if (m_pDirector->GetPrefs()->m_bShareDownloadDir)
		m_pSearchSharedThread->AddSharedDir(m_pDirector->GetPrefs()->m_szDownloadPath,false);
	ED().PostEvent( new MShareCmd(SHCMD_LOAD) );
}

void MGnuShare::ResetDirectories(DWORD &EventCount, LPHANDLE EventList)
{
	// Close wait events on all directories
	assert(0);
	/*for(int i = 1; i < EventCount; i++)
		FindCloseChangeNotification( EventList[i]);

	EventCount = 1;

	// Reset partial watch
	CString PartialPath = m_pDoc->m_pPrefs->m_PartialsInDir ? m_pDoc->m_pPrefs->m_DownloadPath + "\\Partials" : ".\\Partials";
	EventList[1] = FindFirstChangeNotification(PartialPath, false, FILE_NOTIFY_CHANGE_FILE_NAME | FILE_NOTIFY_CHANGE_DIR_NAME);

	if(EventList[1] != INVALID_HANDLE_VALUE)
		EventCount++;

	// Reset wait events on all shared directories
	for(i = 0; i < SharedDirectories.size(); i++)
	{
		if(EventCount >= MAX_EVENTS)
			break;

		EventList[EventCount] = FindFirstChangeNotification(SharedDirectories[i].Name, SharedDirectories[i].Recursive, FILE_NOTIFY_CHANGE_FILE_NAME | FILE_NOTIFY_CHANGE_DIR_NAME);

		if(EventList[EventCount] != INVALID_HANDLE_VALUE)
			EventCount++;
	}*/
}

void MShareThread::AddSharedDir(CString path, bool bDeep)
{
	MLock lock(m_filesmutex);
	SharedDirectory sd;	
	sd.Name = ExpandPath(path);
	sd.Recursive = bDeep;
	sd.Size = 0;
	sd.FileCount = 0;
	
	m_SharedDirectories.push_back(sd);
}

void MShareThread::ResetSharedDirs()
{
	MLock lock(m_filesmutex);
	
	m_SharedDirectories.clear();
}

void MShareThread::LoadFiles()
{
	m_pWordHash->ClearLocalTable();
	
	MLock lock(m_filesmutex);
	
	m_SharedFiles.clear();
	m_dwTotalFiles = 0;
	m_dwTotalSize  = 0;

	for(int i = 0; i < m_SharedDirectories.size(); i++)
	{
		m_SharedDirectories[i].FileCount = 0;
		m_SharedDirectories[i].Size = 0;
		RecurseLoad(m_SharedDirectories[i].Name, "", m_SharedDirectories[i].Recursive, m_SharedDirectories[i].FileCount, m_SharedDirectories[i].Size);
		m_dwTotalFiles += m_SharedDirectories[i].FileCount;
		m_dwTotalSize  += m_SharedDirectories[i].Size/1024;
	}
	
	m_pShare->UpdateSizeAndCount(m_dwTotalSize, m_dwTotalFiles);
	//
	CString stmp;
	stmp.format("directories:%d  files:%d  size:%d Mbytes", m_SharedDirectories.size(), m_dwTotalFiles, m_dwTotalSize/1024);
	POST_EVENT( MStringEvent(
				ET_MESSAGE,
				ES_GOODTOKNOW,
				stmp,
				MSG_SHARE_SCANNED,
				MSGSRC_SHARE
			));

}

void MShareThread::RecurseLoad(CString FullPath, CString DirPath, bool doRecurse, DWORD &DirCount, long long &DirSize)
{
	DirEntryVec files;
	if (!ScanDir(files, FullPath, m_pShare->m_pDirector->GetPrefs()->m_szShareFilter, DirEntry::regular|DirEntry::dir))
	{
		// TODO: messageID
		POST_ERROR(ES_GOODTOKNOW, CString("RecurseLoad: error scanning the dir ") + FullPath);
		return;
	}
	if (!files.size())
	{
		// TODO: messageID
		POST_ERROR(ES_UNIMPORTANT, CString("RecurseLoad: no files at ") + FullPath);
		return;
	}
	for (int i = 0; i<files.size(); ++i)
	{
		if (!m_pShare->m_pDirector->GetPrefs()->m_bShareDotFiles && files[i].Name[0]=='.')
			continue;
		if (files[i].Type & DirEntry::dir)
		{
			if(doRecurse)
				RecurseLoad(files[i].Path, DirPath + "/" + files[i].Name, true, DirCount, DirSize);
		}
		else if (files[i].Type & DirEntry::regular)
		{
			DWORD   FileSize = files[i].Size;
			CString FileName = files[i].Name;
			CString FilePath  = files[i].Path;

			if(m_pShare->m_pDirector->GetPrefs()->m_bReplyFilePath && DirPath.length())
				FileName = DirPath + "/" + FileName;

			// add the file
			AddFile(FileName, FilePath, FileSize, false);

			DirCount++;
			DirSize += FileSize;
		}
	}
}

void MShareThread::AddFile(CString sFileName, const CString& sFilePath, DWORD dwFileSize, bool bLock /*=true*/ )
{
	MLock lock(m_filesmutex, bLock);
	// some cleanups
	if(sFileName[0] == '/')
		sFileName = sFileName.substr(1);

	// Make a file item
	SharedFile addFile;
	addFile.Path    = sFilePath;
	addFile.Name	= sFileName;
	MakeLower(sFileName);
	addFile.NameLower   = sFileName;
	addFile.Size		= dwFileSize;
	addFile.Matches		= 0;
	addFile.Uploads		= 0;

	m_SharedFiles.push_back(addFile);

	// queue for hashing
	POST_EVENT( MHashRequest(
		addFile.Name,
		addFile.Path,
		m_SharedFiles.size() - 1
	));

	// add to the word lookup table
	m_pWordHash->InsertString(addFile.Name, m_SharedFiles.size() - 1);
}

void MGnuShare::GetShareCopy(vector<SharedFile>& ShareCopy)
{
	m_pSearchSharedThread->GetShareCopy(ShareCopy);
}

void MShareThread::GetShareCopy(vector<SharedFile>& ShareCopy)
{
	MLock FilesAccess(m_filesmutex);
	ShareCopy = m_SharedFiles; //TODO: check if it really copies the list, not just references it
}

DWORD MGnuShare::GetIndexBySha1(const CString& sSha1)
{
	return m_pSearchSharedThread->GetIndexBySha1(sSha1);
}

int MShareThread::GetIndexBySha1(const CString& sSha1)
{
	// Look up sha1 in shared files
	list<UINT> Indexes;
	m_pWordHash->LookupLocalSha1(sSha1, Indexes);
	if (Indexes.empty())
		return -1;
	// find first not hidden file
	MLock FilesAccess(m_filesmutex);
	for ( ; Indexes.size(); Indexes.pop_front())
	{
		ASSERT(m_SharedFiles[Indexes.front()].Sha1Hash == sSha1);
		if (!m_SharedFiles[Indexes.front()].Hidden)
			return Indexes.front();
	}
	return -1;
}

bool MGnuShare::GetFileByIndex(DWORD index, CString &file, CString &filePath, bool bCheckName)
{
	return m_pSearchSharedThread->GetFile(index, file, filePath, bCheckName);
}

bool MShareThread::GetFile(DWORD index, CString &fileName, CString &filePath, bool bCheckName)
{
	CString file = fileName;
	MakeLower(file);
	ReplaceSubStr(file,"\\","/");// for Windows dudes
	ReplaceSubStr(file,"//","/");// for any case
	MLock FilesAccess(m_filesmutex);
	if (index < 0 || index >= m_SharedFiles.size())
		return false;
	SharedFile* pFile = &m_SharedFiles[index];
	if (pFile->Hidden)
		return false;
	if (bCheckName && file != pFile->NameLower)
	{
		// lets give them one more chance -- some clients ignore the directory path
		CString s = pFile->NameLower;
		s = s.substr(s.rfind("/")+1);
		if (file != s)
			return false;
	}
	pFile->Uploads++; // TODO: this is not correct for pushes
	filePath = pFile->Path;
	if (!bCheckName)
		fileName = pFile->Name;
	return true;
}

void MShareThread::run()
{
	ASSERT(m_pShare);
	MGnuDirector*  pComm  = m_pShare->m_pDirector;;
	ASSERT(pComm);
	
	BYTE* QueryReply = new BYTE[65000]; // 1000 bytes extra for overrun
	ASSERT(QueryReply);
	BYTE* QueryReplyNext;
	int QueryReplyLength, TotalReplyCount, ReplyCount, MaxReplies;
	CString MetaTail;
	
	vector<char*> QWords;
	vector<char*>::iterator itWord;
	int nQLen;
	char c;
	char* szWord;
	int i;

	bool bIndexingQuery;

	list<UINT>  MatchingIndexes;
    list<DWORD>   MatchingNodes;

    list<UINT>::iterator itIndex;
    list<DWORD>::iterator  itNodeID;

    int FilesScanned;

	vector<SharedFile>::iterator itFile;
	
	struct timespec rqtp, rmtp;
	
	rqtp.tv_sec  = 1;
	rqtp.tv_nsec = rmtp.tv_sec = rmtp.tv_nsec = 0;
	safe_nanosleep(&rqtp, &rmtp);
	
	for(;;)
	{
		m_EventQueue.Poll();
		// now we do have an event!
		TSmartPtr<MEvent> spEvent = m_EventQueue.SafeFront();
		m_EventQueue.Pop();
		switch (spEvent->GetID())
		{
			case MSG_SHARE_COMMAND:{
				switch(((MShareCmd*) (MEvent*) spEvent)->GetValue())
				{
					case SHCMD_LOAD:
						POST_EVENT( MStringEvent(
							ET_MESSAGE,
							ES_NONE,
							"Reset hasing queue",
							MSG_SHA1_RESET_QUEUE,
							MSGSRC_SHARE
						));
						//
						LoadFiles();
						pComm->OnShareChanged();
						//
						POST_EVENT( MStringEvent(
							ET_MESSAGE,
							ES_NONE,
							"Scanning is over",
							MSG_SHA1_LAST_IN_THE_ROW,
							MSGSRC_SHARE
						));
						break;
					case SHCMD_STOP:
						delete [] QueryReply;
						ED().RemoveReceiver(&m_EventQueue);
						return;
				}
			}
			break;
			case MSG_DOWNLOAD_FILE_MOVED:
			if (pComm->GetPrefs()->m_bShareDownloadDir)
			{
				MEvFileMoved* pEFM = (MEvFileMoved*) (MEvent*) spEvent;
				CString sPath = pEFM->GetValue();
				int nSlashPos = sPath.rfind('/');
				if (nSlashPos>=0)
				{
					AddFile(sPath.substr(nSlashPos+1), sPath, pEFM->m_nFileSize);

					m_dwTotalSize += pEFM->m_nFileSize;
					++m_dwTotalFiles;
					m_pShare->UpdateSizeAndCount(m_dwTotalSize, m_dwTotalFiles);
					pComm->OnShareChanged();
				}
			}
			break;
			case MSG_UPLOAD_FILE_OPEN_FAILED:{
				int index = ((MIntEvent*) (MEvent*) spEvent)->GetValue();
				// unshare the file
				m_filesmutex.lock();
				if (index >= 0 && index < m_SharedFiles.size())
					m_SharedFiles[index].Hidden = true;
					#warning we shall remove the file from the hash here!
				m_filesmutex.unlock();
			}
			break;
			case MSG_SHA1_PROGRESS_UPDATE:{
				MHashProgress* pEHP = (MHashProgress*) (MEvent*) spEvent;
				m_filesmutex.lock();
				if (pEHP->m_nShareIndex >= 0                                 &&
					pEHP->m_nShareIndex < m_SharedFiles.size()               &&
					pEHP->m_sPath == m_SharedFiles[pEHP->m_nShareIndex].Path )
				{
					m_SharedFiles[pEHP->m_nShareIndex].Sha1Hash = "... " + DWrdtoStr(pEHP->m_nPercentDone) + "% done ...";
				}
				m_filesmutex.unlock();
			}
			break;
			case MSG_SHA1_READY:{
				MHashReady* pEHR = (MHashReady*) (MEvent*) spEvent;
				m_filesmutex.lock();
				if (pEHR->m_nShareIndex >= 0                                 &&
					pEHR->m_nShareIndex < m_SharedFiles.size()               &&
					pEHR->m_sPath == m_SharedFiles[pEHR->m_nShareIndex].Path )
				{
					m_SharedFiles[pEHR->m_nShareIndex].Sha1Hash = pEHR->m_sSha1;
					m_pWordHash->InsertString("sha1:" + pEHR->m_sSha1, pEHR->m_nShareIndex, false);
				}
				m_filesmutex.unlock();
			}
			break;
			case MSG_SHARE_REQUEST:{
				QueryComp& SearchQuery = ((MShareReq*) (MEvent*) spEvent)->m_query;
				
				//TRACE4("MShareThread: query= ", SearchQuery.Text, " Ext= ", SearchQuery.ExtendedPart);

				MatchingIndexes.clear();
                MatchingNodes.clear();

                // A four space query is an index query
                bIndexingQuery = ( strcmp(SearchQuery.Text, "    ") == 0 && SearchQuery.nHops == 1);

                if (!bIndexingQuery)
                {
					// Look up query in shared files and remote indexes
					m_pWordHash->LookupQuery(SearchQuery, MatchingIndexes, MatchingNodes);
				}

                // Search Local Files
                FilesScanned = 0;
                QueryReplyNext          = QueryReply + 34;
                QueryReplyLength        = 0;
                TotalReplyCount         = 0;
                ReplyCount              = 0;
                MaxReplies              = pComm->GetPrefs()->m_nMaxReplies;
                MetaTail                = "";

                m_filesmutex.lock();
                // this loop is either loop for all the files, if bIndexingQuery==true or
                // just a loop over the matches... going to be a bit ugly construct
                int nPos;
                int nSharedFiles = m_SharedFiles.size();
                nPos = 0;
                itIndex = MatchingIndexes.begin();
                for( ;  bIndexingQuery ? nPos < nSharedFiles : itIndex != MatchingIndexes.end();
                		++nPos, ++itIndex /* i hope it's safe to increment invalid iterator */ )
                {
                    if (!bIndexingQuery)
                    	nPos = *itIndex;

                    if (nPos >= nSharedFiles)
                    	break;

                    // limit number of replies
                    if(!bIndexingQuery && MaxReplies && MaxReplies <= TotalReplyCount)
                        break;

					if (m_SharedFiles[nPos].Hidden) // it's a quick workaround
						continue;

                    FilesScanned++;

                    if (!bIndexingQuery)
                    	m_SharedFiles[nPos].Matches++;

                    packet_QueryHitItem* pQHI = (packet_QueryHitItem*) QueryReplyNext;
					pQHI->le_Index = nPos;
					pQHI->le_Size = m_SharedFiles[nPos].Size;

					QueryReplyNext   += 8;
					QueryReplyLength += 8;

                    // File Name
                    strcpy ((char*) QueryReplyNext, m_SharedFiles[nPos].Name.c_str());
                    QueryReplyNext   += m_SharedFiles[nPos].Name.size() + 1;
                    QueryReplyLength += m_SharedFiles[nPos].Name.size() + 1;

                    // File Hash
                    if( m_SharedFiles[nPos].Sha1Hash.length()==32 ) // I also abuse Sha1Hash member to show the hashing progress
                    {
                        strcpy ((char*) QueryReplyNext, "urn:sha1:");
                        QueryReplyNext   += 9;
                        QueryReplyLength += 9;

                        strcpy ((char*) QueryReplyNext, m_SharedFiles[nPos].Sha1Hash.c_str());
                        QueryReplyNext   += m_SharedFiles[nPos].Sha1Hash.size() + 1;
                        QueryReplyLength += m_SharedFiles[nPos].Sha1Hash.size() + 1;
                    }
                    else
                    {
                        *QueryReplyNext = '\0';

                        QueryReplyNext++;
                        QueryReplyLength++;
                    }

                    // File Meta
                    // AddMeta(m_SharedFiles[pos], MetaTail, ReplyCount);

                    ReplyCount++;
                    TotalReplyCount++;

                    if(QueryReplyLength > 2048 || ReplyCount == 255)
                    {
                        m_filesmutex.unlock();
                        //SendResults(SearchQuery, QueryReply, QueryReplyLength, ReplyCount, MetaTail);
                        pComm->Post_QueryHit(&SearchQuery, QueryReply, QueryReplyLength, ReplyCount);
                        m_filesmutex.lock();

                        QueryReplyNext   = QueryReply + 34;
                        QueryReplyLength = 0;
                        ReplyCount               = 0;
                        MetaTail                 = "";
                    }
                }
                m_filesmutex.unlock();

                if(ReplyCount > 0)
                {
                    //SendResults(SearchQuery, QueryReply, QueryReplyLength, ReplyCount, MetaTail);
                    pComm->Post_QueryHit(&SearchQuery, QueryReply, QueryReplyLength, ReplyCount);
                }

                // Forward query to child nodes that match the query
                // When UDP queries results are in, forward ttl 0, hops 7 to children
                if(SearchQuery.nHops < MAX_TTL_SELF)
                    pComm->Send_ForwardQuery(SearchQuery, MatchingNodes);

#if 0
				// Clear word list
				QWords.clear();
				nQLen = strlen(pSearchQuery->Text);
				ASSERT(nQLen <= MAX_QUERY_LEN);
				// Break Query into individual words
				MakeWordList(pSearchQuery->Text, QWords);
				// Filter out searches for file types
				// or other crap which causes flood
				if( QWords.size() >= 1 &&
				    (QWords.size() > 1 || strlen(QWords[0]) > 3) )
				{
					// Search files
					BYTE* QueryReplyNext	= QueryReply;
					DWORD QueryReplyLength	= 0;
					BYTE  ReplyCount		= 0;
					bool  QueryMatch		= false;
					int i					= 0;
					MLock lock(m_filesmutex);

					for(itFile = m_SharedFiles.begin(); itFile != m_SharedFiles.end(); itFile++)
					{
						if ((*itFile).Hidden)
						{
							++i;
							continue;
						}
						// if matched -- add to search reply
						if(MatchWordList((*itFile).NameLower, QWords))
						{
							if(pComm->GetPrefs()->m_nMaxReplies >= 0)
								if(pComm->GetPrefs()->m_nMaxReplies <= ReplyCount)	
									break;
							if(QueryReplyLength > 64000)
								break;
							
							ReplyCount++;
							(*itFile).Matches++;
							
							packet_QueryHitItem* pQHI = (packet_QueryHitItem*) QueryReplyNext;
							pQHI->le_Index = i;
							pQHI->le_Size = (*itFile).Size;
							
							QueryReplyNext   += 8;
							QueryReplyLength += 8;
							
							strcpy ((char*) QueryReplyNext, (*itFile).Name.c_str());
							QueryReplyNext   += (*itFile).Name.size() + 1;
							QueryReplyLength += (*itFile).Name.size() + 1;
							*QueryReplyNext = '\0';
							QueryReplyNext++;
							QueryReplyLength++;
						}
						++i;
					}
					
					// Send reply
					if(ReplyCount > 0)
					{
						pComm->Post_QueryHit(pSearchQuery, QueryReply, QueryReplyLength, ReplyCount);
						// too many quiery replies cause remote nodes to disconnect
						// sleep for 100ms
						rqtp.tv_nsec = 100*1000000L;
						rqtp.tv_sec = rmtp.tv_sec = rmtp.tv_nsec = 0;
						safe_nanosleep(&rqtp, &rmtp);
					}
				}
#endif
				// kinda hack, but this effectively limits number of searches to ~100 per second
				// sleep for 10ms
				//rqtp.tv_nsec = 10*1000000L;
				//rqtp.tv_sec = rmtp.tv_sec = rmtp.tv_nsec = 0;
				//safe_nanosleep(&rqtp, &rmtp);
			}
			break;
		}
	}
}

