#define _LARGEFILE64_SOURCE     /* required for GLIBC to enable stat64 and friends */
#include <sys/types.h>
#include <regex.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <netdb.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include "mt.h"
#include "mem.h"
#include "error.h"
#include "globals.h"


int cv_offsets_compare(const void *a, const void *b)
{
	cv_off *pa = (cv_off *)a, *pb = (cv_off *)b;

	if (pa -> start > pb -> start)
		return -1;
	else if (pa -> start == pb -> start)
	{
		if (pa -> end > pb -> end)
			return -1;
	}

	return 0;
}

char *do_convert(char *what, int type)
{
	switch(type)
	{
		case CONVTYPE_IP4TOHOST:
			{
				struct hostent *ht;
				in_addr_t addr = inet_addr(what);
				if ((int)addr == -1)
					return mystrdup(what, "do_convert: unaltered string");

				if ((ht = gethostbyaddr((char *)&addr, sizeof(addr), AF_INET)) == NULL)
					return mystrdup(what, "do_convert: unaltered string");

				return mystrdup(ht -> h_name, "do_convert: looked-up hostname");
			}
			break; /* this redundant break-statement is what we call "defensive programming" */

		case CONVTYPE_EPOCHTODATE:
			{
				char *new_string;
				time_t ts = (time_t)atol(what);
				struct tm *ptm = localtime(&ts);
				if (!ptm)
					return mystrdup(what, "do_convert: unaltered string");

				new_string = mymalloc(4096, "do_convert: new string");
				if (!strftime(new_string, 4096, cnv_ts_format, ptm))
					error_exit("do_convert: error converting timestamp format '%s'\n", cnv_ts_format);

				return new_string;
			}
			break;

		case CONVTYPE_ERRNO:
			{
				return mystrdup(strerror(atoi(what)), "do_convert: error string");
			}

		case CONVTYPE_HEXTODEC:
			{
				long long int result = strtoll(what, NULL, 16);
				char result_str[128];

				snprintf(result_str, sizeof(result_str), "%lld", result);

				return mystrdup(result_str, "do_convert: decimal string");
			}

		case CONVTYPE_DECTOHEX:
			{
				long long int result = atoll(what);
				char result_str[128];

				snprintf(result_str, sizeof(result_str), "%llx", result);

				return mystrdup(result_str, "do_convert: hex string");
			}

		default:
			error_exit("do_convert: internal error, unknown conversion type (%d)\n", type);
	}

	return "do_convert: INTERNAL ERROR";
}

char *convert(proginfo *cur, char *line)
{
	conversion *cur_conv = NULL;
	cv_off *cv_offsets = NULL;
	int conv_index;
	int new_len = 0;
	int max_n_cv_matches = 0, cur_n_cv_matches = 0;
	char *new_string = NULL;
	int offset_old = 0, offset_new = 0;
	int old_len = strlen(line);

	if (cur -> conversion_index == -1)
		return line;

	cur_conv = &conversions[cur -> conversion_index];

	max_n_cv_matches = cur_conv -> n * MAX_N_RE_MATCHES;
	cv_offsets = (cv_off *)mymalloc(sizeof(cv_off) * max_n_cv_matches, "convert: array of offsets for conversions");

	/* find where they match */
	for(conv_index=0; conv_index<cur_conv -> n && cur_n_cv_matches < max_n_cv_matches; conv_index++)
	{
		regmatch_t matches[MAX_N_RE_MATCHES];

		if (regexec(&(cur_conv -> regex)[conv_index], line, MAX_N_RE_MATCHES, matches, 0) == 0)
		{
			int cur_match_index;
			LOG("convert match\n");
			for(cur_match_index=1; cur_match_index<MAX_N_RE_MATCHES && cur_n_cv_matches < max_n_cv_matches; cur_match_index++)
			{
				char *dummy;
				int dummylen;

				if (matches[cur_match_index].rm_so == -1)
					break;

				(cur_conv -> match_count)[conv_index]++;

				cv_offsets[cur_n_cv_matches].start = matches[cur_match_index].rm_so;
				cv_offsets[cur_n_cv_matches].end   = matches[cur_match_index].rm_eo;

				dummylen = matches[cur_match_index].rm_eo - matches[cur_match_index].rm_so;
				dummy = mymalloc(dummylen + 1, "convert: replace by");
				memcpy(dummy, &line[matches[cur_match_index].rm_so], dummylen);
				dummy[dummylen] = 0x00;
				cv_offsets[cur_n_cv_matches].newstr= do_convert(dummy, (cur_conv -> type)[conv_index]);
				LOG("convert newstr: %s\n", cv_offsets[cur_n_cv_matches].newstr);
				myfree(dummy);

				cur_n_cv_matches++;
			}
		}
	}

	if (cur_n_cv_matches)
	{
		int n_copy;

		/* sort */
		if (cur_n_cv_matches > 1)
			qsort(cv_offsets, cur_n_cv_matches, sizeof(cv_off), cv_offsets_compare);

		/* create new string */
		for(conv_index=0; conv_index < cur_n_cv_matches; conv_index++)
		{
			n_copy = cv_offsets[conv_index].start - offset_old;
			LOG("convert replace by: %s\n", cv_offsets[conv_index].newstr);
			if (n_copy > 0)
			{
				new_string = myrealloc(new_string, new_len + n_copy + 1, "convert: new_string");
				memcpy(&new_string[offset_new], &line[offset_old], n_copy);
				new_string[offset_new + n_copy] = 0x00;
				new_len += n_copy;
				offset_new += n_copy;
			}
			offset_old = cv_offsets[conv_index].end;

			n_copy = strlen(cv_offsets[conv_index].newstr);
			new_string = myrealloc(new_string, new_len + n_copy + 1, "convert: new_string");
			memcpy(&new_string[offset_new], cv_offsets[conv_index].newstr, n_copy);
			new_string[offset_new + n_copy] = 0x00;
			myfree(cv_offsets[conv_index].newstr);
			new_len += n_copy;
			offset_new += n_copy;
		}
		LOG("convert new string: %s\n", new_string);

		n_copy = old_len - offset_old;
		if (n_copy)
		{
			new_string = myrealloc(new_string, new_len + n_copy + 1, "convert: new_string");
			memcpy(&new_string[offset_new], &line[offset_old], n_copy);
			new_string[offset_new + n_copy] = 0x00;
		}
	}
	else
	{
		new_string = line;
	}

	myfree(cv_offsets);

	return new_string;
}
