#include <ctype.h>
#include <errno.h>
#include <sys/time.h>
#include <sys/types.h>
#include <regex.h>
#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

#include "mt.h"
#include "colors.h"
#include "error.h"
#include "term.h"
#include "utils.h"

extern char use_colors;
extern int n_colors;
extern char mail;
extern char do_refresh;
extern int max_y, max_x;

int term_type = TERM_MISC;

int wait_for_keypress()
{
	fd_set rfds;
	int rc, c;

	FD_ZERO(&rfds);
	FD_SET(0, &rfds);

	for(;;)
	{
		rc = select(1, &rfds, NULL, NULL, NULL);
		if (rc == -1)
		{
			if (errno != EINTR)
				error_exit("select failed");
		}
		else
			break;
	}

	c = getch();
	if (c == 3)	/* ^C */
	{
		do_exit(-1);
		error_exit("this should not be reached\n");
	}
	else if (c == 26)/* ^Z */
	{
		endwin();

		if (kill(getpid(), SIGTSTP) == -1)
			error_exit("problem suspending\n");

		doupdate();
	}

	return c;
}

void wrong_key(void)
{
	flash();
	flushinp();
}

char ask_yes_no(void)
{
	for(;;)
	{
		int c = toupper(wait_for_keypress());

		switch(c)
		{
		case 'Y':
		case 'J':
			return 1;

		case 'N':
			return 0;

		case 'Q':
			return -1;
		}

		wrong_key();
	}
}

void color_on(WINDOW *win, int index)
{
#ifdef N_CURSES
	if (use_colors)
		wattron(win, COLOR_PAIR(index));
#endif
}

void color_off(WINDOW *win, int index)
{
#ifdef N_CURSES
	if (use_colors)
		wattroff(win, COLOR_PAIR(index));
#endif
}

void draw_line(WINDOW *win, char where)
{
	int loop;
	int mx = getmaxx(win), my = getmaxy(win);

	getmaxyx(win, my, mx);

	wattron(win, A_REVERSE);

	if (where == LINE_LEFT)
		mvwvline(win, 0, 0, ' ', my);
	else if (where == LINE_RIGHT)
		mvwvline(win, 0, mx-1, ' ', my);
	else if (where == LINE_TOP)
		mvwhline(win, 0, 0, ' ', mx);
	else if (where == LINE_BOTTOM)
	{
		if (check_date())
		{
			for(loop=0; loop<mx; loop++)
			{
				color_on(win, loop % n_colors);
				mvwprintw(win, my-1, loop, " ");
				color_off(win, loop % n_colors);
			}
		}
		else
		{
			mvwhline(win, my-1, 0, ' ', mx);
		}
	}

	wattroff(win, A_REVERSE);
}

char * edit_string(WINDOW *win, int y, int win_width, int max_width, char numbers_only, char *input_string)
{
	char *string = (char *)mymalloc(max_width + 1, "edit buffer");
	int str_pos = 0, x = 0;
	int line_width = win_width - 4;

	wattron(win, A_REVERSE);
	mvwprintw(win, y, 1, ">");
	mvwprintw(win, y, win_width - 2, "<");
	wattroff(win, A_REVERSE);

	if (input_string)
	{
		int dummy = max(0, str_pos - line_width);
		strcpy(string, input_string);
		str_pos = dummy;
		mvwprintw(win, y, 2, &string[dummy]);
		x = strlen(string) - dummy;
	}
	else
	{
		string[0] = 0x00;
	}
	wmove(win, y, 2 + x);

	mydoupdate(win);

	for(;;)
	{
		char force_redraw = 0;
		int prev_str_pos = str_pos;
		int c = wait_for_keypress();

		/* confirm */
		if (c == KEY_ENTER || c == 13 || c == 10 )
			break;

		/* abort */
		if (c == 17 || c == 24) /* ^q / ^x */
		{
			string[0] = 0x00;
			break;
		}

		switch(c)
		{
		case 1:			/* ^A */
			str_pos = x = 0;
			break;
		case 5:			/* ^E */
			{
				int dummy = strlen(string);

				if (dummy > line_width)
				{
					str_pos = dummy - (line_width / 2);
					x = (line_width / 2);
				}
				else
				{
					str_pos = 0;
					x = dummy;
				}
			}
			break;
		case 9:			/* tab (filename completion) */
			if (numbers_only)
			{
				wrong_key();
			}
			else
			{
				int dummy = strlen(string);
				char *file = select_file(string);

				if (file)
				{
					strcpy(string, file);
					free(file);
					str_pos = 0;
				}

				dummy = strlen(string);
				if (dummy > line_width)
				{
					str_pos = dummy - (line_width / 2);
					x = (line_width / 2);
				}
				else
				{
					str_pos = 0;
					x = dummy;
				}

				force_redraw = 1;
			}
			break;
		case 23:		/* ^W */
			string[0] = 0x00;
			str_pos = x = 0;
			break;
		case 127:		/* DEL */
		case KEY_BACKSPACE:
			if ((str_pos + x) > 0)
			{
				memmove(&string[str_pos + x - 1], &string[str_pos + x], (max_width - (str_pos + x)) + 1);
				if (x > 0)
				{
					x--;
				}
				else
				{
					str_pos--;
				}

				force_redraw = 1;
			}
			break;
		case KEY_LEFT:
			if (x > 0)
			{
				x--;
			}
			else if (str_pos > 0)
			{
				str_pos--;
			}
			break;
		case KEY_RIGHT:
			if ((x + str_pos) < strlen(string))
			{
				if (x < line_width)
					x++;
				else
					str_pos++;
			}
			else
			{
				wrong_key();
			}
			break;
		default:
			{
				int len = strlen(string);

				/* only allow valid ASCII */
				if (c < 32)
				{
					wrong_key();
					break;
				}

				if (numbers_only && (c < '0' || c > '9'))
				{
					wrong_key();
					break;
				}

				if (len == max_width)
				{
					wrong_key();
					break;
				}

				/* cursor at end of string? */
				if (str_pos == len)
				{
					string[str_pos + x] = c;
					string[str_pos + x + 1] = 0x00;
					waddch(win, c);
				}
				else /* add character to somewhere IN the string */
				{
					memmove(&string[str_pos + x + 1], &string[str_pos + x], strlen(&string[str_pos + x]) + 1);
					string[str_pos + x] = c;
					force_redraw = 1;
				}

				if ((x + str_pos) < max_width)
				{
					if (x < line_width)
						x++;
					else
						str_pos++;
				}
				else
				{
					wrong_key();
				}
			}
			break;
		}

		if (str_pos != prev_str_pos || force_redraw)
		{
			int loop;
			char *dummy = mystrdup(&string[str_pos]);
			dummy[min(strlen(dummy), line_width)] = 0x00;
			for(loop=strlen(dummy); loop<line_width; loop++)
				mvwprintw(win, y, 2 + loop, " ");
			mvwprintw(win, y, 2, dummy);
			free(dummy);
			force_redraw = 0;
		}
		wmove(win, y, 2 + x);
		mydoupdate(win);
	}

	if (strlen(string) == 0)
	{
		free(string);
		string = NULL;
	}

	return string;
}

NEWWIN * create_popup(int n_lines, int n_colls)
{
	NEWWIN *newwin = mymalloc(sizeof(NEWWIN), "NEWWIN-structure");
	int x, y, index = 0;

	newwin -> ocols  = (max_x/2) - (n_colls/2);
	newwin -> olines = (max_y/2) - (n_lines/2);
	newwin -> ncols  = n_colls;
	newwin -> nlines = n_lines;

	/* copy old contents of screen */
	newwin -> pwc = (prev_win_contents *)mymalloc(n_lines * n_colls * sizeof(prev_win_contents), "previous window-contents");
	for(y=0; y<n_lines; y++)
	{
		for(x=0; x<n_colls; x++)
		{
			(newwin -> pwc)[index].prevchar = mvwinch(stdscr, newwin -> olines + y, newwin -> ocols + x);
			index++;
		}
	}

	/* create new window */
	newwin -> win = mysubwin(stdscr, n_lines, n_colls, newwin -> olines, newwin -> ocols);
	werase(newwin -> win);
	box(newwin -> win, 0, 0);

	return newwin;
}

void delete_popup(NEWWIN *mywin)
{
	int x, y;
	int index = 0;

	mydelwin(mywin -> win);

	for(y = 0; y < mywin -> nlines; y++)
	{
		for(x = 0; x < mywin -> ncols; x++)
		{
			mvwaddch(stdscr, mywin -> olines + y, mywin -> ocols + x, (mywin -> pwc)[index].prevchar);
			index++;
		}
	}

	mydoupdate(stdscr);

	free(mywin -> pwc);

	free(mywin);
}

void draw_marker_line(WINDOW *win)
{
	int mx = getmaxx(win);
	int loop;

	wattron(win, A_REVERSE);

	if (check_date())
	{
		for(loop=0; loop<mx; loop++)
		{
			color_on(win, loop % n_colors);
			waddch(win, '-');
			color_off(win, loop % n_colors);
		}
	}
	else
	{
		color_on(win, MY_RED);
		for(loop=0; loop<mx; loop++)
				waddch(win, '-');
		color_off(win, MY_RED);
	}

	wattroff(win, A_REVERSE);
}

void mydelwin(WINDOW *win)
{
	if (ERR == delwin(win))
		error_exit("delwin() failed\n");
}

void mydoupdate(WINDOW *win)
{
	wnoutrefresh(win);
	doupdate();
}

WINDOW * mysubwin(WINDOW *orig, int nlines, int ncols, int begin_y, int begin_x)
{
	WINDOW *dummy = subwin(orig, nlines, ncols, begin_y, begin_x);
	if (!dummy)
		error_exit("failed to create window (subwindow) with dimensions %d-%d at offset %d,%d (terminal size: %d,%d)\n", ncols, nlines, begin_x, begin_y, COLS, LINES);

	return dummy;
}

void escape_print(WINDOW *win, int y, int x, char *str)
{
	int loop, index = 0, len = strlen(str);
	char inv = 0;

	for(loop=0; loop<len; loop++)
	{
		if (str[loop] == '^')
		{
			if (!inv)
				wattron(win, A_REVERSE);
			else
				wattroff(win, A_REVERSE);

			inv = 1 - inv;
		}
		else
		{
			mvwprintw(win, y, x + index++, "%c", str[loop]);
		}
	}

	if (inv) wattroff(win, A_REVERSE);
}

void win_header(WINDOW *win, char *str)
{
	wattron(win, A_REVERSE);
	mvwprintw(win, 1, 2, str);
	wattroff(win, A_REVERSE);
}

void gui_window_header(char *string)
{
	if (term_type == TERM_XTERM)
	{
		/* \033]0;%s\007 */
		putp("\033]0;");
		putp(string);
		putp("\007");
	}
}
