#include "version.h"

#include <ctype.h>
#include <sys/types.h>
#include <dirent.h>
#include <errno.h>
#include <fcntl.h>
#include <limits.h>
#include <regex.h>
#ifndef __APPLE__
#include <search.h>
#endif
#include <math.h>
#include <signal.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#ifndef AIX
#include <sys/termios.h> /* needed on Solaris 8 */
#endif
#include <sys/time.h>
#include <sys/wait.h>
#include <unistd.h>
#if defined(__GLIBC__) && ( __GLIBC__ < 2 || (__GLIBC__ == 2 && __GLIBC_MINOR__ < 2))
	#include <sys/sysinfo.h>
#endif
#include <sys/utsname.h>

#include "mt.h"
#include "error.h"
#include "my_pty.h"
#include "utils.h"
#include "term.h"
#include "colors.h"
#include "scrollback.h"


/* #define KEYB_DEBUG */

proginfo *pi = NULL;
buffer *lb = NULL;
int nfd = 0;
int max_y, max_x;
char terminal_changed = 0;
char split = 0;
char banner = 1;
WINDOW *splitline = NULL;
char mode_statusline = 1;
char warn_closed = 1;
int n_colors = -1;
char use_colors = 0;
int min_n_bufferlines;
int path_max = 0;
int heartbeat = 0;
color_scheme *cschemes = NULL;
int n_cschemes = 0;
scheme_per_file *spf = NULL;
int n_scheme_per_file = 0;
char do_refresh = 0;
char mail = 0;
char check_for_mail = 5;	/* check for mail every 5 seconds */
char tab_width = 4;		/* some people use 8 */
time_t mt_started;
int vertical_split = -1;
int terminal_index = -1;
char prev_term_char = -1;
int n_keybindings = 0;
keybinding *keybindings = NULL;
pid_t children_list[MAX_N_SPAWNED_PROCESSES];
int n_children = 0;
extern int term_type;
char *set_title = NULL;

#ifdef _DEBUG
void LOG(char *s, ...)
{
        va_list ap;
	FILE *fh = fopen("log.log", "a+");
	if (!fh)
	{
		endwin();
		printf("error\n");
	}

        va_start(ap, s);
        vfprintf(fh, s, ap);
        va_end(ap);

	fclose(fh);
}
#else
#define LOG(x)
#endif

char *keys[] = {
		"Keys while the program runs:",
		"	q / x	exit program",
		"	r	redraw",
		"	e	edit regular expression for a given window",
		"	d	delete window",
		"	a	add window",
		"	s	swap windows",
		"	c	toggle colors",
		"	v	toggle vertical screen split",
		"	z	hide/unhide window",
		"	w	write commandline",
		"	0...9	set mark",
		"	b	scrollback",
		"	p	pause",
		"	i	info",
		"	h	help",
		NULL
		};

void init_curses(void)
{
	initscr();
	if (use_colors)
	{
		start_color(); /* don't care if this one failes */
	}
	keypad(stdscr, TRUE);
	cbreak();
	intrflush(stdscr, FALSE);
	leaveok(stdscr, TRUE);
	noecho();
	nonl();
	refresh();
	nodelay(stdscr, FALSE);
	meta(stdscr, TRUE);	/* enable 8-bit input */
	raw();			/* to be able to catch ctrl+c */
	idlok(stdscr, TRUE);	/* may give a little clunky screenredraw */

#ifdef N_CURSES
	if (use_colors)
	{
		init_pair(MY_RED, COLOR_RED, COLOR_BLACK);
		init_pair(MY_GREEN, COLOR_GREEN, COLOR_BLACK);
		init_pair(MY_YELLOW, COLOR_YELLOW, COLOR_BLACK);
		init_pair(MY_BLUE, COLOR_BLUE, COLOR_BLACK);
		init_pair(MY_MAGENTA, COLOR_MAGENTA, COLOR_BLACK);
		init_pair(MY_CYAN, COLOR_CYAN, COLOR_BLACK);
		init_pair(MY_WHITE, COLOR_WHITE, COLOR_BLACK);
	}
	n_colors = 8;
#endif

	max_y = LINES;
	max_x = COLS;
}

/** do_exit
 * - in:      int sig
 * - returns: nothing (doesn't return!)
 * this function is called when MultiTail receives the TERM-signal. it is also
 * called by, for example, wait_for_keypress when ^c is pressed. it stops all
 * child-processes, ends the curses-library and exits with SUCCESS status
 */
void do_exit(int sig)
{
	proginfo *cur;
	int loop;

	/* kill tail processes */
	for(loop=0; loop<nfd; loop++)
	{
		cur = &pi[loop];

		do
		{
			stop_process(cur -> pid);
			cur = cur -> next;
		}
		while(cur);
	}

	endwin();

	exit(EXIT_SUCCESS);
}

int execute_program(char *execute, char bg)
{
	int status;
	pid_t child;

	if (bg)
	{
		/* to prevent meltdowns, only a limited number of
		 * processes can be executed
		 */
		if (n_children >= MAX_N_SPAWNED_PROCESSES)
			return 0;
	}
	else
		endwin();

	child = fork();
	if (child == 0)
	{
		if (bg)
		{
			int fd = open("/dev/null", O_RDWR);
			if (fd == -1)
				error_exit("Failed to open /dev/null\n");

			close(2);
			close(1);
			close(0);
			dup(fd);
			dup(fd);
			dup(fd);
		}

		/* start process */
		if (-1 == execlp("/bin/sh", "/bin/sh", "-c", execute, (void *)NULL)) error_exit("execlp of %s failed\n", execute);

		/* if execlp returns, an error occured */
		error_exit("error while starting process!\n");
	}
	else if (child == -1)
	{
		error_exit("failed to fork child process\n");
	}

	if (bg)
	{
		/* remember this childprocess: we'll see if it has
		 * exited in the main-loop
		 */
		children_list[n_children++] = child;
	}
	else
	{
		/* wait for the childprocess to exit */
		if (waitpid(child, &status, 0) == -1)
			error_exit("failt waiting for process to exit!\n");

		/* restore (n)curses */
		doupdate();
	}

	return 0;
}

/** create_subwindow_list
 * - in:      int f_index     window number
 *            char ***swlist  pointer to an array of strings
 * - returns: int             number of elements in the array of strings
 * this function creates for a given window (f_index) a list of subwindows:
 * subwindows are created when you're merging the output of several files/
 * commands
 */
int create_subwindow_list(int f_index, char ***swlist)
{
	char **list = NULL;
	int n = 0;
	proginfo *cur = &pi[f_index];

	do
	{
		list = (char **)myrealloc(list, (n + 1) * sizeof(char *), "subwindow list");

		list[n] = mystrdup(cur -> filename);

		n++;

		cur = cur -> next;
	}
	while(cur);

	*swlist = list;

	return n;
}

void free_re(re *cur_re)
{
	free(cur_re -> regex_str);
	if (cur_re -> use_regex)
		regfree(&cur_re -> regex);
	free(cur_re -> cmd);
}

void free_subentry(proginfo *entry)
{
	int loop;

	/* free all those allocated memory blocks */
	free(entry -> filename);
	free(entry -> field_del);
	if (entry -> status)
		mydelwin(entry -> status);
	if (entry -> data)
		mydelwin(entry -> data);

	/* free buffers for diff (if any) */
	if (entry -> bcur)
		delete_array(entry -> bcur, entry -> ncur);
	if (entry -> bprev)
		delete_array(entry -> bprev, entry -> nprev);

	/* delete regular expressions */
	for(loop=0; loop<entry -> n_re; loop++)
		free_re(&(entry -> pre)[loop]);
	free(entry -> pre);

	/* stop process */
	stop_process(entry -> pid);
	if (waitpid(entry -> pid, NULL, WNOHANG | WUNTRACED) == -1)
	{
		if (errno != ECHILD)
			error_exit("waitpid failed\n");
	}

	/* close pipe to (tail) process */
	close(entry -> fd);
	close(entry -> wfd);
}

void store_for_diff(proginfo *cur, char *string)
{
	cur -> bcur = (char **)myrealloc(cur -> bcur, sizeof(char *) * (cur -> ncur + 1), "difference list");

	(cur -> bcur)[cur -> ncur] = mystrdup(string);

	cur -> ncur++;
}

void buffer_replace_pi_pointers(int f_index, proginfo *org, proginfo *new)
{
	int loop;

	for(loop=0; loop<lb[f_index].curpos; loop++)
	{
		if ((lb[f_index].pi)[loop] == org)
		{
			if (!new)
			{
				free((lb[f_index].Blines)[loop]);
				(lb[f_index].Blines)[loop] = NULL;
				(lb[f_index].pi)[loop] = NULL;
			}
			else
			{
				(lb[f_index].pi)[loop] = new;
			}
		}
	}
}

char delete_entry(int f_index, proginfo *sub)
{
	char delete_all = 0;

	/* no children? then sub must be pointing to current */
	if (pi[f_index].next == NULL)
	{
		sub = NULL;
	}

	/* stop the process(es) we're watching ('tail' most of the time) */
	if (sub == NULL) /* delete all? */
	{
		proginfo *cur = &pi[f_index];

		do
		{
			free_subentry(cur);
			cur = cur -> next;
		}
		while(cur);

		/* free the subwindows (if any) */
		cur = pi[f_index].next;
		while(cur)
		{
			proginfo *dummy = cur -> next;
			free(cur);
			cur = dummy;
		}

		delete_all = 1;
	}
	else
	{
		free_subentry(sub);
	}

	/* remove entry from array */
	if (sub == NULL) /* delete entry in the main array */
	{
		int n_to_move = (nfd - f_index) - 1;

	        /* free buffers */
		delete_array(lb[f_index].Blines, lb[f_index].curpos);

		if (n_to_move > 0)
		{
			int loop;

			/* update buffer proginfo-pointers */
			for(loop=f_index + 1; loop<nfd; loop++)
			{
				/* replace lb[f_index].pi -> pi[loop] met pi[loop-1] */
				buffer_replace_pi_pointers(loop, &pi[loop], &pi[loop - 1]);
			}

			/* prog info */
			memmove(&pi[f_index], &pi[f_index+1], sizeof(proginfo) * n_to_move);
			/* buffers */
			memmove(&lb[f_index], &lb[f_index+1], sizeof(buffer) * n_to_move);
		}

		nfd--;

		/*  shrink array */
		if (nfd == 0)	/* empty? */
		{
			free(pi);
			pi = NULL;
			free(lb);
			lb = NULL;
		}
		else		/* not empty, just shrink */
		{
			pi = (proginfo *)myrealloc(pi, nfd * sizeof(proginfo), "proginfo list");
			lb = (buffer *)myrealloc(lb, nfd * sizeof(buffer), "nfd list");
		}
	}
	else		/* delete sub */
	{
		if (sub != &pi[f_index])	/* not in main array? */
		{
			proginfo *parent = &pi[f_index];

			/* find parent of 'sub' */
			while (parent -> next != sub)
				parent = parent -> next;

			parent -> next = sub -> next;

			buffer_replace_pi_pointers(f_index, sub, NULL);

			free(sub);
		}
		else				/* main array, find next */
		{
			proginfo *oldnext = pi[f_index].next;
			/* first, delete the entries of that entry (which is in the array) */
			buffer_replace_pi_pointers(f_index, &pi[f_index], NULL);
			/* then, 'rename' the pointers (original address -> address in array)... */
			buffer_replace_pi_pointers(f_index, oldnext, &pi[f_index]);
			/* and move the object to the array */
			memmove(&pi[f_index], oldnext, sizeof(proginfo));
			/* free the obsolete entry */
			free(oldnext);
		}
	}

	return delete_all;
}

int start_tail(proginfo *cur, int initial_tail)
{
	if (cur -> is_command)
	{
		int fd_master, fd_slave;

		/* allocate pseudo-tty & fork*/
		cur -> pid = get_pty_and_fork(&fd_master, &fd_slave);
		if (-1 == cur -> pid) error_exit("failed to create pseudo-tty & fork\n");

		/* child? */
		if (cur -> pid == 0)
		{
			/* reset signal handler for SIGTERM*/
			signal(SIGTERM, SIG_DFL);

			/* sleep if requested and only when 2nd or 3d (etc.) execution time */
			if (cur -> restart && cur -> first == 0)
				sleep(cur -> restart);

			/* connect slave-fd to stdin/out/err */
			if (-1 == close(0)) error_exit("close failed\n");
			if (-1 == close(1)) error_exit("close failed\n");
			if (-1 == close(2)) error_exit("close failed\n");
			if (-1 == dup(fd_slave)) error_exit("dup failed\n");
			if (-1 == dup(fd_slave)) error_exit("dup failed\n");
			if (-1 == dup(fd_slave)) error_exit("dup failed\n");

			/* set terminal */
			if (putenv("TERM=dumb") == -1)
			{
				fprintf(stderr, "Could not set TERM environment-variable\n");
			}

			/* start process */
			if (-1 == execlp("/bin/sh", "/bin/sh", "-c", cur -> filename, (void *)NULL)) error_exit("execlp of %s failed\n", cur -> filename);

			/* if execlp returns, an error occured */
			error_exit("error while starting process!\n");
		}

		/* remember master-fd (we'll read from that one) */
		cur -> fd = fd_master;
		cur -> wfd = fd_master;

		/* next time, sleep */
		cur -> first = 0;
	}
	else
	{
		int pipefd[2];

		/* create a pipe, will be to child-process */
		if (-1 == pipe(pipefd)) error_exit("error creating pipe\n");

		/* start child process */
		if ((cur -> pid = fork()) == 0)
		{
			char par1[32] = { 0 };
			char par2[32] = { 0 };

			if (-1 == close(1)) error_exit("close failed\n");	/* close stdout */
			if (-1 == close(2)) error_exit("close failed\n");	/* close stderr */
			if (-1 == dup(pipefd[1])) error_exit("dup failed\n");	/* pipe-write is connected to stdout */
			if (-1 == dup(pipefd[1])) error_exit("dup failed\n");	/* pipe-write is connected to stderr */

			/* Linux' tail has the --retry option, but not all
			 * other UNIX'es have this, so I implemented it
			 * myself
			 */
			if (cur -> retry_open)
			{
				struct stat buf;

				for(;;)
				{
					int rc = stat(cur -> filename, &buf);
					if (rc == -1)
					{
						if (errno != ENOENT)
						{
							fprintf(stderr, "Error while looking for file %s: %d\n", cur -> filename, errno);
							exit(EXIT_FAILURE);
						}
					}
					else if (rc == 0)
						break;

					usleep(WAIT_FOR_FILE_DELAY * 1000);
				}
			}

			/* create command for take last n lines & follow and start tail */
#if defined(__FreeBSD__) || defined(__NetBSD__) || defined(__OpenBSD__) || defined(linux) || defined(__CYGWIN__) || defined(__APPLE__)
			if (cur -> follow_filename)
			{
	#if defined(linux) || defined(__CYGWIN__)
				sprintf(par1, "--follow=name");
	#else
				sprintf(par1, "-F");
	#endif
				sprintf(par2, "-n %d", initial_tail);
			}
			else
#endif
			{
				sprintf(par1, "-%dlf", initial_tail);
			}

			/* run tail! */
			if (strlen(par2))
			{
				if (-1 == execlp("tail", "tail", par1, par2, cur -> filename, (void *)NULL)) error_exit("execlp of tail failed");
			}
			else
			{
				if (-1 == execlp("tail", "tail", par1, cur -> filename, (void *)NULL)) error_exit("execlp of tail failed");
			}

			/* if execlp returns, an error occured */
			error_exit("error while starting process!\n");
		}

		cur -> fd = pipefd[0];
		cur -> wfd = pipefd[1];
	}

	if (cur -> pid == -1)
		return -1;

	return 0;
}

int gen_color(char *start, char *end)
{
	char *loop;
	int chk = 0;

	for(loop=start; loop<end; loop++)
	{
		chk ^= *loop;
	}

	return abs(chk) % n_colors;
}

int compare_filenames(const void *arg1, const void *arg2)
{
	return strcmp(*(char **)arg1, *(char **)arg2);
}

int match_files(char *search_for, char **path, char ***found)
{
	DIR *dir;
	struct dirent *entry;
	char *fname;
	char **list = NULL;
	int nfound = 0;
	size_t fname_size;
	char *slash = strrchr(search_for, '/');
	if (slash)
	{
		fname = mystrdup(slash + 1);
		*(slash + 1) = 0x00;
		*path = search_for;
	}
	else
	{
		*path = "./";
		fname = mystrdup(search_for);
	}
	fname_size = strlen(fname);

	dir = opendir(*path);
	if (!dir)
	{
		return 0;
	}

	while((entry = readdir(dir)) != NULL)
	{
		if ((fname_size == 0 || strncmp(entry -> d_name, fname, fname_size) == 0) && strcmp(entry -> d_name, ".") != 0 &&
											     strcmp(entry -> d_name, "..") != 0)
		{
			list = (char **)myrealloc(list, (nfound + 1) * sizeof(char *), "directory list");

			list[nfound] = mystrdup(entry -> d_name);
			nfound++;
		}
	}

	if (closedir(dir) == -1)
		error_exit("closedir failed");

	qsort( (void *)list, (size_t)nfound, sizeof(char *), compare_filenames);

	*found = list;

	free(fname);

	return nfound;
}

char generate_string(char *whereto, void **list, char type, int wcols, int index)
{
	char invert = 0;

	if (type == SEL_WIN)
	{
		snprintf(whereto, wcols + 1, "%02d %s", index, ((proginfo *)pi)[index].filename);
		invert = pi[index].hidden;
	}
	else if (type == SEL_SUBWIN || type == SEL_FILES)
		strncpy(whereto, ((char **)list)[index], min(strlen(((char **)list)[index]), wcols) + 1);
	else if (type == SEL_CSCHEME)
		strncpy(whereto, ((color_scheme *)list)[index].name, min(strlen(((color_scheme *)list)[index].name), wcols) + 1);
	else
		error_exit("internal error (%d is an unknown selectionbox type)\n", type);

	whereto[min(strlen(whereto), wcols)] = 0x00;

	return invert;
}

int selection_box(void **list, int nlines, char type)
{
	NEWWIN *mywin;
	int wlines = min(nlines, (max_y - 1) - 3);
	int total_win_size = wlines + 3;
	int wcols = (max_x / 4) - 4;
	int pos = 0, ppos = -1, offs = 0, poffs = -1;
	int loop, sel = -1;
	char invert;
	char *dummy = (char *)mymalloc(wcols + 1, "windowname string");

	mywin = create_popup(total_win_size, max_x / 4);

	for(;;)
	{
		int c;

		/* draw list */
		if (pos != ppos)
		{
			int entries_left = (nlines - pos);

			werase(mywin -> win);

			win_header(mywin -> win, "Select file");

			for(loop=0; loop<min(entries_left, wlines); loop++)
			{
				invert = generate_string(dummy, list, type, wcols, loop + pos);
				if (loop == offs)
					wattron(mywin -> win, A_REVERSE);
				if (invert)
					color_on(mywin -> win, 3);
				mvwprintw(mywin -> win, loop + 2, 2, "%s", dummy);
				if (invert)
					color_off(mywin -> win, 3);
				if (loop == offs)
					wattroff(mywin -> win, A_REVERSE);
			}

			box(mywin -> win, 0, 0);

			mydoupdate(mywin -> win);

			ppos = pos;
			poffs = offs;
		}
		else if (poffs != offs)
		{
			generate_string(dummy, list, type, wcols, poffs + pos);
			mvwprintw(mywin -> win, poffs + 2, 2, "%s", dummy);

			invert = generate_string(dummy, list, type, wcols, offs + pos);

			wattron(mywin -> win, A_REVERSE);
			if (invert)
				color_on(mywin -> win, 3);
			mvwprintw(mywin -> win, offs + 2, 2, "%s", dummy);
			if (invert)
				color_off(mywin -> win, 3);
			wattroff(mywin -> win, A_REVERSE);

			mydoupdate(mywin -> win);

			poffs = offs;
		}

		c = wait_for_keypress();

		if (c == KEY_UP)
		{
			if ((offs + pos) > 0)
			{
				if (offs)
					offs--;
				else
					pos--;
			}
			else
			{
				wrong_key();
			}
		}
		else if (c == KEY_DOWN)
		{
			if ((pos + offs) < (nlines-1))
			{
				if (offs < (wlines-1))
					offs++;
				else
					pos++;
			}
			else
			{
				wrong_key();
			}
		}
		else if (c == KEY_NPAGE)
		{
			if ((pos + offs) < (nlines - 1))
			{
				pos += min(wlines, (nlines - 1) - (pos + offs));
			}
			else
			{
				wrong_key();
			}
		}
		else if (c == KEY_PPAGE)
		{
			if ((pos + offs - wlines) >= 0)
			{
				if (pos > wlines)
				{
					pos -= wlines;
				}
				else
				{
					pos -= (wlines - offs);
					offs = 0;
				}
			}
			else if (offs > 0)
			{
				offs = 0;
			}
			else if (pos > 0)
			{
				pos = 0;
			}
			else
			{
				wrong_key();
			}
		}
		else if (c == KEY_ENTER || c == 13 || c == 10)
		{
			sel = pos + offs;
			break;
		}
		else if (toupper(c) == 'Q' || toupper(c) == 'X')
		{
			break;
		}
		else
		{
			wrong_key();
		}
	}

	delete_popup(mywin);

	free(dummy);

	return sel;
}

int select_window(void)
{
	return selection_box((void **)pi, nfd, SEL_WIN);
}

proginfo * select_subwindow(int f_index)
{
	proginfo *cur = NULL;
	char **list;
	int list_n, index, loop;
	
	if (f_index == -1)
		return NULL;

	list_n = create_subwindow_list(f_index, &list);

	index = selection_box((void **)list, list_n, SEL_SUBWIN);

	if (index != -1)
	{
		cur = &pi[f_index];
		for(loop=0; loop<index; loop++)
		{
			cur = cur -> next;
		}
	}

	delete_array(list, list_n);

	return cur;
}

char * select_file(char *input)
{
	char **list = NULL, *path;
	char *new_fname = NULL;
	struct stat statbuf;
	int list_n, index;

	list_n = match_files(input, &path, &list);
	if (list_n == 0)
	{
		flash();
		return NULL;
	}

	index = selection_box((void **)list, list_n, SEL_FILES);
	if (index != -1)
	{
		new_fname = (char *)mymalloc(path_max + 1, "new filename");

		sprintf(new_fname, "%s%s", path, list[index]);

		if (stat(new_fname, &statbuf) == -1)
		{
			free(new_fname);
			new_fname = NULL;
			flash();
		}
		else
		{
			if (S_ISDIR(statbuf.st_mode))
			{
				strcat(new_fname, "/");
			}
		}
	}

	delete_array(list, list_n);

	return new_fname;
}

int choose_color(char *string, proginfo *cur, WINDOW *win, color_offset_in_line *cmatches, int *n_cmatches)
{
	int color = -1, loop;
	char *sp = string, *dummy = NULL;
	regmatch_t colormatches[MAX_N_RE_MATCHES];

	/* find color */
	switch(cur -> colorize)
	{
	case 's':
		if (strlen(string) >= 16)
		{
			sp = strchr(&string[16], ' ');
		}

		while(sp && *sp == ' ') sp++;

		if (sp)
		{
			char *end1 = strchr(sp, '[');
			char *end2 = strchr(sp, ' ');
			char *end3 = strchr(sp, ':');
			char *end = NULL;

			end = end1;
			if ((end2 && end2 < end) || (end == NULL))
				end = end2;
			if ((end3 && end3 < end) || (end == NULL))
				end = end3;

			if (end)
				color = gen_color(sp, end);
			else
				color = gen_color(sp, &sp[strlen(sp)]);
		}
		else
		{
			color = 0;
		}
		break;

	case 'f':
		{
			int nlen = strlen(cur -> field_del);

			for(loop=0; loop<cur -> field_nr; loop++)
			{
				sp = strstr(sp, cur -> field_del);
				while(sp)
				{
					if (strncmp(sp, cur -> field_del, nlen) == 0)
					{
						sp += nlen;
					}
					else
					{
						break;
					}
				}

				if (!sp)
					break;
			}

			if (sp)
				dummy = strstr(sp, cur -> field_del);

			if (dummy != NULL && sp != NULL)
				color = gen_color(sp, dummy);
			else if (sp)
				color = gen_color(sp, &sp[strlen(sp)]);
			else
				color = 0;
		}
		break;

	case 'm':
		color = gen_color(string, &string[strlen(string)]);
		break;

	case 'S':
		for(loop=0; loop<cschemes[cur -> color_scheme].n; loop++)
		{
			int offset = 0;
			char match = 1;

			while(offset < strlen(string) && match)
			{
				int rc;

				match = 0;

				if ((rc = regexec(&cschemes[cur -> color_scheme].regex[loop], &string[offset], MAX_N_RE_MATCHES, colormatches, offset?REG_NOTBOL:0)) == 0)
				{
					int loop2;

					match = 1;

					for(loop2=0; loop2<MAX_N_RE_MATCHES; loop2++)
					{
						if (colormatches[loop2].rm_so == -1)
							break;

						cmatches[*n_cmatches].start = colormatches[loop2].rm_so + offset;
						cmatches[*n_cmatches].end   = colormatches[loop2].rm_eo + offset;
						cmatches[*n_cmatches].color_index = cschemes[cur -> color_scheme].color[loop];

						offset = max(offset, colormatches[loop2].rm_eo + offset);

						(*n_cmatches)++;
						if (*n_cmatches == MAX_COLORS_PER_LINE)
							return -1;
					}
				}
				else	/* an error occured */
				{
					char *error = convert_regexp_error(rc, &cschemes[cur -> color_scheme].regex[loop]);

					if (error)
					{
						wprintw(win, "%s", error);
						free(error);
					}
				}
			}
		}
		break;

	}

	return color;
}

void select_display_start_and_end(char *string, char mode, int wrap_offset, int win_width, int *prt_start, int *prt_end)
{
	int nbytes = strlen(string);

	*prt_start = 0;
	*prt_end = nbytes;

	/* figure out what to display (because of linewraps and such) */
	if (mode == 'a')
	{
		/* default is everything */
	}
	else if (mode == 'l')
	{
		*prt_end = min(nbytes, win_width);
	}
	else if (mode == 'r')
	{
		*prt_start = max(0, *prt_end - win_width);
	}
	else if (mode == 'S')
	{
		*prt_start = find_char_offset(string, ':');
		if (*prt_start == -1)
			*prt_start = 0;
	}
	else if (mode == 's')
	{
		if (nbytes > 16)
			*prt_start = find_char_offset(&string[16], ' ');
		else
			*prt_start = -1;

		if (*prt_start == -1)
			*prt_start = 0;
	}
	else if (mode == 'o')
	{
		*prt_start = min(wrap_offset, nbytes);
	}
}

int find_in_regexec_resultset(regmatch_t *matches, int set_size, int offset)
{
	int index = -1;
	int loop;

	for(loop=0; loop<set_size; loop++)
	{
		if (offset >= matches[loop].rm_so && offset < matches[loop].rm_eo)
		{
			index = loop;
			break;
		}
		else if (matches[loop].rm_so == -1)
		{
			break;
		}
	}

	return index;
}

void draw_tab(WINDOW *win)
{
	if (tab_width)
	{
		int cury, curx, move, loop;

		/* get current coordinate */
		getyx(win, cury, curx);

		/* calculate new coordinate */
    		move = (((curx / tab_width) + 1) * tab_width) - curx;

		for(loop=0; loop<move; loop++)
			waddch(win, ' ');
	}
}

void color_print(WINDOW *win, proginfo *cur, unsigned char *string, regmatch_t *matches, int matching_regex)
{
	char reverse = 0;
	int loop;
	int color = -1;
	int prt_start = 0, prt_end;
	int mx = getmaxx(win);
	char use_regex = 0;
	color_offset_in_line cmatches[MAX_COLORS_PER_LINE]; /* max. 80 colors per line */
	int n_cmatches = 0;

	/* cur == NULL? Markerline! */
	if (!cur)
	{
		draw_marker_line(win);

		return;
	}

	/* select color or generate list of offset for colors */
	if (use_colors && cur -> colorize) /* no need to do this for b/w terminals */
	{
		color = choose_color(string, cur, win, cmatches, &n_cmatches);

		/* if not using colorschemes (which can have more then one color
		 * per line), set the color
		 */
		if (cur -> colorize != 'S')
			color_on(win, color);
	}

	/* select start and end of string to display */
	select_display_start_and_end(string, cur -> line_wrap, cur -> line_wrap_offset, mx, &prt_start, &prt_end);

	if (matching_regex != -1)
	{
		use_regex = (cur -> pre)[matching_regex].use_regex;
	}

	/* print text */
	for(loop=prt_start; loop<prt_end; loop++)
	{
		char re_inv = 0;

		/* find things to invert */
		if (matches != NULL && (use_regex == 'c' || use_regex == 'C' || use_regex == 'b'))
		{
			/* is this offset in the string in the given (-> function-parameter)
			 * resultset? then invert
			 */
			if (find_in_regexec_resultset(matches, MAX_N_RE_MATCHES, loop) != -1)
				re_inv = 1;
		}

		/* find things to colorize */
		if (use_colors && cur -> colorize == 'S')
		{
			int loop2, new_color = -1;

			for(loop2=0; loop2<n_cmatches; loop2++)
			{
				if (loop >= cmatches[loop2].start && loop < cmatches[loop2].end)
				{
					new_color = cmatches[loop2].color_index;
					break;
				}
			}

			if (color != new_color)
			{
				color_off(win, color);
				color = -1;
			}

			if (new_color != -1 && color != new_color)
			{
				color = new_color;
				color_on(win, color);
			}
		}

		/* control-characters will be displayed as an inverse '.' */
		if (string[loop] < 32  || string[loop] == 127)
		{
			if (string[loop] != 10 && (string[loop] != 9 && tab_width > 0))
			{
				re_inv = 1;
			}
		}

		if (re_inv)
		{
			if (!reverse)
			{
				wattron(win, A_REVERSE);
				reverse = 1;
			}
		}

		if (string[loop] == 9)	/* TAB? */
		{
			draw_tab(win);
		}
		else if (string[loop] < 32 || string[loop] == 127)	/* control-characters? */
		{
			waddch(win, '.');
		}
		else
		{
			waddch(win, string[loop]);
		}

		if (reverse)
		{
			wattroff(win, A_REVERSE);
			reverse = 0;
		}
	}

	if (cur -> colorize && color != -1)
	{
		color_off(win, color);
	}
}

/* checks if this window has regexps that are only for actions (bell, exec proc, etc.)
 * because for those: things should still print
 */
char check_no_suppress_lines_filter(proginfo *cur)
{
	int loop;
	char print_only_when_matches = 0;

	/* do this regular expression? */
	for(loop=0; loop<cur -> n_re; loop++)
	{
		char cur_regexp_type = (cur -> pre)[loop].use_regex;

		if (cur_regexp_type != 'C' && cur_regexp_type != 'b' && cur_regexp_type != 'x')
		{
			print_only_when_matches = 1;
			break;
		}
	}

	return print_only_when_matches;
}

char check_filter(proginfo *cur, char *string, regmatch_t **pmatch, char **error, int *matching_regex)
{
	int loop;
	/* if no regexps, then always matches so it's the default */
	char matches = 1;

	*error = NULL;
	*pmatch = NULL;

	/* do this regular expression? */
	for(loop=0; loop<cur -> n_re; loop++)
	{
		if ((cur -> pre)[loop].use_regex)
		{
			int rc;

			matches = 0;

			if (pmatch)
			{
				if (*pmatch == NULL)
					*pmatch = (regmatch_t *)mymalloc(sizeof(regmatch_t) * MAX_N_RE_MATCHES, "matching regular expressions");

				rc = regexec(&(cur -> pre)[loop].regex, string, MAX_N_RE_MATCHES, *pmatch, 0);
			}
			else
			{
				rc = regexec(&(cur -> pre)[loop].regex, string, 0, NULL, 0);
			}

			/* matches? */
			if ((rc == 0 && (cur -> pre)[loop].invert_regex == 0) || (rc == REG_NOMATCH && (cur -> pre)[loop].invert_regex == 1))
			{
				if (rc == REG_NOMATCH && (cur -> pre)[loop].invert_regex == 1)
				{
					if (pmatch)
					{
						free(*pmatch);
						*pmatch = NULL;
					}
				}

				*matching_regex = loop;

				return 1;
			}
			/* no match: error actually! */
			else if (rc != REG_NOMATCH && rc != 0)
			{
				*error = convert_regexp_error(rc, &(cur -> pre)[loop].regex);
				break;
			}
		}
	}

	free(*pmatch);
	*pmatch = NULL;
	*matching_regex = -1;

	return matches;
}

void print_and_buffer(int f_index, proginfo *cur, char *string)
{
	regmatch_t *pmatch = NULL;
	int matching_regex = -1;
	char ok;
	char *error = NULL;
	WINDOW *win = pi[f_index].data;
	int x, y;
	if (!win)
	{
		return;
	}

	/* only proceed to the next line if we're not already there (for example
	 * when the previous line had the same length as the windowsize)
	 */
	getyx(win, y, x);
	if (x)
		wprintw(win, "\n");

	/* check filter */
	/* if cur == NULL, then marker line: don't check against filter */
	if (cur)
	{
		ok = check_filter(cur, string, &pmatch, &error, &matching_regex);
		if (ok)
		{
			color_print(win, cur, (unsigned char *)string, pmatch, matching_regex);

			/* did check_filter return ok because there was a matching regexp or just
			 * because there were no regexps at all?
			 */
			if (matching_regex != -1)
			{
				/* execute command if regexp matches? */
				if ((cur -> pre)[matching_regex].use_regex == 'x')
				{
					char *command = mymalloc(strlen((cur -> pre)[matching_regex].cmd) + 1/* cmd + space */
							+ 1 + (strlen(string) * 2) + 1 + 1, "command string");     /* "string"\0 */
					int str_index, par_len = strlen(string);
					int loop;

					strcpy(command, (cur -> pre)[matching_regex].cmd);
					str_index = strlen(command);
					command[str_index++] = ' ';
					command[str_index++] = '\"';
					for(loop=0; loop<par_len; loop++)
					{
						if (string[loop] == '"' ||
						    string[loop] == '`'
						   )
						{
							command[str_index++] = '\\';
						}

						command[str_index++] = string[loop];
					}
					command[str_index++] = '\"';
					command[str_index] = 0x00;

					if (execute_program(command, 1) == -1)
						error_exit("Failed to execute command '%s'!", command);

					free(command);
				}
				/* sound bell when matches */
				else if (toupper((cur -> pre)[matching_regex].use_regex) == 'B')
				{
					beep();
				}
			}
		}
		/* no matches because everything was suppressed by the regexps? if that
		 * is NOT the case, DO print the line
		 */
		else if (check_no_suppress_lines_filter(cur) == 0)
		{
			color_print(win, cur, (unsigned char *)string, NULL, -1);
			ok = 1;	/* DO remember this line */
		}
		else if (error)
		{
			color_print(win, cur, error, NULL, -1);
			free(error);
		}
	}
	else
	{
		ok = 1; /* markerline: DO remember */
		color_print(win, NULL, NULL, NULL, -1);
	}

	/* remember string */
	if (ok || lb[f_index].markset == 'a')
	{
		int curline;

		if (lb[f_index].curpos < lb[f_index].maxnlines || lb[f_index].maxnlines == 0)
		{
			lb[f_index].Blines = (char **)myrealloc(lb[f_index].Blines, sizeof(char *) *
				(lb[f_index].curpos + 1), "lines buffer");

			lb[f_index].pi    = (proginfo **)myrealloc(lb[f_index].pi, sizeof(proginfo *) *
				(lb[f_index].curpos + 1), "lines buffer progrinfo");

			curline = lb[f_index].curpos;
			lb[f_index].curpos++;
		}
		else
		{
			/* delete oldest */
			free((lb[f_index].Blines)[0]);
			curline = lb[f_index].maxnlines-1;
			memmove(&(lb[f_index].Blines)[0], &(lb[f_index].Blines)[1], (lb[f_index].maxnlines-1) * sizeof(char *));
			memmove(&(lb[f_index].pi)[0], &(lb[f_index].pi)[1], (lb[f_index].maxnlines-1) * sizeof(proginfo *));
		}

		if (string)
			(lb[f_index].Blines)[curline] = mystrdup(string);
		else
			(lb[f_index].Blines)[curline] = NULL;
		(lb[f_index].pi   )[curline] = cur;
	}
}

void update_statusline(WINDOW *status, int win_nr, proginfo *cur)
{
	if (mode_statusline > 0)
	{
		char *dummy;
		int dx, dy;
		int color = -1;
		int statusline_len;
		off_t	fsize = (off_t)-1;
		time_t	ts = (time_t)0;


		if (win_nr == terminal_index)
			color = MY_RED;
		else if (mail)
			color = MY_GREEN;

		if (color != -1)
			color_on(status, color);

		draw_line(status, LINE_BOTTOM);

		wattron(status, A_REVERSE);

		mvwprintw(status, 0, 0, "%02d] %s%s", win_nr, cur -> filename, win_nr == terminal_index ? ", press <CTRL>+<A>, <D> to exit" : mail ? " You've got mail!" : "");

		if (cur -> is_command)
			time(&ts);
		else
			(void)file_info(cur -> filename, &fsize, &ts);
		dummy = ctime(&ts);

		statusline_len = 4 + strlen(cur -> filename) + strlen(dummy) + 1; /* 4: '%02d] '!! (window nr.) */

		getmaxyx(status, dy, dx);
		if (dx >= (statusline_len + 13))
		{
			if (cur -> is_command)
			{
				int vmsize = get_vmsize(cur -> pid);

				if (vmsize != -1 && dx >= (statusline_len + 30))
				{
					char *vmsize_str = amount_to_str(vmsize);

					mvwprintw(status, 0, dx - strlen(dummy) - 30, "%6s (VMsize) %5d (PID) - %s", vmsize_str, cur -> pid, dummy);

					free(vmsize_str);
				}
				else
					mvwprintw(status, 0, dx - strlen(dummy) - 12, "%5d (PID) - %s", cur -> pid, dummy);
			}
			else if (fsize == -1)
			{
				mvwprintw(status, 0, dx - strlen(dummy) - 6, "??? - %s", dummy);
			}
			else
			{
				/* remove the lf at the end as it isn't viewable and should not take up space */
				char *lf = strchr(dummy, '\n');
				if (lf) *lf = 0x00;

				/* this trick is because on MacOS X 'off_t' is specified as a 64 bit integer */
				if (sizeof(off_t) == 8)
					mvwprintw(status, 0, dx - strlen(dummy) - 13, "%10lld - %s", fsize, dummy);
				else
					mvwprintw(status, 0, dx - strlen(dummy) - 13, "%10ld - %s", fsize, dummy);
			}
		}

		wattroff(status, A_REVERSE);

		if (color != -1)
			color_off(status, color);

		wnoutrefresh(status);
	}
}

void create_window_set(int startx, int width, int nwindows, int indexoffset)
{
	int loop;
	int n_window_lines, add_lines;
	int n_not_hidden = 0, win_counter=0;
	int added = 0;
	int term_offset = 0;

	for(loop=0; loop<nwindows; loop++)
	{
		if (pi[loop + indexoffset].hidden == 0)
			n_not_hidden++;
	}

	if (n_not_hidden == 0)
		return;

	/* calculate the size of each window */
	n_window_lines = max_y / n_not_hidden;

	/* each window is less then 3 lines? then hide one or more
	 * windows
	 */
	if (n_window_lines < 3)
	{
		n_not_hidden = max_y / 3;
		n_window_lines = 3;
	}
	/* the number of windows multiplied with their sizes may be
	 * less then the height of the terminalwindow, in that case
	 * one or more windows can be 1 line longer
	 */
	add_lines = max_y - (n_window_lines * n_not_hidden);

	/* re-create windows */
	for(loop=0; loop<nwindows; loop++)
	{
		int cur_index = loop + indexoffset;

		if (pi[cur_index].hidden == 0)
		{
			int add = 0;
			int cur_win_length;
			int cur_term_size = max_y - term_offset;
			int loop2;
			regmatch_t *pmatch = NULL;
			char *error = NULL, ok;
			int n_windows_left = n_not_hidden - win_counter;

			if (add_lines > 0)
			{
				add = 1;
				add_lines--;
			}

			cur_win_length = (n_window_lines - (mode_statusline >= 0?1:0)) + add;

			if (pi[cur_index].win_height != -1)
			{
				int dummy = pi[cur_index].win_height - (mode_statusline >= 0?1:0);

				if (dummy < cur_win_length)
				{
					add_lines += (cur_win_length - dummy);
				}
				else if (dummy > cur_win_length)
				{
					add_lines -= (dummy - cur_win_length);
					if (add_lines < 0)
						add_lines = 0;
				}

				cur_win_length = dummy;
			}

			if (cur_term_size < (cur_win_length + 1))
			{
				cur_win_length = (cur_term_size / n_windows_left) - 1;
				add_lines = max_y - ((cur_win_length + 1) * n_windows_left);
			}

			if (cur_win_length < 2)
			{
				if (n_windows_left > 1 && cur_win_length == 1)
					cur_win_length = 2;
				else
					continue;
			}

			if (win_counter == (n_not_hidden - 1))
				cur_win_length = cur_term_size -1;

			/* creat data window, clear en set scroll ok */
			pi[cur_index].data   = mysubwin(stdscr, cur_win_length, width, term_offset, startx);
			werase(pi[cur_index].data);

			scrollok(pi[cur_index].data, TRUE);/* supposed to always return OK, according to the manpage */

			/* create status window */
			if (mode_statusline >= 0)
			{
				pi[cur_index].status = mysubwin(stdscr, 1, width, term_offset + cur_win_length, startx);
				werase(pi[cur_index].status);

				color_on(pi[cur_index].status, 0);

				/* create status-line */
				update_statusline(pi[cur_index].status, cur_index, &pi[cur_index]);
			}

			/* display old strings */
			for(loop2=max(0, lb[cur_index].curpos - cur_win_length); loop2<lb[cur_index].curpos; loop2++)
			{
				wprintw(pi[cur_index].data, "\n");

				if (!lb[cur_index].pi[loop2] && lb[cur_index].Blines[loop2] != NULL)
				{
					color_print(pi[cur_index].data, NULL, NULL, NULL, -1);
				}
				else if ((lb[cur_index].Blines)[loop2])
				{
					int matching_regex = -1;

					/* check filter */
					ok = check_filter(lb[cur_index].pi[loop2], lb[cur_index].Blines[loop2], &pmatch, &error, &matching_regex);
					if (ok)
					{
						color_print(pi[cur_index].data, (lb[cur_index].pi)[loop2], (lb[cur_index].Blines)[loop2], pmatch, matching_regex);
					}
					else if (check_no_suppress_lines_filter(lb[cur_index].pi[loop2]) == 0)
					{
						color_print(pi[cur_index].data, (lb[cur_index].pi)[loop2], (lb[cur_index].Blines)[loop2], NULL, -1);
					}
					else if (error)
					{
						color_print(pi[cur_index].data, (lb[cur_index].pi)[loop2], error, NULL, -1);
						free(error);
					}
					free(pmatch);
				}
			}
			wnoutrefresh(pi[cur_index].data);

			if (add)
			{
				added++;
			}

			/* keep track of the number windows displayed on the terminal-
			 * window
			 */
			win_counter++;

			term_offset += cur_win_length + 1;

			/* n_not_hidden not only indicates the number of windows that are
			 * not hidden, it also indicates the number of windows that *can*
			 * be put in the terminal (due to the height of the terminal).
			 * so if the number of displayed windows is equal to the number of
			 * windows that fit in the terminal, break out of the loop
			 */
			if (win_counter == n_not_hidden)
				break;
		}
	}
}

void create_windows(void)
{
	int loop;
	int split_x;

	/* close windows */
	for(loop=0; loop<nfd; loop++)
	{
		if (pi[loop].status)
		{
			mydelwin(pi[loop].status);
			pi[loop].status = NULL;
		}
		if (pi[loop].data)
		{
			mydelwin(pi[loop].data);
			pi[loop].data = NULL;
		}
	}

	if (splitline)
	{
		mydelwin(splitline);
		splitline = NULL;
	}

	/* here it is ok if ncurses decides to rebuild the whole
	 * terminal, so not using werase here
	 */
	wclear(stdscr);

	if (vertical_split != -1)
	{
		split_x = vertical_split;

		if (split_x > max_x)
			split_x = max_x / 2;

		if (split_x < 4 || (max_x - split_x) < 4)
			split_x = max_x / 2;

		if (split_x < 4)
			split_x = 0;
	}
	else
	{
		split_x = max_x / 2;
	}

	if (split && nfd > 1)
	{
		int half_nfd = nfd / 2;
		int left_n = nfd - half_nfd;
		int n_not_hidden = 0;

		for(loop=left_n; loop<nfd; loop++)
		{
			if (pi[loop].hidden == 0)
				n_not_hidden++;
		}

		if (n_not_hidden > 0)
		{
			create_window_set(0, split_x, left_n, 0);
			create_window_set(split_x + 1, (max_x - split_x) - 1, half_nfd, left_n);

			splitline = mysubwin(stdscr, max_y, 1, 0, split_x);

			draw_line(splitline, LINE_LEFT);
			wnoutrefresh(splitline);
		}
		else
		{
			create_window_set(0, max_x, nfd, 0);
		}
	}
	else if (nfd != 0)
	{
		create_window_set(0, max_x, nfd, 0);
	}

	doupdate();
}

void do_resize(int s)
{
        struct winsize size;

        if (ioctl(STDIN_FILENO, TIOCGWINSZ, &size) == 0)
	{
		terminal_changed = 1;

		max_y = size.ws_row;
		max_x = size.ws_col;
	}
	else
	{
		error_exit("ioctl TIOCGWINSZ failed\n");
	}

	signal(SIGWINCH, do_resize);
}

int swap_window(void)
{
	int swapped = 0;
	NEWWIN *mywin = create_popup(13 + nfd, 40);

	win_header(mywin -> win, "Swap windows");

	if (nfd > 1)
	{
		int f1_index, f2_index;

		mvwprintw(mywin -> win, 2, 2, "Window 1:");
		wnoutrefresh(mywin -> win);
		f1_index = select_window();
		if (f1_index != -1)
		{
			mvwprintw(mywin -> win, 3, 2, "%s", pi[f1_index].filename);
			mvwprintw(mywin -> win, 4, 2, "Window 2:");
			wnoutrefresh(mywin -> win);

			f2_index = select_window();
			if (f2_index != -1)
			{
				proginfo dummy;
				buffer dummy2;

				buffer_replace_pi_pointers(f1_index, &pi[f1_index], &pi[f2_index]);
				buffer_replace_pi_pointers(f2_index, &pi[f2_index], &pi[f1_index]);

				memmove(&dummy, &pi[f1_index], sizeof(proginfo));
				memmove(&pi[f1_index], &pi[f2_index], sizeof(proginfo));
				memmove(&pi[f2_index], &dummy, sizeof(proginfo));

				memmove(&dummy2, &lb[f1_index], sizeof(buffer));
				memmove(&lb[f1_index], &lb[f2_index], sizeof(buffer));
				memmove(&lb[f2_index], &dummy2, sizeof(buffer));

				swapped = 1;
			}
		}
	}
	else
	{
		mvwprintw(mywin -> win, 2, 2, "Only 1 window!");
		mvwprintw(mywin -> win, max_y/2 - 2, 1, "Press any key to exit this screen");
		mydoupdate(mywin -> win);
		wait_for_keypress();
	}

	delete_popup(mywin);

	return swapped;
}

int delete_window(void)
{
	int deleted = 0;
	int f_index = 0;
	NEWWIN *mywin = create_popup(13 + nfd, 24);

	win_header(mywin -> win, "Delete window");
	wnoutrefresh(mywin -> win);

	/* only popup selectionbox when more then one window */
	if (nfd > 1)
		f_index = select_window();

	/* user did not press q/x? */
	if (f_index != -1)
	{
		char all = 1;

		/* multiple files for this window? then ask if delete all
		 * or just one
		 */
		if (pi[f_index].next)
		{
			escape_print(mywin -> win, 3, 2, "Delete all? (^y^/^n^)");
			mydoupdate(mywin -> win);

			all = ask_yes_no();
		}

		if (all == 1)	/* pressed 'Y' */
		{
			delete_entry(f_index, NULL);
			deleted = 1;
		}
		else if (all == 0) /* pressed 'N' */
		{
			proginfo *cur = select_subwindow(f_index);

			if (cur)
			{
				delete_entry(f_index, cur);
				deleted = 1;
			}
		}
		/* otherwhise: pressed 'Q' */
	}

	delete_popup(mywin);

	return deleted;
}

int hide_window(void)
{
	int f_index = -1;
	NEWWIN *mywin = create_popup(9 + nfd, 40);

	win_header(mywin -> win, "Hide/unhide window");
	wnoutrefresh(mywin -> win);

	f_index = select_window();
	if (f_index != -1)
	{
		pi[f_index].hidden = 1 - pi[f_index].hidden;
	}

	delete_popup(mywin);

	return f_index != -1 ? 1 : 0;
}

void info(void)
{
	NEWWIN *mywin = create_popup(15, 49);
	char *dummy;
	int line = 6;
	struct utsname uinfo;

	mvwprintw(mywin -> win, 1, 2, "-=* MultiTail " VERSION " *=-");
	mvwprintw(mywin -> win, 3, 2, "Written by folkert@vanheusden.com");
	mvwprintw(mywin -> win, 4, 2, "Website: http://www.vanheusden.com/multitail/");

#if defined(__FreeBSD__) || defined(linux) || defined(__OpenBSD__) || defined(__NetBSD__) || defined(__APPLE__) || defined(sun)
	dummy = get_load();
	mvwprintw(mywin -> win, line++, 2, "Current load of system: %s", dummy);
	free(dummy);
#endif
	if (!use_colors)
	{
		mvwprintw(mywin -> win, line++, 2, "Your terminal doesn't support colors");
	}

	if (uname(&uinfo) == -1)
		error_exit("uname() failed");

	line++;
	mvwprintw(mywin -> win, line++, 2, "Running on:");
	mvwprintw(mywin -> win, line++, 2, "%s/%s", uinfo.nodename, uinfo.sysname);
	mvwprintw(mywin -> win, line++, 2, "%s %s", uinfo.release, uinfo.version);
#ifdef _GNU_SOURCE
	mvwprintw(mywin -> win, line++, 2, "%s %s", uinfo.machine, uinfo.domainname);
#else
	mvwprintw(mywin -> win, line++, 2, "%s", uinfo.machine);
#endif

	mvwprintw(mywin -> win, 13, 1, "Press any key to exit this screen");

	mydoupdate(mywin -> win);

	wait_for_keypress();

	delete_popup(mywin);
}

char ask_negate_regexp(WINDOW *win, int line)
{
	escape_print(win, line, 2, "Negate regular expression? (^y^/^n^)");
	mydoupdate(win);

	return ask_yes_no();
}

char ask_colors(WINDOW *win, int line, char cur, char *fieldnr, char **fielddel, int *cscheme_index)
{
	char ok = 0;

	escape_print(win, line, 1, "Colors? (^s^yslog/^m^isc/^f^ield/^n^one,^S^cheme)");

	mydoupdate(win);
	for(;;)
	{
		int c = wait_for_keypress();

		switch(c)
		{
		case 'q':
			return -1;

		case 'S':
			{
				int dummy = selection_box((void *)cschemes, n_cschemes, SEL_CSCHEME);
				if (dummy != -1)
				{
					*cscheme_index = dummy;
					return 'S';
				}
			}

		case 's':
		case 'm':
			return 'm';

		case 'n':
			return 0;

		case 'f':
			{
				int loop, fld_dummy;
				char *dummy, nr[5]; /* xxx\0 and a little :-) */

				sprintf(nr, "%d", *fieldnr);

				mvwprintw(win, line + 2, 1, "Enter field number: ");
				dummy = edit_string(win, line + 3, 42, 39, 1, nr);
				if (dummy)
				{
					fld_dummy = atoi(dummy);
					free(dummy);

					mvwprintw(win, line + 4, 1, "Enter delimiter (eg a space):");
					dummy = edit_string(win, line + 5, 42, 39, 0, *fielddel);

					if (dummy)
					{
						free(*fielddel);
						*fielddel = dummy;
						*fieldnr = fld_dummy;
						ok = 1;
					}
				}

				for(loop=0; loop<4; loop++)
					mvwprintw(win, line + 1 + loop, 1, "                              ");
			}

			if (ok)
				return 'f';
			break;

		case 13:
			if (cur != -1)
				return cur;
		}

		wrong_key();
	}
}

char ask_regex_type(WINDOW *win, int line)
{
	escape_print(win, line+0, 2, "Usage of regexp? (^m^atch, match + ^c^olor,");
	escape_print(win, line+1, 2, "^C^olor, ^B^ell, ^b^ell + colorize, e^x^ecute)");
	mydoupdate(win);

	for(;;)
	{
		int c = wait_for_keypress();

		if (c == 'm' || toupper(c) == 'C' || toupper(c) == 'B' || c == 'x')
			return c;

		if (toupper(c) == 'q')
			return -1;

		wrong_key();
	}
}

void will_not_fit(void)
{
	NEWWIN *mywin = create_popup(3, 40);

	color_on(mywin -> win, 1);
	box(mywin -> win, 0, 0);
	mvwprintw(mywin -> win, 1, 2, "I'm sorry, that won't fit!");
	color_off(mywin -> win, 1);

	mydoupdate(mywin -> win);
	wrong_key();
	wait_for_keypress();

	delete_popup(mywin);
}

int add_window(void)
{
	int added = 0;
	char *fname;
	NEWWIN *mywin = create_popup(11, 45);
	int col, fc;
	proginfo *cur = NULL;
	char ask_add_to;

	if (nfd)
		ask_add_to = 1;
	else
		ask_add_to = 0;

	for(;;)
	{
		int cscheme = -1;
		char field_nr = 0;
		char *field_del = NULL;

		win_header(mywin -> win, "Add window");

		if (ask_add_to)
		{
			int rc;

			ask_add_to = 0;
			mvwprintw(mywin -> win, 2, 2, "Add (merge) to existing window?");
			mydoupdate(mywin -> win);

			rc = ask_yes_no();

			if (rc == 1)	/* pressed 'Y' */
			{
				int index = 0;

				if (nfd > 1)
					index = select_window();

				if (index == -1)
					break;

				cur = &pi[index];
			}
			else if (rc == -1)	/* pressed 'Q' */
			{
				break;
			}
		}

		/* check if this extra window will still fit */
		if (!cur)
		{
			if ((max_y / (nfd + 1)) < 3)
			{
				will_not_fit();
				break;
			}
		}

		escape_print(mywin -> win, 2, 2, "File or command? (^f^/^c^)         ");
		mydoupdate(mywin -> win);
		for(;;)
		{
			fc = toupper(wait_for_keypress());

			if (fc == 'F' || fc == 'C' || fc == 'Q')
				break;

			wrong_key();
		}

		if (fc == 'Q')
			break;
		else if (fc == 'F')
			mvwprintw(mywin -> win, 2, 2, "Enter filename:       ");
		else
			mvwprintw(mywin -> win, 2, 2, "Enter command:        ");

		fname = edit_string(mywin -> win, 3, 41, path_max, 0, NULL);
		if (!fname)
			break;

		col = ask_colors(mywin -> win, 4, -1, &field_nr, &field_del, &cscheme);
		if (col == -1)
			break;

		if (cur == NULL)
		{
			int loop;
			proginfo *newpi = (proginfo *)myrealloc(pi, (nfd + 1) * sizeof(proginfo), "proginfo list");
			lb = (buffer *)myrealloc(lb, (nfd + 1) * sizeof(buffer), "buffer list");

			/* 'lb' has pointers to 'pi'-structures. now since we've realloced the pi-
			 * array, we need to update to pointers to pi in the lb-array
			 */
			for(loop=0; loop<nfd; loop++)
			{
				buffer_replace_pi_pointers(loop, &pi[loop], &newpi[loop]);
			}
			/* now that everything is updated, we can safely forget the previous
			 * pointer
			 */
			pi = newpi;

			/* initialize the new structure */
			memset(&lb[nfd], 0x00, sizeof(buffer));
			lb[nfd].maxnlines = min_n_bufferlines;
			lb[nfd].markset = 'm';

			cur = &pi[nfd];
			nfd++;

			added = 1;
		}
		else
		{
			/* skip to end of chain */
			while (cur -> next)
				cur = cur -> next;

			/* allocate new entry */
			cur -> next = (proginfo *)mymalloc(sizeof(proginfo), "proginfo");
			/* and set pointer to the newly allocated entry */
			cur = cur -> next;
		}
		memset(cur, 0x00, sizeof(proginfo));

		cur -> restart = -1;
		if (fc == 'C')
		{
			char *dummy;

			mvwprintw(mywin -> win, 5, 2, "Repeat command interval: (empty for don't)");
			dummy = edit_string(mywin -> win, 6, 10, 10, 1, NULL);
			if (dummy)
			{
				cur -> restart = atoi(dummy);
				free(dummy);
			}
		}
		cur -> filename = fname;
		cur -> status = cur -> data = NULL;
		cur -> is_command = fc == 'F' ? 0 : 1;
		cur -> retry_open = cur -> follow_filename = 0;
		cur -> next = NULL;
		cur -> first = 1;
		cur -> line_wrap = 'a';
		cur -> win_height = -1;

		cur -> colorize = col;
		cur -> field_nr = field_nr;
		cur -> field_del = field_del;
		cur -> color_scheme = cscheme;

		/* start tail-process */
		if (start_tail(cur, max_y / nfd) == -1)
		{
			mvwprintw(mywin -> win, 7, 2, "error opening file!");
			mydoupdate(mywin -> win);
			wait_for_keypress();
			nfd--;
			break;
		}

		mvwprintw(mywin -> win, 9, 2, "Add another to this new window?");
		mydoupdate(mywin -> win);
		if (ask_yes_no() <= 0)
			break;

		werase(mywin -> win);
		box(mywin -> win, 0, 0);
	}

	delete_popup(mywin);

	return added;
}

void toggle_colors(void)
{
	NEWWIN *mywin = create_popup(9 + nfd, 42);
	int f_index = 0;

	win_header(mywin -> win, "Toggle colors");
	wnoutrefresh(mywin -> win);

	if (nfd > 1)
		f_index = select_window();

	if (f_index != -1)
	{
		char col;
		proginfo *cur = &pi[f_index];
		char *dummy = mystrdup(pi[f_index].filename);
		dummy[min(strlen(dummy), 40)] = 0x00;
		mvwprintw(mywin -> win, 3, 1, dummy);
		free(dummy);

		if (cur -> next)
			cur = select_subwindow(f_index);

		col = ask_colors(mywin -> win, 4, cur -> colorize, &cur -> field_nr, &cur -> field_del, &cur -> color_scheme);
		if (col != -1)
			cur -> colorize = col;
	}

	delete_popup(mywin);
}

char zerotomin(char c)
{
	if (c == 0)
		return 'n';
	if (c == 1)
		return 'y';

	return c;
}

int enter_regexp(void)
{
	int changed = 0;
	NEWWIN *mywin;
	proginfo *cur;
	int f_index = 0;
	int cur_re = 0;

	/* select window */
	if (nfd > 1)
		f_index = select_window();

	if (f_index == -1)	/* user pressed Q/X */
		return 0;

	/* select subwindow */
	cur = &pi[f_index];
	if (cur -> next)
		cur = select_subwindow(f_index);
	if (!cur)
		return 0;

	/* create window */
	mywin = create_popup(24, 60);
	win_header(mywin -> win, "Edit reg.exp.");
	mvwprintw(mywin -> win, 2, 2, "%s", cur -> filename);
	escape_print(mywin -> win, 3, 2, "^A^dd, ^E^dit, ^D^elete, e^X^it");

	for(;;)
	{
		int c, key;
		int loop;
		char buffer[58 + 1];
		buffer[58] = 0x00;

		/* clear */
		memset(buffer, ' ', 58);
		for(loop=4; loop<23; loop++)
			mvwprintw(mywin -> win, loop, 1, buffer);

		/* display them lines */
		for(loop=0; loop<cur -> n_re; loop++)
		{
			strncpy(buffer, (cur -> pre)[loop].regex_str, 34);
			if (loop == cur_re)
				wattron(mywin -> win, A_REVERSE);
			mvwprintw(mywin -> win, 4 + loop, 1, "%c%c %s", 
					zerotomin((cur -> pre)[loop].invert_regex),
					zerotomin((cur -> pre)[loop].use_regex),
					buffer);
			if ((cur -> pre)[loop].use_regex == 'x')
			{
				char dummy[18];
				strncpy(dummy, (cur -> pre)[loop].cmd, min(17, strlen((cur -> pre)[loop].cmd)));
				dummy[17]=0x00;
				mvwprintw(mywin -> win, 4 + loop, 42, dummy);
				wmove(mywin -> win, 4 + loop, 41);
			}
			if (loop == cur_re)
				wattroff(mywin -> win, A_REVERSE);
		}
		mydoupdate(mywin -> win);

		/* wait for key */
		for(;;)
		{
			key = wait_for_keypress();
			c = toupper(key);

			/* convert return to 'E'dit */
			if (key == 13)
				key = c = 'E';

			/* any valid keys? */
			if (c == 'Q' || c == 'X' || c == 'A' || c == 'E' || c == 'D' || key == KEY_DOWN || key == KEY_UP)
				break;

			wrong_key();
		}
		/* exit this screen? */
		if (c == 'Q' || c == 'X')
			break;

		if (key == KEY_UP)
		{
			if (cur_re > 0)
				cur_re--;
			else
				wrong_key();
		}
		else if (key == KEY_DOWN)
		{
			if (cur_re < (cur -> n_re -1))
				cur_re++;
			else
				wrong_key();
		}

		/* add or edit */
		if (c == 'A' || c == 'E')
		{
			int invert_regex, regex_type;
			int rc;
			char *regex_str, *cmd_str = NULL;

			/* max. 10 regular expressions */
			if (c == 'A' && cur -> n_re == 10)
			{
				wrong_key();
				continue;
			}
			if (c == 'E' && cur -> n_re == 0)
			{
				wrong_key();
				continue;
			}

			/* ask new reg exp */
			mvwprintw(mywin -> win, 15, 2, "Edit regular expression:");
			if (c == 'E')
				regex_str = edit_string(mywin -> win, 16, 58, 128, 0, (cur -> pre)[cur_re].regex_str);
			else
				regex_str = edit_string(mywin -> win, 16, 58, 128, 0, NULL);

			/* user did not abort edit? */
			if (regex_str == NULL)
				continue;

			invert_regex = ask_negate_regexp(mywin -> win, 17);
			if (invert_regex == -1)	/* pressed 'Q' */
				continue;

			regex_type = ask_regex_type(mywin -> win, 18);
			if (regex_type == -1)
				continue;

			if (regex_type == 'x')
			{
				mvwprintw(mywin -> win, 20, 2, "Edit command:");
				if (c == 'E')
					cmd_str = edit_string(mywin -> win, 21, 58, 128, 0, (cur -> pre)[cur_re].cmd);
				else
					cmd_str = edit_string(mywin -> win, 21, 58, 128, 0, NULL);

				/* did the user cancel things? */
				if (!cmd_str)
				{
					/* then free the memory for the new regexp
					 * and continue with the menu
					 */
					free(regex_str);
					continue;
				}
			}

			changed = 1;

			/* edit: free previous */
			if (c == 'E')
			{
				regfree(&(cur -> pre)[cur_re].regex);
				free((cur -> pre)[cur_re].regex_str);
			}
			/* add: allocate new */
			else
			{
				cur_re = cur -> n_re++;
				cur -> pre = (re *)myrealloc(cur -> pre, cur -> n_re * sizeof(re), "list of regular expressions");
				memset(&(cur -> pre)[cur_re], 0x00, sizeof(re));
			}

			/* wether to negate this expression or not */
			(cur -> pre)[cur_re].invert_regex = invert_regex;

			/* compile */
			if ((rc = regcomp(&(cur -> pre)[cur_re].regex, regex_str, REG_EXTENDED)))
			{
				char *error = convert_regexp_error(rc, &(cur -> pre)[cur_re].regex);

				mvwprintw(mywin -> win, 17, 2, "Failed to compile regular expression!");

				if (error)
				{
					if (strlen(error) > 58)
					{
						mvwprintw(mywin -> win, 18, 2, "%58", error);
						mvwprintw(mywin -> win, 19, 2, "%s", &error[58]);
					}
					else
					{
						mvwprintw(mywin -> win, 18, 2, "%s", error);
					}
				}

				mvwprintw(mywin -> win, 22, 2, "Press any key...");
				mydoupdate(mywin -> win);
				wait_for_keypress();
				free(regex_str);

				if (c == 'A')
				{
					cur -> n_re--;
					if (cur -> n_re == cur_re)
						cur_re--;
				}

				free(error);
			}
			else
			{
				/* compilation went well, remember everything */
				(cur -> pre)[cur_re].regex_str = regex_str;
				(cur -> pre)[cur_re].use_regex = regex_type;
				(cur -> pre)[cur_re].cmd       = cmd_str;
			}
		}

		/* delete entry */
		if (c == 'D')
		{
			changed = 1;

			/* delete entry */
			free_re(&(cur -> pre)[cur_re]);

			/* update administration */
			if (cur -> n_re == 1)
			{
				free(cur -> pre);
				cur -> pre = NULL;
			}
			else
			{
				int n_to_move = (cur -> n_re - cur_re) - 1;

				if (n_to_move > 0)
					memmove(&(cur -> pre)[cur_re], &(cur -> pre)[cur_re+1], sizeof(re) * n_to_move);
			}
			cur -> n_re--;

			/* move cursor */
			if (cur_re > 0 && cur_re == cur -> n_re)
			{
				cur_re--;
			}
		}
	}

	delete_popup(mywin);

	return changed;
}

int toggle_vertical_split(void)
{
	int changed = 0;

	if (split)
	{
		if ((max_y / nfd) < 3)
		{
			will_not_fit();
		}
		else
		{
			split = 0;
			changed = 1;
		}
	}
	else
	{
		split = 1;
		changed = 1;
	}

	return changed;
}

void show_help(void)
{
	NEWWIN *mywin = create_popup(16, 36);

	escape_print(mywin -> win, 1, 2, "^q^ exit");
	escape_print(mywin -> win, 2, 2, "^r^ redraw the screen");
	escape_print(mywin -> win, 3, 2, "^e^ enter/edit regular expression");
	escape_print(mywin -> win, 4, 2, "^d^ delete window");
	escape_print(mywin -> win, 5, 2, "^a^ add window");
	escape_print(mywin -> win, 6, 2, "^s^ swap windows");
	escape_print(mywin -> win, 7, 2, "^v^ toggle vertical split");
	escape_print(mywin -> win, 8, 2, "^w^ write startupscript");
	escape_print(mywin -> win, 9, 2, "^z^ hide/unhide window");
	escape_print(mywin -> win, 10, 2, "^0^...^9^ set mark");
	escape_print(mywin -> win, 11, 2, "^b^ scrollback ^k^ terminal mode");
	escape_print(mywin -> win, 12, 2, "^t^ statistics ^j^ set window sizes");
	escape_print(mywin -> win, 13, 2, "^i^ info       ^h^ this screen");
	mvwprintw(mywin -> win, 14, 1, "Press any key to exit this screen");
	mydoupdate(mywin -> win);

	wait_for_keypress();

	delete_popup(mywin);
}

void do_set_mark(int f_index, char store_what_lines, int maxnlines)
{
	lb[f_index].markset = store_what_lines;

	if (maxnlines < lb[f_index].maxnlines)
	{
		delete_array(lb[f_index].Blines, lb[f_index].curpos);
		/* delete array already frees the Blines-array */
		lb[f_index].Blines = NULL;

		free(lb[f_index].pi);
		lb[f_index].pi = NULL;

		lb[f_index].curpos = 0;
	}

	lb[f_index].maxnlines = maxnlines;
}

void do_pause(void)
{
	NEWWIN *mywin = create_popup(3, 8);

	color_on(mywin -> win, 1);
	box(mywin -> win, 0, 0);
	mvwprintw(mywin -> win, 1, 1, "Paused");
	color_off(mywin -> win, 1);

	wmove(mywin -> win, 1, 2);
	mydoupdate(mywin -> win);

	wait_for_keypress();

	delete_popup(mywin);
}

void set_mark(void)
{
	char winchoice, whatlines, maxnlines = 0;
	NEWWIN *mywin = create_popup(13, 40);

	win_header(mywin -> win, "Set mark");

	if (nfd > 1)
	{
		escape_print(mywin -> win, 3, 2, "Set on ^a^ll or ^o^ne");
		mvwprintw(mywin -> win, 4, 2, "window(s)?");
		mydoupdate(mywin -> win);

		for(;;)
		{
			winchoice = toupper(wait_for_keypress());

			if (winchoice == 'Q' || winchoice == 'X')
			{
				winchoice = 'Q';
				break;
			}

			if (winchoice == 'A' || winchoice == 'O')
				break;

			wrong_key();
		}
	}
	else
	{
		winchoice = 'A';
	}

	/* ask wether to store all lines or just the ones matchine to the
	 * regular expression (if any)
	 */
	if (winchoice != 'Q')
	{
		escape_print(mywin -> win, 5, 2, "Store ^a^ll or ^m^atching regular");
		mvwprintw(mywin -> win, 6, 2, "expressions (if any)?");
		mydoupdate(mywin -> win);
		for(;;)
		{
			whatlines = toupper(wait_for_keypress());
			if (whatlines == 'Q' || whatlines == 'X')
			{
				winchoice = 'Q';
				break;
			}

			if (whatlines == 'A' || whatlines == 'M')
				break;

			wrong_key();
		}
	}

	/* get number of lines to store */
	if (winchoice != 'Q')
	{
		char *dummy;

		mvwprintw(mywin -> win, 7, 2, "Store how many lines? (0 for all)");

		dummy = edit_string(mywin -> win, 8, 40, 7, 1, NULL);

		if (!dummy)
		{
			winchoice = 'Q';
		}
		else
		{
			maxnlines = atoi(dummy);

			free(dummy);
		}
	}

	/* do set mark */
	if (winchoice != 'Q')
	{
		if (winchoice == 'A')
		{
			int loop;

			for(loop=0; loop<nfd; loop++)
			{
				do_set_mark(loop, whatlines == 'A' ? 'a' : 'm', maxnlines);
			}
		}
		else
		{
			int window = select_window();

			if (window != -1)
			{
				do_set_mark(window, whatlines == 'A' ? 'a' : 'm', maxnlines);
			}
		}
	}

	delete_popup(mywin);
}

void generate_diff(int winnr, proginfo *cur)
{
	int loop;

	/* calculate & print difference */
	if (cur -> nprev == 0 && cur -> ncur > 0)
	{
		for(loop=0; loop<cur -> ncur; loop++)
		{
			print_and_buffer(winnr, cur, cur -> bcur[loop]);
		}
	}
	else if (cur -> nprev > 0 && cur -> ncur > 0)
	{
		int curprev = 0;

		for(loop=0; loop<cur -> ncur; loop++)
		{
			int indexfound = -1, loop2;

			for(loop2=curprev; loop2<cur -> nprev; loop2++)
			{
				if (strcmp((cur -> bprev)[loop2], (cur -> bcur)[loop]) == 0)
				{
					indexfound = loop2;
					break;
				}
			}

			if (indexfound == -1)
			{
				/* output cur[loop] */
				print_and_buffer(winnr, cur, (cur -> bcur)[loop]);
			}
			else
			{
				curprev = indexfound + 1;
			}
		}
	}
	wnoutrefresh(pi[winnr].data);

	/* free up previous */
	delete_array(cur -> bprev, cur -> nprev);

	/* remember current */
	cur -> bprev = cur -> bcur;
	cur -> nprev = cur -> ncur;
	cur -> bcur = NULL;
	cur -> ncur = 0;

	/* update statusline */
	update_statusline(pi[winnr].status, winnr, cur);
}

char close_window(int winnr, proginfo *cur)
{
	if (winnr == terminal_index)
		terminal_index = -1;

	/* make sure it is really gone */
	stop_process(cur -> pid);

	/* wait for the last remainder of the exited process to go away,
	 * otherwhise we'll find zombies on our way
	 */
	if (waitpid(cur -> pid, NULL, WNOHANG | WUNTRACED) == -1)
	{
		if (errno != ECHILD)
			error_exit("waitpid failed\n");
	}

	/* restart window? */
	if (cur -> restart >= 0)
	{
		int initial_n_lines_tail;

		/* close old fds */
		if (close(cur -> fd) == -1) error_exit("closing read filedescriptor failed\n");
		if (close(cur -> wfd) == -1) error_exit("closing write filedescriptor failed\n");

		/* do diff */
		if (cur -> do_diff)
			generate_diff(winnr, cur);

		if (cur -> initial_n_lines_tail == -1)
			initial_n_lines_tail = max_y / (nfd + 1);
		else
			initial_n_lines_tail = cur -> initial_n_lines_tail;

                /* restart tail-process */
                if (start_tail(cur, initial_n_lines_tail) != -1)
			return -1;

		/* ...and if that fails, go on with the closing */
	}

	if (warn_closed)
	{
		char buffer[128];
		NEWWIN *mywin;

		sprintf(buffer, " Window %d closed ", winnr);
		mywin = create_popup(3, strlen(buffer) + 4);

		color_on(mywin -> win, 1);
		box(mywin -> win, 0, 0);
		mvwprintw(mywin -> win, 1, 2, "%s", buffer);
		color_off(mywin -> win, 1);

		wmove(mywin -> win, 1, 2);
		mydoupdate(mywin -> win);

		wait_for_keypress();

		delete_popup(mywin);
	}

	/* file no longer exists (or so) */
	return delete_entry(winnr, cur);
}

void usage(void)
{
	int index = 0;

	printf(version_str, VERSION);
	printf("\n\nmultitail [-cs|-Cs|-c-] [-i] inputfile [-i anotherinputfile] [...]\n");
	printf("-l	parameter is a command to be executed\n");
	printf("-L	see -l but add to the previous window\n");
	printf("-r interval\n");
	printf("	restart the command when it exited after `interval' seconds\n");
	printf("-R interval\n");
	printf("	same as -r, only with this one only the difference is displayed\n");
	printf("-i	the following parameter is a filename (in case it starts with a dash)\n");
	printf("-n	initial number of lines to tail\n");
	printf("-I	see -i but add to the previous window\n");
	printf("-s	vertical split screen\n");
	printf("-sw x	at what column to split the screen\n");
	printf("-wh x	height of window\n");
	printf("-f	follow the following filename, not the descriptor\n");
	printf("--retry	keep trying to open the following file if it is inaccessible\n");
	printf("-e	use regular expression on following file\n");
	printf("-ec	use regular expression but display the matches inverted on following file\n");
	printf("-eC	use regexp, display everything but matches inverted on following file\n");
	printf("-ex	execute command ('-ex regexp command') when matches\n");
	printf("-E	use regular expression on following files\n");
	printf("-Ec	use regular expression but display the matches inverted on following files\n");
	printf("-EC	use regexp, display everything but matches inverted on following files\n");
	printf("-v      invert next regular expression\n");
	printf("-cs	colorize current with syslog-scheme\n");
	printf("-c	colorize current\n");
	printf("-cs scheme\n");
	printf("	use colorscheme 'scheme' (as defined in multitail.conf)\n");
	printf("-Cs	colorize all following files with syslog-scheme\n");
	printf("-C	colorize all following files\n");
	printf("-Cf/-cf field delimiter\n");
	printf("        colorize next/all file(s) depending on the given field number. fields\n");
	printf("        are delimited with the given field-delimiter\n");
	printf("-c-	do NOT colorize the following file\n");
	printf("-C-	do NOT colorize the following files\n");
	printf("-d	do NOT update the status-line\n");
	printf("-D	do not display a status-line at all\n");
	printf("-z	do not \"window closed\" windows\n");
	printf("-w	do not use colors\n");
	printf("-m nlines\n");
	printf("	immediately set mark\n");
	printf("-u	set update interval (for slow links)\n");
	printf("-H interval\n");
	printf("	visual heart-beat (usefull when you want to keep your session alive)\n");
	printf("-p x [y]\n");
	printf("	set linewrap (l=left/a=all/r=right/s=syslog,S=syslog w/o procname, o=offset -> 'y')\n");
	printf("-b n	set TAB-width\n");
	printf("-x str	switch on the xtermtitlebar stuff\n");
	printf("-h	this help\n");
	printf("\n");
	printf("You can have multiple regular expressions per file/command. Be warned: if\n");
	printf("you define multiple and one of them is specified with '-E' (=for every\n");
	printf("following file), _all_ of the current regular expressions are for all\n");
	printf("following files!\n");
	printf("\n");

	do
	{
		printf("%s\n", keys[index++]);
	} while (keys[index]);
}

void write_escape_str(FILE *fh, char *string)
{
	int loop, len = strlen(string);

	fprintf(fh, "\"");
	for(loop=0; loop<len; loop++)
	{
		if (string[loop] == '\"')
			fprintf(fh, "\\");
		fprintf(fh, "%c", string[loop]);
	}
	fprintf(fh, "\"");
}

void write_script(void)
{
	char *fname;
	NEWWIN *mywin = create_popup(12, 42);

	win_header(mywin -> win, "Write script");

	mvwprintw(mywin -> win, 2, 2, "Enter filename:");

	fname = edit_string(mywin -> win, 3, 41, path_max, 0, NULL);
	if (fname)
	{
		FILE *fh = fopen(fname, "w");
		if (fh)
		{
			int loop;

			fprintf(fh, "#!/bin/sh\n\n");

			fprintf(fh, "multitail");

			if (heartbeat)
				fprintf(fh, " -H %d", heartbeat);

			for(loop=0; loop<nfd; loop++)
			{
				proginfo *cur = &pi[loop];
				char first = 1;

				do
				{
					if (cur -> line_wrap != 'a')
					{
						fprintf(fh, " -p %c", cur -> line_wrap);
					}

					if (cur -> win_height != -1)
						fprintf(fh, " -wh %d", cur -> win_height);

#if 0
					if (cur -> regex_str)
					{
						if (cur -> invert_regex)
							fprintf(fh, " -v");
						if (cur -> use_regex == 'm')
							fprintf(fh, " -e ");
						else if (cur -> use_regex)
							fprintf(fh, " -e%c ", cur -> use_regex);
						write_escape_str(fh, cur -> regex_str);
					}
#endif

					if (cur -> colorize)
					{
						if (cur -> colorize == 's')
							fprintf(fh, " -cs");
						else if (cur -> colorize == 'm')
							fprintf(fh, " -c");
						else if (cur -> colorize == 'f')
						{
							fprintf(fh, " -cf %d ", cur -> field_nr);
							write_escape_str(fh, cur -> field_del);
						}
					}

					if (first)
					{
						first = 0;

						if (cur -> is_command)
						{
							if (cur -> restart != -1)
								fprintf(fh, " -%c %d", cur -> do_diff?'R':'r', cur -> restart);

							fprintf(fh, " -l ");
						}
						else if ((cur -> filename)[0] == '-')
							fprintf(fh, " -i ");
						else
							fprintf(fh, " ");
					}
					else
					{
						if (cur -> is_command)
							fprintf(fh, " -L ");
						else
							fprintf(fh, " -I ");
					}

					write_escape_str(fh, cur -> filename);

					cur = cur -> next;
				}
				while(cur);
			}
			if (split)
				fprintf(fh, " -s");

			if (fchmod(fileno(fh), S_IRUSR | S_IWUSR | S_IXUSR | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH) == -1)
			{
				error_exit("error setting mode-bits on file");
			}

			fprintf(fh, "\n");

			fclose(fh);
		}
		else
		{
			mvwprintw(mywin -> win, 4, 2, "Error creating file!");
			mydoupdate(mywin -> win);

			wait_for_keypress();
		}

		free(fname);
	}

	delete_popup(mywin);
}

int find_colorscheme(char *name)
{
	int loop;

	for(loop=0; loop<n_cschemes; loop++)
	{
		if (strcasecmp(cschemes[loop].name, name) == 0)
			return loop;
	}

	return -1;
}

/* returns the default color scheme or -1 if none */
int load_config(char *file)
{
	int fd = -1;
	char *cur_scheme = NULL;
	char *defaultcscheme = NULL;
	int defaultcindex = -1;
	int linenr = 0;

	/* given file */
	fd = open(file, O_RDONLY);
	if (fd == -1)
	{
		if (errno == ENOENT)	/* file does not exist, not an error */
			return -1;

		return -2;
	}

	for(;;)
	{
		char *dummy, *par, *cmd;
		char *cmdin = read_line_fd(fd);

		cmd = cmdin;
		if (!cmdin)
			break;

		linenr++;

		/* remove spaces at the beginning of the line */
		while (isspace(*cmd)) cmd++;

		/* see if there's anything in this line anway */
		if (strlen(cmd) == 0)
			continue;

		/* skip comments */
		if (cmd[0] == '#' || cmd[0] == ';')
			continue;

		/* lines are in the format of command:parameter */
		dummy = strchr(cmd, ':');
		if (!dummy)
			error_exit("Malformed configline found: %s (command delimiter (:) missing)\n", cmd);
		*dummy = 0x00;
		par = dummy + 1; 

		if (strcasecmp(cmd, "defaultcscheme") == 0)	/* default colorscheme */
		{
			defaultcscheme = mystrdup(par);
		}
		else if (strcasecmp(cmd, "colorscheme") == 0)	/* name for colorscheme */
		{
			if (cur_scheme)
				free(cur_scheme);

			cur_scheme = mystrdup(par);
		}
		else if (strcasecmp(cmd, "cs_re") == 0)		/* ColorScheme_RegularExpression */
		{
			char *re;
			int sn, rc;

			dummy = strchr(par, ':');		/* format: cs_re:color:regular expression */
			if (!dummy)
				error_exit("cs_re-entry malformed: color or regular expression missing (%s:%s)\n", cmd, par);
			*dummy = 0x00;
			re = dummy + 1;

			/* find colorscheme */
			sn = find_colorscheme(cur_scheme);
			if (sn == -1)
			{
				sn = n_cschemes++;
				cschemes = (color_scheme *)myrealloc(cschemes, n_cschemes * sizeof(color_scheme), "list of colorschemes");

				cschemes[sn].name = mystrdup(cur_scheme);
				cschemes[sn].n = 0;
				cschemes[sn].color = NULL;
				cschemes[sn].regex = NULL;
			}

			/* add to list */
			cschemes[sn].color = (int *)myrealloc(cschemes[sn].color, (cschemes[sn].n + 1) * sizeof(int), "color for regexp");
			cschemes[sn].regex = (regex_t *)myrealloc(cschemes[sn].regex, (cschemes[sn].n + 1) * sizeof(regex_t), "regexp for colorscheme");
			if (strcasecmp(par, "red") == 0)
				cschemes[sn].color[cschemes[sn].n] = MY_RED;
			else if (strcasecmp(par, "green") == 0)
				cschemes[sn].color[cschemes[sn].n] = MY_GREEN;
			else if (strcasecmp(par, "yellow") == 0)
				cschemes[sn].color[cschemes[sn].n] = MY_YELLOW;
			else if (strcasecmp(par, "blue") == 0)
				cschemes[sn].color[cschemes[sn].n] = MY_BLUE;
			else if (strcasecmp(par, "magenta") == 0)
				cschemes[sn].color[cschemes[sn].n] = MY_MAGENTA;
			else if (strcasecmp(par, "cyan") == 0)
				cschemes[sn].color[cschemes[sn].n] = MY_CYAN;
			else if (strcasecmp(par, "white") == 0)
				cschemes[sn].color[cschemes[sn].n] = MY_WHITE;
			else
				error_exit("%s is not a known color\n", par);

			/* compile regular expression */
			if ((rc = regcomp(&cschemes[sn].regex[cschemes[sn].n], re, REG_EXTENDED)))
				error_exit("While loading configurationfile: failed to compile regular expression '%s': %s\n", re, convert_regexp_error(rc, &cschemes[sn].regex[cschemes[sn].n]));

			cschemes[sn].n++;
		}
		else if (strcasecmp(cmd, "scheme") == 0)	/* default colorscheme for file */
		{
			char *re;
			int cs, rc;

			dummy = strchr(par, ':');		/* format: cs_re:color:regular expression */
			if (!dummy)
				error_exit("scheme-entry malformed: schemename or regular expression missing (%s:%s)\n", cmd, par);
			*dummy = 0x00;
			re = dummy + 1;

			/* find colorscheme */
			cs = find_colorscheme(par);
			if (cs == -1)
				error_exit("colorscheme %s is unknown: you should first define a colorscheme before using it\n", par);

			/* add to list */
			spf = (scheme_per_file *)myrealloc(spf, sizeof(scheme_per_file) * (n_scheme_per_file + 1), "scheme_per_file struct");
			/* store internal scheme-nr */
			spf[n_scheme_per_file].scheme_nr = cs;

			/* compile & store regular expression */
			rc = regcomp(&spf[n_scheme_per_file].regex, re, REG_EXTENDED);
			if (rc != 0)
				error_exit("failed to compile regular expression '%s': %s\n", re, convert_regexp_error(rc, &spf[n_scheme_per_file].regex));

			n_scheme_per_file++;
		}
		else if (strcasecmp(cmd, "check_mail") == 0)	/* how often to check for mail */
		{
			check_for_mail = atoi(par);
		}
		else if (strcasecmp(cmd, "tab_stop") == 0)	/* length of a tab */
		{
			tab_width = atoi(par);
		}
		else if (strcasecmp(cmd, "bind") == 0)		/* program to execute when user presses a key */
		{
			char *proc;

			dummy = strchr(par, ':');		/* format: cs_re:color:regular expression */
			if (!dummy) error_exit("bind-entry malformed: key or binding missing (%s:%s)\n", cmd, par);
			*dummy = 0x00;
			proc = dummy + 1;

			if (strlen(par) > 2) error_exit("bind-entry malformed: unknown keyselection (%s:%s)\n", cmd, par);

			keybindings = (keybinding *)myrealloc(keybindings, sizeof(keybinding) * (n_keybindings + 1), "keybindings array");

			if (par[0] == '^')
				keybindings[n_keybindings].key = toupper(par[1]) - 'A' + 1;
			else
				keybindings[n_keybindings].key = par[0];

			keybindings[n_keybindings].command = strdup(proc);
			n_keybindings++;
		}
		else if (strcasecmp(cmd, "titlebar") == 0)	/* do things with xterm title bar */
		{
			set_title = mystrdup(par);
		}
		else
			error_exit("Configurationparameter %s:%s is unknown (file: %s, line: %d)\n", cmd, par, file, linenr);

		free(cmd);
	}

	close(fd);

	if (defaultcscheme)
	{
		defaultcindex = find_colorscheme(defaultcscheme);

		if (defaultcindex == -1)
			error_exit("Default colorscheme '%s' is not defined! Check multitail.conf\n", defaultcscheme);
	}

	return defaultcindex;
}

int do_load_configfile(char *config_file)
{
	/* load configurationfile (if any) */
	int default_color_scheme = load_config("/etc/multitail.conf");

	if (config_file)
		default_color_scheme = load_config(config_file);
	else
	{
		char *path = mymalloc(path_max + 1, "path");
		char *dummy = getenv("HOME");

		if (dummy)
		{
			snprintf(path, path_max, "%s/.multitailrc", dummy);

			default_color_scheme = load_config(path);
		}

		free(path);
	}

	return default_color_scheme;
}

void get_terminal_type(void)
{
	char *dummy = getenv("TERM");

	if (dummy && strstr(dummy, "xterm") != NULL)
	{
		term_type = TERM_XTERM;
	}
}

void statistics(void)
{
	NEWWIN *mywin = create_popup(nfd + 6, 65);
	int loop;

	escape_print(mywin -> win, 1, 2, "^Window^");
	escape_print(mywin -> win, 1, 36, "^Average^");
	escape_print(mywin -> win, 1, 44, "^std dev^");
	escape_print(mywin -> win, 1, 54, "^Tendency^");

	for(loop=0; loop<nfd; loop++)
	{
		int y = 2 + loop;
		char *copy = mystrdup(pi[loop].filename);
		char *dummy = copy;
		double avg, dev;

		if (strlen(copy) > 30)
			dummy += strlen(copy) - 30;

		mvwprintw(mywin -> win, y, 2, "%02d %s", loop, dummy);

		if (pi[loop].n_events == 0)
		{
			mvwprintw(mywin -> win, y, 36, "Not yet available");
		}
		else
		{
			avg = pi[loop].med / (double)pi[loop].n_events;
			mvwprintw(mywin -> win, y, 36, "%.2f", avg);

			dev = sqrt((pi[loop].dev / (double)pi[loop].n_events) - pow(avg, 2.0));
			mvwprintw(mywin -> win, y, 44, "%.2f", dev);

			if (pi[loop].n_events > 1)
				mvwprintw(mywin -> win, y, 54, "%.2f", pi[loop].total_deltat / ((double)pi[loop].n_events - 1));
		}
	}

	mvwprintw(mywin -> win, nfd+3, 2, "Run-time: %.2f hours", difftime(time(NULL), mt_started) / 3600.0);

	escape_print(mywin -> win, nfd+4, 2, "Press ^r^ to reset counters, ^q^ to exit");

	mydoupdate(mywin -> win);

	for(;;)
	{
		int c = toupper(wait_for_keypress());

		if (c == 'R')
		{
			for(loop=0; loop<nfd; loop++)
			{
				pi[loop].med = pi[loop].dev = 0;
				pi[loop].n_events = 0;
				pi[loop].prev_deltat = pi[loop].total_deltat = 0;
				pi[loop].lastevent = 0;
			}

			break;
		}

		if (c == 'Q')
			break;

		wrong_key();
	}

	delete_popup(mywin);
}

void redraw_statuslines(void)
{
	int loop;

	/* update statuslines */
	for(loop=0; loop<nfd; loop++)
		update_statusline(pi[loop].status, loop, &pi[loop]);
}

void store_statistics(int index, time_t now)
{
	if (pi[index].lastevent)
	{
		double cur_deltat = difftime(now, pi[index].lastevent);

		if (pi[index].n_events == 1)
		{
			pi[index].total_deltat += (cur_deltat - pi[index].prev_deltat);
			pi[index].prev_deltat = cur_deltat;
		}

		pi[index].med += cur_deltat;
		pi[index].dev += pow(cur_deltat, 2.0);
		pi[index].n_events++;
	}

	pi[index].lastevent = now;
}

void do_heartbeat(void)
{
	static int hb_x = 0, hb_y = 0, hb_dx = 1, hb_dy = 1;
	int delta = myrand(2) + 1;

	move(hb_y, hb_x);

	hb_y += hb_dy;
	hb_x += hb_dx;

	if (hb_y >= max_y - 1)
	{
		hb_y = max_y - 1;
		hb_dy = -delta;
	}
	else if (hb_y < 0)
	{
		hb_y = 0;
		hb_dy = delta;
	}
	if (hb_x >= max_x - 1)
	{
		hb_x = max_x - 1;
		hb_dx = -delta;
	}
	else if (hb_x < 0)
	{
		hb_x = 0;
		hb_dx = delta;
	}
}

int set_window_sizes(void)
{
	int resized = 0;
	NEWWIN *mywin = create_popup(7, 35);

	escape_print(mywin -> win, 1, 2, "^w^ set width of windows");
	escape_print(mywin -> win, 2, 2, "^h^ set height of a window");
	escape_print(mywin -> win, 3, 2, "^q^ quit this menu");
	mydoupdate(mywin -> win);

	for(;;)
	{
		int key;

		wmove(mywin -> win, 4, 2);

		key = toupper(wait_for_keypress());
		if (key == 'Q')
			break;

		if (key == 'W')
		{
			char oldval[5];
			char *str;

			if (vertical_split == -1)
				sprintf(oldval, "%d", max_x / 2);
			else
				snprintf(oldval, 5, "%d", vertical_split);

			mvwprintw(mywin -> win, 4, 2, "Enter column:");
			mydoupdate(mywin -> win);
			str = edit_string(mywin -> win, 5, 20, 3, 1, oldval);
			if (str)
			{
				vertical_split = atoi(str);
				free(str);
				resized = 1;
				break;
			}
		}
		else if (key == 'H')
		{
			int window = select_window();
			if (window != -1)
			{
				char *str;
				char oldval[5];
				char *spaces = "                                 ";

				snprintf(oldval, 5, "%d", pi[window].win_height);

				mvwprintw(mywin -> win, 4, 2, "Enter number of lines:");
				mvwprintw(mywin -> win, 5, 1, spaces);
				mydoupdate(mywin -> win);
				str = edit_string(mywin -> win, 5, 20, 3, 1, oldval);
				mvwprintw(mywin -> win, 4, 1, spaces);
				mvwprintw(mywin -> win, 5, 1, spaces);
				if (str)
				{
					int dummy = atoi(str);

					if (dummy >= -1 && dummy < (max_y - ((nfd - 1) * 3)) && dummy != 0 && dummy != 1)
					{
						pi[window].win_height = dummy;
						resized = 1;
						break;
					}

					mvwprintw(mywin -> win, 5, 2, "Invalid height!");
				}

				mydoupdate(mywin -> win);
			}
		}
		else
		{
			wrong_key();
		}
	}

	delete_popup(mywin);

	return resized;
}

void terminal_mode(void)
{
	int index = 0;

	if (nfd > 1)
	{
		NEWWIN *mywin = create_popup(6 + nfd, 40);
		win_header(mywin -> win, "Enter terminal mode");
		mydoupdate(mywin -> win);

		index = select_window();

		delete_popup(mywin);
	}

	if (index != -1)
	{
		terminal_index = index;
		prev_term_char = -1;

		redraw_statuslines();
	}
}

int exec_bind(char key)
{
	int executed = 0;
	int loop;

	for(loop=0; loop<n_keybindings; loop++)
	{
		if (keybindings[loop].key == key)
		{
			if (execute_program(keybindings[loop].command, 0) == 0)
				executed = 1;

			break;
		}
	}

	return executed;
}

void draw_gui_window_header(proginfo *last_changed_window)
{
	char *in = mystrdup(set_title);
	int pos = 0;
	int str_len = strlen(in);
	struct utsname uinfo;

	for(;pos < str_len;)
	{
		if (in[pos] == '%')
		{
			char *new_str = NULL;
			char *repl_str = NULL;

			switch(in[pos+1])
			{
			case 'f':	/* changed file */
				repl_str = mystrdup(last_changed_window?last_changed_window -> filename:"");
				break;

			case 'h':	/* hostname */
				{
					if (uname(&uinfo) == -1)
						error_exit("uname() failed");

					repl_str = mystrdup(uinfo.nodename);
				}
				break;

			case 'l':	/* current load */
				{
					char *dummy;
					repl_str = get_load();
					dummy = strchr(repl_str, ' ');
					if (dummy) *dummy = 0x00;
				}
				break;

			case 'm':
				repl_str = mystrdup(mail?"New mail!":"");
				break;

			case 'u':	/* username */
				{
					char *dummy = getenv("USER");
					if (dummy)
						repl_str = mystrdup(dummy);
					else
						new_str = mystrdup("???");
				}
				break;

			case 't':	/* atime */
				if (last_changed_window)
				{
					char *dt = ctime(&last_changed_window -> lastevent);
					char *dummy = strchr(dt, '\n');
					if (dummy) *dummy = 0x00;
					repl_str = mystrdup(dt);
				}
				else
				{
					repl_str = mystrdup("");
				}
				break;

			case '%':	/* % */
				repl_str = mystrdup("%");
				break;
			}

			if (repl_str)
			{
				new_str = replace_string(in, pos, pos + 1, repl_str);
				free(in);
				in = new_str;
				str_len = strlen(in);
				pos += strlen(repl_str);
				free(repl_str);
			}
			else
			{
				pos += 2;
			}
		}
		else
		{
			pos++;
		}
	}

	gui_window_header(in);

	free(in);
}

int main(int argc, char *argv[])
{
	int loop;
	char curcolor = 'n', allcolor = 'n';
	char regex_mode = 0;
	char fi = 0;
	char *fd = NULL;
	char follow_filename = 0, retry = 0, invert_regex = 0;
	char *expr = NULL, exprall = 0;
	time_t lastupdate = 0;
	int update_interval = 0;
	char *nsubwindows = NULL;
	int maxlines = 0;
	char setmark = 0, allmark = 0;
	int restart = -1;
	char do_diff = 0;
	time_t hb_time;
	char line_wrap = 'a';
	int line_wrap_offset = 0;
	re *pre = NULL;
	int n_re = 0;
	char config_loaded = 0;
	char *config_file = NULL;
	int cur_color_scheme = -1;
	int default_color_scheme = -1;
	char *mail_spool_file = getenv("MAIL");
	struct stat msf_info;
	off_t msf_prev_size = 0;
	time_t msf_last_check = 0;
	int window_height = -1;
	int initial_n_lines_tail = -1;

	time(&mt_started);

	if (SIG_ERR == signal(SIGTERM, do_exit)) error_exit("setting of signal failed");

	/* determine PATH_MAX */
	path_max = pathconf("/", _PC_PATH_MAX);
	if (path_max == -1)
	{
		if (errno) error_exit("pathconf(_PC_PATH_MAX) failed\n");

		path_max = 255;
	}
	else
	{
		path_max++; /* since its relative to root */
	}

	/* calc. buffer length (at least a complete terminal screen) */
	initscr();
	min_n_bufferlines = max(MIN_N_BUFFERLINES, LINES);
#ifdef N_CURSES
	if (has_colors())
	{
		use_colors = 1;
	}
#endif

	/* verify size of terminal window */
	if (LINES < 24 || COLS < 78)
		error_exit("Your terminal(-window) is %dx%d. That is too small for MultiTail (at least 78x24 is required).\n", COLS, LINES);

	endwin();

	/* init random thing */
	time(&hb_time);
	srand((unsigned int)hb_time);

	/* parse commandline */
	for(loop=1; loop<argc; loop++)
	{
		if (strcmp(argv[loop], "-V") == 0)
		{
			printf(version_str, VERSION);
			printf("\n\nThank you for using MultiTail.\n");
			printf("If you have any suggestion on how I can improve this program,\n");
			printf("do not hesitate to contact me at folkert@vanheusden.com\n");
			printf("Website is available at: http://www.vanheusden.com/multitail/\n\n\n");

			return 0;
		}
		else if (strcmp(argv[loop], "-x") == 0)
		{
			set_title = mystrdup(argv[++loop]);
		}
		else if (strcmp(argv[loop], "-p") == 0)
		{
			line_wrap = argv[++loop][0];
			if (line_wrap == 'o')
				line_wrap_offset = atoi(argv[++loop]);
		}
		else if (strcmp(argv[loop], "--retry") == 0)
		{
			retry = 1;
		}
		else if (strcmp(argv[loop], "-n") == 0)
		{
			initial_n_lines_tail = atoi(argv[++loop]);
		}
		else if (strcmp(argv[loop], "-b") == 0)
		{
			tab_width = atoi(argv[++loop]);
		}
		else if (strcmp(argv[loop], "-u") == 0)
		{
			update_interval = atoi(argv[++loop]);
		}
		else if (strcasecmp(argv[loop], "-r") == 0)
		{
			if (argv[loop][1] == 'R')
				do_diff = 1;

			restart = atoi(argv[++loop]);
		}
		else if (strcmp(argv[loop], "-s") == 0)
		{
			split = 1;
		}
		else if (strcmp(argv[loop], "-sw") == 0)
		{
			vertical_split = atoi(argv[++loop]);
		}
		else if (strcmp(argv[loop], "-wh") == 0)
		{
			window_height = atoi(argv[++loop]);
		}
		else if (argv[loop][0] == '-' && toupper(argv[loop][1]) == 'E')
		{
			int rc;
			char *cmd = NULL;

			/* -e => only for this file, -E => for all following files */
			if (argv[loop][1] == 'E')
				exprall = 1;

			/* c/C/m define colors, x says: execute */
			if (toupper(argv[loop][2]) == 'C')
				regex_mode = argv[loop][2];
			else if (toupper(argv[loop][2]) == 'B')
				regex_mode = argv[loop][2];
			else if (argv[loop][2] == 'm' || argv[loop][2] == 0x00)
				regex_mode = 'm';	/* m = match, only print when matches */
			else if (argv[loop][2] == 'x')
				regex_mode = 'x';	/* x = execute */
			else
				error_exit("%s is an unknown switch", argv[loop]);

			/* get expression */
			expr = argv[++loop];

			/* and if there's anything to execute, get commandline */
			if (regex_mode == 'x')
				cmd = argv[++loop];

			/* compile & set expression */
			/* allocate new structure */
			pre = (re *)myrealloc(pre, sizeof(re) * (n_re + 1), "list of regular expressions");

			/* initialize structure */
			memset(&pre[n_re], 0x00, sizeof(re));

			/* compile */
			if ((rc = regcomp(&pre[n_re].regex, expr, REG_EXTENDED)))
				error_exit("failed to compile regular expression \"%s\": %s\n", expr, convert_regexp_error(rc, &pre[n_re].regex));

			/* remember string for later edit */
			pre[n_re].regex_str = mystrdup(expr);

			/* set flag on current file */
			pre[n_re].use_regex = regex_mode;
			if (exprall == 0)
				regex_mode = 0;

			/* wether to invert the reg exp or not */
			pre[n_re].invert_regex = invert_regex;
			if (exprall == 0)
				invert_regex = 0;

			/* what to execute (if...) */
			if (cmd)
				pre[n_re].cmd = mystrdup(cmd);
			else
				pre[n_re].cmd = NULL;

			n_re++;

		}
		else if (strcmp(argv[loop], "-v") == 0)
		{
			invert_regex = 1;
		}
		else if (argv[loop][0] == '-' && toupper(argv[loop][1]) == 'C')
		{
			char dummy = -1, doall = 0;

			if (argv[loop][1] == 'C')
				doall = 1;

			if (argv[loop][2] == 's')	/* syslog-file coloring? */
				dummy = 's';
			else if (argv[loop][2] == 'S')	/* use colorscheme */
			{
				char *cur_cscheme = argv[++loop];

				if (config_loaded == 0)
				{
					/* load configurationfile (if any) */
					default_color_scheme = do_load_configfile(config_file);

					config_loaded = 1;
				}

				cur_color_scheme = find_colorscheme(cur_cscheme);
				if (cur_color_scheme == -1)
					error_exit("Color scheme %s not found! Check your configfile (%s)\n", cur_cscheme, config_file);
				dummy = 'S';
			}
			else if (argv[loop][2] == '-')	/* do not color current */
				dummy = 'n';
			else if (argv[loop][2] == 'f')	/* select field for coloring */
			{
				dummy = 'f';
				fi = atoi(argv[++loop]);
				fd = argv[++loop];
			}
			else				/* use complete line for coloring */
				dummy = 'm';

			if (doall)
				allcolor = dummy;
			else
				curcolor = dummy;
		}
		else if (strcmp(argv[loop], "-f") == 0)
		{
			follow_filename = 1;
		}
		else if (strcmp(argv[loop], "-w") == 0)
		{
			use_colors = 0;
		}
		else if (strcmp(argv[loop], "-m") == 0 || strcmp(argv[loop], "-M") == 0)
		{
			setmark = 1;
			if (argv[loop][1] == 'M')
				allmark = 1;
			maxlines = atoi(argv[++loop]);
		}
		else if (strcasecmp(argv[loop], "-i") == 0 || argv[loop][0] != '-' ||
			 strcasecmp(argv[loop], "-l") == 0)
		{
			struct stat buf;
			char *dummy;
			char is_cmd = 0;
			char is_sub = 0;
			proginfo *cur;

			if (config_loaded == 0)
			{
				/* load configurationfile (if any) */
				default_color_scheme = do_load_configfile(config_file);
				if (default_color_scheme == -2)
					error_exit("could not load configfile %s", config_file);

				config_loaded = 1;
			}

			if (strcasecmp(argv[loop], "-l") == 0)
			{
				is_cmd = 1;
			}

			if (strcmp(argv[loop], "-L") == 0 || strcmp(argv[loop], "-I") == 0)
			{
				is_sub = 1;
			}

			if (argv[loop][0] == '-')
			{
				loop++;
			}

			dummy = argv[loop];

			if (is_sub == 1 && nfd > 0)
			{
				cur = &pi[nfd - 1];

				while(cur -> next)
				{
					cur = cur -> next;
				}

				cur -> next = (proginfo *)mymalloc(sizeof(proginfo), "proginfo");

				cur = cur -> next;

				nsubwindows[nfd-1]++;
			}
			else
			{
				pi = (proginfo *)myrealloc(pi, (nfd + 1) * sizeof(proginfo), "proginfo");

				lb = (buffer *)myrealloc(lb, (nfd + 1) * sizeof(buffer), "buffers");

				nsubwindows = (char *)myrealloc(nsubwindows, (nfd + 1) * sizeof(char), "subwindows");
				nsubwindows[nfd] = 1;
				memset(&lb[nfd], 0x00, sizeof(buffer));
				if (setmark)
				{
					lb[nfd].maxnlines = maxlines;

					if (!allmark)
						setmark = 0;
				}
				else
				{
					lb[nfd].maxnlines = min_n_bufferlines;
				}
				lb[nfd].markset = 'm';

				cur = &pi[nfd];
				nfd++;
			}
			memset(cur, 0x00, sizeof(proginfo));

			/* see if file exists */
			if (is_cmd == 0 && retry == 0 && stat(dummy, &buf) == -1)
			{
				error_exit("error opening %s (%d)\n", dummy, errno);
			}

			/* init. struct. for this file */
			cur -> filename = mystrdup(dummy);
			cur -> is_command = is_cmd;

			/* initial number of lines to tail */
			cur -> initial_n_lines_tail = initial_n_lines_tail;
			initial_n_lines_tail = -1;

			/* default window height */
			cur -> win_height = window_height;
			window_height = -1;

			/* store regular expression(s) */
			cur -> n_re = n_re;
			cur -> pre = pre;
			if (exprall == 0)
			{
				n_re = 0;
				pre = NULL;
			}

			/* hide this window? */
			cur -> hidden = 0;

			/* line wrap */
			cur -> line_wrap = line_wrap;
			line_wrap = 'a';
			cur -> line_wrap_offset = line_wrap_offset;

			cur -> retry_open = retry;
			cur -> follow_filename = follow_filename;

			/* 'watch' functionality configuration (more or less) */
			cur -> restart = restart;
			restart = -1; /* invalidate for next parameter */
			cur -> first = 1;
			cur -> do_diff = do_diff;
			do_diff = 0;

			if (cur_color_scheme == -1)
				cur_color_scheme = default_color_scheme;

			cur -> color_scheme = cur_color_scheme;
			if (curcolor != 'n' && curcolor != 0)
			{
				cur -> colorize = curcolor;
			}
			else
			{
				if (allcolor == 'n' && curcolor == 'n' && default_color_scheme != -1)
				{
					cur -> colorize = 'S';
				}
				else if (allcolor == 'n' || curcolor == 'n')
				{
					cur -> colorize = 0;
				}
				else
				{
					cur -> colorize = allcolor;
				}
			}
			cur -> field_nr = fi;
			if (fd)
			{
				cur -> field_del = mystrdup(fd);
			}
			else
			{
				cur -> field_del = NULL;
			}

			curcolor = 'n';
			retry = follow_filename = 0;
		}
		else if (strcmp(argv[loop], "-d") == 0)
		{
			mode_statusline = 0;
		}
		else if (strcmp(argv[loop], "-D") == 0)
		{
			mode_statusline = -1;
		}
		else if (strcmp(argv[loop], "-z") == 0)
		{
			warn_closed = 0;
		}
		else if (strcmp(argv[loop], "-H") == 0)
		{
			heartbeat = atoi(argv[++loop]);
		}
		else
		{
			if (strcmp(argv[loop], "-h") != 0)
			{
				fprintf(stderr, "unknown parameter '%s'\n", argv[loop]);
			}

			usage();

			return 1;
		}
	}

	if (config_loaded == 0)
	{
		do_load_configfile(config_file);
		if (default_color_scheme == -2)
			error_exit("could not load configfile %s", config_file);

		config_loaded = 1;
	}

	/* init list of pids to watch for exit */
	memset(children_list, 0x00, sizeof(children_list));

	/* initialise mail-stuff */
	if (check_for_mail > 0 && mail_spool_file != NULL)
	{
		if (stat(mail_spool_file, &msf_info) == -1)
		{
			check_for_mail = 0;

			printf("Could not determine size of file '%s' (which is supposed to be ", mail_spool_file);
			printf("your mailfile): mail-check is disabled.\n");
			printf("You can prevent this message by adding the line 'check_mail:0' ");
			printf("in /etc/multitail.conf or in .multitailrc in your home-directory.\n\n");
			printf("Press enter to continue...");
			fflush(NULL);
			getchar();
		}
		else
		{
			msf_prev_size = msf_info.st_size;

			msf_last_check = time(NULL);
		}
	}

	/* start curses library */
	init_curses();

	/* get terminal type */
	get_terminal_type();

	/* start processes */
	for(loop=0; loop<nfd; loop++)
	{
		proginfo *cur = &pi[loop];
		int cur_win_size = lb[loop].maxnlines / nsubwindows[loop];

		do
		{
			int initial_n_lines_tail;

			/* find colorscheme if not selected */
			if (cur -> color_scheme == -1 && cur -> is_command == 0)
			{
				int loop2;

				for(loop2=0; loop2<n_scheme_per_file; loop2++)
				{
					int rc;

					if ((rc = regexec(&spf[loop2].regex, cur -> filename, 0, NULL, 0)) == 0)
					{
						cur -> color_scheme = spf[loop2].scheme_nr;
						cur -> colorize = 'S';
						break;
					}
					/* we should inform the user of any errors while executing
					 * the regexp! */
					else
					{
						char *error = convert_regexp_error(rc, &spf[loop2].regex);

						if (error)
							error_exit("%s", error);
					}
				}
			}

			if (cur -> initial_n_lines_tail == -1)
				initial_n_lines_tail = cur_win_size == 0 ? min_n_bufferlines : cur_win_size;
			else
				initial_n_lines_tail = cur -> initial_n_lines_tail;

			/* start the tail process for this file/command */
			if (start_tail(cur, initial_n_lines_tail) == -1)
				error_exit("failed to start process! (%s)\n", cur -> filename);

			cur = cur -> next;
		} while(cur);
	}
	free(nsubwindows);

	/* set signalhandler for terminal resize */
	if (SIG_ERR ==  signal(SIGWINCH, do_resize)) error_exit("signal failed");

	/* create windows */
	do_refresh = 2;

	for(;;)
	{
		int last_fd = 0, rc;
		fd_set rfds;
		struct timeval tv;
		time_t now;
		proginfo *last_changed_window = NULL;
		char prev_mail_status = mail;

		time(&now);

		tv.tv_sec = 0;
		tv.tv_usec = 100000;

		FD_ZERO(&rfds);

		/* add stdin to fd-set: needed for monitoring key-presses */
		FD_SET(0, &rfds);

		/* add fd's of pipes to fd-set */
		for(loop=0; loop<nfd; loop++)
		{
			proginfo *cur = &pi[loop];

			do
			{
				if (cur -> fd != -1)
				{
					FD_SET(cur -> fd, &rfds);
					last_fd = max(last_fd, cur -> fd);
				}

				cur = cur -> next;
			}
			while(cur);
		}

		/* heartbeat? */
		if (heartbeat)
		{
			if (difftime(now, hb_time) >= heartbeat)
			{
				do_heartbeat();

				hb_time = now;

				if (do_refresh != 2) do_refresh = 1;
			}
		}

		/* check for mail */
		if (check_for_mail > 0 && mail_spool_file != NULL && difftime(now, msf_last_check) >= check_for_mail)
		{
			/* get current filesize */
			if (stat(mail_spool_file, &msf_info) == -1)
				error_exit("Error doing stat() on file %s.", mail_spool_file);

			/* filesize changed? */
			if (msf_info.st_size != msf_prev_size)
			{
				/* file became bigger: new mail
				 * if it became less, the file changed because
				 * mail was deleted or so
				 */
				if (msf_info.st_size > msf_prev_size)
				{
					mail = 1;

					redraw_statuslines();

					if (do_refresh != 2) do_refresh = 1;
				}

				msf_prev_size = msf_info.st_size;

				msf_last_check = now;
			}
		}

		/* update screen? */
		if (do_refresh)
		{
			if (do_refresh == 2)
			{
				/* no windows? display list of keys */
				if (nfd == 0)
				{
					int index = 0;

					werase(stdscr);

					wprintw(stdscr, version_str, VERSION);
					wprintw(stdscr, "\n\n");

					do
					{
						wprintw(stdscr, "%s\n", keys[index++]);
					} while (keys[index]);

					mydoupdate(stdscr);
				}
				else
				{
					create_windows();
				}

				do_refresh = 1;
			}

			if (difftime(now, lastupdate) >= update_interval)
			{
				do_refresh = 0;
				lastupdate = now;
				doupdate();
			}
		}

		/* wait for any data or key press */
		if ((rc = select(last_fd + 1, &rfds, NULL, NULL, &tv)) == -1)
		{
			if (errno != EINTR)
				error_exit("select returned an error! (%d)\n", errno);

			continue;
		}

		if (terminal_changed)
		{
			terminal_changed = 0;

#ifdef N_CURSES
			if (ERR == resizeterm(max_y, max_x)) error_exit("problem resizeing terminal\n");
#endif

			touchwin(stdscr);
			endwin();
			doupdate();

			create_windows();
		}

		/* see if any processes have exited (processes started
		 * by matchin regular expressions)
		 */
		for(loop=0; loop<n_children; loop++)
		{
			pid_t rc = waitpid(children_list[loop], NULL, WNOHANG | WUNTRACED);

			/* error while waiting? */
			if (rc == -1 && errno != ECHILD)
	                        error_exit("waitpid for pid %d failed\n", children_list[loop]);

			if (rc != 0)
			{
				int n_pids_to_move = (n_children - loop) - 1;

				if (n_pids_to_move > 0)
					memmove(&children_list[loop], &children_list[loop + 1], sizeof(pid_t) * n_pids_to_move);

				n_children--;
				/* also decrease pointer! as we move the other pids up! */
				loop--;
			}
		}
			

		/* any fd's set? */
		if (rc == 0)
		{
			/* verify if any of the processes exited */
			for(loop=0; loop<nfd; loop++)
			{
				char deleted_entry_in_array = 0;
				proginfo *cur = &pi[loop];

				do
				{
					/* see if the process exited */
					pid_t rc = waitpid(cur -> pid, NULL, WNOHANG | WUNTRACED);

					/* error while waiting? */
					if (rc == -1 && errno != ECHILD)
			                        error_exit("waitpid for pid %d failed\n", cur -> pid);

					/* did it exit? */
					if (rc != 0) /* equal to: rc == cur -> pid */
					{
						deleted_entry_in_array = close_window(loop, cur);
						/* is an entry deleted? (>=0) or restarted? (==-1) */
						if (deleted_entry_in_array >= 0)
						{
							do_refresh = 2;
							break;
						}
						else if (cur -> do_diff && do_refresh == 0)
						{
							if (do_refresh != 2) do_refresh = 1;
						}
					}

					cur = cur -> next;
				}
				while(cur);

				if (deleted_entry_in_array > 0)
					break;
			}

			/* and since no fd's were set, redo the select()-call */
			continue;
		}

		/* any key pressed? */
		if (FD_ISSET(0, &rfds))
		{
			int c = wait_for_keypress();

			if (terminal_index == -1)
			{
				int uc = toupper(c);

				do_refresh = 1;

				if (mail)
				{
					mail = 0;
					redraw_statuslines();
				}

				if (uc == 'Q' || uc == 'X')
				{
					break;
				}
				else if (uc == 'A')
				{
					if (add_window())
						do_refresh = 2;
					continue;
				}
				else if (uc == 'H' || uc == '?')
				{
					show_help();
					continue;
				}
				else if (uc == 'I')
				{
					info();
					continue;
				}
				else if (uc == 'T')
				{
					statistics();
					continue;
				}
				else if (uc == 'J')
				{
					if (set_window_sizes())
						do_refresh = 2;
					continue;
				}

				if (nfd == 0)
				{
					wrong_key();
					do_refresh = 0;
					continue;
				}

#ifdef N_CURSES
				if (c == KEY_RESIZE || uc == 'R')
#else
				if (uc == 'R')
#endif
				{
					mail = 0;
					do_refresh = 2;
				}
				else if (uc == 'E' || uc == '\\')
				{
					if (enter_regexp())
						do_refresh = 2;
				}
					else if (uc == 'D')
				{
					if (delete_window())
						do_refresh = 2;
					continue;
				}
				else if (uc == 'V')
				{
					if (toggle_vertical_split())
						do_refresh = 2;
				}
				else if (uc == 'C' && use_colors)
				{
					toggle_colors();
				}
				else if (uc == 'S')
				{
					if (swap_window())
						do_refresh = 2;
					continue;
				}
				else if (uc == 'Z')
				{
					if (hide_window())
						do_refresh = 2;
				}
				else if (uc == 'W')
				{
					write_script();
				}
				else if (uc == 'M')
				{
					set_mark();
				}
				else if (uc == 'N')
				{
					delete_mark();
				}
				else if (uc == 'B')
				{
					scrollback();
				}
				else if (uc == 'P')
				{
					do_pause();
				}
				else if (uc >= '0' && uc <= '9')
				{
					int index = uc - '0';
	
					if (index < nfd)
					{
						print_and_buffer(index, NULL, "");
	
						wnoutrefresh(pi[index].data);
					}
					else
					{
						wrong_key();
					}
				}
				else if (uc == 'K')
				{
					terminal_mode();
				}
				else if (uc == 26)	/* ^Z */
				{
					/* we're back from suspend */
					redraw_statuslines();
				}
				else
				{
					if (exec_bind(c))
						do_refresh = 1;
					else
					{
						wrong_key();
						do_refresh = 0;
					}
				}
			}
			else
			{
				char byte = (char)c;

				if (prev_term_char == 1)	/* ^A */
				{
					if (c == 'd')
					{
						terminal_index = -1;
						redraw_statuslines();
						mydoupdate(stdscr);
					}
					else
					{
						WRITE(pi[terminal_index].wfd, &prev_term_char, 1);
						WRITE(pi[terminal_index].wfd, &byte, 1);
					}
				}
				else
				{
					if (byte != 1)	/* ^A */
						WRITE(pi[terminal_index].wfd, &byte, 1);
				}

				prev_term_char = byte;
			}
		}

		/* go through all fds */
		for(loop=0; loop<nfd; loop++)
		{
			char deleted_entry_in_array = 0;
			proginfo *cur = &pi[loop];
			WINDOW *data = pi[loop].data, *status = pi[loop].status;

			do
			{
				if (cur -> fd == -1)
					continue;

				if (FD_ISSET(cur -> fd, &rfds))
				{
					char buffer[65536];
					int nbytes;

					nbytes = read(cur -> fd, buffer, 65535);

					/* read error or EOF? */
					if ((nbytes == -1 && errno != EINTR) || nbytes == 0)
					{
						deleted_entry_in_array = close_window(loop, cur);

						if (deleted_entry_in_array >= 0)
						{
							do_refresh = 2;
							break;
						}
						else if (cur -> do_diff && do_refresh == 0)
						{
							if (do_refresh != 2) do_refresh = 1;
						}
					}
					else if (nbytes != -1)	/* if nbytes == -1 it must be an interrupt while READ */
					{
						char *pnt = buffer;
						buffer[nbytes] = 0x00;

						/* remember this window as it might be displayed in the GUI
						 * terminal window
						 */
						last_changed_window = cur;

						/* gen stats */
						store_statistics(loop, now);

						/* display lines */
						for(;*pnt != 0x00;)
						{
							char org = 0;
							char *end = strchr(pnt, '\n');
							if (end)
								*end = 0x00;

							/* is this the output of a program which we should diff and such? */
							if (cur -> do_diff)
							{
								store_for_diff(cur, pnt);
							}
							else /* no, just output */
							{
								if (do_refresh == 0)
									do_refresh = 1;	/* after update interval, update screen */

								/* output new text */
								print_and_buffer(loop, cur, pnt);
							}

							if (end)
								pnt = end + 1;
							else
								break;
						}

						if (do_refresh)
						{
							/* display statusline? */
							update_statusline(status, loop, cur);

							wnoutrefresh(data);
						}
					}
				}

				cur = cur -> next;
			}
			while(cur);

			if (deleted_entry_in_array > 0)
				break;
		}

		/* any window changed? then we may want to update the terminal window header */
		if ((last_changed_window != NULL || mail != prev_mail_status) && set_title != NULL)
		{
			prev_mail_status = mail;

			draw_gui_window_header(last_changed_window);
		}
	}

	endwin();

	/* kill tail processes */
	do_exit(0);

	return 0;
}
