/* File rw.c
 *
 * Due to some problems with the 2.6 kernels, I had to find a way in
 * order to make reading and writing from/to the device file /dev/lp0 and
 * /dev/usb/lp0 working in the same way.
 * With the 2.6 kernel, the read() call will no more be interrupted by
 * an alarm timer. Read on /dev/lp0 is a blocking call, read on 
 * /dev/usb/lp0 will never block.
 * Both device has different behaviour and the kernel people seem not to
 * be able to understood the problem and fix it.
 *
 * Solution: If we open the device file with the O_NDELAY flag set,
 * both interface will return immediatly with the value -1 and errno set
 * to EAGAIN if nothing is to be read. 
 *
 * Writing may be working for now without changes.
 */
#ifndef MACOS
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/time.h>
#include <sys/poll.h>

/* this variable is set before a read or write and reset
 * after the read is done or by the signal handler 
 */
extern int doRw;

int devRead(int fd, unsigned char *buf, size_t sz)
{
   int rd = 0;
   int ms;

   errno = 0;
   while(doRw && rd != -1 )
   {
      /* wait a little bit */
      poll(NULL, 0, 10);
      rd = read(fd, buf, sz);
      if ( rd > 0 )
      {
         return rd;
      }
   }
   /* at this point we are probably interrupted */
   if ( rd == 0 )
   {
      errno = EINTR;
   }
   return -1;
}
#endif
