#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include "doc_bootstub.h"

unsigned char buf[0x3000];
unsigned char spl_sig[16] =
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x55, 0x55,
	  0x84, 0xa8, 0xac, 0xa0, 0x30, 0x30, 0x30, 0x30 };
unsigned char cmdline_sig[16] =
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x55, 0x55,
	  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff };
unsigned char kern_sig[16] =
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdb, 0xb1,
	  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff };

void writeblocks(int outfd, unsigned char *buf, int len, unsigned char *oob) {
	int ret;

	while (len) {
		if (len < 512) {
			memset(buf + len, 0xff, 512 - len);
			len = 512;
		}
		ret = write(outfd, buf, 512);
		if (ret < 0) {
			perror("write output file");
			exit(1);
		}
		if (ret < 512) {
			fprintf(stderr, "short write of output file (%d bytes < 512)\n", ret);
			exit(1);
		}
		ret = write(outfd, oob, 16);
		if (ret < 0) {
			perror("write output file");
			exit(1);
		}
		if (ret < 16) {
			fprintf(stderr, "short write of output file (%d bytes < 16)\n", ret);
			exit(1);
		}
		len -= 512;
		buf += 512;
	}
}

int main(int argc, char **argv)
{
	int len, i, stubfd, imgfd, outfd, biosfd;
	int ret;
	char bios_extension;
	unsigned char checksum=0;
	int imglen;
	int total_sects, setup_sects, kernel_sects;

	//memset(buf, 0xff, sizeof(buf));

	if (argc < 4) {
		fprintf(stderr, "Usage: makespl <stubfile> <kernelfile> <outfile> [bios_extensionfile]\n");
		exit(1);
	}

	if (argv[4])
		bios_extension=1;
	else
		bios_extension=0;
	
	stubfd = open(argv[1], O_RDONLY);
	if (stubfd < 0) {
		perror("open stub file");
		exit(1);
	}

	imgfd = open(argv[2], O_RDONLY);
	if (imgfd < 0) {
		perror("open kernel image file");
		exit(1);
	}

	// get image length
	imglen = lseek(imgfd, 0, SEEK_END);
	lseek(imgfd, 0, SEEK_SET);
	total_sects = (imglen + 511) >> 9;

	outfd = open(argv[3], O_WRONLY | O_CREAT | O_TRUNC, 0664);
	if (outfd < 0) {
		perror("open output file");
		exit(1);
	}

	if (bios_extension) {
		biosfd = open(argv[4], O_WRONLY | O_CREAT | O_TRUNC, 0664);
		if (biosfd < 0) {
			perror("open BIOS extension output file");
			exit(1);
		}
	}

	/* Read the bootstub */
	len = read(stubfd, buf, 512);
	if (len < 0) {
		perror("read from stub file");
		exit(1);
	}
	if (len != 512) {
		fprintf(stderr, "Unexpected EOF in stub file\n");
		exit(1);
	}
	close(stubfd);

	if (isatty(0))
		fprintf(stderr, "Enter commandline: ");
	len = strlen(fgets(buf + 512, 256, stdin));
	len--;
	if (buf[512+len] == '\n')
		buf[512+len] = 0;
	fprintf(stderr, "Commandline is \"%s\"\n", buf + 512);

	/* Read the kernel */
	len = read(imgfd, buf + 1024, sizeof(buf) - 1024);
	if (len < 0) {
		perror("read from kernel");
		exit(1);
	}

	setup_sects = buf[1024 + SETUP_SECTS_LOCATION];
	setup_sects++;
	kernel_sects = total_sects - setup_sects;

	fprintf(stderr, "2 bootstub sectors, %d real-mode sectors, %d kernel sectors\n",
		setup_sects, kernel_sects);

	*((unsigned short *) &buf[CHECKSUM_LOCATION+2]) = setup_sects;
	*((unsigned short *) &buf[CHECKSUM_LOCATION+4]) = kernel_sects;

	/* Calculate the csum */
	buf[CHECKSUM_LOCATION] = 0;

	if (bios_extension) {
		for (i = 0; i < 1024; i++)
			checksum += buf[i];

		/* Set the slack byte to fix the csum */
		buf[CHECKSUM_LOCATION] = 0 - checksum;
	} else {
		for (i = 0; i < sizeof(buf); i++)
			checksum += buf[i];

		/* Set the slack byte to fix the csum */
		buf[CHECKSUM_LOCATION] = 0x55 - checksum;
	}

	if (bios_extension) {
		/* Write the bootstub and commandline to one file */
		ret = write(biosfd, buf, 1024);
		if (ret < 0) {
			perror("write BIOS extension output file");
			exit(1);
		}
	} else {
		/* Write the bootstub page */
		writeblocks(outfd, buf, 512, spl_sig);
		/* Write the commandline page */
		writeblocks(outfd, buf + 512, 512, cmdline_sig);
	}
	/* Write the rest of the first buffer */
	writeblocks(outfd, buf + 1024, sizeof(buf) - 1024, kern_sig);

	/* Now chuck out the rest of the kernel */

	while (1) {
		len = read(imgfd, buf, sizeof(buf));
		if (len < 0) {
			perror("read from kernel");
			exit(1);
		}
		if (len == 0)
			break;
		
		writeblocks(outfd, buf, len, kern_sig);
	}
	return 0;
}
