/* Time-stamp: <2007-10-05 02:01:30 poser> */

/*
 * Copyright (C) 1993-2007 William J. Poser.
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of version 3 of the GNU General Public License
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "config.h"
#include "compdefs.h"
#include <stdlib.h>
#ifdef HAVE_STDINT_H
#include <stdint.h>
#endif
#include <stdio.h>
#include <string.h>
#ifdef HAVE_LOCALE_H
#include <locale.h>
#endif
#ifdef LOCALE_GETTEXT
#include <libintl.h>
#define _(x) gettext(x)
#else
#define _(x) (x)
#endif
#include <time.h>
#include <wchar.h>
#include <errno.h>
#ifndef USEUTF8PROC
#include <unicode/umachine.h>
#include <unicode/unorm.h>
#else
#include <utf8proc.h>
#include "unorm.h"
#endif
#include <tre/regex.h>
#include "key.h"
#ifdef HAVE_UNINUM_UNICODE_H
#include <uninum/unicode.h>
#else
#include "unicode.h"
#endif
#include <uninum/nsdefs.h>
#include <uninum/uninum.h>
#include "dstr.h"
#include "exitcode.h"
#include "retcodes.h"
#include "ex_codes.h"

#ifdef HAVE_LONGDOUBLE
#define DOUBLE (long double)
#else
#define DOUBLE double
#endif

void
DumpExclusions(unsigned char *exct,long exctsize,FILE *fp)
{
  unsigned long i;
  unsigned char flags;
	
  fprintf(fp,_("Characters excluded:\n"));
  if(exctsize > 1) printf("\n");
  for(i = 0L;i < exctsize; i++){
    if(exct[i]){
      flags = exct[i];
      fprintf(fp,"\t0x%04lX",i);
      if(flags & EX_INITIAL) fprintf(fp,_("\tinitial"));
      if(flags & EX_MEDIAL) fprintf(fp,_("\tmedial"));
      if(flags & EX_FINAL) fprintf(fp,_("\tfinal"));
      fprintf(fp,"\n");
    }
  }
}


/*
 * Overwrite a textual key with itself, skipping characters specified
 * in the exclusion table.
 */

int
Exclude(wchar_t *str, int len, wchar_t *exct)
{
  int i;
  int cnt;

  cnt = 0;
  if(!(exct[str[0]] & EX_INITIAL)){
    str[cnt++] = str[0];
  }
  for(i = 1; i < len-1; i++){
    if(exct[str[i]] & EX_MEDIAL) continue;
    else str[cnt++] = str[i];
  }
  if(!(exct[str[len-1]] & EX_FINAL)){
    str[cnt++] = str[len-1];
  }
  str[cnt] = ((wchar_t) 0); /* Null terminate */
  return(cnt);
}

void
KeyCopy(wchar_t *t,wchar_t *s,int len)
{
  register int i;
  for(i = 0;i < len;i++) *t++ = *s++;
}

void
RevKeyCopy(wchar_t *t,wchar_t *s,int len)
{
  register int i;
	
  s += (len-2);
	
  for(i = 0;i < len-1;i++) *t++ = *s--;
  *t= (wchar_t) 0;
}

void
CopyCommandLine(FILE *fp,int ac, char **av)
{
  int i;
	
  for(i=0; i < ac-1; i++){
    fprintf(fp,"%s ",av[i]);
  }
  fprintf(fp,"%s\n",av[ac-1]);
}


int mywcscasecmp(wchar_t *a, wchar_t *b) {
  struct dstr *ad;
  struct dstr *bd;
  int Result;
  extern void UnicodeFoldCase(struct dstr *);

  ad = MakeDynamicString(a);
  bd = MakeDynamicString(b);
  UnicodeFoldCase(ad);
  UnicodeFoldCase(bd);
  Result = wcscmp(ad->s,bd->s);
  FreeDynamicString(ad);
  FreeDynamicString(bd);
  return(Result);
}

DOUBLE GetMonthKey(wchar_t *mn,wchar_t **mtbl) {
  int i;
  for(i=0;i<MONTHNAMES;i++) {
#if defined HAVE_WCSCASECMP && defined __USE_GNU
    if(wcscasecmp(mn,mtbl[i]) ==0) return((DOUBLE) ((i/2)+1));
#else
    if(mywcscasecmp(mn,mtbl[i]) ==0) return((DOUBLE) ((i/2)+1));
#endif
  }
  return((DOUBLE)(-1));
}


/* Given a date string, write the equivalent double into key. */

int
GetDateKey(wchar_t *field, DOUBLE *key, struct ymdinfo *ymd,int KeyNumber)
{
  long int num[3];
  int month;
  int day;
  int year;
  wchar_t *sepptr1;
  wchar_t *sepptr2;
  wchar_t *endptr;
  wchar_t *sub[3];
#ifdef USE_UNINUM
  union ns_rval UninumValue;
#endif

  extern int GetMonthKeyFromMap (wchar_t *, int);
  extern int errno;
  extern struct keyinfo **KeyInfo;

  errno = 0;

  if(ymd->m < 0) {		/* Day of year format */
    sepptr1 = wcschr(field,ymd->sep1);
    if(sepptr1 == NULL) return(ERROR);
    *sepptr1 = L'\0';
    sub[0] = field;
    sub[1] = sepptr1+1;

#ifdef USE_UNINUM
    StringToInt(&UninumValue,(UTF32 *) (sub[ymd->d]),NS_TYPE_ULONG,NS_ALL);
    if (uninum_err != 0) return(ERROR);
    day = (int) UninumValue.u;

    StringToInt(&UninumValue,(UTF32 *) (sub[ymd->y]),NS_TYPE_ULONG,NS_ALL);
    if (uninum_err != 0) return(ERROR);
    year = (int) UninumValue.u;
#else
    day = wcstol(sub[ymd->d],NULL,10);
    if((errno == EINVAL)||(errno == ERANGE)) return (ERROR);
    year = wcstol(sub[ymd->y],NULL,10);
    if((errno == EINVAL)||(errno == ERANGE)) return (ERROR);
#endif
    *key = (366 * (DOUBLE) year) + (DOUBLE) day;
    return(SUCCESS);
  }

  /* Full ymd format */
  sepptr1 = wcschr(field,ymd->sep1);
  sepptr2 = wcschr(sepptr1+1,ymd->sep2);
  if( (sepptr1 == NULL) || (sepptr2 == NULL) ) {
    return(ERROR);
  }
  *sepptr1 = L'\0';
  *sepptr2 = L'\0';
  sub[0] = field;
  sub[1] = sepptr1+1;
  sub[2] = sepptr2+1;

  if(KeyInfo[KeyNumber]->MapTable == NULL) 
    month = (int) GetMonthKey(sub[ymd->m],KeyInfo[KeyNumber]->MonthNames);
  else month =  GetMonthKeyFromMap(sub[ymd->m],KeyNumber);
#ifdef USE_UNINUM
  if (month < 0) {
    StringToInt(&UninumValue,(UTF32 *) (sub[ymd->m]),NS_TYPE_ULONG,NS_ALL);
    if (uninum_err != 0) return(ERROR);
    month = (int) UninumValue.u;
  }

  StringToInt(&UninumValue,(UTF32 *) (sub[ymd->d]),NS_TYPE_ULONG,NS_ALL);
  if (uninum_err != 0) return(ERROR);
  day = (int) UninumValue.u;

  StringToInt(&UninumValue,(UTF32 *) (sub[ymd->y]),NS_TYPE_ULONG,NS_ALL);
  if (uninum_err != 0) return(ERROR);
  year = (int) UninumValue.u;
#else
  day = wcstol(sub[ymd->d],NULL,10);
  if((errno == EINVAL)||(errno == ERANGE)) return (ERROR);
  month = wcstol(sub[ymd->m],NULL,10);
  if((errno == EINVAL)||(errno == ERANGE)) return (ERROR);
  year = wcstol(sub[ymd->y],NULL,10);
  if((errno == EINVAL)||(errno == ERANGE)) return (ERROR);
#endif

  if(month < 0 || day < 0) return (ERROR);
  if(month > 12) return (ERROR);
  if(day > 31) return (ERROR);
  *key = (DOUBLE) ((month * 31) + day) + (DOUBLE) (366 *year);
  return(SUCCESS);
}

/* Given a time string, write the equivalent double into key. */

int
GetTimeKey(wchar_t *field, DOUBLE *key)
{
  int hours;
  int minutes;
  int iseconds;
  double seconds = 0.0;
  UTF8 *cfield;
  int FieldsFound;
  long TZOffset;		/* Number of seconds to add to normalize to UTC */
  short err;

  extern UTF8 * ws2u8(wchar_t *);
  extern long ExtractTimeZone(UTF8 *, short *);

  cfield = ws2u8(field);
  TZOffset = ExtractTimeZone(cfield,&err);
  if(err) return(ERROR);

  /* HH:MM:SS(.SSS) */
  FieldsFound = sscanf((char *)cfield,"%d:%d:%lf",&hours,&minutes,&seconds);
  if(FieldsFound == 3) goto validate;

  /* HH:MM.SS */
  FieldsFound = sscanf((char *)cfield,"%d:%d.%d",&hours,&minutes,&iseconds);
  if(FieldsFound == 3) {
    seconds = (double) iseconds;
    goto validate;
  }

  /* HH:MM */
  FieldsFound = sscanf((char *)cfield,"%d:%d",&hours,&minutes);
  if(FieldsFound == 2) {
    seconds = 0.0;
    goto validate;
  }

  free(cfield);  
  return(ERROR); 

validate:
  free(cfield);
  if(hours < 0 || minutes < 0 || seconds < 0) return (ERROR);
  if(hours > 24 || minutes > 59 || seconds >= 60) return (ERROR);
  *key = (DOUBLE) (((((DOUBLE) hours * 60) + (DOUBLE) minutes) * 60) + (DOUBLE) seconds + (DOUBLE) TZOffset);
  return(SUCCESS);
}


/* 
 * This produces a value that is suitable for ordering dates and times.
 * It is NOT accurate for computing differences between dates and times
 * because the conversion of dates to days is not exact. For simplicity's
 * sake it treats years as containing 366 days and months as all containing
 * 31 days.
 */
#define DTITYPE unsigned long
#define SECONDSINDAY (60*60*24)
int
GetISO8601Key(wchar_t *field, DOUBLE *key)
{
  unsigned int year = 0;
  unsigned int month =0;
  unsigned int day = 0;
  unsigned int hour = 0;
  unsigned int minute = 0;
  unsigned int second = 0;

  long TZOffset;		/* Number of seconds to add to normalize to UTC */
  DTITYPE Days;
  DTITYPE DaySeconds;
  int f1, f2;
  char *cfield;
  char * Tloc;
  short err;

  extern UTF8 * ws2u8(wchar_t *);
  extern long ExtractTimeZone(UTF8 *, short *);
  extern char *StripTimeDateSeparators(char *, char *);


  cfield = (char *) ws2u8(field);
  Tloc = strchr(cfield,'T');
  if(!Tloc) return(ERROR);
  Tloc = StripTimeDateSeparators(cfield,Tloc);
  TZOffset = ExtractTimeZone(((UTF8 *)Tloc)+1,&err);
  if(err) return(ERROR);

  f1=sscanf(cfield,"%4u%2u%2uT%2u%2u.%2u", &year,&month,&day,&hour,&minute,&second);
  if(f1 != 6) {
    f2=sscanf(cfield,"%4u%2u%2uT%2u%2u", &year,&month,&day,&hour,&minute);
    if(f2 != 5) return(ERROR);
  }
  if(month < 0 || day < 0) return (ERROR);
  if(month > 12) return (ERROR);
  if(day > 31) return (ERROR);
  if(hour < 0 || minute < 0 || second < 0) return (ERROR);
  if(hour > 24) return(ERROR);
  if(minute > 60 || second > 60) return (ERROR);

  Days = (DTITYPE) ( (366 * (DTITYPE) year) + (31 * month) + day);
  DaySeconds = (DTITYPE) (((((DTITYPE) hour * 60) + (DTITYPE) minute) * 60) + (DTITYPE) second);
  *key = (DOUBLE) ( ( (DOUBLE)Days * (DOUBLE)SECONDSINDAY) + (DOUBLE)DaySeconds + (DOUBLE)TZOffset);
  return(SUCCESS);
}


/* Given an angle string, write the equivalent double into key. */

int
GetAngleKey(wchar_t *field, DOUBLE *key)
{
  int degrees;
  int minutes;
  double seconds;
  UTF8 *cfield;
  int FieldsFound;
  short NegativeP = 0;
  DOUBLE Value;

  extern UTF8 * ws2u8(wchar_t *);

  cfield = ws2u8(field);
  FieldsFound = sscanf((char *)cfield,"%4d:%2d:%lf",&degrees,&minutes,&seconds);
  if(FieldsFound!=3) {
    FieldsFound = sscanf((char *)cfield,"%4d %2d %lf",&degrees,&minutes,&seconds);
  }
  free(cfield);
  if(FieldsFound != 3)return(ERROR);
  if(minutes < 0 || seconds < 0) return (ERROR);
  if( (minutes > 59) || (seconds >= 60)) return (ERROR);

  if(degrees < 0) {
    degrees *= (-1);
    NegativeP = 1;
  }
  degrees %= 360;
  if(NegativeP) degrees = 360 -degrees;
  Value = ((((DOUBLE) degrees * 60) + (DOUBLE) minutes) * 60) + seconds;
  *key = Value;
  return(SUCCESS);
}

#ifdef HAVE_GNU_LIBC_VERSION_H
#include <gnu/libc-version.h>
#endif

void
ShowVersion(FILE *fp)   
{
  extern char progname[];
  extern char version[];
  extern char* compdate;
  char *vp;
  char vnum[11+1];

  fprintf(fp,"%s %s\n",progname,version);
  vp = tre_version();
    sscanf(vp,"%*s %11[0-9.] %*s",vnum);
#ifdef USE_UNINUM
  fprintf(fp,_("  lib gmp      %s\n"),gmp_version);
#else
  fprintf(fp,_("  lib gmp not linked\n"));
#endif
#ifndef USEUTF8PROC
  fprintf(fp,_("  lib icu      %s\n"),U_ICU_VERSION);
#else
  fprintf(fp,_("  lib utf8proc\n"));
#endif
  fprintf(fp,_("  lib tre      %s\n"),vnum);
#ifdef USE_UNINUM
  fprintf(fp,_("  lib uninum   %s\n"),uninum_version());
#else
  fprintf(fp,_("  lib uninum not linked\n"));
#endif
#ifdef HAVE_GNU_LIBC_VERSION_H
  fprintf(fp,_("  glibc        %s\n"),gnu_get_libc_version());
#endif
  fprintf(fp,_("Compiled %s\n"),compdate);
}

FILE *
OpenFile(char *file,char *mode,char *pgname)
{
  FILE *fp;
  extern FILE *fopen();
   
  if((fp = fopen(file,mode)) != NULL) return fp;
  else{
    switch(*mode){
    case 'r': 
      fprintf(stderr,_("%s: can't open file %s for reading.\n"),pgname,file);
      break;
    case 'w':
      fprintf(stderr,_("%s: can't open file %s for writing.\n"),pgname,file);
      break;
    default:
      fprintf(stderr,_("%s: can't open file %s for appending.\n"),pgname,file);
    }
    exit(OPENERROR);
  }
  /* NOTREACHED */
}


/*
 * Rewrite a string, replacing backslash escapes.
 *
 */

#define DEFAULT        0
#define READBACKSLASH  1
#define READONEDIGIT   2
#define READTWODIGITS  3

int
EvalEscapes(wchar_t *s)
{
  int state;
  wchar_t c;
  wchar_t odigit1 = 0L;
  wchar_t odigit2 = 0L;
  wchar_t sum = 0L;
  wchar_t *SStart;
  wchar_t *TStart;
  wchar_t *t;

  if(s == NULL) return(SUCCESS);

  /* Allocate storage for de-escaped copy of string */
	
  t = (wchar_t *) malloc(sizeof(wchar_t) * (wcslen(s)+1));
  if(t == NULL) return(ERROR);

  SStart = s;
  TStart = t;

  /* Work through string */
	
  state = 0;
  while((c = *s++) != L'\0'){
    switch(c){
    case '\\':
      if(state == READBACKSLASH){
	*t++ = L'\\'; /* Literal backslash */
	state = DEFAULT;
      }
      else if(state  == READONEDIGIT){
	*t++ = L'\\';
	*t++ = odigit1;
	state = DEFAULT;
      }
      else if(state  == READTWODIGITS){
	*t++ = L'\\';
	*t++ = odigit1;
	*t++ = odigit2;
	state = DEFAULT;
      }
      else{				/* state == DEFAULT */
	state = READBACKSLASH;
      }
      break;
    case 'n':
      if(state == READBACKSLASH){
	*t++ = L'\n';
	state = DEFAULT;
      }
      else{
	if(state  == READONEDIGIT){
	  *t++ = L'\\';
	  *t++ = odigit1;
	}
	else if(state  == READTWODIGITS){
	  *t++ = L'\\';
	  *t++ = odigit1;
	  *t++ = odigit2;
	}
	*t++ = c;
	state = DEFAULT;
      }
      break;
    case ' ':
      if(state == READBACKSLASH){
	*t++ = L' ';
	state = DEFAULT;
      }
      else{
	if(state  == READONEDIGIT){
	  *t++ = L'\\';
	  *t++ = odigit1;
	}
	else if(state  == READTWODIGITS){
	  *t++ = L'\\';
	  *t++ = odigit1;
	  *t++ = odigit2;
	}
	*t++ = c;
	state = DEFAULT;
      }
      break;
    case 't':
      if(state == READBACKSLASH){
	*t++ = L'\t';
	state = DEFAULT;
      }
      else{
	if(state  == READONEDIGIT){
	  *t++ = L'\\';
	  *t++ = odigit1;
	}
	else if(state  == READTWODIGITS){
	  *t++ = L'\\';
	  *t++ = odigit1;
	  *t++ = odigit2;
	}
	*t++ = c;
	state = DEFAULT;
      }
      break;
    case '0':
    case '1':
    case '2':
    case '3':
      if(state == READBACKSLASH){
	odigit1 = c;
	sum = ( (c - L'0') * 64);
	state = READONEDIGIT;
      }
      else if(state == READONEDIGIT){
	odigit2 = c;
	sum+=( (c - L'0') * 8);
	state = READTWODIGITS;
      }
      else if(state == READTWODIGITS){
	sum+= (c - L'0');
	*t++ = sum;
	state = DEFAULT;
      }
      else *t++ = c;
      break;
    case '4':
    case '5':
    case '6':
    case '7':
      /* These are not possible leading digits */
      if(state == READBACKSLASH){
	*t++ = L'\\';
	*t++ = c;
	state = DEFAULT;
      }
      else if(state == READONEDIGIT){
	odigit2 = c;
	sum+=( (c - L'0') * 8);
	state = READTWODIGITS;
      }
      else if(state == READTWODIGITS){
	sum+= (c - L'0');
	*t++ = sum;
	state = DEFAULT;
      }
      else *t++ = c;
      break;
    default:
      if(state == READBACKSLASH){
	*t++ = L'\\';
      }
      else if(state == READONEDIGIT){
	*t++ = L'\\';
	*t++ = odigit1;
      }
      else if(state == READTWODIGITS){
	*t++ = L'\\';
	*t++ = odigit1;
	*t++ = odigit2;
      }
      *t++ = c;
      state = DEFAULT;
      break;
    }
  }
  
  /* Null terminate copy and overwrite original */
  
  *t = L'\0';
  wcscpy(SStart,TStart);
  free((void *) TStart);
  return(SUCCESS);
}

/*
 * Overwrite a textual key with itself, skipping characters specified
 * in the exclusion table. This version differs from the main Exclude
 * function in operating on chars and in not overwriting its input.
 */

int
ExcludeChar(wchar_t *str, wchar_t **out, int len, wchar_t *exct)
{
  int i;
  int cnt;
  static wchar_t *OutStr;

  cnt = 0;
  if(OutStr != NULL) free((void *) OutStr);
  OutStr = (wchar_t *) malloc ( (size_t) (len+1) * sizeof(wchar_t));
  if (OutStr == NULL) {
    fprintf(stderr,_("ExcludeChar: out of memory\n"));
    exit(OUTOFMEMORY);
  }
  if(!(exct[str[0]] & EX_INITIAL) ){
    OutStr[cnt++] = str[0];
  }
  for(i = 1; i < len-1; i++) {
    if(exct[str[i]] & EX_MEDIAL) continue;
    else OutStr[cnt++] = str[i];
  }
  if(!(exct[str[len-1]] & EX_FINAL)) {
    OutStr[cnt++] = str[len-1];
  }
  OutStr[cnt] = ((wchar_t) 0); /* Null terminate */
  *out = OutStr;
  return(cnt);
}

#ifndef USEUTF8PROC

/* 
 * Return the normalized input string in fresh storage.
 * It is the responsability of the caller to free the original
 * un-normalized string, if it is no longer needed.
 * The length of the normalized string is returned via
 * the pointer ResultLength.
 */

UTF32 * NormalizeString(UTF32 *s, int32_t *ResultLength, UNormalizationMode type) {
  UChar *Result;
  UErrorCode status;
  int CharsNormalized;
  UTF16 *s16;
  UTF32 *r32;

  extern UTF16 *ConvertUTF32toUTF16 (const UTF32*, int *);
  extern UTF32 *ConvertUTF16toUTF32 (const UTF16*, int *);

  s16 = ConvertUTF32toUTF16(s,&status);
  if(!s16) {
    *ResultLength = 0;
    return NULL;
  }
  status = U_ZERO_ERROR;
  *ResultLength = unorm_normalize(s16,-1,type, 0, NULL, 0, &status);
  Result = (UChar*) malloc(sizeof(UChar) * ((*ResultLength) + 1));
  if (!Result) {
    *ResultLength = 0;
    return NULL;
  }
  status = U_ZERO_ERROR;
  /* Add 1 to *ResultLength to allow addition of terminal null and prevent error code */
  CharsNormalized = unorm_normalize(s16, -1,type, 0, Result, (*ResultLength+1), &status);
  free((void *) s16);
  if(status != U_ZERO_ERROR) {
    *ResultLength = -1;
    return NULL;
  }
  r32 = ConvertUTF16toUTF32(Result,&status);
  if(!r32) {
    *ResultLength = 0;
    return NULL;
  }
  free((void *) Result);
  *ResultLength = CharsNormalized;
  return (r32);
}

#endif
