/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */
#ifndef RandomGenerator_H
#define RandomGenerator_H

#include <mrpt/utils/utils_defs.h>
#include <mrpt/math/CMatrixTemplateNumeric.h>

namespace mrpt
{
	namespace math
	{
		class CMatrix;
		class CMatrixD;
	}

	/** A namespace of pseudo-random numbers genrators of diferent distributions.
	 */
	namespace random
	{
		using namespace mrpt::utils;
		using namespace mrpt::math;

		/** Generate a normalized normally distributed pseudo-random number.
		 *  \param likelihood If desired, pass a pointer to a double which will receive the likelihood of the given sample to have been obtained, that is, the value of the normal pdf at the sample value.
		 */
		double MRPTDLLIMPEXP normalizedGaussian( double *likelihood = NULL);

		/** Generate a normally distributed pseudo-random number.
		 * \param mean The mean value of desired normal distribution
		 * \param std  The standard deviation value of desired normal distribution
		 */
		double MRPTDLLIMPEXP  RandomNormal( double mean = 0, double std = 1);

		/** Generate a uniformly distributed pseudo-random number using the MT19937 algorithm, in the whole range of 32-bit integers.
		  *  See: http://en.wikipedia.org/wiki/Mersenne_twister
		  * \sa RandomUni, Randomize
		  */
		uint32_t MRPTDLLIMPEXP  RandomUniInt();

		/** Generate a uniformly distributed pseudo-random number using the MT19937 algorithm, scaled to the selected range.
		  *  This function uses internally RandomUniInt to generate the number, then scales it to the desired range.
		  *  Since MRPT 0.6.0 the MT19937 algorithm is used instead of C runtime library "rand" version.
		  *  See: http://en.wikipedia.org/wiki/Mersenne_twister
		  * \sa RandomUniInt, Randomize
		  */
		inline double RandomUni( const double &min, const double &max)
		{
			return min + (max-min)* RandomUniInt() * 2.3283064370807973754314699618685e-10; // 0xFFFFFFFF ^ -1
		}


		/** Fills the given matrix with independent, uniformly distributed samples.
		  * \sa matrixRandomNormal
		  */
		template <class T>
		void matrixRandomUni(
			CMatrixTemplateNumeric<T> &matrix,
			const  T& unif_min = 0,
			const  T& unif_max = 1 )
		{
			for (size_t r=0;r<matrix.getRowCount();r++)
				for (size_t c=0;c<matrix.getColCount();c++)
					matrix.set_unsafe(r,c, RandomUni(unif_min,unif_max) );
		}

		/** Fills the given matrix with independent, uniformly distributed samples.
		  * \sa vectorRandomNormal
		  */
		template <class T>
		void vectorRandomUni(
			std::vector<T> &v_out,
			const  T& unif_min = 0,
			const  T& unif_max = 1 )
		{
			size_t n = v_out.size();
			for (size_t r=0;r<n;r++)
				v_out[r] = RandomUni(unif_min,unif_max);
		}

		/** Fills the given matrix with independent, normally distributed samples.
		  * \sa matrixRandomUni
		  */
		template <class T>
		void matrixRandomNormal(
			CMatrixTemplateNumeric<T> &matrix,
			const  T& mean = 0,
			const  T& std = 1 )
		{
			for (size_t r=0;r<matrix.getRowCount();r++)
				for (size_t c=0;c<matrix.getColCount();c++)
					matrix.set_unsafe(r,c, mean + std*normalizedGaussian() );
		}

		/** Generates a random vector with independent, normally distributed samples.
		  * \sa matrixRandomUni
		  */
		template <class T>
		void vectorRandomNormal(
			std::vector<T> &v_out,
			const  T& mean = 0,
			const  T& std = 1 )
		{
			size_t n = v_out.size();
			for (size_t r=0;r<n;r++)
				v_out[r] = mean + std*normalizedGaussian();
		}

		/** Randomize the generators.
		 *   A seed can be providen, or a current-time based seed can be used (default)
		 */
		void	 MRPTDLLIMPEXP  Randomize(const uint32_t &seed);
		void	 MRPTDLLIMPEXP  Randomize();

		/** Returns a random permutation of a vector: all the elements of the input vector are in the output but at random positions.
		  */
		template <class T>
		void  randomPermutation(
			const std::vector<T> &in_vector,
			std::vector<T>       &out_result)
		{
			out_result = in_vector;
			std::random_shuffle( out_result.begin(),out_result.end()  );
		}

		/** Generate multidimensional random samples according to a given covariance matrix.
		 * \exception std::exception On invalid covariance matrix
		 * \sa randomNormalMultiDimensionalMany
		 */
		 template <typename T>
		 void  MRPTDLLIMPEXP  randomNormalMultiDimensional(
			const CMatrixTemplateNumeric<T>	&cov,
			std::vector<T>		&out_result);

		/** Generate a given number of multidimensional random samples according to a given covariance matrix.
		 * \param cov The covariance matrix where to draw the samples from.
		 * \param desiredSamples The number of samples to generate.
		 * \param samplesLikelihoods If desired, set to a valid pointer to a vector, where it will be stored the likelihoods of having obtained each sample: the product of the gaussian-pdf for each independent variable.
		 * \param ret The output list of samples
		 *
		 * \exception std::exception On invalid covariance matrix
		 *
		 * \sa randomNormalMultiDimensional
		 */
		 template <typename T>
		 void  MRPTDLLIMPEXP  randomNormalMultiDimensionalMany(
			const CMatrixTemplateNumeric<T>	&cov,
			size_t							desiredSamples,
			std::vector< std::vector<T> >	&ret,
			std::vector<T>					*samplesLikelihoods = NULL);


	}// End of namespace

} // End of namespace

#endif
