/*	picture.c
 *
 *	Various funtions for saving/loading pictures.
 *	Copyright 2002 by Jeroen Vreeken (pe1rxq@amsat.org)
 *	Portions of this file are Copyright by Lionnel Maugis
 *	This software is distributed under the GNU public license version 2
 *	See also the file 'COPYING'.
 *
 */
#include "motion.h"
#include "picture.h"
#include "event.h"

#include <jpeglib.h>


// twice as fast as RGB jpeg 
void put_jpeg_yuv420p (FILE *fp, unsigned char *image, int width, int height, int quality)
{
	int i,j;

	JSAMPROW y[16],cb[16],cr[16]; // y[2][5] = color sample of row 2 and pixel column 5; (one plane)
	JSAMPARRAY data[3]; // t[0][2][5] = color sample 0 of row 2 and column 5

	struct jpeg_compress_struct cinfo;
	struct jpeg_error_mgr jerr;

	data[0] = y;
	data[1] = cb;
	data[2] = cr;

	cinfo.err = jpeg_std_error(&jerr);  // errors get written to stderr 
	
	jpeg_create_compress (&cinfo);
	cinfo.image_width = width;
	cinfo.image_height = height;
	cinfo.input_components = 3;
	jpeg_set_defaults (&cinfo);

	jpeg_set_colorspace(&cinfo, JCS_YCbCr);

	cinfo.raw_data_in = TRUE; // supply downsampled data
	cinfo.comp_info[0].h_samp_factor = 2;
	cinfo.comp_info[0].v_samp_factor = 2;
	cinfo.comp_info[1].h_samp_factor = 1;
	cinfo.comp_info[1].v_samp_factor = 1;
	cinfo.comp_info[2].h_samp_factor = 1;
	cinfo.comp_info[2].v_samp_factor = 1;

	jpeg_set_quality (&cinfo, quality, TRUE);
	cinfo.dct_method = JDCT_FASTEST;

	jpeg_stdio_dest (&cinfo, fp);  	  // data written to file
	jpeg_start_compress (&cinfo, TRUE);

	for (j=0;j<height;j+=16) {
		for (i=0;i<16;i++) {
			y[i] = image + width*(i+j);
			if (i%2 == 0) {
				cb[i/2] = image + width*height + width/2*((i+j)/2);
				cr[i/2] = image + width*height + width*height/4 + width/2*((i+j)/2);
			}
		}
		jpeg_write_raw_data (&cinfo, data, 16);
	}

	jpeg_finish_compress (&cinfo);
	jpeg_destroy_compress (&cinfo);
}

void put_jpeg_grey (FILE *picture, char *image, int width, int height, int quality)
{
	int y;
	JSAMPROW row_ptr[1];
	struct jpeg_compress_struct cjpeg;
	struct jpeg_error_mgr jerr;

	cjpeg.err = jpeg_std_error(&jerr);
	jpeg_create_compress (&cjpeg);
	cjpeg.image_width = width;
	cjpeg.image_height = height;
	cjpeg.input_components = 1; /* one colour component */
	cjpeg.in_color_space = JCS_GRAYSCALE;

	jpeg_set_defaults (&cjpeg);

	jpeg_set_quality (&cjpeg, quality, TRUE);
	cjpeg.dct_method = JDCT_FASTEST;
	jpeg_stdio_dest (&cjpeg, picture);

	jpeg_start_compress (&cjpeg, TRUE);

	row_ptr[0]=image;
	for (y=0; y<height; y++) {
		jpeg_write_scanlines(&cjpeg, row_ptr, 1);
		row_ptr[0]+=width;
	}
	jpeg_finish_compress(&cjpeg);
	jpeg_destroy_compress(&cjpeg);
}

void put_ppm_bgr24 (FILE *picture, char *image, int width, int height)
{
	int x, y;
	unsigned char *l=image;
	unsigned char *u=image+width*height;
	unsigned char *v=u+(width*height)/4;
	int r, g, b;
	unsigned char rc, gc, bc;

	/*	ppm header
	 *	width height
	 *	maxval
	 */
	fprintf(picture, "P6\n");
	fprintf(picture, "%d %d\n", width, height);
	fprintf(picture, "%d\n", 255);
	for (y=0; y<height; y++) {
		for (x=0; x<width; x++) {
			r = 76283*(((int)*l)-16)+104595*(((int)*u)-128);
			g = 76283*(((int)*l)-16)- 53281*(((int)*u)-128)-25625*(((int)*v)-128);
			b = 76283*(((int)*l)-16)+132252*(((int)*v)-128);
			r = r>>16;
			g = g>>16;
			b = b>>16;
			if (r<0)
				r=0;
			else if (r>255)
				r=255;
			if (g<0)
				g=0;
			else if (g>255)
				g=255;
			if (b<0)
				b=0;
			else if (b>255)
				b=255;
			rc=r;
			gc=g;
			bc=b;

			l++;
			if (x&1) {
				u++;
				v++;
			}
			/* ppm is rgb not bgr */
			fwrite(&bc, 1, 1, picture);
			fwrite(&gc, 1, 1, picture);
			fwrite(&rc, 1, 1, picture);
		}
		if (y&1) {
			u-=width/2;
			v-=width/2;
		}
	}
}

void put_picture_fd (struct context *cnt, FILE *picture, char *image, int quality)
{
	if (cnt->conf.ppm) {
		put_ppm_bgr24(picture, image, cnt->conf.width, cnt->conf.height);
	} else {
//		printf("type: %d %d\n", cnt->imgs.type, VIDEO_PALETTE_YUV420P);
		switch (cnt->imgs.type) {
			case VIDEO_PALETTE_YUV420P:
				put_jpeg_yuv420p(picture, image, cnt->conf.width, cnt->conf.height, quality);
				break;
			case VIDEO_PALETTE_GREY:
				put_jpeg_grey(picture, image, cnt->conf.width, cnt->conf.height, quality);
				break;
		}
	}
}

void put_picture (struct context *cnt, char *file, char *image, int ftype)
{
	FILE *picture;

	picture=myfopen(file, "w");
	if (!picture) {
		/* Report to syslog - suggest solution if the problem is access rights to target dir */
		if (errno ==  EACCES){
			syslog(LOG_ERR, "can't write picture to file %s - check access rights to target directory: %m", file);
			cnt->finish = 1;
			return;
		} else {
			/* If target dir is temporarily unavailable we may survive */
			syslog(LOG_ERR, "can't write picture to file %s: %m", file);
			return;
		}
	}

	put_picture_fd(cnt, picture, image, cnt->conf.quality);
	event(EVENT_FILECREATE, cnt, file, (void *)ftype,NULL);
	fclose(picture);
}
