/*
  **
  ** conf.c
  **
  ** I originaly wrote conf.c as part of the drpoxy package
  ** thanks to Matthew Pratt and others for their additions.
  **
  ** Copyright 1999 Jeroen Vreeken (pe1rxq@chello.nl)
  **
  ** This software is licensed under the terms of the GNU General
  ** Public License (GPL). Please see the file COPYING for details.
  **
  **
*/

/**
 * How to add a config option :
 *
 *   1. think twice, there are settings enough
 *
 *   2. add a field to 'struct config' (conf.h) and 'struct config conf'
 *
 *   4. add a entry to the config_params array below, if your
 *      option should be configurable by the config file.
 */
#include "motion.h"

#ifdef __freebsd__
#include "video_freebsd.h"
#else
#include "video.h"
#endif /* __freebsd__ */

#include "conf.h"
#include "track.h"

#define stripnewline(x) {if ((x)[strlen(x)-1]=='\n') (x)[strlen(x) - 1] = 0; }


struct config conf_template = {
	width:                 DEF_WIDTH,
	height:                DEF_HEIGHT,
	quality:               DEF_QUALITY,
	rotate_deg:            0,
	max_changes:           DEF_CHANGES,
	threshold_tune:        1,
	always_diff:           0,
	new_img:               1,
	motion_img:            0,
	output_all:            0,
	gap:                   DEF_GAP,
	maxmpegtime:           DEF_MAXMPEGTIME,
	snapshot_interval:     0,
	locate:                0,
	input:                 IN_DEFAULT,
	norm:                  0,
	frame_limit:           DEF_MAXFRAMERATE,
	mpeg:                  0,
	quiet:                 0,
	ppm:                   0,
	noise:                 DEF_NOISELEVEL,
	noise_tune:            1,
	mingap:                0,
	lightswitch:           0,
	jpg_cleanup:           0,
	nightcomp:             0,
	adjustrate:            0,
	oldlayout:             0,
	low_cpu:               0,
	nochild:               0,
	autobright:            0,
	roundrobin_frames:     1,
	roundrobin_skip:       1,
	pre_capture:           0,
	post_capture:          0,
	switchfilter:          0,
	ffmpeg_cap_new:        0,
	ffmpeg_cap_motion:     0,
	ffmpeg_bps:            DEF_FFMPEG_BPS,
	ffmpeg_vbr:            DEF_FFMPEG_VBR,
	ffmpeg_video_codec:    DEF_FFMPEG_CODEC,
	webcam_port:           0,
	webcam_quality:        30,
	webcam_motion:         0,
	webcam_maxrate:        100,
	webcam_localhost:      1,
	webcam_limit:          0,
	control_port:          0,
	control_localhost:     1,
	frequency:             0,
	timelapse:             0,
	timelapse_mode:        DEF_TIMELAPSE_MODE,
	predict_threshold:     DEF_PREDICT_THRESHOLD,
	predict_enable:        0,
#ifdef __freebsd__
	tuner_device:          TUNER_DEVICE,
#endif
	video_device:          VIDEO_DEVICE,
	vidpipe:               NULL,
	mail_address:          NULL,
	sms_nr:                NULL,
	filepath:              NULL,
	jpegpath:              DEF_JPEGPATH,
	mpegpath:              DEF_MPEGPATH,
	snappath:              DEF_SNAPPATH,
	timepath:              DEF_TIMEPATH,
	predictpath:           DEF_PREDICTPATH,
	externcommand:         NULL,
	mask_file:             NULL,
	sql_log_image:         1,
	sql_log_snapshot:      1,
	sql_log_mpeg:          0,
	sql_log_timelapse:     0,
	sql_log_prediction:    0,
	mysql_db:              NULL,
	mysql_host:            NULL,
	mysql_user:            NULL,
	mysql_password:        NULL,
	onsavecommand:         NULL,
	onmpegcommand:         NULL,
	onffmpegclosecommand:  NULL,
	motionvidpipe:         NULL,
	netcam_url:            NULL,
	netcam_userpass:       NULL,
	pgsql_db:              NULL,
	pgsql_host:            NULL,
	pgsql_user:            NULL,
	pgsql_password:        NULL,
	pgsql_port:            5432,
	mpeg_encode_bin:       "/usr/local/bin/mpeg_encode",
	text_changes:          0,
	text_left:             NULL,
	text_right:            DEF_TIMESTAMP,
	despeckle:             NULL,
	minimum_motion_frames: 1,
	debug_parameter:       0
};



static struct context **copy_bool(struct context **, char *, int);
static struct context **copy_int(struct context **, char *, int);
struct context **copy_string(struct context **, char *, int);
static struct context **config_thread(struct context **cnt, char *str, int val);
static struct context **load_desc(struct context **cnt, char *, int);

static char *print_bool(struct context **, char **, int, int);
static char *print_int(struct context **, char **, int, int);
static char *print_string(struct context **, char **, int, int);
static char *print_desc(struct context **, char **, int, int);
static char *print_thread(struct context **, char **, int, int);

static void usage(void);

/* Pointer magic to determine relative addresses of variables to a
   struct context pointer */
#define CNT_OFFSET(varname) ( (int)&((struct context *)NULL)->varname )
#define CONF_OFFSET(varname) ( (int)&((struct context *)NULL)->conf.varname ) 
#define TRACK_OFFSET(varname) ( (int)&((struct context *)NULL)->track.varname ) 

config_param config_params[] = {
	{
	"daemon",
	"############################################################\n"
	"# Daemon\n"
	"############################################################\n\n"
	"#Start in daemon (background) mode and release terminal (default: off)",
	CNT_OFFSET(daemon),
	copy_bool,
	print_bool
	},
	{
	"videodevice",
	"\n###########################################################\n"
	"# Capture device options\n"
	"############################################################\n\n"
	"#Videodevice to be used for capturing  (default /dev/video0)\n"
	"#for FreeBSD default is /dev/bktr0",
	CONF_OFFSET(video_device),
	copy_string,
	print_string
	},
#ifdef __freebsd__
	{
	"tunerdevice",
	"#Tuner device to be used for capturing using tuner as source (default /dev/tuner0)\n"
	"#This is ONLY used for FreeBSD. Leave it commented out for Linux",
	CONF_OFFSET(tuner_device),
	copy_string,
	print_string
	},
#endif
	{
	"input",
	"#The video input to be used (default: 8)\n"
	"#Should normally be set to 1 for video/TV cards, and 8 for USB cameras",
	CONF_OFFSET(input),
	copy_int,
	print_int
	},
	{
	"norm",
	"#The video norm to use (only for video capture and TV tuner cards)\n"
	"#Values: 0 (PAL), 1 (NTSC), 2 (SECAM), 3 (PAL NC no colour). Default: 0 (PAL)",
	CONF_OFFSET(norm),
	copy_int,
	print_int
	},
	{
	"frequency",
	"#The frequency to set the tuner to (kHz) (only for TV tuner cards) (default: 0)",
	CONF_OFFSET(frequency),
	copy_int,
	print_int
	},
	{
	"rotate",
	"#Rotate image this number of degrees. The rotation affects all saved images as\n"
	"#well as mpeg movies. Valid values: 0 (default = no rotation), 90, 180 and 270.",
	CONF_OFFSET(rotate_deg),
	copy_int,
	print_int
	},
	{
	"width",
	"#Image width (pixels). Valid range: Camera dependent, default: 352",
	CONF_OFFSET(width),
	copy_int,
	print_int
	},
	{
	"height",
	"#Image height (pixels). Valid range: Camera dependent, default: 288",
	CONF_OFFSET(height),
	copy_int,
	print_int
	},
	{
	"framerate",
	"#Maximum number of frames to be captured per second.\n"
	"#Valid range: 2-100. Default: 100 (almost no limit).",
	CONF_OFFSET(frame_limit),
	copy_int,
	print_int
	},
#ifdef HAVE_CURL
	{
	"netcam_url",
	"#URL to use if you are using a network camera, size will be autodetected (incl http://)\n"
	"#Must be a URL that returns single jpeg pictures. Default: Not defined",
	CONF_OFFSET(netcam_url),
	copy_string,
	print_string
	},
	{
	"netcam_userpass",
	"#Username and password for network camera (only if required). Default: not defined",
	CONF_OFFSET(netcam_userpass),
	copy_string,
	print_string
	},
#endif /* HAVE CURL */
	{
	"auto_brightness",
	"#Let motion regulate the brightness of a video device (default: off)\n"
	"#Only recommended for cameras without auto brightness",
	CONF_OFFSET(autobright),
	copy_bool,
	print_bool
	},

	{
	"roundrobin_frames",
	"\n############################################################\n"
	"# Round Robin (multiple inputs on same video device name)\n"
	"############################################################\n\n"
	"#Number of frames to capture in each roundrobin step (default: 1)",
	CONF_OFFSET(roundrobin_frames),
	copy_int,
	print_int
	},
	{
	"roundrobin_skip",
	"#Number of frames to skip before each roundrobin step (default: 1)",
	CONF_OFFSET(roundrobin_skip),
	copy_int,
	print_int
	},
	{
	"switchfilter",
	"#Try to filter out noise generated by roundrobin (default: 0)",
	CONF_OFFSET(switchfilter),
	copy_bool,
	print_int
	},

	{
	"threshold",
	"\n############################################################\n"
	"# Motion Detection Settings:\n"
	"############################################################\n\n"
	"#Threshold for number of changed pixels in an image that\n"
	"#triggers motion detection (default: 1500)",
	CONF_OFFSET(max_changes),
	copy_int,
	print_int
	},
	{
	"threshold_tune",
	"#Automatically tune the threshold down if possible (default: on)",
	CONF_OFFSET(threshold_tune),
	copy_bool,
	print_bool
	},
	{
	"noise_level",
	"#Noise threshold for the motion detection (default: 32)",
	CONF_OFFSET(noise),
	copy_int,
	print_int
	},
	{
	"noise_tune",
	"#Automatically tune the noise threshold (default: on)",
	CONF_OFFSET(noise_tune),
	copy_bool,
	print_bool
	},
	{
	"night_compensate",
	"#Enables motion to adjust its detection/noise level for very dark frames\n"
	"#don't mix with noise_tune (default: off)",
	CONF_OFFSET(nightcomp),
	copy_bool,
	print_bool
	},
	{
	"despeckle",
	"#Despeckle motion image using (e)rode or (d)ilate or (l)abel (Default: not defined)\n"
	"#Recommended value is EedDl. Any combination (and number of) of E, e, d, and D is valid.\n"
	"#(l)abeling must only be used once and in the last place.\n"
	"#Comment out to disable",
	CONF_OFFSET(despeckle),
	copy_string,
	print_string
	},
	{
	"mask_file",
	"#PGM file to use as a sensitivity mask.\n"
	"#Full path name to. (Default: not defined)",
	CONF_OFFSET(mask_file),
	copy_string,
	print_string
	},
	{
	"lightswitch",
	"#Ignore sudden massive light intensity changes given as a percentage of the picture\n"
	"#area that changed intensity. Valid range: 0 - 100 , default: 0 = disabled",
	CONF_OFFSET(lightswitch),
	copy_int,
	print_int
	},
	{
	"minimum_motion_frames",
	"#Picture frames must contain motion at least the specified number of frames\n"
	"#in a row before they are detected as true motion. At the default of 1, all\n"
	"#motion is detected. Valid range: 1 to thousands, recommended 1-10",
	CONF_OFFSET(minimum_motion_frames),
	copy_int,
	print_int
	},
	{
	"pre_capture",
	"#Specifies the number of pre-captured (buffered) pictures from before motion\n"
	"#was detected that will be output at motion detection (default: 0)",
	CONF_OFFSET(pre_capture),
	copy_int,
	print_int
	},
	{
	"post_capture",
	"#Number of frames to capture after motion is no longer detected (default: 0)",
	CONF_OFFSET(post_capture),
	copy_int,
	print_int
	},
	{
	"gap",
	"#Minimum gap between two events (seconds) (default: 60)\n"
	"#An event is defined as a series of motion images taken within a short timeframe.",
	CONF_OFFSET(gap),
	copy_int,
	print_int
	},
	{
	"minimum_gap",
	"#Minimum gap in seconds between the storing pictures while detecting motion.\n"
	"#Default: 0 = as fast as possible (given by the camera framerate)",
	CONF_OFFSET(mingap),
	copy_int,
	print_int
	},
	{
	"max_mpeg_time",
	"#Maximum length in seconds of an mpeg movie (common for ffmpeg and Berkeley types)\n"
	"#When value is exceeded a new mpeg file is created. (Default: 0 = infinite)",
	CONF_OFFSET(maxmpegtime),
	copy_int,
	print_int
	},
	{
	"low_cpu",
	"#Number of frames per second to capture when not detecting\n"
	"#motion (saves CPU load) (Default: 0 = disabled)",
	CONF_OFFSET(low_cpu),
	copy_int,
	print_int
	},
	{
	"output_all",
	"#Always save images even if there was no motion (default: off)",
	CONF_OFFSET(output_all),
	copy_bool,
	print_bool
	},

	{
	"output_normal",
	"\n############################################################\n"
	"# Image File Output\n"
	"############################################################\n\n"
	"#Output 'normal' pictures when motion is detected (default: on)",
	CONF_OFFSET(new_img),
	copy_bool,
	print_bool
	},
	{
	"output_motion",
	"#Output pictures with only the pixels moving object (green ghost images) (default: off)",
	CONF_OFFSET(motion_img),
	copy_bool,
	print_bool
	},
	{
	"quality",
	"#The quality (in percent) to be used by the jpeg compression (default: 75)",
	CONF_OFFSET(quality),
	copy_int,
	print_int
	},
	{
	"ppm",
	"#Output ppm images instead of jpeg (default: off)",
	CONF_OFFSET(ppm),
	copy_bool,
	print_bool
	},

#ifdef HAVE_FFMPEG
	{
	"ffmpeg_cap_new",
	"\n############################################################\n"
	"# Film (mpeg) File Output - ffmpeg based\n"
	"############################################################\n\n"
	"#Use ffmpeg to encode mpeg movies in realtime (default: off)",
	CONF_OFFSET(ffmpeg_cap_new),
	copy_bool,
	print_bool
	},
	{
	"ffmpeg_cap_motion",
	"#Use ffmpeg to make movies with only the pixels moving\n"
	"#object (green ghost images) (default: off)",
	CONF_OFFSET(ffmpeg_cap_motion),
	copy_bool,
	print_bool
	},
	{
	"ffmpeg_timelapse",
	"#Use ffmpeg to encode a timelapse movie\n"
	"#Default value 0 = off - else save frame every Nth second",
	CONF_OFFSET(timelapse),
	copy_int,
	print_int
	},
	{
	"ffmpeg_timelapse_mode",
	"#The file rollover mode of the timelapse video\n"
	"#Valid values: hourly, daily (default), weekly-sunday, weekly-monday, monthly, manual",
	CONF_OFFSET(timelapse_mode),
	copy_string,
	print_string
	},
	{
	"ffmpeg_bps",
	"#Bitrate to be used by the ffmpeg encoder (default: 400000)\n"
	"#This option is ignored if ffmpeg_variable_bitrate is not 0 (disabled)",
	CONF_OFFSET(ffmpeg_bps),
	copy_int,
	print_int
	},
	{
	"ffmpeg_variable_bitrate",
	"#Enables and defines variable bitrate for the ffmpeg encoder.\n"
	"#ffmpeg_bps is ignored if variable bitrate is enabled.\n"
	"#Valid values: 0 (default) = fixed bitrate defined by ffmpeg_bps,\n"
	"#or the range 2 - 31 where 2 means best quality and 31 is worst.",
	CONF_OFFSET(ffmpeg_vbr),
	copy_int,
	print_int
	},
	{
	"ffmpeg_video_codec",
	"#Codec to used by ffmpeg for the video compression.\n"
	"#Timelapse mpegs are always made in mpeg1 format independent from this option.\n"
	"#Supported formats are: mpeg1 (default), mpeg4, and msmpeg4.\n"
	"#mpeg1 gives you files with extension .mpg\n"
	"#mpeg4 or msmpeg4 give you files with extension .avi\n"
	"#msmpeg4 is recommended for use with Windows Media Player because\n"
	"#it requires no installation of codec on the Windows client.",
	CONF_OFFSET(ffmpeg_video_codec),
	copy_string,
	print_string
	},
#endif /* HAVE_FFMPEG */

	{
	"mpeg_encode",
	"\n############################################################\n"
	"# Film (mpeg) File Output - Berkeley mpeg_encode Based\n"
	"############################################################\n\n"
	"#Use mpeg_encode from Berkeley to encode movies  (default: off)",
	CONF_OFFSET(mpeg),
	copy_bool,
	print_bool
	},
	{
	"mpeg_encode_bin",
	"#Path to the mpeg_encode binary (default: /usr/bin/mpeg_encode)",
	CONF_OFFSET(mpeg_encode_bin),
	copy_string,
	print_string
	},
	{
	"adjust_rate",
	"#Adjust the number of frames for an mpeg movie (only mpeg_encode) to get 25fps  (default: off)",
	CONF_OFFSET(adjustrate),
	copy_bool,
	print_bool
	},
	{
	"jpg_cleanup",
	"#Delete jpeg images after they have been converted to movies by mpeg_encode (default: off)\n"
	"#This option is only used when mpeg_encode is on",
	CONF_OFFSET(jpg_cleanup),
	copy_bool,
	print_bool
	},

	{
	"snapshot_interval",
	"\n############################################################\n"
	"# Snapshots (Traditional Periodic Webcam File Output)\n"
	"############################################################\n\n"
	"#Make automated snapshot every N seconds (default: 0 = disabled)",
	CONF_OFFSET(snapshot_interval),
	copy_int,
	print_int
	},

	{
	"locate",
	"\n############################################################\n"
	"# Text Display\n"
	"# %Y = year, %m = month, %d = date,\n"
	"# %H = hour, %M = minute, %S = second, %T = HH:MM:SS\n"
	"# %v = event, %q = frame number, \\n = new line\n"
	"# You can put quotation marks around the text to allow\n"
	"# leading spaces\n"
	"############################################################\n\n"
	"#Locate and draw a box around the moving object (default: off)",
	CONF_OFFSET(locate),
	copy_bool,
	print_bool
	},
	{
	"text_right",
	"#Draws the timestamp using same options as C function strftime(3)\n"
	"#Default: %Y-%m-%d\\n%T = date in ISO format and time in 24 hour clock\n"
	"#Text is placed in lower right corner",
	CONF_OFFSET(text_right),
	copy_string,
	print_string
	},
	{
	"text_left",
	"#Draw a user defined text on the images using same options as C function strftime(3)\n"
	"#Default: Not defined = no text\n"
	"#Text is placed in lower left corner",
	CONF_OFFSET(text_left),
	copy_string,
	print_string
	},
 	{
	"text_changes",
	"#Draw the number of changed pixed on the images (default: off)\n"
	"#Will normally be set to off except when you setup and adjust the motion settings\n"
	"#Text is placed in upper right corner",
	CONF_OFFSET(text_changes),
	copy_bool,
	print_bool
	},

	{
	"target_dir",
	"\n############################################################\n"
	"# Target Directories and filenames For Images And Films\n"
	"# For the options snapshot_, jpeg_, mpeg_ and timelapse_filename\n"
	"# you can use conversion specifiers\n"
	"# %Y = year, %m = month, %d = date,\n"
	"# %H = hour, %M = minute, %S = second,\n"
	"# %v = event, %q = frame number\n"
	"# Quotation marks round string are allowed.\n"
	"############################################################\n\n"
	"#Target base directory for pictures and films",
	CONF_OFFSET(filepath),
	copy_string,
	print_string
	},
	{
	"snapshot_filename",
	"#File path for snapshots (jpeg or ppm) relative to target_dir\n"
	"#Default: "DEF_SNAPPATH"\n"
	"#Default value is equivalent to legacy oldlayout option\n"
	"#For Motion 3.0 compatible mode choose: %Y/%m/%d/%H/%M/%S-snapshot\n"
	"#File extension .jpg or .ppm is automatically added so do not include this.\n"
	"#Note: A symbolic link called lastsnap.jpg created in the target_dir will always\n"
	"#point to the latest snapshot, unless snapshot_filename is exactly 'lastsnap'",
	CONF_OFFSET(snappath),
	copy_string,
	print_string
	},
	{
	"jpeg_filename",
	"#File path for motion triggered images (jpeg or ppm) relative to target_dir\n"
	"#Default: "DEF_JPEGPATH"\n"
	"#Default value is equivalent to legacy oldlayout option\n"
	"#For Motion 3.0 compatible mode choose: %Y/%m/%d/%H/%M/%S-%q\n"
	"#File extension .jpg or .ppm is automatically added so do not include this\n"
	"#For use of Berkeley mpeg_encode this parameter but be set as follows:\n"
	"#If berkeley_single_directory = on: %v-%Y%m%d%H%M%S-%q\n"
	"#If berkeley_single_directory = off: %Y/%m/%d/%H/%M/%S-%q",
	CONF_OFFSET(jpegpath),
	copy_string,
	print_string
	},
#ifdef HAVE_FFMPEG
	{
	"ffmpeg_filename",
	"#File path for motion triggered ffmpeg films (mpeg) relative to target_dir\n"
	"#Default: "DEF_MPEGPATH"\n"
	"#Default value is equivalent to legacy oldlayout option\n"
	"#For Motion 3.0 compatible mode choose: %Y/%m/%d/%H%M%S\n"
	"#File extension .mpg or .avi is automatically added so do not include this",
	CONF_OFFSET(mpegpath),
	copy_string,
	print_string
	},
	{
	"timelapse_filename",
	"#File path for timelapse mpegs relative to target_dir (ffmpeg only)\n"
	"#Default: "DEF_TIMEPATH"\n"
	"#Default value is near equivalent to legacy oldlayout option\n"
	"#For Motion 3.0 compatible mode choose: %Y/%m/%d-timelapse\n"
	"#File extension .mpg or .avi is automatically added so do not include this",
	CONF_OFFSET(timepath),
	copy_string,
	print_string
	},
#endif /* HAVE_FFMPEG */
	{
	"berkeley_single_directory", 
	"#Use single directory structure (For Berkeley mpeg_encode feature only)\n"
	"#on = place mpeg_encode file in target_dir with name YYYYMMDDHHMMSS-n\n"
	"#off (default) = place files as target_dir/YYYY/MM/DD/SS-n",
	CONF_OFFSET(oldlayout),
	copy_bool,
	print_bool
	},
	{
	"predict_filename",
	"#File path for prediction files. (Default: "DEF_PREDICTPATH")\n"
	"#Default value is equivalent to legacy oldlayout option\n"
	"#For Motion 3.0 compatible mode choose: %Y/%m/%d/%H%M%S\n"
	"#File extension .desc is automatically added so do not include this",
	CONF_OFFSET(predictpath),
	copy_string,
	print_string
	},

	{
	"webcam_port",
	"\n############################################################\n"
	"# Live Webcam Server\n"
	"############################################################\n\n"
	"#The mini-http server listens to this port for requests (default: 0 = disabled)",
	CONF_OFFSET(webcam_port),
	copy_int,
	print_int
	},
	{
	"webcam_quality",
	"#Quality of the jpeg images produced (default: 30)",
	CONF_OFFSET(webcam_quality),
	copy_int,
	print_int
	},
	{
	"webcam_motion",
	"#Only output frames if motion is detected (default: off)",
	CONF_OFFSET(webcam_motion),
	copy_bool,
	print_bool
	},
	{
	"webcam_maxrate",
	"#Maximum framerate for webcam streams (default: 100)",
	CONF_OFFSET(webcam_maxrate),
	copy_int,
	print_int
	},
	{
	"webcam_localhost",
	"#Restrict webcam connections to localhost only (default: on)",
	CONF_OFFSET(webcam_localhost),
	copy_bool,
	print_bool
	},
	{
	"webcam_limit",
	"#Limits the number of images per connection (default: 0 = unlimited)\n"
	"#Number can be defined by multiplying actual webcam rate by desired number of seconds\n"
	"#Actual webcam rate is the smallest of the numbers framerate and webcam_maxrate",
	CONF_OFFSET(webcam_limit),
	copy_int,
	print_int
	},
	{
	"control_port",
	"\n############################################################\n"
	"# XML-RPC Based Control\n"
	"############################################################\n\n"
	"#TCP/IP port for the xml-rpc server to listen on (default: 0 = disabled)",
	CONF_OFFSET(control_port),
	copy_int,
	print_int
	},
	{
	"control_localhost",
	"#Restrict control connections to localhost only (default: on)",
	CONF_OFFSET(control_localhost),
	copy_bool,
	print_bool
	},

	{
	"track_type",
	"\n############################################################\n"
	"# Tracking (Pan/Tilt)\n"
	"############################################################\n\n"
	"#Type of tracker (0=none (default), 1=stepper, 2=iomojo, 3=pwc)",
	TRACK_OFFSET(type),
	copy_int,
	print_int
	},
	{
	"track_port",
	"#Serial port of motor (default: none)",
	TRACK_OFFSET(port),
	copy_string,
	print_string
	},
	{
	"track_motorx",
	"#Motor number for x-axis (default: -1)",
	TRACK_OFFSET(motorx),
	copy_int,
	print_int
	},
	{
	"track_maxx",
	"#Maximum value on x-axis (default: 0)",
	TRACK_OFFSET(maxx),
	copy_int,
	print_int
	},
	{
	"track_iomojo_id",
	"#ID of an iomojo camera if used (default: 0)",
	TRACK_OFFSET(iomojo_id),
	copy_int,
	print_int
	},
	{
	"track_speed",
	"#Speed to set the motor to (default: 255)",
	TRACK_OFFSET(speed),
	copy_int,
	print_int
	},
	{
	"track_stepsize",
	"#Number of steps to make (default: 40)",
	TRACK_OFFSET(stepsize),
	copy_int,
	print_int
	},

	{
	"quiet",
	"\n############################################################\n"
	"# External Commands, Warnings and Logging:\n"
	"############################################################\n\n"
	"#Output less information and don't sound beeps when detecting motion (default: off)",
	CONF_OFFSET(quiet),
	copy_bool,
	print_bool
	},
	{
	"always_changes",
	"#Always display the difference between captured and reference frame (default: off)",
	CONF_OFFSET(always_diff),
	copy_bool,
	print_bool
	},
	{
	"mail",
	"#Email address to send a warning to when detecting motion (default: none)",
	CONF_OFFSET(mail_address),
	copy_string,
	print_string
	},
	{
	"execute",
	"#External command to be executed when detecting motion (default: none)",
	CONF_OFFSET(externcommand),
	copy_string,
	print_string
	},
	{
	"sms",
	"#Number to send an sms to with sms_client (default: none)",
	CONF_OFFSET(sms_nr),
	copy_string,
	print_string
	},
	{
	"onsave",
	"#Command to be executed each time an image is saved (default: none)",
	CONF_OFFSET(onsavecommand),
	copy_string,
	print_string
	},
	{
	"onmpeg",
	"#Command to be executed each time an mpeg movie is created (default: none)",
	CONF_OFFSET(onmpegcommand),
	copy_string,
	print_string
	},
#ifdef HAVE_FFMPEG
	{
	"onffmpegclose",
	"#Command to be executed each time a file generated by ffmpeg is closed (default: none)",
	CONF_OFFSET(onffmpegclosecommand),
	copy_string,
	print_string
	},
#endif /* HAVE_FFMPEG */

#if defined(HAVE_MYSQL) || defined(HAVE_PGSQL)
	{
	"sql_log_image",
	"\n############################################################\n"
	"# Common Options For MySQL and PostgreSQL database features.\n"
	"# Options require the MySQL/PostgreSQL options to be active also.\n"
	"############################################################\n\n"
	"#Log to the database when creating motion triggered image file  (default: on)",
	CONF_OFFSET(sql_log_image),
	copy_bool,
	print_bool
	},
	{
	"sql_log_snapshot",
	"#Log to the database when creating a snapshot image file (default: on)",
	CONF_OFFSET(sql_log_snapshot),
	copy_bool,
	print_bool
	},
	{
	"sql_log_mpeg",
	"#Log to the database when creating motion triggered mpeg file (default: off)",
	CONF_OFFSET(sql_log_mpeg),
	copy_bool,
	print_bool
	},
	{
	"sql_log_timelapse",
	"#Log to the database when creating timelapse mpeg file (default: off)",
	CONF_OFFSET(sql_log_timelapse),
	copy_bool,
	print_bool
	},	
	{
	"sql_log_prediction",
	"#Log to the database when creating a prediction file (default: off)",
	CONF_OFFSET(sql_log_prediction),
	copy_bool,
	print_bool
	},	
#endif /* defined(HAVE_MYSQL) || defined(HAVE_PGSQL) */

#ifdef HAVE_MYSQL
	{
	"mysql_db",
	"\n############################################################\n"
	"# Database Options For MySQL\n"
	"############################################################\n\n"
	"#Mysql database to log to (default: not defined)",
	CONF_OFFSET(mysql_db),
	copy_string,
	print_string
	},
	{
	"mysql_host",
	"#The host on which the database is located (default: not defined)",
	CONF_OFFSET(mysql_host),
	copy_string,
	print_string
	},
	{
	"mysql_user",
	"#User account name for MySQL database (default: not defined)",
	CONF_OFFSET(mysql_user),
	copy_string,
	print_string
	},
	{
	"mysql_password",
	"#User password for MySQL database (default: not defined)",
	CONF_OFFSET(mysql_password),
	copy_string,
	print_string
	},
#endif /* HAVE_MYSQL */

#ifdef HAVE_PGSQL
	{
	"pgsql_db",
	"\n############################################################\n"
	"# Database Options For PostgreSQL\n"
	"############################################################\n\n"
	"#PostgreSQL database to log to (default: not defined)",
	CONF_OFFSET(pgsql_db),
	copy_string,
	print_string
	},
	{
	"pgsql_host",
	"#The host on which the database is located (default: not defined)",
	CONF_OFFSET(pgsql_host),
	copy_string,
	print_string
	},
	{
	"pgsql_user",
	"#User account name for PostgreSQL database (default: not defined)",
	CONF_OFFSET(pgsql_user),
	copy_string,
	print_string
	},
	{
	"pgsql_password",
	"#User password for PostgreSQL database (default: not defined)",
	CONF_OFFSET(pgsql_password),
	copy_string,
	print_string
	},
	{
	"pgsql_port",
	"#Port on which the PostgreSQL database is located (default: 5432)",
	CONF_OFFSET(pgsql_port),
	copy_int,
	print_int
	},
#endif /* HAVE_PGSQL */
	
	{
	"video_pipe",
	"\n############################################################\n"
	"# Video Loopback Device (vloopback project)\n"
	"############################################################\n\n"
	"#Output images to a video4linux loopback device\n"
	"#The value '-' means next available (default: not defined)",
	CONF_OFFSET(vidpipe),
	copy_string,
	print_string
	},
	{
	"motion_video_pipe",
	"#Output motion images to a video4linux loopback device\n"
	"#The value '-' means next available (default: not defined)",
	CONF_OFFSET(motionvidpipe),
	copy_string,
	print_string
	},

	{
	"predict_enable",
	"\n############################################################\n"
	"# Prediction Feature (experimental and incomplete)\n"
	"############################################################\n\n"
	"#Enable prediction",
	CONF_OFFSET(predict_enable),
	copy_bool,
	print_bool
	},
	{
	"predict_threshold",
	"#Threshold for prediction (default: 90)",
	CONF_OFFSET(predict_threshold),
	copy_int,
	print_int
	},
	{
	"predict_description (default: not defined)",
	"#Description for prediction",
	0,
	load_desc,
	print_desc
	},

	{
	"debug_parameter",
	"\n############################################################\n"
	"# Debugging for Programmers - Ignore\n"
	"# Thread Config Files - One for each camera\n"
	"# If Only One Camera - Use default values in this file\n"
	"############################################################\n\n"
	"#Debug option for programmers - not normally used",
	CONF_OFFSET(debug_parameter),
	copy_int,
	print_int
	},
	{
	"thread",
	"",
	0,
	config_thread,
	print_thread
	},
	{ NULL, NULL, 0 , NULL, NULL }
};

/* conf_cmdline sets the conf struct options as defined by the command line.
 * Any option already set from a config file are overridden.
 */
static void conf_cmdline (struct context *cnt)
{
	struct config *conf=&cnt->conf;
	int c;

	/* For the string options, we free() if necessary and malloc()
	 * if necessary. This is accomplished by calling mystrcpy();
	 * see this function for more information.
	 */
	while ((c=getopt(conf->argc, conf->argv, "a:BCc:d:DE:F:f:G:g:hi:lL:M:mNn:O:P:pQq:r:S:s:t:u:U:V:w"))!=EOF)
		switch (c) {
			case 'a':
				sscanf (optarg, "%d", &conf->snapshot_interval);
				break;
			case 'B':
				conf->mpeg=1;
				break;
			case 'C':
				conf->always_diff=1;
				break;
			case 'c':
				sscanf (optarg, "%d", &conf->max_changes);
				break;
			case 'd':
				conf->video_device = mystrcpy(conf->video_device, optarg);
				break;
			case 'E':
				conf->externcommand = mystrcpy(conf->externcommand, optarg);
				break;
			case 'D':
				cnt->daemon=1;
				break;
			case 'F':
				conf->mask_file = mystrcpy(conf->mask_file, optarg);
				break;
			case 'f':
				sscanf (optarg, "%d", &conf->frame_limit);
				break;
			case 'G':
				sscanf (optarg, "%d", &conf->mingap);
				break;
			case 'g':
				sscanf (optarg, "%d", &conf->gap);
				break;
			case 'i':
				sscanf (optarg, "%d", &conf->input);
				break;
			case 'l':
				conf->locate=0;
				break;
			case 'L':
				sscanf (optarg, "%d", &conf->noise);
				break;
			case 'M':
				conf->mail_address = mystrcpy(conf->mail_address, optarg);
				break;
			case 'm':
				conf->motion_img=1;
				break;
			case 'N':
				conf->new_img=0;
				break;
			case 'n':
				if (!strcmp(optarg, "pal")) conf->norm=NORM_PAL;
				if (!strcmp(optarg, "pal-nc")) conf->norm=NORM_PAL_NC;
				if (!strcmp(optarg, "ntsc")) conf->norm=NORM_NTSC;
				if (!strcmp(optarg, "secam")) conf->norm=NORM_SECAM;
				break;
			case 'O':
				conf->onsavecommand = mystrcpy(conf->onsavecommand, optarg);
				break;
			case 'P':
				conf->vidpipe = mystrcpy(conf->vidpipe, optarg);
				break;
			case 'p':
				conf->ppm=1;
				break;
			case 'S':
				conf->sms_nr = mystrcpy(conf->sms_nr, optarg);
				break;
			case 'q':
				sscanf (optarg, "%d", &conf->quality);
				break;
			case 'r':
				/* Should already be handled */
				break;
			case 'Q':
				conf->quiet=1;
				break;
			case 's':
				sscanf (optarg, "%dx%d", &conf->width, &conf->height);
				break;
			case 't':
				conf->filepath = mystrcpy(conf->filepath, optarg);
				break;
			case 'u':
				conf->netcam_userpass = mystrcpy(conf->netcam_userpass, optarg);
				break;
			case 'U':
				conf->netcam_url = mystrcpy(conf->netcam_url, optarg);
				break;
			case 'V':
				conf->motionvidpipe = mystrcpy(conf->motionvidpipe, optarg);
				break;
			case 'w':
				conf->lightswitch=1;
				break;
			case 'h':
			/* Do the same as an invallid option, but without the
			 * 'invalid option' error....
			 */
			default:
				usage();
				exit(1);
		}
	optind=1;
}


/* conf_cmdparse sets a config option given by 'cmd' to the value given by 'arg1'.
 * Based on the name of the option it searces through the struct 'config_params'
 * for an option where the config_params[i].param_name matches the option.
 * By calling the function pointed to by config_params[i].copy the option gets
 * assigned.
 */
struct context **conf_cmdparse(struct context **cnt, char *cmd, char *arg1)
{
	int i = 0;

	if(!cmd)
		return cnt;

	/* We search through config_params until we find a param_name that matches
	 * our option given by cmd (or reach the end = NULL)
	 */
	while( config_params[i].param_name != NULL ) {
		if(!strncasecmp(cmd, config_params[i].param_name , 255 + 50)) { // Why +50?
			if(config_params[i].conf_value && !arg1)
				return cnt;
			/* We call the function given by the pointer config_params[i].copy
			 * If the option is a bool, copy_bool is called.
			 * If the option is an int, copy_int is called.
			 * If the option is a string, copy_string is called.
			 * If the option is a thread, config_thread is called.
			 * The arguments to the function are:
			 *  cnt  - a pointer to the context structure
			 *  arg1 - a pointer to the new option value (represented as string)
			 *  config_params[i].conf_value - an integer value which is a pointer
			 *    to the context structure member relative to the pointer cnt.
			 */
			cnt=config_params[i].copy( cnt, arg1, config_params[i].conf_value );
			return cnt;
		}
		i++;
	}

	/* We reached the end of config_params without finding a matching option */
	syslog(LOG_ERR, "Unknown config option: \"%s\"", cmd );

	return cnt;
}

/* conf_process walks through an already open config file line by line
 * Any line starting with '#' or ';' or empty lines are ignored as a comments.
 * Any non empty line is process so that the first word is the name of an option 'cnd'
 * and the rest of the line is the argument 'arg1'
 * White space before the first word, between option and argument and end of the line
 * is discarded. A '=' between option and first word in argument is also discarded.
 * Quotation marks round the argument are also discarded.
 * For each option/argument pair the function conf_cmdparse is called which takes
 * care of assigning the value to the option in the config structures.
 */
static struct context **conf_process(struct context **cnt, FILE *fp)
{
	/* process each line from the config file */
	
	char line[PATH_MAX], *cmd = NULL, *arg1 = NULL;
	char *beg = NULL, *end = NULL;

	while (fgets(line, PATH_MAX-1, fp)) {
		if(!(line[0]=='#' || line[0]==';' || strlen(line)< 2)) {/* skipcomment */
			
			arg1 = NULL;

			/* trim white space and any CR or LF at the end of the line */
			end = line + strlen(line) - 1; /* Point to the last non-null character in the string */
			while (*end == ' ' || *end == '\t' || *end == '\n' || *end == '\r') {
				end--;
			}
			*(end+1) = '\0';
			
			/* If line is only whitespace we continue to the next line */
			if (strlen(line) == 0)
				continue;

			/* trim leading whitespace from the line and find command */
			beg = line;
			while (*beg == ' ' || *beg == '\t') {
				beg++;
			}

			cmd = beg; /* command starts here */

			while (*beg != ' ' && *beg != '\t' && *beg != '=' && *beg != '\0') {
				beg++;
			}
			*beg = '\0'; /* command string terminates here */

			/* trim space between command and argument */
			beg++;

			if (strlen(beg) > 0) {
				while (*beg == ' ' || *beg == '\t' || *beg == '=') {
					beg++;
				}

				/* If argument is in "" we will strip them off
				   It is important that we can use "" so that we can use
				   leading spaces in text_left and text_right */
				if ((beg[0]=='"' && beg[strlen(beg)-1]=='"') ||
				    (beg[0]=='\'' && beg[strlen(beg)-1]=='\'')) {
					beg[strlen(beg)-1]='\0';
					beg++;
				}
				
				arg1 = beg; /* Argument starts here */
			}
			/* else arg1 stays null pointer */

			cnt=conf_cmdparse(cnt, cmd, arg1);
		}
	}

	return cnt;
}


/* conf_print is used to write out the config file(s) motion.conf and any thread
 * config files. The function is called when using XMLRPC.
 */
void conf_print(struct context **cnt)
{
	char *val;
	int i, thread;
	FILE *conffile;

	for (thread=0; cnt[thread]; thread++) {
		syslog(LOG_INFO, "Writing config file to: %s", cnt[thread]->conf_filename);
		conffile=myfopen(cnt[thread]->conf_filename, "w");
		if (!conffile)
			continue;
		fprintf(conffile, "# %s\n", cnt[thread]->conf_filename);
		fprintf(conffile, "#\n# This config file was generated by motion " VERSION "\n");
		fprintf(conffile, "\n\n");
		for (i=0; config_params[i].param_name; i++) {
			val=config_params[i].print(cnt, NULL, i, thread);
			/*If config parameter has a value (not NULL) print it to the config file*/
			if (val) {
				fprintf(conffile, "%s\n", config_params[i].param_help);
				fprintf(conffile, "%s %s\n\n", config_params[i].param_name, val);
			} else {
				config_params[i].print(cnt, &val, i, thread);
				/* It can either be a thread file parameter or a disabled parameter
				   If it is a thread parameter write it out
				   Else write the disabled option to the config file but with a
				   comment mark in front of the parameter name */
				if (val) {
					fprintf(conffile, "%s\n", val);
					free(val);
				} else if (thread==0) {
					fprintf(conffile,"%s\n", config_params[i].param_help);
					fprintf(conffile,"; %s value\n\n", config_params[i].param_name);
				}
			}
		}
		fprintf(conffile, "\n");
		fclose(conffile);
		conffile=NULL;
	}
}

/**************************************************************************
 * conf_load is the main function, called from motion.c
 * The function sets the important context structure "cnt" including
 * loading the config parameters from config files and command line.
 * The following takes place in the function:
 * - The default start values for cnt stored in the struct conf_template
 *   are copied to cnt[0] which is the default context structure common to
 *   all threads.
 * - All config (cnt.conf) struct members pointing to a string are changed
 *   so that they point to a malloc'ed piece of memory containing a copy of
 *   the string given in conf_template.
 * - motion.conf is opened processed. The process populates the cnt[0] and
 *   for each thread config file it populates a cnt[1], cnt[2]... for each
 *   thread
 * - Finally it process the options given in the command line. This is done
 *   for each thread cnt[i] so that the command line options overrides any
 *   option given by motion.conf or a thread config file.
 **************************************************************************/
struct context ** conf_load (struct context **cnt)
{
	FILE *fp;
	char filename[PATH_MAX];
	int i;
	/* We preserve argc and argv because they get overwritten by the memcpy command */
	char **argv=cnt[0]->conf.argv;
	int argc=cnt[0]->conf.argc;

	/* Copy the template config structure with all the default config values
	 * into cnt[0]->conf
	 */
	memcpy(&cnt[0]->conf, &conf_template, sizeof(struct config));
	
	/* For each member of cnt[0] which is a pointer to a string
	 * if the member points to a string in conf_template and is not NULL
	 * 1. Reserve (malloc) memory for the string
	 * 2. Copy the conf_template given string to the reserved memory
	 * 3. Change the cnt[0] member (char*) pointing to the string in reserved memory
	 * This ensures that we can free and malloc the string when changed
	 * via xmlrpc or config file or command line options
	 */
	malloc_strings(cnt[0]);

	/* Restore the argc and argv */
	cnt[0]->conf.argv=argv;
	cnt[0]->conf.argc=argc;

	/* Open the motion.conf file. We try in this sequence:
	 * 1. current working directory
	 * 2. $HOME/.motion/motion.conf
	 * 3. sysconfig/motion.conf
	 */
	sprintf(filename, "motion.conf");
	fp = fopen (filename, "r");
	if (!fp) {	/* specified file does not exist... try default file */
		sprintf(filename, "%s/.motion/motion.conf", getenv("HOME"));
		fp = fopen (filename, "r");
		if (!fp) {
			sprintf(filename, "%s/motion.conf", sysconfdir);
			fp = fopen (filename, "r");
			if (!fp) {	/* there is no config file.... use defaults */
				syslog(LOG_ERR, "could not open configfile %s: %m", filename);
			}
		}
	}

	/* Now we process the motion.conf config file and close it*/
	if (fp) {
		strcpy(cnt[0]->conf_filename, filename);
		printf("Processing thread 0 - config file %s\n", filename);
		syslog(LOG_INFO, "Processing thread 0 - config file %s\n", filename);
		cnt=conf_process(cnt, fp);
		fclose(fp);
	}
	

	/* For each thread (given by cnt[i] being not null
	 * cnt is an array of pointers to a context type structure
	 * cnt[0] is the default context structure
	 * cnt[1], cnt[2], ... are context structures for each thread
	 * Command line options always wins over config file options
	 * so we go through each thread and overrides any set command line
	 * options
	 */
	i=-1;
	while(cnt[++i])
	{
		conf_cmdline(cnt[i]);

		/* If rotation is 90 or 270 degrees, then swap width and height
		 * in the configuration struct. This way, all functions that rely
		 * on the width and height will get the correct values. The 
		 * functions that fetch video must swap them back, however.
		 */
		if((cnt[i]->conf.rotate_deg == 90) || (cnt[i]->conf.rotate_deg == 270))
			myswap(cnt[i]->conf.width, cnt[i]->conf.height);
	}

	return cnt;
}

/* malloc_strings goes through the members of a context structure.
 * For each context structure member which is a pointer to a string it does this:
 * If the member points to a string and is not NULL
 * 1. Reserve (malloc) memory for the string
 * 2. Copy the original string to the reserved memory
 * 3. Change the cnt member (char*) pointing to the string in reserved memory
 * This ensures that we can free and malloc the string if it is later changed
 */
void malloc_strings (struct context * cnt)
{
	int i = 0;
	void *val;
	while( config_params[i].param_name != NULL ) {
		if (config_params[i].copy == copy_string) { /* if member is a string */
			/* val is made to point to a pointer to the current string */
			val=(void *)cnt+config_params[i].conf_value;
			
			/*printf("config_params[i].param_name is string: %s\n",config_params[i].param_name);
			printf("string value: %s\n",*(char **)val);
			printf("string address: %p\n",*(char **)val);*/

			/* if there is a string, malloc() space for it, copy
			 * the string to new space, and point to the new
			 * string. we don't free() because we're copying a
			 * static string.
			 */
			*(char **)val = mystrdup(*(char **)val);
		}
		i++;
	}
}

/************************************************************************
 * copy functions
 *
 *   copy_bool   - convert a bool representation to int
 *   copy_int    - convert a string to int
 *   copy_string - just a string copy
 *
 * @param str     - A char *, pointing to a string representation of the
 *                  value.
 * @param val_ptr - points to the place where to store the value relative
 *                  to pointer pointing to the given context structure
 * @cnt           - points to a context structure for a thread
 *
 * The function is given a pointer cnt to a context structure and a pointer val_ptr
 * which is an integer giving the position of the structure member relative to the
 * pointer of the context structure.
 * If the context structure is for thread 0 (cnt[0]->threadnr is zero) then the
 * function also sets the value for all the child threads since thread 0 is the
 * global thread.
 * If the thread given belongs to a child thread (cnt[0]->threadnr is not zero)
 * the function will only assign the value for the given thread.
 ***********************************************************************/

/* copy_bool assigns a config option to a new boolean value.
 * The boolean is given as a string in str which is coverted to 0 or 1 
 * by the function. Values 1, yes and on are converted to 1 ignoring case.
 * Any other value is converted to 0.
 */
static struct context **copy_bool (struct context **cnt, char *str, int val_ptr)
{
	void *tmp;
	int i;

	i=-1;
	while(cnt[++i]) {
		tmp=(void *)cnt[i]+(int)val_ptr;
		if ( !strcmp(str, "1") ||
		     !strcasecmp(str, "yes") ||
		     !strcasecmp(str,"on"))
		{
			*((int *)tmp) = 1;
		} else {
			*((int *)tmp) = 0;
		}
		if (cnt[0]->threadnr)
			return cnt;
	}
	return cnt;
}

/* copy_int assigns a config option to a new interger value.
 * The integer is given as a string in str which is coverted to integer by the function.
 */
static struct context ** copy_int(struct context **cnt, char *str, int val_ptr)
{
	void *tmp;
	int i;

	i=-1;
	while(cnt[++i]) {
		tmp=(void *)cnt[i]+val_ptr;
		*((int *)tmp) = atoi(str);
		if (cnt[0]->threadnr)
			return cnt;
	}
	return cnt;
}

/* copy_string assigns a config option to a new interger value.
 * Strings are handled differently from bool and int.
 * the char *conf->option that we are working on is free()'d
 * (if memory for it has already been malloc()'d), and set to
 * a freshly malloc()'d string with the value from str,
 * or NULL if str is blank
 */
struct context **copy_string(struct context **cnt, char *str, int val_ptr)
{
	void *tmp;
	int i;

	i=-1;
	while(cnt[++i]) {
		tmp=(void *)cnt[i]+val_ptr;

		/* mystrcpy assigns the new string value
		 * including free'ing and reserving new memory for it.
		 */
		*(char **)tmp = mystrcpy(*(char **)tmp, str);

		/* set the option on all threads if setting the option
		 * for thread 0; otherwise just set that one thread's option
		 */
		if (cnt[0]->threadnr)
			return cnt;
	}
	return cnt;
}


/* mystrcpy is used to assign string type fields (e.g. config options)
 * In a way so that we the memory is malloc'ed to fit the string.
 * If a field is already pointing to a string (not NULL) the memory of the
 * old string is free'd and new memory is malloc'ed and filled with the
 * new string is copied into the the memory and with the char pointer
 * pointing to the new string.
 *
 * from - pointer to the new string we want to copy
 * to   - the pointer to the current string (or pointing to NULL)
 *        If not NULL the memory it points to is fre'ed.
 * function returns pointer to the new string which is in malloc'ed memory
 * FIXME The strings that are malloc'ed with this function should be freed
 * when the motion program is terminated normally instead of relying on the
 * OS to clean up.
 */
char *mystrcpy(char *to, char *from)
{
	/* free the memory used by the to string, if such memory exists,
	 * and return a pointer to a freshly malloc()'d string with the
	 * same value as from.
	 */

	if (to != NULL) 
		free(to);

	return mystrdup(from);
}


/* mystrdup return a pointer to a freshly malloc()'d string with the same
 * value as the string that the input parameter 'from' points to,
 * or NULL if the from string is 0 characters.
 * The function truncates the string to the length given by the environment
 * variable PATH_MAX to ensure that config options can always contain
 * a really long path but no more than that.
 */
char *mystrdup(char *from)
{
	char *tmp;
	int stringlength;

	if (from == NULL || !strlen(from)) {
		tmp = NULL;
	} else {
		stringlength = strlen(from);
		stringlength = (stringlength < PATH_MAX ? stringlength : PATH_MAX);
		tmp = (char *)mymalloc(stringlength + 1);
		strncpy(tmp, from, stringlength);

		/* We must ensure the string always has a NULL terminator.
		 * This necessary because strncpy will not append a NULL terminator
		 * if the original string is greater than stringlength.
		 */
		tmp += stringlength;
		*tmp = '\0';
		tmp -= stringlength;
	}
	return tmp;
}

char *config_type(config_param *configparam)
{
	if (configparam->copy==copy_string)
		return "string";
	if (configparam->copy==copy_int)
		return "int";
	if (configparam->copy==copy_bool)
		return "bool";
	return "unknown";
}

static char *print_bool(struct context **cnt, char **str, int parm, int threadnr)
{
	int val=config_params[parm].conf_value;

	if (threadnr &&
	    *(int*)((void*)cnt[threadnr]+(int)val)==
	    *(int*)((void*)cnt[0]+(int)val))
		return NULL;

	if (*(int*)((void*)cnt[threadnr]+(int)val))
		return "on";
	else
		return "off";
}

/* print_string returns a pointer to a string containing the value of the config option
 * If the option is not defined NULL is returned.
 * If the thread number is not 0 the string is compared with the value of the same
 * option in thread 0. If the value is the same, NULL is returned which means that
 * the option is not written to the thread config file.
 */
static char *print_string(struct context **cnt, char **str, int parm, int threadnr)
{
	int val=config_params[parm].conf_value;
	
	/* strcmp does not like NULL so we have to check for this also */
	if (threadnr && *(char **)((void*)cnt[threadnr]+(int)val)!=NULL &&
	    *(char **)((void*)cnt[0]+(int)val)!=NULL &&	    
	    !strcmp(*(char **)((void*)cnt[threadnr]+(int)val), *(char **)((void*)cnt[0]+(int)val)))
		return NULL;

	return *(char **)((void*)cnt[threadnr]+(int)val);
}

static char *print_int(struct context **cnt, char **str, int parm, int threadnr)
{
	static char retval[20];
	int val=config_params[parm].conf_value;

	if (threadnr &&
	    *(int*)((void*)cnt[threadnr]+(int)val)==
	    *(int*)((void*)cnt[0]+(int)val))
		return NULL;

	sprintf(retval, "%d", *(int*)((void*)cnt[threadnr]+(int)val));

	return retval;
}

static char *print_thread(struct context **cnt, char **str, int parm, int threadnr)
{
	char *retval=mymalloc(1);
	int i=0;

	if (!str || threadnr || !retval)
		return NULL;

	retval[0]=0;
	while (cnt[++i]) {
		retval=myrealloc(retval, strlen(retval)+strlen(cnt[i]->conf_filename)+10, "print_thread");
		sprintf(retval+strlen(retval), "thread %s\n", cnt[i]->conf_filename);
	}
	*str=retval;

	return NULL;
}

static char *print_desc(struct context **cnt, char **str, int parm, int threadnr)
{
	char *retval=mymalloc(1);
	struct predict_set *tmp;

	if (!str || !retval)
		return NULL;
	if (!threadnr && cnt[1]!=NULL)
		return NULL;

	retval[0]=0;
	for (tmp=cnt[threadnr]->predict_sets; tmp; tmp=tmp->next) {
		retval=myrealloc(retval, strlen(retval)+strlen(tmp->filename)+22, "print_desc");
		sprintf(retval+strlen(retval), "predict_description %s\n", tmp->filename);
	}
	*str=retval;

	return NULL;
}

/* config_thread() is called during initial config file loading each time Motion
 * finds a thread option in motion.conf
 * The size of the context array is increased and the main context's values are
 * copied to the new thread.
 *
 * cnt  - pointer to the array of pointers pointing to the context structures
 * str  - pointer to a string which is the filename of the thread config file
 * val  - is not used. It is defined to be function header compatible with
 *        copy_int, copy_bool and copy_string.
 */
static struct context **config_thread(struct context **cnt, char *str, int val)
{
	int i;
	FILE *fp;
	
	if (cnt[0]->threadnr)
		return cnt;

	fp=fopen(str, "r");
	if (!fp) {
		syslog(LOG_ERR, "Thread config file %s not found: %m", str);
		return cnt;
	}

	/* Find the current number of threads defined. */
	i=-1;
	while (cnt[++i]);

	/* Make space for the threads + the terminating NULL pointer
	 * in the array of pointers to context structures
	 * First thread is 0 so the number of threads is i+1
	 * plus an extra for the NULL pointer. This gives i+2
	 */
	cnt=myrealloc(cnt, sizeof(struct context *)*(i+2), "config_thread");

	/* Now malloc space for an additional context structure for thread nr. i */
	cnt[i]=mymalloc(sizeof(struct context));
	
	/* And make this an exact clone of the context structure for thread 0 */
	memcpy(cnt[i], cnt[0], sizeof(struct context));

	/* All the integers are copies of the actual value.
	 * The strings are all pointers to strings so we need to create
	 * unique malloc'ed space for all the strings that are not NULL and
	 * change the string pointers to point to the new strings.
	 * malloc_strings takes care of this.
	 */
	malloc_strings(cnt[i]);
	
	/* Set the current thread number */
	cnt[i]->threadnr=i;
	
	/* Mark the end if the array of pointers to context structures */
	cnt[i+1]=NULL;

	/* process the thread's config file and notify user on console */
	strcpy(cnt[i]->conf_filename, str);
	printf("Processing thread %d - config file %s\n", i, str);
	syslog(LOG_INFO, "Processing thread %d - config file %s\n", i, str);
	conf_process(cnt+i, fp);
	
	/* Finally we close the thread config file */
	fclose(fp);

	return cnt;
}

static struct context **load_desc(struct context **cnt, char *str, int val)
{
	struct predict_set *newset;
	char line[256];
	FILE *fp;
	int i;

	fp=fopen(str, "r");
	if (!fp) {
		syslog(LOG_ERR, "Couldn't open %s: %m", str);
		return cnt;
	}

	newset=mymalloc(sizeof(struct predict_set));
	memset(newset, 0x00, sizeof(struct predict_set));

	if (fp) {
		if (fgets(line, 256, fp)) {
			stripnewline(line);
			newset->name=mymalloc(strlen(line)+1);
			strcpy(newset->name, line);
			newset->filename=mymalloc(strlen(str)+1);
			strcpy(newset->filename, str);
		}
		if (fgets(line, 256, fp)) {
			sscanf(line, "%d\t%d\t%d\t%d\t%d\t%d\t%d\t%d\n",
			    &newset->min.length,
			    &newset->min.x,
			    &newset->min.y,
			    &newset->min.width,
			    &newset->min.height,
			    &newset->min.diffs,
			    &newset->min.dx,
			    &newset->min.dy
			);
		}
		if (fgets(line, 256, fp)) {
			sscanf(line, "%d\t%d\t%d\t%d\t%d\t%d\t%d\t%d\n",
			    &newset->max.length,
			    &newset->max.x,
			    &newset->max.y,
			    &newset->max.width,
			    &newset->max.height,
			    &newset->max.diffs,
			    &newset->max.dx,
			    &newset->max.dy
			);
		}
		fclose(fp);
		i=-1;
		while (cnt[++i]) {
			struct predict_set *tmpset=cnt[i]->predict_sets;
			cnt[i]->predict_sets=mymalloc(sizeof(struct predict_set));
			memcpy(cnt[i]->predict_sets, newset, sizeof(struct predict_set));
			cnt[i]->predict_sets->next=tmpset;
		}
		free(newset);
	}
	return cnt;
}

static void usage (void)
{
	printf("motion Version "VERSION", Copyright 2000 Jeroen Vreeken\n");
	printf("\nusage:\tmotion [options]\n");
	printf("\n");
	printf("Overall settings:\n");
	printf("-C\t\tOutput changes count for every frame, usable for tuning\n");
	printf("-D\t\tDeamonize\n");
	printf("-a seconds\ttime between two automated snapshots, default: 0\n");
	printf("-c changes\tthreshold for detecting a change, default: %d\n", DEF_CHANGES);
	printf("-F file\t\tppm mask file to use as a motion filter.\n");
	printf("-f nr\t\tMaximum nr of frames per second, default: none\n");
	printf("-G sec\t\tMinimum gap between two shots in seconds, default: 0\n");
	printf("-L noise\tNoise level, default: %d\n", DEF_NOISELEVEL);
	printf("-l\t\tDo not locate and mark motion on output pictures\n");
	printf("-m\t\tOutput 'motion' images\n");
	printf("-N\t\tDon't output normal images\n");
	printf("-p\t\tOutput ppm images instead of jpeg, uses a lot of diskspace\n");
	printf("-t target dir\tdestination for snapshots.\n");
	printf("-Q\t\tBe quiet, don't output beeps when detecting motion\n");
	printf("-w\t\tActivate light switch filter\n");
	printf("\nDevice settings:\n");
	printf("-d device\tvideo4linux capture device, default: "VIDEO_DEVICE"\n");
	printf("-i input\tinput channel to use, default: not set\n");
	printf("-n norm\t\tnorm to use (pal/ntsc/secam), default: pal\n");
	printf("-P device\tvideo4linux video loopback input device, default: not set\n");
	printf("-q quality\tJPEG image quality, default: %d\n", DEF_QUALITY);
	printf("-s widthxheight\tPicture size, default: %dx%d\n", DEF_WIDTH, DEF_HEIGHT);
#ifdef HAVE_CURL
	printf("-u user:pass\tNetwork camera username and password, default: none\n");
	printf("-U url\t\tNetwork camera url for download, default: none\n");
#endif
	printf("-V device\tvideo4linux video loopback input device for motion frames,\n");
	printf("\t\tdefault: not set.");
	printf("\nActions:\n");
	printf("-B\t\tEncode all jpg images to mpeg after an event using mpeg_encode\n");
	printf("-E command\tExecute 'command' when detecting motion.\n");
	printf("-M address\tSend a mail to 'address' when detecting motion.\n");
	printf("-O command\tExecute 'command' when saving an image.\n");
	printf("-S nr\t\tSend a SMS to nr using sms_client when detecting motion.\n");
	printf("-g seconds\tminimum gap between events, default %d\n", DEF_GAP);
	printf("\n");
	printf("Will read motion.conf from current directory, ~ and %s.\n", sysconfdir);
	printf("\n");
}
