/*	video.c
 *
 *	Video stream functions for motion.
 *	Copyright 2000 by Jeroen Vreeken (pe1rxq@amsat.org)
 *	This software is distributed under the GNU public license version 2
 *	See also the file 'COPYING'.
 *
 */

/* Common stuff: */
#include "motion.h"
#include "video.h"
#include "conf.h"

/* for the v4l stuff: */
#include "sys/mman.h"
#include "sys/ioctl.h"

void yuv422to420p(unsigned char *map, unsigned char *cap_map, int width, int height)
{
	unsigned char *src, *dest, *src2, *dest2;
	int i, j;

	/* Create the Y plane */
	src=cap_map;
	dest=map;
	for (i=width*height; i; i--) {
		*dest++=*src;
		src+=2;
	}
	/* Create U and V planes */
	src=cap_map+1;
	src2=cap_map+width*2+1;
	dest=map+width*height;
	dest2=dest+(width*height)/4;
	for (i=height/2; i; i--) {
		for (j=width/2; j; j--) {
			*dest=((int)*src+(int)*src2)/2;
			src+=2;
			src2+=2;
			dest++;
			*dest2=((int)*src+(int)*src2)/2;
			src+=2;
			src2+=2;
			dest2++;
		}
		src+=width*2;
		src2+=width*2;
	}
}


/*******************************************************************************************
	Video4linux capture routines
*/


static char *v4l_start (struct video_dev *viddev, int width, int height, int input, int norm, unsigned long freq)
{
	int dev=viddev->fd;
        struct video_capability vid_caps;
	struct video_channel vid_chnl;
	struct video_tuner vid_tuner;
	struct video_window vid_win;
	struct video_mbuf vid_buf;
	struct video_mmap vid_mmap;
	char *map;

	if (ioctl (dev, VIDIOCGCAP, &vid_caps) == -1) {
		perror ("ioctl (VIDIOCGCAP)");
		return (NULL);
	}
	if (vid_caps.type & VID_TYPE_MONOCHROME) viddev->v4l_fmt=VIDEO_PALETTE_GREY;
        if (input != IN_DEFAULT) {
                vid_chnl.channel = -1;
                if (ioctl (dev, VIDIOCGCHAN, &vid_chnl) == -1) {
                        perror ("ioctl (VIDIOCGCHAN)");
                } else {
                        vid_chnl.channel = input;
                        vid_chnl.norm    = norm;
                        if (ioctl (dev, VIDIOCSCHAN, &vid_chnl) == -1) {
                                perror ("ioctl (VIDIOCSCHAN)");
                                return (NULL);
                        }
                }
        }
	if (freq) {
		if (ioctl (dev, VIDIOCGTUNER, &vid_tuner)==-1) {
			perror("ioctl (VIDIOCGTUNER)");
		} else {
			if (vid_chnl.channel & VIDEO_TUNER_LOW) {
				freq=freq*16; /* steps of 1/16 KHz */
			} else {
				freq=(freq*10)/625;
			}
			if (ioctl(dev, VIDIOCSFREQ, &freq)==-1) {
				perror("ioctl (VIDIOCSFREQ)");
				return (NULL);
			}
			printf("Frequency set\n");
		}
	}
	if (ioctl (dev, VIDIOCGMBUF, &vid_buf) == -1) {
		fprintf(stderr, "\tno mmap falling back on read\n");
		if (ioctl (dev, VIDIOCGWIN, &vid_win)== -1) {
			perror ("ioctl VIDIOCGWIN");
			return (NULL);
		}
		vid_win.width=width;
		vid_win.height=height;
		vid_win.clipcount=0;
		if (ioctl (dev, VIDIOCSWIN, &vid_win)== -1) {
			perror ("ioctl VIDIOCSWIN");
			return (NULL);
		}
		viddev->v4l_read_img=1;
		map=malloc(width*height*3);
		viddev->v4l_maxbuffer=1;
		viddev->v4l_curbuffer=0;
		viddev->v4l_buffers[0]=map;
		goto done;
	}
	map=mmap(0, vid_buf.size, PROT_READ|PROT_WRITE, MAP_SHARED, dev, 0);
	if (vid_buf.frames>1) {
		viddev->v4l_maxbuffer=2;
		viddev->v4l_buffers[0]=map;
		viddev->v4l_buffers[1]=map+vid_buf.offsets[1];
	} else {
		viddev->v4l_buffers[0]=map;
		viddev->v4l_maxbuffer=1;
	}

	if ((unsigned char *)-1 == (unsigned char *)map) {
		return (NULL);
	}
	viddev->v4l_curbuffer=0;
	vid_mmap.format=viddev->v4l_fmt;
	vid_mmap.frame=viddev->v4l_curbuffer;
	vid_mmap.width=width;
	vid_mmap.height=height;
	if (ioctl(dev, VIDIOCMCAPTURE, &vid_mmap) == -1) {
		printf("Failed with YUV420P, trying YUV422 palette\n");
		viddev->v4l_fmt=VIDEO_PALETTE_YUV422;
		vid_mmap.format=viddev->v4l_fmt;
		/* Try again... */
		if (ioctl(dev, VIDIOCMCAPTURE, &vid_mmap) == -1) {
			printf("Failed with YUV422, trying RGB24 palette\n");
			viddev->v4l_fmt=VIDEO_PALETTE_RGB24;
			vid_mmap.format=viddev->v4l_fmt;
			/* Try again... */
			if (ioctl(dev, VIDIOCMCAPTURE, &vid_mmap) == -1) {
				perror("ioctl VIDIOCMCAPTURE");
				return (NULL);
			}
		}
	}
done:
	switch (viddev->v4l_fmt) {
		case VIDEO_PALETTE_YUV420P:
			viddev->v4l_bufsize=(width*height*3)/2;
			break;
		case VIDEO_PALETTE_YUV422:
			viddev->v4l_bufsize=(width*height*2);
			break;
		case VIDEO_PALETTE_RGB24:
			viddev->v4l_bufsize=(width*height*3);
			break;
		case VIDEO_PALETTE_GREY:
			viddev->v4l_bufsize=width*height;
			break;
	}
	return map;
}

static char *v4l_next (struct video_dev *viddev, char *map, int width, int height)
{
	int dev=viddev->fd;
	int frame=viddev->v4l_curbuffer;
	struct video_mmap vid_mmap;
	char *cap_map=NULL;

    	sigset_t    set, old;

	if (viddev->v4l_read_img) {
		if (read(dev, map, viddev->v4l_bufsize) != viddev->v4l_bufsize)
			return NULL;
	} else {
		vid_mmap.format=viddev->v4l_fmt;
		vid_mmap.width=width;
		vid_mmap.height=height;
    		/* Block signals during IOCTL */
		sigemptyset (&set);
		sigaddset (&set, SIGCHLD);
		sigaddset (&set, SIGALRM);
		sigaddset (&set, SIGUSR1);
		sigaddset (&set, SIGTERM);
		sigaddset (&set, SIGHUP);
		pthread_sigmask (SIG_BLOCK, &set, &old);

		cap_map=viddev->v4l_buffers[viddev->v4l_curbuffer];
		viddev->v4l_curbuffer++;
		if (viddev->v4l_curbuffer >= viddev->v4l_maxbuffer)
			viddev->v4l_curbuffer=0;
		vid_mmap.frame=viddev->v4l_curbuffer;

		if (ioctl(dev, VIDIOCMCAPTURE, &vid_mmap) == -1) {
	        	sigprocmask (SIG_UNBLOCK, &old, NULL);
			printf("mcatpure error %d %d %d\n", errno, EAGAIN, getpid());
			perror("mcapture");
			return (NULL);
		}

		vid_mmap.frame=frame;
		if (ioctl(dev, VIDIOCSYNC, &vid_mmap.frame) == -1) {
	        	sigprocmask (SIG_UNBLOCK, &old, NULL);
			printf("sync error %d\n", getpid());
			perror("sync");
		}
		
        	pthread_sigmask (SIG_UNBLOCK, &old, NULL);	/*undo the signal blocking*/
	}
	if (!viddev->v4l_read_img) {
		if (viddev->v4l_fmt==VIDEO_PALETTE_YUV422) {
			yuv422to420p(map, cap_map, width, height);
		} else
			memcpy(map, cap_map, viddev->v4l_bufsize);
	}

	return map;
}

void v4l_set_input (struct video_dev *viddev, char *map, int width, int height, int input, int norm, int skip, long freq)
{
	int dev=viddev->fd;
	int i;
	struct video_channel vid_chnl;
	struct video_tuner vid_tuner;

        if (input != viddev->input || width != viddev->width || height!=viddev->height || freq!=viddev->freq) {
		if (freq) {
			if (ioctl (dev, VIDIOCGTUNER, &vid_tuner)==-1) {
				perror("ioctl (VIDIOCGTUNER)");
			} else {
				if (vid_chnl.channel & VIDEO_TUNER_LOW) {
					freq=freq*16; /* steps of 1/16 KHz */
				} else {
					freq=(freq*10)/625;
				}
				if (ioctl(dev, VIDIOCSFREQ, &freq)==-1) {
					perror("ioctl (VIDIOCSFREQ)");
					return;
				}
			}
		}
                vid_chnl.channel = -1;
                if (ioctl (dev, VIDIOCGCHAN, &vid_chnl) == -1) {
                        perror ("ioctl (VIDIOCGCHAN)");
                } else {
			printf("input: %d\n", input);
                        vid_chnl.channel = input;
			vid_chnl.norm = norm;
                        if (ioctl (dev, VIDIOCSCHAN, &vid_chnl) == -1) {
                                perror ("ioctl (VIDIOCSCHAN)");
                                return;
                        }
                }
		viddev->input=input;
		viddev->width=width;
		viddev->height=height;
		/* skip a few frames if needed */
		for (i=0; i<skip; i++)
			v4l_next (viddev, map, width, height);
        }
}

static int v4l_open_vidpipe(void)
{
	int pipe;
	FILE *vloopbacks;
	char pipepath[255];
	char buffer[255];
	char *loop;
	char *input;
	char *istatus;
	char *output;
	char *ostatus;
	
	vloopbacks=fopen("/proc/video/vloopback/vloopbacks", "r");
	if (!vloopbacks) {
		perror ("Failed to open '/proc/video/vloopback/vloopbacks");
		return -1;
	}
	/* Read vloopback version*/
	fgets(buffer, 255, vloopbacks);
	printf("\t%s", buffer);
	/* Read explaination line */
	fgets(buffer, 255, vloopbacks);
	while (fgets(buffer, 255, vloopbacks)) {
		if (strlen(buffer)>1) {
			buffer[strlen(buffer)-1]=0;
			loop=strtok(buffer, "\t");
			input=strtok(NULL, "\t");
			istatus=strtok(NULL, "\t");
			output=strtok(NULL, "\t");
			ostatus=strtok(NULL, "\t");
			if (istatus[0]=='-') {
				sprintf(pipepath, "/dev/%s", input);
				pipe=open(pipepath, O_RDWR);
				if (pipe>=0) {
					printf("\tInput: /dev/%s\n", input);
					printf("\tOutput: /dev/%s\n", output);
					return pipe;
				}
			}
		}
	}
	return -1;
}

static int v4l_startpipe (char *devname, int width, int height, int type)
{
	int dev;
	struct video_picture vid_pic;
	struct video_window vid_win;

	if (!strcmp(devname, "-")) {
		dev=v4l_open_vidpipe();
	} else {
		dev=open(devname, O_RDWR);
	}
	if (dev < 0)
		return(-1);

	if (ioctl(dev, VIDIOCGPICT, &vid_pic)== -1) {
		perror("ioctl VIDIOCGPICT");
		return(-1);
	}
	vid_pic.palette=type;
	if (ioctl(dev, VIDIOCSPICT, &vid_pic)== -1) {
		perror("ioctl VIDIOCSPICT");
		return(-1);
	}
	if (ioctl(dev, VIDIOCGWIN, &vid_win)== -1) {
		perror("ioctl VIDIOCGWIN");
		return(-1);
	}
	vid_win.height=height;
	vid_win.width=width;
	if (ioctl(dev, VIDIOCSWIN, &vid_win)== -1) {
		perror("ioctl VIDIOCSWIN");
		return(-1);
	}
	return dev;
}

static int v4l_putpipe (int dev, char *image, int size)
{
	return write(dev, image, size);
}

static void v4l_autobright (int dev, unsigned char *image, int height, int width)
{
	struct video_picture vid_pic;
	int i, j=0, avg=0, offset=0;
	
	for (i=0; i<width*height*3; i+=100) {
		avg+=*image++;
		j++;
	}
	avg=avg/j;

	if (avg > 140 || avg < 64) {
		if (ioctl(dev, VIDIOCGPICT, &vid_pic)==-1) {
			perror("ioctl VIDIOCGPICT");
		}
		if (avg > 140) {
			offset=avg-140;
			if (vid_pic.brightness > 100*offset)
				vid_pic.brightness-=100*offset;
		}
		if (avg < 64) {
			offset=64-avg;
			if (vid_pic.brightness < 65535-100*offset)
				vid_pic.brightness+=100*offset;
		}
		printf("auto_brightness: %d\n", vid_pic.brightness);
		if (ioctl(dev, VIDIOCSPICT, &vid_pic)==-1) {
			perror("ioctl VIDIOCSPICT");
		}
	}
}



/*****************************************************************************
	Wrappers calling the actual capture routines
 *****************************************************************************/

/* big lock for vid_start */
pthread_mutex_t vid_mutex;
/* structure used for per device locking */
struct video_dev **viddevs=NULL;

void vid_init(void)
{
	if (!viddevs) {
		viddevs=malloc(sizeof(struct video_dev *));
		viddevs[0]=NULL;
	}

	pthread_mutex_init(&vid_mutex, NULL);
}

/* Called by childs to get rid of open video devices */
void vid_close(void)
{
	int i=-1;

	if (viddevs) {
		while(viddevs[++i]) {
			close(viddevs[i]->fd);
		}
	}
}

void vid_cleanup(void)
{
	int i=-1;
	if (viddevs) {
		while(viddevs[++i])
			free(viddevs[i]);
		free(viddevs);
		viddevs=NULL;
	}
}

int vid_start (struct context *cnt, int width, int height, int input, int norm)
{
	struct config *conf=&cnt->conf;
	int i, dev;

#ifdef HAVE_CURL
	if (conf->netcam_url)
		return netcam_start(cnt);
#endif

	pthread_mutex_lock(&vid_mutex);

	cnt->imgs.width=cnt->conf.width;
	cnt->imgs.height=cnt->conf.height;

	i=-1;
	while (viddevs[++i]) {
		if (!strcmp(conf->device, viddevs[i]->device)) {
			pthread_mutex_unlock(&vid_mutex);
			cnt->imgs.type=viddevs[i]->v4l_fmt;
			switch (cnt->imgs.type) {
				case VIDEO_PALETTE_RGB24:
					cnt->imgs.motionsize=width*height*3;
					cnt->imgs.size=width*height*3;
					break;
				case VIDEO_PALETTE_GREY:
					cnt->imgs.motionsize=width*height;
					cnt->imgs.size=width*height;
					break;
				case VIDEO_PALETTE_YUV422:
					cnt->imgs.type=VIDEO_PALETTE_YUV420P;
				case VIDEO_PALETTE_YUV420P:
					cnt->imgs.motionsize=width*height;
					cnt->imgs.size=(width*height*3)/2;
					break;
			}
			return viddevs[i]->fd;
		}
	}

	viddevs=realloc(viddevs, sizeof(struct video_dev *)*(i+2));
	viddevs[i]=malloc(sizeof(struct video_dev));
	viddevs[i+1]=NULL;

	pthread_mutexattr_init(&viddevs[i]->attr);
	pthread_mutex_init(&viddevs[i]->mutex, NULL);

	dev=open(conf->device, O_RDWR);
	if (dev <0) {
		perror("failed to open video device");
		exit(1);
	}

	viddevs[i]->device=conf->device;
	viddevs[i]->fd=dev;
	viddevs[i]->input=input;
	viddevs[i]->height=height;
	viddevs[i]->width=width;
	viddevs[i]->owner=-1;

	viddevs[i]->v4l_fmt=VIDEO_PALETTE_YUV420P;
	viddevs[i]->v4l_read_img=0;
	viddevs[i]->v4l_curbuffer=0;
	viddevs[i]->v4l_maxbuffer=1;

	if (!v4l_start (viddevs[i], width, height, input, norm, cnt->conf.frequency)) {
		pthread_mutex_unlock(&vid_mutex);
		return -1;
	}
	cnt->imgs.type=viddevs[i]->v4l_fmt;
	switch (cnt->imgs.type) {
		case VIDEO_PALETTE_RGB24:
			cnt->imgs.size=width*height*3;
			cnt->imgs.motionsize=width*height*3;
			break;
		case VIDEO_PALETTE_GREY:
			cnt->imgs.size=width*height;
			cnt->imgs.motionsize=width*height;
			break;
		case VIDEO_PALETTE_YUV422:
			cnt->imgs.type=VIDEO_PALETTE_YUV420P;
		case VIDEO_PALETTE_YUV420P:
			cnt->imgs.size=(width*height*3)/2;
			cnt->imgs.motionsize=width*height;
			break;
	}

	pthread_mutex_unlock(&vid_mutex);
	
	return dev;
}

char *vid_next (struct context *cnt, int dev, char *map, int width, int height)
{
	struct config *conf=&cnt->conf;
	int i;
	char *ret;

#ifdef HAVE_CURL
	if (conf->netcam_url) {
		while (!(ret=netcam_next(cnt))) {
			printf("Error capturing netcam image, wait and retry...\n");
			sleep(1);
		}
		return ret;
	}
#endif

	i=-1;
	while (viddevs[++i])
		if (viddevs[i]->fd==dev)
			break;
	if (!viddevs[i])
		return NULL;

	if (viddevs[i]->owner!=cnt->threadnr) {
		pthread_mutex_lock(&viddevs[i]->mutex);
		viddevs[i]->owner=cnt->threadnr;
		viddevs[i]->frames=conf->roundrobing_frames;
		cnt->switched=1;
	}

	v4l_set_input (viddevs[i], map, width, height, conf->input, conf->norm, conf->roundrobing_skip, conf->frequency);
	ret=v4l_next (viddevs[i], map, width, height);

	if (--viddevs[i]->frames <= 0) {
		viddevs[i]->owner=-1;
		pthread_mutex_unlock(&viddevs[i]->mutex);
	}
	
	return ret;
}

int vid_startpipe (char *devname, int width, int height, int type)
{
	return v4l_startpipe (devname, width, height, type);
}

int vid_putpipe (int dev, char *image, int size)
{
	return v4l_putpipe (dev, image, size);
}

void vid_autobright (struct config *conf, int dev, char *image, int height, int width)
{
	if (!conf->netcam_url) 
		v4l_autobright(dev, image, height, width);
}

char *vid_keepalive(struct context *cnt, struct images *imgs, int dev, int pipe, int mpipe)
{
	struct config *conf=&cnt->conf;

	imgs->new=vid_next(cnt, dev, imgs->new, imgs->width, imgs->height);
	if (imgs->new) {
		memcpy(imgs->ref, imgs->new, imgs->height*imgs->width*3);
		if (conf->vidpipe)
			vid_putpipe(pipe, imgs->new, imgs->size);
		if (conf->motionvidpipe)
			vid_putpipe(mpipe, imgs->out, imgs->size);
	}
	return imgs->new;
}

