/*	picture.c
 *
 *	Various funtions for saving/loading pictures.
 *	Copyright 2002 by Jeroen Vreeken (pe1rxq@amsat.org)
 *	Portions of this file are Copyright by Lionnel Maugis
 *	This software is distributed under the GNU public license version 2
 *	See also the file 'COPYING'.
 *
 */
#include "motion.h"
#include "picture.h"
#include "event.h"

#include <jpeglib.h>

void put_jpeg_bgr24 (FILE *picture, char *image, int width, int height, int quality)
{
	int y, x, line_width;
	JSAMPROW row_ptr[1];
	struct jpeg_compress_struct cjpeg;
	struct jpeg_error_mgr jerr;
	unsigned char *line;

	line=malloc(width *3);
	if (!line)
		return;
	cjpeg.err = jpeg_std_error(&jerr);
	jpeg_create_compress (&cjpeg);
	cjpeg.image_width = width;
	cjpeg.image_height = height;
	cjpeg.input_components = 3;
	cjpeg.in_color_space = JCS_RGB;
	jpeg_set_defaults (&cjpeg);

	jpeg_set_quality (&cjpeg, quality, TRUE);
	cjpeg.dct_method = JDCT_FASTEST;
	jpeg_stdio_dest (&cjpeg, picture);

	jpeg_start_compress (&cjpeg, TRUE);

	row_ptr[0] = line;
	line_width = width * 3;
	for (y=0; y<height; y++) {
		for (x=0; x<line_width; x+=3) {
			line[x]   = image[x+2];
			line[x+1] = image[x+1];
			line[x+2] = image[x];
		}
		jpeg_write_scanlines (&cjpeg, row_ptr, 1);
		image += line_width;
	}
	jpeg_finish_compress (&cjpeg);
	jpeg_destroy_compress (&cjpeg);
	free(line);
}

// twice as fast as RGB jpeg 
void put_jpeg_yuv420p (FILE *fp, unsigned char *image, int width, int height, int quality)
{
	int i,j;

	JSAMPROW y[16],cb[16],cr[16]; // y[2][5] = color sample of row 2 and pixel column 5; (one plane)
	JSAMPARRAY data[3]; // t[0][2][5] = color sample 0 of row 2 and column 5

	struct jpeg_compress_struct cinfo;
	struct jpeg_error_mgr jerr;

	data[0] = y;
	data[1] = cb;
	data[2] = cr;

	cinfo.err = jpeg_std_error(&jerr);  // errors get written to stderr 
	
	jpeg_create_compress (&cinfo);
	cinfo.image_width = width;
	cinfo.image_height = height;
	cinfo.input_components = 3;
	jpeg_set_defaults (&cinfo);

	jpeg_set_colorspace(&cinfo, JCS_YCbCr);

	cinfo.raw_data_in = TRUE; // supply downsampled data
	cinfo.comp_info[0].h_samp_factor = 2;
	cinfo.comp_info[0].v_samp_factor = 2;
	cinfo.comp_info[1].h_samp_factor = 1;
	cinfo.comp_info[1].v_samp_factor = 1;
	cinfo.comp_info[2].h_samp_factor = 1;
	cinfo.comp_info[2].v_samp_factor = 1;

	jpeg_set_quality (&cinfo, quality, TRUE);
	cinfo.dct_method = JDCT_FASTEST;

	jpeg_stdio_dest (&cinfo, fp);  	  // data written to file
	jpeg_start_compress (&cinfo, TRUE);
  
	for (j=0;j<height;j+=16) {
		for (i=0;i<16;i++) {
			y[i] = image + width*(i+j);
			if (i%2 == 0) {
				cb[i/2] = image + width*height + width/2*((i+j)/2);
				cr[i/2] = image + width*height + width*height/4 + width/2*((i+j)/2);
			}
		}
		jpeg_write_raw_data (&cinfo, data, 16);
  	}

	jpeg_finish_compress (&cinfo);
	jpeg_destroy_compress (&cinfo);
}

void put_jpeg_grey (FILE *picture, char *image, int width, int height, int quality)
{
	int y;
	JSAMPROW row_ptr[1];
	struct jpeg_compress_struct cjpeg;
	struct jpeg_error_mgr jerr;

	cjpeg.err = jpeg_std_error(&jerr);
	jpeg_create_compress (&cjpeg);
	cjpeg.image_width = width;
	cjpeg.image_height = height;
	cjpeg.input_components = 1; /* one colour component */
	cjpeg.in_color_space = JCS_GRAYSCALE;

	jpeg_set_defaults (&cjpeg);

	jpeg_set_quality (&cjpeg, quality, TRUE);
	cjpeg.dct_method = JDCT_FASTEST;
	jpeg_stdio_dest (&cjpeg, picture);

	jpeg_start_compress (&cjpeg, TRUE);

	row_ptr[0]=image;
	for (y=0; y<height; y++) {
		jpeg_write_scanlines(&cjpeg, row_ptr, 1);
		row_ptr[0]+=width;
	}
	jpeg_finish_compress(&cjpeg);
	jpeg_destroy_compress(&cjpeg);
}

void put_ppm_bgr24 (FILE *picture, char *image, int width, int height)
{
	int x, y;

	/*	ppm header
	 *	width height
	 *	maxval
	 */
	fprintf(picture, "P6\n");
	fprintf(picture, "%d %d\n", width, height);
	fprintf(picture, "%d\n", 255);
	for (y=0; y<height; y++) {
		for (x=0; x<width; x++) {
			/* ppm is rgb not bgr */
			fwrite(&image[y*width*3+x*3+2], 1, 1, picture);
			fwrite(&image[y*width*3+x*3+1], 1, 1, picture);
			fwrite(&image[y*width*3+x*3+0], 1, 1, picture);
		}
	}
}

void put_picture_fd (struct context *cnt, FILE *picture, char *image, int quality)
{
	if (cnt->conf.ppm) {
		put_ppm_bgr24(picture, image, cnt->conf.width, cnt->conf.height);
	} else {
		switch (cnt->imgs.type) {
			case VIDEO_PALETTE_RGB24:
				put_jpeg_bgr24(picture, image, cnt->conf.width, cnt->conf.height, quality);
				break;
			case VIDEO_PALETTE_YUV420P:
				put_jpeg_yuv420p(picture, image, cnt->conf.width, cnt->conf.height, quality);
				break;
			case VIDEO_PALETTE_GREY:
				put_jpeg_grey(picture, image, cnt->conf.width, cnt->conf.height, quality);
				break;
		}
	}
}

void put_picture (struct context *cnt, char *file, char *image)
{
	FILE *picture;

	picture=fopen(file, "w");
	if (!picture) {
		perror("can't write picture!");
		return;
	}
	put_picture_fd(cnt, picture, image, cnt->conf.quality);
	if (cnt->conf.ppm) {
		event(EVENT_FILECREATE, cnt, file, "image/ppm", cnt->currenttime);
	} else {
		event(EVENT_FILECREATE, cnt, file, "image/jpg", cnt->currenttime);
	}
	fclose(picture);
}
