/*	alg.c
 *
 *	Detect changes in a video stream.
 *	Copyright 2001 by Jeroen Vreeken (pe1rxq@amsat.org)
 *	This software is distributed under the GNU public license version 2
 *	See also the file 'COPYING'.
 *
 */
#include "motion.h"
#include "video.h"
#include "conf.h"
#include "alg.h"
#include "event.h"

/* locate the center of the movement. */
struct coord alg_locate_center (struct images *imgs, int width, int height)
{
	char *out=imgs->out;
	int x, y, centc=0;
	struct coord cent;

	cent.x=0;
	cent.y=0;
	/* Locate movement */
	for (y=0; y<height; y++) {
		for (x=0; x<width; x++) {
			if (imgs->type==VIDEO_PALETTE_RGB24) {
				if (*(out++) + *(out++) + *(out++)) {
					cent.x+=x;
					cent.y+=y;
					centc++;
				}
			} else {
				if (*(out++)) {
					cent.x+=x;
					cent.y+=y;
					centc++;
				}
			}
		}
	}
	if (centc) {
		cent.x=cent.x/centc;
		cent.y=cent.y/centc;
	}
	return (cent);
}

/* draw a box around the movement */
void alg_locate (struct coord cent, struct images *imgs, int width, int height)
{
	char *out=imgs->out, *new=imgs->new;
	int x, y, maxx=0, maxy=0, minx=width, miny=height;
	int centc=0, xdist=0, ydist=0;

	centc=0;
	for (y=0; y<height; y++) {
		for (x=0; x<width; x++) {
			if (imgs->type==VIDEO_PALETTE_RGB24) {
				if (*(out++) + *(out++) + *(out++)) {
					if (x > cent.x) xdist+=x-cent.x;
					if (y > cent.y) ydist+=y-cent.y;
					if (x < cent.x) xdist+=cent.x-x;
					if (y < cent.y) ydist+=cent.y-y;
					centc++;
				}
			} else {
				if (*(out++)) {
					if (x > cent.x) xdist+=x-cent.x;
					if (y > cent.y) ydist+=y-cent.y;
					if (x < cent.x) xdist+=cent.x-x;
					if (y < cent.y) ydist+=cent.y-y;
					centc++;
				}
			}
		}	
	}
	if (centc) {
		minx=cent.x-xdist/centc*2;
		maxx=cent.x+xdist/centc*2;
		/* Make the box a litle bigger in y direction to make sure the
		 * heads fit in */
		miny=cent.y-ydist/centc*3;
		maxy=cent.y+ydist/centc*2;
	}
	if (minx < 0) minx=0;
	if (miny < 0) miny=0;
	if (maxx > width-1) maxx=width-1;
	if (maxy > height-1) maxy=height-1;
	if (maxx < 0) maxx=0;
	if (maxy < 0) maxy=0;
	if (minx > width-1) minx=width-1;
	if (miny > height-1) miny=height-1;
	out=imgs->out;
	
	/* Draw a box around the movement */
	if (imgs->type==VIDEO_PALETTE_RGB24) {
		for (x=minx; x<=maxx; x++) {
			new[(x+width*miny)*3+0]=~new[(x+width*miny)*3+0];
			new[(x+width*miny)*3+1]=~new[(x+width*miny)*3+1];
			new[(x+width*miny)*3+2]=~new[(x+width*miny)*3+2];
			new[(x+width*maxy)*3+0]=~new[(x+width*maxy)*3+0];
			new[(x+width*maxy)*3+1]=~new[(x+width*maxy)*3+1];
			new[(x+width*maxy)*3+2]=~new[(x+width*maxy)*3+2];
			out[(x+width*miny)*3+0]=~out[(x+width*miny)*3+0];
			out[(x+width*miny)*3+1]=~out[(x+width*miny)*3+1];
			out[(x+width*miny)*3+2]=~out[(x+width*miny)*3+2];
			out[(x+width*maxy)*3+0]=~out[(x+width*maxy)*3+0];
			out[(x+width*maxy)*3+1]=~out[(x+width*maxy)*3+1];
			out[(x+width*maxy)*3+2]=~out[(x+width*maxy)*3+2];
		}
		for (y=miny; y<=maxy; y++) {
			new[(minx+y*width)*3+0]=~new[(minx+y*width)*3+0];
			new[(minx+y*width)*3+1]=~new[(minx+y*width)*3+1];
			new[(minx+y*width)*3+2]=~new[(minx+y*width)*3+2];
			new[(maxx+y*width)*3+0]=~new[(maxx+y*width)*3+0];
			new[(maxx+y*width)*3+1]=~new[(maxx+y*width)*3+1];
			new[(maxx+y*width)*3+2]=~new[(maxx+y*width)*3+2];
			out[(minx+y*width)*3+0]=~out[(minx+y*width)*3+0];
			out[(minx+y*width)*3+1]=~out[(minx+y*width)*3+1];
			out[(minx+y*width)*3+2]=~out[(minx+y*width)*3+2];
			out[(maxx+y*width)*3+0]=~out[(maxx+y*width)*3+0];
			out[(maxx+y*width)*3+1]=~out[(maxx+y*width)*3+1];
			out[(maxx+y*width)*3+2]=~out[(maxx+y*width)*3+2];
		}
	} else {
		for (x=minx; x<=maxx; x++) {
			new[x+width*miny]=~new[x+width*miny];
			new[x+width*maxy]=~new[x+width*maxy];
			out[x+width*miny]=~out[x+width*miny];
			out[x+width*maxy]=~out[x+width*maxy];
		}
		for (y=miny; y<=maxy; y++) {
			new[minx+y*width]=~new[minx+y*width];
			new[maxx+y*width]=~new[maxx+y*width];
			out[minx+y*width]=~out[minx+y*width];
			out[maxx+y*width]=~out[maxx+y*width];
		}
	}
}

// NOT UP TO DATE (RGB/YUV)
int alg_diff_box(int x0, int y0, int x1, int y1, char *image, int width, int height)
{
	int diffs=0;
	int x, y;

	for (x=x0; x<=x1; x++)
		for (y=y0; y<=y1; y++)
			if (image[(width*y+x)*3+0] ||
			    image[(width*y+x)*3+1] ||
			    image[(width*y+x)*3+2])
			    	diffs++;
	return diffs;
}

// NOT UP TO DATE (RGB/YUV)
// If no longer experimental move it to draw.c
void draw_box(int x0, int y0, int x1, int y1, char *image, int width, int height)
{
	int x;
	int y;
	
	if (x0<0)
		x0=0;
	if (x1<0)
		x1=0;
	if (y0<0)
		y0=0;
	if (y1<0)
		y1=0;
	if (x0>width-1)
		x0=width-1;
	if (x1>width-1)
		x1=width-1;
	if (y0>height-1)
		y0=height-1;
	if (y1>height-1)
		y1=height-1;

	for (x=x0; x<=x1; x++) {
		image[(x+width*y0)*3+0]=~image[(x+width*y0)*3+0];
		image[(x+width*y0)*3+1]=~image[(x+width*y0)*3+1];
		image[(x+width*y0)*3+2]=~image[(x+width*y0)*3+2];
		image[(x+width*y1)*3+0]=~image[(x+width*y1)*3+0];
		image[(x+width*y1)*3+1]=~image[(x+width*y1)*3+1];
		image[(x+width*y1)*3+2]=~image[(x+width*y1)*3+2];
	}
	for (y=y0; y<=y1; y++) {
		image[(x0+y*width)*3+0]=~image[(x0+y*width)*3+0];
		image[(x0+y*width)*3+1]=~image[(x0+y*width)*3+1];
		image[(x0+y*width)*3+2]=~image[(x0+y*width)*3+2];
		image[(x1+y*width)*3+0]=~image[(x1+y*width)*3+0];
		image[(x1+y*width)*3+1]=~image[(x1+y*width)*3+1];
		image[(x1+y*width)*3+2]=~image[(x1+y*width)*3+2];
	}

}

// NOT UP TO DATE (RGB/YUV)
// Experimental
void alg_locate_new (struct coord cent, char *out, char *new, int width, int height)
{
	int diffline;
	int lastline;
	int i, j, k;
	int diffs=0;
	int object;
	int avgv=1;
	int avgh=1;
	int size=0;
	int *uph=NULL;
	int *upv=NULL;
	int *downh=NULL;
	int *downv=NULL;
	int uphc=0;
	int upvc=0;
	int downhc=0;
	int downvc=0;
	int boxx0;
	int boxx1;
	int boxy0;
	int boxy1;
	
	// get existing diffs instead of getting our own
	for (i=0; i<width*height*3; i++)
		if (out[i]) diffs++;

	// check if diffs!=0
	if (diffs) {
		avgv=diffs/width;
		avgh=diffs/height;
	}

	printf("%d\n", avgv);
	size=0;
	object=0;
	lastline=-80;
	for (i=4; i<width; i++){
		diffline=0;
		for (j=0; j<height; j++) {
			if (out[j*width*3+i*3+0] || out[j*width*3+i*3+1] || out[j*width*3+i+3+2])
				diffline++;
		}
		if (!object && size>avgv*2 && i-80>lastline) {
			object=1;
			upvc++;
			upv=realloc(upv, sizeof(int)*upvc);
			upv[upvc-1]=i;
			for (j=0; j<height; j++) {
				new[j*width*3+(i-4)*3+2]=255;
			}
		}
		size=size+diffline;
		if ((diffline<=avgv)) {
			size=size*3/4;
			if (object && size<avgv) {
				object=0;
				lastline=i;
				if (size>0)
					size=0;
			}
		}
	}
	size=0;
	object=0;
	lastline=width+80;
	for (i=width-4; i>=0; i--){
		diffline=0;
		for (j=0; j<height; j++) {
			if (out[j*width*3+i*3+0] || out[j*width*3+i*3+1] || out[j*width*3+i+3+2])
				diffline++;
		}
		if (!object && size>avgv*2 && i+80<lastline) {
			object=1;
			downvc++;
			downv=realloc(downv, sizeof(int)*downvc);
			downv[downvc-1]=i;
			for (j=0; j<height; j++) {
				new[j*width*3+(i+4)*3+0]=255;
			}
		}
		size=size+diffline;
		if ((diffline<=avgv)) {
			size=size*3/4;
			if (object && size<avgv) {
				object=0;
				lastline=i;
				if (size>0)
					size=0;
			}
		}
	}
	size=0;
	object=0;
	lastline=-80;
	for (i=4; i<height; i++){
		diffline=0;
		for (j=0; j<width; j++) {
			if (out[i*width*3+j*3+0] || out[i*width*3+j*3+1] || out[i*width*3+j+3+2])
				diffline++;
		}
		if (!object && size>avgh*2 && i-80>lastline) {
			object=1;
			uphc++;
			uph=realloc(uph, sizeof(int)*uphc);
			uph[uphc-1]=i;
			for (j=0; j<width; j++) {
				new[(i-4)*width*3+j*3+2]=255;
			}
		}
		size=size+diffline;
		if ((diffline<=avgh)) {
			size=size*3/4;
			if (object && size<avgh) {
				object=0;
				lastline=i;
				if (size>0)
					size=0;
			}
		}
	}
	size=0;
	object=0;
	lastline=height+80;
	for (i=height-4; i>=0; i--){
		diffline=0;
		for (j=0; j<width; j++) {
			if (out[i*width*3+j*3+0] || out[i*width*3+j*3+1] || out[i*width*3+j+3+2])
				diffline++;
		}
		if (!object && size>avgh*2 && i+80<lastline) {
			object=1;
			downhc++;
			downh=realloc(downh, sizeof(int)*downhc);
			downh[downhc-1]=i;
			for (j=0; j<width; j++) {
				new[(i+4)*width*3+j*3+0]=255;
			}
		}
		size=size+diffline;
		if ((diffline<=avgh)) {
			size=size*4/5;
			if (object && size<avgh) {
				object=0;
				lastline=i;
				if (size>0)
					size=0;
			}
		}
	}
	printf("upvc: %d downvc: %d uphc: %d downhc: %d\n", upvc, downvc, uphc, downhc);
	size=0;
	for (i=0; i<upvc; i++) {
		boxx0=upv[i];
		boxx1=width-1;
		for (j=0; j<downvc; j++)
			if (downv[j]>boxx0)
				boxx1=downv[j];
		boxy0=0;
		boxy1=height-1;
		for (j=0; j<uphc; j++) {
			boxy0=uph[j];
			for (k=0; k<downhc; k++)
				if (downh[k]>boxy0)
					boxy1=downh[k];
			size+=(boxx1-boxx0)*(boxy1-boxy0);
		}
	}
	for (i=0; i<upvc; i++) {
		boxx0=upv[i];
		boxx1=width-1;
		for (j=0; j<downvc; j++)
			if (downv[j]>boxx0)
				boxx1=downv[j];
		boxy0=0;
		boxy1=height-1;
		for (j=0; j<uphc; j++) {
			boxy0=uph[j];
			for (k=0; k<downhc; k++)
				if (downh[k]>boxy0)
					boxy1=downh[k];
			if (alg_diff_box(boxx0, boxy0, boxx1, boxy1, out, width, height)>diffs*((boxx1-boxx0)*(boxy1-boxy0)/(diffs*2)))
				draw_box(boxx0-8, boxy0-8, boxx1+8, boxy1+8, new, width, height);
		}
	}
	if (upv) free(upv);
	if (uph) free(uph);
	if (downv) free(downv);
	if (downh) free(downh);

	printf("\n");
}

/*	The heart of the motion detector
 *	Basicly: ref-new and count the pixels that are over a noise threshold
 */
int alg_diff_standard (struct context *cnt)
{
	struct images *imgs=&cnt->imgs;
	int i, diffs=0;
	long int level=0;
	int noise=cnt->conf.noise;
	char *ref=imgs->ref;
	char *new=imgs->new;
	char *out=imgs->out;
	unsigned char *mask=imgs->mask;


	/* If the average level of the picture is to low, compensate by 
	 * lowering the noise threshold
	 */
	if (cnt->conf.nightcomp) {
		i=imgs->motionsize;
		for (i--; i>=0; i--) {
			level+=(unsigned char)new[i];
		}
		level/=imgs->motionsize;
		if (level < noise*2)
			noise*=2;
	}

	i=imgs->motionsize;
	for (; i>0; i--) {
		*out=*ref-*new;
		if (mask)
			*out=((int)((char)*out**mask++)/255);
		if (*out >  noise ||
		    *out < -noise ) {
			if (!cnt->conf.realmotion) *out=*new;
			diffs++;
		} else {
			if (!cnt->conf.realmotion) *out=0;
		}
		out++;
		ref++;
		new++;
	}

	return diffs;
}

/*
	Very fast diff function, does not do nightcompensation or mask
	overlaying.
*/
int alg_diff_fast (struct context *cnt)
{
	struct images *imgs=&cnt->imgs;
	int i, diffs=0;
	int noise=cnt->conf.noise;
	char *out=imgs->out;
	char *ref=imgs->ref;
	char *new=imgs->new;

	memset(out, 0, imgs->size);
	i=imgs->motionsize;
	for (; i>0; i-=40) {
		*out=*ref-*new;
		if (*out >  noise ||
		    *out < -noise ) {
			if (!cnt->conf.realmotion) *out=*new;
			diffs++;
		} else {
			if (!cnt->conf.realmotion) *out=0;
		}
		out+=40;
		ref+=40;
		new+=40;
	}
	diffs=diffs*40;
	return diffs;
}

/*
	diff_hybrid uses diff_fast to quickly decide if there is anything worth
	sending to diff_standard.
*/
int alg_diff_hybrid (struct context *cnt)
{
	int diffs;
	
	diffs=alg_diff_fast(cnt);
	if (diffs > cnt->conf.max_changes/2)
		diffs=alg_diff_standard(cnt);
	return diffs;
}

int alg_diff (struct context *cnt)
{
	return alg_diff_hybrid(cnt);
}


/* Detect a sudden massive change in the picture.
   It is assumed to be the light being switched on or a camera displacement.
   In any way the user doesn't think it is worth capturing.
 */
int alg_lightswitch (struct context *cnt, int diffs, int dev, int pipe, int mpipe)
{
	struct images *imgs=&cnt->imgs;
	int i, j;
	
	/* is 2/3 of the image right?  */
	if (diffs > imgs->width*imgs->height*2) {
		printf("lightswitch detected\n");
		/* wait for the device to settle and update the
		 * reference frame to prevent the next frames
		 * from being detected
		 * if 1/3 of the frame is still moving asume that the camera
		 * has not yet compensated.
		 */
		i=10;
		while(i && (imgs->new=vid_next(cnt, dev, imgs->new, imgs->width, imgs->height))) {
			if (alg_diff(cnt)
			    > imgs->width*imgs->height)
				i=10;
			j=imgs->width*imgs->height;
			if (imgs->type==VIDEO_PALETTE_RGB24)
				j*=3;
			for (j--; j>=0; j--) {
				imgs->ref[j]=imgs->new[j];
			}
			event(EVENT_IMAGE, cnt, imgs->new, &pipe, cnt->currenttime);
			event(EVENT_IMAGEM, cnt, imgs->out, &mpipe, cnt->currenttime);
			i--;
		}
		return 0;
	}
	return diffs;
}

int alg_switchfilter (struct context *cnt, int diffs, struct coord *cent)
{
	int linediff=diffs/cnt->conf.height;
	struct images *imgs=&cnt->imgs;
	char *out=cnt->imgs.out;
	int y, x, line;
	int lines=0, vertlines=0;
	char tmp[80];

	if (imgs->type==VIDEO_PALETTE_RGB24)
		for (y=0; y<cnt->conf.height; y++) {
			line=0;
			for (x=0; x<cnt->conf.width*3; x++) {
				if (*(out++)) {
					line++;
				}
			}
			if (line>cnt->conf.width/6) {
				vertlines++;
			}
			if (line>linediff*2) {
				lines++;
			}
		}
	else
		for (y=0; y<cnt->conf.height; y++) {
			line=0;
			for (x=0; x<cnt->conf.width; x++) {
				if (*(out++)) {
					line++;
				}
			}
			if (line>cnt->conf.width/18) {
				vertlines++;
			}
			if (line>linediff*2) {
				lines++;
			}
		}
//	printf("%d %d %d %d  %d\n", lines, vertlines, linediff, lines-vertlines, diffs);
	if (vertlines>cnt->conf.height/10 && lines<vertlines/3 && 
	    (vertlines>cnt->conf.height/4 || lines-vertlines>lines/2)) {
	    	sprintf(tmp, "%d %d", lines, vertlines);
	    	draw_text(cnt->imgs.new, 10, 10, 288, 352, tmp, cnt->imgs.type);
		return diffs;
	}
	return 0;
}
