#ifndef __MOOD_H
#define __MOOD_H

#define PIDFILE PIDDIR "/mood.pid"
#define FIRSTUID 60000
#define LASTUID 60500
#define RETSIZE 8
#define PID_MAX 0x8000

#include "../mooix.h"
#include "moocallstack.h"

#include <sys/time.h>
#include <sys/resource.h>
#include <unistd.h>
#include <sys/param.h>
#include <sys/socket.h>
#ifndef __linux__
#include <sys/ucred.h>
#endif
#include <grp.h>

extern uid_t mooadminuid;
extern gid_t mooadmingid;
extern char *libmooproxy;
struct callstack *mooadminstack;
extern char *perlpreload;

struct rlimit_name_t {
	char *name;
	int resource;
};

/* mood.c */
struct embedded_lang {
	char *interp;
	void (*init_func)(void);
	void (*run_func)(int argc, char **argv, const char *this, 
	                 const char *moosock, const char *method, 
	                 const char *mooix_debug);
	void (*end_func)(void);
};

/* stackinfo.c */
enum tristate {
	false,
	true,
	unknown,
};
/* Hold a callstack and some precalculated data that is frequently needed
 * about it. */
typedef struct stackinfo_t {
	enum tristate important_ok; /* is whole stack contents topmost object
				  of stack and parents of that object */
	struct callstack *stack;
	struct callstack *callerstack;
	uid_t uid; /* the uid of the method that the stack is being used for */
} stackinfo;
stackinfo *stackinfo_create (void);
enum tristate important_ok(stackinfo *info);

/* proxy.c */
int proxycommand (int client, stackinfo *info, int privs);

/* object.c */
int in_mooix_obj (void);
int is_object_file (const char *string);
enum method_type {
	stackless,
	normal,
	invalid,
};
enum method_type get_method_type (const char *filename);
struct callstack *object_owner_stack (int objectfd);
struct callstack *object_parents_stack (int objectfd, int nochdir);
int valid_file_mode (stackinfo *info, mode_t mode);
int valid_dir_mode (stackinfo *info, mode_t mode);
int valid_file_mode (stackinfo *info, mode_t mode);
int valid_dir_mode (stackinfo *info, mode_t mode);
int can_open (stackinfo *info, const char *file, int flags, mode_t mode);
int can_create (stackinfo *info, const char *file);
int can_unlink (stackinfo *info, const char *file);
int can_rmdir (stackinfo *info, const char *dir);
int can_chmod (stackinfo *info, const char *file, int mode);
int can_signal (stackinfo *info);

/* uids.c */
int validuid (uid_t uid);
int validate (struct ucred *);
uid_t uidalloc (void);
void reclaim (uid_t uid, int reallykill);
void clearuidspace (uid_t low, uid_t high, unsigned int reallykill);
void checkuidspace (uid_t firstuid, uid_t lastuid);
void uidspace (uid_t first, uid_t last, uid_t entry);

/* socket.c */
int bindsocket (char *path, int perms);
struct ucred *getcreds (int);
void sendfd(int client, int send_fd);
int getfd (int);

/* clientcomm.c */
void freecommand (struct command_t *);
struct command_t *getcommand (int client);
void result (int client, int ret, int err);

/* noembed.c*/
void run_noembed (int my_argc, char **my_argv, const char *this, 
		  const char *moosock, const char *method,
		  const char *mooix_debug);

/* perl.c */
void init_perl (void);
void run_perl (int my_argc, char **my_argv, const char *this, 
               const char *moosock, const char *method,
	       const char *mooix_debug);
void end_perl (void);

#ifdef EMBED_PYTHON
/* python.c */
void init_python (void);
void run_python (int my_argc, char **my_argv, const char *this,
		 const char *moosock, const char *method,
		 const char *mooix_debug);
void end_python (void);
#endif

#ifdef EMBED_RUBY
/* ruby.c */
void init_ruby (void);
void run_ruby (int my_argc, char **my_argv, const char *this,
               const char *moosock, const char *method,
	       const char *mooix_debug);
void end_ruby (void);
#endif

/*
 * This is similar to assert(), except if the expression fails the client
 * is passed a message containing errno. Also, unlike assert() the
 * expression is always evaluated, even if debugging is turned off, so
 * side-effects are fine.
 */  
#ifdef ASSERT
#include <errno.h>
#define rassert(expr) errno=0;_rassert(client, __STRING(expr), (expr), __func__, __FILE__, __LINE__);
int _rassert (int client, const char *expr, int ret, const char *func, const char *file, int line);
#else
#define rassert(expr) if ((expr)) {};
#endif

#ifdef DEBUG
#include <stdio.h>
#define debug(msg, args...) do { fprintf(stderr, msg, ##args); fprintf(stderr, "\n"); } while (0);
#else
#define debug(msg, args...) /* nothing */
#endif

#ifdef PROFILE
FILE *profilef;
void profile (const char *startstop, const char *object, const char *what);
#else
#define profile(args...) /* nothing */
#endif

/* This one goes to the client. */
#define gripe(msg, args...) fprintf(stderr, "mood: " msg "\n", ##args)

#define warn(msg, args...) syslog(LOG_WARNING, "warning: " msg, ##args)

#define die(msg, args...) do { syslog(LOG_ERR, "error: " msg, ##args); exit(1); } while (0);

#endif /* __MOOD_H */
