/*
 * Mood, the mooix daemon.
 * 
 * Copyright 2001-2003 by Joey Hess <joey@mooix.net>
 * under the terms of the GNU GPL.
 */

#include <stdlib.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/un.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <sys/wait.h>
#include <assert.h>
#include <pwd.h>
#include <syslog.h>
#include <fcntl.h>
#include <errno.h>
#include <syslog.h>
#include <stdarg.h>
#include <libgen.h>
#include <signal.h>

#include "mood.h"

char *sockfile=MOOSOCK;
char *libmooproxy=LIBMOOPROXY;
unsigned int daemonize = 1;
unsigned int clearspace = 1;
uid_t firstuid = FIRSTUID;
uid_t lastuid = LASTUID;
gid_t entrygid = -1;
uid_t moouid = -1;
gid_t moogid = -1;
uid_t mooadminuid = -1;
gid_t mooadmingid = -1;
struct callstack *mooadminstack = NULL;
char *perlpreload;
char **real_argv;

/* embedded language hooks */
struct embedded_lang embedded[] = {
	{ "/usr/bin/perl", init_perl, run_perl, end_perl },
#ifdef EMBED_PYTHON
	{ "/usr/bin/python", init_python, run_python, end_python },
#endif
#ifdef EMBED_RUBY
	{ "/usr/bin/ruby", init_ruby, run_ruby, 0 },
#endif
	{ "", 0, 0, 0 }
};

/* Maps pids that we fork off to their uid's. */
uid_t pid2uid[PID_MAX] = {0};

void usage (void) {
	fprintf(stderr,
		"Usage: mood [options]]\n"
		"\n"
		" -h            Display this help text\n"
		" -n            Do not daemonize\n"
		" -s sockfile   Filename of communication socket (default: %s)\n"
		" -L libfile    Library to preload for methods (default: %s)\n"
		" -k            Don't kill everything in the uid space\n",
		sockfile, libmooproxy
	);
}

/* The config file is sequence of shell variable settings, with comments. */
void readconfig (char *file) {
	FILE *f;
	size_t ret;
	int c;
	int lineno = 0;
	char *key, *key_p = malloc(20 * sizeof(char));
	char *value, *value_p = malloc(256 * sizeof(char));
	struct passwd *pw;
	struct group *gr;
	struct rlimit_name_t rlimits[] = {
		{"CPU", RLIMIT_CPU},
		{"FSIZE", RLIMIT_FSIZE},
		{"DATA", RLIMIT_DATA},
		{"STACK", RLIMIT_STACK},
		{"CORE", RLIMIT_CORE},
		{"RSS", RLIMIT_RSS},
		{"MEMLOCK", RLIMIT_MEMLOCK},
		{"NPROC", RLIMIT_NPROC},
		{"NOFILE", RLIMIT_NOFILE},
		{"AS", RLIMIT_AS},
		{NULL, 0},
	};

	if ((f = fopen(file, "r")) == NULL) {
		fprintf(stderr, "%s\n", file);
		perror("config file");
		return;
	}

	while ((ret = fscanf(f, "%20[A-Z_a-z]=%256[^\n]\n", key_p, value_p)) != EOF) {
		value = value_p;
		key = key_p;
		lineno++;
		if (ret != 2) {
			c=fgetc(f);
			if (c == '#') {
				/* Read comment line and discard. */
				fscanf(f, "%*[^\n]\n");
			}
			else if (c != '\n') {
				fprintf(stderr, "mood: %s: parse error on line %i, near (%c)\n",
					        file, lineno, c);
				exit(1);
			}
			continue;
		}
		
		if (value[0] == '"') {
			value++;
			value[strlen(value) - 1]='\0';
		}
		
		if (strcasecmp(key, "MOOENTRYGROUP") == 0) {
			if ((gr = getgrnam(value)) == NULL) {
				perror("getgrnam");
				exit(1);
			}
			entrygid = gr->gr_gid;
		}
		else if (strcasecmp(key, "MOOADMIN") == 0) {
			if ((gr = getgrnam(value)) == NULL) {
				perror("getgrnam");
				exit(1);
			}
			mooadmingid = gr->gr_gid;
			if ((pw = getpwnam(value)) == NULL) {
				perror("getpwnam");
				exit(1);
			}
			mooadminuid = pw->pw_uid;
		}
		else if (strcasecmp(key, "MOOADMINOBJ") == 0) {
			int objectfd = open(value, O_RDONLY);
			assert(objectfd != -1);
			mooadminstack = callstack_push(mooadminstack, objectfd, NULL);
			close(objectfd);
		}
		else if (strcasecmp(key, "LOWUID") == 0) {
			firstuid = atoi(value);
		}
		else if (strcasecmp(key, "HIGHUID") == 0) {
			lastuid = atoi(value);
		}
		else if (strcasecmp(key, "PERLPRELOAD") == 0) {
			perlpreload = strdup(value);
		}
		else if (strncasecmp(key, "RLIMIT_", strlen("RLIMIT_")) == 0) {
			int i;
			struct rlimit rl;
			for (i = 0; rlimits[i].name; i++) {
				if (strcasecmp(key + strlen("RLIMIT_"), rlimits[i].name) == 0) {
					rl.rlim_max = rl.rlim_cur = atoi(value);
					if (setrlimit(rlimits[i].resource, &rl) != 0) {
						fprintf(stderr, "mood: failed setting relimit %s: %s\n",
								key, strerror(errno));
					}

					break;
				}
			}
			if (! rlimits[i].name) {
				fprintf(stderr, "mood: %s: on line %i, bad resource limit name, (%s)\n", file, lineno, key);
			}
		}
	}
	
	callstack_dedup(mooadminstack);

	free(key_p);
	free(value_p);
	fclose(f);
}

void parseopt (int argc, char **argv) {
	extern char *optarg;
	int c = 0;
	
	while (c != -1) {
		c = getopt(argc, argv, "hns:kL:m:");
		switch (c) {
			case 's':
				sockfile = strdup(optarg);
				break;
			case 'L':
				libmooproxy = strdup(optarg);
				break;
			case 'n':
				daemonize=0;
				break;
			case 'k':
				clearspace = 0;
				break;
			case 'h':
				usage();
				exit(1);
		}
	}
	if (optind < argc) {
		usage();
		exit(1);
	}
}

/* Write a pid file. */
void pidfile (const char *file) {
	FILE *f;

	if ((f=fopen(file, "w")) == NULL)
		die("unable to write %s", file);
	fprintf(f, "%i\n", getpid());
	fclose(f);
}

/* See if any children have exited and clean up. Options is as with waitpid. */
/* Returns true if any kid did exit. */
int waitonkid (int options) {
	pid_t pid;
	int status = 0;
	pid = waitpid(-1, &status, options);
	if (pid > 0) {
		if (WIFSIGNALED(status) && WTERMSIG(status) != SIGPIPE)
			warn("child process died from signal %i",
				WTERMSIG(status));
		else if (WEXITSTATUS(status) != 0)
			warn("child process returned non-zero exit code: %i",
				WEXITSTATUS(status));
		assert(pid2uid[pid] != 0);
		reclaim(pid2uid[pid], 1);
		pid2uid[pid] = 0;
		return 1;
	}
	return 0;
}

/* Listen for connections to the socket and dispatch children. */
void mainloop (int socket) {
	struct sockaddr addr;
	socklen_t addrlen = sizeof(addr);
	pid_t pid;
	uid_t newuid;
	int client;
	struct ucred *creds;
	
	debug("entering the main loop");
	
	while ((client = accept(socket, &addr, &addrlen)) != -1) {
		creds = getcreds(client);
		if (validate(creds)) {
			/* Has to come before the fork as the main process
			 * needs to track which uids are used. */
			while ((newuid = uidalloc()) == -1) {
				warn("Out of uids!");
				waitonkid(0); // blocking wait
			}
			/* Fork now to prevent clients from blocking mood. */
			rassert((pid = fork()) != -1);
			if (pid) {
				rassert(pid2uid[pid] == 0);
				pid2uid[pid] = newuid;
			}
			else {
				stackinfo *info;
				int thisfd;
				
				close(socket);

/* "portable" access to uid and gid field */
#ifdef __linux__
#define creds_uid uid
#define creds_gid gid
#else
#define creds_uid cr_uid
#define creds_gid cr_gid
#endif
				
				/* Load up the caller stack associated with
				 * the caller's uid (if there is one). */
				info = stackinfo_create();
				info->stack = callstack_load(callstack_file(creds->creds_uid), 0);
				info->uid = creds->creds_uid;

				/* Receive fd from the client, that should
				 * be open to the object's directory. Set
				 * up THISFD to point to it. */
				thisfd = getfd(client);
				rassert(fchdir(thisfd) != -1);
				rassert(in_mooix_obj());
				rassert(dup2(thisfd, THISFD) == THISFD);
				close(thisfd);
			
				/* Handle the command. */
				proxycommand(client, info, newuid);
				exit(0);
			}
		}
		else {
			result(client, -1, EPERM);
		}
		close(client);

		while (waitonkid(WNOHANG));
	}
	perror("accept");
}

int main (int argc, char **argv) {
	int socket, index;
	
	real_argv = argv;
	
	if (geteuid() != 0) {
		fprintf(stderr, "mood must be run by root\n");
		exit(1);
	}
		
#ifdef PROFILE
	profilef=fopen("mooprofile", "a");
#endif
	
	readconfig(CONFFILE);
	parseopt(argc, argv);
	
	/* Log goes to console if not in daemon mode. */
	openlog("mood", LOG_PID | (daemonize ? 0 : LOG_PERROR), LOG_DAEMON);
	
	umask(022);
	uidspace(firstuid, lastuid, entrygid);
	clearuidspace(firstuid, lastuid, clearspace);

	/* Unset TERM just because things like the ttysession should not
	 * try to use the terminal type mood was started up on, but should
	 * handle it themselves. */
	unsetenv("TERM");
	
	/* initialize all the embedded languages */
	for (index = 0; embedded[index].run_func != 0; index++) {
		if (embedded[index].init_func != 0) {
			(*embedded[index].init_func)();
		}
	}
	
	socket = bindsocket(sockfile, 0666);
	if (daemonize) {
		if (daemon(0,0) == -1)
			perror("daemon");
		pidfile(PIDFILE);
	}
	mainloop(socket);
	return -1;
}
