"""Result.py
This module implements Result classes for mooix methods.
 
This file is part of the Python 2.2.1+ binding for mooix
Copyright (c)2002,2003 Nicholas D. Borko.  All Rights Reserved.
The author can be reached at nick@dd.revealed.net
 
The Python binding for mooix is free software; you can redistribute
it and/or modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.
 
The Python binding for mooix is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.
 
You should have received a copy of the GNU Lesser General Public
License along with mooix; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
02111-1307 USA."""

import os, os.path, sys
from debug import *
from util import *

class Result(object):

    """A Result object is returned by Thing.__getattr__ whenever an
    attribute access is required, effectively making all properties and
    methods transparent in Python and look like any other "normal"
    Python property or method call.  It encapsulates the return value
    from a mooix method so you can retrieve the result in any type you
    want, since mooix method results are typeless.  It also tries to
    mimic the behavior of perl, in that coercing the object to a type
    will call the method with no arguments and convert the result. """

    def __init__(self, this, fieldfile):
        self._this = this
        self._fieldfile = fieldfile
        self.__input = []

    def _init_input(self, input, strip_quotes = True,
                    dereference_things = True):
        """Initialize the list that contains the input, since this can
        be delayed in the case of methods."""
        self.__input = [ parse_mooix(line, strip_quotes, dereference_things)
                         for line in input ]

    def _check_init(self):
        """Make sure the input list is initialized before accessing it
        using one of the conversion properties."""
        # override this if you want some default action if the input is not
        # initialized (e.g., before a method call)
        pass

    def int():
        def fget(self):
            self._check_init()
            if len(self) == 1:
                try:
                    return int(self.__input[0])
                except:
                    return 0
        def fset(self, value):
            setattr(self._this, os.path.basename(self._fieldfile), value)
        doc = 'Return the result as an integer.'
        return locals()
    int = property(**int())

    def float():
        def fget(self):
            self._check_init()
            if len(self) == 1:
                try:
                    return float(self.__input[0])
                except: 
                    return 0
        def fset(self, value):
            setattr(self._this, os.path.basename(self._fieldfile), value)
        doc = 'Return the result as a floating point number.'
        return locals()
    float = property(**float())

    def number():
        def fget(self): 
            self._check_init()
            if len(self) == 1:
                try:
                    return int(self.__input[0])
                except:
                    try:
                        return float(self.__input[0])
                    except:
                        return 0
            else:
                return 0
        def fset(self, value):
            setattr(self._this, os.path.basename(self._fieldfile), value)
        doc = 'Return the result as a numeric value, either int or float.'
        return locals()
    number = property(**number())

    def string():
        def fget(self):
            self._check_init()
            return '\n'.join([ str(part) for part in self.__input ])
        def fset(self, value):
            setattr(self._this, os.path.basename(self._fieldfile), value)
        doc = 'Return the result as single string.'
        return locals()
    string = property(**string())

    def list():
        def fget(self):
            self._check_init()
            return list(self.__input)
        def fset(self, value):
            setattr(self._this, os.path.basename(self._fieldfile), value)
        doc = 'Return the result as a list.'
        return locals()
    list = property(**list())

    def dict():
        def fget(self):
            self._check_init()
            if len(self.__input) % 2:
                input = self.__input + [ None ]
            else:
                input = self.__input
            return dict([ (input[i], input[i+1])
                          for i in range(0, len(input), 2) ])
        def fset(self, value):
            setattr(self._this, os.path.basename(self._fieldfile), value)
        doc = 'Return the result as a dictionary.'
        return locals()
    dict = property(**dict())

    def __int__(self):
        return self.int

    def __long__(self):
        return self.int #this is handy because int returns a long, if necessary

    def __float__(self):
        return self.float

    def __str__(self):
        return self.string
    
    def __nonzero__(self):
        return len(self) > 0

    # emulate a list to be backward compatible, or at least be what mooix
    # returns by default
    def __len__(self):
        self._check_init()
        return len(self.__input)

    def __getitem__(self, key):
        self._check_init()
        if type(key) is types.SliceType:
            # always return a copy of the list
            if not key.start and not key.stop: 
                return self.list
            elif not key.stop:
                return self.list[key.start:]
            elif not key.start: 
                return self.list[:key.stop]
            else:
                return self.list[key.start:key_stop]
        else:
            return self.__input[key]

    def __contains__(self, value):
        self.check_init()
        return value in self.__input

    def __iter__(self):
        return iter(self.list)

    def __call__(self, *arg, **kw):
        return self

class Arguments(Result):
    def __init__(self, stdin):
        self._init_input(iter(stdin), True, True)
        # since we'll be referenced as a dict most of the time, make it once
        self.__dict = self.dict
        # pass through the common dictionary methods we'll support
        for method in ('keys', 'values', 'items', 'has_key', 'get', 'iterkeys',
                       'itervalues', 'iteritems', 'copy'):
            setattr(self, method, getattr(self.__dict, method))

    # this is usually (supposed to be) a mapping, so make it look like a dict
    # by default
    def __len__(self):
        return len(self.__dict.keys())

    def __getitem__(self, key):
        # be like perl, return None if key doesn't exist instead of excepting
        return self.__dict.get(key)

    def __contains__(self, value):
        return value in self.__dict

    def __iter__(self):
        return self.iterkeys()

class Property(Result):
    def __init__(self, this, fieldfile, is_list):
        Result.__init__(self, this, fieldfile)
        self.__is_list = is_list

    def _check_init(self):
        # always retrieve the value from the file, every time
        # there can be problems if the stickyness changes, or if the
        # property changes to a method between calls: caveat programmer
        self._init_input(file(self._fieldfile), False, self.__is_list)

    def __call__(self, value = ''):
        # if you "call" a property, you're calling setattr for this
        setattr(self._this, os.path.basename(self._fieldfile), value)

class Method(Result):
    def __init__(self, this, fieldfile):
        Result.__init__(self, this, fieldfile)
        self.__status = 0
        self.__was_called = False

    def _check_init_(self):
        if not self.__was_called:
            self.__call__()

    def __call__(self, *arg, **kw):
        """Execute an external mooix method and return the result."""
        debug('called method %s with arg %s and kw %s' % \
              (self._fieldfile, arg, kw))

        # be like Popen3 but debuggable
        stdin_reader, stdin_writer = os.pipe()
        stdout_reader, stdout_writer = os.pipe()
        sys.stdout.flush()
        sys.stderr.flush()
        pid = os.fork()
        if not pid: # child process
            debug('in child: %s' % self._fieldfile)
            os.dup2(stdin_reader, 0)
            os.dup2(stdout_writer, 1)
            os.close(stdin_writer)
            os.close(stdout_reader)

            # chdir only at this point (per joey)
            os.chdir(self._this.path)
            debug('chdir: %s' % os.getcwd())
            debug('file: %s' % os.path.join(os.getcwd(), self._fieldfile))

            os.execvpe(self._fieldfile,
                       (os.path.basename(self._fieldfile),),
                       os.environ)
            os._exit(-1) #should never get here
        # parent process
        os.close(stdin_reader)
        tochild = os.fdopen(stdin_writer, 'w')
        os.close(stdout_writer)
        fromchild = os.fdopen(stdout_reader)

        if len(arg):
            debug('raw method access, using args %s' % str(arg))
            tochild.writelines('\n'.join([ mooix_parse(a, raw = True) \
                                           for a in arg ]))
        elif len(kw):
            debug('keyword method access, using args %s' % kw)
            debug('\n'.join([ '%s\n%s' % ( mooix_parse(k),
                                           mooix_parse(v, raw = True) ) \
                              for k, v in kw.iteritems() ]))
            tochild.writelines('\n'.join([ '%s\n%s' % ( mooix_parse(k),
                                                        mooix_parse(v, raw = True) ) \
                                           for k, v in kw.iteritems() ]))
        tochild.close()

        self._init_input(iter(fromchild), True, True)

        fromchild.close()

        debug('returned %s' % self.list)

        # wait to prevent race conditions
        (pid, status) = os.waitpid(pid, os.WNOHANG)
        self.__status = status >> 8

        self.__was_called = True
        return Result.__call__(self)

    def status():
        def fget(self):
            return self.__status
        doc = 'The exit status of the method.'
        return locals()
    status = property(**status())

class NullMethod(Result):
    def __init__(self, this):
        Result.__init__(self, this, '')

    def status():
        def fget(self):
            return 0
        doc = 'The exit status of the method, always 0 for NullMethod'
        return locals()
    status = property(**status())

__all__ = [ 'Arguments', 'Property', 'Method', 'NullMethod' ]
