#!/usr/bin/perl

package Mooix::Thing;

#Turn off in production code!
#use strict;
#use warnings;

=head1 NAME

Mooix::Thing - perl binding for mooix objects

=head1 SYNOPSIS

  #!/usr/bin/perl
  use Mooix::Thing;
  run sub {
  	my $this=shift;
	my %params=@_;
	$this->super(%params);
	$this->field($value);
	$this->method(param => 'value', param2 => 'value');
	return 1;
  }

=head1 DESCRIPTION

This is the mooix object class for perl. It provides a standard perl
interface to the decidedly non-standard mooix objects.

Please do not change directory after instantiating objects in this
class; relative paths are used extensively, for security reasons.

=head1 INTERNALS AND TEMPORARY STATE

Internally, Mooix::Thing objects are blessed array references that are used
to hold various information about the object's temporary state (its
persistent state is stored on disk of course).

If you need to store temporary state of a Mooix::Thing, you can appropriate
your own fields of the array. Any array field after Mooix::Thing::_LAST_FIELD
is fair game.

$Mooix::Thing::debugging is set if debugging is enabled.

=cut

# Things are blessed array references. The array elements are:
use constant {
	_USEPATH	=> 0,	# The path to use when accessing files.
	 			# Generally absolute, can be set relative in
	 			# some cases.
	_ABSPATH	=> 1,	# The absolute path to the object.
	_INDEX		=> 2,	# An index, guarenteed to be unique for
	 			# unique objects.
	_EXEC_NEXT 	=> 3,	# Exec next method flag.
	_LAST_FIELD	=> 4
};

our $debugging = 0;

=head1 METHODS

=over 4

=item import

When imported, Mooix::Thing exports into your namespace a function called
"run".

Since this class is most often used to implement mooix object methods,
which run as individual programs, it's useful to have a way to write a
mooix method much as you would any perl object method, without having to
fiddle around with the underlying details of the mooix method calling
convention. The run function does this. You pass it a sub reference,
which is run much as a normal method would run -- it gets parameters in @_
(and these parameters may include references to other Mooix::Things); and
it can return a value or a list (including references to Mooix::Things).

If your method does not take any parameters besides the first, "$this"
parameter, it is worthwhile to take the time to prototype it so, like this:

  run sub ($) {
	  ...
  }

The advantage of doing this is that it will make the subroutine not bother
to parse any input data. This can be quicker, and also makes such methods
easier to run from outside the moo with runmeth.

Mooix::Thing also exports into your namespace a function called "fail". This
function can be used by verbs (and occasional other methods) to exit with a
numeric exit code that indicates failure, and at the same time return a value
to the caller.

=cut

# Note that this import happens at mood startup time, for embedded perl.
sub import {
	my $pkg = shift;
	my $callpkg = caller;
	
	# This should not be necessary, but on at least Mandrake 9.1 (perl
	# 5.8.0), if atdout is not autofushed, then methods that print to
	# stdout and exit will not send the stdout to their caller. This
	# makes it work, at the expense of slowing down the system a bit.
	# Configure --with-perl-stdout-flush-hack to enable.
	#PERL_STDOUT_FLUSH_HACK $|=1;
	
	# When methods are run by mood, $0 is set to just the bare method
	# name. Change it to the full method name. This is required to make
	# super work. This is not done for embedded perl, which sets $0
	# properly manually.
	if (exists $ENV{METHOD}) {
		$0=$ENV{METHOD};
	}

#	no strict 'refs';
	*debuglog = *_debuglog_null;
	*{"${callpkg}::fail"} = sub {
		print join("\n",map { "\"$_\"" } @_)."\n";
		exit 10; # FAIL
	};
	*{"${callpkg}::run"} = sub {
		if (! exists $ENV{THIS}) {
			die "\$ENV{THIS} is not set. Not running under mooix?\n";
		}
		
		# Go to the object directory, so relative paths can be used;
		# that is marginally faster and safer.
		chdir($ENV{THIS});
		my $this = Mooix::Thing->get($ENV{THIS});
		$this->[_USEPATH] = '.';

		if (exists $ENV{MOOIX_DEBUG}) {
			$this->debugger($ENV{MOOIX_DEBUG});
		}
		
		my @params=$this;
		# Skip reading stdin if the prototype says it's not needed.
		if (prototype($_[0]) ne '$') {
			push @params, map {
				chomp;
				if (/^mooix:/) {
					$_=Mooix::Thing->get($_);
				}
				elsif (/^"(.*)"$/) {
					$_=$1;
					s/(?<!\\)\\n/\n/g;  # unescape newlines
					s/\\\\/\\/g; # and slashes in parameters
				}
				$_
			} <STDIN>;
		}
		# This is where the first parameter to the run function
		# (the body of the method to run) is called, passed the 
		# parameters read from stdin, and its return value sent
		# to stdout.
		for ($_[0]->(@params)) {
			if (! ref $_) {
				# Return each value quoted, which means
				# escaping newlines is called for.
				# The temp $f variable is used because this
				# value might be a constant, and we modify
				# it.
				my $f=$_;
				$f =~ s/\\/\\\\/g;
				$f =~ s/\n/\\n/g;
				print "\"$f\"\n";
			}
			else {
				print $_, "\n"; # uses stringification..
			}
		}
	}
}

=item get

Returns an existing object. Pass the object id.

Returns undef if the id does not correspond to an object.

=cut

sub get {
	my $proto=shift;
	my $id=shift;
	
	# These mooix: tags are used to identify random strings as object
	# id's, and can be removed.
	$id=~s/^mooix://;
	return unless -e "$id/.mooix";

	# Get an absolute path for the object. If the object is referenced
	# through a link, use the link content, which should be absolute.
	# Otherwise, occasionally use pwd.
	my $abspath=$id;
	if (-l $abspath) {
		$abspath=readlink($id);
		if ($abspath !~ m:^/:) {
			print STDERR "warning: $abspath of $id is not absolute.\n";
			my ($dir)=$id=~m!(.*)/[^/]!;
			$abspath="$dir/$abspath";
		}
	}
	elsif ($abspath !~ m:^/:) {
		my $pwd;
		# If you know the pwd, you can pass it in as a second
		# parameter.
		if (@_ && $_[0] =~ m:^/:) {
			$pwd=$_[0];
		}
		else {
			$pwd=`pwd`;
			chomp $pwd;
#			print STDERR "warning: had to use pwd to figure out location of $pwd for $abspath\n";
		}
		$abspath="$pwd/$abspath";
	}
	
	my @s=stat("$id/.");
	#      _USEPATH  _ABSPATH  _INDEX
	bless([$abspath, $abspath, "$s[0].$s[1]"], ref($proto) || $proto);
}

=item id

Returns the id of an object. This will always be an absolute path to the
object, and it should always be a realpath() as well, with no intervening
symlinks.

=cut

sub id {
	$_[0]->[_ABSPATH];
}

=item index

Returns a unique index for the object. If you need to store some objects in
a hash, use these indexes for has keys and you'll be guaranteed that there
will be no hash key collisions.

=cut

sub index {
	$_[0]->[_INDEX];
}

=item usage

Displays usage help on the currently running method (by calling the
object's getusage method (which must exist if you want this to work)), and
exits. Any parameters are printed out.

=cut

sub usage {
	my $this=shift;
	my ($caller, $method) = $0 =~ m!^(.*)/(.*)!;
	
	$method = $0 if $method eq '';
	die join("\n", @_, "Usage: ", $this->getusage(field => $method));
}

=item fieldfile

Return the filename that contains a field, or undef if there are none.
Does inheritence from parents.

=cut
# For the sake of optimization, let's say the last thing it does a lstat
# or will always be the filename it returns. And it'll always do a lstat,
# not a stat. Do not depend on that in code external to this module.
sub fieldfile {
	my $this=shift;
	my $field=shift;
	$field=~s/^_/./;
	
	# Test for links too, because it could be a dangling symlink, and
	# that should still be returned (really happens).
	if (-l "$$this[0]/$field" || -e _) {
		return "$$this[0]/$field";
	}
	elsif ($field ne 'parent') {
		# Inheritance.
		my $dir="$$this[0]/parent";
		my $c=0;
		while (-e "$dir/.mooix") {
			lstat("$dir/$field"); # lstat first..
			return "$dir/$field" if -e _;
			$dir.="/parent";
			if ($c++ > 200) {
				die "possible recursive parent loop: $dir";
			}
		}
	}

	# Mixin support.
	if ($field =~ /_.+/) {
		my ($mixin, $field)=split(/_/, $field, 2);
		my $dir=$this->[_USEPATH];
		# Try to find the mixin object, recursing to parents as
		# necessary.
		my $c=0;
		do {
			if (-e "$dir/$mixin/.mooix") {
				# Now find the field in the mixin or in one
				# of its parents.
				if (-l "$dir/$mixin/$field" || -e _) {
					return "$dir/$mixin/$field";
				}
				$dir.="/$mixin/parent";
				my $c2=0;
				while (-e "$dir/.mooix") {
					lstat("$dir/$field"); # lstat first..
					return "$dir/$field" if -e _;
					$dir.="/parent";
					if ($c2++ > 200) {
						die "possible recursive parent loop: $dir";
					}
				}
				
				# If the mixin doesn't have it, failure
				# (don't recurse to other parents).
				return undef;
			}
			$dir.="/parent";
			if ($c++ > 200) {
				die "possible recursive parent loop: $dir";
			}
		} until (! -e "$dir/.mooix");
	}

	return undef;
}

=item super 

Call parent's implementation of the currently running method.
(Be sure to pass in the method parameters.)

=cut

sub super {
	my $this=shift;
	
	# $0 provides state about what method called this function.
	my ($caller, $method) = $0 =~ m!^(.*)/(.*)!;
	$method = $0 if $method eq '';
	
	# Search for implementation in parent. Since super calls may be
	# stacked, determine _which_ parent by just prefixing 'parent' to 
	# the current caller. Since super calls may jump back to
	# grandparents or even further, need to loop.
	my $prefix="$$this[0]/$caller/parent";
	while (-e "$prefix/.mooix") {
		lstat("$prefix/$method"); # lstat first for _readfield's sake..
		if (-f _) {
			if (-x _) {
				$this->debuglog(type => "call", field => "$prefix/$method", message => \@_) if $debugging;
				return $this->_runmethod("$prefix/$method", @_);
			}
			else {
				# Parent has a field by that name, but no
				# method. This is ill-defined territory.
				# I will _try_ to get the field.
				if (@_) {
					$this->croak("super is a field");
				}
				else {
					$this->debuglog(type => "read", field => $method) if $debugging;
					return $this->_readfield("$prefix/$method");
				}
			}
		}
		$prefix.="/parent";
	}
}

=item exec

This internal method makes the next external method call be run in a way
similar to exec(), so the caller of the method that calls it gets back
whatever it returns, and sees its numeric return code too. 

It is currently implemented in a much less optimal way than real exec()
however.

The method returns the object it was called on, so you can say:

  $this->exec->method(@params);

=cut

sub exec {
	$_[0]->[_EXEC_NEXT] = 1;
	$_[0];
}

=item getlock

This method returns an open filehandle which represents a lock; close the
filehandle to drop the lock. The first parameter is like the operation
parameter  to flock(). If the second parameter is passed, it is the name of
the field to lock; by default the .mooix field is locked, which locks the
object's position. The field that is locked must exist on the object, and
not be inherited, or the method will fail and return undef.

=cut

sub getlock {
	my $this=shift;
	my $locktype=shift;
	my $file=shift;
	
	$file='.mooix' if ! defined $file;
	my $lock;
	open ($lock, "$$this[0]/$file") || return undef;
	flock ($lock, $locktype) || return undef;

	$this->debuglog(type => "lock", field => $file, message => "type: $locktype") if $debugging;
	
	return $lock;
}

=item isa

This method should be passed a Mooix::Thing, and it returns true if the
object on which it is called is derived from that Mooix::Thing.

Note that if it is passed a scalar, it just calls UNIVERSAL::isa instead.

=cut

sub isa {
	my $this=shift;
	my $class=shift;

	if (! ref $class) {
		return UNIVERSAL::isa($this, $class);
	}
	
	while (ref $this) {
		return 1 if ($this == $class);
		$this = $this->parent;
	}

	return 0; # is not
}

=item msg

This convenience method calls the real msg method of an object, passing it 
the message field specified as the first parameter, and passing additional
fields in as additional parameters.

A sample invocation:

  $this->msg('bounce', %_);

=cut

sub msg {
	my $this=shift;
	$this->debuglog(type => "call", field => "msg", message => ["event", @_]) if $debugging;
	$this->_runmethod($this->fieldfile("msg"), event => @_);
}

=item croak

Throws a Mooix::Error object. Pass the base error message as the sole
parameter; the invoking method, $!, and a calltrace will be added when 
it's displayed.

=cut

sub croak {
	my $this=shift;
	require Carp;
	my $message="Error: ".Carp::shortmess(shift().(length $! ? " ($!)" : ""));
	# Munge filename of method in message from carp to use object
	# syntax.
	my ($file) = $0 =~ /.*\/+(.*)/;
	$obj = Mooix::Thing->get($ENV{THIS});
	$message=~s/\Q$0\E (line \d+)/$obj->$file $1/;
	chomp $message;
	require Mooix::Error;
	die Mooix::Error->new($this, $message);
}

=item debuglog

If debugging is enabled (the MOOIX_DEBUG environment variable is set), it
passes its parameters to the debug object specified by the MOOIX_DEBUG
environemtn variable. Otherwise, it does nothing, and calls to this method
should be very fast with debugging disabled.

Parameters are named, and are: type, accessee, field, message, replace, value.
See the usage information for mooix:abstract/debug->log for details. 

The message parameter may be an array reference, if so the debug object will
be passed a series of messages.

Or the message parameter can be a subroutine reference. If so, the referenced
sub will be called to generate the message, only if the message is actually
sent to the debugger.

If a value parameter is passed, it should be an array reference. The array
may be changed by the debugger to insert a new return value.

=cut

# This stub is replaced with a real method when debugging is enabled. Note
# that this is as fast as I can make it (a constant sub does not work). It
# is marginally faster to do $this->debug(..) if $debugging, but not enough
# so that external methods should do it. debug calls in this module will
# add the $debugging test.
sub _debuglog_null { 0 }

# Variables used by _debuglog_real.
my ($debugobj, $actingobj, $methodfile, $method, @debug_wanted);
# This is the real debugging method, which is only bound to the debug
# method when debugging is enabled.
sub _debuglog_real {
	my $this=shift;
	
	# See if the debugger has indicated it wants messages like this
	# one. This is much faster than calling it (sigh, external
	# methods..).
	return unless ref $debugobj && $debugging;
	my %params=@_;
	# Make the field relative.
	if (exists $params{field}) {
		$params{field}=~s/.*\///;
	}
	my $expr=$params{type}." ".(exists $params{field} ? $params{field} : "");
	return unless grep { $expr=~/^$_$/ } @debug_wanted;
	
	# Save miscallaneous perl variables that might get clobbered during
	# the debug logging.
	# Note that any calls to methods that call fieldfile, or other
	# things that might clobber perl's _ stat cache are not allowed in
	# here. (I don't know how to backup and restore the _ stat cache).
	# That's why the log method is called using _runmethod and
	# $methodfile below..
	my $savedstatuscode=$?;
	# set flag variable to prevent deep recursion when the method
	# runner calls debug..
	$debugging=0;

	my @params;
	if (ref $params{message} eq 'ARRAY') {
		foreach my $param (@{$params{message}}) {
			push @params, message => $param;
		}
		delete $params{message};
	}
	elsif (ref $params{message} eq 'CODE') {
		$params{message}=$params{message}->();
	}
	my $value;
	if ($params{replace}) {
		foreach my $param (@{$params{value}}) {
			push @params, value => $param;
		}
		$value=delete $params{value};
	}
	my @ret = $debugobj->_runmethod(
		$methodfile, %params, @params,
		object => $actingobj,
		method => $method,
		accessee => $this
	);
	if ($params{replace}) {
		@{$value} = @ret;
	}
	else {
		@debug_wanted=map {
			s/(^| )\*( |$)/$1.*$2/g; # convert lone "*" to ".*"
			s/[^A-Za-z0-9_.* ]//g; # sanitize
			$_
		} @ret;
	}
	$debugging = @debug_wanted;
	
	$?=$savedstatuscode;
	return $ret;
}

=item debugger

Get/set the object that debug messages are sent to. Setting it to an object
makes debugging be turned on for the current method, and any methods it
calls. Setting it to false turns off debugging.

=cut

sub debugger {
	my $this=shift;
	if (@_) {
		my $debugger=shift;
		if (! $debugger) {
			delete $ENV{MOOIX_DEBUG};
			$debugging=0;
			*debuglog = *_debuglog_null;
		}
		else {
			$ENV{MOOIX_DEBUG}=$debugger;
			$debugging=1;
			*debuglog = *_debuglog_real;
			# Populate all the variables _debuglog_real uses.
			$debugobj = Mooix::Thing->get($debugger);
			if (! ref $debugobj) {
				$this->croak("bad debug object, $debugger");
			}
			$actingobj = Mooix::Thing->get($ENV{THIS});
			$method=$0;
			$method=~s/^.*\///;
			$methodfile = $debugobj->fieldfile("log");
			# To start off, assume the debugger wants all messages.
			@debug_wanted = '.*';
		}
	}
	if ($debugging) {
		return $this->get($ENV{MOOIX_DEBUG});
	}
	else {
		return;
	}
}

=item implements

Returns true if the object implements the given method. Like the builtin
perl can method, it returns a subroutine reference if the method is
implemented as a built-in method. If the method is implemented as an
external method, it returns the filename of the method.

=cut

sub implements {
	my $this=shift;
	my $can=$this->can(@_);
	return $can if $can;
	my $file=$this->fieldfile(@_);
	return $file if length $file && -x $file && ! -d _;
}

=item defines

Checks to see if an object contains a field. Inheriting the field from a
parent does not count; the field must be part of the object the method is
called on. If the object does define the field, returns the filename of
the field in the object.

=cut

sub defines {
	my $file=$_[0]->[_ABSPATH]."/$_[1]";
	return $file if -e $file;
}

=item encapsulator

Returns the mooix object that the object is encapsulated in. That is, the
object that the mooix object is a subdirectory underneath.

=cut

sub encapsulator {
	my $this=shift;
	return $this->get($this->[_ABSPATH]."/..");
}

=item safegetfield

This method can be used to safely get the value of a field or method,
which might be supplied by a non-programmer (eg, as part of a message), 
without accidentially calling destructive methods like destroy.

It only allows getting values of fields that are not private.

It only allows calling of methods that are marked as safe by the existence
of a field named .<method>-safe with a true value.

=cut

sub safegetfield {
	my $this = shift;
	my $field = shift;

	return if $field=~/^[_.]/; # private field
	return if $this->can($field); # internal method
	my $file=$this->fieldfile($field);
	# Using fieldfile's guarantee that the last thing lstatted will be
	# the returned file..
	return if defined $file && -d _ || -k _; # reference
	if (defined $file && -f _ && -x _) { # method
		my $safefield=".$field-safe";
		return unless $this->$safefield;
        }
	
	return eval {$this->$field};
}

=item background

Fork to the background. It takes care of some details, and returns 1 if
the method is in the background; 0 if it is parent and not in the
background. For example:

  if ($this->background) {
  	$this->long_operation;
  }

Or:

  exit unless $this->background
  # Rest of method in background.

=cut

sub background {
	return 0 if fork;
	close STDIN;
	close STDOUT;
	# See changelog entry 0.3.0.7.0. It seems to be necessary to open
	# something here to keep the low fd's from being used.
	open STDIN, "/dev/null";
	open STDOUT, ">/dev/null";
	return 1;
}

=item prettylist

Generates a very pretty-printed list of objects, and returns it. The
object it's run on will appear in the list as "you".

=cut

sub prettylist {
	my $this=shift;
	my @objects=@_;

	return "nothing" if ! @objects;
	@objects = map { $_ == $this ? 'you' : $_->prettyname }
		grep ref, @objects;
	$objects[$#objects] = 'and '.$objects[$#objects] if @objects > 1;
	return join((@objects > 2) ? ', ' : ' ', @objects);
}

=item prettyname

Returns the object's name with any article prepended.

=cut

sub prettyname {
	my $a = $_[0]->article;
	$a.=" " if length $a;
	return $a.$_[0]->name;
}

=item untaint

If perl is run with taint checking enabled, and some method returns a mooix
object, perl will rightly consider that object's path to be tainted, and will
flag attempts to access fields of the object and so on. This method may
be used to untaint an object. It does no sanity checking, and must be used
with due caution.

The method returns the object it is run on.

=cut

sub untaint {
	($_[0]->[_USEPATH]) = $_[0]->[_USEPATH] =~ /(.*)/;
	($_[0]->[_ABSPATH]) = $_[0]->[_ABSPATH] =~ /(.*)/;
	return $_[0];
}

=back

Additional methods can be present in objects, of course. The ones above are
built-in, and can be used with no speed penalty. Additional, user-defined
methods are run as separate processes.

=head1 FIELDS

Fields may be retrieved and set in the standard perl OO-ish manner:

	$value=$thing->field;
	$thing->field($value);

Note that fields whose names start with '_' are stored on the disk as files
starting with '.', and are object internal fields.

Note that permissions may prevent some methods from retreiving or setting
some fields. If so, an exception will be thrown.

=cut

# Perl object destruction is a no-op; objects persist across perl
# invocations.
sub DESTROY {}

sub AUTOLOAD {
	(my $field = our $AUTOLOAD) =~ s/.*://;
	$field=~s/^_/./;
	my $this=shift;
	
	my $file=$this->fieldfile($field);
	if (defined $file && -f _ && -x _) {
		$this->debuglog(type => "call", field => $field, message => \@_) if $debugging;
		return $this->_runmethod($file, @_);
	}
	elsif (@_) {
		$this->debuglog(type => "write", field => $field, message => \@_) if $debugging;
		return $this->_writefield("$$this[0]/$field", @_);
	}
	elsif (defined $file && -e _) {
		$this->debuglog(type => "read", field => $field) if $debugging;
		return $this->_readfield($file);
	}
	return;
}

# Reads a field from a file. If the file is a object ref, returns the
# object ref. Otherwise, in a list context, returns an array of the 
# file's lines; in scalar context, the whole file.
#
# Note that it assumes that the file has been lstatted before it is called.
sub _readfield {
	my $this=shift;
	my $file=shift;
	
	if ((-d _ || -l _)) {
		if (-e "$file/.mooix") {
			return Mooix::Thing->get($file, $this->[_ABSPATH]);
		}
		else {
			# broken object reference
			return undef if -l $file && ! -d _ && ! -f _;
		}
	}
	
	my $fh;
	if (! open ($fh, $file)) {
		$this->croak($file);
	}
	if (wantarray) {
		my $sticky=-k _;
		my @ret = grep defined, map {
			chomp;
			if ($sticky) { # sticky = reference file
				$_ = Mooix::Thing->get($_, $this->[_ABSPATH]);
			}
			$_
		} <$fh>;
		# This explicit close is done because perl
		# seemingly does not always close $fh for me. See
		# changelog entry 0.3.0.6.3. Probably a perl bug?
		close $fh;
		return @ret;
	}
	local $/=undef;
	my $ret=<$fh>;
	close $fh;
	# I go to the trouble of chomping off the \n if any,
	# because it's too confusing for it to matter, when fields
	# are built up by echoing to them or whatever.
	local $/="\n";
	chomp $ret;
	return $ret;
}

# Writes out a field to a specified file. If the field value is an
# object ref, sets up a symlink to the object. Otherwise, write out
# a value to a file. If a whole list of values is passed, write out
# one per line.
#
# Note that if this is called in an attempt to write a field of an object
# that is not the object that runs the currently executing method, it's
# going to fail -- mood does not let that happen, ever. The solution is to
# use the setfield method of the object whose field we are trying to write.
# This function will detect such cases, and call the setfield method
# transparently, so programmers do not have to know about this quirk of
# mooix's security model.
sub _writefield {
	my $this=shift;
	my $file=shift;

	if ($this->[_USEPATH] ne '.') {
		# Non-relative usepath, so need to use setfield.
		my ($field)=$file=~m:/([^/]+)$:;
		if ($field eq 'setfield') {
			$this->croak("cannot setfield a field named setfield");
		}
		my @ret=$this->setfield($field, @_);
		if (($? >> 8) != 0) {
			$!=""; # whatever's in here is not significant
			$this->croak("setfield $field failed");
		}
	}
	else {
		if (@_ == 1 && ref $_[0] eq __PACKAGE__) {
			my $src=$_[0]->[_ABSPATH];
			# Use a tmp file and rename so that symlink
			# replacement is atomic. A good idea if eg, the
			# parent link is being changed.
			my $tmp="$file-new.$$";
			symlink($src, $tmp) || $this->croak($tmp);
			rename($tmp, $file) || $this->croak($file);
		}
		else {
			if (-l $file) {
				# A link is being converted to a non-link..
				# XXX might be good to make this be done
				# atomically too. Esp if I ever get
				# multiple inheritence.
				unlink "$file" || $this->croak("unlink $file");
			}
			open (my $fh, ">", "$file") || $this->croak($file);
			print $fh join("\n", @_);
			close $fh;
			
			if (@_ > 1 && ! grep { ref $_[0] ne __PACKAGE__} @_) {
				# The list is all objects, so mark it as a
				# reference file.
				chmod(01644, $file) || $this->croak("chmod $file");
			}
		}
	}

	# Return our input.
	return @_ if wantarray;
	return $_[0];
}

# Run the specified file as a method. The rest of @_ is passed into it
# a line at a time on stdin, and its stdout is read, processed (split into
# lines, object id's replaced with instantiated objects), and returned.
#
# This should work inside or outside the moo: outside the moo it uses
# runmeth to run moo methods.
sub _runmethod {
	my $this=shift;
	my $file=shift;

	# We have to use the dreaded bi-directional pipes, and IPC::Open2
	# is too meaty, so I'll do it myself. Use lexical filehandles to
	# allow multiple methods to be run at once.
	my ($child_wtr, $child_rdr);
	pipe(PARENT_RDR, $child_wtr);
	pipe($child_rdr, PARENT_WTR);

	my $pid;
	if ($pid = fork) {
		# Ignore sigpipes, which can easily occur if the child is
		# very quick to run and does not read its input.
		$SIG{PIPE}=sub {};
		
		close PARENT_RDR;
		close PARENT_WTR;
		
		# Argument passing.
		select $child_wtr; $|=1; select STDOUT; # autoflush
		foreach (@_) {
			if (! ref $_) {
				my $f=$_;
				chomp $f;
				$f =~ s/\\/\\\\/g; # eacape slashes
				$f =~ s/\n/\\n/g;  # and newlines in parameter values
				print $child_wtr "\"$f\"\n";
			}
			else {
				print $child_wtr "$_\n";
			}
		}
		close $child_wtr; # let child know we're done so it can run
		
		# Handle exec flag. 
		if ($this->[_EXEC_NEXT]) {
			print <$child_rdr>; # pass on to caller parent method
			close $child_rdr;
			waitpid($pid, 0);
			exit $? >> 8;
		}

		# Result processing.
		my @ret = map {
			chomp;
			if (/^mooix/) {
				$_=Mooix::Thing->get($_)
			}
			elsif (/^"(.*)"$/) {
				$_=$1;
				s/(?<!\\)\\n/\n/g;  # Unescape newlines
				s/\\\\/\\/g; # and slashes.
			}
			$_
		} <$child_rdr>;
		close $child_rdr;
		
		waitpid($pid, 0);
		
		# Only log returns if the caller is paying attention to the
		# return value. Since @ret is passed as the value, it may
		# be changed by this call.
		if ($debugging && defined wantarray) {
			$this->debuglog(type => "return", field => $file,
			                message => \@ret, replace => 1,
					value => \@ret)
		}
		
		return @ret if wantarray;
		# So what to do if they don't want an array? If I return
		# @ret, in a scalar context they'll get a number, which is
		# wrong. I'd like to return the entire method output, but
		# that is not workable because I've changed some mooix:
		# items in it into object references. So I've settled for
		# just returning the first thing.
		return $ret[0];
	}
	else {
		close $child_rdr;
		close $child_wtr;
	
		open(STDIN, "<&PARENT_RDR");
		open(STDOUT, ">&PARENT_WTR");
		
		chdir($this->[_ABSPATH]) || $this->croak("chdir");
		# If the filename is absolute, remove the dir
		# we just cd'd into from it.
		$file=~s/^\Q$$this[0]\E\///;
		# './' for PATH issues
		if (exists $ENV{THIS}) { # in the moo
			CORE::exec("./$file") || $this->croak("exec ./$file");
		}
		else {
			CORE::exec("runmeth", "./$file") || $this->croak("runmeth ./$file");
		}
	}
}

=head1 OPERATOR OVERLOADING

=over 4

Some operator overloading is done so that operations on Mooix::Thing objects
behave in a useful way:

=item stringification

Stringifying a Mooix::Thing will result in the thing's id.

=item numeric comparisons

Comparing two Mooix::Things numerically will make sure they were instantiated
as the same directory (by comparing dev and inode numbers).

=back

=cut

use overload
	'""' => sub { "mooix:".$_[0]->[_ABSPATH] },
	'<=>' => sub {
		my $a = shift; # prevent autovivification
		my $b = shift; # of aliases..
		$a->[_INDEX] cmp $b->[_INDEX]
	},
	'eq' => sub {
		my $a = shift;
		my $b = shift;
		$a->[_ABSPATH] eq $b->[_ABSPATH]
	};

=head1 COPYRIGHT
	
Copyright 2001-2003 by Joey Hess <joey@mooix.net>
under the terms of the modified BSD license given in full in the file
COPYRIGHT.

=head1 AUTHOR

Joey Hess <joey@mooix.net>

=cut

1
