#!/usr/bin/perl

package Mooix::Error;

#Turn off in production code!
#use strict;
#use warnings;

=head1 NAME

Mooix::Error - mooix error exception class

=head1 DESCRIPTION

This is a class that is thrown as an exception.

=head1 METHODS

=over 4

=item new

Return a new error object. Pass in the mooix object that is experiencing the
error, and a string describing the error.

=cut

my $dying=0;

sub new {
	my $proto = shift;
	my $class = ref($proto) || $proto;
	my $self  = {};
	$self->{object} = shift;
	$self->{message} = shift;
	bless ($self, $class);
	return $self;
}

=item get

Gets the error message, and a mooix callstack trace too.

=cut

sub get {
	my $this=shift;
	return $this->{message}."\ncallstack:\n";
	       join("\n\t", $this->{object}->calltrace)."\n";
}

=head1 OPERATOR OVERLOADING

=over 4

When a Mooix::Error object is stringified, it returns the result of the
get method, so if one is thrown and not caught, the program dies as expected.

=back

=cut

use overload '""' => sub {
	my $this=shift;
	
	# This is completly gross beyond belief. The problem is that
	# the calltrace is very expensive to get. We'd like to only get it
	# if the message will really be displayed to the user, and not if
	# the exception is caught and discarded. However, it seems that
	# perl currently stringifies excpetions each time they go through
	# die, even if that string is just thrown away. Which eliminated
	# that hoped for speed gain. To get back the desired behavior, I
	# use this utterly gross hack of hooking into the global __DIE__
	# handler, and displaying the calltrace if the message we're dying
	# of contains the special token, "\ncallstack:\n". Of course, we
	# add that to the message when stringifying, below.
	#
	# XXX This is in the pipeline to be fixed in perl 5.8.1
	$SIG{__DIE__} = sub {
		my $error=shift;
		return if $error !~ /\ncallstack:\n/s;
		return if ! defined $^S || $^S; # inside eval
		if (! $dying) {
			$dying=1;
			die $error."\t".join("\n\t", $this->{object}->calltrace)."\n"
		}
		else {
			die "error generating calltrace";
		}
			
	};

	$this->{message}."\ncallstack:\n";
};

=head1 COPYRIGHT
	
Copyright 2003 by Joey Hess <joey@mooix.net>
under the terms of the modified BSD license given in full in the file
COPYRIGHT.

=head1 AUTHOR

Joey Hess <joey@mooix.net>

=cut

1
