# copyright (C) 1997-2004 Jean-Luc Fontaine (mailto:jfontain@free.fr)
# this program is free software: please read the COPYRIGHT file enclosed in this package or use the Help Copyright menu

# $Id: interrupts.tcl,v 1.14 2004/01/01 11:39:06 jfontain Exp $


package provide interrupts [lindex {$Revision: 1.14 $} 1]
package require network 1


namespace eval interrupts {

    variable nextIndex 0
    variable cpuColumn 3                                                                         ;# CPU(s) data start at this column

    array set data {
        updates 0
        0,label number 0,type dictionary 0,message {interrupt number or identification: NMI (Non Maskable Interrupt), LOC (local interrupt counter of the internal APIC) or ERR (incremented in the case of errors in the IO-APIC bus)}
        1,label device 1,type dictionary 1,message {device name}
        2,label type 2,type ascii 2,message {interrupt type}
        switches {-r 1 --remote 1}
        sort {0 increasing}
    }
    set file [open interrupts.htm]
    set data(helpText) [read $file]                                                           ;# initialize HTML help data from file
    close $file

    proc initialize {optionsName} {
        upvar 1 $optionsName options
        variable remote
        variable data
        variable last
        variable interruptsFile
        variable cpuColumn
        variable cpus

        if {![catch {set locator $options(--remote)}] || ![catch {set locator $options(-r)}]} {                 ;# remote monitoring
            set data(pollTimes) {20 10 30 60 120 300 600}                                ;# poll less often when remotely monitoring
            foreach {remote(protocol) remote(user) remote(host)} [network::parseRemoteLocator $locator] {}
            network::checkRemoteOutputEmptiness $remote(protocol) $remote(user) $remote(host)
            set data(identifier) interrupts($remote(host))
            if {[string equal $::tcl_platform(platform) unix]} {
                set remote(command) "$remote(protocol) -n -l $remote(user) $remote(host) cat /proc/interrupts"
            } else {                                                                                                      ;# windows
                set remote(command) "plink -batch $remote(host) cat /proc/interrupts"              ;# host is rather a putty session
            }
            set file [open "| $remote(command)"]
            fileevent $file readable {set ::interrupts::remote(busy) 0}
            vwait ::interrupts::remote(busy)
            # detect errors early (but avoid write on pipe with no readers errors by reading whole data)
            if {[catch {gets $file line} message] || [catch {read $file} message] || [catch {close $file} message]} {
                error "on remote host $remote(host) as user $remote(user): $message"
            }
        } else {
            set data(pollTimes) {10 5 20 30 60 120 300 600}
            set interruptsFile [open /proc/interrupts]                                ;# keep local file open for better performance
            gets $interruptsFile line                                                     ;# only first line is needed at this point
        }
        set cpus 0
        set column $cpuColumn                                                                 ;# add CPU columns to existing columns
        foreach cpu $line {                                                    ;# line format: CPU0 CPU1 ... (with blank separators)
            scan $cpu CPU%u index
            set data($column,label) "count($index)"
            set data($column,type) real
            set data($column,message) "interrupts count for CPU $index"
            incr column
            set data($column,label) "rate($index)"
            set data($column,type) real
            set data($column,message) "interrupts per second for CPU $index"
            incr column
            incr cpus
        }
        if {$cpus == 0} {
            error "invalid /proc/interrupts first line: \"$line\""
        }
    }

    proc update {} {
        variable remote
        variable interruptsFile
        variable index
        variable nextIndex
        variable cpus
        variable data
        variable last
        variable cpuColumn

        if {[info exists remote]} {
            if {![info exists interruptsFile]} {                           ;# start data gathering process in a non blocking fashion
                if {$remote(busy)} return                                           ;# core invocation while waiting for remote data
                set remote(busy) 1
                set file [open "| $remote(command)"]
                # do not hang GUI, allow other modules updates
                fileevent $file readable "set ::interrupts::interruptsFile $file; ::interrupts::update"
                return                                                                                       ;# wait for remote data
            }                                                                                 ;# else continue below to process data
        } else {
            seek $interruptsFile 0                                                                  ;# rewind before retrieving data
        }
        gets $interruptsFile line                                                                                ;# skip header line
        set clock [expr {[clock clicks -milliseconds] / 1000.0}]                       ;# immediately store current clock in seconds
        if {[info exists last(clock)]} {
            set period [expr {$clock - $last(clock)}]
        }
        while {[gets $interruptsFile line] >= 0} {
            # examples:
            #    1:       8949       8958    IO-APIC-edge  keyboard
            #  NMI:    2457961    2457959 
            #  LOC:    2457882    2457881 
            #  ERR:       2155
            set column 0
            set device {}
            foreach value $line {
                if {$column == 0} {                                                     ;# number or identification is followed by :
                    set value [string trimright $value :]
                    if {[catch {set row $index($value)}]} {                                                             ;# new entry
                        set row [set index($value) $nextIndex]
                        incr nextIndex
                    }
                    set data($row,0) $value
                    set data($row,2) {}                                                  ;# type is undefined for NMI, LOC, ERR, ...
                    for {set cpu 0; set number $cpuColumn} {$cpu < $cpus} {incr cpu} {
                        # in case of data error or NMI, LOC, ERR, ... with single value even if several CPUs:
                        set data($row,$number) ?; incr number
                        set data($row,$number) ?; incr number
                    }
                } elseif {$column <= $cpus} {                                                              ;# CPU interrupt counters
                    set cpu [expr {$column - 1}]
                    set number [expr {$cpuColumn + (2 * $cpu)}]
                    set data($row,$number) $value                                                                           ;# count
                    incr number
                    if {[info exists last($row,$cpu)]} {
                        set data($row,$number) [format %.1f [expr {int($value - $last($row,$cpu)) / $period}]]  ;# rate (per second)
                    }
                    set last($row,$cpu) $value
                } elseif {$column == $cpus+1} {                                                                              ;# type
                    set data($row,2) $value
                } else {                                                                         ;# device name (may contain spaces)
                    if {[string length $device] > 0} {
                        append device { }
                    }
                    append device $value
                }
                incr column
            }
            set data($row,1) $device
        }
        if {[info exists remote]} {                                                 ;# closing is necessary since seek does not work
            read $interruptsFile                             ;# avoid write on pipe with no readers errors by reading remaining data
            if {[catch {close $interruptsFile} message]} {                               ;# communication error can be detected here
                foreach {name row} [array get index] {                                      ;# set all statistics data cells to void
                    for {set cpu 0; set number $cpuColumn} {$cpu < $cpus} {incr cpu} {
                        set data($row,$number) ?; incr number
                        set data($row,$number) ?; incr number
                        catch {unset last($row,$cpu)}                            ;# reset last values since they are no longer valid
                    }
                }
                flashMessage "error: $message"
            }
            unset interruptsFile
            set remote(busy) 0
        }
        set last(clock) $clock
        incr data(updates)
    }

}
