# copyright (C) 1997-2004 Jean-Luc Fontaine (mailto:jfontain@free.fr)
# this program is free software: please read the COPYRIGHT file enclosed in this package or use the Help Copyright menu

# $Id: gui.tcl,v 2.103 2004/01/01 11:39:06 jfontain Exp $


set menuHelp(file,new) [mc {unload all modules and remove all viewers}]
set menuHelp(file,open) [mc {load and replace modules from a file}]
set menuHelp(file,saveAs) [mc {input file for saving configuration and viewers layout}]
set menuHelp(file,print) [mc {print or save canvas area to file in postscript}]
set menuHelp(file,exit) [mc {close main window and exit program}]
set menuHelp(edit,configuration) [mc {edit current configuration}]
set menuHelp(view,refresh) [mc {refresh display of all synchronous modules}]
set menuHelp(view,refreshDatabase) [mc {update display from database according to cursors positions}]
set menuHelp(help,global) [mc {global help for user}]
set menuHelp(file,database,load) [mc {select stored module instances}]
set menuHelp(file,database,start) [mc {start recording cells data history in database}]
set menuHelp(file,database,stop) [mc {stop recording cells data history in database}]


namespace eval controls {
    variable fileDatabaseMenu
    variable fileDatabaseMenuStartIndex
    variable fileDatabaseStartButton
}

proc updateTitle {} {                                                          ;# show modules and poll time or mode in window title
    foreach {loadedUserModules anySynchronousModules databaseConnected databaseMode} [userInputStates] {}
    if {$databaseMode} {
        set title [mc {moodss: database history}]
        wm title . $title
        wm iconname . $title
    } elseif {$loadedUserModules} {
        set names [modules::names]
        if {$anySynchronousModules} {
            wm title . [format [mc {moodss: %s data (every %u seconds)}] [commaSeparatedString $names] $global::pollTime]
        } else {
            wm title . [format [mc {moodss: %s data (asynchronous)}] [commaSeparatedString $names]]
        }
        wm iconname . "moodss $names"
    } else {
        wm title . moodss
        wm iconname . moodss                                                                      ;# give the icon a meaningful name
    }
}

proc underlineRemoved {string indexName} {     ;# underline is placed after character to underline, underlines are escaped with self
    upvar 1 $indexName index                         ;# for example: passing A_bcd results in index set to 0 and Abcd to be returned

    set indices 0
    regexp -indices _ $string indices
    set index [expr {[lindex $indices 0] - 1}]
    regsub _ $string {} string
    return $string
}

proc updateMenuWidget {{parentPath .}} {
    static help
    static cache

    foreach {loadedUserModules anySynchronousModules databaseConnected databaseMode} [userInputStates] {}
    set moduleNames [modules::names]
    if {[info exists cache]} {
        set change 0
        foreach {name value} [array get cache] {
            if {![string equal [set $name] $value]} {set change 1; break}
        }
        if {!$change} return                                                                ;# avoid updates when there is no change
    }
    foreach name {loadedUserModules anySynchronousModules databaseMode databaseConnected global::readOnly moduleNames} {
        set cache($name) [set $name]                           ;# the menu configuration is uniquely determined from these variables
    }

    set menu .[string trimleft $parentPath.menu .]                                   ;# remove double dot when parent is main widget
    if {[winfo exist $menu]} {                                                   ;# delete all entries and destroy all related menus
        for {set entry 0} {[string length [$menu type $entry]]} {incr entry} {
            destroy [$menu entrycget $entry -menu]
            $menu delete $entry
        }
    } else {                                                                                                 ;# create menu bar once
        menu $menu -tearoff 0
        frame $menu.bound                              ;# create a frame for bindings that otherwise would propagate to all children
    }
    if {![info exists help(bar)]} {
        set help(bar) [new menuContextHelp $menu]
    }
    set index(bar) -1

    menu $menu.file -tearoff 0
    if {![info exists global::fileMenuContextHelper]} {
        set global::fileMenuContextHelper [new menuContextHelp $menu.file]
    }
    set index(file) -1

    set string [underlineRemoved [mc F_ile] underline]
    $menu add cascade -label $string -menu $menu.file -underline $underline
    menuContextHelp::set $help(bar) [incr index(bar)] [mc {file related operations}]

    catch {unset global::fileMenuContextHelperSaveIndex}
    bind $parentPath <Control-n> {}                                        ;# first reset bindings which may be optionally set below
    bind $parentPath <Control-o> {}
    bind $parentPath <Control-s> {}
    bind $parentPath <Control-a> {}
    if {!$global::readOnly} {
        set string [underlineRemoved [mc N_ew] underline]
        $menu.file add command -label $string... -command clear -underline $underline -accelerator Ctrl+N
        bind $parentPath <Control-n> clear
        menuContextHelp::set $global::fileMenuContextHelper [incr index(file)] $::menuHelp(file,new)

        set string [underlineRemoved [mc O_pen] underline]
        $menu.file add command -label $string... -command reload -underline $underline -accelerator Ctrl+O
        bind $parentPath <Control-o> reload
        menuContextHelp::set $global::fileMenuContextHelper [incr index(file)] $::menuHelp(file,open)

        set string [underlineRemoved [mc S_ave] underline]
        $menu.file add command -label $string -command {save 0} -underline $underline -accelerator Ctrl+S
        bind $parentPath <Control-s> {save 0}
        set global::fileMenuContextHelperSaveIndex [incr index(file)]
        if {($loadedUserModules == 0) && !$databaseMode} {                                ;# saving is also allowed in database mode
            $menu.file entryconfigure $index(file) -state disabled
        }
        set string [underlineRemoved [mc {Save A_s}] underline]
        $menu.file add command -label $string... -command {save 1} -underline $underline -accelerator Ctrl+A
        bind $parentPath <Control-a> {save 1}
        menuContextHelp::set $global::fileMenuContextHelper [incr index(file)] $::menuHelp(file,saveAs)
        if {($loadedUserModules == 0) && !$databaseMode} {
            $menu.file entryconfigure $index(file) -state disabled
        }
    }

    set string [underlineRemoved [mc M_odules] underline]
    $menu.file add cascade -label $string -menu [menu $menu.file.modules -tearoff 0] -underline $underline
    menuContextHelp::set $global::fileMenuContextHelper [incr index(file)] [mc {operations on modules}]

    if {![info exists help(modules)]} {
        set help(modules) [new menuContextHelp $menu.file.modules]
    }
    set index(modules) -1
    set disable 1
    if {!$databaseMode} {                                  ;# disallow managing instance module and loading modules in database mode
        if {$global::readOnly} {
            if {[llength $moduleNames] > 0} {
                set disable 0
                set string [underlineRemoved [mc Load_ed] underline]
                $menu.file.modules add command -label $string... -command {new moduleOperations display} -underline $underline
                menuContextHelp::set $help(modules) [incr index(modules)] [mc {view loaded modules and their options}]
            }
        } else {
            set disable 0
            set string [underlineRemoved [mc L_oad] underline]
            $menu.file.modules add command -label $string... -command {new moduleOperations load} -underline $underline
            menuContextHelp::set $help(modules) [incr index(modules)] [mc {load new module instances}]
            if {[llength $moduleNames] > 0} {
                set string [underlineRemoved [mc M_anage] underline]
                $menu.file.modules add command -label $string... -command {new moduleOperations manage} -underline $underline
                menuContextHelp::set $help(modules) [incr index(modules)] [mc {unload, reconfigure modules or create new instances}]
            }
        }
    }
    if {$disable} {
        $menu.file entryconfigure $index(file) -state disabled
    }

    if {!$global::readOnly} {
        set string [underlineRemoved [mc D_atabase] underline]
        $menu.file add cascade -label $string -menu [menu $menu.file.database -tearoff 0] -underline $underline
        set ::controls::fileDatabaseMenu $menu.file.database
        menuContextHelp::set $global::fileMenuContextHelper [incr index(file)] [mc {operations on database}]
        if {![info exists help(database)]} {
            set help(database) [new menuContextHelp $menu.file.database]
        }
        set index(database) -1
        set string [underlineRemoved [mc L_oad] underline]
        $menu.file.database add command -label $string... -command "loadFromDatabase [expr {!$global::readOnly}] $global::static"\
            -underline $underline
        menuContextHelp::set $help(database) [incr index(database)] $::menuHelp(file,database,load)
        if {!$databaseMode && ($loadedUserModules > 0)} {
            set string [underlineRemoved [mc Sta_rt] underline]
            $menu.file.database add command -label $string -command {databaseRecording 1} -underline $underline
            menuContextHelp::set $help(database) [incr index(database)] $::menuHelp(file,database,start)
            set ::controls::fileDatabaseMenuStartIndex $index(database)
            if {$global::database != 0} {                             ;# connected to database: currently storing data cells history
                $menu.file.database entryconfigure $index(database) -state disabled
            }
            set string [underlineRemoved [mc Sto_p] underline]
            $menu.file.database add command -label $string -command {databaseRecording 0} -underline $underline
            menuContextHelp::set $help(database) [incr index(database)] $::menuHelp(file,database,stop)
            if {$global::database == 0} {                              ;# disconnected from database: not storing data cells history
                $menu.file.database entryconfigure $index(database) -state disabled
            }
        }
    }

    set string [underlineRemoved [mc P_rint] underline]
    $menu.file add command -label $string... -command $global::printDialogCommand -underline $underline -accelerator Ctrl+P
    menuContextHelp::set $global::fileMenuContextHelper [incr index(file)] $::menuHelp(file,print)
    bind $parentPath <Control-p> {}                                         ;# first reset binding which may be optionally set below
    if {[string equal $::tcl_platform(platform) unix] || ![catch {package require Tkprint 1.1}]} {
        bind $parentPath <Control-p> $global::printDialogCommand
    } else {
        $menu.file entryconfigure $index(file) -state disabled
    }
    $menu.file add separator
    incr index(file)
    set string [underlineRemoved [mc E_xit] underline]
    $menu.file add command -label $string -command exit -underline $underline -accelerator Ctrl+Q
    menuContextHelp::set $global::fileMenuContextHelper [incr index(file)] $::menuHelp(file,exit)
    bind $parentPath <Control-q> exit

    if {!$global::readOnly} {
        set string [underlineRemoved [mc E_dit] underline]
        $menu add cascade -label $string -menu [menu $menu.edit -tearoff 0] -underline $underline
        set index(edit) -1
        if {![info exists help(edit)]} {
            set help(edit) [new menuContextHelp $menu.edit]
        }
        menuContextHelp::set $help(bar) [incr index(bar)] [mc {content editing, configuration and preferences}]

        set string [underlineRemoved [mc T_hresholds] underline]
        $menu.edit add command -label $string... -command {thresholds::edit} -underline $underline
        menuContextHelp::set $help(edit) [incr index(edit)] [mc {edit data thresholds}]
        if {$databaseMode || ($loadedUserModules == 0)} {
            $menu.edit entryconfigure $index(edit) -state disabled
        } else {
            $menu.edit entryconfigure $index(edit) -state normal
        }

        set string [underlineRemoved [mc C_onfiguration] underline]
        $menu.edit add command -label $string... -command {configuration::edit 0} -underline $underline
        menuContextHelp::set $help(edit) [incr index(edit)] $::menuHelp(edit,configuration)

        set string [underlineRemoved [mc {New P_age}] underline]
        $menu.edit add command\
            -label $string... -command {pages::edit [new page . -draggable [expr {!$global::readOnly}]]} -underline $underline
        menuContextHelp::set $help(edit) [incr index(edit)] [mc {create a new page}]

        set string [underlineRemoved [mc {New V_iewer}] underline]
        $menu.edit add cascade -label $string -menu [menu $menu.edit.new -tearoff 0] -underline $underline
        if {![info exists help(new)]} {
            set help(new) [new menuContextHelp $menu.edit.new]
        }
        menuContextHelp::set $help(edit) [incr index(edit)] [mc {create empty data viewers}]

        set string [underlineRemoved [mc {G_raph Chart}] underline]
        $menu.edit.new add command -label $string... -underline $underline\
            -command "createCellsViewer dataGraph {} 1 $global::static \$global::pollTime"
        menuContextHelp::set $help(new) 0 [mc {create an empty graph chart data viewer}]

        set string [underlineRemoved [mc {Stacked Graph C_hart}] underline]
        $menu.edit.new add command -label $string... -underline $underline\
            -command "createCellsViewer dataStackedGraph {} 1 $global::static \$global::pollTime"
        menuContextHelp::set $help(new) 1 [mc {create an empty stacked graph chart data viewer}]

        set string [underlineRemoved [mc {O_verlap Bar Chart}] underline]
        $menu.edit.new add command -label $string... -underline $underline\
            -command "createCellsViewer dataOverlapBarChart {} 1 $global::static"
        menuContextHelp::set $help(new) 2 [mc {create an empty overlap bar chart data viewer}]

        set string [underlineRemoved [mc {Side B_ar Chart}] underline]
        $menu.edit.new add command -label $string... -underline $underline\
            -command "createCellsViewer dataSideBarChart {} 1 $global::static"
        menuContextHelp::set $help(new) 3 [mc {create an empty side bar chart data viewer}]

        set string [underlineRemoved [mc {S_tacked Bar Chart}] underline]
        $menu.edit.new add command -label $string... -underline $underline\
            -command "createCellsViewer dataStackedBarChart {} 1 $global::static"
        menuContextHelp::set $help(new) 4 [mc {create an empty stacked bar chart data viewer}]

        set string [underlineRemoved [mc {2_D Pie Chart}] underline]
        $menu.edit.new add command -label $string... -underline $underline\
            -command "createCellsViewer data2DPieChart {} 1 $global::static"
        menuContextHelp::set $help(new) 5 [mc {create an empty 2D pie chart data viewer}]

        set string [underlineRemoved [mc {3_D Pie Chart}] underline]
        $menu.edit.new add command -label $string... -underline $underline\
            -command "createCellsViewer data3DPieChart {} 1 $global::static"
        menuContextHelp::set $help(new) 6 [mc {create an empty 3D pie chart data viewer}]

        set string [underlineRemoved [mc {V_alues Table}] underline]
        $menu.edit.new add command -label $string... -underline $underline\
            -command "createCellsViewer currentValueTable {} 1 $global::static \$global::pollTime"
        menuContextHelp::set $help(new) 7 [mc {create an empty values table data viewer}]

        set string [underlineRemoved [mc {Statistics T_able}] underline]
        $menu.edit.new add command -label $string... -underline $underline\
            -command "createCellsViewer summaryTable {} 1 $global::static \$global::pollTime"
        menuContextHelp::set $help(new) 8 [mc {create an empty statistics table data viewer}]

        set string [underlineRemoved [mc {F_ree Text}] underline]
        $menu.edit.new add command -label $string... -underline $underline\
            -command "createCellsViewer freeText {} 1 $global::static"
        menuContextHelp::set $help(new) 9 [mc {create an empty free text data viewer}]

        set string [underlineRemoved [mc {D_atabase}] underline]
        $menu.edit add command -label $string... -underline $underline
        menuContextHelp::set $help(edit) [incr index(edit)] [mc {select cells the history of which to store in database}]
        if {$databaseMode || ($loadedUserModules == 0)} {
            $menu.edit entryconfigure $index(edit) -state disabled
        } else {
            $menu.edit entryconfigure $index(edit) -state normal
            if {$global::database == 0} {                              ;# disconnected from database: not storing data cells history
                $menu.edit entryconfigure $index(edit)\
                    -command {updateDatabaseStart disabled; store::edit 1 {updateDatabaseStart normal}}
            } elseif {$loadedUserModules > 0} {                                              ;# currently storing data cells history
                $menu.edit entryconfigure $index(edit) -command {store::edit 0 {}}               ;# only display archived data cells
            }
        }

        $menu.edit add separator
        incr index(edit)
        set string [underlineRemoved [mc P_references] underline]
        $menu.edit add command -label $string... -command {configuration::edit 1} -underline $underline
        menuContextHelp::set $help(edit) [incr index(edit)] [mc {edit application-wide preferences}]
    }

    set string [underlineRemoved [mc V_iew] underline]
    $menu add cascade -label $string -menu [menu $menu.view -tearoff 0] -underline $underline
    if {![info exists help(options)]} {
        set help(options) [new menuContextHelp $menu.view]
    }
    menuContextHelp::set $help(bar) [incr index(bar)] [mc {data visualization settings}]
    set index(view) -1
    bind $parentPath <Control-r> {}                                        ;# first reset bindings which may be optionally set below
    if {!$global::readOnly && ((($loadedUserModules > 0) && $anySynchronousModules) || $databaseMode)} {
        set string [underlineRemoved [mc R_efresh] underline]
        $menu.view add command -label $string -command {after idle ::refresh} -underline $underline -accelerator Ctrl+R
        bind $parentPath <Control-r> {after idle ::refresh}
        if {$databaseMode} {
            menuContextHelp::set $help(options) [incr index(view)] $::menuHelp(view,refreshDatabase)
        } else {
            menuContextHelp::set $help(options) [incr index(view)] $::menuHelp(view,refresh)
        }
    }
    if {$databaseMode} {
        set string [underlineRemoved [mc {D_atabase Range}] underline]
        $menu.view add command -label $string... -command database::displayAndSelectRange -underline $underline
        menuContextHelp::set $help(options) [incr index(view)] [mc {select time range for database history views}]
    }
    if {$loadedUserModules > 0} {                                            ;# only useful when there is at least one loaded module
        if {!$global::readOnly && $anySynchronousModules} {
            set string [underlineRemoved [mc {P_oll Time}] underline]
            $menu.view add command -label $string... -command inquirePollTime -underline $underline
            menuContextHelp::set $help(options) [incr index(view)] [mc {change poll time for all synchronous modules}]
        }
        set string [underlineRemoved [mc T_race] underline]
        $menu.view add checkbutton -label $string -command {residentTraceModule $global::showTrace} -underline $underline\
            -variable global::showTrace -offvalue 0 -onvalue 1
        menuContextHelp::set $help(options) [incr index(view)] [mc {show or hide loaded modules informational, error, ... messages}]
    }
    set string [underlineRemoved [mc {Tool B_ar}] underline]
    $menu.view add checkbutton -label $string\
        -command manageToolBar -underline $underline -variable global::showToolBar -offvalue 0 -onvalue 1
    menuContextHelp::set $help(options) [incr index(view)] [mc {show or hide the bar with tool icons}]

    if {[string equal $::tcl_platform(platform) windows]} {
        set string ?
        set underline 0
    } else {
        set string [underlineRemoved [mc H_elp] underline]
    }
    $menu add cascade -label $string -menu [menu $menu._help -tearoff 0] -underline $underline
    if {![info exists help(help)]} {
        set help(help) [new menuContextHelp $menu._help]                  ;# do not use help name which would right justify the menu
    }
    set index(help) -1
    menuContextHelp::set $help(bar) [incr index(bar)] [mc {help on moodss and modules}]

    set string [underlineRemoved [mc G_lobal] underline]
    $menu._help add command -label $string... -underline 0 -accelerator F1 -command generalHelpWindow
    bind $parentPath <F1> generalHelpWindow
    menuContextHelp::set $help(help) [incr index(help)] $::menuHelp(help,global)

    if {([llength $moduleNames] > 0) && !$databaseMode} {
        # display modules help menu only if they are any loaded and not in database mode (instance module help not available)
        set string [underlineRemoved [mc M_odules] underline]
        $menu._help add cascade -label $string -menu [menu $menu._help.modules -tearoff 0] -underline $underline
        menuContextHelp::set $help(help) [incr index(help)] [mc {help on loaded modules}]
        if {![info exists help(modulesHelp)]} {
            set help(modulesHelp) [new menuContextHelp $menu._help.modules]
        }
        set index(modulesHelp) -1
        foreach module $moduleNames {
            $menu._help.modules add command -label $module... -command "moduleHelpWindow $module \[modules::helpHTMLData $module\]"
            menuContextHelp::set $help(modulesHelp) [incr index(modulesHelp)]\
                [format [mc {display %s module documentation}] $module]
        }
    }

    set string [underlineRemoved [mc A_bout] underline]
    $menu._help add command -label $string... -underline $underline -command aboutDialogBox
    menuContextHelp::set $help(help) [incr index(help)] [mc {display author and general information}]

    $parentPath configure -menu $menu
}

proc updateFileSaveHelp {file} {
    if {![info exists global::fileMenuContextHelperSaveIndex]} return                                  ;# file save menu not visible
    if {[string length $file] == 0} {
        set string [mc {input file for saving configuration and viewers layout}]
    } else {
        set string [format [mc {into %s file, save configuration and viewers layout}] $file]
    }
    menuContextHelp::set $global::fileMenuContextHelper $global::fileMenuContextHelperSaveIndex $string
    switched::configure $global::fileSaveHelpTip -text $string
}

proc createMessageWidget {parentPath} {
    # make sure eventual initial messages are visible by presetting width:
    set frame [frame $parentPath.messenger -borderwidth $widget::option(button,borderwidth) -relief sunken]
    set global::messenger [new lifoLabel $frame -font $font::(mediumNormal) -borderwidth 0]
    composite::configure $global::messenger base -width 300                       ;# make sure eventual initial messages are visible
    pack $widget::($global::messenger,path) -side right -fill both -expand 1                        ;# save room for threshold label
    return $frame
}

proc dragEcho {data format} {
    return $data
}

proc updateDragAndDropZone {{parentPath .}} {
    static drop
    static cache

    set databaseMode [lindex [userInputStates] end]
    if {[info exists cache] && [string equal $cache(databaseMode) $databaseMode]} return    ;# avoid updates when there is no change
    set cache(databaseMode) $databaseMode            ;# the drag'n'drop zone configuration is uniquely determined from this variable

    set frame .[string trimleft $parentPath.drops .]                                 ;# remove double dot when parent is main widget
    if {![info exists drop]} {
        frame $frame

        set label [label $frame.graph -image applicationIcon -relief sunken]
        pack $label -pady 1 -side left
        new dropSite -path $label -formats DATACELLS\
            -command "createCellsViewer dataGraph \$dragSite::data(DATACELLS) 1 $global::static \$global::pollTime"
        new widgetTip -path $label -text [mc {graph chart drag'n'drop site}]
        set drag [new dragSite -path $label]
        dragSite::provide $drag VIEWER {dragEcho ::dataGraph}

        set label [label $frame.stackedGraph -image [image create photo -data [dataStackedGraph::iconData]] -relief sunken]
        pack $label -pady 1 -side left
        new dropSite -path $label -formats DATACELLS\
            -command "createCellsViewer dataStackedGraph \$dragSite::data(DATACELLS) 1 $global::static \$global::pollTime"
        new widgetTip -path $label -text [mc {stacked graph chart drag'n'drop site}]
        set drag [new dragSite -path $label]
        dragSite::provide $drag VIEWER {dragEcho ::dataStackedGraph}

        set label [label $frame.overlapBarChart -image [image create photo -data [dataOverlapBarChart::iconData]] -relief sunken]
        pack $label -pady 1 -side left
        new dropSite -path $label -formats DATACELLS\
            -command "createCellsViewer dataOverlapBarChart \$dragSite::data(DATACELLS) 1 $global::static"
        new widgetTip -path $label -text [mc {overlap bar chart drag'n'drop site}]
        set drag [new dragSite -path $label]
        dragSite::provide $drag VIEWER {dragEcho ::dataOverlapBarChart}

        set label [label $frame.sideBarChart -image [image create photo -data [dataSideBarChart::iconData]] -relief sunken]
        pack $label -pady 1 -side left
        new dropSite -path $label -formats DATACELLS\
            -command "createCellsViewer dataSideBarChart \$dragSite::data(DATACELLS) 1 $global::static"
        new widgetTip -path $label -text [mc {side bar chart drag'n'drop site}]
        set drag [new dragSite -path $label]
        dragSite::provide $drag VIEWER {dragEcho ::dataSideBarChart}

        set label [label $frame.stackedBarChart -image [image create photo -data [dataStackedBarChart::iconData]] -relief sunken]
        pack $label -pady 1 -side left
        new dropSite -path $label -formats DATACELLS\
            -command "createCellsViewer dataStackedBarChart \$dragSite::data(DATACELLS) 1 $global::static"
        new widgetTip -path $label -text [mc {stacked bar chart drag'n'drop site}]
        set drag [new dragSite -path $label]
        dragSite::provide $drag VIEWER {dragEcho ::dataStackedBarChart}

        set label [label $frame.2DPieChart -image [image create photo -data [data2DPieChart::iconData]] -relief sunken]
        pack $label -pady 1 -side left
        new dropSite -path $label -formats DATACELLS\
            -command "createCellsViewer data2DPieChart \$dragSite::data(DATACELLS) 1 $global::static"
        new widgetTip -path $label -text [mc {2D pie chart drag'n'drop site}]
        set drag [new dragSite -path $label]
        dragSite::provide $drag VIEWER {dragEcho ::data2DPieChart}

        set label [label $frame.3DPieChart -image [image create photo -data [data3DPieChart::iconData]] -relief sunken]
        pack $label -pady 1 -side left
        new dropSite -path $label -formats DATACELLS\
            -command "createCellsViewer data3DPieChart \$dragSite::data(DATACELLS) 1 $global::static"
        new widgetTip -path $label -text [mc {3D pie chart drag'n'drop site}]
        set drag [new dragSite -path $label]
        dragSite::provide $drag VIEWER {dragEcho ::data3DPieChart}

        set label [label $frame.currentValueTable -image [image create photo -data [currentValueTable::iconData]] -relief sunken]
        pack $label -pady 1 -side left
        new dropSite -path $label -formats DATACELLS\
            -command "createCellsViewer currentValueTable \$dragSite::data(DATACELLS) 1 $global::static \$global::pollTime"
        new widgetTip -path $label -text [mc {values table drag'n'drop site}]
        set drag [new dragSite -path $label]
        dragSite::provide $drag VIEWER {dragEcho ::currentValueTable}

        set label [label $frame.summaryTable -image [image create photo -data [summaryTable::iconData]] -relief sunken]
        pack $label -pady 1 -side left
        new dropSite -path $label -formats DATACELLS\
            -command "createCellsViewer summaryTable \$dragSite::data(DATACELLS) 1 $global::static \$global::pollTime"
        new widgetTip -path $label -text [mc {statistics table drag'n'drop site}]
        set drag [new dragSite -path $label]
        dragSite::provide $drag VIEWER {dragEcho ::summaryTable}

        set label [label $frame.freeText -image [image create photo -data [freeText::iconData]] -relief sunken]
        pack $label -pady 1 -side left
        new dropSite -path $label -formats DATACELLS\
            -command "createCellsViewer freeText \$dragSite::data(DATACELLS) 1 $global::static"
        new widgetTip -path $label -text [mc {free text drag'n'drop site}]
        set drag [new dragSite -path $label]
        dragSite::provide $drag VIEWER {dragEcho ::freeText}

        set label [label $frame.threshold -image [image create photo -data [thresholds::iconData]] -relief sunken]
        pack $label -pady 1 -side left
        set drop(thresholds,label) $label
        set drop(thresholds,site) [new dropSite\
            -path $label -formats DATACELLS\
            -command {thresholds::edit; viewer::view $thresholds::singleton $dragSite::data(DATACELLS)}\
        ]
        new widgetTip -path $label -text [mc {threshold drop site}]

        set eraserData {
            R0lGODdhJAAkAKUAAPj4+Hh4eLi4uMiAKLhQAPDouPDksOjYoOjQmOjMkODAgOC8eNiwaPj8+Ojw6ODg4NjY2MDIwNioYNikWLCwsNCYSFCQqJigmJCQkNCQ
            QODo6NDg6Mjg4IiIiAAAAHB4cMiIMGhoaMDY2MjY4GBgYMh8IEhQSLDQ2EBAQKjI0LjQ2Dg4OCAoIKDAyJC4yIi4wICAgHiwuKDI0IiQiJjAyJCYkJiYmKCg
            oKioqICwwMDAwMjQyNDY0HiouOjo6AAAACwAAAAAJAAkAAAG/kCAcCgMEI/G4zBZVBKZSijy6VxWAVKqFRvoer9dwZULBlfF1cGAwJ5ux0p1wWA4HNrNN3xI
            KMzrBwgICQR5eV5oVX10doIJCQoKhWRfewAEX46QCgsLDAQNWnsBDg8PEBBdm50Mn25wpKaoERECAawMEhITk4cBiU+yELQCAhQUAbm7ExWFZVcW0dG1xhQX
            FxgBE8wVGb1YYxYaG+Qc0cfXGBgdHhXd3qJK4uQb5hoaFtjrHR8eGRkgQHxL8kWMhXIc7I3bYKEDvw8hQngAocaDh2dEDibkIOIgvREjLEAMQYKEhwElCFzU
            I8RCQhEwPZIDCdICSRImPKS8+MtQ/kuYIk6ckEmzpgWcJlB4UInIl4WgQlMcLApSRTRzKFCssMjVohkLUVNIpaqi7NWQK1aw4BlmC1ixKVpYoFlWhdCzVtWy
            7blFbIu/c0fUFTo0sF0Lawu6+evChYXBhAsLLju0q+XLjV88jhw1GuW7e8uAeaGZM9xohMVa2AujtevXrWOADZtChgzUcOXuncG7t+/es+HaphEtt+4vNZIr
            X75c+HDiFuL+Jb7Xxo3r2LNnty2DhnfHFqZ7X/3FBo7z6NOr906jMXj2jcl7sVGsvv37xdzHt9C+cQ75Xdigw4AEFmjggO79Z0GC/1W3w4MQRighhC7koGCF
            FspWHQ8cJ3bo4Yce5iCbBRnGoGF5pqSo4oosjmhiDD30AGAANvhg44045qhjEAA7
        }
        set label [label $frame.eraser -image [image create photo -data $eraserData] -relief sunken]
        pack $label -pady 1 -side left
        new dropSite -path $label -formats OBJECTS -command "eval delete \$dragSite::data(OBJECTS)"
        new widgetTip -path $label -text [mc {objects deletion drag'n'drop site}]
        set drag [new dragSite -path $label]
        dragSite::provide $drag KILL list                                       ;# drag data is unimportant, only the kill action is
    }
    if {$databaseMode} {
        $drop(thresholds,label) configure -state disabled
        switched::configure $drop(thresholds,site) -state disabled
    } else {
        $drop(thresholds,label) configure -state normal
        switched::configure $drop(thresholds,site) -state normal
    }
    return $frame
}

proc inquirePollTime {} {
    set dialog [new dialogBox .grabber\
        -buttons hoc -default o -title [mc {moodss: Poll Time}] -die 0 -x [winfo pointerx .] -y [winfo pointery .]\
        -helpcommand {generalHelpWindow #menus.view.polltime} -deletecommand {grab release .grabber}\
    ]
    grab .grabber                                                          ;# grab siblings such as help window so that it is usable
    lappend objects [linkedHelpWidgetTip $composite::($dialog,help,path)]
    set frame [frame $widget::($dialog,path).frame]
    set minimum [lindex $global::pollTimes 0]
    set message [message $frame.message\
        -width [winfo screenwidth .] -font $font::(mediumNormal) -justify center\
        -text [format [mc {Enter new poll time (greater than %u):}] $minimum]
    ]
    pack $message

    if {[package vcompare $::tcl_version 8.4] < 0} {
        set entry [new spinEntry $frame -width 5 -list $global::pollTimes -side right]
        spinEntry::set $entry $global::pollTime
        setupEntryValidation $composite::($entry,entry,path) {{checkUnsignedInteger %P}}              ;# filter on positive integers
        pack $widget::($entry,path) -anchor e -side left -expand 1 -padx 2   ;# evenly pack entry and label together near the center
        lappend objects $entry
    } else {                                                                                        ;# use native widget if possible
        set entry [spinbox $frame.spinbox -font $font::(mediumBold) -width 5 -values $global::pollTimes]
        $entry set $global::pollTime
        setupEntryValidation $entry {{checkUnsignedInteger %P}}                                       ;# filter on positive integers
        pack $entry -anchor e -side left -expand 1 -padx 2                   ;# evenly pack entry and label together near the center
    }

    pack [label $frame.label -text [mc seconds]] -anchor w -side right -expand 1 -padx 2
    dialogBox::display $dialog $frame
    widget::configure $dialog -command "
        if {[package vcompare $::tcl_version 8.4] < 0} {
            set time \[spinEntry::get $entry\]
        } else {
            set time \[$entry get\]
        }
        if {\$time < $minimum} {                                                                  ;# check against minimum poll time
            bell
            $message configure -text \[format \[mc {Enter new poll time\n(must be greater than %u):}\] $minimum\]
        } else {                                                                                                ;# new time is valid
            if {\$time != \$global::pollTime} {                                   ;# but check that it actually differs from current
                set global::pollTime \$time
                viewer::updateInterval \$time
                updateTitle
                refresh                                      ;# update immediately in case poll time was set to a much greater value
            }
            delete $dialog                                                                                     ;# destroy dialog box
        }
    "
    bind $frame <Destroy> "delete $objects"                                          ;# delete objects not managed by the dialog box
}

proc updateToolBar {{parentPath .}} {
    static button
    static tip
    static cache

    set frame .[string trimleft $parentPath.tools .]                                 ;# remove double dot when parent is main widget
    if {![info exists button]} {                                                                          ;# create all widgets once
        set dark [lindex [3DBorders . $widget::option(menu,background)] 0]
        frame $frame
        set data {
            R0lGODlhGAAXAOcAAPgA+FBQUEhMSEBAQDg8ODAwMCgoKCAgIBgYGBAQEAgICAAAANDQ0FBMUODc4ODg4NjU2MjIyLC0sJiYmMDEwEBEQPj8+Ojs6BAUEDAs
            MMjMyDg4ODg0ONjc2FhUWEA8QJCUkCAgAPD08PDw8NDIwLiwqIB8eIiIiCgkAMDIwNDU0CAkAPj4EPj4+Ojg4KCcmNjYSKisqPj8AMDAwPj48Pj08Pjw6Ojg
            2EhMANjYULi4uPjs6Pjk2Mi4sEhIAKikOPj82KioqMC8wPjs4PDk2Pjg0NDAsLCoOPj84PDs6Pjw8PDcyNC8sPj8yPj8wODU0Pjo4PDYwNC4oLCsqMjIwPDg
            2Pjo2PDg0NC4qHBwcJicmLCoqLi4sNjQyOjg0PDUwMi0mFBMAMC4sOjc0Pjc0OjMsNCwoFBUUNDIuOjIsMiskFhcWGhsaPj4GLi0sLCkoMi8sODUwPDYyPDM
            sOjEqMiokKiwqICEgNjMyNDEsOjUwPDUuOjEoMikgLCooMC4uNDEuNDAuMi4qNC8qNC0oMikiMigiAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
            AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
            AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
            AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
            AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAACH5BAEAAAAALAAAAAAYABcAAAj+AAEIHBhAwAAC
            BQwcQJBAwYIFDAZKlNjAwQOLFiFEkLBgwgIKEydWeGChpMkLDzBE8AgypMABDkzKvJDBggaWLgFsICnTJAGSKz+65NChp0kPHzJgeAgi4sQQDEQYLTmCRAkT
            C05oEIgihYoVLFY8EIFiRAuZLVyQeJF1KwoYK2KgkLFixluxZy3QqGHjRom2AFLMxQFDRg4cMlDogDDCQgu+O3j0AKwihg8ZP4D8kOEjiJAOImrUGDKESBEj
            gFe8PQIECZAjOVCscJBEyY4hPHgsYdI2BAsZMoA0AeIECHAWIZ4QgWJ6SRQpbVOozsHa9REYsqdQqWLFypUiUbCntK2c5XLmzT60bOHSxQuPK86/gAHMYG6Y
            HDJghEkspsuY92RE8UUZZgB21xlzraDFXWiMscQSXwyYhhqACbYGGyu0scIWbqDwBhxxyCFgGXPQUQdldtwRgwQhTOAGHmjAkYceeuxRRhp88NEHYACo4BUD
            C/jxByCBCMLEIFgQYoYadRRiCI8SPSTllFQ+pFVIDExwwpZcdtnlBE5NxIAGZJZpppkRBQQAOw==
        }
        set button(new) [frame $frame.new -borderwidth 1]
        pack [button $button(new).button -image [image create photo -data $data] -command clear]
        new widgetTip -path $button(new) -text $::menuHelp(file,new)
        set data {
            R0lGODlhFwAWAOcAAPgA+NDU0AAAAMDEwMjMyNjY2MDAwLi4uLCwsKisqKioqKCkoMjIyLi8uLjAuNDY0MDIwNDQ0CgYECgcENjc2DgkEPisaPjAgPjEgPi0
            cEAoGDgoEOioaPjMkPjUoPjIkBgQCBAQCBgQEPiwaPisYMDAuPjIiPjQmPjEiPCgWIBUMEgwGFAwGFAwIEgsGEAsGDgkGDAgEKigmPi0aKhwQFBQSPj8+PD0
            8Pj0+PDw8Ojs6ODg4ODc4MjIwLi0uDA0MKCUkFg8KPioYOjo6Ojk6KCcoJiYmIiMiIiEgGBcYBgcGJCIeGBEKMCIUFhUUODk4PDs6ODc2LCsqKikqJCQkHh4
            eDg4OJCEeHhYOPi8eOicWHBQMJCMiNjU2MjEyLC0sKCgoHBwcJCYkIiEeGhMMMCEUFhUSOjk4KikoJiUkICAgCgoIHB0cJCIgFhAKPikYGhIKIiIiNDMyGBc
            WBgYGJCUkJCMgLB0QFhQUNjU0MC8uJicmGhwaMjEwLCssGhoaLi4sAgICGhoYLi0sCgsKHh8eNjY0KCcmKCgmJiUmJCMkFhYWBgUGDAwMDAsMCgoKCAkIBAQ
            EJigmMjQyAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
            AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
            AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
            AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAACH5BAEAAAAALAAAAAAXABYAAAj+AAEIHDgwgICD
            BwcQCFCgAMGHAAognHjQgICFEAVKFHAAQQIFCxZMPEDAIUQBDBocaODAQYMGCBAqGBDg4YMABAYMgLAwQIAIExcYqKlRwoSjDQZEeECh6QMCB0AiYGCyQAUL
            FzBk0HCAQQSfPwkYeAmBKIANHDp46PBhKwgQIUS8jdsAQoSqGi6sxTCChN+/gLl69Vlig4kTKFKoWMFiRQsXLzbAiBFDgoSJMlaY6DCDRg0bN3Dk0LGDx88e
            Aw74SIDgB5AgJi4IWcFgCJEdFCIwMNBRQREjR5AkUbKEyYUZTZzkeDIEyo4oEcQekDLFCJUqVo5cwZJFyxYuXXbP6BgyhEcBAl4OfFEA5kgYAWLGkJlRxsyZ
            ADuePKFwnkGJAwmgkYYaa7CxQBtuvAFHHAx0QYF5BUQgx1gISAEGFXPQUccXdqxwBx4UEEBAHhFEt5seFYKxhxpW8CGUFBu4QMVKXjDAQB9ejPWFH2i094cA
            GxIASCCCMJDAF18csNIXg3wERhFUqEFIIUIFYIiGRhyyBxiIgHHIIUYkQoUicaixCCNGfFESAIw0QgghjsApJyGPPAKJnZFQMRNRDCCwwB6ABipooJIo0MAk
            AgUEADs=
        }
        set button(open) [frame $frame.open -borderwidth 1]
        pack [button $button(open).button -image [image create photo -data $data] -command reload]
        new widgetTip -path $button(open) -text $::menuHelp(file,open)
        set data {
            R0lGODlhFgAWAMYAAPgA+GhsaAAAAMjMyNDU0MDAwEBAQEhISHBwcIiIiMjEyFhcWGhkaFBQUHh0eFhYWPj8+ODg4HBscEBEQAgECGBkYFBUUJiYmDg8OGBg
            YHB0cFBMUMDEwNjo6FhUWHh4eKiwqBgUGLCwsHh8eLi8uKCkoICAgLi0uPDw8Pj4+MC8wIiMiKioqEhMSEA8QMjIyJicmLi4uBAMEJCQkJiUmIB8gKCgoJCM
            kIiEiICEgDg4ODAwMAgICOjs6JCUkGhoaNjY2KCcoPj0+EhESAAEACgoKPD08Ojk6GBcYNjU2NDM0JigmAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
            AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
            AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAACH5BAEAAAAALAAAAAAWABYAAAf+gACCggECAwSDiYqLAgUGBwiHi5MACAkKCwwLBg0OkpSC
            BgUFDxARCxKbExKfixSkERCypgsVmxaeiIMSFwUYs8C0GRoCDwODGxykwcAdHh8cAiCDISIPscyyFiMaJAclghImJwYo2RApBh8qFCssgi0XFy4B5xEWHy8u
            MDGCMjM0ZtSwYCBBsGcaVDSwwQGABA0rVsxYcWOGgwYFTRnoJmAGi2MGcOQ4oOPBigQ4EuRA8GBjvh0XSCDiMQLBhx4+DrTI8CHHhw8aEPwo0IIhEAk/LgaB
            IAQFkBxDhnxAgCCACiLujhWRwOCAkR5AjvSIYKKBjiELfryYwA8REgFmPwwISQJEyQsSHnLgwJAhg4oHRgEMkDBBwAQWHFTEWDGhxgoPDUgIyCqIQAEWSyQM
            ETCExZAECSZY8PBCx4UYugAQGMCBBGbChi/osBD5QGBFq1u/3jzkARGPx0Dldp1ZwIKYqQMBADs=
        }
        set button(save) [frame $frame.save -borderwidth 1]
        pack [button $button(save).button -image [image create photo -data $data] -command {save 0}]
        set global::fileSaveHelpTip [new widgetTip -path $button(save)]
        set data {
            R0lGODlhFwAWAOcAAPjUoNiMSJhgOPgA+GhoaAAAAPiECPDcqPDUoNiQSJhgMMDAwDhAOEhISHBwcNhsAPiEANiMULC0sFhYWGBgYFhcYEBAQFBQUNhsCNhw
            APCEAOjw6NjY2GhsaGBcWEBESHBscMDIwMjQyODk4Pj8+Njc2FhcWFBUUHh0eNhoOPC8oNiggPj4+GBcYGh0aDg4OMhkGMB4WLh8WOCggNicePD48NDo4FhU
            WHh4eKisqBAUEJicmEA8QEBAOHhMMHBQMJBYOPD4+LC4sJikmICAgKiwqDg8ONDU0KioqLjAuPDw8PD08MC4wAgECICMgKCkoEhMSJiYmOjs6ODs4JCckLi0
            uBAMEIiQiJCQkHh8eICIiEhQSKCcoLjEuGhwaIiMiICIgJCIkIiEiHB0cHiAeICEgIiIiDAwMJCUkLC8sAgICHB4cODo4EBIQGhkaLi8uJigmNjU2KCYoPD8
            8EhESAAEAMDMwCgoKHBocMDEwEBEQJCYkNDQ0MjIyLi4uKCooDhEOIB4gEhASMjUyMjMyAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
            AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
            AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
            AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
            AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAACH5BAEAAAMALAAAAAAXABYAAAj+AAEEEDCgIIEC
            BRMqXDjAwAEECRQMKLCAQQMHDBk+gOAQQAQFEiZQqGDhAsaMEzFk0OAQooANHCZ08GDhA4gQIhYWKIDhAcsDAkeQIFFCpAkLJ1DgVLhTJYQUKlaMYDGUqIkW
            LgpMCKHwxU4YMWTMoDF1aA0bN3AsKJBDoY4dPAr08PEDSIERQUjUOIHDhZAGQxJ2IFLEyBEJSBLnSKKExBILOJg0cfIkIZQoUYwQkMJ5CuehJfiG4EGlSkIr
            V7BgyXKCh5aqQ2+gcMFkC5cuBUF4+QLmipMwVxxcsCCmRgkLY4QUwPKE6wDiZBq8uOFETBkzZBxMsOAFR4gzaNLQ5BygZo0DHGzQtIHSYk2ZMTjGOHDzBgqc
            JHEGgCCA4oKcOUpIcUQZdNCBgwMgdPBGHV88YUdBd+DhRgMbsHHECGyUkMUFL9BhAgF56LGHBHwU1EIBbligBB9H9JHHGzcQUYYRLVDgxwT3JRQCCHoUoMcf
            C7whwReABOLEDVv4UUCDfSQkwht/7NABHQXQ8YcgYJShxwk35PHCiIModEQISfjxBxU8+rjHCydc8Nd9R2Q0CJlCIIEmlXTcUAdzhKCU0JxJ1ElFBwWYgIYQ
            4w0QEAA7
        }
        set button(saveAs) [frame $frame.saveAs -borderwidth 1]
        pack [button $button(saveAs).button -image [image create photo -data $data] -command {save 1}]
        new widgetTip -path $button(saveAs) -text $::menuHelp(file,saveAs)
        set data {
            R0lGODlhGAAYAMYAAPgA+FhYWDAwMGBQSMDEwMjMyNDU0NDY0PDgyOjcyFhQSDg4MFBIKPDg0Pj08Pj4+OjgyPD08Pj0+PDs6Dg0MFhMQFBMKPjw8PDs4OjY
            uAAAALCwsMjIyNjY2FBUUPDw6Pjw6PDk2ODYyOjMqKCkoMDAwPjs6Ojg0OjYyIiIiDA0MEBAQGBcWPDo4ODIqAgECEhIKLi4uMjQyEhISMC8wKioqFBISODI
            oPDo2Pjs4OjYwJCACNjUeEhMSLi8uHBwcLiYcOjAiOjUuOjQuLi0SJCQkKisqDg4OKCgoGhkaOjAkOjMsFhQQJCAENjQeJiYmLC0sPj8+LCssJicmLCYaOjI
            qLiwSLjAuEhESOjk6PDs8JCUkJiUmODc4ODg4ODk4FhcWHh4eNDQ0NjU2ICEgIiEiHh8eMjEyICAgIB8gEBEQJigmNDM0Njc2EA8QGhsaLi0uKCooGBgYKik
            qACIAAD8ADg0OGhoaIiMiHBscAAAAAAAAAAAAAAAAAAAAAAAACH5BAEAAAAALAAAAAAYABgAAAf+gACCg4QBAgMEBQYHhI2OAAYCCAkKCwqJi4+ODA0OD58Q
            CoeYmoIGARGfDxISERMJFBUEBqUWCKqfFw4YDRkaGxwdjwYeuA8fIBghIiMaJCW0hAcGDCapEg8OyRgnKAopz9EAHSoqKywQLRMgyBgtJw0uLzAMMTKCHTM0
            NTY3DSEtcOTAgeNEAh0KdvBgAGxcDx8kfqwAEmQEPB0odAgZ0owIgyJGgh2hQQIJkiQaAgBR4kLjkCVDmDRxYuEJlAIaokSQMuWJz5MpqSi5UcWFBisaUiC5
            YgBLFi1RokiQsoVLkSlAVWLRwEODM2geunj54iVL1J1FiqR4giSAhhXAYH6E0RCjwAwOYjqMGdOlw5eoU1OkIFPGTIoiKzTUIKAB5JkSHApENmCgy1kpaNKk
            WKFGw5oSjDWE4VIjBpQSqAmwEdMmyoMzbtRsneJDjI8anTW8KYNkQw04UOD4IFBii5ojAlZ87iCDAJQ4U7ZqkIOGy5w5dOpISTLDTuPFpgoQ8LEBemINYMwU
            eVLkTmI8SOxJMyCevHmvYPL80AC/rjBH09RX3hS5FRFfAf+VEuB45ZFQGoKlAEgfcQTIkGAgADs=
        }
        set button(print) [frame $frame.print -borderwidth 1]
        pack [button $button(print).button -image [image create photo -data $data] -command $global::printDialogCommand]
        new widgetTip -path $button(print) -text $::menuHelp(file,print)
        set data {
            R0lGODlhGAAXAOcAAPgA+EBAQEhESGBgYEBESNDQ0Hh8eKioqLi4uKCcoGhoaAgICMDEwPj8+MjEyIiEiEhMSEA8QGhkYJCUkHBwcDAwMKCkoMDAwEhISIiI
            iIiMiBA8OFiEiChQUHh4eLCwsMjIyFBQUDg0OICAgBg8OOj0+GCsqBg8QJiYmJCQkPj4+KCkqEhMUHiAeOj08FCgmBA4OPj0+JicoIh4KGhgONjc2Kjo4BhI
            SLD0+CBAQKCgoLCYQGBYIFg4EKjo6IDIyDBUUEigoBAsMMjMyGhkaFhUWMisUNjAgIjQ0HC8wJDc2KDo6CBQUHB0cODg4FBUUEBASNjAeIBYIMiseIhsKIDU
            2ECQkCBcWCBMUBhQUAgcIJiUmFBMUDA0OLiQYPDgyMikcIhwKFg8ECBoaAgQEGBkYNjY2DAgCLiMYNCkeIBsKNCsgIjc4DhwcMCgeIjY2FikqAgAAMCgcHjE
            wEiUmChoaAAECOjMqIDQ0EiYmCBcYAgUGDAkCCgsMJDY2OjQqLiYYCgwOLj8+FCgoCBgYMCYYFhAGCA0OBAsKAgYGJBsOFA0CDgkEAAAAAAAAAAAAAAAAAAA
            AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
            AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
            AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
            AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAACH5BAEAAAAALAAAAAAYABcAAAj+AAEIHEgQQAAB
            AwgUKMiQYQADBxAkULCAQcOLBhs4eAAhgoQJCzEKpCChgoULGB5m0LCBQ4eQDT18AJEhhAgMIyYwIFHCxAkUMAmmKNBAxQoWLFro3ODiBQwDEzaQCApgAIgY
            MgLMoJGBQY0TNm5APYEjB1CCI3SsCNBgB48eXjf4+AGEaRAhGoZUHUBExIAiPIw0OPI2RwkkSZQsMcEEaoEmToYceCICigEeUaRModKjipUrWLJogWpxwAMV
            W7h0gYqAh5cvYMKIGUPGwEqdAgsMGDGkDIauZhicQfMljZoeORkMCcpAQZcijr+WOPN6jRoxCMw0ZDBBg84aTNl+tBn+xQ1niw0LLJf+Bk6WONS/yJGN/iLT
            OXTq2LE9/I6c8xedUAIeeeixB2kM8OHFHVPQx1AfLviRRx214QaAcGj8AQiABAUiyCCEHGihQMJ5UYghZ4BEUAGHIJIIaQwJp8giZ+RVEHfe1XfjGYw4xpB6
            VN04wYgiXVRASAEBADs=
        }
        set button(configuration) [frame $frame.configuration -borderwidth 1]
        pack [button $button(configuration).button -image [image create photo -data $data] -command {configuration::edit 0}]
        new widgetTip -path $button(configuration) -text $::menuHelp(edit,configuration)
        set data {
            R0lGODlhFgAWAOcAAPgA+NDU0ChQGCBQGCBAEBhAECg8GBg0EBgUAMjIyNDQ0NjY2BhEECh0MGDAgNjw2LjkuIDMkDiMSEh8QEBwMCBIGBg8EMDEwNjg2Dio
            YPj8+LjkwEi0cDCEQCBEEBAkCBAUACgkCCBMGBAsCMjQyLDguEC0cFCESJicmKCkoKioqKisqBgsCBAYAGjAgNDs0DiAQBgwCJiYmLC0sLjAuMjEyMDAwBgs
            ENDY0ChsKKDYqMC8wMjMyNjc2OD02GjEiEiQUJigmLi4uODg4ODk4ChIGDBQICigUMDkwFC8eBgYAODc4CiAOLDkuNj02HDEiECYWDiIQAAAABgYGDBsKDCo
            YDC0YDCgUDiQSGjEgChgIKikqKCgoEhESDhsMDCQSHDIiChgKKCcoLCssHBscIiEiEBEQEB0ODCMQGhsaDg4OEhISJCMkEhMSHBwcCAgIDAsMFBMUJCQkCgo
            KGhoaCgsKDAwMAgMCAgECHh4eDg8OFBUUCAkIOjo6IiIiEBAQCgkKOjk6FhUWLCwsAgICODo4ICEgBgUGCAcIICAgFBQUAAEAGBcYBAQEBgcGAAAAAAAAAAA
            AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
            AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
            AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
            AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAACH5BAEAAAAALAAAAAAWABYAAAj+AAEIHAgggIAB
            BAoYOIAggYIABCMuYNDAwQMIESRMoFDBwgUFCyJiAMAgg4YNHDp4+AAiBAgRIy6QCClwZIAGGkqYOAECAYoUKlaAYNFCJk2BFVy8cAAjhowUM2jUuGBjxg0Q
            FyAKxJFDg44OIGSo2MFjQY8eAXjYQHCBB4AFAUjk8PEDCIsgQsoOIcIXLQ8eCkAUKGLkCJIkHVgo6UmiB18iQ3oswLEEAZMmPpw8gRIlwosOMWwEGCJlCuQh
            e3kgoFLFyhUsWZhoAbFFiAIpXLqYRj0kwFovX6yAYRIGhJgxCaQIIVPGzBTeABSsPYNGdtgVF6RcSKNmDZs2z1Gt91BAA4GX4jKwK3ej5g2cOHJ0Y4CMdi0I
            LtjnoKDTxUwdOHaY4UYdd5yGQQAXCCFEAgtIIQUeUuShxx58QOjgfKctoABIRPThoRR+/LEHIDUJRIQUj6EWiIcfbqGHIFNEBICDj7H44SB0mPEHITKS5mAh
            LDo4iCFdrHGIjAINQQgfhCDiYB6DJKKIInAsguRAUrzRBhmJqMBIF4o0IsWVBA1xRyNTOPKGI2MiGRAAOw==
        }
        set button(refresh) [frame $frame.refresh -borderwidth 1]
        pack [button $button(refresh).button -image [image create photo -data $data] -command {after idle ::refresh}]
        set tip(refresh) [new widgetTip -path $button(refresh)]
        set data {
            R0lGODlhGAAYAOcAAAgECAA0OMjIyNDQ0NjY2PgA+AAAAHisqLi8uMjMyIC0sCB4eLi4uNDU0CAgIBh0cPj8+AAEAPDo2GhgSLC0sNDY0HCsqPj4+HBkUKCg
            oHBoUMDEwGioqPjgyPjw6HhsWLCokOjc0Pj08GCoqNjEmPDcwPj06ODYyPj48AhoaMjQyFikoLCgiNi8mIB0YBBscKioqFCkoPD08KCYgOjcwNjUwJigmMDA
            wEicmNjAmIh8aBBsaJicmECYkPDs4KCQcPj04Pj88DCQiPDk0KCQeJCEcCiIgPDw4Pjw4Pjs2OjkyJiMcPj46Bh8eOjUuPjo2OjYyKCUgBCIgAhsaODUsODU
            uJiIaPDs2MC0mKCUeKigiMjAqAgAAODMsPDYuKCMcPDgyPDk2Ojg0JiQeLiokMC0oNjIsODUyNDAoODMoJCEYNjAkKiYeMi4mNC4kLCggMDAsMjAuMjAsMCw
            mIh8YJCIaHh4YDCEgDCAgBCMiKCkoAAwMAg0MAAsKIiMiJCUkJiYmLCwsKCooAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
            AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
            AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
            AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
            AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAACH5BAEAAAUALAAAAAAYABgAAAj+AAEEEDCAQIGD
            CBMqTGjggAEECQwuREhAYkIAChYYYBBxYgMDDhI0UGhAwQMIGzsibBDBgIQJFARUuGhhAQSUHA0SMABAAoQLGDJwnHnQgIWTN1MOcHnzpoaWETYULACAg82m
            BigY6OCh6YUPIEKIAABx5wikNy9EIFHCRFMIYE+gSBEBgQoAK67eFMGiBdemIlyAOCHihQEYGwzEQAtBxgwSNEyggDBZcI25EWzcAIBDLwQPM3L8FXFBhI7B
            InYY4IHAQI+TFyCI8PEjRwkgIoKgQHH6Ml3WAIQsEIFChIghRPx6QLHcRBHUhlkbMPIAyBEkSZQssQ3EAxMPzkHM+I4AvMkCJ0meJIHyQ7kJ8B6ioFbNekME
            KVOoDKFRxQr3KyZgkYUWW4zHmgoMcCHFC1140cUXLYARhhhjkFGGGWdIUNhqCBCQQIL4oZGGGmuUMAMbbdQggWSS/YZAAR4yAMCCbvhHxBtmwBGHHHHoOAd9
            L8L44X1T0FGHHXfgkeSSSuJBXpBCyrjgFHnwZCUAVxqgxw0JxRjAHnz04ccfgPBg5pk86BGIAArFCEMGNggSCAMI1GknAjcIMFKbKmxwww0biNTAoIQOalBA
            ADs=
        }
        set button(help) [frame $frame.help -borderwidth 1]
        pack [button $button(help).button -image [image create photo -data $data] -command generalHelpWindow]
        new widgetTip -path $button(help) -text $::menuHelp(help,global)
        set data {
            R0lGODlhFgAWAIQAAPgA+MjMyMDEwLi8uLi4uMjIyNjY2MDAwAAAAHh4eHh0eHB0cMDIwDAwMNDU0NDQ0NDM0MjQyNDY0Njc2LjAuODg4ODc4ODk4Ojk6Ojo
            6Njg2AAAAAAAAAAAAAAAAAAAACH5BAEAAAAALAAAAAAWABYAAAX4ICCOQCAMBDEIAemORSrPRPGKxkEjyOD7B4PLgJoNeL9fkKRL9Xy8gxQhPQhgMl7vwBMI
            tF6WSJbgKay8QqGMWBQEjFIW0bAWeIF8Ax8oOAQpCzwJAmo8D4iCCAl5D0VgBQEQPA6VDlqIESlHbZERDzwSBqNslQ8qnAoFEA+XCKMGE4oGlT4UWgGtrhMT
            FRZao38DFAo8C62jvBUVxW0TowEDaHQOohMWFRcXewi+vADSAooNC7HZCtwKFxUTIgF2zYOKgxjrFiMMhRDzWmYZGetIGFCTywEbY/8y1GsnsFE1c9owKLzA
            8IUDZLGwaQt4Q2AyZRpuhAAAOw==
        }
        set button(exit) [frame $frame.exit -borderwidth 1]
        pack [button $button(exit).button -image [image create photo -data $data] -command exit]
        new widgetTip -path $button(exit) -text $::menuHelp(file,exit)
        set data {
            R0lGODlhFgAWAKUAAJgPDwAAAG4hIb82NudAQOM8PLcuLmYZGSAKCsI6OtosLMMQELsBAcABAcoNDdYhIa4mJhwGBs0fH7YAALoAAL8AAMMAAMcAAMsAANMU
            FKggINAAANQAANIYGF0PD1UHB9gAANkICKAYGN0AAOAAAMEaGuEAAOQAAL0XF+UAAN8FBekAAM8NDdgJCZQNDf//////////////////////////////////
            /////////////////////////////////yH5BAEKAD8ALAAAAAAWABYAAAauwJ9wSCwaj8jkEcBsKonNqFPZDFiv0yUzIBgQCoZDIAtlIhKKBaPheEAiZOE2
            IZlQKpYLJqMZA4pbAgp2eHobHB0eH0xlAQMLd3kYhyAhIot/Q1sEDIWTHCAjJCWYjQUNkpQjJicopZoAAQYOhqCrKSoAr3KxBw+foSYpKywfu7wBEBm2wist
            Lse8ABEaHSEkJyos0HHSAR4iJSi6mJla5MbG3YBSUU+w7u/y8+9BADs=
        }
        set button(record) [frame $frame.record -borderwidth 1]
        frame $frame.recordSeparator -bg $dark -width 1                                                    ;# thin vertical darkline
        pack [button $button(record).button -image [image create photo -data $data] -command {databaseRecording 1}]
        set ::controls::fileDatabaseStartButton $button(record).button
        new widgetTip -path $button(record) -text $::menuHelp(file,database,start)
        set data {
            R0lGODlhFgAWAKUAAP+goP9AAGAAAF4QEHcVFYwcHIMYGHoVFXgXF4ATE3gJCXYEBHcEBHoICH4PD3UREaAAAIMcHHsODnQEBHUEBHgEBHoEBHsEBH4LC3QQ
            EF8SEn0EBH4EBH0MDFcJCYMbG38EBH8HB20MDKYmJoEEBIIEBHkNDYMEBHYMDHUTE4IFBWoJCVwODoUEBHwICHMREW0ICG8PD2wICH0NDW4JCVUHB///////
            /////////////////////////////////yH5BAEKAD8ALAAAAAAWABYAAAa+wJ9wSCwaj0KAcok0KgPQKBP5jFqhymM1IOh6pQBnVDAgFAyHgQBMBIwRCcWC
            0XA8INI2VBCRTCgMFRYXGBl4AVk/blwaCX+BgxscHR6HYYp7HwqAgheSICEilkl7I3OdnyQlJqOYXAaQnhwgqicorYsCKQ2Rs7UqK7h7LA6ytCUnLS4Qra4C
            Lxi+yC0hMJaXrgEQMR0hyCouMoeIRYvaHiIzKDQ144lDW8zy7tht5ldY9U73+fpUS1OaCEQSBAA7
        }
        set button(recording) [frame $frame.recording -borderwidth 1]
        pack [button $button(recording).button -image [image create photo -data $data]]
        bindtags $button(recording).button {. all}          ;# completely remove button specific bindings since this is an indicator
        new widgetTip -path $button(recording) -text [mc {currently recording cells data history in database}]
        set data {
            R0lGODlhFgAWAOcAAAAAAAEBAQICAgMDAwQEBAUFBQYGBgcHBwgICAkJCQoKCgsLCwwMDA0NDQ4ODg8PDxAQEBERERISEhMTExQUFBUVFRYWFhcXFxgYGBkZ
            GRoaGhsbGxwcHB0dHR4eHh8fHyAgICEhISIiIiMjIyQkJCUlJSYmJicnJygoKCkpKSoqKisrKywsLC0tLS4uLi8vLzAwMDExMTIyMjMzMzQ0NDU1NTY2Njc3
            Nzg4ODk5OTo6Ojs7Ozw8PD09PT4+Pj8/P0BAQEFBQUJCQkNDQ0REREVFRUZGRkdHR0hISElJSUpKSktLS0xMTE1NTU5OTk9PT1BQUFFRUVJSUlNTU1RUVFVV
            VVZWVldXV1hYWFlZWVpaWltbW1xcXF1dXV5eXl9fX2BgYGFhYWJiYmNjY2RkZGVlZWZmZmdnZ2hoaGlpaWpqamtra2xsbG1tbW5ubm9vb3BwcHFxcXJycnNz
            c3R0dHV1dXZ2dnd3d3h4eHl5eXp6ent7e3x8fH19fX5+fn9/f4CAgIGBgYKCgoODg4SEhIWFhYaGhoeHh4iIiImJiYqKiouLi4yMjI2NjY6Ojo+Pj5CQkJGR
            kZKSkpOTk5SUlJWVlZaWlpeXl5iYmJmZmZqampubm5ycnJ2dnZ6enp+fn6CgoKGhoaKioqOjo6SkpKWlpaampqenp6ioqKmpqaqqqqurq6ysrK2tra6urq+v
            r7CwsLGxsbKysrOzs7S0tLW1tba2tre3t7i4uLm5ubq6uru7u7y8vL29vb6+vr+/v8DAwMHBwcLCwsPDw8TExMXFxcbGxsfHx8jIyMnJycrKysvLy8zMzM3N
            zc7Ozs/Pz9DQ0NHR0dLS0tPT09TU1NXV1dbW1tfX19jY2NnZ2dra2tvb29zc3N3d3d7e3t/f3+Dg4OHh4eLi4uPj4+Tk5OXl5ebm5ufn5+jo6Onp6erq6uvr
            6+zs7O3t7e7u7u/v7/Dw8PHx8fLy8vPz8/T09PX19fb29vf39/j4+Pn5+fr6+vv7+/z8/P39/f7+/v///yH5BAEKAP4ALAAAAAAWABYAAAi0AP0JHEiwoMGD
            CBMqXMiwob8oECNKjEgCYZRIixA1gsSR0ps1YUhUPBjFkbN3/1KmNMdNTpSRBqMoGrcsnE2b3HjxeWmRz79cuWYJnQWrkUuYBaP4fHWoKaJEmIzyJGmHXjCn
            iRQ1EvRmasyqV58qWrS1K1KCUcCKJVvWa9Kqy7KSdYSJ0lGLa8LRwlSqbylQmO6SjPJKjuHDhqM0OTswSo6JEpMUYSxQpOXLmB1q3sy5M8OAADs=
        }
        set button(stop) [frame $frame.stop -borderwidth 1]
        pack [button $button(stop).button -image [image create photo -data $data] -command {databaseRecording 0}]
        new widgetTip -path $button(stop) -text $::menuHelp(file,database,stop)
        set data {
            R0lGODlhFgAWAKUAAFeLV0V5RVGFUQA0ACdbJ5zQnEp+SrXptSNXI5HFkb7yvq/jrx9TH4u/i7LmshxQHJ7SnnquenGlcRhMGIO3g3mteWygbBVJFYm9iWmd
            aWicaGqeanSodJnNmRFFEXKmclyQXC5iLg5CDgo+CmGVYW2hbQc7BwM3A1mNWTJmMv//////////////////////////////////////////////////////
            /////////////////////////////////yH5BAEKAD8ALAAAAAAWABYAAAZ0wJ9wSCwaj8hkERBQKgOCgRNJKBik02L1cM0SEQnFouv9MRqOxBibfTQgEcna
            O6FAKhY52XmJYDIaGxwde0keHxEaGhIYICFsSSIZFYuNj1kjJCWMjpBKJiR+nWUnIBGjZQMoqKkAl2VCAymeXgO0sLi5P0EAOw==
        }
        set button(play) [frame $frame.play -borderwidth 1]
        frame $frame.playSeparator -bg $dark -width 1                                                      ;# thin vertical darkline
        # use file database load menu command:
        pack [button $button(play).button\
            -image [image create photo -data $data] -command "loadFromDatabase [expr {!$global::readOnly}] $global::static"\
        ]
        new widgetTip -path $button(play) -text $::menuHelp(file,database,load)
        foreach name [array names button] {
            $button($name).button configure\
                -activebackground $widget::option(button,background) -relief flat -takefocus 0 -highlightthickness 0 -pady 0
            bind $button($name) <Leave> {%W configure -relief flat}
            # make sure frame relief disappears whenever button is depressed:
            bind $button($name).button <ButtonPress-1> "$button($name) configure -relief flat"
        }
        grid columnconfigure $frame 100 -weight 1                                                 ;# so that icons stick to the left
    }
    foreach {loadedUserModules anySynchronousModules databaseConnected databaseMode} [userInputStates] {}
    if {[info exists cache]} {
        set change 0
        foreach {name value} [array get cache] {
            if {![string equal [set $name] $value]} {set change 1; break}
        }
        if {!$change} {return $frame}                                                       ;# avoid updates when there is no change
    }
    foreach name {loadedUserModules anySynchronousModules databaseMode databaseConnected global::readOnly} {
        set cache($name) [set $name]                       ;# the tool bar configuration is uniquely determined from these variables
    }
    switched::configure $tip(refresh) -text $::menuHelp(view,refresh)
    if {$global::readOnly} {
        if {$databaseMode} {
            set list {print refresh help exit}
            switched::configure $tip(refresh) -text $::menuHelp(view,refreshDatabase)
        } else {
            set list {print help exit}
        }
    } elseif {$loadedUserModules == 0} {
        if {$databaseMode} {
            set list {new open save saveAs play print configuration refresh help exit}
            switched::configure $tip(refresh) -text $::menuHelp(view,refreshDatabase)
        } else {
            set list {new open save saveAs play print configuration help exit}
            array set disable {save {} saveAs {}}                                                                 ;# behave as menus
        }
    } else {
        set list {new open save saveAs}
        if {$global::database == 0} {                                  ;# disconnected from database: not storing data cells history
            lappend list record
            set disable(stop) {}
        } else {                                                      ;# connected to database: currently storing data cells history
            lappend list recording
            set disable(recording) {}
        }
        lappend list stop play print configuration
        if {$anySynchronousModules} {
            lappend list refresh
        }
        lappend list help exit
    }
    if {[string equal $::tcl_platform(platform) windows] && [catch {package require Tkprint 1.1}]} {
        ldelete list print                        ;# do not display print icon when printing is not available on the windows machine
    }
    eval grid forget [winfo children $frame]                                                       ;# first unmap all existing icons
    set column 0
    foreach name $list {
        if {[string match record* $name]} {                                         ;# always insert separator defore record buttons
            grid $frame.recordSeparator -row 0 -sticky ns -column $column
            incr column
        }
        if {[info exists disable($name)]} {
            if {![string equal $name recording]} {                          ;# recording button should not be grayed since it is lit
                $button($name).button configure -state disabled
            }
            bind $button($name) <Enter> {}
        } else {
            $button($name).button configure -state normal
            bind $button($name) <Enter> {%W configure -relief raised}
        }
        grid $button($name) -row 0 -sticky ns -column $column
        incr column
        if {[string equal $name play] && ([llength [grid info $frame.recordSeparator]] > 0)} {
            # always add separator after play button but only if record and stop buttons are displayed
            grid $frame.playSeparator -row 0 -sticky ns -column $column
            incr column
        }
    }
    return $frame
}

proc updateDatabaseStart {state} {                                                    ;# set start database recording controls state
    $::controls::fileDatabaseMenu entryconfigure $::controls::fileDatabaseMenuStartIndex -state $state
    $::controls::fileDatabaseStartButton configure -state $state
}

# return list of booleans: any loaded user modules, any loaded synchronous modules, connected to database, database history mode
proc userInputStates {} {
    set history [llength $modules::(instances)]                ;# there must be at least one loaded module to determine history mode
    foreach instance $modules::(instances) {
        if {![string equal [lindex [modules::decoded $modules::instance::($instance,namespace)] 0] instance]} {
            set history 0                                            ;# in history mode, all loaded modules must be instance modules
            break
        }
    }
    if {$history || [info exists databaseInstances::singleton]} {     ;# history mode (also implied by instances viewer's existence)
        return {0 0 1 1}                                                                                         ;# implies the rest
    } else {
        return [list\
            [expr {[llength $modules::(instances)] > 0}] [expr {[llength $global::pollTimes] > 0}]\
            [expr {$global::database != 0}] 0\
        ]
    }
}
