# copyright (C) 1997-2004 Jean-Luc Fontaine (mailto:jfontain@free.fr)
# this program is free software: please read the COPYRIGHT file enclosed in this package or use the Help Copyright menu

# $Id: canvaswm.tcl,v 2.36 2004/01/01 11:39:06 jfontain Exp $


class canvasWindowManager {

    proc canvasWindowManager {this canvas} {
        set ($this,bindings) [new bindings $canvas end]
        bindings::set $($this,bindings) <Configure> "canvasWindowManager::rearrangeIcons $this %w %h"
        set ($this,canvas) $canvas
    }

    proc ~canvasWindowManager {this} {
        variable ${this}data
        variable ${this}handleIcon
        variable ${this}handleCoordinates
        variable ${this}handleIconCoordinates

        delete $($this,bindings)
        foreach {handle icon} [array get ${this}handleIcon] {
            delete $icon
        }
        foreach {name handle} [array get ${this}data handle,*] {
            delete $handle
        }
        catch {unset ${this}data}
        catch {unset ${this}handleIcon ${this}handleCoordinates ${this}handleIconCoordinates}
    }

    proc manage {this path viewer} {                                           ;# viewer, table or database cell histories container
        variable ${this}data

        set handle [new handles $($this,canvas) $this -path $path]
        set ${this}data(handle,$path) $handle
        set ${this}data(viewerHandle,$viewer) $handle                                           ;# a viewer can have one handle only
    }

    proc unmanage {this path} {
        variable ${this}data
        variable ${this}handleIcon
        variable ${this}handleCoordinates
        variable ${this}handleIconCoordinates

        set handle [set ${this}data(handle,$path)]
        if {[info exists ${this}handleIcon($handle)]} {                                                                 ;# minimized
            delete [set ${this}handleIcon($handle)]                                                               ;# delete icon and
            unset ${this}handleIcon($handle) ${this}handleCoordinates($handle)                                       ;# related data
            catch {unset ${this}handleIconCoordinates($handle)}
        }
        foreach {name value} [array get ${this}data viewerHandle,*] {
            if {$value == $handle} {array unset ${this}data $name; break}
        }
        delete $handle
        unset ${this}data(handle,$path) ${this}data(relativeStackingLevel,$path)
    }

    proc configure {this path args} {
        variable ${this}data

        set handle [set ${this}data(handle,$path)]
        array set value $args
        if {![catch {string length $value(-level)} length] && ($length > 0)} {
            # find out which managed widget to stack right below, if any. if none is found, widget defaults to top stack level
            set names [array names ${this}data relativeStackingLevel,*]
            if {[llength $names] > 0} {                                                           ;# there are other managed widgets
                foreach name $names {                                                               ;# build path from level mapping
                    set pathFrom([set ${this}data($name)]) [lindex [split $name ,] end]
                }
                foreach level [lsort -integer [array names pathFrom]] {
                    if {$level > $value(-level)} {
                        handles::stackLower $handle [set ${this}data(handle,$pathFrom($level))]
                        break                                                            ;# found the handles for widget right above
                    }
                }
            }
            set ${this}data(relativeStackingLevel,$path) $value(-level)
        }
        catch {set xIcon $value(-iconx); set yIcon $value(-icony)}
        catch {unset value(-level)}                                                        ;# handles do not handle the level option
        catch {unset value(-iconx) value(-icony)}                                                        ;# nor the icon coordinates
        if {![catch {set object $value(-dragobject)}]} {
            composite::configure $handle -dragobject $object
            unset value(-dragobject)
        }
        eval composite::configure $handle [array get value]
        ::update idletasks                    ;# so that handles return correct geometry even when immediately minimized right below
        if {[info exists xIcon] && ([string length $xIcon] > 0)} {                       ;# icon coordinates are defined so minimize
            minimize $this $handle [composite::cget $handle -title] $xIcon $yIcon $value(-static)
        }
    }

    proc getGeometry {this path} {                                                        ;# return x, y, width and height as a list
        variable ${this}data
        variable ${this}handleIcon
        variable ${this}handleCoordinates

        set handle [set ${this}data(handle,$path)]
        set geometry [handles::getGeometry $handle]
        if {[info exists ${this}handleIcon($handle)]} {                                                                 ;# minimized
            # return coordinates before minimization:
            return [eval lreplace [list $geometry] 0 1 [set ${this}handleCoordinates($handle)]]
        } else {
            return $geometry
        }
    }

    proc getStackLevel {this path} {                                                               ;# return relative stacking level
        variable ${this}data

        return [set ${this}data(relativeStackingLevel,$path)]
    }

    proc iconCoordinates {this path} {
        variable ${this}data
        variable ${this}handleIcon
        variable ${this}handleCoordinates

        set handle [set ${this}data(handle,$path)]
        if {[catch {set icon [set ${this}handleIcon($handle)]}]} {                                                  ;# not minimized
            return {{} {}}                                                                            ;# list of 2 empty coordinates
        } else {
            return [$($this,canvas) coords icon($icon)]
        }
    }

    proc relativeStackingLevels {this} {                                  ;# return paths relative levels sorted in increasing order
        variable ${this}data

        set list {}
        foreach {name value} [array get ${this}data relativeStackingLevel,*] {
            lappend list $value
        }
        return [lsort -integer $list]
    }

    proc stacked {this path raised} {                           ;# parameter is a boolean: either raised to top or lowered to bottom
        variable ${this}data

        set levels [relativeStackingLevels $this]
        if {[llength $levels] == 0} {                                                              ;# first widget to be positionned
            set ${this}data(relativeStackingLevel,$path) 0
        } elseif {$raised} {
            set ${this}data(relativeStackingLevel,$path) [expr {[lindex $levels end] + 1}]        ;# place right above maximum level
        } else {
            set ${this}data(relativeStackingLevel,$path) [expr {[lindex $levels 0] - 1}]          ;# place right below minimum level
        }
    }

    proc raisedOnTop {this path} {
        variable ${this}data

        return [expr {[set ${this}data(relativeStackingLevel,$path)] >= [lindex [relativeStackingLevels $this] end]}]
    }

    proc raise {this next} {                                                              ;# next is a boolean: false means previous
        variable ${this}data
        variable ${this}handleIcon

        set canvas $($this,canvas)
        set bounds [bounds $this]                                                                                ;# for current page
        set handles {}
        foreach {name handle} [array get ${this}data handle,*] {
            if {[info exists ${this}handleIcon($handle)]} continue                                       ;# ignore minimized handles
            if {![intersect [$canvas bbox $handles::($handle,item)] $bounds]} continue              ;# ignore handles in other pages
            set path($handle) [scan $name handle,%s]
            lappend handles $handle
        }
        set length [llength $handles]
        if {$length < 2} return                                                                  ;# there can be no next or previous
        set handles [lsort -integer $handles]                                                           ;# sort in order of creation
        set maximum $global::integerMinimum
        set index 0
        foreach handle $handles {
            set level [set ${this}data(relativeStackingLevel,$path($handle))]
            if {$level > $maximum} {
                set maximum $level
                set top $index
            }
            incr index
        }
        if {$next} {
            if {[incr top] >= $length} {set top 0}                                          ;# eventually circle around to beginning
        } else {
            if {[incr top -1] < 0} {set top end}                                             ;# eventually circle around back to end
        }
        handles::stack [lindex $handles $top] raise
    }

    proc minimize {this handle title {xIcon {}} {yIcon {}} {static 0}} {
        variable ${this}handleIcon
        variable ${this}handleCoordinates
        variable ${this}handleIconCoordinates

        foreach {x y} [handles::getGeometry $handle] break                                           ;# retrieve current coordinates
        set ${this}handleCoordinates($handle) [list $x $y]                                                          ;# remember them
        handles::move $handle $global::integerMinimum $global::integerMinimum                   ;# make window invisible to the user
        set icon [new icon $($this,canvas) $title "canvasWindowManager::deIconify $this $handle" $static]
        if {[string length $xIcon] > 0} {
            $($this,canvas) move icon($icon) $xIcon $yIcon                                                ;# pre-defined coordinates
        } elseif {[info exists ${this}handleIconCoordinates($handle)]} {                                            ;# if available,
            eval $($this,canvas) move icon($icon) [set ${this}handleIconCoordinates($handle)]       ;# use previous icon coordinates
        } else {
            stowIcon $this $icon
        }
        set ${this}handleIcon($handle) $icon
    }

    proc deIconify {this handle} {
        variable ${this}handleIcon
        variable ${this}handleCoordinates
        variable ${this}handleIconCoordinates

        set icon [set ${this}handleIcon($handle)]
        if {$icon::($icon,moved)} {                                                                        ;# icon was moved by user
            set ${this}handleIconCoordinates($handle) [$($this,canvas) coords icon($icon)]               ;# remember its coordinates
        }
        delete $icon
        eval handles::move $handle [set ${this}handleCoordinates($handle)]                        ;# place back in original position
        handles::stack $handle raise                                      ;# but on top of the others so that it can be easily found
        unset ${this}handleIcon($handle) ${this}handleCoordinates($handle)
    }

    proc stowIcon {this identifier} {                ;# place icon at the bottom left of the canvas current page where there is room
        set canvas $($this,canvas)
        set padding $global::iconPadding
        ::update idletasks                                                                      ;# needed to get canvas correct size
        set bounds [bounds $this]                                                                                ;# for current page
        foreach {region(left) region(top) region(right) region(bottom)} $bounds {}
        foreach item [$canvas find all] {
            set index 0
            foreach tag [$canvas gettags $item] {
                if {[scan $tag icon(%u) index]>0} break                                                             ;# found an icon
            }
            if {($index == 0) || ($index == $identifier)} continue                     ;# ignore items other than icons and new icon
            if {![intersect [$canvas bbox icon($index)] $bounds]} continue                            ;# ignore icons in other pages
            set found($index) {}             ;# eliminate duplicates (an icon may be composed of several elements with the same tag)
        }
        set coordinates {}
        foreach index [array names found] {
            lappend coordinates [$canvas bbox icon($index)]
        }
        set coordinates [lsort -integer -index 0 $coordinates]                                                  ;# order by abscissa
        foreach {left top right bottom} [$canvas bbox icon($identifier)] {}                        ;# all icons have the same height
        set height [expr {$bottom - $top + (2 * $padding)}]  ;# icons are initially placed is slices with a reasonable padding value
        set width [expr {$right - $left + (2 * $padding)}]
        set maximum $region(bottom)
        while {[set minimum [expr {$maximum - $height}]] >= 0} {                 ;# look for room in slices starting from the bottom
            set spaces {}                                                              ;# build a list a empty segments in the slice
            set x $region(left)                                                          ;# the right side of the last occupied area
            foreach list $coordinates {
                foreach {left top right bottom} $list {}
                if {($top > $maximum) || ($bottom < $minimum)} continue                    ;# icon area does not intersect the slice
                if {$left > $x} {                                                                           ;# an empty space exists
                    lappend spaces $x $left
                }
                set x $right
            }
            if {$x < $region(right)} {
                lappend spaces $x $region(right)                                  ;# space remaining on the right slice of the slice
            }
            # find out whether there is a wide enough empty area in the slice to contain the new icon:
            foreach {left right} $spaces {
                if {($right - $left) > $width} {
                    set position(x) $left
                    set position(y) $minimum
                    break                                                                           ;# a wide enough space was found
                }
            }
            if {[info exists position]} break                                                             ;# a valid space was found
            set maximum $minimum                                                                        ;# look in the next slice up
        }
        # if no large enough empty area was found (unlikely), place the icon at the bottom left corner, maybe on top of other ones:
        if {![info exists position]} {
            set position(x) [expr {$region(left)}]
            set position(y) [expr {$region(bottom) - $height}]
        }
        foreach {x y} [$canvas coords icon($identifier)] break
        $canvas move icon($identifier) [expr {$position(x) - $x + $padding}] [expr {$position(y) - $y + $padding}]
    }

    proc rearrangeIcons {this width height} {                           ;# make sure icons are always visible when canvas is resized
        set canvas $($this,canvas)
        foreach {left top right bottom} [$canvas cget -scrollregion] {}
        set width [maximum $width [expr {$right - $left}]]                     ;# in case canvas was sized pass its scrolling region
        set height [maximum $height [expr {$bottom - $top}]]
        foreach item [$canvas find all] {
            set identifier 0
            foreach tag [$canvas gettags $item] {
                if {[scan $tag icon(%u) identifier] > 0} break                                                      ;# found an icon
            }
            if {$identifier == 0} continue                                                                    ;# item is not an icon
            # find the page that the icon belongs to and calculate its bounds:
            foreach {left top} [pages::closestPageTopLeftCorner [lindex [$canvas coords icon($identifier)] 0]] {}
            set right [expr {$left + $width}]
            set bottom [expr {$top + $height}]
            foreach {icon(left) icon(top) icon(right) icon(bottom)} [$canvas bbox icon($identifier)] {}
            # move the icon inside its page boundaries if necessary:
            if {$icon(right) > $right} {$canvas move icon($identifier) [expr {$right - $icon(right)}] 0}
            if {$icon(bottom) > $bottom} {$canvas move icon($identifier) 0 [expr {$bottom - $icon(bottom)}]}
            if {$icon(left) < $left} {$canvas move icon($identifier) [expr {$left - $icon(left)}] 0}
            if {$icon(top) < $top} {$canvas move icon($identifier) 0 [expr {$top - $icon(top)}]}
        }
    }

    proc bounds {this} {                                            ;# returns the current canvas bounds (changes when page changes)
        set canvas $($this,canvas)
        foreach {left top dummy dummy} [$canvas cget -scrollregion] {}                             ;# current page upper left corner
        # actual size may be larger than user specified size:
        return [list\
            $left $top\
            [expr {$left + [maximum $global::canvasWidth [winfo width $canvas]]}]\
            [expr {$top + [maximum $global::canvasHeight [winfo height $canvas]]}]\
        ]
    }

    proc currentPageEmpty {this} {
        variable ${this}data
        variable ${this}handleIcon

        set canvas $($this,canvas)
        set bounds [bounds $this]                                                                                ;# for current page
        foreach {name handle} [array get ${this}data handle,*] {
            set tag $handles::($handle,item)
            catch {set tag icon([set ${this}handleIcon($handle)])}                                        ;# handle may be minimized
            if {[intersect [$canvas bbox $tag] $bounds]} {                                                      ;# item lies in page
                return 0
            }
        }
        return 1
    }

    # return the page where the viewer or table lies or nothing if failure, such as the object not being managed
    proc viewerPage {this object} {                                                                                    ;# identifier
        variable ${this}data
        variable ${this}handleIcon

        if {[catch {set handle [set ${this}data(viewerHandle,$object)]}]} {
            return {}                                                                                       ;# object is not managed
        }
        set tag $handles::($handle,item)
        catch {set tag icon([set ${this}handleIcon($handle)])}                                            ;# handle may be minimized
        return [pages::tagOrItemPage $tag]
    }

    proc handles {this} {                                                                  ;# returns a list of the existing handles
        variable ${this}data

        set list {}
        foreach {name handle} [array get ${this}data handle,*] {
            lappend list $handle
        }
        return $list
    }

    # list of rectangles (borders of managed windows) from the same page, including visible canvas rectangle
    proc rectangles {this exclude} {                                                                            ;# handle to exclude
        variable ${this}handleIcon

        set canvas $($this,canvas)
        set page [pages::tagOrItemPage $handles::($exclude,item)]
        set list {}
        foreach handle [handles $this] {
            if {($handle == $exclude) || [info exists ${this}handleIcon($handle)]} continue                             ;# minimized
            set item $handles::($handle,item)
            if {![string equal [pages::tagOrItemPage $item] $page]} continue              ;# page may be empty if there are no pages
            foreach {x y} [coordinates $canvas $item] {}
            lappend list [list $x $y [winfo width $widget::($handle,path)] [winfo height $widget::($handle,path)]]
        }
        return $list
    }

    proc coordinates {canvas itemOrTag} {                                                ;# return coordinates in pixels as integers
        set values {}
        foreach value [$canvas coords $itemOrTag] {
            lappend values [expr {round($value)}]
        }
        return $values
    }

}
