/* libmondo-stream.c

- tools for talking to tapes, Monitas streams, etc.

12/13
- cdrecord -scanbus call was missing 2> /dev/null

11/24
- disabled fatal_error("Bad marker")

10/29
- replaced convoluted grep wih wc (KP)

09/07
- removed g_end_of_tape_reached
- lots of multitape-related fixes

08/01 - 08/31
- write 16MB of zeroes to end of tape when closing(out)
- compensate for 'buffer' and its distortion of speed of writing/reading
  when writing/reading data disks at start of tape
- rewrote lots of multitape stuff
- wrote workaround to allow >2GB of archives w/buffering
- do not close/reopen tape when starting to read/write from
  new tape: no need! 'buffer' handles all that; we're writing
  to/reading from a FIFO, so no need to close/reopen when new tape
- write 8MB of zeroes at end of tape, just in case
- re-enable various calls to *_evalcall_form
- added g_end_of_tape_reached
- fixed bugs in start_to_[read|write]_[to|from]_next_tape
- added internal buffering, replacing the external 'buffer' exe
- wait 10 seconds (after user inserts new tape), to
  let tape stabilize in drive
- added insist_on_this_tape_number()
- worked on find_tape_device_and_size()
- cleaned up some log_it() calls
- added find_tape_device_and_size()
- replaced using_cdstream with backup_media_type
- replace *_from_tape with *_from_stream
- replace *_to_stream with *_to_stream

07/01 - 07/31
- leave 32MB at end of tape, to avoid overrunning
- started [07/24]
*/



#include "my-stuff.h"
#include "mondostructures.h"
#include "libmondo-devices.h"
#include "lib-common-externs.h"
#include "libmondo-stream.h"
#include "libmondo-string-EXT.h"
#include "libmondo-files-EXT.h"
#include "libmondo-gui-EXT.h"
#include "libmondo-fork-EXT.h"
#include "libmondo-tools-EXT.h"
#include "libmondo-fifo-EXT.h"

#define EXTRA_TAPE_CHECKSUMS


FILE *g_tape_stream = NULL;
long long g_tape_posK = 0;
int g_current_media_number=-1;







void insist_on_this_tape_number(int tapeno)
{
  int i;
  char tmp[MAX_STR_LEN];

  log_it("Insisting on tape #%d", tapeno);
  if (g_current_media_number != tapeno)
    {
      //      log_it("g_current_media_number = %d", g_current_media_number);
      sprintf(tmp, "Please insert volume %d in this series.", tapeno);
      popup_and_OK (tmp);
      open_evalcall_form ("Waiting while the tape drive settles");
      for(i=0; i<10; i+=2)
        {
          sleep(1);
          update_evalcall_form (i*10);
        }
      close_evalcall_form();
      log_it("I assume user has inserted it. They _say_ they have...");
      g_current_media_number = tapeno;
    }
  //  log_it("g_current_media_number = %d", g_current_media_number);
  log_it("OK, I've finished insisting. On with the revelry.");
}








/*************************************************************************
 * ) -- Hugo Rabson
 *                                                                       *
 * Purpose:  
 * Called by:
 * Params:
 * Returns:
 * NB:
 *************************************************************************/
int
closein_cdstream (struct s_bkpinfo *bkpinfo)
{
  return (closein_tape (bkpinfo));
}



/*************************************************************************
 * closein_tape() -- Hugo Rabson                                         *
 *                                                                       *
 * Purpose:                                                              *
 * Called by:                                                            *
 * Returns:                                                              *
 *************************************************************************/
int
closein_tape (struct s_bkpinfo *bkpinfo)
{
	/** int's ********************************************************/
  int retval = 0;
  int res = 0;
  int ctrl_chr = '\0';

	/** buffers ******************************************************/
  char fname[MAX_STR_LEN];

	/** long long's **************************************************/
  long long size;
  char *blk;
  int i;

  blk = (char*)malloc(256*1024);

  log_it("closein_tape() -- entering");
  res = read_header_block_from_stream (&size, fname, &ctrl_chr);
  retval += res;
  if (ctrl_chr != BLK_END_OF_BACKUP)
    {
      wrong_marker (BLK_END_OF_BACKUP, ctrl_chr);
    }
  res = read_header_block_from_stream (&size, fname, &ctrl_chr);
  retval += res;
  if (ctrl_chr != BLK_END_OF_TAPE)
    {
      wrong_marker (BLK_END_OF_TAPE, ctrl_chr);
    }
  for(i=0; i<8 && !feof(g_tape_stream); i++)
    {
        fread (blk, 1, 256*1024, g_tape_stream);
    }
  sleep(1);
  system("sync");
  sleep(1);
  fclose (g_tape_stream);
  g_tape_stream = NULL;
  free(blk);
  stop_buffer_process(FALSE); // false==reading from tape
  log_it("closein_tape() -- leaving");
  return (retval);
}




/*************************************************************************
 * closeout_tape() -- Hugo Rabson                                        *
 *                                                                       *
 * Purpose:                                                              *
 * Called by:                                                            *
 * Returns:                                                              *
 *************************************************************************/
int
closeout_tape (struct s_bkpinfo *bkpinfo)
{
	/** int's ********************************************************/
  int retval = 0;
//  int res = 0;
//  int ctrl_chr = '\0';

	/** buffers ******************************************************/
//  char fname[MAX_STR_LEN];

	/** long long's **************************************************/
//  long long size;
  int i;
  char *blk;

  blk = (char*)malloc(256*1024);

  sleep(1);
  system("sync");
  sleep(1);
  log_it("closeout_tape() -- entering");
    retval+=write_header_block_to_stream (0, "end-of-backup", BLK_END_OF_BACKUP);
    retval+=write_header_block_to_stream (0, "end-of-tape", BLK_END_OF_TAPE);	/* just in case */
/* write 1MB of crap */
    for (i = 0; i < 256*1024; i++)
      {
        blk[i] = (int)(random()&0xFF);
      }
    for (i = 0; i < 4*8; i++)
      {
        fwrite (blk, 1, 256*1024, g_tape_stream);
        if (should_we_write_to_next_tape (bkpinfo->media_size[g_current_media_number], 256*1024))
          {
            start_to_write_to_next_tape (bkpinfo);
          }
      }
/* write 1MB of zeroes */
/*
    for (i = 0; i < 256*1024; i++)
      {
        blk[i] = 0;
      }
    for (i = 0; i < 4; i++)
      {
        fwrite (blk, 1, 256*1024, g_tape_stream);
        if (should_we_write_to_next_tape (bkpinfo->media_size[g_current_media_number], 256*1024))
          {
            start_to_write_to_next_tape (bkpinfo);
          }
      }
*/
  sleep(2);
  fclose (g_tape_stream);
  g_tape_stream = NULL;
  stop_buffer_process(TRUE); // true==writing from tape
  log_it("closeout_tape() -- leaving");
  free(blk);
  return (retval);
}








/*************************************************************************
 * find_tape_device_and_size() -- Hugo Rabson                            *
 *                                                                       *
 * Purpose:                                                              *
 * Called by:                                                            *
 * Returns:                                                              *
 *************************************************************************/
int
find_tape_device_and_size (char*dev, char*siz)
{
  char tmp[MAX_STR_LEN];
  char command[MAX_STR_LEN*2];
  char cdr_exe[MAX_STR_LEN];
//  char tape_description_[MAX_STR_LEN];
//  char tape_manufacturer_cdr[MAX_STR_LEN];
  FILE*fin;
  int res;

  dev[0] = siz[0] = '\0';
  if (!system("which dvdrecord > /dev/null 2> /dev/null"))
    { strcpy(cdr_exe, "dvdrecord"); }
  else
    { strcpy(cdr_exe, "cdrecord"); }
  sprintf(command, "%s -scanbus 2> /dev/null | grep -i tape | wc -l", cdr_exe);
  strcpy (tmp, call_program_and_get_last_line_of_output(command));
  if (atoi(tmp)!=1)
    {
      log_it("Either too few or too many tape streamers for me to detect...");
      strcpy(dev, "/dev/st0");
      return 1;
    }
  sprintf(command, "%s -scanbus 2> /dev/null | tr -s '\t' ' ' | grep \"[0-9]*,[0-9]*,[0-9]*\" | grep -v \"[0-9]*) \\*\" | grep -i TAPE | cut -d' ' -f2 | head -n1", cdr_exe);
  strcpy (tmp, call_program_and_get_last_line_of_output(command));
  if (strlen (tmp) < 2)
    {
      log_it ("Could not find tape device");
      return 1;
    }
  sprintf(command, "%s -scanbus 2> /dev/null | tr -s '\t' ' ' | grep \"[0-9]*,[0-9]*,[0-9]*\" | grep -v \"[0-9]*) \\*\" | grep -i TAPE | cut -d' ' -f3 | cut -d')' -f1 | head -n1", cdr_exe);
  strcpy (tmp, call_program_and_get_last_line_of_output(command));
  sprintf(dev, "/dev/st%s", tmp);
  res=0; if (!(fin = fopen(dev, "r"))) { res++; } else { fclose(fin); }
  if (res)
    {
      strcpy(dev, "/dev/ht0");
      res=0; if (!(fin = fopen(dev, "r"))) { res++; } else { fclose(fin); }
      log_it ("Cannot openin %s", dev); dev[0]='\0'; return(1);
    }
  strcpy(tmp, call_program_and_get_last_line_of_output("\
cdrecord -scanbus 2> /dev/null | tr -s '\t' ' ' | \
grep \"[0-9]*,[0-9]*,[0-9]*\" | grep -v \"[0-9]*) \\*\" | grep -i TAPE | \
awk '{for(i=1; i<NF; i++) { if (index($i, \"GB\")>0) { print $i;};};};'"));
  if (strlen(tmp)<2) { siz[0]='\0'; log_it("Warning - size of tape unknown"); return(0); }
  else { strcpy(siz, tmp); return(0); }
}





/*************************************************************************
 * openin_cdstream() -- Hugo Rabson                                      *
 *                                                                       *
 * Purpose:                                                              *
 * Called by:                                                            *
 * Returns:                                                              *
 *************************************************************************/
int
openin_cdstream (struct s_bkpinfo *bkpinfo)
{
  return (openin_tape (bkpinfo));
}


char g_tape_fifo[MAX_STR_LEN];


/*************************************************************************
 * openin_tape() -- Hugo Rabson                                          *
 *                                                                       *
 * Purpose:   Open tape device for input                                 *
 * Called by: ...                                                        *
 * Params:    bkpinfo             backup information structure           *
 * Returns:   0=success, nonzero=failure                                 *
 *************************************************************************/
int
openin_tape (struct s_bkpinfo *bkpinfo)
{
	/** buffer ******************************************************/
  char fname[MAX_STR_LEN];
  char *datablock;
  char tmp[MAX_STR_LEN];
  char old_cwd[MAX_STR_LEN];
  char outfname[MAX_STR_LEN];
	/** int ********************************************************/
  int i;
  int j;
  int res;
  long length, k, templong;
  int retval = 0;
  int ctrl_chr;

	/** long long **************************************************/
  long long size;

	/** pointers ***************************************************/
  FILE *fout;

	/** end vars ****************************************************/

  g_tape_posK = 0;
  if (g_tape_stream) { log_it("FYI - I won't 'openin' the tape. It's already open."); return(0); }
  insist_on_this_tape_number(1);
  sprintf (outfname, "%s/tmp/all.tar.gz", bkpinfo->tmpdir);
  make_hole_for_file (outfname);
  system("rm -Rf /tmp/mondo-tape-fifo.dev.*");
  make_fifo( g_tape_fifo, "/tmp/mondo-tape-fifo.dev.");
  start_buffer_process( g_tape_fifo, bkpinfo->media_device, FALSE );

//  start_buffer_process( bkpinfo->media_device, g_tape_fifo, FALSE);

  log_it ("Opening IN tape");
  if (!(g_tape_stream = fopen(g_tape_fifo, "r"))) { log_to_screen ("Cannot openin stream device"); return(1); }
  log_to_screen ("Reading stream");
  log_it ("stream device = '%s'", bkpinfo->media_device);
  g_tape_posK = 0;
/* skip data disks */
  open_evalcall_form ("Skipping data disks on stream");
  log_to_screen ("Skipping data disks on stream");
  if (!(fout = fopen (outfname, "w")))
    {
      log_to_screen ("Cannot openout datadisk all.tar.gz file");
      return(-1);
    }
  if (!(datablock = (char*)malloc(256*1024))) { log_to_screen("Unable to malloc 256*1024"); exit(1); }
  for (i = 0; i < 32; i++)
    {
      for (j = 0; j < 4; j++)
	{
          for(length=k=0; length<256*1024; length+=k)
            {
	      k=fread(datablock+length, 1, 256*1024 - length, g_tape_stream);
            }
          fwrite (datablock, 1, length, fout);
	  g_tape_posK += length/1024;
	}
      if (i>8) // otherwise, 'buffer' distorts calculations
        {
          templong = ((i-8) * 4 + j) * 100 / (128-8*4);
          update_evalcall_form ((int)(templong));
        }
    }
  fclose (fout);
  free(datablock);
  datablock=NULL;
/* find initial blocks */
  res = read_header_block_from_stream (&size, fname, &ctrl_chr);
  retval += res;
  if (ctrl_chr != BLK_START_OF_TAPE)
    {
      wrong_marker (BLK_START_OF_TAPE, ctrl_chr);
    }
  res = read_header_block_from_stream (&size, fname, &ctrl_chr);
  retval += res;
  if (ctrl_chr != BLK_START_OF_BACKUP)
    {
      wrong_marker (BLK_START_OF_BACKUP, ctrl_chr);
    }
  close_evalcall_form ();
  log_it ("Saved all.tar.gz to '%s'", outfname);
/*
  sprintf(tmp,"cp -f %s /tmp", outfname);
  system(tmp);
*/
  getcwd (old_cwd, MAX_STR_LEN);
  chdir (bkpinfo->tmpdir);
  sprintf (tmp, "tar -zxf %s tmp/mondo-restore.cfg 2> /dev/null", outfname);
  system(tmp);
  system ("cp -f tmp/mondo-restore.cfg . 2> /dev/null");
  chdir (old_cwd);
  unlink (outfname);
  free (datablock);
  return (retval);
}




/*************************************************************************
 * openout_cdstream() -- Hugo Rabson                                     *
 *                                                                       *
 * Purpose:    Commence writing to CD (stream, not regular ISO)          *
 * Called by:  ...                                                       *
 * Params:     cddev                 SCSI node, e.g. "0,0,0"             *
 *             speed                 write speed                         *
 * Returns:    0=success; nonzero=failure                                *
 *************************************************************************/
int
openout_cdstream (char *cddev, int speed)
{
	/** buffers ******************************************************/
  char command[MAX_STR_LEN*2];

	/** end vars ****************************************************/

/*  add 'dummy' if testing */
  sprintf (command,
	   "cdrecord -eject dev=%s speed=%d fs=24m -waiti - >> %s 2>> %s",
	   cddev, speed, MONDO_LOGFILE, MONDO_LOGFILE);
  log_it ("Opening OUT cdstream with the command");
  log_it (command);
/*  log_it("Let's see what happens, shall we?"); */
  g_tape_stream = popen (command, "w");
  if (g_tape_stream)
    {
      return (0);
    }
  else
    {
      log_to_screen ("Failed to openout to cdstream (fifo)");
      return (1);
    }
}


/*************************************************************************
 * openout_tape() -- Hugo Rabson                                         *
 *                                                                       *
 * Purpose:   Open tape device for output                                *
 * Called by: ...                                                        *
 * Params:    tapdev           tape streamer device (e.g. /dev/st0)      *
 * Returns:   0=success, nonzero=failure                                 *
 *************************************************************************/
int
openout_tape (char *tapedev)
{
  g_current_media_number = 1;
  if (g_tape_stream) { log_it("FYI - I won't 'openout' the tape. It's already open."); return(0); }

  log_it ("Opening OUT tape");
  system("rm -Rf /tmp/mondo-tape-fifo.dev.*");
  make_fifo( g_tape_fifo, "/tmp/mondo-tape-fifo.dev.");
  start_buffer_process( g_tape_fifo, tapedev, TRUE );

  if (!(g_tape_stream = fopen(g_tape_fifo, "w"))) { log_to_screen ("Cannot openin stream device"); return(1); }
  return(0);
}




/*************************************************************************
 * read_file_from_stream_to_file() -- Hugo Rabson                        *
 *                                                                       *
 * Purpose:                                                              *
 * Called by:                                                            *
 * Returns:                                                              *
 *************************************************************************/
int
read_file_from_stream_to_file (struct s_bkpinfo *bkpinfo, char *outfile,
			     long long size)
{

	/** int *********************************************************/
  int res;

	/** end vars ****************************************************/

  res = read_file_from_stream_FULL (bkpinfo, outfile, NULL, size);

  return (res);
}




/*************************************************************************
 * read_file_from_stream_to_stream() -- Hugo Rabson                        *
 *                                                                       *
 * Purpose:                                                              *
 * Called by:                                                            *
 * Returns:                                                              *
 *************************************************************************/
int
read_file_from_stream_to_stream (struct s_bkpinfo *bkpinfo, FILE * fout,
			       long long size)
{

	/** int *********************************************************/
  int res;

	/** end vars ****************************************************/

  res = read_file_from_stream_FULL (bkpinfo, NULL, fout, size);
/*  fflush(g_tape_stream);
  fflush(fout);*/
  return (res);
}

/*************************************************************************
 * read_file_from_stream_FULL() -- Hugo Rabson                             *
 *                                                                       *
 * Purpose:                                                              *
 * Called by:                                                            *
 * Returns:                                                              *
 *************************************************************************/
int
read_file_from_stream_FULL (struct s_bkpinfo *bkpinfo, char *outfname,
			  FILE * foutstream, long long orig_size)
{
	/** buffers ******************************************************/
  char tmp[MAX_STR_LEN];
  char datablock[TAPE_BLOCK_SIZE];
  char temp_fname[MAX_STR_LEN];
  char temp_cksum[MAX_STR_LEN];
  char actual_cksum[MAX_STR_LEN];

	/** int **********************************************************/
  int retval = 0;
#ifdef EXTRA_TAPE_CHECKSUMS
  int i, ch;
#endif
  int noof_blocks;
  int ctrl_chr;
  int res;

	/** pointers *****************************************************/
  FILE *fout;

	/** long    ******************************************************/
  long bytes_to_write = 0 /*,i */ ;

	/** long long ****************************************************/
  long long temp_size, size;

	/** unsigned int *************************************************/
  /*  unsigned int ch; */
  unsigned int crc16;
  unsigned int crctt;

	/** init  ********************************************************/
  crc16 = 0;
  crctt = 0;
  size = orig_size;


	/** end vars ****************************************************/
  res = read_header_block_from_stream (&temp_size, temp_fname, &ctrl_chr);
  if (orig_size != temp_size && orig_size != -1)
    {
      sprintf (tmp,
	       "output file's size should be %ld K but is apparently %ld K",
	       (long) size >> 10, (long) temp_size >> 10);
      log_to_screen (tmp);
    }
  if (ctrl_chr != BLK_START_FILE)
    {
      wrong_marker (BLK_START_FILE, ctrl_chr);
      return(1);
    }
  sprintf (tmp, "Reading file from tape; writing to '%s'; %ld KB", outfname,
	   (long) size >> 10);

  if (foutstream)
    {
      fout = foutstream;
    }
  else
    {
      fout = fopen (outfname, "w");
    }
  if (!fout)
    {
      log_it (outfname);
      log_to_screen ("Cannot openout file");
      return(1);
    }
  for (noof_blocks = 0; size > 0; noof_blocks++, size -= bytes_to_write)
    {
      if (size < TAPE_BLOCK_SIZE)
	{
	  bytes_to_write = size;
	}
      else
	{
	  bytes_to_write = TAPE_BLOCK_SIZE;
	}

      g_tape_posK += fread (datablock, 1, /*bytes_to_write */ TAPE_BLOCK_SIZE,
			    g_tape_stream) / 1024;
      fwrite (datablock, 1, bytes_to_write, fout);


#ifdef EXTRA_TAPE_CHECKSUMS
         for(i=0;i<bytes_to_write;i++)
         {
         ch=datablock[i];
         crc16=updcrcr(crc16,ch);
         crctt=updcrc(crctt,ch);
         }
#endif
    }
  sprintf (actual_cksum, "%04x%04x", crc16, crctt);
  if (foutstream)
    {				/*log_it("Finished writing to foutstream"); */
    }
  else
    {
      fclose (fout);
    }
  res = read_header_block_from_stream (&temp_size, temp_cksum, &ctrl_chr);
  if (ctrl_chr != BLK_STOP_FILE)
    {
      wrong_marker (BLK_STOP_FILE, ctrl_chr);
      //      fatal_error("Bad marker"); // return(1);
    }
  if (strcmp (temp_cksum, actual_cksum))
    {
      sprintf (tmp, "actual cksum=%s; recorded cksum=%s", actual_cksum,
	       temp_cksum);
      log_to_screen (tmp);
      sprintf (tmp, "%s (%ld K) is corrupt on tape", temp_fname,
	       (long) orig_size >> 10);
      log_to_screen (tmp);
      retval++;
    }
  else
    {
      sprintf (tmp, "%s is GOOD on tape", temp_fname);
      /*      log_it(tmp); */
    }
  return (retval);
}



/*************************************************************************
 * read_header_block_from_stream() -- Hugo Rabson                          *
 *                                                                       *
 * Purpose:                                                              *
 * Called by:                                                            *
 * Returns:                                                              *
 *************************************************************************/
int
read_header_block_from_stream (long long *plen, char *filename,
			     int *pcontrol_char)
{

	/** buffers ******************************************************/
  char *tempblock;

	/** int **********************************************************/
  int i, retval;

	/** end vars ****************************************************/

  tempblock = (char*)malloc(TAPE_BLOCK_SIZE);

  for (i = 0; i < TAPE_BLOCK_SIZE; i++)
    {
      tempblock[i] = 0;
    }
  while(!(*pcontrol_char = tempblock[7000]))
    {
      g_tape_posK += fread (tempblock, 1, TAPE_BLOCK_SIZE, g_tape_stream) / 1024;
    }
/*  memcpy((char*)plength_of_incoming_file,(char*)tempblock+7001,sizeof(long long)); */
/*  for(*plen=0,i=7;i>=0;i--) {*plen<<=8; *plen |= tempblock[7001+i];} */
  memcpy ((char *) plen, tempblock + 7001, sizeof (long long));
  if (strcmp (tempblock + 6000 + *pcontrol_char, "Mondolicious, baby"))
    {
      log_it ("Bad header block at %ld K", (long) g_tape_posK);
    }
  strcpy (filename, tempblock + 1000);
/*  strcpy(cksum,tempblock+5555);*/
/*  log_it( "%s  (reading) fname=%s, filesize=%ld K",
	   marker_to_string (*pcontrol_char), filename,
	   (long) ((*plen) >> 10));
*/
  if (*pcontrol_char == BLK_ABORTED_BACKUP)
    {
      log_to_screen ("I can't verify an aborted backup.");
      retval=1;
    }
  else 
    { retval=0; }
  for(i=1000; i<1020; i++) { if (tempblock[i]<32 || tempblock[i]>126) { tempblock[i]=' '; } }
  tempblock[i] = '\0';
//  log_it( "%s (fname=%s, size=%ld K)", marker_to_string (*pcontrol_char),
//	   tempblock+1000, (long) (*plen) >> 10);
  free(tempblock);
  return (retval);
}



/*************************************************************************
 * should_we_write_to_next_tape() -- Hugo Rabson                         *
 *                                                                       *
 * Purpose:                                                              *
 * Called by:                                                            *
 * Returns:                                                              *
 *************************************************************************/
bool
should_we_write_to_next_tape (long mediasize,
			      long long length_of_incoming_file)
{
	/** bool's ******************************************************/
  bool we_need_a_new_tape = FALSE;

	/** end vars ****************************************************/

  if (mediasize==0) { return(FALSE); }
  if (mediasize>0 && (g_tape_posK >> 10 >= mediasize))
    {
      log_it("mediasize = %ld", mediasize);
      we_need_a_new_tape = TRUE;
      log_to_screen ("Should have started a new tape/CD already");
    }
  if ((g_tape_posK + length_of_incoming_file / 1024) >> 10 >= mediasize-(SLICE_SIZE*4/1024))
    {
      log_it("g_tape_posK = %ld\nmediasize = %ld\n", g_tape_posK, mediasize);
      we_need_a_new_tape = TRUE;
    }
  return (we_need_a_new_tape);
}


/*************************************************************************
 * start_to_read_from_next_tape() -- Hugo Rabson                         *
 *                                                                       *
 * Purpose:                                                              *
 * Called by:                                                            *
 * Returns:                                                              *
 *************************************************************************/
int
start_to_read_from_next_tape (struct s_bkpinfo *bkpinfo)
{
	/** int **********************************************************/
  int res = 0;
//  int ctrl_chr = '\0';

  /** buffers ************************************************************/
//  char fname[MAX_STR_LEN];	/*cksum[MAX_STR_LEN]; */

  /** long long *********************************************************/
//  long long size;

	/** end vars ****************************************************/

  log_it ("Next tape requested.");
  insist_on_this_tape_number(g_current_media_number+1); // will increment it, too
  g_tape_posK = 0;
  return (res);
}




/*************************************************************************
 * start_to_write_to_next_tape() -- Hugo Rabson                          *
 *                                                                       *
 * Purpose:                                                              *
 * Called by:                                                            *
 * Returns:                                                              *
 *************************************************************************/
int
start_to_write_to_next_tape (struct s_bkpinfo *bkpinfo)
{
  int res=0;

  log_it ("New tape requested.");
  insist_on_this_tape_number(g_current_media_number+1); // will increment g_current_media, too
  if (g_current_media_number > MAX_NOOF_MEDIA) { res++; log_to_screen("Too many tapes. Man, you need to use nfs!"); }
  log_it ("Opening OUT additional tape");
  g_tape_posK = 0;
  return (res);
}




/*************************************************************************
 * write_data_disk_to_stream() -- Hugo Rabson                            *
 *                                                                       *
 * Purpose:                                                              *
 * Called by:                                                            *
 * Returns:                                                              *
 *************************************************************************/
int
write_data_disks_to_stream (char *fname)
{
	/** pointers ****************************************************/
  FILE *fin;
  char tmp[MAX_STR_LEN];

	/** long ********************************************************/
  long m = -1;
  long templong;

	/** int *********************************************************/
  int i,j;

	/** buffers *****************************************************/
  char tempblock[256 * 1024];

	/** end vars ****************************************************/

  open_evalcall_form ("Writing data disks to tape");
  log_to_screen ("Writing data disks to tape");
  if (!does_file_exist (fname))
    {
      sprintf (tmp, "Cannot find %s", fname);
      log_to_screen (tmp);
      return(1);
    }
  if (!(fin = fopen (fname, "r")))
    {
      fatal_error ("Cannot openin the data disk");
    }
  for (i = 0; i < 32; i++)	/* 32MB */
    {
      for (j = 0; j < 4; j++)	/* 256K x 4 = 1MB (1024K) */
	{
	  if (!feof (fin))
	    {
	      m = fread (tempblock, 1, 256 * 1024, fin);
	    }
	  else
	    {
	      m = 0;
	    }
	  for (; m < 256 * 1024; m++)
	    {
	      tempblock[m] = '\0';
	    }
	  g_tape_posK +=
	    fwrite (tempblock, 1, 256 * 1024, g_tape_stream) / 1024;
	}
      if (i>6) // otherwise, 'buffer' distorts calculations
        {
          templong = ((i-8) * 4 + j) * 100 / (128-8*4);
          update_evalcall_form ((int)(templong));
        }
    }
  fclose (fin);
  close_evalcall_form ();
  return (0);
}



/*************************************************************************
 * write_file_to_stream_from_file() -- Hugo Rabson                         *
 *                                                                       *
 * Purpose:                                                              *
 * Called by:                                                            *
 * Returns:                                                              *
 *************************************************************************/
int
write_file_to_stream_from_file (struct s_bkpinfo *bkpinfo, char *infile)
{
	/** buffers *****************************************************/
  char tmp[MAX_STR_LEN];
  char datablock[TAPE_BLOCK_SIZE];
  char checksum[MAX_STR_LEN];

	/** int *********************************************************/
  int retval = 0;
  int noof_blocks;

  /*  unsigned int ch; */
  unsigned int crc16;
  unsigned int crctt;

	/** pointers ****************************************************/
  FILE *fin;
  char *p;

	/** long ********************************************************/
  long bytes_to_read = 0;
  long i;

	/** long long ***************************************************/
  long long filesize;

#ifdef EXTRA_TAPE_CHECKSUMS
  int ch;
#endif

	/** initialize *************************************************/
  crc16 = 0;
  crctt = 0;


 
	/** end vars ****************************************************/
  filesize = length_of_file (infile);
  if (should_we_write_to_next_tape (bkpinfo->media_size[g_current_media_number], filesize))
    {
      start_to_write_to_next_tape (bkpinfo);
    }
  p = strrchr (infile, '/');
  if (!p)
    {
      p = infile;
    }
  else
    {
      p++;
    }
  sprintf (tmp, "Writing file '%s' to tape (%ld KB)", p,
	   (long) filesize >> 10);
  log_it (tmp);
  write_header_block_to_stream (filesize, infile, BLK_START_FILE);
  fin = fopen (infile, "r");
  for (noof_blocks = 0; filesize > 0;
       noof_blocks++, filesize -= bytes_to_read)
    {
      if (filesize < TAPE_BLOCK_SIZE)
	{
	  bytes_to_read = filesize;
	  for (i = 0; i < TAPE_BLOCK_SIZE; i++)
	    {
	      datablock[i] = '\0';
	    }
	}
      else
	{
	  bytes_to_read = TAPE_BLOCK_SIZE;
	}
      fread (datablock, 1, bytes_to_read, fin);
      g_tape_posK += fwrite (datablock, 1, /*bytes_to_read */ TAPE_BLOCK_SIZE,
			     g_tape_stream) / 1024;

#ifdef EXTRA_TAPE_CHECKSUMS
         for(i=0;i<bytes_to_read;i++)
         {
         ch=datablock[i];
         crc16=updcrcr(crc16,ch);
         crctt=updcrc(crctt,ch);
         }
#endif
    }
  fclose (fin);
  sprintf (checksum, "%04x%04x", crc16, crctt);
  write_header_block_to_stream (g_current_media_number, checksum,
			      BLK_STOP_FILE);
//  log_it("File '%s' written to tape.", infile);
  return (retval);
}






/*************************************************************************
 *write_header_block_to_stream() -- Hugo Rabson                            *
 *                                                                       *
 * Purpose:                                                              *
 * Called by:                                                            *
 * Returns:                                                              *
 *************************************************************************/
int
write_header_block_to_stream (long long length_of_incoming_file, char *filename,
			    int control_char)
{
	/** buffers *****************************************************/
  char tempblock[TAPE_BLOCK_SIZE];
  char tmp[MAX_STR_LEN];
  char *p;

	/** int *********************************************************/
  int i;

	/** long long ***************************************************/
  long long olen;

	/** end vars ****************************************************/

 
  olen = length_of_incoming_file;
  p = strrchr (filename, '/');	/* Make 'em go, "Unnnh!" Oh wait, that was _Master_ P... */
  if (!p)
    {
      p = filename;
    }
  else
    {
      p++;
    }
  if (!g_tape_stream)
    {
      log_to_screen
	("You're not backing up to tape. Why write a tape header?");
      return(1);
    }
  for (i = 0; i < TAPE_BLOCK_SIZE; i++)
    {
      tempblock[i] = 0;
    }
  sprintf (tempblock + 6000 + control_char, "Mondolicious, baby");
  tempblock[7000] = control_char;
/*  for(i=0;i<8;i++) {tempblock[7001+i]=olen&0xff; olen>>=8;} */
  memcpy (tempblock + 7001, (char *) &olen, sizeof (long long));
/*  if (length_of_incoming_file) {memcpy(tempblock+7001,(char*)&length_of_incoming_file,sizeof(long long));} */
  strcpy (tempblock + 1000, filename);
/*  strcpy(tempblock+5555,cksum); */
  g_tape_posK += fwrite (tempblock, 1, TAPE_BLOCK_SIZE, g_tape_stream) / 1024;
  sprintf (tmp, "%s (fname=%s, size=%ld K)", marker_to_string (control_char),
	   p, (long) length_of_incoming_file >> 10);
  /*  log_it(tmp); */
/*  log_tape_pos(); */
  return (0);
}




/*************************************************************************
 * wrong_marker() -- Hugo Rabson                                         *
 *                                                                       *
 * Purpose:                                                              *
 * Called by:                                                            *
 * Returns:                                                              *
 *************************************************************************/
void
wrong_marker (int should_be, int it_is)
{
	/** buffer ******************************************************/
  char tmp[MAX_STR_LEN];


	/** end vars ****************************************************/
  sprintf (tmp, "Wrong marker! (Should be %s, ",
	   marker_to_string (should_be));
  sprintf (tmp + strlen (tmp), "is actually %s)", marker_to_string (it_is));
  log_to_screen (tmp);
}






/*************************************************************************
 * log_tape_pos() -- Hugo Rabson                                         *
 *                                                                       *
 * Purpose:                                                              *
 * Called by:                                                            *
 * Returns:                                                              *
 *************************************************************************/
void
log_tape_pos (void)
{
	/** buffers ******************************************************/


	/** end vars ****************************************************/

  log_it ("Tape position -- %ld KB (%ld MB)", (long) g_tape_posK,
	   (long) g_tape_posK >> 10);
}







