/* libmondo-mountlist.c                            subroutines for handling mountlist


02/12/2003
- added code for LVM (Brian Borgeson)

10/19/2002
- added some comments

07/24
- created
*/


#include "my-stuff.h"
#include "mondostructures.h"
#include "libmondo-mountlist.h"
#include "lib-common-externs.h"
#include "libmondo-raid-EXT.h"
#include "libmondo-devices-EXT.h"
#include "libmondo-tools-EXT.h"
#include "libmondo-string-EXT.h"
#include "libmondo-gui-EXT.h"


/*************************************************************************
 * evaluate_drive_within_mountlist() -- Hugo Rabson                      *
 *                                                                       *
 * Purpose:  See if there is something missing from a particular drive   *
 *           in the mountlist. For example, if /dev/hda6 is the first    *
 *           partition beginning with '/dev/hda' then that's bad! You    *
 *           need at least one primary ('1') and the first logical should*
 *           be '5'. This subroutine will check for mistakes such as that*
 * Called by:evaluate_mountlist                                          *
 * Params:   mountlist            array containing mountlist             *
 *           drive                drive to examine (e.g. /dev/hda)       *
 *           flaws_str            [returned] string listing flaws        *
 * Returns:  0=OK; nonzero=number of errors found                        *
 *************************************************************************/
int
evaluate_drive_within_mountlist (struct mountlist_itself *mountlist,
				 char *drive, char *flaws_str)
{

	/** int **************************************************************/
  int prev_part_no = 0;
	int curr_part_no = 0;
	int pos = 0;
	int res = 0;
	int mountpoint_copies = 0;
	int device_copies = 0;
	int i = 0;

	/** buffers *********************************************************/
  char tmp[MAX_STR_LEN];
	char device[MAX_STR_LEN];
	char mountpoint[MAX_STR_LEN];

	/** long ************************************************************/
  long physical_drive_size = 0;
	long amount_allocated = 0;

	/** initialize ******************************************************/
  flaws_str[0] = '\0';
  prev_part_no = 0;
  tmp[0] = '\0';


  physical_drive_size = get_phys_size_of_drive (drive);

  if (physical_drive_size < 0)
    {
      sprintf (tmp, " %s does not exist.", drive);
      strcat (flaws_str, tmp);
    }
  else
    {
      sprintf (tmp, "%s is %ld MB", drive, physical_drive_size);
    }
  log_it (tmp);

  for (curr_part_no = 1; curr_part_no < 99; curr_part_no++)
    {
      sprintf (device, "%s%d", drive, curr_part_no);
      pos = find_device_in_mountlist (mountlist, device);
      if (pos < 0)
	{
	  continue;
	}
      strcpy (mountpoint, mountlist->el[pos].mountpoint);
      /* gap in the partition list? */
      if (curr_part_no - prev_part_no > 1)
	{
	  if (prev_part_no == 0)
	    {
	      sprintf (tmp, " Gap prior to %s.", device);
	      log_it (tmp);
	      strcat (flaws_str, tmp);
	      res++;
	    }
	  else if (curr_part_no > 5
		   || (curr_part_no <= 4 && prev_part_no > 0))
	    {
	      sprintf (tmp, " Gap between %s%d and %d.", drive, prev_part_no,
		       curr_part_no);
	      log_it (tmp);
	      strcat (flaws_str, tmp);
	      res++;
	    }
	}
      /* no spare primary partitions to help accommodate the logical(s)? */
      if (curr_part_no >= 5 && prev_part_no == 4)
	{
	  sprintf (tmp, " Partition %s4 is occupied.", drive);
	  log_it (tmp);
	  strcat (flaws_str, tmp);
	  res++;
	}
      /* does partition /dev/hdNX exist more than once in the mountlist? */
      for (i = 0, mountpoint_copies = 0, device_copies = 0;
	   i < mountlist->entries; i++)
	{
	  if (!strcmp (device, mountlist->el[i].device))
	    {
	      device_copies++;
	    }
	}
      if (device_copies > 1)
	{
	  sprintf (tmp, " %s %s's.", number_to_text (device_copies), device);
	  if (!strstr (flaws_str, tmp))
	    {
	      log_it (tmp);
	      strcat (flaws_str, tmp);
	      res++;
	    }
	}
      /* silly partition size? */
      if (mountlist->el[pos].size < 8192
	  && strcmp (mountlist->el[pos].mountpoint, "lvm"))
	{
	  sprintf (tmp, " %s is tiny!", device);
	  log_it (tmp);
	  strcat (flaws_str, tmp);
	  res++;
	}
      /* mountpoint should begin with / unless it is swap, lvm or raid */
      if (strcmp (mountlist->el[pos].mountpoint, "swap")
	  && strcmp (mountlist->el[pos].mountpoint, "lvm")
	  && strcmp (mountlist->el[pos].mountpoint, "raid")
	  && strcmp (mountlist->el[pos].mountpoint, "image")
	  && mountlist->el[pos].mountpoint[0] != '/')
	{
	  sprintf (tmp, " %s has a weird mountpoint.", device);
	  log_it (tmp);
	  strcat (flaws_str, tmp);
	  res++;
	}
      /* is format sensible? */
      if (!is_this_a_valid_disk_format (mountlist->el[pos].format))
	{
	  sprintf (tmp, " %s has unsupported format.", device);
	  log_it (tmp);
	  strcat (flaws_str, tmp);
	  res++;
	}
      /* OK, continue with main loop */
      amount_allocated += mountlist->el[pos].size / 1024;
      prev_part_no = curr_part_no;
    }

  /* Over-allocated the disk? Unallocated space on disk? */
  if (amount_allocated > physical_drive_size + 1)
    {
      sprintf (tmp, " %ld MB over-allocated on %s.",
	       amount_allocated - physical_drive_size, drive);
      log_it (tmp);
      strcat (flaws_str, tmp);
      res++;
    }
  else if (amount_allocated < physical_drive_size - 1)
    {				/* NOT AN ERROR, JUST A WARNING :-) */
      sprintf (tmp, " %ld MB unallocated on %s.",
	       physical_drive_size - amount_allocated, drive);
      log_it (tmp), strcat (flaws_str, tmp);
    }
  if (res)
    {
      return (FALSE);
    }
  else
    {
      return (TRUE);
    }
}





/*************************************************************************
 * evaluate_mountlist() -- Hugo Rabson                                   *
 *                                                                       *
 * Purpose:  See if there is something missing from the mountlist.       *
 *           in the mountlist. For example, if /dev/hda6 is the first    *
 *           partition beginning with '/dev/hda' then that's bad! You    *
 *           need at least one primary ('1') and the first logical should*
 *           be '5'. This subroutine will check for mistakes such as that*
 *           for every drive listed in the mountlist.                    *
 * Called by:mountlist           array containing mountlist              *
 *           flaws_str_A         [ret'd] 1st line listing errors found   *
 *           flaws_str_B         [ret'd] 2nd line listing errors found   *
 *           flaws_str_C         [ret'd] 3rd line listing errors found   *
 * Returns:  0=ok; nonzero=number of errors                              *
 *************************************************************************/
int
evaluate_mountlist (struct mountlist_itself *mountlist, char *flaws_str_A,
		    char *flaws_str_B, char *flaws_str_C)
{

	/** buffer ************************************************************/
  char list_of_drives[ARBITRARY_MAXIMUM][MAX_STR_LEN];
	char tmp[MAX_STR_LEN];
	char flaws_str[MAX_STR_LEN];

	/** int ***************************************************************/
  int noof_drives	= 0;
	int i = 0;
	int res = 0;

	/** initialize ********************************************************/
  flaws_str[0] = '\0';

	noof_drives = make_list_of_drives_in_mountlist (mountlist, list_of_drives);

  log_it ("Evaluating mountlist...");

 for (i = 0; i < noof_drives; i++)
    {
      if (strstr (list_of_drives[i], "/dev/md"))
	{
	  sprintf (tmp, " Not evaluating %s (I don't know how yet)",
		   list_of_drives[i]);
	  log_it (tmp);
	  tmp[0] = '\0';
	}
      else
	{
	  if (!evaluate_drive_within_mountlist
	      (mountlist, list_of_drives[i], tmp))
	    {
	      res++;
	    }
	}
      strcat (flaws_str, tmp);
    }
  res += look_for_duplicate_mountpoints (mountlist, flaws_str);
/*  res+=look_for_weird_formats(mountlist,flaws_str); .. not necessary, now that we can check to see
 which formarts are actually _supported_ by the kernel */
  /* log_it(flaws_str); */
  return (spread_flaws_across_three_lines
	  (flaws_str, flaws_str_A, flaws_str_B, flaws_str_C, res));
}



/*************************************************************************
 * find_device_in_mountlist() -- Hugo Rabson                             *
 *                                                                       *
 * Purpose:  Find the index# of device, e.g. /dev/hda1, in mountlist     *
 * Called by:                                                            *
 * Params:   mountlist                      array containing mountlist   *
 *           device                         device to search for         *
 * Returns:  element#, or -1 if not found                                *
 *************************************************************************/
int
find_device_in_mountlist (struct mountlist_itself *mountlist, char *device)
{

	/** int ***************************************************************/
  int i = 0;


  for (i = 0;
       i < mountlist->entries
       && strcmp (mountlist->el[i].device, device) != 0; i++);
  if (i == mountlist->entries)
    {
      return (-1);
    }
  else
    {
      return (i);
    }
}






/*************************************************************************
 * look_for_duplicate_mountpoints() -- Hugo Rabson                       *
 *                                                                       *
 * Purpose:  Look in mountlist for duplicate /dev entries in it          *
 * Called by:evaluate_mountlist                                          *
 * Params:   mountlist             array containing mountlist            *
 *           flaws_str             [returned] flaws found in mountlist   *
 * Returns:  0=ok, nonzero=number of errors found                        *
 *************************************************************************/
int
look_for_duplicate_mountpoints (struct mountlist_itself *mountlist,
				char *flaws_str)
{

	/** int **************************************************************/
  int res = 0;
	int currline = 0;
	int i = 0;
	int copies = 0;
	int last_copy = 0;

	/** buffetr **********************************************************/
  char curr_mountpoint[MAX_STR_LEN];
	char tmp[MAX_STR_LEN];


  for (currline = 0; currline < mountlist->entries; currline++)
    {
      strcpy (curr_mountpoint, mountlist->el[currline].mountpoint);
      for (i = 0, copies = 0, last_copy = -1; i < mountlist->entries; i++)
	{
	  if (!strcmp (mountlist->el[i].mountpoint, curr_mountpoint)
	      && strcmp (mountlist->el[i].mountpoint, "lvm")
	      && strcmp (mountlist->el[i].mountpoint, "swap"))
	    {
	      last_copy = i;
	      copies++;
	    }
	}
      if (copies > 1 && last_copy == currline
	  && strcmp (curr_mountpoint, "raid"))
	{
	  sprintf (tmp, " %s %s's.", number_to_text (copies),
		   curr_mountpoint);
	  strcat (flaws_str, tmp);
	  log_it (tmp);
	  res++;
	}
    }
  return (res);
}


/*************************************************************************
 * look_for_weird_formats() -- Hugo Rabson                               *
 *                                                                       *
 * Purpose:  See if any partition on mountlist is due to be formatted in *
 *           some way which the user's kernel simply cannot accommodate. *
 * Called by:evaluate_mountlist                                          *
 * Params:   mountlist             array containing mountlist            *
 *           flaws_str             [returned] flaws found in mountlist   *
 * Returns:  0=ok, nonzero=number of errors found                        *
 *************************************************************************/
int
look_for_weird_formats (struct mountlist_itself *mountlist, char *flaws_str)
{

	/** int **************************************************************/
  int i	= 0;
	int res = 0;

	/** buffers **********************************************************/
  char tmp[MAX_STR_LEN];
	char format_sz[MAX_STR_LEN];



  for (i = 0; i < mountlist->entries; i++)
    {
      sprintf (format_sz, " %s ", mountlist->el[i].format);
      if (!strstr
	  ("swap image vfat ext2 ext3 xfs vfs jfs reiserfs dos minix coda nfs ntfs hpfs raid lvm",
	   format_sz) && strcmp (mountlist->el[i].mountpoint, "image") != 0)
	{
	  sprintf (tmp, " %s has unknown format.", mountlist->el[i].device);
	  log_it (tmp);
	  strcat (flaws_str, tmp);
	  res++;
	}
      else
	if ((!strcmp (mountlist->el[i].format, "swap")
	     && strcmp (mountlist->el[i].mountpoint, "swap"))
	    || (strcmp (mountlist->el[i].format, "swap")
		&& !strcmp (mountlist->el[i].mountpoint, "swap")))
	{
	  sprintf (tmp, " %s is half-swap.", mountlist->el[i].device);
	  log_it (tmp);
	  strcat (flaws_str, tmp);
	  res++;
	}
    }
  return (res);
}




/*************************************************************************
 * make_list_of_drives_in_mountlist) -- Hugo Rabson                      *
 *                                                                       *
 * Purpose:   Make list of drives mentioned in mountlist                 *
 * Called by: evaluate_mountlist()                                       *
 * Params:    mountlist                       mountlist structure        *
 *            drivelist                       (returned)                 *
 * Returns:   struct drivelist[][]            list of drives in mountlist*
 *            noof_drives                     0 or less if failure       *
 *************************************************************************/
int
make_list_of_drives_in_mountlist (struct mountlist_itself *mountlist,
		     char drivelist[ARBITRARY_MAXIMUM][MAX_STR_LEN])
{

		/** int **************************************************************/
  int lino;
  int noof_drives = 0;
  int j;

		/** buffers **********************************************************/
  char drive[MAX_STR_LEN];
  long size;
  char tmp[MAX_STR_LEN];

  log_it ("Making list of drives");
  for (lino = 0; lino < mountlist->entries; lino++)
    {
      strcpy (drive, mountlist->el[lino].device);
      if (!strncmp (drive, "/dev/md", 7))
	{
	  sprintf (tmp,
		   "Not putting %s in list of drives: it's a virtual drive",
		   drive);
	  log_it (tmp);
	  continue;
	}

// added -- BB, 02/12/2003
      size=mountlist->el[lino].size;
      if ( size == 0 )
        {
          sprintf (tmp,
                   "Not putting %s in list of drives: it has zero size (maybe an LVM volume)",
                   drive);
          log_it (tmp);
          continue;
        }
// :-)

/*
      for (i = strlen (drive); isdigit (drive[i - 1]); i--);
      drive[i] = '\0';
      if (get_phys_size_of_drive (drive) <= 0 && drive[i - 1] == 'p')
	{
	  i--;
	  drive[i] = '\0';
	}
      for (j = 0; j < noof_drives && strcmp (drivelist[j], drive) != 0; j++);
*/

      truncate_to_drive_name (drive);
      for (j = 0; j < noof_drives && strcmp (drivelist[j], drive) != 0; j++)
	continue;
      if (j == noof_drives)
	{
	  strcpy (drivelist[noof_drives++], drive);
	}
    }
  log_it ("Made list of drives");
  return (noof_drives);
}







/*************************************************************************
 * make_list_of_unallocated_raid_partitions() -- Hugo Rabson  	         *
 *                                                                       *
 * Purpose:  Make list of RAID slices, listed in mountlist, which are    *
 *           not currently associated with any software RAID devices     *
 * Called by:select_raid_disks                                           *
 * Params:   output_list        [returned]                               *
 *           mountlist          array containing mountlist               *
 *           raidlist           array of RAID devices and assoc'd slices *
 * Returns:  output_list        list of non-associated slices            *
 *************************************************************************/
void
make_list_of_unallocated_raid_partitions (struct mountlist_itself
					  *output_list,
					  struct mountlist_itself *mountlist,
					  struct raidlist_itself *raidlist)
{

	/** int **************************************************************/
  int items = 0;
	int i = 0;
	int used_by = 0;

	/** buffers **********************************************************/
  char tmp[MAX_STR_LEN];


  log_it ("MLOURP -- starting");
  items = 0;


  for (i = 0; i < mountlist->entries; i++)
    {
      if (strstr (mountlist->el[i].mountpoint, "raid"))
	{
	  used_by =
	    which_raid_device_is_using_this_partition (raidlist,
						       mountlist->el[i].
						       device);
	  if (used_by < 0)
	    {
	      memcpy ((void *) &output_list->el[items++],
		      (void *) &mountlist->el[i],
		      sizeof (struct mountlist_line));
	      sprintf (tmp,
		       "%s is available; user may choose to add it to raid device",
		       output_list->el[items - 1].device);
	      log_it (tmp);
	    }
	}
    }
  output_list->entries = items;
  log_it ("MLUORP -- ending");
}






/*************************************************************************
 * size_of_specific_device_in_mountlist() -- Hugo Rabson                 *
 *                                                                       *
 * Purpose:  Return the 'size' field of a specific mountlist record.     *
 *           Search by device (/dev) entry.                              *
 * Called by:                                                            *
 * Params:   mountlist                  array containing mountlist       *
 *           device                     device, e.g. /dev/hda1           *
 * Returns:  Length in KBytes, or -1 if device not found in mountlist.   *
 *************************************************************************/
long
size_of_specific_device_in_mountlist (struct mountlist_itself *mountlist, char *device)
{
	/** int ***************************************************************/
  int i = 0;



  for (i = 0;
       i < mountlist->entries && strcmp (mountlist->el[i].device, device);
       i++);
  if (i == mountlist->entries)
    {
      return (-1);
    }
  else
    {
      return (mountlist->el[i].size);
    }
}


