/* libmondo-fifo.c


04/22/2003
- copy_file_from_here_to_there() --- added a bit of fault tolerance;
  if write fails, retry a few times before reporting error

11/30/2002
- is_incoming_block_valid() --- always make
  checksums %65536, just in case int size is
  odd (GRRR, ArkLinux)

11/22
- disabled rotor-related superfluous warnings
- added INTERNAL_TAPE_BLK_SIZE

11/10
- do_sem() now returns int explicitly
- changed internal_block_size

10/04
- some irregularities (e.g. bad 'type'-ing) found by Kylix; fixed by Hugo

09/01 - 09/30
- change 64k to TAPE_BLOCK_SIZE
- added internal_block_size; set it to TAPE_BLOCK_SIZE*2
- if data is flowing FROM tape TO hard disk then set the threshold to 10 (not 75)
- lots of multitape-related fixes
- finally caught & fixed the 'won't finish unzipping last bigfile' bug
- added WIFEXITED() after waitpid(), to improve multi-tape support

08/01 - 08/31
- trying to catch & stop 'won't finish unzipping last bigfile' bug by
  changing the copy_file_rom_here_to_there code
- changed % threshold from 95 back to 75
- don't insist on full 256K write of last block to tape
- if >25 secs go by & all data (AFAIK) has been copied thru by FIFO wrapper
  and g_tape_stream is _still_ not closed then shrug shoulders & pthread_exit
  anyway...
- change fprintf()'s to log_it()'s
- added a header+footer to each block as it is read/written to/from tape
  by copy_file_from_here_to_there
- wrote workaround to allow >2GB of archives w/buffering
- changed % threshold from 75 to 95
- added calls to set_signals(); 'buffer' was killing mondoarchive as
  it terminated
- cleaned up struct-passing, to improve reliability and eliminate
  some race conditions
- changed some forks to pthreads
- added some comments
- created libfifo{.c,.h,-EXT.h}
- copied core of 'buffer' here
- added some other, Mondo-specific functions
- hacked 'buffer' into user-friendliness
*/


/* Almost everything after this line comes straight from the original 'buffer' tgz/rpm */


/*
    Buffer.  Very fast reblocking filter speedy writing of tapes.
    Copyright (C) 1990,1991  Lee McLoughlin

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    Lee McLoughlin.
    Dept of Computing, Imperial College,
    180 Queens Gate, London, SW7 2BZ, UK.

    Email: L.McLoughlin@doc.ic.ac.uk
*/

/* This is a reblocking process, designed to try and read from stdin
 * and write to stdout - but to always try and keep the writing side
 * busy.  It is meant to try and stream tape writes.
 *
 * This program runs in two parts.  The reader and the writer.  They
 * communicate using shared memory with semaphores locking the access.
 * The shared memory implements a circular list of blocks of data.
 *
 * L.McLoughlin, Imperial College, 1990
 *
 * 
 */
#include <unistd.h>
#include <stdio.h>
#include <signal.h>
#include <fcntl.h>
#include <stdio.h>

#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <sys/sem.h>
#include <sys/wait.h>
#include <pthread.h>

#include "my-stuff.h"
#include "mondostructures.h"
#include "libmondo.h"


#define PRINT_EVERY 10 K


//#ifndef lint
//static char *rcsid = "$Header: /mondorescue/mondo-stable/mondo/common/libmondo-fifo.c,v 1.2 2003/06/06 14:52:28 hugo Exp $";
//#endif

#ifndef __alpha
//extern char *shmat();
#endif /* __alpha */

/* General macros */
#define TRUE 1
#define FALSE 0
#define K *1024
#define M *1024*1024




/* This is the inter-process buffer - it implements a circular list
 * of blocks. */

#ifdef AMPEX
#define MAX_BLOCKSIZE (4 M)
#define DEF_BLOCKSIZE MAX_BLOCKSIZE
#define DEF_SHMEM (32 M)
#endif


#ifndef MAX_BLOCKSIZE
#define MAX_BLOCKSIZE (512 K)
#endif
#ifndef DEF_BLOCKSIZE
#define DEF_BLOCKSIZE (10 K)
#endif



#define FINAL_BLOCK_MARKER_SZ "talibkweli"


#define MAX_BLOCKS 2048
int blocks = 1;
/* Circular increment of a buffer index */
#define INC(i) (((i)+1) == blocks ? 0 : ((i)+1))

/* Max amount of shared memory you can allocate - can't see a way to look
 * this up.
 */
#ifndef DEF_SHMEM
#define DEF_SHMEM (1 K K)
#endif
int max_shmem = DEF_SHMEM;

/* Just a flag to show unfilled */
#define NONE (-1)

/* the shared memory id of the buffer */
int buffer_id = NONE;
struct block {
	int bytes;
	char *data;
} *curr_block;

#define NO_BUFFER ((struct buffer *)-1)
struct buffer {
	/* Id of the semaphore group */
	int semid;

	/* writer will hang trying to lock this till reader fills in a block */
	int blocks_used_lock;
	/* reader will hang trying to lock this till writer empties a block */
	int blocks_free_lock;

	int next_block_in;
	int next_block_out;

	struct block block[ MAX_BLOCKS ];

	/* These actual space for the blocks is here - the array extends
	 * pass 1 */
	char data_space[ 1 ];
} *pbuffer = NO_BUFFER;
int buffer_size;

int fdin	= 0;
int fdout	= 1;
int in_ISCHR	= 0;
int out_ISCHR	= 0;
int padblock	= FALSE;
int writer_pid	= 0;
int reader_pid	= 0;
int free_shm	= 1;
int percent	= 0;
int debug	= 0;
int Zflag	= 0;
int writer_status = 0;
char *progname = "buffer";
void byee(int);
void start_reader_and_writer();
void parse_args();
void set_handlers();
void buffer_allocate();
void report_proc();
int do_size(char*); 
void get_buffer();
void reader();
void writer();
void writer_end();
void wait_for_writer_end();
void get_next_free_block();
void test_writer();
int fill_block();
void get_next_filled_block();
int data_to_write();
void write_blocks_to_stdout(int, int);
void write_block_to_stdout();
void pr_out();
void end_writer();

long blocksize = DEF_BLOCKSIZE;

/* Which process... in error reports*/
char *proc_string = "buffer";

/* Numbers of blocks in the queue. 
 */

/* When showing print a note every this many bytes writen */
int showevery = 0;

/* Pause after every write */
unsigned write_pause;

char print_total = 0;
/* Number of K output */
unsigned long outk = 0;


/* Allocate new semaphores */
int new_sems();

/* Perform actions on semaphores */
void sem_set();
void lock();
void unlock();
void remove_sems();




extern long long g_tape_posK;
extern void set_signals(bool);
extern int g_current_media_number;


int configure_BUFFER_for_mondo(char*incoming_device, char*outgoing_device, bool writing_to_tape);
bool g_end_of_tape_reached=FALSE;





/*************************************************************************
 * wait_until_process_done() -- Hugo Rabson                              *
 *                                                                       *
 * Purpose:   Loop until the specified process has finished, then return.*
 * Called by: mastermind_the_buffer_stuff()                              *
 * Params:    pid               process ID to watch                      *
 * Returns:   none                                                       *
 *************************************************************************/
void wait_until_process_done(pid_t pid)
{
  int stat_val;
  pid_t child_pid;

//  open_evalcall_form("Waiting for buffer to empty");
  child_pid = wait(&stat_val);
//  close_evalcall_form();
  return;  

/*
  open_evalcall_form("Waiting for buffer to empty");
  while(1)
    {
      sleep(1);
      child_pid = waitpid(pid, &stat_val, WNOHANG | WUNTRACED);
      if (g_end_of_tape_reached)
        {log_it("End of tape reached. Starting a new one.\n"); g_end_of_tape_reached = FALSE; sleep(1); }
          if (WIFEXITED(stat_val) || WIFSTOPPED(stat_val))
        {log_it("Child exited with code %d\n", WEXITSTATUS(stat_val)); break; }
      update_evalcall_form(0);
    }
  close_evalcall_form();
*/
}






char *resolve_softlinks_to_get_to_actual_device_file(char*incoming)
{
  static char output[MAX_STR_LEN];
  char command[MAX_STR_LEN*2];

  if (!does_file_exist(incoming))
    {
      log_it("resolve_softlinks_to_get_to_actual_device_file --- device not found");
      strcpy(output, incoming);
    }
  else
    {
      sprintf(command, "i=%s; while [ -h $i ] ; do r=`dirname $i`; s=`file $i | gawk '{print $NF;}'`; j=$r/$s; [ -e $j ] && i=$j || i=$s ; done ; echo $i", incoming);
      strcpy(output, call_program_and_get_last_line_of_output(command));
      log_it("resolved %s to %s", incoming, output);
    }
  return(output);
}



pid_t g_mastermind_pid=0;
extern FILE*g_tape_stream;




int is_incoming_block_valid( char*buffer, long internal_block_size, int rotor )
{
  int tmp_int_A, tmp_int_B;

  memcpy((void*)&tmp_int_A, (void*)(buffer+2), 2);
  memcpy((void*)&tmp_int_B, (void*)(buffer+internal_block_size-2), 2);
  tmp_int_A = tmp_int_A & 0xFFFF;
  tmp_int_B = tmp_int_B & 0xFFFF;

  //  tmp_int_B = tmp_int_A; // FIXME
  if (buffer[0] != 0x11 || buffer[1] != 0x22 || buffer[internal_block_size-4] != 0x11 || buffer[internal_block_size-3] != 0x33)
    {
      log_it("Warning - expecting block#%d but got flotsam ('%x %x %x %x')", rotor, buffer[0], buffer[1], buffer[internal_block_size-4], buffer[internal_block_size-3]);
      return(-1);
    }
  else if (tmp_int_A != rotor || tmp_int_B != rotor)
    {
      if (tmp_int_A != tmp_int_B)
        {
	  log_it("Rotors %d and %d found but I was expecting %d", tmp_int_A, tmp_int_B, rotor);
	}
      else
        {
          log_it("Good rotors, up to %d; now, %d has a problem", rotor-1, rotor);
          if (tmp_int_A == rotor-1)
            {
//              log_it("_Not_ writing block %d again (this one's a duplicate)", tmp_int_A);
              return(99);
            }
          log_it("WARNING! Out-of-sequence rotor (last was %d; this is %d)", rotor, tmp_int_A);
          return(2);
        }
    }
  return(1);
}





FILE*pipe_open_file(char*fname, char*direction)
{
  char command[MAX_STR_LEN];
  if (strchr(direction,'w'))
    {
      sprintf(command, "dd bs=%ld of=%s 2> /dev/null", TAPE_BLOCK_SIZE, fname);
    }
  else if (strchr(direction,'r'))
    {
      sprintf(command, "dd bs=%ld if=%s 2> /dev/null", TAPE_BLOCK_SIZE, fname);
    }
  else
    {
      fatal_error("pipe_open_file -- unknown direction");
    }
  return(popen(command, direction));
}




long try_hard_to_fread (char*buffer, int s, long bufsize, FILE*fin)
{
  long length;
  int retries, e;

  for (length=retries=0; retries<5 && length<bufsize; retries++)
    {
      length += fread
	(buffer + length, s, bufsize - length, fin);
      e = errno;
      if (e && e!=ECHILD)
        {
          log_it("line %d: Error reported while reading: %s", __LINE__, strerror(e));
          if (length==bufsize)
            { log_it("...but at least I managed to read the whole block. :-)"); }
          if (e == EPIPE)
            { log_it("Error is serious. I'll not retry. Returning -1."); return(-1); }
        }
    }
  if (retries>1)
    {
      if (length==bufsize)
	{ 
	  log_it("It took %d tries to read the whole of block",
		 retries); 
	}
      else
	{
	  log_it("Even after %d tries, I couldn't read all of blk; only %ld bytes of a desired %ld",
		 retries, length, bufsize);
	}
    }
  return(length);
}




long try_hard_to_fwrite (char*buffer, int s, long bufsize, FILE*fout)
{
  long length;
  int retries, e;

  for (length=retries=0; retries<5 && length<bufsize; retries++)
    {
      length += fwrite
	(buffer + length, s, bufsize - length, fout);
      e = errno;
      if (e && e!=ECHILD)
        {
          log_it("line %d: Error reported while writing: %s", __LINE__, strerror(e));
          if (length==bufsize)
            { log_it("...but at least I managed to write the whole block. :-)"); }
          if (e == EPIPE)
            { log_it("Error is serious. I'll not retry. Returning -1."); return(-1); }
        }
    }
  if (retries>1)
    {
      if (length==bufsize)
	{ 
	  log_it("It took %d tries to write the whole of block",
		 retries); 
	}
      else
	{
	  log_it("Even after %d tries, I couldn't write all of blk; only %ld bytes of a desired %ld",
		 retries, length, bufsize);
	}
    }
  return(length);
}



/*************************************************************************
 * copy_file_from_here_to_there() -- Hugo Rabson                         *
 *                                                                       *
 * Purpose:   Cat file/device A to file/device B, 256K at a time         *
 * Called by: wrap_FIFO_around_device()                                  *
 * Params:    src              source file/device                        *
 *            dest             destination file/device                   *
 *            writing_to_priv  for handling multiple tapes; if TRUE then *
 *                             pause, request next tape, and reopen the  *
 *                             output; if FALSE then [...] the _input_   *
 * Returns:   none                                                       *
 * NB:        uses global var g_end_of_tape_reached sometimes            *
 *************************************************************************/
void copy_file_from_here_to_there(char *src, char *dest, bool writing_to_private_device)
{
  int stat_val, i;
  pid_t returned_pid;
// int i;
  bool really_stop = FALSE;
  FILE*fin, *fout;
  long length, templong;
  char*buffer;
  long long bytes_written=0;
  long written;
  int rotor;
  long internal_block_size = INTERNAL_TAPE_BLK_SIZE;
  //  long internal_block_size = 2048;

  buffer = (char*)malloc(internal_block_size);
  if (!(fin = fopen(src, "r"))) { log_it( "Can't openin %s\n", src); exit(1); }
  if (!(fout = fopen(dest, "w"))) { log_it("Can't openout %s\n", dest); exit(1); }
  sleep(3);

// first 32MB will be copied as-is, to allow the data disks to be put at the start of the tape

  for(templong=0; templong<8; templong++) { buffer[templong]=buffer[internal_block_size-1-templong]=0; }
  for(bytes_written = 0; bytes_written < 32L*1024*1024; bytes_written+=written)
    {
      length = try_hard_to_fread(buffer, 1, internal_block_size, fin);
      written = try_hard_to_fwrite(buffer, 1, length, fout);
    }

  for(rotor=1; !really_stop; rotor=(rotor>=30000)?1:rotor+1)
    {
/* read block */
      if (writing_to_private_device)
/* archiving to tape */
        {
          if (feof(fin))
            {
              log_it("ALL ARCHIVES HAVE BEEN READ IN AND WRITTEN (last block: #%d)", rotor-1);
              buffer[0] = buffer[internal_block_size-4] = 0x11;
              buffer[1] = 0x66; buffer[internal_block_size-3] = 0x77;
              strcpy(buffer+2, FINAL_BLOCK_MARKER_SZ);
              for(templong=0; templong<(1024L*1024 / internal_block_size); templong++)
                {
                  if (try_hard_to_fwrite(buffer, 1, internal_block_size, fout) != internal_block_size)
                    { log_it("Warning - failed to write FINAL_BLOCK_MARKER block to end of stream"); }
                }
              really_stop=TRUE; break;
            }
          /* archiving to tape */
          length = try_hard_to_fread(buffer+4, 1, internal_block_size-8, fin);
          buffer[0] = buffer[internal_block_size-4] = 0x11;
          buffer[1] = 0x22; buffer[internal_block_size-3] = 0x33;
          memcpy((void*)(buffer+2), (void*)&rotor, 2);
          memcpy((void*)(buffer+internal_block_size-2), (void*)&rotor, 2);
          if (length <= 0 && feof(fin)) { log_it("(cffhth) Failed to read block in, while archiving to tape"); continue; }
          length += 8;
        }
      else
        {
          length = try_hard_to_fread(buffer, 1, internal_block_size, fin);
          if (length < internal_block_size || !is_incoming_block_valid(buffer, internal_block_size, rotor))
            {
	      if (length==internal_block_size) { log_it("Bad block, man."); }
              system("sync");
              sleep(8);
              returned_pid = waitpid(g_mastermind_pid, &stat_val, WNOHANG | WUNTRACED);
              if (!g_tape_stream)
                {
                  log_it("g_tape_stream is closed. So, I think this is the end of _all_ tape volumes. Set over.");
                  really_stop = TRUE;
                  break;
                }
              g_end_of_tape_reached = TRUE; 
              log_it("End of tape reached (CFFHTT) - rotor=%d", rotor);
	      log_it("Skipping rotor %d because it probably ended prematurely.", rotor);
	      rotor--;
	      length=0;
            }
          else
            {
              i = is_incoming_block_valid(buffer, internal_block_size, rotor);
              if (i==99)
                {
                  log_it("Good blocks thru %d. However, %d is a duplicate. Skipping it.", rotor-1, rotor);
//                  log_it("Skipping this duplicate block and continuing to read.");
//                  length = 0;
//                  rotor--;
		  //                  log_it("OK, I think I skipped the bad (former) copy & this one (%d) is the _good_ copy.", rotor);
                }
              else if (i!=0)
                {
//                  log_it("Good block (rotor=%d)", rotor);
                }
              else
                {
                  log_it("Good blocks until %d, which is bad / too short / something.", rotor);
                  length = 0;
                  g_end_of_tape_reached = TRUE;
                  log_it("Bad incoming block. End of tape, methinks.");
                }
            }
          length -= 8;
          if (length<0) { length=0; }
        }
/* OK, now we'll write the block */
      if (length>0) { written = try_hard_to_fwrite(buffer+(writing_to_private_device?0:4), 1, length, fout); }
      else { written = 0; }
      if (written < length) { log_it("Can't write whole block to output (rotor %d)", rotor); g_end_of_tape_reached = TRUE; }
//      else {log_it("Written block %d OK", rotor);}
/* Handle end-of-tape scenario */
      if (g_end_of_tape_reached)
        {
          log_it("END OF TAPE REACHED");
          sleep(6);
          returned_pid = waitpid(g_mastermind_pid, &stat_val, WNOHANG | WUNTRACED);
          if (WIFEXITED(stat_val) || WIFSTOPPED(stat_val))
            { log_it("Buffer has ended, so I'm (here_to_there) closing, too."); really_stop = TRUE; break; }
          if (!g_tape_stream && !writing_to_private_device)
            { log_it("Ach, goo awee! Tha teep is oover! OOVER@"); really_stop = TRUE; break; }
          log_it("Pausing; telling parent to change tapes");
          if (writing_to_private_device) { fclose(fout); } else { /*while(!feof(fin)) { fread(buffer, 1, internal_block_size, fin); } ; */ fclose(fin); }
          insist_on_this_tape_number(g_current_media_number+1); // will increment it, too

//          g_tape_posK = 0;
          g_end_of_tape_reached = FALSE;
//          while(g_end_of_tape_reached) { sleep(5); log_it("Still waiting for Enter to be pushed"); }

          log_it("Thankyou. Continuing...");
          sleep(2);
          if (writing_to_private_device)
            {
              log_it("Will re-try to write rotor %d", rotor);
              if (!(fout = fopen(dest, "w"))) { log_it( "Can't reopenout %s", dest); finish(1); }
              written = try_hard_to_fwrite(buffer+(writing_to_private_device?0:4), 1, length, fout);
            }
          else
            {
              if (!(fin = fopen(src, "r"))) { log_it( "Can't reopenin %s", src); finish(1); }
	      //              log_it("Skipping rotor %d (it ended prematurely, probably because the tape ended)", rotor);
            }
        }
      bytes_written += written;
    }
  log_it("Copied file from %s to %s; %lld bytes written", src, dest, bytes_written/1024);
  fclose(fin);


//  log_it("(here_to_there) fin closed");

/*
	  if (!writing_to_private_device)
	    {
	      for(i=0; i<internal_block_size; i++) { buffer[i]=0; }
	      fwrite(buffer, 1, internal_block_size, fout);
	      fwrite(buffer, 1, internal_block_size, fout);
	      fwrite(buffer, 1, internal_block_size, fout);
	      fwrite(buffer, 1, internal_block_size, fout);
	    }
*/

  fclose(fout);
//  log_it("(here_to_there) fout closed");
  free(buffer);
}




/*************************************************************************
 * wrap_FIFO_around_device() -- Hugo Rabson                              *
 *                                                                       *
 * Purpose:   Copy all data from the output of 'buffer' to the relevant  *
 *            device (public/private, depending on archive_to_priv_dev)  *
 *            another place, to help 'buffer' to do its job.             *
 * Called by: start_buffer_processes()                                   *
 * Params:    incmg            ptr to structure 's_wrapinfo' containing:-*
 *            int_buf_IN_fifo  'buffer' reads from this dev and writes...*
 *            int_buf_OUT_fifo ...to this device, in the background      *
 *            public_device    the published device for subroutines to   *
 *                             interact with                             *
 *            private_device   the actual device, e.g. /dev/st0          *
 *            archive_to_pri.. writing from public device to private dev,*
 *                             then TRUE; else, FALSE (priv-->pub dev)   *
 * Returns:   none                                                       *
 * NB:        Filenames must already have been populated in struct.      *
 *************************************************************************/
void* wrap_FIFO_around_device (void*incmg)
{
  struct s_wrapfifo inforec;
  char src[MAX_STR_LEN], dest[MAX_STR_LEN];
//  char command[MAX_STR_LEN*2];
  bool writing_to_private_device;
  int i;

  memcpy( &inforec, (struct s_wrapfifo*)incmg, sizeof(struct s_wrapfifo));
  writing_to_private_device = inforec.writing_to_private_device;
  if (writing_to_private_device)
    { strcpy(src, inforec.internal_buffer_OUT_fifo); strcpy(dest, inforec.private_device); }
  else
    { strcpy(src, inforec.private_device); strcpy(dest, inforec.internal_buffer_IN_fifo); }
  log_it("Creating copierdaemon (%s to %s)", src, dest);


/*
  sprintf(command, "cat %s | dd bs=%ld > %s", src, dest, TAPE_BLOCK_SIZE);
  system(command);
*/


  copy_file_from_here_to_there(src, dest, writing_to_private_device);
  log_to_screen("Waiting for tape streamer to close");
  for(i=0; i<150 && g_tape_stream; i+=5) { sleep(1); }
  if (g_tape_stream)
    {
      log_it("I (wrap_around_FIFO...) have waited 30 seconds for g_tape_stream to close.");
      log_it("It hasn't. I suspect that all data _has_ been read to / written from tape.");
      log_it("I also suspect that there is a 'bzip2 -dc' hanging around somewhere because");
      log_it("'buffer' is being an ass. I'll keep waiting...");
      while(g_tape_stream) { sleep(1); }
    }
  else
    {
      log_it("(wrap_around_FIFO...)");
      log_to_screen("Tape streamer has closed.");
    }
  close_evalcall_form();
  pthread_exit(NULL);
}







pthread_t g_buffer_thread, g_copier_thread;







/*************************************************************************
 * mastermind_the_buffer_stuff() -- Hugo Rabson                          *
 *                                                                       *
 * Purpose:   Fork a process to buffer data between two devices. Return  *
 *            after forking. Forked process will run & then exit, itself.*
 * Called by: start_buffer_process()                                     *
 * Params:    incmg            ptr to structure 's_wrapinfo' containing:-*
 *            int_buf_IN_fifo  'buffer' reads from this dev and writes...*
 *            int_buf_OUT_fifo ...to this device, in the background      *
 *            public_device    the published device for subroutines to   *
 *                             interact with                             *
 *            private_device   the actual device, e.g. /dev/st0          *
 *            archive_to_pri.. writing from public device to private dev,*
 *                             then TRUE; else, FALSE (priv-->pub dev)   *
 * Returns:   none                                                       *
 * NB:        Filenames must already have been populated in struct.      *
 *************************************************************************/
void *mastermind_the_buffer_stuff(void*incmg)
{
  struct s_wrapfifo inforec;

//  log_it("(mastermind_the_buffer_stuff) in=%s, out=%s", inforec.internal_buffer_IN_fifo, inforec.internal_buffer_OUT_fifo );
  memcpy( &inforec, (struct s_wrapfifo*)incmg, sizeof(struct s_wrapfifo));
  g_mastermind_pid = fork();
  signal(SIGPIPE, SIG_IGN);
  switch( g_mastermind_pid )
    {
      case -1: fatal_error("Forking error");
      case 0: // child
	  log_it("Configuring buffer (%s to %s)", inforec.internal_buffer_IN_fifo, inforec.internal_buffer_OUT_fifo );
	  configure_BUFFER_for_mondo( inforec.internal_buffer_IN_fifo, inforec.internal_buffer_OUT_fifo, inforec.writing_to_private_device );
	  set_handlers();
	  buffer_allocate();
	  start_reader_and_writer();
	  log_it("Buffer thread is closing");
	  byee( 0 ); // calls exit()
      default: // parent
	  wait_until_process_done(g_mastermind_pid);
	  log_it("Buffer's PID has stopped");
    }
  pthread_exit(NULL);
}







/*************************************************************************
 * start_buffer_process() -- Hugo Rabson                                 *
 *                                                                       *
 * Purpose:   Fork processes to buffer data between the public device    *
 *            (the one with which the rest of the program will interact) *
 *            and the privaet device (the actual tape streamer). Exit    *
 *            after forking. Forked process will run & then exit, itself.*
 * Called by: openin_tape(); openout_tape()                              *
 * Params:    public_device    the published device for subroutines to   *
 *                             interact with                             *
 *            private_device   the actual device, e.g. /dev/st0          *
 *            archive_to_pri.. writing from public device to private dev,*
 *                             then TRUE; else, FALSE (priv-->pub dev)   *
 * Returns:   none                                                       *
 * NB:        Filenames must already have been populated in struct.      *
 *************************************************************************/
void start_buffer_process(char*pub_device, char*priv_device, bool archive_to_private_device)
{
  int res;
  struct s_wrapfifo copier_inforec, buffer_inforec, inforec;
  char temporary_device[MAX_STR_LEN];
//  char tmp[MAX_STR_LEN];

/*
   If archiving:-
    - data flows from public_device to internal_buffer_IN_fifo
      to internal_buffer_OUT_fifo to private_device
    - strictly, internal_buffer_IN_fifo is the same filename
      as public_device, to save CPU overhead
   If restoring:-
    - data flows from private_device to internal_buffer_IN_fifo
      to internal_buffer_OUT_fifo to publi_device
    - strictly, internal_buffer_OUT_fifo is the same filename
      as public_device, to save CPU overhead
*/

//  log_it("start_buffer_process() --- public device = %s; private device = %s", pub_device, priv_device);
  system("rm -f /tmp/mondorescue-fifo.*");
  inforec.writing_to_private_device = archive_to_private_device;
  strcpy(inforec.public_device, pub_device);
  strcpy(inforec.private_device, priv_device);
  system("rm -Rf /tmp/mondorescue-fifo.*");
  make_fifo( temporary_device, "/tmp/mondorescue-fifo.");
  if (inforec.writing_to_private_device)
    {
      strcpy(inforec.internal_buffer_IN_fifo, inforec.public_device);
      strcpy(inforec.internal_buffer_OUT_fifo, temporary_device);
    }
  else
    {
      strcpy(inforec.internal_buffer_IN_fifo, temporary_device);
      strcpy(inforec.internal_buffer_OUT_fifo, inforec.public_device);
    }
  strcpy( inforec.private_device, resolve_softlinks_to_get_to_actual_device_file( inforec.private_device ) );

//  log_it("public_device = %s", inforec.public_device);
//  log_it("intnl_buf_IN  = %s", inforec.internal_buffer_IN_fifo);
//  log_it("intnl_buf_OUT = %s", inforec.internal_buffer_OUT_fifo);
//  log_it("private_device= %s", inforec.private_device);

  memcpy( (void*) &copier_inforec, (void*) &inforec, sizeof(struct s_wrapfifo) );
  memcpy( (void*) &buffer_inforec, (void*) &inforec, sizeof(struct s_wrapfifo) );

  if ((res = pthread_create(&g_copier_thread, NULL, wrap_FIFO_around_device, (void*)(&copier_inforec)))) { fatal_error( "Unable to start buffer (copying) process in background - thread creation failure" ); }
  if ((res = pthread_create(&g_buffer_thread, NULL, mastermind_the_buffer_stuff, (void*)(&buffer_inforec)))) { fatal_error( "Unable to start buffer (buffering) process in background - thread creation failure" ); }

  sleep(2); // FIXME - lazy! Use locking, semaphores, _something_ to wait until the pthreads have done their job
//  set_signals(FALSE);

  log_it("Buffer has been STARTED.");
}




void stop_buffer_process(bool writing_to_private_device)
{
  void*vp;
  void**pvp;
//  int res;
//  char tmp[MAX_STR_LEN+1], *p;
  char result_str[MAX_STR_LEN+1];

//  log_it("(stop_buffer_process) --- entering");

  pvp=&vp;
  vp=(void*)result_str;

  log_it("Waiting for pthread_join()'s to take effect");
  if (pthread_join(g_copier_thread, pvp)) { fatal_error( "pthread_join(g_copier_thread) failed"); }
  log_it("Copier pthread has returned. Great.");
  if (pthread_join(g_buffer_thread, pvp)) { fatal_error( "pthread_join(g_buffer_thread) failed"); }
  log_it("Buffer pthread has returned. Great.");
  sleep(2);
  log_it("Buffer has been STOPPED");
//  log_it("(stop_buffer_process) --- leaving");

}





/*
int main(int argc, char*argv[])
{
  if (argc!=3)
    {
      log_it( "./fifobuf <input> <output>\n");
      exit(1);
    }
  configure_BUFFER_for_mondo(argv[1], argv[2]);
  set_handlers();
  buffer_allocate();
  start_reader_and_writer();
  byee( 0 );
  // NOTREACHED
  exit( 0 );
}
*/




/* Hugo Rabson */
char g_incoming_dev[MAX_STR_LEN], g_outgoing_dev[MAX_STR_LEN];
int configure_BUFFER_for_mondo(char*incoming_device, char*outgoing_device, bool writing_to_tape)
{
//	extern char *optarg;
//	char blocks_given = '\0';
	struct stat buf;

	strcpy(g_incoming_dev, incoming_device);
	strcpy(g_outgoing_dev, outgoing_device);
	max_shmem = 12*1024*1024;
	write_pause = 0; // microseconds
	padblock = TRUE; // pad last block
	Zflag = TRUE;
	if( (fdin = open( g_incoming_dev, O_RDONLY )) < 0 ){
		perror( "cannot open input file" );
		log_it( "filename: %s\n", optarg );
		byee ( -1 );
		}
	if( (fdout = open( g_outgoing_dev, O_WRONLY | O_CREAT | O_TRUNC, 0666 )) < 0 ){
		perror( "cannot open output file" );
		log_it( "filename: %s\n", optarg );
		byee ( -1 );
		}
	percent = (writing_to_tape) ? 75 : 10; // wait until buffers are N% full before writing
	blocksize = INTERNAL_TAPE_BLK_SIZE/2;
	blocks = (max_shmem - sizeof( struct buffer )) / blocksize;
	if( blocks <= 0 ){
		log_it( "Cannot handle blocks that big, aborting!\n" );
		byee( -1 );
		}
	if( MAX_BLOCKS < blocks  ){
		log_it( "Cannot handle that many blocks, aborting!\n" );
		byee( -1 );
		}
	/* check if fdin or fdout are character special files */
	if( fstat( fdin, &buf ) != 0 ){
		report_proc();
		perror( "can't stat input file" );
		byee( -1 );
	}
	in_ISCHR = S_ISCHR( buf.st_mode );
	if( fstat( fdout, &buf ) != 0 ){
		report_proc();
		perror( "can't stat output file" );
		byee( -1 );
	}
	out_ISCHR = S_ISCHR( buf.st_mode );
	return(0);
}












#if defined(SYS5) || defined(ultrix) || defined(_AIX) || defined _SEM_SEMUN_UNDEFINED
union semun {
	int val;
	struct semid_ds *buf;
	ushort *array;
};
#endif

/* IMPORTS */

/* Used to print error messages */
extern void report_proc();

/* Used to end the program - on error */
//extern void byee();



/* Set a semaphore to a particular value - meant to be used before
 * first lock/unlock */
void
sem_set( sem_id, semn, val )
	int sem_id;
	int semn;
	int val;
{
	union semun arg;
	extern int errno;

	arg.val = val;

	errno = 0;
	semctl( sem_id, semn, SETVAL, arg );
	if( errno != 0 ){
		report_proc();
		perror( "internal error, sem_set" );
		byee( -1 );
	}
}
	
int
new_sems( nsems )
	int nsems;
{
	int sem;
	int i;

	sem = semget( IPC_PRIVATE, nsems, IPC_CREAT|S_IREAD|S_IWRITE );
	if( sem < 0 ){
		report_proc();
		perror( "internal error, couldn't create semaphore" );
		byee( -1 );
	}
	
	for( i = 0; i < nsems; i++ ){
		sem_set( sem, i, 1 );
	}

	return sem;
}

static int
do_sem( sem_id, pbuf, err )
	int sem_id;
	struct sembuf *pbuf;
	char *err;
{
	/* This just keeps us going in case of EINTR */
	while( 1 ){
		if( semop( sem_id, pbuf, 1 ) == -1 ){
			if( errno == EINTR ){
				continue;
			}
			report_proc();
			log_it( "internal error pid %d, lock id %d\n",
				getpid(), sem_id );
			perror( err );
			byee( -1 );
		}
		return(0);
	}
}

void
lock( sem_id, semn )
	int sem_id;
	int semn;
{
	struct sembuf sembuf;

	sembuf.sem_num = semn;
	sembuf.sem_op = -1;
	sembuf.sem_flg = 0;

	do_sem( sem_id, &sembuf, "lock error" );
}

void
unlock( sem_id, semn )
	int sem_id;
	int semn;
{
	struct sembuf sembuf;

	sembuf.sem_num = semn;
	sembuf.sem_op = 1;
	sembuf.sem_flg = 0;

	do_sem( sem_id, &sembuf, "unlock error" );
}

void
remove_sems( sem_id )
	int sem_id;
{
	if( sem_id == -1 )
		return;

	if( semctl( sem_id, 0, IPC_RMID, /*(union semun)*/0 ) == -1 ){
		report_proc();
		perror( "internal error, failed to remove semaphore" );
	}
}

/* Some forward declarations */


/*
int
main_orig( argc, argv ) // was main() but Hugo hacked it --- 2002/08/12
	int argc;
	char **argv;
{
	parse_args( argc, argv );

	set_handlers();

	buffer_allocate();

	start_reader_and_writer();

	byee( 0 );

	// NOTREACHED
	exit( 0 );
}
*/



void
parse_args( argc, argv )
	int argc;
	char **argv;
{
	int c;
	int iflag = 0;
	int oflag = 0;
	int zflag = 0;
//	extern char *optarg;
	char blocks_given = FALSE;
	struct stat buf;


	while( (c = getopt( argc, argv, "BS:Zdm:s:b:p:u:ti:o:z:" )) != -1 ){
		switch( c ){
		case 't': /* Print to stderr the total no of bytes writen */
			print_total++;
			break;
		case 'u': /* pause after write for given microseconds */
			write_pause = atoi( optarg );
			break;
		case 'B':   /* Pad last block */
			padblock = TRUE;
			break;
		case 'Z':   /* Zero by lseek on the tape device */
			Zflag = TRUE;
			break;
		case 'i': /* Input file */
			iflag++;
			if( iflag > 1 ){
				report_proc();
				log_it( "-i given twice\n" );
				byee( -1 );
			}
			if( (fdin = open( optarg, O_RDONLY )) < 0 ){
				report_proc();
				perror( "cannot open input file" );
				log_it( "filename: %s\n", optarg );
				byee ( -1 );
			}
			break;
		case 'o': /* Output file */
			oflag++;
			if( oflag > 1 ){
				report_proc();
				log_it( "-o given twice\n" );
				byee( -1 );
			}
			if( (fdout = open( optarg, O_WRONLY | O_CREAT | O_TRUNC, 0666 )) < 0 ){
				report_proc();
				perror( "cannot open output file" );
				log_it( "filename: %s\n", optarg );
				byee ( -1 );
			}
			break;
		case 'S':
			/* Show every once in a while how much is printed */
			showevery = do_size( optarg );
			if( showevery <= 0 )
				showevery = PRINT_EVERY;
			break;
		case 'd':	/* debug */
			debug++;
			if( debug == 1 ){
				setbuf( stdout, NULL );
				setbuf( stderr, NULL );
				log_it( "debugging turned on\n" );
			}
			break;
		case 'm':
			/* Max size of shared memory lump */
			max_shmem = do_size( optarg );

			if( max_shmem < (sizeof( struct buffer ) + (blocksize * blocks)) ){
				log_it( "max_shmem %d too low\n", max_shmem );
				byee( -1 );
			}
			break;
		case 'b':
			/* Number of blocks */
			blocks_given = TRUE;
			blocks = atoi( optarg );
			if( (blocks <= 0) || (MAX_BLOCKS < blocks) ){
				log_it( "blocks %d out of range\n", blocks );
				byee( -1 );
			}
			break;
		case 'p':	/* percent to wait before dumping */
			percent = atoi( optarg );

			if( (percent < 0) || (100 < percent) ){
				log_it( "percent %d out of range\n", percent );
				byee( -1 );
			}
			if( debug )
				log_it( "percent set to %d\n", percent );
			break;
		case 'z':
			zflag++;
			/* FALL THRU */
		case 's':	/* Size of a block */
			blocksize = do_size( optarg );

			if( (blocksize <= 0) || (MAX_BLOCKSIZE < blocksize) ){
				log_it( "blocksize %ld out of range\n", blocksize );
				byee( -1 );
			}
			break;
		default:
			log_it( "Usage: %s [-B] [-t] [-S size] [-m memsize] [-b blocks] [-p percent] [-s blocksize] [-u pause] [-i infile] [-o outfile] [-z size]\n",
				progname );
			log_it( "-B = blocked device - pad out last block\n" );
			log_it( "-t = show total amount writen at end\n" );
			log_it( "-S size = show amount writen every size bytes\n" );
			log_it( "-m size = size of shared mem chunk to grab\n" );
			log_it( "-b num = number of blocks in queue\n" );
			log_it( "-p percent = don't start writing until percent blocks filled\n" );
			log_it( "-s size = size of a block\n" );
			log_it( "-u usecs = microseconds to sleep after each write\n" );
			log_it( "-i infile = file to read from\n" );
			log_it( "-o outfile = file to write to\n" );
			log_it( "-z size = combined -S/-s flag\n" );
			byee( -1 );
		}
	}

	if (zflag) showevery = blocksize;

	/* If -b was not given try and work out the max buffer size */
	if( !blocks_given ){
		blocks = (max_shmem - sizeof( struct buffer )) / blocksize;
		if( blocks <= 0 ){
			log_it( "Cannot handle blocks that big, aborting!\n" );
			byee( -1 );
		}
		if( MAX_BLOCKS < blocks  ){
			log_it( "Cannot handle that many blocks, aborting!\n" );
			byee( -1 );
		}
	}

	/* check if fdin or fdout are character special files */
	if( fstat( fdin, &buf ) != 0 ){
		report_proc();
		perror( "can't stat input file" );
		byee( -1 );
	}
	in_ISCHR = S_ISCHR( buf.st_mode );
	if( fstat( fdout, &buf ) != 0 ){
		report_proc();
		perror( "can't stat output file" );
		byee( -1 );
	}
	out_ISCHR = S_ISCHR( buf.st_mode );
}

/* The interrupt handler */
void
shutdown(int sig)
{
	static int shutting;
	if( shutting ){
		if( debug )
			log_it( "%s: ALREADY SHUTTING!\n", proc_string );
		return;
	}
	shutting = 1;
	if( debug )
		log_it( "%s: shutdown on signal\n", proc_string );

	byee( -1 );
}

/* Shutdown because the child has ended */
void
child_shutdown(int sig)
{
	/* Find out which child has died.  (They may not be my
	 * children if buffer was exec'd on top of something that had
	 * childred.)
	 */
	int deadpid;

	while( (deadpid = waitpid( -1, &writer_status, WNOHANG | WUNTRACED)) &&
		deadpid != -1 && deadpid != 0 ){
		if( debug > 2 )
			log_it( "child_shutdown %d: 0x%04x\n", deadpid, writer_status );
		if( deadpid == writer_pid ){
			if( debug > 2 )
				log_it( "writer has ended\n" );
			writer_pid = 0;
			byee( 0 );
		}
	}
}

void
set_handlers()
{
	if( debug )
		log_it( "%s: setting handlers\n", proc_string );

	signal( SIGHUP, shutdown );
	signal( SIGINT, shutdown );
	signal( SIGQUIT, shutdown );
	signal( SIGTERM, shutdown );
#ifdef SIGCHLD
	signal( SIGCHLD, child_shutdown );
#else
#ifdef SIGCLD
	signal( SIGCLD, child_shutdown );
#endif
#endif
}

void
buffer_allocate()
{
	/* Allow for the data space */
	buffer_size = sizeof( struct buffer ) +
		((blocks * blocksize) - sizeof( char ));

	/* Create the space for the buffer */
	buffer_id = shmget( IPC_PRIVATE,
			   buffer_size,
			   IPC_CREAT|S_IREAD|S_IWRITE );
	if( buffer_id < 0 ){
		report_proc();
		perror( "couldn't create shared memory segment" );
		byee( -1 );
	}

	get_buffer();

#ifdef SYS5
	memset( (char *)pbuffer, '\0', buffer_size );
#else
	bzero( (char *)pbuffer, buffer_size );
#endif
	pbuffer->semid = -1;
	pbuffer->blocks_used_lock = -1;
	pbuffer->blocks_free_lock = -1;

	pbuffer->semid = new_sems( 2 );	/* Get a read and a write sem */
	pbuffer->blocks_used_lock = 0;
	/* Start it off locked - it is unlocked when a buffer gets filled in */
	lock( pbuffer->semid, pbuffer->blocks_used_lock );

	pbuffer->blocks_free_lock = 1;
	/* start this off so lock() can be called on it for each block
	 * till all the blocks are used up */
	sem_set( pbuffer->semid, pbuffer->blocks_free_lock, blocks - 1 );

	/* Detattach the shared memory so the fork doesnt do anything odd */
	shmdt( (char *)pbuffer );
	pbuffer = NO_BUFFER;
}

void
buffer_remove()
{
	static char removing = FALSE;

	/* Avoid accidental recursion */
	if( removing )
		return;
	removing = TRUE;

	/* Buffer not yet created */
	if( buffer_id == NONE )
		return;

	/* There should be a buffer so this must be after its detached it
	 * but before the fork picks it up */
	if( pbuffer == NO_BUFFER )
		get_buffer();

	if( debug )
		log_it( "%s: removing semaphores and buffer\n", proc_string );
	remove_sems( pbuffer->semid );
	
	if( shmctl( buffer_id, IPC_RMID, (struct shmid_ds *)0 ) == -1 ){
		report_proc();
		perror( "failed to remove shared memory buffer" );
	}
}

void
get_buffer()
{
	int b;

	/* Grab the buffer space */
	pbuffer = (struct buffer *)shmat( buffer_id, (char *)0, 0 );
	if( pbuffer == NO_BUFFER ){
		report_proc();
		perror( "failed to attach shared memory" );
		byee( -1 );
	}

	/* Setup the data space pointers */
	for( b = 0; b < blocks; b++ )
		pbuffer->block[ b ].data =
			&pbuffer->data_space[ b * blocksize ];

}

void
start_reader_and_writer()
{
	fflush( stdout );
	fflush( stderr );

	if( (writer_pid = fork()) == -1 ){
		report_proc();
		perror( "unable to fork" );
		byee( -1 );
	}
	else if( writer_pid == 0 ){
		free_shm = 0;
		proc_string = "buffer (writer)";
		reader_pid = getppid();

		/* Never trust fork() to propogate signals - reset them */
		set_handlers();

		writer();
	}
	else {
		proc_string = "buffer (reader)";
		reader();

		wait_for_writer_end();
	}
}

/* Read from stdin into the buffer */
void
reader()
{
	if( debug )
		log_it( "R: Entering reader\n" );

	get_buffer();

	while( 1 ){
		get_next_free_block();
		if( ! fill_block() )
			break;
	}

	if( debug )
		log_it( "R: Exiting reader\n" );
}

void
get_next_free_block()
{
	test_writer();

	/* Maybe wait till there is room in the buffer */
	lock( pbuffer->semid, pbuffer->blocks_free_lock );

	curr_block = &pbuffer->block[ pbuffer->next_block_in ];

	pbuffer->next_block_in = INC( pbuffer->next_block_in );
}

int
fill_block()
{
	int bytes;
	char *start;
	int toread;
	static char eof_reached = 0;
	
	if( eof_reached ){
		curr_block->bytes = 0;
		unlock( pbuffer->semid, pbuffer->blocks_used_lock );
		return 0;
	}

	start = curr_block->data;
	toread = blocksize;

	/* Fill the block with input.  This reblocks the input. */
	while( toread != 0 ){
		bytes = read( fdin, start, toread );
		if( bytes <= 0 ){
			/* catch interrupted system calls for death
			 * of children in pipeline */
			if( bytes < 0 && errno == EINTR )
				continue;
			break;
		}
		start += bytes;
		toread -= bytes;
	}

	if( bytes == 0 )
		eof_reached = 1;

	if( bytes < 0 ){
		report_proc();
		perror( "failed to read input" );
		byee( -1 );
	}

	/* number of bytes available. Zero will be taken as eof */
	if( !padblock || toread == blocksize )
		curr_block->bytes = blocksize - toread;
	else {
		if( toread ) bzero( start, toread );
		curr_block->bytes = blocksize;
	}

	if( debug > 1 )
		log_it( "R: got %d bytes\n", curr_block->bytes );

	unlock( pbuffer->semid, pbuffer->blocks_used_lock );

	return curr_block->bytes;
}

/* Write the buffer to stdout */
void
writer()
{
	int filled = 0;
	int maxfilled = (blocks * percent) / 100;
	int first_block;

	if( debug )
		log_it( "\tW: Entering writer\n blocks = %d\n maxfilled = %d\n",
			blocks,
			maxfilled );

	get_buffer();

	while( 1 ){
		if( !filled )
			first_block = pbuffer->next_block_out;
		get_next_filled_block();
		if( !data_to_write() )
			break;

		filled++;
		if( debug > 1 )
			log_it( "W: filled = %d\n", filled );
		if( filled >= maxfilled ){
			if( debug > 1 )
				log_it( "W: writing\n" );
			write_blocks_to_stdout( filled, first_block );
			filled = 0;
		}
	}

	write_blocks_to_stdout( filled, first_block );

	if( showevery ){
		pr_out();
		log_it( "\n" );
	}

	if( print_total ){
		log_it( "Kilobytes Out %lu\n", outk );
	}

	if( debug )
		log_it( "\tW: Exiting writer\n" );
}

void
get_next_filled_block()
{
	/* Hang till some data is available */
	lock( pbuffer->semid, pbuffer->blocks_used_lock );

	curr_block = &pbuffer->block[ pbuffer->next_block_out ];

	pbuffer->next_block_out = INC( pbuffer->next_block_out );
}

int
data_to_write()
{
	return curr_block->bytes;
}

void
write_blocks_to_stdout( filled, first_block )
	int filled;
	int first_block;
{
	pbuffer->next_block_out = first_block;

	while( filled-- ){
		curr_block = &pbuffer->block[ pbuffer->next_block_out ];
		pbuffer->next_block_out = INC( pbuffer->next_block_out );
		write_block_to_stdout();
	}
}

void
write_block_to_stdout()
{
	static unsigned long out = 0;
	static unsigned long last_gb = 0;
	static unsigned long next_k = 0;
	int written;

	if( next_k == 0 && showevery ){
		if( debug > 3 )
			log_it( "W: next_k = %lu showevery = %d\n", next_k, showevery );
		showevery = showevery / 1024;
		next_k = showevery;
	}
	if( (written = write( fdout, curr_block->data, curr_block->bytes )) != curr_block->bytes ){
		report_proc();
//		perror( "write of data failed" );
		log_it("(buffer) bytes to write=%d, bytes written=%d, total written %10luK\n", curr_block->bytes, written, outk );
log_it("Exiting here");
		byee( -1 );
	}

	if( write_pause ){
		usleep( write_pause );
	}

	out = curr_block->bytes / 1024;
	outk += out;
	last_gb += out;

	/*
	 * on character special devices (tapes), do an lseek() every 1 Gb,
	 * to overcome the 2Gb limit. This resets the file offset to
	 * zero, but -- at least on exabyte SCSI drives -- does not perform
	 * any actual action on the tape.
	 */
	if( Zflag && last_gb >= 1 K K ){
		last_gb = 0;
		if( in_ISCHR )
			(void) lseek( fdin, 0, SEEK_SET);
		if( out_ISCHR )
			(void) lseek( fdout, 0, SEEK_SET);
	}
	if( showevery ){
		if( debug > 3 )
			log_it( "W: outk = %lu, next_k = %lu\n",
				outk, next_k );
		if( outk >= next_k ){
			pr_out();
			next_k += showevery;
		}
	}

	unlock( pbuffer->semid, pbuffer->blocks_free_lock );
}


void
byee( exit_val )
	int exit_val;
{
	if( writer_pid != 0 ){
		if( exit_val != 0 ){
			/* I am shutting down due to an error.
			 * Shut the writer down or else it will try to access
			 * the freed up locks */
			end_writer();
		}
		wait_for_writer_end();
	}

	if( free_shm ){
		buffer_remove();
	}

#ifdef SIGCHLD
	signal( SIGCHLD, SIG_IGN );
#else
#ifdef SIGCLD
	signal( SIGCLD, SIG_IGN );
#endif
#endif

	/* If the child died or was killed show this in the exit value */
	if( writer_status ){
		if( WEXITSTATUS( writer_status ) || WIFSIGNALED( writer_status ) ){
			if( debug )
				log_it( "writer died badly: 0x%04x\n", writer_status );
			exit( -2 );
		}
	}

        log_it("Thread is exiting\n");
	exit(exit_val);
}

/* Kill off the writer */
void
end_writer()
{
	if( writer_pid )
		kill( writer_pid, SIGHUP );
}

void
wait_for_writer_end()
{
	int deadpid;

	/* Now wait for the writer to finish */
	while( writer_pid && ((deadpid = wait( &writer_status )) != writer_pid) &&
		deadpid != -1 )
		;
}

void
test_writer()
{
	/* Has the writer gone unexpectedly? */
	if( writer_pid == 0 ){
		log_it( "writer has died unexpectedly\n" );
		byee( -1 );
	}
}

/* Given a string of <num>[<suff>] returns a num
 * suff =
 *   m/M for 1meg
 *   k/K for 1k
 *   b/B for 512
 */
int
do_size( arg )
	char *arg;
{
	char format[ 20 ];
	int ret;

	*format = '\0';
	sscanf( arg, "%d%s", &ret, format );

	switch( *format ){
	case 'm':
	case 'M':
		ret = ret K K;
		break;
	case 'k':
	case 'K':
		ret = ret K;
		break;
	case 'b':
	case 'B':
		ret *= 512;
		break;
	}
	
	return ret;
}

void
pr_out()
{
	log_it( " %10luK\r", outk );
}

#ifdef SYS5
#include <sys/time.h>

#ifndef __alpha
bzero( b, l )
	char *b;
	unsigned l;
{
	memset( b, '\0', l );
}
#endif /* __alpha */

usleep_back()
{
}

void
usleep( u )
	unsigned u;
{
	struct itimerval old, t;
	signal( SIGALRM, usleep_back );
	t.it_interval.tv_sec = 0;
	t.it_interval.tv_usec = 0;
	t.it_value.tv_sec = u / 1000000;
	t.it_value.tv_usec = u % 1000000;
	setitimer( ITIMER_REAL, &t, &old );
	pause();
	setitimer( ITIMER_REAL, &old, NULL );
}
#endif

/* Called before error reports */
void
report_proc()
{
	log_it( "%s: ", proc_string );
}































































