/*
 *  mod_bt - Making Things Better For Seeders
 *  Copyright 2004, 2005, 2006 Tyler MacDonald <tyler@yi.org>
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/* libc */
#include <time.h>
#include <sys/types.h>
#include <sys/socket.h>
/* other libs */
#include <apr.h>
#include <apr_pools.h>
#include <apr_strings.h>
#include <db.h>
#ifdef BTT_WITH_LIBXML2
#include <libxml/tree.h>
#endif
/* local */
#include <libbttracker.h>

static const btt_cxn_scrape_config new_btt_cxn_scrape_config = {
    0, 0, 0,        /* full, verbose, xml */
    NULL,           /* hash list */
    0,              /* hashes */
    HTTP_OK,        /* status */
    { 0 },          /* iter_content */
    0, 0, 0, 0      /* t_downloaders, t_seeds, t_shields, t_completed */
};

static const char* btt_cxn_scrape_html_start =
    "<HTML>\n"
    " <HEAD>\n"
    "  <TITLE>BitTorrent Download Info - mod_bt/"VERSION"</TITLE>\n"
    "  <LINK REL=\"stylesheet\" TYPE=\"text/css\" HREF=\"%s\">\n"
    " </HEAD>\n"
    " <BODY>\n"
    "  <H1>BitTorrent Download Info</H1>\n"
    "  <TABLE CLASS=\"info\">\n"
    "   %s\n"
    "  </TABLE>\n"
    "  <BR/>\n%s"
    "  <TABLE CLASS=\"listing\">\n"
    "   <CAPTION>Infohash Listing:</CAPTION>\n"
    "   <TR CLASS=\"heading\">\n"
    "    <TH>Infohash</TH><TH>Filename</TH><TH>Last Act</TH>\n"
    "    <TH>Seeds</TH><TH>Downloaders</TH><TH>Completed</TH>\n"
    "   </TR>\n"
;

static const char* btt_cxn_scrape_html_end =
    "   <TR CLASS=\"total\">\n"
    "    <TD><B>TOTAL:</B></TD><TD CLASS=\"numeric\">%u / %u</TD>\n"
    "    <TD>&nbsp;</TD><TD CLASS=\"numeric\">%s</TD>\n"
    "    <TD CLASS=\"numeric\">%u</TD><TD CLASS=\"numeric\">%u</TD>\n"
    "   </TR>\n"
    "  </TABLE>\n"
    "  <BR/>\n"
    "  <HR/>\n"
    " </BODY>\n"
    "</HTML>\n"
;

int btt_iter_scrape_wantandadd_hash(
    apr_pool_t* p, DB_TXN* txn, DBC* cursor, DBT* key, DBT* val, void* data
) {
    int ret;
    if((ret = btt_iter_scrape_want_hash(p, txn, cursor, key, val, data)) != 0)
        return ret;
    else
        return btt_iter_scrape_add_hash(p, txn, cursor, key, val, data);
}


btt_cxn_scrape_config btt_make_cxn_scrape_config(
    const char* args, apr_pool_t* p
) {
    btt_cxn_scrape_config rv = new_btt_cxn_scrape_config;
    bt_form_pair fp;
    char* aargs;
    char* cp;
    char* xp;
    int nh = 0;

    if(!args || !*args)
        return rv;

    cp = aargs = apr_pstrdup(p, args);
    
    while((rv.status == HTTP_OK) && cp && *cp) {
        fp = bt_next_form_pair_unesc(cp);
  
        if((cp = fp.eop))
            cp++;
   
        if(fp.key && *fp.key) {
            if(!strcmp("info_hash", fp.key)) {
                if(fp.value_len == BT_INFOHASH_LEN)
                    rv.hashes++;
                else
                    rv.status = HTTP_BAD_REQUEST;
            } else if(!strcmp("full", fp.key)) {
                if(rv.full)
                    rv.status = HTTP_BAD_REQUEST;
                else if(!strcmp("1", fp.value))
                    rv.full = 1;
                else
                    rv.status = HTTP_BAD_REQUEST;
            } else if(!strcmp("verbose", fp.key)) {
                if(rv.verbose || rv.xml)
                    rv.status = HTTP_BAD_REQUEST;
                else if(!strcmp("1", fp.value))
                    rv.verbose = 1;
                else
                    rv.status = HTTP_BAD_REQUEST;
#ifdef BTT_WITH_LIBXML2
            } else if(!strcmp("xml", fp.key)) {
                if(rv.verbose || rv.xml)
                    rv.status = HTTP_BAD_REQUEST;
                else if(!strcmp("1", fp.value))
                    rv.xml = BTT_SCRAPE_XML;
                else
                    rv.status = HTTP_BAD_REQUEST;
#endif
           } else if(!strcmp("html", fp.key)) {
                if(rv.xml)
                    rv.status = HTTP_BAD_REQUEST;
                else if(!strcmp("1", fp.value))
                    rv.xml = BTT_SCRAPE_HTML;
                else
                    rv.status = HTTP_BAD_REQUEST;
            } else {
                fprintf(
                    stderr,
                    "btt_make_cxn_scrape_config: Got an unknown URL parameter "
                    "of \"%s\"\n",
                    fp.key
                );
                fflush(stderr);
                rv.status = HTTP_BAD_REQUEST;
            }
        }
    }

    /* bail on any error */
    if(rv.status != HTTP_OK)
        return rv;
 
    /* no hashes, no need to re-iterate */
    if(!rv.hashes)
        return rv;
 
    /* both full and info_hashes specified?? */
    if(rv.full) {
        rv.status = HTTP_BAD_REQUEST;
        return rv;
    }
 
    rv.hashlist = apr_palloc(p, BT_INFOHASH_LEN * rv.hashes);
    memcpy(aargs, args, strlen(args));
    cp = aargs;
    xp = rv.hashlist;

    while(cp && *cp) {
        fp = bt_next_form_pair_unesc(cp);

        if((cp = fp.eop))
            cp++;
  
        if((fp.key && *fp.key) && !strcmp("info_hash", fp.key)) {
            memcpy(xp, fp.value, BT_INFOHASH_LEN);
            xp += BT_INFOHASH_LEN;
            nh++;
        }
    }
 
    /* this should never happen */
    if(nh != rv.hashes) {
        rv.status = HTTP_SERVER_ERROR;
        rv.hashes = 0;
        rv.hashlist = NULL;
        return rv;
    }
 
    return rv;
}

int btt_cxn_scrape(
    btt_tracker* tracker, apr_pool_t* p, DB_TXN* ptxn, const char* args,
    struct sockaddr_in address, char** content, int* content_length
) {
    btt_txn_iterator peerupdate_iterator[] = {
        { btt_iter_check_peer, NULL },
        { btt_iter_peer_stats, NULL },
        { NULL, NULL }
    };
 
    btt_txn_iterator iterator[] = {
        { btt_iter_scrape_want_hash, NULL },
        { btt_iter_scrape_add_hash, NULL },
        { NULL, NULL }
    };
	
    DBT hash_key;
    DBT hash_val;
    DBC* hash_cur = NULL;
    DB_TXN* txn = NULL;
    btt_infohash* hash;
    unsigned char updating = 0;
    int ret = 0;
    int len = 0;
    char* buf = NULL;

    *content = NULL;
    *content_length = 0;

    btt_cxn_scrape_config s_config = btt_make_cxn_scrape_config(args, p);
    s_config.tracker = tracker;
 
    tracker->s->server_time = time(NULL);
    tracker->s->scrapes++;
 
    if(s_config.status != HTTP_OK) {
        tracker->s->bad_scrapes++;
        return s_config.status;
    }

    if(
        (s_config.full && !(tracker->c->flags & BTT_TRACKER_ALLOW_SCRAPE_FULL))
        ||
        (
            s_config.verbose &&
            !(tracker->c->flags & BTT_TRACKER_ALLOW_SCRAPE_VERBOSE)
        )
        ||
        (s_config.xml && !(tracker->c->flags & BTT_TRACKER_ALLOW_SCRAPE_XML))
        ||
        (
            (!s_config.hashes) &&
            !(tracker->c->flags & BTT_TRACKER_ALLOW_SCRAPE_GENERAL)
        )
    ) {
        tracker->s->bad_scrapes++;
        return HTTP_UNAUTHORIZED;
    }

    if(!s_config.hashes)
        tracker->s->full_scrapes++;

    if(!btt_tracker_refresh_stats(tracker)) {
        tracker->db.env->err(
            tracker->db.env, ret, "bt_cxn_scrape(): bt_tracker_refresh_stats()"
        );
        goto err;
    }
    
    updating = 1;

    if((ret = btt_txn_start(tracker, ptxn, &txn, 0)) != 0) {
        tracker->s->bad_scrapes++;
        return HTTP_SERVER_ERROR;
    }
 
    bzero(&hash_key, sizeof(hash_key));
    bzero(&hash_val, sizeof(hash_val));
 
    hash_key.data = apr_palloc(p, BT_INFOHASH_LEN);
    hash_key.ulen = BT_INFOHASH_LEN;
    /* hash_key.size = 0; */
    hash_key.flags = DB_DBT_USERMEM;
 
    hash = hash_val.data = apr_palloc(p, sizeof(btt_infohash));
    hash_val.ulen = sizeof(btt_infohash);
    /* hash_val.size = 0; */
    hash_val.flags = DB_DBT_USERMEM;

    if(
        (ret = tracker->db.hashes->cursor(
            tracker->db.hashes, txn, &hash_cur,
            (updating ? BTT_WRITE_CURSOR(tracker) : 0)
        ))
        != 0
    ) {
        tracker->db.hashes->err(
            tracker->db.hashes, ret, "bt_cxn_scrape(): hashes->cursor()"
        );
        s_config.status = HTTP_SERVER_ERROR;
        goto err;
    }

    apr_pool_create(&s_config.rv.pool, p);
    s_config.rv.buffer_length = (
        (
            (s_config.hashes ? s_config.hashes : tracker->s->num_hashes)
            * BT_SHORT_STRING
        )
        + 10240
    );
    s_config.rv.content = malloc(s_config.rv.buffer_length);
    s_config.rv.hashandpeer.tracker = tracker;

    if(s_config.xml) {
        if(s_config.xml == BTT_SCRAPE_XML) {
            if((len = btt_tracker2xml(s_config.rv.pool, tracker, &buf)) > 0) {
                sprintf(s_config.rv.content, "%s\n", buf);
                s_config.rv.content_length = len + 1;
            }
        } else if(s_config.xml == BTT_SCRAPE_HTML) {
            if(
                (len = btt_tracker2info_trs(s_config.rv.pool, tracker, &buf))
                > 0
            ) {
                sprintf(
                    s_config.rv.content, btt_cxn_scrape_html_start,
                    tracker->c->stylesheet, buf,
                    (
                        tracker->c->root_include[0] ? apr_pstrcat(
                            s_config.rv.pool, "<!--#include virtual=\"",
                            tracker->c->root_include, "\" -->\n", NULL
                        )
                        : ""
                    )
                );
                len = s_config.rv.content_length = strlen(s_config.rv.content);
            }
        } else {
            fprintf(
                stderr,
                "btt_cxn_scrape(%s): Got something that isn't BT_SCRAPE_XML "
                "or BT_SCRAPE_XML in config.xml!\n",
                args
            );
            fflush(stderr);
            s_config.status = HTTP_SERVER_ERROR;
            goto err;
        }
    } else {
        sprintf(s_config.rv.content, "d5:filesd");
        len = s_config.rv.content_length = strlen(s_config.rv.content);
    }
 
    if(!len) {
        fprintf(
            stderr, "bt_cxn_scrape(%s): Didn't get any starting content!\n",
            args
        );
        fflush(stderr); 
        s_config.status = HTTP_SERVER_ERROR;
        goto err;
    }

    peerupdate_iterator[0].data = tracker;
    peerupdate_iterator[1].data = hash;
 
    iterator[0].data = &s_config;
    iterator[1].data = &s_config;
 
    while(!ret) {
        if(
            (ret = hash_cur->c_get(
                hash_cur, &hash_key, &hash_val,
                DB_NEXT | (updating ? DB_RMW : 0)
            ))
            == 0
        ) {
            hash = hash_val.data;
            if(updating == 1) {
                if(
                    (tracker->s->server_time - hash->last_peer_t) >
                    (tracker->c->return_interval * 2)
                ) {
                    updating = 2;
                    if(hash->peers) {
                        hash->peers = hash->seeds = hash->shields = 0;
                        if(
                            (ret = btt_txn_iterate_peerlist(
                                tracker, s_config.rv.pool, hash, txn, 0,
                                DB_RMW, peerupdate_iterator
                            ))
                            != 0
                        ) {
                            if(ret == DB_NOTFOUND) {
                                ret = 0;
                            } else {
                                tracker->s->bad_scrapes++;
                                tracker->db.peers->err(
                                    tracker->db.peers, ret,
                                    "bt_cxn_scrape(): bt_txn_iterate_peerlist"
                                );
                                s_config.status = HTTP_SERVER_ERROR;
                            }
                        }
                    }
     
                    if(!ret)
                        ret = hash_cur->c_put(
                            hash_cur, &hash_key, &hash_val, DB_CURRENT
                        );
                }
            }
   
            if(!ret)
                if(
                    (ret = btt_txn_iterate_callbacks(
                        s_config.rv.pool, txn, hash_cur, &hash_key, &hash_val,
                        iterator))
                    != 0
                )
                    tracker->db.hashes->err(
                        tracker->db.hashes, ret,
                        "bt_cxn_scrape(): bt_txn_iterate_callbacks"
                    );
   
            if(updating == 2) {
                if((!ret) && (!hash->peers))
                    if(
                        (!hash->register_t) &&
                        (
                            (tracker->s->server_time - hash->last_t) >
                            tracker->c->hash_min_age
                        )
                    )
                        if(
                            (
                                tracker->s->num_hashes >
                                tracker->c->hash_watermark
                            )
                            ||
                            (
                                (tracker->s->server_time - hash->last_t) >
                                tracker->c->hash_max_age
                            )
                        ) {
                            if((ret = hash_cur->c_del(hash_cur, 0)) == 0)
                                tracker->s->num_hashes--;
                            else
                                tracker->db.hashes->err(
                                    tracker->db.hashes, ret,
                                    "bt_cxn_scrape(): c_del"
                                );
                        }
  	
                        updating = 1;
            }
        }
    }
 
    hash_cur->c_close(hash_cur);
    hash_cur = NULL;
 
    if(ret != DB_NOTFOUND && ret != BT_CALLBACK_STOP_ALL) {
        s_config.status = HTTP_SERVER_ERROR;
        goto err;
    }

    if(s_config.xml == BTT_SCRAPE_XML) {
        /* TODO: Proper DTD, start/end tags, etc */
    } else if(s_config.xml == BTT_SCRAPE_HTML) {
        if(s_config.t_shields) {
            buf = apr_psprintf(
                s_config.rv.pool, "%u (+%u)", s_config.t_seeds,
                s_config.t_shields
            );
        } else {
            buf = apr_psprintf(s_config.rv.pool, "%u", s_config.t_seeds);
        }
  
        buf = apr_psprintf(
            s_config.rv.pool, btt_cxn_scrape_html_end, s_config.t_hashes,
            tracker->s->num_hashes, buf, s_config.t_downloaders,
            s_config.t_completed
        );
        memcpy(
            &(s_config.rv.content[s_config.rv.content_length]),
            buf, strlen(buf) + 1
        );
        s_config.rv.content_length = strlen(s_config.rv.content);
    } else {
        s_config.rv.content[s_config.rv.content_length] = 'e';
        s_config.rv.content_length++;
        s_config.rv.content[s_config.rv.content_length] = 'e';
        s_config.rv.content_length++;
    }

    s_config.rv.content[s_config.rv.content_length] = 0;

    if(txn)
        if((ret = txn->commit(txn, 0)) != 0) {
            tracker->db.env->err(
                tracker->db.env, ret, "bt_cxn_scrape(): txn->commit()"
            );
            s_config.status = HTTP_SERVER_ERROR;
            goto err;
        }

    *content = apr_palloc(p, s_config.rv.content_length);
    memcpy(*content, s_config.rv.content, s_config.rv.content_length);
    *content_length = s_config.rv.content_length;

    free(s_config.rv.content);
    s_config.rv.content = NULL;
    apr_pool_destroy(s_config.rv.pool);
    s_config.rv.pool = NULL;

    return HTTP_OK;
 
err:

    if(s_config.rv.pool) {
        apr_pool_destroy(s_config.rv.pool);
        s_config.rv.pool = NULL;
    }
 
    if(s_config.rv.content)
        free(s_config.rv.content);
 
    if(hash_cur) {
        hash_cur->c_close(hash_cur);
        hash_cur = NULL;
    }
 
    if(txn)
        txn->abort(txn);

    return s_config.status;
}
