#!/usr/bin/python -ttu
# vim: ai ts=4 sts=4 et sw=4

#    HintWindow.py: A simple window that shows a hint to the user

#    Copyright (c) 2007, 2008 Intel Corporation
#
#    This program is free software; you can redistribute it and/or modify it
#    under the terms of the GNU General Public License as published by the Free
#    Software Foundation; version 2 of the License
#
#    This program is distributed in the hope that it will be useful, but
#    WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
#    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
#    for more details.
#
#    You should have received a copy of the GNU General Public License along
#    with this program; if not, write to the Free Software Foundation, Inc., 59
#    Temple Place - Suite 330, Boston, MA 02111-1307, USA.

import gc
import gobject
import gtk
import gtk.gdk
import gtk.glade
import os
import sys
import time

# Import our app's libraries
import utils

BACKGROUND_COLOR = "black" # the background color of the window and image
SHOW_DURATION = 2500 # 2.5 seconds

class Timer(object):
    """This is simple way of having something happen after so much time has
    passed."""
    def __init__(self, time, callback):
        """Initialize the object by passing the delay time and function to call
        when the time has expired"""
        self.__time = time
        self.__caller_callback = callback
        self.__timeout_id = None
    
    def start(self):
        """Kick off the timer"""
        if not self.__timeout_id:
            self.__timeout_id = gobject.timeout_add(self.__time, self.__callback)
            return True
        return False

    def __callback(self):
        # Call the caller's callback function and then stop ourselves from
        # getting triggered again
        self.__caller_callback()
        self.stop()
        return False
    
    def stop(self):
        if self.__timeout_id:
            gobject.source_remove(self.__timeout_id)
            self.__timeout_id = None
            return True
        return False
    
    def restart(self):
        self.stop()
        self.start()

# Show image in a borderless window for SHOW_DURATION milliseconds.
# FIXME: Do we know that this works correctly?  One question I have is are we
# just counting on garbage collection not happening while the hint is being
# displayed since the object is being created and then immediately discarded in
# our current usage?
class HintWindow(object):
    """A window which shows a hint to the user"""
    def __init__(self, image_file, x = None, y = None):
        """init function: image_file is the image to load. x,y is position to
        show the window (absolute).  If x or y are not passed in then it will
        be centered to the screen size."""
        self.begin_show_time = time.time()        
        # load ui from glade
        self.wTree = gtk.glade.XML("dialogs.glade", 'HintWindow')
        self.ht_window = self.wTree.get_widget('HintWindow')
        # set the background color
        color = gtk.gdk.color_parse(BACKGROUND_COLOR)
        self.ht_window.modify_bg(gtk.STATE_NORMAL , color) 
        self.ht_window.modify_bg(gtk.STATE_ACTIVE , color)
        self.ht_window.modify_bg(gtk.STATE_PRELIGHT , color)
        self.ht_window.modify_bg(gtk.STATE_SELECTED , color)
        self.ht_window.modify_bg(gtk.STATE_INSENSITIVE , color)
        # Load background image
        self.hw_image = self.wTree.get_widget("hw_image")        
        self.hw_image.set_from_file( image_file)        
        pb =  self.hw_image.get_pixbuf()
        h = pb.get_height()
        w = pb.get_width()
        self.ht_window.set_resizable(True)
        self.ht_window.resize( w +10, h+10 )
        self.ht_window.set_resizable(False)
        # set the background color (cover the border)
        self.hw_image.modify_bg(gtk.STATE_NORMAL , color)
        self.hw_image.modify_bg(gtk.STATE_ACTIVE , color)
        self.hw_image.modify_bg(gtk.STATE_PRELIGHT , color)
        self.hw_image.modify_bg(gtk.STATE_SELECTED , color)
        self.hw_image.modify_bg(gtk.STATE_INSENSITIVE , color)
        # the Pixbuf costs memory
        gc.collect()
        # move to the position
        width, height = utils.getScreenSize()
        if x == None:
            x = (width / 2) - (w / 2)
        if y == None:
            y = (height / 2) - (h / 2)
        self.ht_window.move(x, y)
        
    def _hide_timer_cb(self):
        self.ht_window.hide()

    def show_hint(self):       
        self.ht_window.show()
        self.hide_timer = Timer(SHOW_DURATION , self._hide_timer_cb)
        self.hide_timer.start()
